/*ffun evaluate total flux for sum of all continuum sources */
/*ffun1 derive flux at a specific energy, for one continuum */
#include "cddefines.h"
#include "readtable.h"
#include "rfield.h"
#include "physconst.h"
#include "ipoint.h"
#include "bit32.h"
#include "ffun.h"

double ffun(double anu)
{
	double ffun_v;
	static int lgWarn = FALSE;

#	ifdef DEBUG_FUN
	fputs( "<+>ffun()\n", debug_fp );
#	endif

	/* FFUN returns the sum of photons per unit time, area, energy,
	 * for all continuua in the calculation
	 * we loop over all nspec continuum sources - ipspec points to each
	 * */
	ffun_v = 0.;
	for( rfield.ipspec=0; rfield.ipspec < rfield.nspec; rfield.ipspec++ )
	{
		ffun_v += ffun1(anu)*rfield.spfac[rfield.ipspec];
	}

	if( (ffun_v > 1e35 && (!lgWarn)) && bit32.lgBit32 )
	{
		lgWarn = TRUE;
		fprintf( ioQQQ, " FFUN:  The net continuum is very intense for a 32 bit cpu.\n" );
		fprintf( ioQQQ, " I will try to press on, but may have problems.\n" );
	}


#	ifdef DEBUG_FUN
	fputs( " <->ffun()\n", debug_fp );
#	endif
	return( ffun_v );
}

/*ffun1 derive flux at a specific energy, for one continuum */
double ffun1(double xnu)
{
	char chKey[6];
	long int i;
	double fac, 
	  ffun1_v, 
	  y;
	static int lgWarn = FALSE;

#	ifdef DEBUG_FUN
	fputs( "<+>ffun1()\n", debug_fp );
#	endif


	/* confirm that pointer is within range */
	assert( rfield.ipspec >= 0);
	assert( rfield.ipspec < rfield.nspec );

	/* FFUN1 returns photons per unit time, area, energy, for one continuum
	 * ipspec is the pointer to the continuum source, in the order
	 * entered on the command lines */

	/*begin sanity check */
	assert( xnu >= rfield.emm*0.99 );
	assert( xnu <= rfield.egamry*1.01 );
	/*end sanity check */

	strcpy( chKey, rfield.chSpType[rfield.ipspec] );

	if( strcmp(chKey,"AGN  ") == 0 )
	{
		/* power law with cutoff at both ends
		 * nomenclature all screwed up - slope is cutoff energy in Ryd,
		 * cutoff(i,1) is ratio of two continua from alpha ox
		 * cutoff(i,2) is slope of bb temp */
		ffun1_v = pow(xnu,-1. + rfield.cutoff[1][rfield.ipspec])*
		  sexp(xnu/rfield.slope[rfield.ipspec])*sexp(0.01/
		  xnu);
		/* only add on x-ray for energies above 0.1 Ryd */
		if( xnu > 0.1 )
		{
			if( xnu < 7350. )
			{
				/* cutoff is actually normalization constant
				 * below 100keV continuum is nu-1 */
				ffun1_v += pow(xnu,rfield.cutoff[2][rfield.ipspec] - 
				  1.)*rfield.cutoff[0][rfield.ipspec]*sexp(1./
				  xnu);
			}
			else
			{
				ffun1_v += pow(7350.,rfield.cutoff[2][rfield.ipspec] - 
				  1.)*rfield.cutoff[0][rfield.ipspec]/
				  POW3(xnu/7350.);
			}
		}

	}
	else if( strcmp(chKey,"POWER") == 0 )
	{
		/* power law with cutoff at both ends */
		ffun1_v = pow(xnu,-1. + rfield.slope[rfield.ipspec])*
		  sexp(xnu/rfield.cutoff[0][rfield.ipspec]+rfield.cutoff[1][rfield.ipspec]/
		  xnu);

	}
	else if( strcmp(chKey,"BLACK") == 0 )
	{
		const double db_log = log(DBL_MAX);

		/* black body */
		fac = TE1RYD*xnu/rfield.slope[rfield.ipspec];
		/* >>>chng 00 apr 13 from 80 to log(dbl_max) */
		if( fac > db_log )
		{
			ffun1_v = 0.;
		}
		else if( fac > 1.e-5 )
		{
			ffun1_v = xnu*xnu/(exp(fac) - 1.);
		}
		else
		{
			ffun1_v = xnu*xnu/(fac*(1. + fac/2.));
		}

	}
	else if( strcmp(chKey,"INTER") == 0 )
	{
		/* interpolate on input spectrum */
		if( xnu >= rfield.tNuRyd[0][rfield.ipspec] )
		{
			/* loop starts at second array element, [1], since want to 
			 * find next continuum energy greater than desired point */
			for( i=1; i < rfield.nupper; i++ )
			{
				if( xnu < rfield.tNuRyd[i][rfield.ipspec] )
				{
					/* points to interpolate upon are in f_nu */
					y = rfield.tFluxLog[i-1][rfield.ipspec] + 
					  rfield.tslop[i-1][rfield.ipspec]*
					  log10(xnu/rfield.tNuRyd[i-1][rfield.ipspec]);
					assert( y >= MIN2( rfield.tFluxLog[i-1][rfield.ipspec] , rfield.tFluxLog[i][rfield.ipspec] )) ;
					assert( y <= MAX2( rfield.tFluxLog[i-1][rfield.ipspec] , rfield.tFluxLog[i][rfield.ipspec] )) ;

					/* return value is photon density, div by energy */
					ffun1_v = pow(10.,y)/xnu ;
					
#					ifdef DEBUG_FUN
					fputs( " <->ffun1()\n", debug_fp );
#					endif
					return( ffun1_v );
				}
			}
			/* energy above highest in table */
			ffun1_v = 0.;
		}
		else
		{
			/* energy below lowest on table */
			ffun1_v = 0.;
		}

	}
	else if( strcmp(chKey,"BREMS") == 0 )
	{
		/* brems continuum, rough gaunt factor */
		fac = TE1RYD*xnu/rfield.slope[rfield.ipspec];
		ffun1_v = sexp(fac)/pow(xnu,1.2);

	}
	else if( strcmp(chKey,"LASER") == 0 )
	{
#		ifdef BIG
#			undef BIG
#		endif
#		define	BIG	1e10
#		ifdef SMALL
#			undef SMALL
#		endif
#		define	SMALL	1e-25
		/* a laser, mostly one frequency */
		/* >>chng 01 jul 01, was hard-wired 0.05 rel frac, change to optional
		 * second parameter, with default of 0.05 */
		/*if( xnu > 0.95*rfield.slope[rfield.ipspec] && xnu < 
		  1.05*rfield.slope[rfield.ipspec] )*/
		if( xnu > (1.-rfield.cutoff[0][rfield.ipspec])*rfield.slope[rfield.ipspec] && 
			xnu < (1.+rfield.cutoff[0][rfield.ipspec])*rfield.slope[rfield.ipspec] )
		{
			ffun1_v = BIG;
		}
		else
		{
			ffun1_v = SMALL;
		}

	}
	else if( strcmp(chKey,"READ ") == 0 )
	{
		/* if this is first time called then we need to read in file */
		if( rfield.tslop[0][rfield.ipspec] == 0. )
		{
			/* >>chng 01 nov 01, array index for below was bogus, only worked for a single continuum */
			ReadTable(rfield.ioTableRead[rfield.ipspec]);
			rfield.tslop[0][rfield.ipspec] = 1.;
		}
		/* use array of values read in on TABLE READ command */
		if( xnu >= rfield.egamry )
		{
			ffun1_v = 0.;
		}
		else
		{
			i = ipoint(xnu);
			if( i > rfield.nupper || i < 1 )
			{
				ffun1_v = 0.;
			}
			else
			{
				ffun1_v = rfield.ConTabRead[i-1]/rfield.anu[i-1]/rfield.anu[i-1];
			}
		}

	}
	else if( strcmp(chKey,"VOLK ") == 0 )
	{
		/* use array of values read in from Kevin Volk's rebinning of
		 * large atlas grids */
		if( xnu >= rfield.egamry )
		{
			ffun1_v = 0.;
		}
		else
		{
			i = ipoint(xnu);
			if( i > rfield.nupper )
			{
				fprintf( ioQQQ, " ffun1: Too many points - increase ncell\n" );
				fprintf( ioQQQ, " cell needed=%4ld ncell=%4ld\n", 
				  i, rfield.nupper );
				puts( "[Stop in ffun1]" );
				cdEXIT(1);
			}
			if( i > rfield.nupper || i < 1 )
			{
				ffun1_v = 0.;
			}
			else
			{
				/* bug fixed Jul 9 93: FFUN1 = TSLOP(IPSPEC,I) / ANU(I) / ANU(I)
				 *   i has value 939 */
				ffun1_v = rfield.tslop[i-1][rfield.ipspec]/
				  rfield.anu[i-1];
			}
		}
	}
	else
	{
		fprintf( ioQQQ, " ffun1: I do not understand continuum label %5.5s\n", 
		  chKey );
		puts( "[Stop in ffun1]" );
		cdEXIT(1);
	}

	if( (ffun1_v > 1e35 && (!lgWarn)) && bit32.lgBit32 )
	{
		lgWarn = TRUE;
		fprintf( ioQQQ, " FFUN1:  Continuum%4ld is very intense for a 32 bit cpu.\n", 
		  rfield.ipspec );
		fprintf( ioQQQ, " I will try to press on, but may have problems.\n" );
	}

#	ifdef DEBUG_FUN
	fputs( " <->ffun1()\n", debug_fp );
#	endif
	return( ffun1_v );
}

