/*grain main routine to converge grains thermal solution */
#include "cddefines.h"
#include "physconst.h"
#include "atomcwgt.h"
#include "heat.h"
#include "rfield.h"
#include "hmi.h"
#include "trace.h"
#include "converge.h"
#include "phycon.h"
#include "doppvel.h"
#include "taulines.h"
#include "heavy.h"
#include "negdrg.h"
#include "interpolate.h"
#include "hevmolec.h"
#include "ionfracs.h"
#include "recoil.h"
#include "ipoint.h"
#include "grainvar.h"
#include "grain.h"
#include "abundances.h"
#include "yield.h"
#include "photrate.h"

#ifdef INCLUDE_OLD_GRAINS
#include "mie.h"
#include "dustop.h"
#endif


/* the next define is for debugging purposes only, uncomment to activate */
/*  #define WD_TEST2 1 */

#define NINT(X) ((long)((X) < 0. ? (X)-0.5 : (X)+0.5))

/* no braces around PTR needed since it needs to be an lvalue */
#define FREE_CHECK(PTR) { assert( PTR != NULL ); free( PTR ); PTR = NULL; }
#define FREE_SAFE(PTR) { if( PTR != NULL ) free( PTR ); PTR = NULL; }

#define POT2CHRG(X) ((X)*EVRYD/ELEM_CHARGE*gv.bin[nd]->Capacity - 1.)
#define CHRG2POT(X) (((X)+1.)*ELEM_CHARGE/EVRYD/gv.bin[nd]->Capacity)
#define ONE_ELEC   (ELEM_CHARGE/EVRYD/gv.bin[nd]->Capacity)

/*#define EPSM(Z_0,Z) ( ((Z) <= (Z_0)-1) ? 1. : (((Z) >= (Z_0)) ? 0. : (double)((Z_0)-(Z))) )*/
#define EPSP(Z_0,Z) ( ((Z) <= (Z_0)) ? 0. : (((Z) >= (Z_0)+1) ? 1. : (double)((Z)-(Z_0))) )

static const int INCL_TUNNEL = TRUE;
static const int NO_TUNNEL = FALSE;

static const int ALL_STAGES = TRUE;
static const int NONZERO_STAGES = FALSE;

/*================================================================================*/
/* these are used for setting up grain emissivities in InitEmissivities() */

/* NTOP is number of bins for temps between GRAIN_TMID and GRAIN_TMAX */
static const long NTOP = NDEMS/5;

/*================================================================================*/
/* miscellaneous grain physics */

/* constant for thermionic emissions, 7.501e20 e/cm^2/s/K^2 */
static const double THERMCONST = PI4*ELECTRON_MASS*POW2(BOLTZMANN)/POW3(HPLANCK);

/* sticking probabilities */
static const double STICK_ELEC = 0.5;
static const double STICK_ION = 1.0;

/* mean free path for electron penetrating grain, in cm */
static const double MEAN_PATH = 1.e-7;

/*================================================================================*/
/* these are used when iterating the grain charge in GrainCharge() */
static const double TOLER = CONSERV_TOL/10.;
static const double BIG_POTENTIAL = 5.;

static const double BRACKET_STEP = 0.2;
static const long BRACKET_MAX = 50L;

/* maximum number of tries to converge charge/temperature in GrainChargeTemperature() */
static const long CT_LOOP_MAX = 10L;

/*================================================================================*/
/* global variables */

static int lgGvInitialized = FALSE;

/* a_0 thru a_2 constants for calculating IP_V and EA, in cm */
static const double AC0 = 3.e-9;
static const double AC1G = 4.e-8;
static const double AC2G = 7.e-8;

/* max number of global charge states that we can cache simultaneously */
#define NGCS 128

/* >>chng 01 sep 13, create structure for dynamically allocating backup store, PvH */
/* the following are placeholders for intermediate results that depend on grain type,
 * however they are only used inside the main loop over nd in GrainChargeTemp(),
 * so it is OK to reuse the same memory for each grain bin separately. */
/* >>chng 01 dec 19, added entry for Auger rate, PvH */
typedef struct t_BUS {
	int lgGlobalValid;                        /* are next four items properly initialized ? */
	long int RecomZ0[NCHS][LIMELM][LIMELM+1]; /* ionization stage the atom/ion recombines to upon impact */
	float RecomEn[NCHS][LIMELM][LIMELM+1];    /* chemical energy released into grain upon impact, in Ryd */
	float ChemEn[NCHS][LIMELM][LIMELM+1];     /* net contribution of ion recomb to grain heating, in Ryd */
	float *yhat[NCHS]/*[NC_ELL]*/;            /* electron yield per absorbed photon, valence band */
	double *cs_pdt[NCHS]/*[NC_ELL]*/;         /* photo-detachment cross section, for default depl */

	/* these are caches to remember electron emission and recombination rates */
	/* the cache will be filled circularly, if the pointer GlobRPtr or GlobEPtr
	 * overflows, we will simply start overwriting the oldest results. This is
	 * accomplished by using GlobRPtr%NGCS instead of GlobRPtr */

	long GlobLast;       /* pointer to last match */

	long GlobRPtr;       /* pointer for storing next recombination rates */
	long GlobRZg[NGCS];  /* grain charge */
	double RSum1[NGCS];  /* rate from colliding electrons */
	double RSum2[NGCS];  /* donation rate from colliding ions */

	long GlobEPtr;       /* pointer for storing next emission rates */
	long GlobEZg[NGCS];  /* grain charge */
	double ESum1a[NGCS]; /* PE rate from valence band */
	double ESum1b[NGCS]; /* PE rate by Auger electrons */
	double ESum1c[NGCS]; /* PE rate from conduction band */
	double ESum2[NGCS];  /* electron loss due to recombination with colliding ions */
} BackupStore;

static BackupStore *cache = NULL;

/* initialize grain emissivities */
static void InitEmissivities(void);
/* PlanckIntegral compute total radiative cooling due to large grains */
static double PlanckIntegral(double,long,long);
/* allocate backup store for global data */
static void AllocGlobalData(void);
/* invalidate global data to avoid accidental use outside of scope of validity */
static void ReturnGlobalData(void);
/* iterate grain charge and temperature */
static void GrainChargeTemp(void);
/* GrainCharge compute grains charge */
static void GrainCharge(long,/*@out@*/double*,double);
/* bracket the solution for the grain potential */
static void BracketGrainPot(long,/*@in@*/double*,/*@out@*/double*,/*@out@*/double*,/*@in@*/double*,
			    /*@out@*/double*,/*@out@*/double*,/*@in@*/double*,/*@out@*/double*,
			    /*@out@*/double*,/*@out@*/double*);
/* GrainElecRecomb compute electron recombination onto grain surface */
static double GrainElecRecomb(long);
/* grain electron recombination rates for single charge state */
static double GrainElecRecomb1(long,long,/*@out@*/double*,/*@out@*/double*);
/* GrainElecEmis computes electron loss rate for the grain species
 * due to photoelectric effect, positive ion recombinations on the
 * grain surface and thermionic emissions */
static double GrainElecEmis(long,/*@out@*/double*);
/* grain electron emission rates for single charge state */
static double GrainElecEmis1(long,long,/*@out@*/double*,/*@out@*/double*,
			     /*@out@*/double*,/*@out@*/double*,/*@out@*/double*);
/* correction factors for grain charge screening (including image potential
 * to correct for polarization of the grain as charged particle approaches). */
static void GrainScreen(long,long,double,double*,double*);
/* helper function for GrainScreen */
static double ThetaNu(double);
/* update items that depend on grain potential */
static void UpdatePot(long);
/* calculate charge state populations */
static void GetFracPop(long,double,long,/*@out@*/long*,/*@out@*/double[]);
/* this routine updates all quantities that depend on grain charge by calculating them */
static void UpdatePotCalc(long,long,long);
/* this routine updates all quantities that depend on grain charge by copying them */
static void UpdatePotCopy(long,long,long,long);
/* find highest ionization stage with non-zero population */
static long HighestIonStage(void);
/* determine charge Z0 ion recombines to upon impact on grain */
static void UpdateRecomZ0(long,long,int);
/* helper routine for UpdatePot */
static void GetPotValues(long,long,/*@out@*/double*,/*@out@*/double*,/*@out@*/double*,
			 /*@out@*/double*,/*@out@*/double*,int);
/* given grain nd in charge state nz, and incoming ion (ipZ,ion),
 * detemine outgoing ion (ipZ,Z0) and chemical energy ChEn released
 * ChemEn is net contribution of ion recombination to grain heating */
static void GrainIonColl(long,long,long,long,const double[],const double[],/*@out@*/long*,
			 /*@out@*/float*,/*@out@*/float*);
/* initialize ion recombination rates on grain species nd */
static void GrainRecomRates(long);
/* this routine updates all grain quantities that depend on radius */
static void GrainUpdateRadius1(void);
/* this routine adds the photo-dissociation cs to gv.dstab */
static void GrainUpdateRadius2(long nd);
/* GrainTemperature computes grains temperature, and gas cooling */
static void GrainTemperature(long,/*@out@*/float*,/*@out@*/double*,/*@out@*/double*,
			     /*@out@*/double*,/*@out@*/double*);
/* GrainCollHeating computes grains collisional heating cooling */
static void GrainCollHeating(long,/*@out@*/float*,/*@out@*/float*);
/* GrainDrift computes grains drift velocity */
static void GrainDrift(void);
/*GrnVryDpth set grains abundance as a function of depth into cloud*/
static double GrnVryDpth(long);



/* >>chng 01 oct 29, introduced gv.bin[nd]->cnv_H_pGR, cnv_GR_pH, etc. PvH */

/* this routine is called by zero(), so it should contain initializations
 * that need to be done every time before the input lines get parsed */
void GrainZero(void)
{
#	ifdef DEBUG_FUN
	fputs( "<+>GrainZero()\n", debug_fp );
#	endif

	gv.TotalEden = 0.;
	gv.lgQHeatAll = FALSE;
	gv.lgQHeatOn = TRUE;
	gv.lgDHetOn = TRUE;
	gv.lgDColOn = TRUE;
	gv.GrainMetal = 1.;
	gv.lgBakes = FALSE;
	gv.nChrgRequested = NCHRG_DEFAULT;
	gv.qnflux = LONG_MAX;
	gv.ReadPtr = 0L;

	/* >>>chng 01 may 08, return memory possibly allocated in previous calls to cloudy(), PvH
	 * this routine MUST be called before ParseCommands() so that grain commands find a clean slate */
	ReturnGrainBins();

#	ifdef INCLUDE_OLD_GRAINS
	/* >>chng 00 mar 10, moved call to OldStyleGrainBlockDataInit from cdInit, PvH */
	/* initialize some old style grain data before other grains are read in */
	/* this call MUST be made before ParseCommands() is called; ParseGrain() relies on this */
	OldStyleGrainBlockDataInit();
#	endif

#	ifdef DEBUG_FUN
	fputs( " <->GrainZero()\n", debug_fp );
#	endif
	return;
}


/* this routine is called by StartIter(), so anything that needs to be reset before each
 * iteration starts should be put here; typically variables that are integrated over radius */
void GrainStartIter(void)
{
	long nd;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainStartIter()\n", debug_fp );
#	endif

	if( gv.lgDustOn )
	{
		gv.reftot = 0.;
		for( nd=0; nd < gv.nBin; nd++ )
		{
			/* >>chng 97 jul 5, save and reset this
			 * save grain potential */
			gv.bin[nd]->dstpotsav = gv.bin[nd]->dstpot;
			gv.bin[nd]->qtmin = ( gv.bin[nd]->qtmin_zone1 > 0. ) ?
				gv.bin[nd]->qtmin_zone1 : DBL_MAX;
			gv.bin[nd]->avdust = 0.;
			gv.bin[nd]->avdpot = 0.;
			gv.bin[nd]->avdft = 0.;
			gv.bin[nd]->avDGRatio = 0.;
			gv.bin[nd]->TeGrainMax = -1.f;
			gv.bin[nd]->qpres = QHEAT_INIT_RES;
			gv.bin[nd]->lgEverQHeat = FALSE;
			gv.bin[nd]->QHeatFailures = 0L;
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->GrainStartIter()\n", debug_fp );
#	endif
	return;
}


/* this routine is called by RestartIter(), so anything that needs to be
 * reset or saved after an iteration is finished should be put here */
void GrainRestartIter(void)
{
	long nd;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainRestartIter()\n", debug_fp );
#	endif

	if( gv.lgDustOn )
	{
		for( nd=0; nd < gv.nBin; nd++ )
		{
			/* >>chng 97 jul 5, reset grain potential
			 * reset grain to pential to initial value from previous iteration */
			gv.bin[nd]->dstpot = gv.bin[nd]->dstpotsav;
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->GrainRestartIter()\n", debug_fp );
#	endif
	return;
}


/* this routine is called by ParseSet() */
void SetNChrgStates(long nChrg)
{
#	ifdef DEBUG_FUN
	fputs( "<+>SetNChrgStates()\n", debug_fp );
#	endif

	assert( nChrg >= 2 && nChrg <= NCHS );
	gv.nChrgRequested = nChrg;

#	ifdef DEBUG_FUN
	fputs( " <->SetNChrgStates()\n", debug_fp );
#	endif
	return;
}


long NewGrainBin(void)
{
	long nd;

#	ifdef DEBUG_FUN
	fputs( "<+>NewGrainBin()\n", debug_fp );
#	endif

	assert( lgGvInitialized );

	if( gv.nBin >= NDUST ) 
	{
		fprintf( ioQQQ, " The code has run out of grain bins; increase NDUST and recompile.\n" );
		puts( "[Stop in NewGrainBin]" );
		cdEXIT(1);
	}
	nd = gv.nBin;

	assert( gv.bin[nd] == NULL ); /* prevent memory leaks */
	if( ( gv.bin[nd] = (GrainBin *)MALLOC(sizeof(GrainBin)) ) == NULL ) 
		bad_malloc();

	gv.bin[nd]->dstab1 = NULL;
	gv.bin[nd]->dstsc1 = NULL;
	gv.bin[nd]->y1 = NULL;
	gv.bin[nd]->inv_att_len = NULL;
	gv.bin[nd]->phiTilde = NULL;

	gv.lgDustOn = TRUE;
#	ifdef INCLUDE_OLD_GRAINS
	gv.bin[nd]->lgOldGrain1 = FALSE;
	gv.bin[nd]->lgDustOn1 = FALSE;
#	endif
	gv.bin[nd]->lgQHeat = FALSE;
	gv.bin[nd]->lgDustVary = FALSE;
	gv.bin[nd]->DustDftVel = FLT_MAX;
	gv.bin[nd]->TeGrainMax = FLT_MAX;
	/* NB - this number should not be larger than NCHS */
	gv.bin[nd]->nChrg = gv.nChrgRequested;
	/* this must be zero for the first solutions to be able to converge */
	/* >>chng 00 jun 19, tedust has to be greater than zero
	 * to prevent division by zero in GrainElecEmis and GrainCollHeating, PvH */
	gv.bin[nd]->tedust = 1.;
	/* used to check that energy scale in grains opacity files is same as
	 * current cloudy scale */
	gv.bin[nd]->EnergyCheck = 0.;
	gv.bin[nd]->dstAbund = -FLT_MAX;
	gv.bin[nd]->dstfactor = 1.f;
	gv.bin[nd]->cnv_H_pGR = -DBL_MAX;
	gv.bin[nd]->cnv_H_pCM3 = -DBL_MAX;
	gv.bin[nd]->cnv_CM3_pGR = -DBL_MAX;
	gv.bin[nd]->cnv_CM3_pH = -DBL_MAX;
	gv.bin[nd]->cnv_GR_pH = -DBL_MAX;
	gv.bin[nd]->cnv_GR_pCM3 = -DBL_MAX;
	gv.nBin++;

#	ifdef DEBUG_FUN
	fputs( " <->NewGrainBin()\n", debug_fp );
#	endif
	return(nd);
}

void ReturnGrainBins(void)
{
	long nd;

#	ifdef DEBUG_FUN
	fputs( "<+>ReturnGrainBins()\n", debug_fp );
#	endif

	if( lgGvInitialized )
	{
		/* >>chng 01 sep 12, allocate/free [rfield.nupper] arrays dynamically */
		for( nd=0; nd < gv.nBin; nd++ ) 
		{
			assert( gv.bin[nd] != NULL );

			FREE_SAFE( gv.bin[nd]->dstab1 );
			FREE_SAFE( gv.bin[nd]->dstsc1 );
			FREE_SAFE( gv.bin[nd]->y1 );
			FREE_SAFE( gv.bin[nd]->inv_att_len );
			FREE_SAFE( gv.bin[nd]->phiTilde );
			FREE_CHECK( gv.bin[nd] );
		}

		FREE_SAFE( gv.dstab );
		FREE_SAFE( gv.dstsc );
		FREE_SAFE( gv.GrainEmission );
		FREE_SAFE( gv.GraphiteEmission );
		FREE_SAFE( gv.SilicateEmission );
	}
	else
	{
		/* >>chng 01 sep 12, moved initialization of data from NewGrainBin to here, PvH */
		/* >>chng 01 may 08, make sure bin pointers are properly initialized, PvH */
		for( nd=0; nd < NDUST; nd++ )
		{
			gv.bin[nd] = NULL;
		}

		gv.dstab = NULL;
		gv.dstsc = NULL;
		gv.GrainEmission = NULL;
		gv.GraphiteEmission = NULL;
		gv.SilicateEmission = NULL;

		lgGvInitialized = TRUE;
	}

	gv.lgDustOn = FALSE;
	gv.nBin = 0;

#	ifdef DEBUG_FUN
	fputs( " <->ReturnGrainBins()\n", debug_fp );
#	endif
	return;
}

/*InitGrains, called one time at initialization of calculation, not after every iteration */
void InitGrains(void)
{
	long int i,
	  ipZ,
	  nd,
	  nd2;
#	ifdef INCLUDE_OLD_GRAINS
	/* >>chng 01 aug 22, allocate space instead */
	double *InvDepCar/*[NC_ELL]*/=NULL,
	  *InvDepSil/*[NC_ELL]*/=NULL;
#	endif

#	ifdef DEBUG_FUN
	fputs( "<+>InitGrains()\n", debug_fp );
#	endif

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, " InitGrains called.\n" );
#		ifdef INCLUDE_OLD_GRAINS
		/* this is really only needed when old style grains are
		 * turned on, otherwise information below is enough... */
		fprintf( ioQQQ, "    There are %ld grain bins.\n", gv.nBin );
		fprintf( ioQQQ, "    Their names are:" );
		for( nd=0; nd < gv.nBin; nd++ ) 
		{
			fprintf( ioQQQ, "%s %c,", gv.bin[nd]->chDstLab, TorF(gv.bin[nd]->lgDustOn1) );
		}
		fprintf( ioQQQ, "\n" );
#		endif
	}

#	ifdef INCLUDE_OLD_GRAINS
	/* return memory of unused grain bins
	 * the first loop swaps all grain bins until
	 * the used ones are up front.
	 * the second loop frees the unused bins */
	for( nd=0; nd < gv.nBin; nd++ ) 
	{
		if( ! gv.bin[nd]->lgDustOn1 ) 
		{
			for( nd2 = nd+1; nd2 < gv.nBin; nd2++ ) 
			{
				if( gv.bin[nd2]->lgDustOn1 ) 
				{
					break;
				}
			}
			if( nd2 < gv.nBin ) 
			{
				GrainBin *swap;
				swap = gv.bin[nd];
				gv.bin[nd] = gv.bin[nd2];
				gv.bin[nd2] = swap;
			}
			else 
			{
				break;
			}
		}
	}
	for( nd2 = nd; nd2 < gv.nBin; nd2++ )
	{
		FREE_CHECK( gv.bin[nd2] );
	}
	gv.nBin = nd;
	gv.lgDustOn = ( gv.nBin > 0 );
#	endif

	/* >>chng 01 sep 12, allocate/free [rfield.nupper] arrays dynamically */
	assert( gv.dstab == NULL ); /* prevent memory leaks */
	if( ( gv.dstab = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double))) ) == NULL ) 
		bad_malloc();
	assert( gv.dstsc == NULL ); /* prevent memory leaks */
	if( ( gv.dstsc = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double))) ) == NULL ) 
		bad_malloc();
	assert( gv.GrainEmission == NULL ); /* prevent memory leaks */
	if( ( gv.GrainEmission = (float*)MALLOC((size_t)(rfield.nupper*sizeof(float))) ) == NULL ) 
		bad_malloc();
	assert( gv.GraphiteEmission == NULL ); /* prevent memory leaks */
	if( ( gv.GraphiteEmission = (float*)MALLOC((size_t)(rfield.nupper*sizeof(float))) ) == NULL ) 
		bad_malloc();
	assert( gv.SilicateEmission == NULL ); /* prevent memory leaks */
	if( ( gv.SilicateEmission = (float*)MALLOC((size_t)(rfield.nupper*sizeof(float))) ) == NULL ) 
		bad_malloc();
	
	/* sanity check */
	assert( gv.nBin >= 0 && gv.nBin < NDUST );
	for( nd=gv.nBin; nd < NDUST; nd++ ) 
	{
		assert( gv.bin[nd] == NULL );
	}

	/* >>chng 02 jan 15, initialize to zero in case grains are not used, needed in IonIron(), PvH */
	for( ipZ=0; ipZ < LIMELM; ipZ++ )
	{
		gv.elmSumAbund[ipZ] = 0.f;
	}

	for( i=0; i < rfield.nupper; i++ )
	{
		gv.dstab[i] = 0.;
		gv.dstsc[i] = 0.;
		/* >>chng 01 sep 12, moved next three initializations from GrainZero(), PvH */
		gv.GrainEmission[i] = 0.;
		gv.SilicateEmission[i] = 0.;
		gv.GraphiteEmission[i] = 0.;
	}

	if( !gv.lgDustOn )
	{
		/* grains are not on, set all heating/cooling agents to zero */
		gv.GrainHeatInc = 0.;
		gv.GrainHeatDif = 0.;
		gv.GrainHeatLya = 0.;
		gv.GrainHeatCollSum = 0.;
		gv.GrainHeatSum = 0.;
		gv.GasCoolColl = 0.;
		heat.heating[0][13] = 0.;
		heat.heating[0][14] = 0.;
		heat.heating[0][25] = 0.;

		if( trace.lgTrace && trace.lgDustBug )
		{
			fprintf( ioQQQ, " InitGrains exits.\n" );
		}

#		ifdef DEBUG_FUN
		fputs( " <->InitGrains()\n", debug_fp );
#		endif
		return;
	}

#	ifdef INCLUDE_OLD_GRAINS
	/* >>chng 01 aug 22, MALLOC this space */
	if( (InvDepCar = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double)) ) ) == NULL )
		bad_malloc();
	if( (InvDepSil = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double)) ) ) == NULL )
		bad_malloc();

	mie_read_ial("graphite.rfi",rfield.nupper,InvDepCar);
	mie_read_ial("silicate.rfi",rfield.nupper,InvDepSil);
#	endif

	for( nd=0; nd < gv.nBin; nd++ )
	{
		double help,atoms,p_rad,ThresInf,dum[4];
		long low1,low2,low3,lowm;

		/* sanity check */
		assert( gv.bin[nd] != NULL );

		/* this is QHEAT ALL command, will become default !! */
		if( gv.lgQHeatAll )
		{
			gv.bin[nd]->lgQHeat = TRUE;
		}

		/* this is NO GRAIN QHEAT command, always takes precedence */
		if( !gv.lgQHeatOn ) 
		{
			gv.bin[nd]->lgQHeat = FALSE;
		}

		/* this is total grain depletion factor, GrainMetal is depletion
		 * set with metals command, normally one */
		gv.bin[nd]->dstfactor *= gv.GrainMetal;

		/* >>chng 01 nov 21, grain abundance may depend on radius,
		 * invalidate for now; GrainUpdateRadius1() will set correct value */
		gv.bin[nd]->dstAbund = -FLT_MAX;

		gv.bin[nd]->qtmin_zone1 = -1.;

		if( gv.bin[nd]->DustWorkFcn < rfield.anu[0] || gv.bin[nd]->DustWorkFcn > rfield.anu[rfield.nupper] )
		{
			fprintf( ioQQQ, " Grain work function for %s has insane value: %.4e\n",
				 gv.bin[nd]->chDstLab,gv.bin[nd]->DustWorkFcn );
			puts( "[Stop in InitGrains]" );
			cdEXIT(1);
		}

		/* >>chng 01 sep 12, allocate/free [rfield.nupper] arrays dynamically */
		assert( gv.bin[nd]->y1 == NULL ); /* prevent memory leaks */
		if( (gv.bin[nd]->y1 = (float*)MALLOC((size_t)(rfield.nupper*sizeof(float)))) == NULL )
			bad_malloc();
		if( gv.bin[nd]->lgQHeat )
		{
			assert( gv.bin[nd]->phiTilde == NULL ); /* prevent memory leaks */
			if( (gv.bin[nd]->phiTilde = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double)))) == NULL )
				bad_malloc();
		}

#		ifdef INCLUDE_OLD_GRAINS
		if( gv.bin[nd]->lgOldGrain1 ) 
		{
			/* >>chng 01 sep 12, allocate/free [rfield.nupper] arrays dynamically */
			/* only for old style grains has this memory not been allocated yet,
			 * for new style grains it has been allocated in mie_read_opc(), PvH */
			if( (gv.bin[nd]->dstab1 = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double)))) == NULL )
				bad_malloc();
			if( (gv.bin[nd]->dstsc1 = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double)))) == NULL )
				bad_malloc();
			if( (gv.bin[nd]->inv_att_len = (float*)MALLOC((size_t)(rfield.nupper*sizeof(float)))) == NULL )
				bad_malloc();

			/* extrapolate absorption and scattering cross sections */
			OldStyleGrainCSInit(nd);

			/* initialize inverse attenuation length for photons */
			if( gv.bin[nd]->matType == MAT_CAR ) 
			{
				for( i=0; i < rfield.nupper; i++ ) 
				{
					gv.bin[nd]->inv_att_len[i] = (float)InvDepCar[i];
				}
			}
			else if( gv.bin[nd]->matType == MAT_SIL ) 
			{
				for( i=0; i < rfield.nupper; i++ ) 
				{
					gv.bin[nd]->inv_att_len[i] = (float)InvDepSil[i];
				}
			}
			else 
			{
				fprintf( ioQQQ, " InitGrains detected unknown material type: %ld\n" ,
					 gv.bin[nd]->matType );
				puts( "[Stop in InitGrains]" );
				cdEXIT(1);
			}
		}
#		endif

		for( i=0; i < rfield.nupper; i++ )
		{
			double alpha,beta,af,bf;

			beta = gv.bin[nd]->AvRadius*gv.bin[nd]->inv_att_len[i];
			if( beta > 1.e-4 ) 
			{
				bf = POW2(beta) - 2.*beta + 2. - 2.*exp(-beta);
			}
			else 
			{
				bf = POW3(beta)/3.;
			}
			alpha = beta + gv.bin[nd]->AvRadius/MEAN_PATH;
			if( alpha > 1.e-4 ) 
			{
				af = POW2(alpha) - 2.*alpha + 2. - 2.*exp(-alpha);
			}
			else 
			{
				af = POW3(alpha)/3.;
			}

			/* this is size-dependent geometrical yield enhancement
			 * defined in Weingartner & Draine, 2000 */
			gv.bin[nd]->y1[i] = (float)(POW2(beta/alpha)*af/bf);
		}

		assert( gv.bin[nd]->nChrg >= 2 && gv.bin[nd]->nChrg <= NCHS );

		/* >>chng 00 jun 19, this value is absolute lower limit for the grain
		 * potential, electrons cannot be bound for lower values..., PvH */
		if( gv.lgBakes )
		{
			/* this corresponds to >>refer Bakes & Tielens, 1994, ApJ, 427, 822 */
			help = ceil(POT2CHRG(gv.bin[nd]->BandGap-gv.bin[nd]->DustWorkFcn+ONE_ELEC/2.));
			low1 = NINT(help);
		}
		else
		{
			/* >>chng 01 jan 18, the following expressions are taken from Weingartner & Draine, 2001 */
			if( gv.bin[nd]->matType == MAT_CAR )
			{
				help = gv.bin[nd]->AvRadius*1.e7;
				help = ceil(-(1.2*POW2(help)+3.9*help+0.2)/1.44);
				low1 = NINT(help);
/*  			help = POT2CHRG(-0.2866-8.82e5*gv.bin[nd]->AvRadius-1.47e-9/gv.bin[nd]->AvRadius); */
/*  			help = ceil(help) + 1.; */
			}
			else if( gv.bin[nd]->matType == MAT_SIL ) 
			{
				help = gv.bin[nd]->AvRadius*1.e7;
				help = ceil(-(0.7*POW2(help)+2.5*help+0.8)/1.44);
				low1 = NINT(help);
/*  			help = POT2CHRG(-0.1837-5.15e5*gv.bin[nd]->AvRadius-5.88e-9/gv.bin[nd]->AvRadius); */
/*  			help = ceil(help) + 1.; */
			}
			else
			{
				/* revert to 
				 * >>refer Bakes & Tielens, 1994, ApJ, 427, 822 */
				help = ceil(POT2CHRG(gv.bin[nd]->BandGap-gv.bin[nd]->DustWorkFcn+ONE_ELEC/2.));
				low1 = NINT(help);
			}
		}

		/* >>chng 01 apr 20, iterate to get LowestPot such that the exponent in the thermionic
		 * rate never becomes positive; the value can be derived by equating ThresInf >= 0;
		 * the new expression for Emin (see GetPotValues) cannot be inverted analytically,
		 * hence it is necessary to iterate for LowestPot. this also automatically assures that
		 * the expressions for ThresInf and LowestPot are consistent with each other, PvH */
		low2 = low1;
		GetPotValues(nd,low2,&ThresInf,&dum[0],&dum[1],&dum[2],&dum[3],INCL_TUNNEL);
		if( ThresInf < 0. )
		{
			low3 = 0;
			/* do a bisection search for the lowest charge such that
			 * ThresInf >= 0, the end result will eventually be in low3 */
			while( low3-low2 > 1 )
			{
				lowm = (low2+low3)/2;
				GetPotValues(nd,lowm,&ThresInf,&dum[0],&dum[1],&dum[2],&dum[3],INCL_TUNNEL);
				if( ThresInf < 0. )
					low2 = lowm;
				else
					low3 = lowm;
			}
			low2 = low3;
		}

		/* the first term implements the minimum charge due to autoionization
		 * the second term assures that the exponent in the thermionic rate never
		 * becomes positive; the expression was derived by equating ThresInf >= 0 */
		gv.bin[nd]->LowestZg = MAX2(low1,low2);
		gv.bin[nd]->LowestPot = CHRG2POT(gv.bin[nd]->LowestZg);

		/* >>chng 00 jul 13, new sticking probability for electrons */
		/* the second term is chance that electron passes through grain,
		 * 1-p_rad is chance that electron is ejected before grain settles
		 * see discussion in 
		 * >>refer Weingartner & Draine, 2001, ApJS, 134, 263 */
		gv.bin[nd]->StickElecPos = STICK_ELEC*(1. - exp(-gv.bin[nd]->AvRadius/MEAN_PATH));
		atoms = gv.bin[nd]->AvVol*gv.bin[nd]->dustp[0]/ATOMIC_MASS_UNIT/gv.bin[nd]->atomWeight;
		p_rad = 1./(1.+exp(20.-atoms));
		gv.bin[nd]->StickElecNeg = gv.bin[nd]->StickElecPos*p_rad;

		/* >>chng 02 feb 15, these quantities depend on radius and are normally set
		 * in GrainUpdateRadius1(), however, it is necessary to initialize them here
		 * as well so that they are valid the first time hmole is called. */
		gv.bin[nd]->dstAbund = (float)(gv.bin[nd]->dstfactor*GrnVryDpth(nd));
		assert( gv.bin[nd]->dstAbund > 0.f );
		/* grain unit conversion, <unit>/H (default depl) -> <unit>/cm^3 (actual depl) */
		gv.bin[nd]->cnv_H_pCM3 = phycon.hden*gv.bin[nd]->dstAbund;
		gv.bin[nd]->cnv_CM3_pH = 1./gv.bin[nd]->cnv_H_pCM3;
		/* grain unit conversion, <unit>/cm^3 (actual depl) -> <unit>/grain */
		gv.bin[nd]->cnv_CM3_pGR = gv.bin[nd]->cnv_H_pGR/gv.bin[nd]->cnv_H_pCM3;
		gv.bin[nd]->cnv_GR_pCM3 = 1./gv.bin[nd]->cnv_CM3_pGR;
	}

	/* >>chng 01 nov 21, total grain opacities depend on charge and therefore on radius,
	 * invalidate for now; GrainUpdateRadius1() and GrainUpdateRadius2() will set correct values */
	for( i=0; i < rfield.nupper; i++ )
	{
		/* these are total absorption and scattering cross sections */
		gv.dstab[i] = -DBL_MAX;
		gv.dstsc[i] = -DBL_MAX;
	}

	/* invalidate the summed grain abundances for now, they
	 * depend on radius and are set in GrainUpdateRadius1() */
	for( ipZ=0; ipZ < LIMELM; ipZ++ )
	{
		gv.elmSumAbund[ipZ] = -FLT_MAX;
	}

	InitEmissivities();

	/* >>chng 01 may 09, this prevents use of global data before they are properly defined */
	ReturnGlobalData();

	if( trace.lgDustBug && trace.lgTrace )
	{
		fprintf( ioQQQ, "     There are %ld grain types turned on.\n", gv.nBin );

		fprintf( ioQQQ, "     grain depletion factors, dstfactor=" );
		for( nd=0; nd < gv.nBin; nd++ )
		{
			fprintf( ioQQQ, "%10.2e", gv.bin[nd]->dstfactor );
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "     nChrg =" );
		for( nd=0; nd < gv.nBin; nd++ )
		{
			fprintf( ioQQQ, " %ld", gv.bin[nd]->nChrg );
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "     lowest charge (e) =" );
		for( nd=0; nd < gv.nBin; nd++ )
		{
			fprintf( ioQQQ, "%10.2e", POT2CHRG(gv.bin[nd]->LowestPot) );
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "     lgDustVary flag for depth dependence:" );
		for( nd=0; nd < gv.nBin; nd++ )
		{
			fprintf( ioQQQ, "%2c", TorF(gv.bin[nd]->lgDustVary) );
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "     Quantum heating flag:" );
		for( nd=0; nd < gv.nBin; nd++ )
		{
			fprintf( ioQQQ, "%2c", TorF(gv.bin[nd]->lgQHeat) );
		}
		fprintf( ioQQQ, "\n" );

		/* >>chng 01 nov 21, removed total abs and sct cross sections, they are invalid */
		fprintf( ioQQQ, "     NU(Ryd), Abs cross sec per proton\n" );

		fprintf( ioQQQ, "    Ryd   " );
		for( nd=0; nd < gv.nBin; nd++ )
		{
			fprintf( ioQQQ, " %-12.12s", gv.bin[nd]->chDstLab );
		}
		fprintf( ioQQQ, "\n" );

		for( i=0; i < rfield.nupper; i += 40 )
		{
			fprintf( ioQQQ, "%10.2e", rfield.anu[i] );
			for( nd=0; nd < gv.nBin; nd++ )
			{
				fprintf( ioQQQ, " %10.2e  ", gv.bin[nd]->dstab1[i] );
			}
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, "     NU(Ryd), Sct cross sec per proton\n" );

		fprintf( ioQQQ, "    Ryd   " );
		for( nd=0; nd < gv.nBin; nd++ )
		{
			fprintf( ioQQQ, " %-12.12s", gv.bin[nd]->chDstLab );
		}
		fprintf( ioQQQ, "\n" );

		for( i=0; i < rfield.nupper; i += 40 )
		{
			fprintf( ioQQQ, "%10.2e", rfield.anu[i] );
			for( nd=0; nd < gv.nBin; nd++ )
			{
				fprintf( ioQQQ, " %10.2e  ", gv.bin[nd]->dstsc1[i] );
			}
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, "     NU(Ryd), Q abs\n" );

		fprintf( ioQQQ, "    Ryd   " );
		for( nd=0; nd < gv.nBin; nd++ )
		{
			fprintf( ioQQQ, " %-12.12s", gv.bin[nd]->chDstLab );
		}
		fprintf( ioQQQ, "\n" );

		for( i=0; i < rfield.nupper; i += 40 )
		{
			fprintf( ioQQQ, "%10.2e", rfield.anu[i] );
			for( nd=0; nd < gv.nBin; nd++ )
			{
				fprintf( ioQQQ, " %10.2e  ", gv.bin[nd]->dstab1[i]*4./gv.bin[nd]->IntArea );
			}
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, "     NU(Ryd), Q sct\n" );

		fprintf( ioQQQ, "    Ryd   " );
		for( nd=0; nd < gv.nBin; nd++ )
		{
			fprintf( ioQQQ, " %-12.12s", gv.bin[nd]->chDstLab );
		}
		fprintf( ioQQQ, "\n" );

		for( i=0; i < rfield.nupper; i += 40 )
		{
			fprintf( ioQQQ, "%10.2e", rfield.anu[i] );
			for( nd=0; nd < gv.nBin; nd++ )
			{
				fprintf( ioQQQ, " %10.2e  ", gv.bin[nd]->dstsc1[i]*4./gv.bin[nd]->IntArea );
			}
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, "     NU(Ryd), yield enhancement\n" );

		fprintf( ioQQQ, "    Ryd   " );
		for( nd=0; nd < gv.nBin; nd++ )
		{
			fprintf( ioQQQ, " %-12.12s", gv.bin[nd]->chDstLab );
		}
		fprintf( ioQQQ, "\n" );

		for( i=0; i < rfield.nupper; i += 40 )
		{
			fprintf( ioQQQ, "%10.2e", rfield.anu[i] );
			for( nd=0; nd < gv.nBin; nd++ )
			{
				fprintf( ioQQQ, " %10.2e  ", gv.bin[nd]->y1[i] );
			}
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, " InitGrains exits.\n" );
	}

#ifdef INCLUDE_OLD_GRAINS
	FREE_CHECK(InvDepCar);
	FREE_CHECK(InvDepSil);
#endif

#	ifdef DEBUG_FUN
	fputs( " <->InitGrains()\n", debug_fp );
#	endif
	return;
}


static void InitEmissivities()
{
	double fac,
	  fac2,
	  mul,
	  tdust;
	long int i,
	  nd;

#	ifdef DEBUG_FUN
	fputs( "<+>InitEmissivities()\n", debug_fp );
#	endif

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "  InitEmissivities starts\n" );
		fprintf( ioQQQ, "    ND    Tdust       Emis       BB Check   4pi*a^2*<Q>\n" );
	}

	assert( NTOP >= 2 && NDEMS > 2*NTOP );
	fac = exp(log(GRAIN_TMID/GRAIN_TMIN)/(double)(NDEMS-NTOP));
	tdust = GRAIN_TMIN;
	for( i=0; i < NDEMS-NTOP; i++ )
	{
		gv.dsttmp[i] = log(tdust);
		for( nd=0; nd < gv.nBin; nd++ )
		{
			gv.bin[nd]->dstems[i] = log(PlanckIntegral(tdust,nd,i));
		}
		tdust *= fac;
	}

	/* temperatures above GRAIN_TMID are unrealistic -> make grid gradually coarser */
	fac2 = exp(log(GRAIN_TMAX/GRAIN_TMID/powi(fac,NTOP-1))/(double)((NTOP-1)*NTOP/2));
	for( i=NDEMS-NTOP; i < NDEMS; i++ )
	{
		gv.dsttmp[i] = log(tdust);
		for( nd=0; nd < gv.nBin; nd++ )
		{
			gv.bin[nd]->dstems[i] = log(PlanckIntegral(tdust,nd,i));
		}
		fac *= fac2;
		tdust *= fac;
	}

	/* sanity checks */
	mul = 1.;
	assert( fabs(gv.dsttmp[0] - log(GRAIN_TMIN)) < 10.*mul*DBL_EPSILON );
	mul = sqrt((double)(NDEMS-NTOP));
	assert( fabs(gv.dsttmp[NDEMS-NTOP] - log(GRAIN_TMID)) < 10.*mul*DBL_EPSILON );
	mul = (double)NTOP + sqrt((double)NDEMS);
	assert( fabs(gv.dsttmp[NDEMS-1] - log(GRAIN_TMAX)) < 10.*mul*DBL_EPSILON );

	/* now find slopes form spline fit */
	for( nd=0; nd < gv.nBin; nd++ )
	{
		/* set up coefficients for spline */
		spline(gv.bin[nd]->dstems,gv.dsttmp,NDEMS,2e31,2e31,gv.bin[nd]->dstslp);
		spline(gv.dsttmp,gv.bin[nd]->dstems,NDEMS,2e31,2e31,gv.bin[nd]->dstslp2);
	}

#	if 0
	/* test the dstems interpolation */
	nd = NINT(fudge(0));
	assert( nd >= 0 && nd < gv.nBin );
	for( i=0; i < 2000; i++ )
	{
		double x,y,z;
		z = pow(10.,-40. + (double)i/50.);
		splint(gv.bin[nd]->dstems,gv.dsttmp,gv.bin[nd]->dstslp,NDEMS,log(z),&y);
		if( exp(y) > GRAIN_TMIN && exp(y) < GRAIN_TMAX )
		{
			x = PlanckIntegral(exp(y),nd,3);
			printf(" input %.6e temp %.6e output %.6e rel. diff. %.6e\n",z,exp(y),x,(x-z)/z);
		}
	}
	exit(0);
#	endif

#	ifdef DEBUG_FUN
	fputs( " <->InitEmissivities()\n", debug_fp );
#	endif
	return;
}


/* PlanckIntegral compute total radiative cooling due to grains */
static double PlanckIntegral(double tdust, 
			     long int nd, 
			     long int ip)
{
	long int i;

	double arg,
	  ExpM1,
	  integral1 = 0.,  /* integral(Planck) */
	  integral2 = 0.,  /* integral(Planck*abs_cs) */
	  Planck1,
	  Planck2,
	  TDustRyg, 
	  x;

#	ifdef DEBUG_FUN
	fputs( "<+>PlanckIntegral()\n", debug_fp );
#	endif

	/******************************************************************
	 *
	 * >>>chng 99 mar 12, this sub rewritten following Peter van Hoof
	 * comments.  Original coding was in single precision, and for
	 * very low temperature the exponential was set to zero.  As 
	 * a result Q was far too large for grain temperatures below 10K
	 *
	 ******************************************************************/

	/* Boltzmann factors for Planck integration */
	TDustRyg = TE1RYD/tdust;

	x = 0.999*log(DBL_MAX);

	for( i=0; i < rfield.nupper; i++ )
	{
		/* this is hnu/kT for grain at this temp and photon energy */
		arg = TDustRyg*rfield.anu[i];

		/* want the number exp(hnu/kT) - 1, two expansions */
		if( arg < 1.e-5 )
		{
			/* for small arg expand exp(hnu/kT) - 1 to second order */
			ExpM1 = arg*(1. + arg/2.);
		}
		else
		{
			/* for large arg, evaluate the full expansion */
			ExpM1 = exp(MIN2(x,arg)) - 1.;
		}

		Planck1 = PI4*2.*HPLANCK/POW2(SPEEDLIGHT)*POW2(FR1RYD)*POW2(FR1RYD)*
			rfield.anu3[i]/ExpM1*rfield.widflx[i];
		Planck2 = Planck1*gv.bin[nd]->dstab1[i];

		/* add integral over RJ tail, maybe useful for extreme low temps */
		if( i == 0 ) 
		{
			integral1 = Planck1/rfield.widflx[0]*rfield.anu[0]/3.;
			integral2 = Planck2/rfield.widflx[0]*rfield.anu[0]/5.;
		}
		/* if we are in the Wien tail - exit */
		if( Planck1/integral1 < DBL_EPSILON && Planck2/integral2 < DBL_EPSILON )
			break;

		integral1 += Planck1;
		integral2 += Planck2;
	}

	/* this is an option to print out every few steps, when 'trace grains' is set */
	if( trace.lgDustBug && trace.lgTrace && ip%10 == 0 )
	{
		fprintf( ioQQQ, "  %4ld %11.4e %11.4e %11.4e %11.4e\n", nd, tdust, 
		  integral2, integral1/4./5.67051e-5/powi(tdust,4), integral2*4./integral1 );
	}

	assert( integral2 > 0. );

#	ifdef DEBUG_FUN
	fputs( " <->PlanckIntegral()\n", debug_fp );
#	endif
	return( integral2 );
}


/* allocate backup store for global data */
static void AllocGlobalData(void)
{
	long i;

#	ifdef DEBUG_FUN
	fputs( "<+>AllocGlobalData()\n", debug_fp );
#	endif

	assert( cache == NULL ); /* prevent memory leak */
	if( ( cache = (BackupStore *)MALLOC(sizeof(BackupStore)) ) == NULL ) 
		bad_malloc();
	cache->lgGlobalValid = FALSE;
	cache->GlobLast = -1;
	cache->GlobRPtr = 0;
	cache->GlobEPtr = 0;
	for( i=0; i < NCHS; i++ )
	{
		if( ( cache->yhat[i] = (float*)MALLOC((size_t)(rfield.nupper*sizeof(float))) ) == NULL ) 
			bad_malloc();
		if( ( cache->cs_pdt[i] = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double))) ) == NULL ) 
			bad_malloc();
	}

#	ifdef DEBUG_FUN
	fputs( " <->AllocGlobalData()\n", debug_fp );
#	endif
	return;
}


/* return global data to avoid accidental use outside of scope of validity */
static void ReturnGlobalData(void)
{
	long i;

#	ifdef DEBUG_FUN
	fputs( "<+>ReturnGlobalData()\n", debug_fp );
#	endif

	if( cache != NULL )
	{
		for( i=0; i < NCHS; i++ )
		{
			FREE_CHECK( cache->yhat[i] );
			FREE_CHECK( cache->cs_pdt[i] );
		}
		FREE_CHECK( cache );
	}

#	ifdef DEBUG_FUN
	fputs( " <->ReturnGlobalData()\n", debug_fp );
#	endif
	return;
}


/* this is the main routine that drives the grain physics */
void grain(void)
{
#	ifdef DEBUG_FUN
	fputs( "<+>grain()\n", debug_fp );
#	endif

	if( gv.lgDustOn )
	{		
		/* find dust charge and temperature */
		GrainChargeTemp();

		/* find dust drift velocity */
		GrainDrift();
	}

#	ifdef DEBUG_FUN
	fputs( " <->grain()\n", debug_fp );
#	endif
	return;
}

/* iterate grain charge and temperature */
static void GrainChargeTemp(void)
{
	long int i,
	  ion,
	  ion_to,
	  ipZ,
	  nd,
	  nz;
	float dccool = FLT_MAX;
	double BoltzFac,
	  delta,
	  hcon = DBL_MAX,
	  HighEnergy,
	  hla = DBL_MAX,
	  hots = DBL_MAX,
	  oldtemp,
	  ratio,
	  thermionic = DBL_MAX,
	  ThermRatio;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainChargeTemp()\n", debug_fp );
#	endif

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "\n GrainChargeTemp called lgSearch%2c\n\n", TorF(conv.lgSearch) );
	}

	/* these will sum heating agents over grain populations */
	gv.GrainHeatInc = 0.;
	gv.GrainHeatDif = 0.;
	gv.GrainHeatLya = 0.;
	gv.GrainHeatCollSum = 0.;
	gv.GrainHeatSum = 0.;
	gv.GrainHeatChem = 0.;

	gv.GasCoolColl = 0.;
	/* must redo grain solution all the time since they
	 * generally have non-trivial contributions to heating-cooling */
	heat.heating[0][13] = 0.;
	heat.heating[0][25] = 0.;

	gv.TotalEden = 0.;

	for( ipZ=0; ipZ < LIMELM; ipZ++ )
	{
		for( ion=0; ion <= ipZ+1; ion++ )
		{
			for( ion_to=0; ion_to <= ipZ+1; ion_to++ )
			{
				gv.GrainRecom[ipZ][ion][ion_to] = 0.f;
			}
		}
	}

#	if 0
	/* >>chng 02 jan 06, moved this to doppvel since needed by other parts of the code */
	for( ipZ=0; ipZ < LIMELM; ipZ++ ) 
	{
		/* this is average (NOT rms) particle speed for Maxwell distribution, Mihalas 70, 9-70 */
		gv.AveVel[ipZ] = sqrt(8.*BOLTZMANN/PI/ATOMIC_MASS_UNIT*phycon.te/AtomcWgt.AtomicWeight[ipZ]);
	}
#	endif

	gv.HighestIon = HighestIonStage();

	/* determine highest energy to be considered by quantum heating routines.
	 * since the Boltzmann distribution is resolved, the upper limit has to be
	 * high enough that a negligible amount of energy is in the omitted tail */
	BoltzFac = -log(CONSERV_TOL) + 8.;
	HighEnergy = MIN2(BoltzFac*BOLTZMANN*phycon.te/EN1RYD,rfield.anu[rfield.nupper-1]);
	gv.qnflux = ipoint(HighEnergy);
	gv.qnflux = MAX2(rfield.nflux,gv.qnflux);

	/* update total grain opacities in gv.dstab and gv.dstsc,
	 * they depend on grain charge and may depend on depth
	 * this also sets dstAbund and conversion factors */
	GrainUpdateRadius1();

	for( nd=0; nd < gv.nBin; nd++ )
	{
		long relax = ( conv.lgSearch ) ? 3 : 1;
		double tol1 = TOLER;

		/* >>chng 01 sep 13, dynamically allocate backup store, remove ncell dependence, PvH */
		/* allocate data inside loop to avoid accidental spillover to next iteration */
		AllocGlobalData();

		if( trace.lgTrace && trace.lgDustBug )
		{
			fprintf( ioQQQ, " >> GrainChargeTemp starting grain %s\n",
				 gv.bin[nd]->chDstLab );
		}

		delta = 2.*TOLER;
		/* >>chng 01 nov 29, relax max no. of iterations during initial search */
		for( i=0; i < relax*CT_LOOP_MAX && delta > TOLER; ++i )
		{
			double ThresEst = 0.;
			oldtemp = gv.bin[nd]->tedust;

			/* solve for charge using previous estimate for grain temp
			 * grain temp only influences thermionic emissions
			 * Thermratio is fraction thermionic emissions contribute
			 * to the total electron loss rate of the grain */
			GrainCharge(nd,&ThermRatio,tol1);

			/* now solve grain temp using new value for grain potential */
			GrainTemperature(nd,&dccool,&hcon,&hots,&hla,&thermionic);

			/* delta estimates relative change in electron emission rate
			 * due to the update in the grain temperature, if it is small
			 * we won't bother to iterate (which is usually the case)
			 * the formula assumes that thermionic emission is the only
			 * process that depends on grain temperature */
			ratio = gv.bin[nd]->tedust/oldtemp;
			for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
			{
				ThresEst += gv.bin[nd]->FracPop[nz]*gv.bin[nd]->ThresInf[nz];
			}
			delta = ThresEst*TE1RYD/gv.bin[nd]->tedust*(ratio - 1.);
			delta = ( delta < 0.9*log(DBL_MAX) ) ?
				ThermRatio*fabs(POW2(ratio)*exp(delta)-1.) : DBL_MAX;
			/* >>chng 01 jan 19, increase accuracy of dstpot solution if convergence fails.
			 * when grain charge is close to LowestPot, thermionic rates become important
			 * even for low grain temperatures. under those conditions the rate is a very
			 * sensitive function of the grain potential. if tol1 is too large, ThermRatio
			 * will be inaccurate and delta will blow up, PvH */
			if( i > 0 )
				tol1 /= 10.;

			if( trace.lgTrace && trace.lgDustBug )
			{
				fprintf( ioQQQ, " >> GrainChargeTemp finds delta=%.4e, ",delta );
				fprintf( ioQQQ, " old/new temp=%.5e %.5e, ",oldtemp,gv.bin[nd]->tedust );
				if( delta > TOLER ) 
				{
					fprintf( ioQQQ, "doing another iteration\n" );
				}
				else 
				{
					fprintf( ioQQQ, "converged\n\n" );
				}
			}
		}
		if( delta > TOLER )
			fprintf( ioQQQ, " charge/temperature not converged for %s\n", gv.bin[nd]->chDstLab );

		for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
		{
			/* get Z0 for all stages, even those with zero abundance,
			 * so that recombination rate below will be correct */
			UpdateRecomZ0(nd,nz,ALL_STAGES);
		}

		/* add in ion recombination rates on this grain species */
		GrainRecomRates(nd);

		/* add in the photo-dissociation cs in gv/dstab */
		GrainUpdateRadius2(nd);

		/* following used to keep track of heating agents in printout
		 * no physics done with GrainHeatInc
		 * dust heating by incident continuum, and elec friction before ejection */
		gv.GrainHeatInc += hcon;
		/* remember total heating by diffuse fields, for printout (includes Lya) */
		gv.GrainHeatDif += hots;
		/* GrainHeatLya - total heating by LA in this zone, erg cm-3 s-1, only here
		 * for eventual printout, hots is total ots line heating */
		gv.GrainHeatLya += hla;

		/* this will be total collisional heating, for printing in lines */
		gv.GrainHeatCollSum += gv.bin[nd]->GrainHeatColl;

		/* GrainHeatSum is total heating of all grain types in this zone,
		 * will be carried by total cooling, only used in lines to print tot heat
		 * printed as entry "GraT    0 " */
		gv.GrainHeatSum += gv.bin[nd]->GrainHeat;

		/* net amount of chemical energy donated by recombining ions */
		gv.GrainHeatChem += gv.bin[nd]->ChemEn;

		/* dccool is gas cooling due to collisions with grains */
		gv.GasCoolColl += dccool;

		/* rate dust heats gas by photoelectric effect */
		heat.heating[0][13] += gv.bin[nd]->GasHeatPhotoEl;
		/* thermionic is gas heating due to thermionic emissions */
		heat.heating[0][25] += thermionic;

		/* this is grain charge in e/cm^3, positive number means grain supplied free electrons */
		/* >>chng 01 mar 24, changed DustZ+1 to DustZ, PvH */
		for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
		{
			gv.TotalEden +=
				gv.bin[nd]->FracPop[nz]*(double)gv.bin[nd]->DustZ[nz]*gv.bin[nd]->cnv_GR_pCM3;
				
		}

		if( trace.lgTrace && trace.lgDustBug )
		{
			fprintf(ioQQQ," GrainChargeTemp finished %s Pot %.5e Thermal %.5e GasCoolColl %.5e" , 
				gv.bin[nd]->chDstLab, gv.bin[nd]->dstpot, gv.bin[nd]->GrainHeat, dccool );
			fprintf(ioQQQ," GasPEHeat %.5e GasThermHeat %.5e ChemHeat %.5e\n\n" , 
				gv.bin[nd]->GasHeatPhotoEl, thermionic, gv.bin[nd]->ChemEn );
		}

		/* this is to safeguard against accidental use outside of the range of validity */
		ReturnGlobalData();
	}

#ifdef WD_TEST2
	printf("wd test: proton fraction %.5e Total DustZ %.6f heating/cooling rate %.5e %.5e\n",xIonFracs[ipHYDROGEN][2]/xIonFracs[ipHYDROGEN][0],gv.bin[0]->AveDustZ,heat.heating[0][13]/phycon.hden/fudge(0),gv.GasCoolColl/phycon.hden/fudge(0));
#endif

	/* if grains hotter than gas then collisions with gas act
	 * to heat the gas, add this in here
	 * a symmetric statement appears in COOLR, where cooling is added on */
	heat.heating[0][14] = MAX2(0.,-gv.GasCoolColl);

	/* options to force gas heating or cooling by grains to zero - for tests only ! */
	if( !gv.lgDHetOn ) 
	{
		heat.heating[0][13] = 0.;
		heat.heating[0][14] = 0.;
		heat.heating[0][25] = 0.;
	}
	if( !gv.lgDColOn ) 
	{
		gv.GasCoolColl = 0.;
	}

	if( trace.lgTrace )
	{
		if( trace.lgDustBug )
		{
			fprintf( ioQQQ, "     Grain contribution to electron density%10.2e\n", gv.TotalEden );
		}
		else
		{
			fprintf( ioQQQ, "     Grain potentials:" );
			for( nd=0; nd < gv.nBin; nd++ )
			{
				fprintf( ioQQQ, "%10.2e", gv.bin[nd]->dstpot );
			}
			fprintf( ioQQQ, "\n" );
			
			fprintf( ioQQQ, "     Grain temperatures:" );
			for( nd=0; nd < gv.nBin; nd++ )
			{
				fprintf( ioQQQ, "%10.2e", gv.bin[nd]->tedust );
			}
			fprintf( ioQQQ, "\n" );
		}
	}


#	ifdef DEBUG_FUN
	fputs( " <->GrainChargeTemp()\n", debug_fp );
#	endif
	return;
}


#define SORT2(X1,X2,F1,F2,T1,T2) \
	if( X1 > X2 )            \
	{                        \
		double TMP = X1; \
		X1 = X2;         \
		X2 = TMP;        \
		TMP = F1;        \
		F1 = F2;         \
		F2 = TMP;        \
		TMP = T1;        \
		T1 = T2;         \
		T2 = TMP;        \
	}


static void GrainCharge(long int nd,
			/*@out@*/double *ThermRatio, /* ratio of thermionic to total rate */
			double toler)
{
	int lgBigError;
	long int lopbig,
	  luplim,
	  nz;
	double dummy,
	  emission=0.,
	  fhi,
	  flo,
	  fmid,
	  help,
	  norm=0.,
	  old_tol,
	  recombination=0.,
	  renorm,
	  thhi,
	  thlo,
	  thmid,
	  tol=DBL_MAX,
	  xhi,
	  xlo,
	  xmid;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainCharge()\n", debug_fp );
#	endif


	/* find dust charge */
	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "  Starting Charge loop for %s, search%2c\n",
			 gv.bin[nd]->chDstLab, TorF(conv.lgSearch) );
	}

	gv.bin[nd]->lgChrgConverged = FALSE;

	if( conv.lgSearch )
	{
		/* this is limit to number of iterations for getting the grain charge */
		luplim = 100;
		/* set up initial estimates for grain potential, will be checked by BracketGrainPot */
		xlo = gv.bin[nd]->LowestPot;
		xmid = 0.;
		xhi = BIG_POTENTIAL;
	}
	else
	{
		/* >>chng 00 jul 19, convergence around zero pot can be tricky; luplim 10 -> 20, PvH */
		/* >>chng 01 jun 02, due to increased demands on precision, change luplim 20 -> 40, PvH */
		luplim = 40;
		xlo = MAX2(gv.bin[nd]->LowestPot,gv.bin[nd]->dstpot - BRACKET_STEP);
		xmid = gv.bin[nd]->dstpot;
		xhi = gv.bin[nd]->dstpot + BRACKET_STEP;
	}

	/* >>chng 00 jul 18, keep solution bracketed, this is needed since
	 * bandgap introduces discontinuities in emission and recombination rate;
	 * solution can keep oscillating around zero, PvH */
	/* >>chng 01 jan 08, validate bracket and alter if
	 * necessary, it sets first new step in gv.bin[nd]->dstpot 
	 * (usually the value from the previous zone) */
	BracketGrainPot(nd,&xlo,&flo,&thlo,&xmid,&fmid,&thmid,&xhi,&fhi,&thhi,&norm);

	/* if BracketGrainPot accidentally stumbled on the
	 * correct solution, it will be in xmid */

	if( trace.lgTrace && trace.lgDustBug )
	{
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG )
		{
			long i;
			double backup = gv.bin[nd]->dstpot;
			/* switch off trace output temporarily */
			trace.lgDustBug = FALSE;
			for( i=0; i<=100; i++ )
			{
				gv.bin[nd]->dstpot = xlo + (double)i/100.*(xhi-xlo);
				UpdatePot(nd);
				emission = GrainElecEmis(nd,&dummy);
				recombination = GrainElecRecomb(nd);
				printf(" GrainPotTest: %.6e %.6e %.6e\n",
				       POT2CHRG(gv.bin[nd]->dstpot),emission,recombination );
			}
			/* reset to old values */
			gv.bin[nd]->dstpot = backup;
			UpdatePot(nd);
			trace.lgDustBug = TRUE;
		}
	}

	lgBigError = TRUE;

	help = fabs(gv.bin[nd]->LowestPot);
	renorm = MAX2(help,1.);

	/* >>chng 01 jan 09, completely rewritten optimization loop. it now uses
	 * Brent's algorithm instead of Newton-Raphson which is a more stable
	 * choice if the derivative is not known explicitly, PvH */
	for( lopbig=0; lopbig < luplim; lopbig++ )
	{
		int lgSlow;
		double delta,
		  fnew,
		  minstep,
		  thnew,
		  tol1,
		  tol2,
		  xnew;

		char* method;

		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/

		old_tol = tol;
		tol = ( flo*fmid > 0. ) ? xhi-xmid : xlo-xmid;
		tol2 = ( norm > 0. ) ? fmid/norm : 0.;
		lgSlow = fabs(tol/old_tol) > 0.5;

		/* >>chng 01 jan 17, using this instead of tol1 = toler assures that the potential
		 * is very accurate near lowest potential where the emission rate is very steep, PvH */
		/* >>chng 01 may 31, change 10.*DBL_EPSILON -> 10.*DBL_EPSILON*xmid, PvH */
		tol1 = MAX2(toler*(xmid-gv.bin[nd]->LowestPot)/renorm,10.*DBL_EPSILON*xmid);

		if( DEBUG )
		{
			int lgTest = ( fabs(tol) < tol1 );
			printf("   tol %.15e tol1 %.15e -> %c\n",tol,tol1,TorF(lgTest));
			lgTest = ( fabs(tol2) < toler );
			printf("   tol2 %.15e toler %.15e -> %c\n",tol2,toler,TorF(lgTest));
			lgTest = ( fmid == 0. );
			printf("   fmid %.15e norm %.15e -> %c\n",fmid,norm,TorF(lgTest));
			lgTest = ( fmid == 0. || ( fabs(tol) < tol1 && fabs(tol2) < toler ));
			printf("   converged ? --> %c\n",TorF(lgTest));
		}

		/* >>chng 01 may 31, put in additional check to assure that emission equals recombination
		 * this is important for bandgap materials with average charges between -1 and 0; the
		 * emission rates can be strongly non-linear and lead to significant discrepancies, PvH */
		if( fmid == 0. || ( fabs(tol) < tol1 && fabs(tol2) < toler ) )
		{
			lgBigError = FALSE;
			break;
		}

		if( ! lgSlow )
		{
			double r = fmid/fhi;
			double s = fmid/flo;
			double t = flo/fhi;
			double p = s*((1.-r)*(xmid-xlo) - t*(r-t)*(xhi-xmid));
			double q = (1.-t)*(r-1.)*(s-1.);
			/* we are only interested in the ratio p/q, make sure q is positive */
			if( q < 0. )
			{
				p = -p;
				q = -q;
			}
			/* inverse quadratic interpolation is within bounds */
			if( p > q*(xlo-xmid) && p < q*(xhi-xmid) )
			{
				delta = p/q;
				method = "InvQuadr";
			}
			/* inverse quadratic interpolation is out of bounds, use bisection */
			else
			{
				delta = 0.5*tol;
				method = "OBBisect";
			}
		}
		/* progress for inverse quadratic interpolation is too slow, use bisection */
		else
		{
			delta = 0.5*tol;
			method = "SLBisect";
		}

		/* >>chng 00 jul 14, make sure change is not infinitesimally small, PvH */
		/* >>chng 01 may 31, prevent stepsize from becoming too small */
		help = fabs(xmid);
		minstep = 5.*DBL_EPSILON*MAX2(help,1.);
		delta = ( delta < 0. ) ? MIN2(delta,-minstep) : MAX2(delta,minstep);

		/* >>chng 01 jan 08, removed upper limit on change, code above handles this, PvH */
		xnew = MAX2(xmid + delta,gv.bin[nd]->LowestPot);

		if( DEBUG )
		{
			printf(" xlo %.15e xmid %.15e xhi %.15e delta %.15e xnew %.15e\n",xlo,xmid,xhi,delta,xnew);
			printf(" flo %.15e fmid %.15e fhi %.15e\n",flo,fmid,fhi);
		}

		if( xnew < xlo || xnew > xhi )
		{
			fprintf( ioQQQ, " GrainCharge failed to keep solution bracketed.\n" );
			lgBigError = TRUE;
			break;
		}

		if( trace.lgTrace && trace.lgDustBug )
		{
			fprintf( ioQQQ, "  Charge loop %ld %s Vmid %.4f (%.1e) Vnext %.4f Vlo %.4f (%.1e)", 
			  lopbig, gv.bin[nd]->chDstLab, xmid, fmid, xnew, xlo, flo );
			fprintf( ioQQQ, " Vhi %.4f (%.1e) %s tol %.4f Em/Rc %.3e %.3e\n", 
			  xhi, fhi, method, fabs(tol), emission, recombination );
		}

		gv.bin[nd]->dstpot = xnew;
		UpdatePot(nd);

		/* function GrainElecEmis returns electron loss rate for the grain species
		 * due to photoelectric effect, positive ion recombinations on the
		 * grain surface and thermionic emissions */
		emission = GrainElecEmis(nd,&thnew);

		/* the function GrainElecRecomb returns the recombination rate
		 * for this grain and charge */
		recombination = GrainElecRecomb(nd);

		/* this is the difference between emission and recombination
		 * rates - we want this to be small */
		fnew = emission - recombination;
		norm = 0.5*(emission+recombination);

		/* update bracket for the correct solution */
		if( fnew == 0. )
		{
			/* this sometimes happens, really ! */
			xlo = xmid = xhi = xnew;
			flo = fmid = fhi = fnew;
			thlo = thmid = thhi = thnew;
		}
		else if( fnew*fmid > 0. )
		{
			/* there are three points on one side of zero point -> remove farthest value */
			if( flo*fnew > 0. )
			{
				xlo = xnew;
				flo = fnew;
				thlo = thnew;
				SORT2(xlo,xmid,flo,fmid,thlo,thmid);
			}
			else
			{
				xhi = xnew;
				fhi = fnew;
				thhi = thnew;
				SORT2(xmid,xhi,fmid,fhi,thmid,thhi);
			}
		}
		else
		{
			/* there are two points on either side of zero point -> minimize bracket size */
			if( MIN2(xnew,xmid)-xlo > xhi-MAX2(xnew,xmid) )
			{
				xlo = xnew;
				flo = fnew;
				thlo = thnew;
				SORT2(xlo,xmid,flo,fmid,thlo,thmid);
			}
			else
			{
				xhi = xnew;
				fhi = fnew;
				thhi = thnew;
				SORT2(xmid,xhi,fmid,fhi,thmid,thhi);
			}
		}

		/* sanity check */
		/* >>chng 01 jul 26, added "fmid == 0." clause to make it run with gcc 2.96 on Linux-PC's */
		assert( fmid == 0. || ( xlo <= xmid && xmid <= xhi && flo*fhi <= 0. ) );
	}

	*ThermRatio = thmid;
	gv.bin[nd]->dstpot = xmid;
	UpdatePot(nd);

	gv.bin[nd]->lgChrgConverged = !lgBigError;

	if( lgBigError && !conv.lgSearch )
	{
		fprintf( ioQQQ," GrainCharge did not converge ionization of grain species %s\n",
			 gv.bin[nd]->chDstLab );
		++conv.nGrainFail;
	}

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, " >Grain potential:%12.12s %.6fV, charge %.5e (e)", 
		  gv.bin[nd]->chDstLab, gv.bin[nd]->dstpot*EVRYD, POT2CHRG(gv.bin[nd]->dstpot) );
		fprintf( ioQQQ, " emis: %.4e recom: %.4e\n", emission, recombination );
		for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
		{
			fprintf( ioQQQ, "    Thres[%ld]: %.4f V ThresVal[%ld]: %.4f V", 
				 gv.bin[nd]->DustZ[nz], gv.bin[nd]->ThresInf[nz]*EVRYD,
				 gv.bin[nd]->DustZ[nz], gv.bin[nd]->ThresInfVal[nz]*EVRYD );
		}
		fprintf( ioQQQ, "\n" );
	}


#	ifdef DEBUG_FUN
	fputs( " <->GrainCharge()\n", debug_fp );
#	endif
	return;
}

/* bracket the solution for the grain potential */
static void BracketGrainPot(long nd,
			    /*@in@*/double* xlo,
			    /*@out@*/double* flo,
			    /*@out@*/double* thlo,
			    /*@in@*/double* xmid,
			    /*@out@*/double* fmid,
			    /*@out@*/double* thmid,
			    /*@in@*/double* xhi,
			    /*@out@*/double* fhi,
			    /*@out@*/double* thhi,
			    /*@out@*/double* norm)
{
	long loop;
	double emission,
	  recombination,
	  stephi,
	  steplo;

#	ifdef DEBUG_FUN
	fputs( "<+>BracketGrainPot()\n", debug_fp );
#	endif

	steplo = *xmid - *xlo;
	stephi = *xhi - *xmid;

	*flo = -1.;
	*thlo = 0.;
	*fhi = 1.;
	*thhi = 0.;
	*norm = 0.;

	for( loop = 0; *flo < 0. && loop < BRACKET_MAX; ++loop )
	{
		gv.bin[nd]->dstpot = *xlo;
		UpdatePot(nd);
		emission = GrainElecEmis(nd,thlo);
		recombination = GrainElecRecomb(nd);
		*flo = emission - recombination;
		*norm = 0.5*(emission+recombination);
		if( *flo < 0. )
		{
			*xhi = *xlo;
			*fhi = *flo;
			*thhi = *thlo;
			*xlo = MAX2(*xlo-steplo,gv.bin[nd]->LowestPot);
			steplo *= 2.;
			/* value from previous zone is useless now */
			*xmid = (*xlo + *xhi)/2.;
		}
	}
	if( *flo == 0. )
	{
		*xmid = *xlo;
		*fmid = *flo;
		*thmid = *thlo;

#		ifdef DEBUG_FUN
		fputs( " <->BracketGrainPot()\n", debug_fp );
#		endif
		return;
	}
	else if( *flo < 0. )
	{
		if( *xlo == gv.bin[nd]->LowestPot )
		{
			fprintf( ioQQQ, " insanity: grain potential below lowest allowed value for %s\n" ,
				 gv.bin[nd]->chDstLab );
			ShowMe();
		}
		else
		{
			fprintf( ioQQQ, " could not bracket grain potential for %s\n" , gv.bin[nd]->chDstLab );
		}
		puts( "[Stop in BracketGrainPot]" );
		cdEXIT(1);
	}

	for( loop = 0; *fhi > 0. && loop < BRACKET_MAX; ++loop )
	{
		gv.bin[nd]->dstpot = *xhi;
		UpdatePot(nd);
		emission = GrainElecEmis(nd,thhi);
		recombination = GrainElecRecomb(nd);
		*fhi = emission - recombination;
		*norm = 0.5*(emission+recombination);
		if( *fhi > 0. )
		{
			*xlo = *xhi;
			*flo = *fhi;
			*thlo = *thhi;
			*xhi += stephi;
			stephi *= 2.;
			/* value from previous zone is useless now */
			*xmid = (*xlo + *xhi)/2.;
		}
	}
	if( *fhi == 0. )
	{
		*xmid = *xhi;
		*fmid = *fhi;
		*thmid = *thhi;

#		ifdef DEBUG_FUN
		fputs( " <->BracketGrainPot()\n", debug_fp );
#		endif
		return;
	}
	else if( *fhi > 0. )
	{
		fprintf( ioQQQ, " could not bracket grain potential for %s\n" , gv.bin[nd]->chDstLab );
		puts( "[Stop in BracketGrainPot]" );
		cdEXIT(1);
	}

	/* set initial estimate for grain potential */
	gv.bin[nd]->dstpot = *xmid;
	UpdatePot(nd);
	emission = GrainElecEmis(nd,thmid);
	recombination = GrainElecRecomb(nd);
	*fmid = emission - recombination;
	*norm = 0.5*(emission+recombination);

	/* sanity checks */
	assert( (*flo)*(*fhi) <= 0. );
	assert( gv.bin[nd]->LowestPot <= *xlo && *xlo < *xhi );
	assert( *xlo <= gv.bin[nd]->dstpot && gv.bin[nd]->dstpot <= *xhi );

#	ifdef DEBUG_FUN
	fputs( " <->BracketGrainPot()\n", debug_fp );
#	endif
	return;
}


static double GrainElecRecomb(long int nd)
{
	long nz;
	double crate,
	  sum1[NCHS],
	  csum1,
	  sum2[NCHS],
	  csum2;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainElecRecomb()\n", debug_fp );
#	endif

	/* >>chng 01 may 07, this routine now completely supports the hybrid grain
	 * charge model, i.e. all relevant quantities are calculated for Zlo and Zhi,
	 * and the average charge state is not used anywhere anymore, PvH */

	crate = 0.;
	csum1 = 0.;
	csum2 = 0.;

	for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
	{
		crate += gv.bin[nd]->FracPop[nz]*GrainElecRecomb1(nd,nz,&sum1[nz],&sum2[nz]);
		csum1 += gv.bin[nd]->FracPop[nz]*sum1[nz];
		csum2 += gv.bin[nd]->FracPop[nz]*sum2[nz];
	}

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "    GrainElecRecomb finds rate1=%.4e, rate2=%.4e.\n",csum1,csum2 );
		if( crate > 0. ) 
		{
			fprintf( ioQQQ, "    rel. fractions rate1/sum=%.4e, rate2/sum=%.4e.\n",
				 csum1/crate,csum2/crate );
		}
	}


#	ifdef DEBUG_FUN
	fputs( " <->GrainElecRecomb()\n", debug_fp );
#	endif
	return( crate );
}


/* grain electron recombination rates for single charge state */
static double GrainElecRecomb1(long nd,
			       long nz,
			       /*@out@*/ double *sum1,
			       /*@out@*/ double *sum2)
{
	long ion,
	  ipZ;
	double eta,
	  rate,
	  Stick,
	  ve,
	  xi;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainElecRecomb1()\n", debug_fp );
#	endif

	/* >>chng 01 may 31, try to find cached results first */
	if( cache->GlobRPtr > 0 )
	{
		long i;
		long iStart = MAX2(cache->GlobLast,0);
		long nCache = MIN2(cache->GlobRPtr,NGCS);
		cache->GlobLast = -1;
		for( i=iStart; i < iStart+nCache; i++ )
		{
			if( cache->GlobRZg[i%nCache] == gv.bin[nd]->DustZ[nz] )
			{
				cache->GlobLast = i%nCache;
				break;
			}
		}
		if( cache->GlobLast >= 0 )
		{
			*sum1 = cache->RSum1[cache->GlobLast];
			*sum2 = cache->RSum2[cache->GlobLast];
			rate = *sum1 + *sum2;

#			ifdef DEBUG_FUN
			fputs( " <->GrainElecRecomb1()\n", debug_fp );
#			endif
			return( rate );
		}
	}

	/* -1 makes psi correct for impact by electrons */
	ion = -1;
	/* VE is mean (not RMS) electron velocity */
	/*ve = TePowers.sqrte*6.2124e5;*/
	ve = sqrt(8.*BOLTZMANN/PI/ELECTRON_MASS*phycon.te);

	Stick = ( gv.bin[nd]->DustZ[nz] <= -1 ) ? gv.bin[nd]->StickElecNeg : gv.bin[nd]->StickElecPos;
	/* >>chng 00 jul 19, replace classical results with results including image potential
	 * to correct for polarization of the grain as charged particle approaches. */
	GrainScreen(ion,gv.bin[nd]->DustZ[nz],gv.bin[nd]->Capacity,&eta,&xi);
	/* this is grain surface recomb rate for electrons */
	*sum1 = EPSP(gv.bin[nd]->LowestZg,gv.bin[nd]->DustZ[nz])*Stick*phycon.eden*ve*eta;

	/* >>chng 00 jul 13, add in gain rate from atoms and ions, PvH */
	*sum2 = 0.;
	for( ion=0; ion <= LIMELM; ion++ )
	{
		double CollisionRateAll = 0.;

		for( ipZ=MAX2(ion-1,0); ipZ < LIMELM; ipZ++ )
		{
			if( abundances.lgElmtOn[ipZ] && xIonFracs[ipZ][ion+1] > 0. &&
			    cache->RecomZ0[nz][ipZ][ion] > ion )
			{
				/* this is rate with which charged ion strikes grain */
				CollisionRateAll += STICK_ION*xIonFracs[ipZ][ion+1]*DoppVel.AveVel[ipZ]*
					(double)(cache->RecomZ0[nz][ipZ][ion]-ion);
			}
		}

		if( CollisionRateAll > 0. )
		{
			/* >>chng 00 jul 19, replace classical results with results
			 * including image potential to correct for polarization 
			 * of the grain as charged particle approaches. */
			GrainScreen(ion,gv.bin[nd]->DustZ[nz],gv.bin[nd]->Capacity,&eta,&xi);
			*sum2 += CollisionRateAll*eta;
		}
	}

	rate = *sum1 + *sum2;

	/* >>chng 01 may 31, store results so that they may be used agian */
	cache->GlobRZg[cache->GlobRPtr%NGCS] = gv.bin[nd]->DustZ[nz];
	cache->RSum1[cache->GlobRPtr%NGCS] = *sum1;
	cache->RSum2[cache->GlobRPtr%NGCS] = *sum2;
	cache->GlobRPtr++;

#	ifdef DEBUG_FUN
	fputs( " <->GrainElecRecomb1()\n", debug_fp );
#	endif
	return( rate );
}


/* GrainElecEmis computes electron loss rate for this grain species due
 * to photoelectric effect, positive ion recombinations on the
 * grain surface and thermionic emissions */
static double GrainElecEmis(long int nd,
			    double *ThermRatio) /* ratio of thermionic to total emissions */
{
	long int nz;
	double crate,
	  sum1a[NCHS],
	  csum1a,
	  sum1b[NCHS],
	  csum1b,
	  sum1c[NCHS],
	  csum1c,
	  sum2[NCHS],
	  csum2,
	  sum3[NCHS],
	  csum3;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainElecEmis()\n", debug_fp );
#	endif

	/* >>chng 01 may 07, this routine now completely supports the hybrid grain
	 * charge model, i.e. all relevant quantities are calculated for Zlo and Zhi,
	 * and the average charge state is not used anywhere anymore, PvH */

	/* this is the loss rate due to photo-electric effect */
	crate = 0.;
	csum1a = 0.;
	csum1b = 0.;
	csum1c = 0.;
	csum2 = 0.;
	csum3 = 0.;
	for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
	{
		crate += gv.bin[nd]->FracPop[nz]*
			GrainElecEmis1(nd,nz,&sum1a[nz],&sum1b[nz],&sum1c[nz],&sum2[nz],&sum3[nz]);
		csum1a += gv.bin[nd]->FracPop[nz]*sum1a[nz];
		csum1b += gv.bin[nd]->FracPop[nz]*sum1b[nz];
		csum1c += gv.bin[nd]->FracPop[nz]*sum1c[nz];
		csum2 += gv.bin[nd]->FracPop[nz]*sum2[nz];
		csum3 += gv.bin[nd]->FracPop[nz]*sum3[nz];
	}

	*ThermRatio = ( crate > 0. ) ? csum3/crate : 0.;
		
	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "    GrainElecEmis finds rate1a=%.4e, rate1b=%.4e, ", csum1a, csum1b );
		fprintf( ioQQQ, "rate1c=%.4e, rate2=%.4e, rate3=%.4e.\n", csum1c, csum2, csum3 );
		if( crate > 0. ) 
		{
			fprintf( ioQQQ, "    rate1a/sum=%.4e, rate1b/sum=%.4e, rate1c/sum=%.4e, ",
				 csum1a/crate, csum1b/crate, csum1c/crate );
			fprintf( ioQQQ, "rate2/sum=%.4e, rate3/sum=%.4e.\n",
				 csum2/crate, csum3/crate );
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->GrainElecEmis()\n", debug_fp );
#	endif
	return( crate );
}


/* grain electron emission rates for single charge state */
static double GrainElecEmis1(long nd,
			     long nz,
			     /*@out@*/ double *sum1a,
			     /*@out@*/ double *sum1b,
			     /*@out@*/ double *sum1c,
			     /*@out@*/ double *sum2,
			     /*@out@*/ double *sum3)
{
	long int i,
	  ion,
	  ipZ;
	double eta,
	  rate,
	  xi;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainElecEmis1()\n", debug_fp );
#	endif

	/* >>chng 01 may 31, try to find cached results first */
	if( cache->GlobEPtr > 0 )
	{
		long iStart = MAX2(cache->GlobLast,0);
		long nCache = MIN2(cache->GlobEPtr,NGCS);
		cache->GlobLast = -1;
		for( i=iStart; i < iStart+nCache; i++ )
		{
			if( cache->GlobEZg[i%nCache] == gv.bin[nd]->DustZ[nz] )
			{
				cache->GlobLast = i%nCache;
				break;
			}
		}
		if( cache->GlobLast >= 0 )
		{
			*sum1a = cache->ESum1a[cache->GlobLast];
			*sum1b = cache->ESum1b[cache->GlobLast];
			*sum1c = cache->ESum1c[cache->GlobLast];
			*sum2 = cache->ESum2[cache->GlobLast];
			/* don't cache thermionic rates as they depend on grain temp */
			*sum3 = 4.*gv.bin[nd]->ThermRate[nz];
			rate = *sum1a + *sum1b + *sum1c + *sum2 + *sum3;

#			ifdef DEBUG_FUN
			fputs( " <->GrainElecEmis1()\n", debug_fp );
#			endif
			return( rate );
		}
	}

	/* this is the loss rate due to photo-electric effect */
	*sum1a = 0.;
	for( i=gv.bin[nd]->ipThresInfVal[nz]; i < rfield.nflux; i++ )
	{
#ifdef WD_TEST2
		*sum1a += rfield.flux[i]*gv.bin[nd]->dstab1[i]*cache->yhat[nz][i];
#else
		*sum1a += rfield.SummedCon[i]*gv.bin[nd]->dstab1[i]*cache->yhat[nz][i];
#endif
	}
	/* normalize to rates per cm^2 of projected grain area */
	*sum1a /= gv.bin[nd]->IntArea/4.;

	/* >>chng 01 dec 18, added code for modeling secondary electron emissions, PvH */
	/* this code does a crude correction for the Auger effect in grains,
	 * it is roughly valid for neutral and negative grains, but overestimates
	 * the effect for positively charged grains. Many of the Auger electrons have
	 * rather low energies and will not make it out of the potential well for
	 * high grain potentials typical of AGN conditions, see Table 4.1 & 4.2 of
	 * >>refer: Dwek E. & Smith R.K., 1996, ApJ, 459, 686 */

	fixit(); /* note that the number of primary electrons is given by yhat,
		  * which may not be one, so this is not necessarily consistent */
	fixit(); /* avAuger depends on grain charge, this should be treated explicitly here */

	*sum1b = 0.; /* this will be the secondary electron rate, in e/s/H at default depl */
	ion = 0; /* do Auger effect only for neutral atoms */

	/* this can be turned off with the NO AUGER command, since that command sets
	 * the array yield.nyield to zero */

	/* no Auger electrons for first three elements, but there are compton
	 * recoil ionizations for those elements, so start with Hydrogen */
	for( ipZ=ipHYDROGEN; ipZ < LIMELM; ipZ++ )
	{
		long ns, nelec, nej;

		if( gv.bin[nd]->elmAbund[ipZ] > 0. )
		{
			double eject_rate;
			/* loop over all shells, except the valence shell which doesn't produce Auger electrons
			 * and is properly part of the low energy grain opacity */
			for( ns=0; ns < Heavy.nsShells[ion][ipZ]-1; ns++ )
			{
				/* this will become the average number of Auger electrons freed */
				double avAuger = 0.;
				/* following is most number of electrons freed */
				nelec = yield.nyield[ns][ion][ipZ];

				/* following loop is over number of electrons that come out of shell
				 * with multiple electron ejection - use (nej-1) so that primary
				 * photoelectron is not counted, which is already done above. */
				for( nej=2; nej <= nelec; nej++ )
				{
					avAuger += yield.vyield[nej-1][ns][ion][ipZ]*(double)(nej-1);
				}
				/* add on Compton recoil of K electrons, with yield */
				if( ns==0 )
				{
					/* first term is Auger shakeoff electrons */
					eject_rate = PhotRate.PhotoRate[0][ns][ion][ipZ]*avAuger +
						/* this is compton recoil ionization term, which adds
						 * primary plus Auger electrons */
						/* >>chng 02 jan 11, add compton recoil ionization */
						/* add ionization due to compton recoil - 
						 * this is turned off with the NO RECOIL command */
						recoil.CompRecoilIonRate[ipZ]*(avAuger+1.);
				}
				else
				{
					/* not K shell, only include explicit shell */
					eject_rate = PhotRate.PhotoRate[0][ns][ion][ipZ]*avAuger;
				}
				/* PhotRate.PhotoRate is the photo-ionization rate per atom per second,
				 * it is updated after grain() is called but before RT_OTS_Update() */
				*sum1b  += eject_rate * gv.bin[nd]->elmAbund[ipZ];
			}
		}
	}

#ifdef WD_TEST2
	*sum1b = 0.;
#else
	*sum1b /= gv.bin[nd]->IntArea/4.;
#endif

	*sum1c = 0.;
	for( i=gv.bin[nd]->ipThresInf[nz]; i < rfield.nflux; i++ )
	{
		/* >>chng 00 jul 17, use description of Weingartner & Draine, 2001 */
#ifdef WD_TEST2
		*sum1c += rfield.flux[i]*cache->cs_pdt[nz][i];
#else
		*sum1c += rfield.SummedCon[i]*cache->cs_pdt[nz][i];
#endif
	}
	*sum1c /= gv.bin[nd]->IntArea/4.;

	/* >>chng 00 jun 19, add in loss rate due to recombinations with ions, PvH */
	*sum2 = 0.;
	for( ion=0; ion <= LIMELM; ion++ )
	{
		double CollisionRateAll = 0.;

		for( ipZ=MAX2(ion-1,0); ipZ < LIMELM; ipZ++ )
		{
			if( abundances.lgElmtOn[ipZ] && xIonFracs[ipZ][ion+1] > 0. &&
			    ion > cache->RecomZ0[nz][ipZ][ion] )
			{
				/* this is rate with which charged ion strikes grain */
				CollisionRateAll += STICK_ION*xIonFracs[ipZ][ion+1]*DoppVel.AveVel[ipZ]*
					(double)(ion-cache->RecomZ0[nz][ipZ][ion]);
			}
		}

		if( CollisionRateAll > 0. )
		{
			/* >>chng 00 jul 19, replace classical results with results
			 * including image potential to correct for polarization 
			 * of the grain as charged particle approaches. */
			GrainScreen(ion,gv.bin[nd]->DustZ[nz],gv.bin[nd]->Capacity,&eta,&xi);
			*sum2 += CollisionRateAll*eta;
		}
	}

	/* >>chng 01 may 30, moved calculation of ThermRate to UpdatePot */
	/* >>chng 01 jan 19, multiply by 4 since thermionic emissions scale with total grain
	 * surface area while the above two processes scale with projected grain surface area, PvH */
	*sum3 = 4.*gv.bin[nd]->ThermRate[nz];

	rate = *sum1a + *sum1b + *sum1c + *sum2 + *sum3;

	/* >>chng 01 may 31, store results so that they may be used agian */
	cache->GlobEZg[cache->GlobEPtr%NGCS] = gv.bin[nd]->DustZ[nz];
	cache->ESum1a[cache->GlobEPtr%NGCS] = *sum1a;
	cache->ESum1b[cache->GlobEPtr%NGCS] = *sum1b;
	cache->ESum1c[cache->GlobEPtr%NGCS] = *sum1c;
	cache->ESum2[cache->GlobEPtr%NGCS] = *sum2;
	cache->GlobEPtr++;

#	ifdef DEBUG_FUN
	fputs( " <->GrainElecEmis1()\n", debug_fp );
#	endif
	return( rate );
}


/* correction factors for grain charge screening (including image potential
 * to correct for polarization of the grain as charged particle approaches). */
static void GrainScreen(long ion,
			long DustZ,
			double Capacity,
			/*@out@*/ double *eta,
			/*@out@*/ double *xi)
{
#	ifdef DEBUG_FUN
	fputs( "<+>GrainScreen()\n", debug_fp );
#	endif

	/* >>refer Draine & Sutin, 1987, ApJ, 320, 803
	 * eta = J-tilde (eq. 3.3 thru 3.5), xi = Lambda-tilde/2. (eq. 3.8 thru 3.10) */
	if( ion == 0 ) 
	{
		*eta = 1.;
		*xi = 1.;
	}
	else 
	{
		/* >>chng 01 jan 03, assume that grain charge is distributed in two states just below and
		 * above the average charge, instead of the delta function we assume elsewhere. by averaging
		 * over the distribution we smooth out the discontinuities of the the Draine & Sutin expressions
		 * around nu == 0. they were the cause of temperature instabilities in globule.in. PvH */
		/* >>chng 01 may 07, revert back to single charge state since only integral charge states are
		 * fed into this routine now, making the two-charge state approximation obsolete, PvH */
		double nu = (double)DustZ/(double)ion;
		double tau = Capacity*BOLTZMANN*phycon.te*1.e-7/POW2((double)ion*ELEM_CHARGE);
		if( nu < 0. )
		{
			*eta = (1. - nu/tau)*(1. + sqrt(2./(tau - 2.*nu)));
			*xi = (1. - nu/(2.*tau))*(1. + 1./sqrt(tau - nu));
		}
		else if( nu == 0. ) 
		{
			*eta = 1. + sqrt(PI/(2.*tau));
			*xi = 1. + 0.75*sqrt(PI/(2.*tau));
		}
		else 
		{
			double theta_nu = ThetaNu(nu);
			/* >>>chng 00 jul 27, avoid passing functions to macro so set to local temp var */
			double xxx = 1. + 1./sqrt(4.*tau+3.*nu);
			*eta = POW2(xxx)*exp(-theta_nu/tau);
			/* >>chng 01 jan 24, use new expression for xi which only contains the excess
			 * energy above the potential barrier of the incoming particle (accurate to
			 * 2% or better), and then add in potential barrier separately, PvH */
			xxx = 0.25*pow(nu/tau,0.75)/(pow(nu/tau,0.75) + pow((25.+3.*nu)/5.,0.75)) +
				(1. + 0.75*sqrt(PI/(2.*tau)))/(1. + sqrt(PI/(2.*tau)));
			*xi = (MIN2(xxx,1.) + theta_nu/(2.*tau))*(*eta);
#			ifdef WD_TEST2
			*xi = (1. + nu/(2.*tau))*(1. + 1./sqrt(3./(2.*tau)+3.*nu))*exp(-theta_nu/tau);
#			else
/*  			*xi = (1. + nu/(2.*tau))*(1. + 1./sqrt(3./(2.*tau)+3.*nu))*exp(-theta_nu/tau); */
#			endif
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->GrainScreen()\n", debug_fp );
#	endif
	return;
}


static double ThetaNu(double nu)
{
	const double REL_TOLER = 1.e-5;
	double theta_nu,
	  xi_nu,
	  xxx;

#	ifdef DEBUG_FUN
	fputs( "<+>ThetaNu()\n", debug_fp );
#	endif

	if( nu > 0. )
	{
		/* >>chng 01 jan 24, get first estimate for xi_nu and iteratively refine, PvH */
		xi_nu = 1. + 1./sqrt(3.*nu);
		do 
		{
			double old = xi_nu;
			xi_nu = sqrt(1.+sqrt((2.*POW2(xi_nu)-1.)/(nu*xi_nu)));
			xxx = fabs(old-xi_nu)/xi_nu;
		} while( xxx > REL_TOLER );

		theta_nu = nu/xi_nu - 1./(2.*POW2(xi_nu)*(POW2(xi_nu)-1.));
	}
	else
	{
		theta_nu = 0.;
	}

#	ifdef DEBUG_FUN
	fputs( " <->ThetaNu()\n", debug_fp );
#	endif
	return theta_nu;
}


/* update items that depend on grain potential */
static void UpdatePot(long int nd)
{
	long loop,
	  newZlo,
	  nz,
	  offset,
	  Zg,
	  Zlo;
	const long MAXLOOP = 10L;

#	ifdef DEBUG_FUN
	fputs( "<+>UpdatePot()\n", debug_fp );
#	endif

	/* sanity checks */
	assert( gv.bin[nd]->dstpot >= gv.bin[nd]->LowestPot );
	assert( cache != NULL );

	/* MAX2 is to protect against roundoff error when dstpot == LowestPot */
	gv.bin[nd]->AveDustZ = MAX2(POT2CHRG(gv.bin[nd]->dstpot),(double)gv.bin[nd]->LowestZg);

	/* invalidate level populations */
	for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
	{
		gv.bin[nd]->FracPop[nz] = -DBL_MAX;
	}

	/* >>chng 01 may 30, calculation of newZlo is now valid in n-charge state model, unchanged for n = 2 */
	if( gv.bin[nd]->nChrg%2 == 0 )
	{
		/* integer division is intended in next two statements */
		double help = floor(gv.bin[nd]->AveDustZ);
		newZlo = NINT(help) - (gv.bin[nd]->nChrg-2)/2;
	}
	else
	{
		newZlo = NINT(gv.bin[nd]->AveDustZ) - (gv.bin[nd]->nChrg-1)/2;
	}
	newZlo = MAX2(newZlo,gv.bin[nd]->LowestZg);

	loop = 0;

	do
	{
		Zlo = newZlo;

		offset = ( cache->lgGlobalValid ) ? Zlo - gv.bin[nd]->DustZ[0] : LONG_MIN;

		/* >>chng 00 jul 17, use description of Weingartner & Draine, 2001 */
		/* >>chng 01 mar 21, assume that grain charge is distributed in two states just below and
		 * above the average charge. */
		/* >>chng 01 may 07, this routine now completely supports the hybrid grain
		 * charge model, i.e. all relevant quantities are calculated for Zlo and Zhi,
		 * and the average charge state is not used anywhere anymore, PvH */
		/* >>chng 01 may 30, reorganize code such that all relevant data can be copied
		 * when a valid set of data is available from a previous call, this saves CPU time, PvH */

		if( offset >= 0 )
		{
			for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
			{
				Zg = Zlo + nz;

				/* >>chng 01 may 11, update only if no valid copy of data is available */
				if( !cache->lgGlobalValid || nz+offset < 0 || nz+offset >= gv.bin[nd]->nChrg )
				{
					UpdatePotCalc(nd,nz,Zg);
				}
				else
				{
					UpdatePotCopy(nd,nz+offset,nz,Zg);
				}
			}
		}
		else
		{
			for( nz=gv.bin[nd]->nChrg-1; nz >= 0; nz-- )
			{
				Zg = Zlo + nz;

				/* >>chng 01 may 11, update only if no valid copy of data is available */
				if( !cache->lgGlobalValid || nz+offset < 0 || nz+offset >= gv.bin[nd]->nChrg )
				{
					UpdatePotCalc(nd,nz,Zg);
				}
				else
				{
					UpdatePotCopy(nd,nz+offset,nz,Zg);
				}
			}		
		}

		GetFracPop(nd,gv.bin[nd]->AveDustZ,Zlo,&newZlo,gv.bin[nd]->FracPop);

		cache->lgGlobalValid = TRUE;

		++loop;
	}

	while( loop < MAXLOOP && newZlo != Zlo );

	if( newZlo != Zlo )
	{
		fprintf( ioQQQ, " UpdatePot did not converge level populations\n" );
		puts( "[Stop in UpdatePot]" );
		cdEXIT(1);
	}
		
#	ifdef DEBUG_FUN
	fputs( " <->UpdatePot()\n", debug_fp );
#	endif
	return;
}


/* calculate charge state populations */
static void GetFracPop(long nd,
		       double AveZg,
		       long Zlo,
		       /*@out@*/ long *newZlo,
		       /*@out@*/ double FracPop[]) /* FracPop[gv.bin[nd]->nChrg] */
{
	long nz;

#	ifdef DEBUG_FUN
	fputs( "<+>GetFracPop()\n", debug_fp );
#	endif

	if( gv.bin[nd]->nChrg == 2 )
	{
		for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
		{
			long Zg = Zlo + nz;
			FracPop[nz] = 1.-fabs(AveZg-(double)Zg);
		}
		*newZlo = Zlo;
	}
	else
	{
		long i,j,k;
		double avePop[2],d[5],frac,pop[2][NCHS-1],rate_up[NCHS],rate_dn[NCHS];
#		if 0
		double test1,test2;
#		endif

		/* initialize charge transfer rates */
		for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
		{
			rate_up[nz] = GrainElecEmis1(nd,nz,&d[0],&d[1],&d[2],&d[3],&d[4]);
			rate_dn[nz] = GrainElecRecomb1(nd,nz,&d[0],&d[1]);
			/* sanity check */
			assert( rate_up[nz] >= 0. && rate_dn[nz] >= 0. );
		}

		/* solve level populations for levels 0..nChrg-2 (i == 0) and
		 * levels 1..nChrg-1 (i == 1), and determine average charge
		 * for each of those subsystems. Next we demand that
		 *         avePop[0] <= AveZg <= avePop[1]
		 * and determine FracPop by linearly adding the subsystems such that
		 *      AveZg == frac*avePop[0] + (1-frac)*avePop[1]
		 * this assures that all charge state populations are positive and all
		 * emission and recombination rates behave continuously as AveZg changes */
		for( i=0; i < 2; i++ )
		{
			double sum;

			pop[i][0] = 1.;
			sum = pop[i][0];
			for( j=1; j < gv.bin[nd]->nChrg-1; j++ )
			{
				nz = i + j;
				if( rate_dn[nz] > 10.*rate_up[nz-1]/sqrt(DBL_MAX) )
				{
					pop[i][j] = pop[i][j-1]*rate_up[nz-1]/rate_dn[nz];
					sum += pop[i][j];
				}
				else
				{
					for( k=0; k < j; k++ )
					{
						pop[i][k] = 0.;
					}
					pop[i][j] = 1.;
					sum = pop[i][j];
				}
				/* guard against overflow */
				if( pop[i][j] > sqrt(DBL_MAX) )
				{
					for( k=0; k <= j; k++ )
					{
						pop[i][k] /= DBL_MAX/10.;
					}
					sum /= DBL_MAX/10.;
				}
			}
			avePop[i] = 0.;
			for( j=0; j < gv.bin[nd]->nChrg-1; j++ )
			{
				nz = i + j;
				pop[i][j] /= sum;
				avePop[i] += pop[i][j]*(Zlo + nz);
			}
		}

		/* ascertain that the choice of Zlo was correct, this is to ensure positive
		 * level populations and continuous emission and recombination rates */
		if( AveZg < avePop[0] )
		{
			*newZlo = MAX2(Zlo - 1,gv.bin[nd]->LowestZg);
		}
		else if( AveZg > avePop[1] )
		{
			*newZlo = Zlo + 1;
		}
		else
		{
			*newZlo = Zlo;
		}

		frac = (AveZg - avePop[1])/(avePop[0] - avePop[1]);
		FracPop[0] = frac*pop[0][0];
		FracPop[gv.bin[nd]->nChrg-1] = (1.-frac)*pop[1][gv.bin[nd]->nChrg-2];
		for( i=1; i < gv.bin[nd]->nChrg-1; i++ )
		{
			FracPop[i] = frac*pop[0][i] + (1.-frac)*pop[1][i-1];
		}

#		if 0
		test1 = test2 = 0.;
		for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
		{
			test1 += FracPop[nz];
			test2 += FracPop[nz]*(double)(Zlo+nz);
		}
		printf(" GetFracPop test 1: %.15e\n",test1);
		printf(" GetFracPop test 2: %.15e %.15e\n",test2,AveZg);
		printf(" Level Pops %.15e %.15e %.15e Zlo %ld newZlo %ld\n",
		       FracPop[0],FracPop[1],FracPop[2],Zlo,*newZlo);
#		endif
	}

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "      FracPop:" );
		for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
		{
			fprintf( ioQQQ, " Zg %ld %.4e", Zlo+nz, FracPop[nz] );
		}
		fprintf( ioQQQ, "\n" );
	}

#	ifdef DEBUG_FUN
	fputs( " <->GetFracPop()\n", debug_fp );
#	endif
	return;
}


/* this routine updates all quantities that depend on grain charge by calculating them
 *
 * NB NB - this routine should be kept parallel with UpdatePotCopy !!
 *
 * NB NB - All global data in grain.c and grainvar.h that have a dimension [NCHS]
 *         should be calculated here (except gv.bin[nd]->FracPop[nz] which is special). */
static void UpdatePotCalc(long nd,
			  long nz,
			  long Zg)
{
	long i,
	  ipLo,
	  ipHi;
	double d[2],
	  ThermExp;

#	ifdef DEBUG_FUN
	fputs( "<+>UpdatePotCalc()\n", debug_fp );
#	endif

	gv.bin[nd]->DustZ[nz] = Zg;

	GetPotValues(nd,Zg,&gv.bin[nd]->ThresInf[nz],&gv.bin[nd]->ThresInfVal[nz],
		     &gv.bin[nd]->ThresSurf[nz],&gv.bin[nd]->ThresSurfVal[nz],
		     &gv.bin[nd]->PotSurf[nz],INCL_TUNNEL);

	/* >>chng 01 may 09, do not use tunneling corrections for incoming electrons */
	/* >>chng 01 nov 25, add gv.bin[nd]->ThresInfInc[nz], PvH */
	GetPotValues(nd,Zg-1,&gv.bin[nd]->ThresInfInc[nz],&d[0],&gv.bin[nd]->ThresSurfInc[nz],&d[1],
		     &gv.bin[nd]->PotSurfInc[nz],NO_TUNNEL);

	/* >>chng 01 jan 08, ThresInf[nz] and ThresInfVal[nz] may become zero in
	 * initial stages of grain potential search, PvH */
	/* >>chng 01 oct 10, use bisection search to find ipThresInf, ipThresInfVal. On C scale now */
	ipLo = 0;
	/* >>chng 01 nov 29, nflux -> nupper so that pointer is always valid, even if above highest photon energy */
	ipHi = rfield.nupper-1;
	/* find anu[ipLo]-0.5*widflx[ipLo] <= gv.bin[nd]->ThresInf[nz] < anu[ipHi]-0.5*widflx[ipHi] */
	while( ipHi-ipLo > 1 )
	{
		long ipMd = (ipLo+ipHi)/2;
		if( rfield.anu[ipMd]-0.5f*rfield.widflx[ipMd] > (float)gv.bin[nd]->ThresInf[nz] )
			ipHi = ipMd;
		else
			ipLo = ipMd;
	}
	gv.bin[nd]->ipThresInf[nz] = ipLo;
	ipLo = 0;
	/* >>chng 01 nov 29, nflux -> nupper so that pointer is always valid, even if above highest photon energy */
	ipHi = rfield.nupper-1;
	/* find anu[ipLo]-0.5*widflx[ipLo] <= gv.bin[nd]->ThresInfVal[nz] < anu[ipHi]-0.5*widflx[ipHi] */
	while( ipHi-ipLo > 1 )
	{
		long ipMd = (ipLo+ipHi)/2;
		if( rfield.anu[ipMd]-0.5f*rfield.widflx[ipMd] > (float)gv.bin[nd]->ThresInfVal[nz] )
			ipHi = ipMd;
		else
			ipLo = ipMd;
	}
	gv.bin[nd]->ipThresInfVal[nz] = ipLo;

	for( i=0; i < MIN2(gv.bin[nd]->ipThresInfVal[nz],rfield.nflux); i++ )
	{
		cache->yhat[nz][i] = 0.;
	}
	for( i=gv.bin[nd]->ipThresInfVal[nz]; i < rfield.nflux; i++ )
	{
		double xv,yzero,y2,Elo,Ehi;
		xv = MAX2((rfield.anu[i] - gv.bin[nd]->ThresSurfVal[nz])/gv.bin[nd]->DustWorkFcn,0.);
		if( gv.bin[nd]->matType == MAT_CAR ) 
		{
			/* >>refer Bakes & Tielens, 1994, ApJ, 427, 822 */
			xv = POW3(xv)*POW2(xv);
			yzero = 9.e-3*xv/(1.+3.7e-2*xv);
		}
		else if( gv.bin[nd]->matType == MAT_SIL ) 
		{
			/* >>ref Weingartner & Draine, 2000 */
			yzero = 0.5*xv/(1.+5.*xv);
		}
		else
		{
			yzero = 0.;
		}
		if( Zg > -1 )
		{
			Elo = -gv.bin[nd]->PotSurf[nz];
			Ehi = rfield.anu[i] - gv.bin[nd]->ThresInfVal[nz];
			y2 = ( Ehi > 0. ) ? POW2(Ehi) * (Ehi - 3.*Elo) / POW3(Ehi - Elo) : 0.;
		}
		else 
		{
			y2 = 1.;
		}
		cache->yhat[nz][i] = (float)(y2*MIN2(yzero*gv.bin[nd]->y1[i],1.));
	}

	for( i=0; i < MIN2(gv.bin[nd]->ipThresInf[nz],rfield.nflux); i++ )
	{
		cache->cs_pdt[nz][i] = 0.;
	}
	for( i=gv.bin[nd]->ipThresInf[nz]; i < rfield.nflux; i++ )
	{
		if( gv.bin[nd]->DustZ[nz] <= -1 )
		{
			/* constants in the expression for the photodetachment cross section
			 * taken from 
			 * >>refer Weingartner & Draine, ApJS, 2001, 134, 263 */
			const double DELTA_E = 3./EVRYD;
			const double CS_PDT = 1.2e-17;
			
			double x = MAX2((rfield.anu[i] - gv.bin[nd]->ThresInf[nz])/DELTA_E,0.);
			double cs = -CS_PDT*(double)gv.bin[nd]->DustZ[nz]*x/POW2(1.+POW2(x)/3.);
			/* this cross section must be at default depletion for consistency
			 * with dstab1, hence no factor dstAbund should be applied here */
			cache->cs_pdt[nz][i] = MAX2(cs,0.)*gv.bin[nd]->cnv_GR_pH;
		}
		else
		{
			cache->cs_pdt[nz][i] = 0.;
		}
	}

	/* >>chng 00 jul 05, determine ionization stage Z0 the ion recombines to */
	UpdateRecomZ0(nd,nz,NONZERO_STAGES);

	/* >>chng 00 jun 19, add in loss rate due to thermionic emission of electrons, PvH */
	ThermExp = gv.bin[nd]->ThresInf[nz]*TE1RYD/gv.bin[nd]->tedust;
	/* ThermExp is guaranteed to be >= 0. */
	gv.bin[nd]->ThermRate[nz] =
		THERMCONST*gv.bin[nd]->ThermEff*POW2(gv.bin[nd]->tedust)*exp(-ThermExp);
#	ifdef WD_TEST2
	gv.bin[nd]->ThermRate[nz] = 0.;
#	endif

	/* sanity check */
	assert( gv.bin[nd]->ipThresInf[nz] <= gv.bin[nd]->ipThresInfVal[nz] );

#	ifdef DEBUG_FUN
	fputs( " <->UpdatePotCalc()\n", debug_fp );
#	endif
	return;
}


/* this routine updates all quantities that depend on grain charge by copying them
 *
 * NB NB - this routine should be kept parallel with UpdatePotCalc !!
 *
 * NB NB - All global data in grain.c and grainvar.h that have a dimension [NCHS]
 *         should be copied here (except gv.bin[nd]->FracPop[nz] which is special).
 *
 * NB NB - TODO: introduce structures like gv.bin[nd]->chrg[nz]->DustZ
 *         this routine than simply has to swap structures instead of copying them
 *         element by element, this is faster, but more importantly safer ! */
static void UpdatePotCopy(long nd,
			  long nz_from,
			  long nz_to,
			  long Zg)
{
	long i,
	  ion,
	  ipZ;

#	ifdef DEBUG_FUN
	fputs( "<+>UpdatePotCopy()\n", debug_fp );
#	endif

	/* sanity checks */
	assert( nz_from >= 0 && nz_from < gv.bin[nd]->nChrg );
	assert( nz_to >= 0 && nz_to < gv.bin[nd]->nChrg );
	assert( gv.bin[nd]->DustZ[nz_from] == Zg );

	if( nz_to != nz_from )
	{
		gv.bin[nd]->DustZ[nz_to] = gv.bin[nd]->DustZ[nz_from];

		gv.bin[nd]->ThresInf[nz_to] = gv.bin[nd]->ThresInf[nz_from];
		gv.bin[nd]->ThresInfVal[nz_to] = gv.bin[nd]->ThresInfVal[nz_from];
		gv.bin[nd]->ThresSurf[nz_to] = gv.bin[nd]->ThresSurf[nz_from];
		gv.bin[nd]->ThresSurfVal[nz_to] = gv.bin[nd]->ThresSurfVal[nz_from];
		gv.bin[nd]->PotSurf[nz_to] = gv.bin[nd]->PotSurf[nz_from];
		gv.bin[nd]->ThresInfInc[nz_to] = gv.bin[nd]->ThresInfInc[nz_from];
		gv.bin[nd]->ThresSurfInc[nz_to] = gv.bin[nd]->ThresSurfInc[nz_from];
		gv.bin[nd]->PotSurfInc[nz_to] = gv.bin[nd]->PotSurfInc[nz_from];
		gv.bin[nd]->ipThresInf[nz_to] = gv.bin[nd]->ipThresInf[nz_from];
		gv.bin[nd]->ipThresInfVal[nz_to] = gv.bin[nd]->ipThresInfVal[nz_from];

		for( i=0; i < rfield.nflux; i++ )
		{
			cache->yhat[nz_to][i] = cache->yhat[nz_from][i];
			cache->cs_pdt[nz_to][i] = cache->cs_pdt[nz_from][i];
		}

		for( ipZ=0; ipZ < LIMELM; ipZ++ )
		{
			if( abundances.lgElmtOn[ipZ] )
			{
				for( ion=0; ion <= ipZ+1; ion++ )
				{
					cache->RecomZ0[nz_to][ipZ][ion] = cache->RecomZ0[nz_from][ipZ][ion];
					cache->RecomEn[nz_to][ipZ][ion] = cache->RecomEn[nz_from][ipZ][ion];
					cache->ChemEn[nz_to][ipZ][ion] = cache->ChemEn[nz_from][ipZ][ion];
				}
			}
		}

		gv.bin[nd]->ThermRate[nz_to] = gv.bin[nd]->ThermRate[nz_from];
	}

#	ifdef DEBUG_FUN
	fputs( " <->UpdatePotCopy()\n", debug_fp );
#	endif
	return;
}

/* find highest ionization stage with non-zero population */
static long HighestIonStage(void)
{
	long high,
		ion,
		ipZ;

#	ifdef DEBUG_FUN
	fputs( "<+>HighestIonStage()\n", debug_fp );
#	endif

	high = 0;
	for( ipZ=LIMELM-1; ipZ >= 0; ipZ-- )
	{
		if( abundances.lgElmtOn[ipZ] )
		{
			for( ion=ipZ+1; ion >= 0; ion-- )
			{
				if( ion == high || xIonFracs[ipZ][ion+1] > 0. )
					break;
			}
			high = MAX2(high,ion);
		}
		if( ipZ <= high )
			break;
	}

#	ifdef DEBUG_FUN
	fputs( " <->HighestIonStage()\n", debug_fp );
#	endif
	return high;
}


static void UpdateRecomZ0(long nd,
			  long nz,
			  int lgAllIonStages)
{
	long hi_ion,
	  i,
	  ion,
	  ipZ,
	  Zg;
	double d[4],
	  phi_s_up[LIMELM+1],
	  phi_s_dn[2];

#	ifdef DEBUG_FUN
	fputs( "<+>UpdateRecomZ0()\n", debug_fp );
#	endif

	Zg = gv.bin[nd]->DustZ[nz];

	hi_ion = ( lgAllIonStages ) ? LIMELM : gv.HighestIon;

	phi_s_up[0] = gv.bin[nd]->ThresSurf[nz];
	for( i=1; i <= LIMELM; i++ )
	{
		if( i <= hi_ion )
			GetPotValues(nd,Zg+i,&d[0],&d[1],&phi_s_up[i],&d[2],&d[3],INCL_TUNNEL);
		else
			phi_s_up[i] = -DBL_MAX;
	}
	phi_s_dn[0] = gv.bin[nd]->ThresSurfInc[nz];
	GetPotValues(nd,Zg-2,&d[0],&d[1],&phi_s_dn[1],&d[2],&d[3],NO_TUNNEL);

	/* >>chng 01 may 09, use GrainIonColl which properly tracks step-by-step charge changes */
	for( ipZ=0; ipZ < LIMELM; ipZ++ )
	{
		if( abundances.lgElmtOn[ipZ] )
		{
			for( ion=0; ion <= ipZ+1; ion++ )
			{
				if( lgAllIonStages || xIonFracs[ipZ][ion+1] > 0. )
				{
					GrainIonColl(nd,nz,ipZ,ion,phi_s_up,phi_s_dn,
						     &cache->RecomZ0[nz][ipZ][ion],
						     &cache->RecomEn[nz][ipZ][ion],
						     &cache->ChemEn[nz][ipZ][ion]);
				}
				else
				{
					cache->RecomZ0[nz][ipZ][ion] = ion;
					cache->RecomEn[nz][ipZ][ion] = 0.f;
					cache->ChemEn[nz][ipZ][ion] = 0.f;
				}
			}
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->UpdateRecomZ0()\n", debug_fp );
#	endif
	return;
}

static void GetPotValues(long nd,
			 long Zg,
			 /*@out@*/ double *ThresInf,
			 /*@out@*/ double *ThresInfVal,
			 /*@out@*/ double *ThresSurf,
			 /*@out@*/ double *ThresSurfVal,
			 /*@out@*/ double *PotSurf,
			 int lgUseTunnelCorr)
{
	double dstpot,
	  dZg = (double)Zg,
	  IP_v;

#	ifdef DEBUG_FUN
	fputs( "<+>GetPotValues()\n", debug_fp );
#	endif

	/* >>chng 01 may 07, this routine now completely supports the hybrid grain charge model,
	 * the change for this routine is that now it is only fed integral charge states; calculation
	 * of IP has also been changed in accordance with Weingartner & Draine, 2001, PvH */

	/* this is average grain potential in Rydberg */
	dstpot = CHRG2POT(dZg);

	/* >>chng 01 mar 20, include O(a^-2) correction terms in ionization potential */
	/* these are correction terms for the ionization potential that are
	 * important for small grains. See Weingartner & Draine, 2000, Eq. 2 */
	IP_v = gv.bin[nd]->DustWorkFcn + dstpot - 0.5*ONE_ELEC + (dZg+2.)*AC0/gv.bin[nd]->AvRadius*ONE_ELEC;

	/* >>chng 01 mar 01, use new expresssion for ThresInfVal, ThresSurfVal following the discussion
	 * with Joe Weingartner. Also include the Schottky effect (see 
	 * >>refer Spitzer, 1948, ApJ, 107, 6,
	 * >>refer Draine & Sutin, 1987, ApJ, 320, 803), PvH */
	if( Zg <= -1 )
	{
		double Emin,IP;

		IP = gv.bin[nd]->DustWorkFcn - gv.bin[nd]->BandGap + dstpot - 0.5*ONE_ELEC;
		if( gv.bin[nd]->matType == MAT_CAR )
		{
			IP -= AC1G/(gv.bin[nd]->AvRadius+AC2G)*ONE_ELEC;
		}

		/* prevent valence electron from becoming less bound than attached electron; this
		 * can happen for very negative, large graphitic grains and is not physical, PvH */
		IP_v = MAX2(IP,IP_v);

		if( Zg < -1 )
		{
			/* >>chng 01 apr 20, use improved expression for tunneling effect, PvH */
			double help = fabs(dZg+1);
			/* this is the barrier height solely due to the Schottky effect */
			Emin = -ThetaNu(help)*ONE_ELEC;
			if( lgUseTunnelCorr )
			{
				/* this is the barrier height corrected for tunneling effects */
				Emin *= 1. - 2.124e-4/(pow(gv.bin[nd]->AvRadius,0.45f)*pow(help,0.26));
			}
		}
		else
		{
			Emin = 0.;
		}

		*ThresInf = IP - Emin;
		*ThresInfVal = IP_v - Emin;
		*ThresSurf = *ThresInf;
		*ThresSurfVal = *ThresInfVal;
		*PotSurf = Emin;
	}
	else
	{
		*ThresInf = IP_v;
		*ThresInfVal = IP_v;
		*ThresSurf = *ThresInf - dstpot;
		*ThresSurfVal = *ThresInfVal - dstpot;
		*PotSurf = dstpot;
	}

#	ifdef DEBUG_FUN
	fputs( " <->GetPotValues()\n", debug_fp );
#	endif
	return;
}


/* given grain nd in charge state nz, and incoming ion (ipZ,ion),
 * detemine outgoing ion (ipZ,Z0) and chemical energy ChEn released
 * ChemEn is net contribution of ion recombination to grain heating */
static void GrainIonColl(long int nd,
			 long int nz,
			 long int ipZ,
			 long int ion,
			 const double phi_s_up[], /* phi_s_up[LIMELM+1] */
			 const double phi_s_dn[], /* phi_s_dn[2] */
			 /*@out@*/long *Z0,
			 /*@out@*/float *ChEn,
			 /*@out@*/float *ChemEn)
{
	long Zg;
	double d[4];
	double phi_s;
	
#	ifdef DEBUG_FUN
	fputs( "<+>GrainIonColl()\n", debug_fp );
#	endif

	long save = ion;
	if( ion > 0 && rfield.anu[Heavy.ipHeavy[ion-1][ipZ]-1] > (float)phi_s_up[0] )
	{
		/* ion will get electron(s) */
		*ChEn = 0.f;
		*ChemEn = 0.f;
		Zg = gv.bin[nd]->DustZ[nz];
		phi_s = phi_s_up[0];
		do 
		{
			*ChEn += rfield.anu[Heavy.ipHeavy[ion-1][ipZ]-1] - (float)phi_s;
			*ChemEn += rfield.anu[Heavy.ipHeavy[ion-1][ipZ]-1];
			/* this is a correction for the imperfections in the n-charge state model:
			 * since the transfer gets modeled as n single-electron transfers, instead of one
			 * n-electron transfer, a correction for the difference in binding energy is needed */
			*ChemEn -= (float)(phi_s - phi_s_up[0]);
			--ion;
			++Zg;
			phi_s = phi_s_up[save-ion];
		} while( ion > 0 && rfield.anu[Heavy.ipHeavy[ion-1][ipZ]-1] > (float)phi_s );

		*Z0 = ion;
	}
	else if( ion <= ipZ && gv.bin[nd]->DustZ[nz] > gv.bin[nd]->LowestZg &&
		 rfield.anu[Heavy.ipHeavy[ion][ipZ]-1] < (float)phi_s_dn[0] )
	{
		/* grain will get electron(s) */
		*ChEn = 0.f;
		*ChemEn = 0.f;
		Zg = gv.bin[nd]->DustZ[nz];
		phi_s = phi_s_dn[0];
		do 
		{
			*ChEn += (float)phi_s - rfield.anu[Heavy.ipHeavy[ion][ipZ]-1];
			*ChemEn -= rfield.anu[Heavy.ipHeavy[ion][ipZ]-1];
			/* this is a correction for the imperfections in the n-charge state model:
			 * since the transfer gets modeled as n single-electron transfers, instead of one
			 * n-electron transfer, a correction for the difference in binding energy is needed */
			*ChemEn += (float)(phi_s - phi_s_dn[0]);
			++ion;
			--Zg;

			if( ion-save < 2 )
				phi_s = phi_s_dn[ion-save];
			else
				GetPotValues(nd,Zg-1,&d[0],&d[1],&phi_s,&d[2],&d[3],NO_TUNNEL);

		} while( ion <= ipZ && Zg > gv.bin[nd]->LowestZg &&
			 rfield.anu[Heavy.ipHeavy[ion][ipZ]-1] < (float)phi_s );
		*Z0 = ion;
	}
	else
	{
		/* nothing happens */
		*ChEn = 0.f;
		*ChemEn = 0.f;
		*Z0 = ion;
	}
/*  	printf(" GrainIonColl: ipZ %ld ion %ld -> %ld, ChEn %.6f\n",ipZ,save,*Z0,*ChEn); */

#	ifdef DEBUG_FUN
	fputs( " <->GrainIonColl()\n", debug_fp );
#	endif
	return;
}


/* initialize ion recombination rates on grain species nd */
static void GrainRecomRates(long nd)
{
#	ifdef DEBUG_FUN
	fputs( "<+>GrainRecomRates()\n", debug_fp );
#	endif

	long ion,
	  ipZ,
	  nz;
	double eta,
	  xi;

	for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
	{
		for( ion=0; ion <= LIMELM; ion++ )
		{
			/* >>chng 00 jul 19, replace classical results with results including image potential
			 * to correct for polarization of the grain as charged particle approaches. */
			GrainScreen(ion,gv.bin[nd]->DustZ[nz],gv.bin[nd]->Capacity,&eta,&xi);

			for( ipZ=MAX2(0,ion-1); ipZ < LIMELM; ipZ++ )
			{
				if( abundances.lgElmtOn[ipZ] && ion != cache->RecomZ0[nz][ipZ][ion] )
				{
					gv.GrainRecom[ipZ][ion][cache->RecomZ0[nz][ipZ][ion]] +=
						(float)(gv.bin[nd]->FracPop[nz]*STICK_ION*DoppVel.AveVel[ipZ]*eta*
							gv.bin[nd]->IntArea/4.*gv.bin[nd]->cnv_H_pCM3);
				}
			}
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->GrainRecomRates()\n", debug_fp );
#	endif
	return;
}


/* this routine updates all grain quantities that depend on radius */
static void GrainUpdateRadius1(void)
{
	long i,
	  ipZ,
	  nd;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainUpdateRadius1()\n", debug_fp );
#	endif

	for( i=0; i < rfield.nupper; i++ )
	{
		/* these are total absorption and scattering cross sections */
		gv.dstab[i] = 0.;
		gv.dstsc[i] = 0.;
	}

	for( ipZ=0; ipZ < LIMELM; ipZ++ )
	{
		gv.elmSumAbund[ipZ] = 0.f;
	}

	/* grain abundance may be a function of depth */
	for( nd=0; nd < gv.nBin; nd++ )
	{
		/* refresh grain abundance if variable grain abundances enabled */
		if( gv.bin[nd]->dstAbund < 0.f || gv.bin[nd]->lgDustVary )
		{
			gv.bin[nd]->dstAbund = (float)(gv.bin[nd]->dstfactor*GrnVryDpth(nd));
			assert( gv.bin[nd]->dstAbund > 0.f );
		}

		/* this is effective area per proton, scaled by depletion
		 * dareff(nd) = darea(nd) * dstAbund(nd) */
		/* >>chng 01 mar 26, from nupper to nflux */
		for( i=0; i < rfield.nflux; i++ )
		{
			/* these are total absorption and scattering cross sections */
			/* we do not know the grain charge yet, so we neglect photo-detachment
			 * here, these cross sections will be added in later once charge is known */
			gv.dstab[i] += gv.bin[nd]->dstab1[i]*gv.bin[nd]->dstAbund;
			gv.dstsc[i] += gv.bin[nd]->dstsc1[i]*gv.bin[nd]->dstAbund;
		}

		/* grain unit conversion, <unit>/H (default depl) -> <unit>/cm^3 (actual depl) */
		gv.bin[nd]->cnv_H_pCM3 = phycon.hden*gv.bin[nd]->dstAbund;
		gv.bin[nd]->cnv_CM3_pH = 1./gv.bin[nd]->cnv_H_pCM3;
		/* grain unit conversion, <unit>/cm^3 (actual depl) -> <unit>/grain */
		gv.bin[nd]->cnv_CM3_pGR = gv.bin[nd]->cnv_H_pGR/gv.bin[nd]->cnv_H_pCM3;
		gv.bin[nd]->cnv_GR_pCM3 = 1./gv.bin[nd]->cnv_CM3_pGR;

		/* >>chng 01 dec 05, calculate the number density of each element locked in grains,
		 * summed over all grain bins. this number uses the actual depletion of the grains
		 * and is already multiplied with hden, units cm^-3. */
		for( ipZ=0; ipZ < LIMELM; ipZ++ )
		{
			gv.elmSumAbund[ipZ] += gv.bin[nd]->elmAbund[ipZ]*(float)gv.bin[nd]->cnv_H_pCM3;
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->GrainUpdateRadius1()\n", debug_fp );
#	endif
	return;
}


/* this routine adds the photo-dissociation cs to gv.dstab, this could not be
 * done in GrainUpdateRadius1 since charge and FracPop must be converged first */
static void GrainUpdateRadius2(long nd)
{
	long i,
	  nz_max;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainUpdateRadius2()\n", debug_fp );
#	endif

	/* >>chng 01 may 30, cs_pdt is only non-zero for negative charge states */
	nz_max = MIN2(MAX2(-gv.bin[nd]->DustZ[0],0),gv.bin[nd]->nChrg);

	/* >>chng 01 mar 26, from nupper to nflux */
	for( i=0; i < rfield.nflux; i++ )
	{
		long nz;
		/* >>chng 01 mar 24, added in cs for photodetachment from negative grains, PvH */
		/* >>chng 01 may 07, use two charge state approximation */
		/* >>chng 01 may 30, replace upper limit of loop gv.bin[nd]->nChrg -> nz_max */
		for( nz=0; nz < nz_max; nz++ )
		{
			gv.dstab[i] +=
				gv.bin[nd]->FracPop[nz]*cache->cs_pdt[nz][i]*gv.bin[nd]->dstAbund;
		}
		/* this must be positive, zero in case of uncontrolled underflow */
		assert( gv.dstab[i] > 0. );
	}

#	ifdef DEBUG_FUN
	fputs( " <->GrainUpdateRadius2()\n", debug_fp );
#	endif
	return;
}


/* GrainTemperature computes grains temperature, and gas cooling */
static void GrainTemperature(long int nd,
			     /*@out@*/ float *dccool,
			     /*@out@*/ double *hcon,
			     /*@out@*/ double *hots,
			     /*@out@*/ double *hla,
			     /*@out@*/ double *thermionic)
{
	long int i,
	  ipLya,
	  nz;
	double EhatThermionic,
	  norm,
	  rate,
	  x, 
	  y;
	float dcheat;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainTemperature()\n", debug_fp );
#	endif

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "  GrainTemperature starts for grain %s\n", gv.bin[nd]->chDstLab );
	}

	/* >>chng 01 may 07, this routine now completely supports the hybrid grain
	 * charge model, i.e. all relevant quantities are calculated for Zlo and Zhi,
	 * and the average charge state is not used anywhere anymore, PvH */

	/* direct heating by incident continuum (all energies) */
	*hcon = 0.;
	/* heating by diffuse ots fields */
	*hots = 0.;
	/* heating by Ly alpha alone, for output only, is already included in hots */
	*hla = 0.;
	ipLya = EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].ipCont;

	/* integrate over ionizing continuum; energy goes to dust and gas
	 * GasHeatPhotoEl is what heats the gas */
	gv.bin[nd]->GasHeatPhotoEl = 0.;

	gv.bin[nd]->GrainCoolTherm = 0.;
	*thermionic = 0.;

	dcheat = 0.f;
	*dccool = 0.f;

	if( gv.bin[nd]->lgQHeat )
	{
		/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
		for( i=0; i < gv.qnflux; i++ )
		{
			gv.bin[nd]->phiTilde[i] = 0.;
		}
	}

	gv.bin[nd]->BolFlux = 0.;

	for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
	{
		double hcon1 = 0.;
		double hots1 = 0.;
		double hla1 = 0.;
		double bolflux = 0.;

		/* integrate over incident continuum for non-ionizing energies */
		for( i=0; i < MIN2(gv.bin[nd]->ipThresInf[nz],rfield.nflux); i++ )
		{
			hcon1 += rfield.flux[i]*gv.bin[nd]->dstab1[i]*rfield.anu[i];
			hots1 += rfield.SummedDif[i]*gv.bin[nd]->dstab1[i]*rfield.anu[i];
			bolflux += rfield.SummedCon[i]*gv.bin[nd]->dstab1[i]*rfield.anu[i];

			/* >>chng 00 apr 18, moved calculation of hla, by PvH */
			if( i == ipLya-1 ) 
			{
				/*  heating by Ly A on dust in this zone,
				 *  only used for printout; Ly-a is already in OTS fields */
				hla1 = rfield.otslin[i]*gv.bin[nd]->dstab1[i]*0.75;
			}

			/* fill in auxiliary array for quantum heating routine
			 * it reshuffles the input spectrum times the cross section to take
			 * the photo-electric effect into account. this prevents the quantum
			 * heating routine from having to calculate this effect over and over
			 * again; it can do a straight integration instead, making the code
			 * a lot simpler and faster. this initializes the array for non-ionizing
			 * energies, the reshuffling for higher energies is done in the next loop
			 * phiTilde has units events/H/s/cell at default depletion */
			if( gv.bin[nd]->lgQHeat ) 
			{
				gv.bin[nd]->phiTilde[i] +=
					gv.bin[nd]->FracPop[nz]*rfield.SummedCon[i]*gv.bin[nd]->dstab1[i];
			}
		}

		/* >>chng 01 mar 02, use new expresssions for grain cooling and absorption
		 * cross sections following the discussion with Joe Weingartner, PvH */
		for( i=gv.bin[nd]->ipThresInf[nz]; i < rfield.nflux; i++ )
		{
			double cool1,cool2,ehat1,ehat2;

			/* effective cross sections for photo-ejection, 1 = valence band, 2 = conduction band */
			double cs1 = gv.bin[nd]->dstab1[i]*cache->yhat[nz][i];
			double cs2 = cache->cs_pdt[nz][i];
			double cs_tot = gv.bin[nd]->dstab1[i] + cache->cs_pdt[nz][i];

			/* >>chng 00 jul 17, use description of Weingartner & Draine, 2000 */
			/* ehat1a,1b,2 is the average energy of the esacping electron at the grain surface */
			if( gv.bin[nd]->DustZ[nz] > -1 ) 
			{
				/* contribution from valence band */
				double Elo = -gv.bin[nd]->PotSurf[nz];
				double Ehi = rfield.anu[i] - gv.bin[nd]->ThresInfVal[nz];
				ehat1 = ( Ehi > 0. ) ?
					Ehi/2. * (Ehi-2.*Elo)/(Ehi-3.*Elo)+gv.bin[nd]->PotSurf[nz] : 0.;
				ehat2 = 0.;
			}
			else 
			{
				/* contribution from valence band */
				ehat1 = MAX2(rfield.anu[i] - gv.bin[nd]->ThresSurfVal[nz],0.)/2.;
				/* contribution from conduction band */
				ehat2 = MAX2(rfield.anu[i] - gv.bin[nd]->ThresInf[nz],0.);
			}

			/* cool1,2 is the amount by which photo-ejection cools the grain */
			cool1 = gv.bin[nd]->ThresSurfVal[nz] + ehat1;
			/* this term in cool1 describes the heating by an electron that de-excites
			 * from the conduction band into the valence band. it is assumed that all
			 * energy is absorbed by the grain, i.e. no photons are emitted */
			/* >>chng 01 nov 27, changed de-excitation energy to conserve energy, PvH */
			if( gv.bin[nd]->DustZ[nz] <= -1 )
				cool1 += gv.bin[nd]->ThresSurf[nz] - gv.bin[nd]->ThresSurfVal[nz];

			cool2 = gv.bin[nd]->ThresSurf[nz] + ehat2;

			/* sanity checks */
			assert( ehat1 > 0. || ( ehat1 == 0. && cs1 == 0. ) );
			assert( ehat2 > 0. || ( ehat2 == 0. && cs2 == 0. ) );
			assert( cool1 >= 0. && cool2 >= 0. );

			/* this is heating by incident radiation field */
			/* >>chng 01 mar 24, use MAX2 (4 times) to protect against roundoff error, PvH */
			hcon1 += rfield.flux[i]*MAX2(cs_tot*rfield.anu[i]-cs1*cool1-cs2*cool2,0.);

			/*  this is heating by all diffuse fields:
			 *  SummedDif has all continua and lines */
			hots1 += rfield.SummedDif[i]*MAX2(cs_tot*rfield.anu[i]-cs1*cool1-cs2*cool2,0.);

			bolflux += rfield.SummedCon[i]*cs_tot*rfield.anu[i];

			/* >>chng 00 apr 18, include photo-electric effect, by PvH */
			if( i == ipLya-1 ) 
			{
				/*  heating by Ly A on dust in this zone,
				 *  only used for printout; Ly-a is already in OTS fields */
				hla1 = rfield.otslin[i]*MAX2(cs_tot*0.75-cs1*cool1-cs2*cool2,0.);
			}

			/*  GasHeatPhotoEl is rate grain photoionization heats the gas */
#ifdef WD_TEST2
			gv.bin[nd]->GasHeatPhotoEl += gv.bin[nd]->FracPop[nz]*rfield.flux[i]*
				(cs1*(ehat1-gv.bin[nd]->PotSurf[nz]) + cs2*(ehat2-gv.bin[nd]->PotSurf[nz]));
#else
			gv.bin[nd]->GasHeatPhotoEl += gv.bin[nd]->FracPop[nz]*rfield.SummedCon[i]*
				(cs1*(ehat1-gv.bin[nd]->PotSurf[nz]) + cs2*(ehat2-gv.bin[nd]->PotSurf[nz]));
#endif

			if( gv.bin[nd]->lgQHeat )
			{
				/* this accounts for the photons that are fully absorbed by grain */
				gv.bin[nd]->phiTilde[i] +=
					gv.bin[nd]->FracPop[nz]*rfield.SummedCon[i]*MAX2(gv.bin[nd]->dstab1[i]-cs1,0.);

				/* >>chng 01 oct 10, use bisection search to find ip. On C scale now */

				/* this accounts for photons that eject an electron from the valence band */
				if( cs1 > 0. )
				{
					/* we treat photo-ejection from the valence band and de-excitation
					 * from the conduction band as one simultaneous event */
					long ipLo = 0;
					long ipHi = i;
					x = rfield.anu[i] - cool1;
					/* find anu[ipLo]-0.5*widflx[ipLo] <= x < anu[ipHi]-0.5*widflx[ipHi] */
					while( ipHi-ipLo > 1 )
					{
						long ipMd = (ipLo+ipHi)/2;
						if( rfield.anu[ipMd]-0.5f*rfield.widflx[ipMd] > (float)x )
							ipHi = ipMd;
						else
							ipLo = ipMd;
					}
					gv.bin[nd]->phiTilde[ipLo] +=
						gv.bin[nd]->FracPop[nz]*rfield.SummedCon[i]*cs1;
				}

				/* this accounts for photons that eject an electron from the conduction band */
				/* >>chng 01 dec 11, cool2 always equal to rfield.anu[i] -> no grain heating */
#				if 0
				if( cs2 > 0. )
				{
					long ipLo = 0;
					long ipHi = i;
					x = rfield.anu[i] - cool2;
					/* find anu[ipLo]-0.5*widflx[ipLo] <= x < anu[ipHi]-0.5*widflx[ipHi] */
					while( ipHi-ipLo > 1 )
					{
						long ipMd = (ipLo+ipHi)/2;
						if( rfield.anu[ipMd]-0.5f*rfield.widflx[ipMd] > (float)x )
							ipHi = ipMd;
						else
							ipLo = ipMd;
					}
					gv.bin[nd]->phiTilde[ipLo] +=
						gv.bin[nd]->FracPop[nz]*rfield.SummedCon[i]*cs2;
				}
#				endif
			}
		}

#		if 0
		{
			long i;
			double integral = 0.;
			for( i=0; i < gv.qnflux; i++ )
			{
				integral += gv.bin[nd]->phiTilde[i]*gv.bin[nd]->cnv_H_pCM3*
					rfield.anu[i]*EN1RYD;
			}
			printf(" integral test 1: integral %.6e\n",integral);
		}
#		endif

		*hcon += gv.bin[nd]->FracPop[nz]*hcon1;
		*hots += gv.bin[nd]->FracPop[nz]*hots1;
		*hla += gv.bin[nd]->FracPop[nz]*hla1;

		gv.bin[nd]->BolFlux += gv.bin[nd]->FracPop[nz]*bolflux;

		if( trace.lgTrace && trace.lgDustBug )
		{
			fprintf( ioQQQ, "    Zg %ld bolflux: %.4e\n", gv.bin[nd]->DustZ[nz],
			  gv.bin[nd]->FracPop[nz]*bolflux*EN1RYD*gv.bin[nd]->cnv_H_pCM3 );
		}

		/* add in thermionic emissions (thermal evaporation of electrons), it gives a cooling
		 * term for the grain. thermionic emissions will not be treated separately in quantum
		 * heating since they are only important when grains are heated to near-sublimation 
		 * temperatures; under those conditions quantum heating effects will never be important.
		 * in order to maintain energy balance they will be added to ion contribution though */
		/* ThermRate is normalized per cm^2 of grain surface area, scales with total grain area */
		rate = gv.bin[nd]->FracPop[nz]*gv.bin[nd]->ThermRate[nz]*gv.bin[nd]->IntArea*
			gv.bin[nd]->cnv_H_pCM3;
		/* >>chng 01 mar 02, PotSurf[nz] term was incorrectly taken into account, PvH */
		EhatThermionic = 2.*BOLTZMANN*gv.bin[nd]->tedust + MAX2(gv.bin[nd]->PotSurf[nz]*EN1RYD,0.);
		gv.bin[nd]->GrainCoolTherm += rate * (EhatThermionic + gv.bin[nd]->ThresSurf[nz]*EN1RYD);
		*thermionic += rate * (EhatThermionic - gv.bin[nd]->PotSurf[nz]*EN1RYD);
	}

	/*  heating by thermal collisions with gas does work
	 *  DCHEAT is grain collisional heating by gas
	 *  DCCOOL is gas cooling due to collisions with grains
	 *  they are different since grain surface recombinations
	 *  heat the grains, but do not cool the gas ! */
	GrainCollHeating(nd,&dcheat,dccool);

	/* norm is used to convert all heating rates to erg/cm^3/s */
	norm = EN1RYD*gv.bin[nd]->cnv_H_pCM3;

	/* hcon is radiative heating by incident radiation field */
	*hcon *= norm;

	/* hots is total heating of the grain by diffuse fields */
	*hots *= norm;

	/* heating by Ly alpha alone, for output only, is already included in hots */
	*hla *= norm;

	gv.bin[nd]->BolFlux *= norm;

	/* GasHeatPhotoEl is what heats the gas */
	if( gv.lgDHetOn )
	{
		gv.bin[nd]->GasHeatPhotoEl *= norm;
	}
	else
	{
		gv.bin[nd]->GasHeatPhotoEl = 0.;
	}

	/* >>chng 01 nov 29, removed next statement, PvH */
	/*  dust often hotter than gas during initial TE search */
	/* if( nzone <= 2 ) */
	/* 	dcheat = MAX2(0.f,dcheat); */

	/*  find power absorbed by dust and resulting temperature
	 *
	 * hcon is heating from incident continuum (all energies)
	 * hots is heating from ots continua and lines
	 * dcheat is net grain collisional and chemical heating by
	 *    particle collisions and recombinations
	 * GrainCoolTherm is grain cooling by thermionic emissions
	 *
	 * GrainHeat is net heating of this grain type,
	 *    to be balanced by radiative cooling */
	gv.bin[nd]->GrainHeat = *hcon + *hots + dcheat - gv.bin[nd]->GrainCoolTherm;

	/* remember collisional heating for this grain species */
	gv.bin[nd]->GrainHeatColl = dcheat;

	/* in case the thermionic rates become very large, this may become negative */
	assert( gv.bin[nd]->GrainHeat > 0. );

	/*  now find temperature, GrainHeat is sum of total heating of grain
	 *  >>chng 97 jul 17, divide by abundance here */
	x = log(MAX2(DBL_MIN,gv.bin[nd]->GrainHeat*gv.bin[nd]->cnv_CM3_pH));

	/* >>chng 96 apr 27, as per Peter van Hoof comment */
	if( x < gv.bin[nd]->dstems[0] )
	{
		gv.bin[nd]->tedust = (float)GRAIN_TMIN;
	}
	else if( x > gv.bin[nd]->dstems[NDEMS-1] )
	{
		gv.bin[nd]->tedust = (float)GRAIN_TMAX;
	}
	else
	{
		splint(gv.bin[nd]->dstems,gv.dsttmp,gv.bin[nd]->dstslp,NDEMS,x,&y);
		gv.bin[nd]->tedust = (float)exp(y);
	}

	/*  save for later possible printout */
	gv.bin[nd]->TeGrainMax = (float)MAX2(gv.bin[nd]->TeGrainMax,gv.bin[nd]->tedust);

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, " >GrainTemperature finds %s Tdst %.5e hcon %.4e ",
			 gv.bin[nd]->chDstLab, gv.bin[nd]->tedust, *hcon);
		fprintf( ioQQQ, "hots %.4e dcheat %.4e GrainCoolTherm %.4e\n", 
			 *hots, dcheat, gv.bin[nd]->GrainCoolTherm );
	}

#	ifdef DEBUG_FUN
	fputs( " <->GrainTemperature()\n", debug_fp );
#	endif
	return;
}

/* GrainCollHeating compute grains collisional heating cooling */
static void GrainCollHeating(long int nd,
			     /*@out@*/ float *dcheat, 
			     /*@out@*/ float *dccool)
{
	long int i,
	  ion, 
	  ipZ,
	  nz;
	double Accommodation,
	  CollisionRateElectr,      /* rate electrons strike grains */
	  CollisionRateMol,         /* rate molecules strike grains */
	  CollisionRateIon,         /* rate ions strike grains */
	  CoolTot,
	  CoolBounce,
	  CoolEmitted,
	  CoolElectrons,
	  CoolMolecules,
	  CoolPotential,
	  CoolPotentialGas,
	  eta,
	  HeatTot,
	  HeatBounce,
	  HeatCollisions,
	  HeatElectrons,
	  HeatIons,
	  HeatMolecules,
	  HeatRecombination, /* sum of abundances of ions times velocity times ionization potential times eta */
	  HeatChem,
	  HeatCor,
	  Stick,
	  ve,
	  WeightMol,
	  xi;


#	ifdef DEBUG_FUN
	fputs( "<+>GrainCollHeating()\n", debug_fp );
#	endif


	/* >>chng 01 may 07, this routine now completely supports the hybrid grain
	 * charge model, i.e. all relevant quantities are calculated for Zlo and Zhi,
	 * and the average charge state is not used anywhere anymore, PvH */

	/* this subroutine evaluates the gas heating-cooling rate
	 * (erg cm^-3 s^-1) due to grain gas collisions.
	 * the net effect can be positive or negative,
	 * depending on whether the grains or gas are hotter
	 * the physics is described in 
	 * >>refer Baldwin, Ferland, Martin et al., 1991, ApJ 374, 580 */

	HeatTot = 0.;
	CoolTot = 0.;

	HeatIons = 0.;

	gv.bin[nd]->ChemEn = 0.;

	/* loop over the charge states */
	for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
	{
		/* HEAT1 will be rate collisions heat the grain
		 * COOL1 will be rate collisions cool the gas kinetics */
		double Heat1 = 0.;
		double Cool1 = 0.;
		double ChemEn1 = 0.;

		/* ============================================================================= */
		/* heating/cooling due to neutrals and positive ions */

		/* loop over all stages of ionization */
		for( ion=0; ion <= LIMELM; ion++ )
		{
			/* this is heating of grains due to recombination energy of species,
			 * and assumes that every ion is fully neutralized upon striking the grain surface.
			 * all radiation produced in the recombination process is absorbed within the grain
			 *
			 * ion=0 are neutrals, ion=1 are single ions, etc
			 * each population is weighted by the AVERAGE velocity
			 * */
			CollisionRateIon = 0.;
			CoolPotential = 0.;
			CoolPotentialGas = 0.;
			HeatRecombination = 0.;
			HeatChem = 0.;

			/* >>chng 00 jul 19, replace classical results with results including image potential
			 * to correct for polarization of the grain as charged particle approaches. */
			GrainScreen(ion,gv.bin[nd]->DustZ[nz],gv.bin[nd]->Capacity,&eta,&xi);

			for( ipZ=MAX2(0,ion-1); ipZ < LIMELM; ipZ++ )
			{
				if( abundances.lgElmtOn[ipZ] && xIonFracs[ipZ][ion+1] > 0. )
				{
					double CollisionRateOne;

					/* >>chng 00 apr 05, use correct accomodation coefficient, by PvH
					 * the coefficient is defined at the end of appendix A.10 of BFM
					 * assume ion sticking prob is unity */
#ifdef WD_TEST2
					Stick = ( ion == cache->RecomZ0[nz][ipZ][ion] ) ?
						0. : STICK_ION;
#else
					Stick = ( ion == cache->RecomZ0[nz][ipZ][ion] ) ?
						gv.bin[nd]->AccomCoef[ipZ] : STICK_ION;
#endif
					/* this is rate with which charged ion strikes grain */
					/* >>chng 00 may 02, this had left 2./SQRTPI off */
					/* >>chng 00 may 05, use average speed instead of 2./SQRTPI*Doppler, PvH */
					CollisionRateOne = Stick*xIonFracs[ipZ][ion+1]*DoppVel.AveVel[ipZ];
					CollisionRateIon += CollisionRateOne;
					/* >>chng 01 nov 26, use PotSurfInc when appropriate:
					 * the values for the surface potential used here make it
					 * consistent with the rest of the code and preserve energy.
					 * NOTE: For incoming particles one should use PotSurfInc with
					 * Schottky effect for positive ion, for outgoing particles
					 * one should use PotSurf for Zg+ion-Z_0-1 (-1 because PotSurf
					 * assumes electron going out), these corrections are small
					 * and will be neglected for now, PvH */
					if( ion >= cache->RecomZ0[nz][ipZ][ion] )
					{
						CoolPotential += CollisionRateOne * (double)ion *
							gv.bin[nd]->PotSurf[nz];
						CoolPotentialGas += CollisionRateOne *
							(double)cache->RecomZ0[nz][ipZ][ion] *
							gv.bin[nd]->PotSurf[nz];
					}
					else
					{
						CoolPotential += CollisionRateOne * (double)ion *
							gv.bin[nd]->PotSurfInc[nz];
						CoolPotentialGas += CollisionRateOne *
							(double)cache->RecomZ0[nz][ipZ][ion] *
							gv.bin[nd]->PotSurfInc[nz];
					}
					/* this is sum of all energy liberated as ion recombines to Z0 in grain */
					/* >>chng 00 jul 05, subtract energy needed to get 
					 * electron out of grain potential well, PvH */
					/* >>chng 01 may 09, chemical energy now calculated in GrainIonColl, PvH */
					HeatRecombination += CollisionRateOne * cache->RecomEn[nz][ipZ][ion];
					HeatChem += CollisionRateOne * cache->ChemEn[nz][ipZ][ion];
				}
			}

			/* >>chng 00 may 01, boltzmann factor had multiplied all of factor instead
			 * of only first and last term.  pvh */

			/* equation 29 from Balwin et al 91 */
			/* this is direct collision rate, 2kT * xi, first term in eq 29 */
			HeatCollisions = CollisionRateIon * 2.*BOLTZMANN*phycon.te*xi;
			/* this is change in energy due to charge acceleration within grain's potential 
			 * this is exactly balanced by deceleration of incoming electrons and accelaration
			 * of outgoing photo-electrons and thermionic emissions; all these terms should
			 * add up to zero (total charge of grain should remain constant) */
			CoolPotential *= eta*EN1RYD;
			CoolPotentialGas *= eta*EN1RYD;
			/* this is recombination energy released within grain */
			HeatRecombination *= eta*EN1RYD;
			HeatChem *= eta*EN1RYD;
			/* energy carried away by neutrals after recombination, so a cooling term */
			CoolEmitted = CollisionRateIon * 2.*BOLTZMANN*gv.bin[nd]->tedust*eta;

			/* total GraC 0 in the emission line output */
			Heat1 += HeatCollisions - CoolPotential + HeatRecombination - CoolEmitted;

			/* rate kinetic energy lost from gas - gas cooling - eq 32 in BFM */
			/* this GrGC 0 in the main output */
			/* >>chng 00 may 05, reversed sign of gas cooling contribution */
			Cool1 += HeatCollisions - CoolEmitted - CoolPotentialGas;

			ChemEn1 += HeatChem;
		}

		/* remember grain heating by ion collisions for quantum heating treatment below */
		HeatIons += gv.bin[nd]->FracPop[nz]*Heat1;

		if( trace.lgTrace && trace.lgDustBug )
		{
			fprintf( ioQQQ, "    Zg %ld ions heat/cool: %.4e %.4e\n", gv.bin[nd]->DustZ[nz],
			  gv.bin[nd]->FracPop[nz]*Heat1*gv.bin[nd]->IntArea/4.*gv.bin[nd]->cnv_H_pCM3,
			  gv.bin[nd]->FracPop[nz]*Cool1*gv.bin[nd]->IntArea/4.*gv.bin[nd]->cnv_H_pCM3 );
		}

		/* ============================================================================= */
		/* heating/cooling due to electrons */

		ion = -1;
		Stick = ( gv.bin[nd]->DustZ[nz] <= -1 ) ? gv.bin[nd]->StickElecNeg : gv.bin[nd]->StickElecPos;
		/* VE is mean (not RMS) electron velocity */
		/*ve = TePowers.sqrte*6.2124e5;*/
		ve = sqrt(8.*BOLTZMANN/PI/ELECTRON_MASS*phycon.te);

		/* electron arrival rate - eqn 29 again */
		CollisionRateElectr = Stick*phycon.eden*ve;

		/* >>chng 00 jul 19, replace classical results with results including image potential
		 * to correct for polarization of the grain as charged particle approaches. */
		GrainScreen(ion,gv.bin[nd]->DustZ[nz],gv.bin[nd]->Capacity,&eta,&xi);

		if( gv.bin[nd]->DustZ[nz] > gv.bin[nd]->LowestZg )
		{
			HeatCollisions = CollisionRateElectr*2.*BOLTZMANN*phycon.te*xi;
			/* this is change in energy due to charge acceleration within grain's potential 
			 * this term (perhaps) adds up to zero when summed over all charged particles */
			CoolPotential = CollisionRateElectr * (double)ion*gv.bin[nd]->PotSurfInc[nz]*eta*EN1RYD;
			/* >>chng 00 jul 05, this is term for energy released due to recombination, PvH */
			HeatRecombination = CollisionRateElectr * gv.bin[nd]->ThresSurfInc[nz]*eta*EN1RYD;
			HeatBounce = 0.;
			CoolBounce = 0.;
		}
		else
		{
			HeatCollisions = 0.;
			CoolPotential = 0.;
			HeatRecombination = 0.;
			/* >>chng 00 jul 05, add in terms for electrons that bounce off grain, PvH */
			/* >>chng 01 mar 09, remove these terms, their contribution is negligible, and replace
			 * them with similar terms that describe electrons that are captured by grains at Z_min,
			 * these electrons are not in a bound state and the grain will quickly autoionize, PvH */
			HeatBounce = CollisionRateElectr * 2.*BOLTZMANN*phycon.te*xi;
			/* >>chng 01 mar 14, replace (2kT_g - phi_g) term with -EA; for autoionizing states EA is
			 * usually higher than phi_g, so more energy is released back into the electron gas, PvH */ 
			CoolBounce = CollisionRateElectr *
				(-gv.bin[nd]->ThresSurfInc[nz]-gv.bin[nd]->PotSurfInc[nz])*EN1RYD*eta;
			CoolBounce = MAX2(CoolBounce,0.);
		}
			
		/* >>chng 00 may 02, CoolPotential had not been included */
		/* >>chng 00 jul 05, HeatRecombination had not been included */
		HeatElectrons = HeatCollisions-CoolPotential+HeatRecombination+HeatBounce-CoolBounce;
		Heat1 += HeatElectrons;

		CoolElectrons = HeatCollisions+HeatBounce-CoolBounce;
		Cool1 += CoolElectrons;

		if( trace.lgTrace && trace.lgDustBug )
		{
			fprintf( ioQQQ, "    Zg %ld electrons heat/cool: %.4e %.4e\n", gv.bin[nd]->DustZ[nz],
			  gv.bin[nd]->FracPop[nz]*HeatElectrons*gv.bin[nd]->IntArea/4.*gv.bin[nd]->cnv_H_pCM3,
			  gv.bin[nd]->FracPop[nz]*CoolElectrons*gv.bin[nd]->IntArea/4.*gv.bin[nd]->cnv_H_pCM3 );
		}

		/* add quantum heating due to recombination of electrons, subtract thermionic cooling */
		if( gv.bin[nd]->lgQHeat ) 
		{
			double fac = BOLTZMANN/EN1RYD*phycon.te;
			/* E0 is barrier that electron needs to overcome, zero for positive grains */
			double E0 = MAX2(-gv.bin[nd]->PotSurfInc[nz],0.);
			/* >>chng 01 mar 02, this should be energy gap between top electron and infinity, PvH */
			/* >>chng 01 nov 21, use correct barrier: ThresInf[nz] -> ThresInfInc[nz], PvH */
			double Einf = gv.bin[nd]->ThresInfInc[nz] - E0;
			/* calculate heating rate in erg/H/s at standard depl
			 * include contributions for recombining electrons, autoionizing electrons
			 * and subtract thermionic emissions here since it is inverse process
			 *
			 * NB - in extreme conditions this rate may become negative (if there
			 * is an intense radiation field leading to very hot grains, but no ionizing
			 * photons, hence very few free electrons). we assume that the photon rates
			 * are high enough under those circumstances to avoid phiTilde becoming negative,
			 * but we will check that in qheat1 anyway. */
			double HeatingRate = HeatElectrons*gv.bin[nd]->IntArea/4. -
				gv.bin[nd]->GrainCoolTherm*gv.bin[nd]->cnv_CM3_pH;
			/* this is average energy deposited by one event, in erg
			 * this value is derived from distribution assumed here, and may
			 * not be the same as HeatElectrons/(CollisionRateElectr*eta) !! */
			/* >>chng 01 nov 21, use correct barrier: ThresInf[nz] -> ThresInfInc[nz], PvH */
			double E_av = gv.bin[nd]->ThresInfInc[nz]*EN1RYD + 2.*BOLTZMANN*phycon.te;
			/* this is rate in events/H/s at standard depletion */
			double rate = HeatingRate/E_av;

			double ylo = -exp(-E0/fac);
			/* this is highest kinetic energy of electron that can be represented in phiTilde */
			/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
			double Ehi = rfield.anu[gv.qnflux-1] + 0.5*rfield.widflx[gv.qnflux-1] - Einf;
			double yhi = ((E0-Ehi)/fac-1.)*exp(-Ehi/fac);
			/* renormalize rate so that integral over phiTilde*anu gives correct total energy */
			rate /= yhi-ylo;

			/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
			for( i=0; i < gv.qnflux; i++ ) 
			{
				Ehi = rfield.anu[i] + 0.5*rfield.widflx[i] - Einf;
				if( Ehi >= E0 ) 
				{
					/* Ehi is kinetic energy of electron at infinity */
					yhi = ((E0-Ehi)/fac-1.)*exp(-Ehi/fac);
					/* >>chng 01 mar 24, use MAX2 to protect against roundoff error, PvH */
					gv.bin[nd]->phiTilde[i] +=
						gv.bin[nd]->FracPop[nz]*rate*MAX2(yhi-ylo,0.);
					ylo = yhi;
				}
			}
#			if 0
			{
				long i;
				double integral = 0.;
				for( i=0; i < gv.qnflux; i++ )
				{
					integral += gv.bin[nd]->phiTilde[i]*gv.bin[nd]->cnv_H_pCM3*
						rfield.anu[i]*EN1RYD;
				}
				printf(" integral test 2: integral %.6e\n",integral);
			}
#			endif
		}

		/* heating/cooling above is in erg/s/cm^2 -> multiply with projected grain area per cm^3 */
		/* GraC 0 is integral of dcheat, the total collisional heating of the grain */
		HeatTot += gv.bin[nd]->FracPop[nz]*Heat1;

		/* GrGC 0 total cooling of gas integrated */
		CoolTot += gv.bin[nd]->FracPop[nz]*Cool1;

		gv.bin[nd]->ChemEn += gv.bin[nd]->FracPop[nz]*ChemEn1;
	}

	/* ============================================================================= */
	/* heating/cooling due to molecules */

	/* these rates do not depend on charge, hence they are outside of nz loop */

	/* sticking prob for H2 onto grain,
	 * estimated from accomodation coefficient defined at end of A.10 in BFM */
	WeightMol = 2.*AtomcWgt.AtomicWeight[0];
	Accommodation = 2.*gv.bin[nd]->atomWeight*WeightMol/POW2(gv.bin[nd]->atomWeight+WeightMol);
	/* molecular hydrogen onto grains */
	CollisionRateMol = Accommodation*hmi.htwo*
		sqrt(8.*BOLTZMANN/PI/ATOMIC_MASS_UNIT/WeightMol*phycon.te);

	/* now add in CO */
	WeightMol = AtomcWgt.AtomicWeight[5] + AtomcWgt.AtomicWeight[7];
	Accommodation = 2.*gv.bin[nd]->atomWeight*WeightMol/POW2(gv.bin[nd]->atomWeight+WeightMol);
	CollisionRateMol += Accommodation*hevmolec.hevmol[ipCO]*
		sqrt(8.*BOLTZMANN/PI/ATOMIC_MASS_UNIT/WeightMol*phycon.te);

	/* xi and eta are unity for neutrals and so ignored */
	HeatCollisions = CollisionRateMol * 2.*BOLTZMANN*phycon.te;
	CoolEmitted = CollisionRateMol * 2.*BOLTZMANN*gv.bin[nd]->tedust;

	HeatMolecules = HeatCollisions - CoolEmitted;
	HeatTot += HeatMolecules;

	/* >>chng 00 may 05, reversed sign of gas cooling contribution */
	CoolMolecules = HeatCollisions - CoolEmitted;
	CoolTot += CoolMolecules;

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "    molecules heat/cool: %.4e %.4e\n",
			 HeatMolecules*gv.bin[nd]->IntArea/4.*gv.bin[nd]->cnv_H_pCM3,
			 CoolMolecules*gv.bin[nd]->IntArea/4.*gv.bin[nd]->cnv_H_pCM3 );
	}

	/* >>chng 01 nov 24, correct for imperfections in the n-charge state model,
	 * these corrections should add up to zero, but are actually small but non-zero, PvH */
	gv.bin[nd]->RateUp = 0.;
	gv.bin[nd]->RateDn = 0.;
	HeatCor = 0.;
	for( nz=0; nz < gv.bin[nd]->nChrg; nz++ )
	{
		double d[4],Auger;
		double rate_dn = GrainElecRecomb1(nd,nz,&d[0],&d[1]);
		double rate_up = GrainElecEmis1(nd,nz,&d[0],&Auger,&d[1],&d[2],&d[3]);

		gv.bin[nd]->RateUp += gv.bin[nd]->FracPop[nz]*rate_up;
		gv.bin[nd]->RateDn += gv.bin[nd]->FracPop[nz]*rate_dn;

		/* >>chng 01 dec 19, subtract the Auger rate, it should not be used in heating correction */
		fixit(); /* a self-consistent treatment for the heating by Auger electrons should be used */
		fixit(); /* a self-consistent treatment for the heating by Compton recoil electrons should be used */
		HeatCor += (gv.bin[nd]->FracPop[nz]*(rate_up-Auger)*gv.bin[nd]->ThresSurf[nz] -
			    gv.bin[nd]->FracPop[nz]*rate_dn*gv.bin[nd]->ThresSurfInc[nz] +
			    gv.bin[nd]->FracPop[nz]*(rate_up-Auger)*gv.bin[nd]->PotSurf[nz] -
			    gv.bin[nd]->FracPop[nz]*rate_dn*gv.bin[nd]->PotSurfInc[nz])*EN1RYD;
	}
	HeatTot += HeatCor;

	*dcheat = (float)(HeatTot*gv.bin[nd]->IntArea/4.*gv.bin[nd]->cnv_H_pCM3);
	*dccool = (float)(CoolTot*gv.bin[nd]->IntArea/4.*gv.bin[nd]->cnv_H_pCM3);

	gv.bin[nd]->ChemEn *= gv.bin[nd]->IntArea/4.*gv.bin[nd]->cnv_H_pCM3;

	/* ============================================================================= */

	/* add quantum heating due to molecule/ion collisions */
	if( gv.bin[nd]->lgQHeat )
	{
		double Ehi,E_av,fac,rate,ylo,yhi,HeatingRate;

		/* if gas temperature is higher than grain temperature we will
		 * consider Maxwell-Boltzmann distribution of incoming particles
		 * and ignore distribution of outgoing particles, if grains
		 * are hotter than ambient gas, we use reverse treatment */
		fac = BOLTZMANN/EN1RYD*MAX2(phycon.te,gv.bin[nd]->tedust);
		/* calculate heating rate in erg/H/s at standard depl
		 * include contributions from molecules/neutral atoms and recombining ions
		 *
		 * in fully ionized conditions electron heating rates will be much higher
		 * than ion and molecule rates since electrons are so much faster and grains
		 * tend to be positive. in non-ionized conditions the main contribution will
		 * come from neutral atoms and molecules, so it is appropriate to treat both
		 * the same. in fully ionized conditions we don't care since unimportant.
		 *
		 * NB - if grains are hotter than ambient gas, the heating rate may become negative.
		 * if photon rates are not high enough to prevent phiTilde from becoming negative,
		 * we will raise a flag while calculating the quantum heating in qheat1 */
		/* >>chng 01 nov 26, add in HeatCor as well, otherwise energy imbalance will result, PvH */
		HeatingRate = (HeatMolecules+HeatIons+HeatCor)*gv.bin[nd]->IntArea/4.;
		/* this is average energy deposited/extracted by one event, in erg */
		E_av = 2.*BOLTZMANN*MAX2(phycon.te,gv.bin[nd]->tedust);
		/* this is rate in events/H/s at standard depletion */
		rate = HeatingRate/E_av;

		ylo = -1.;
		/* this is highest energy of incoming/outgoing particle that can be represented in phiTilde */
		/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
		Ehi = rfield.anu[gv.qnflux-1] + 0.5*rfield.widflx[gv.qnflux-1];
		yhi = -(Ehi/fac+1.)*exp(-Ehi/fac);
		/* renormalize rate so that integral over phiTilde*anu gives correct total energy */
		rate /= yhi-ylo;

		/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
		for( i=0; i < gv.qnflux; i++ ) 
		{
			/* Ehi is kinetic energy of incoming/outgoing particle
			 * we assume that Ehi-E0 is deposited/extracted from grain */
			Ehi = rfield.anu[i] + 0.5*rfield.widflx[i];
			yhi = -(Ehi/fac+1.)*exp(-Ehi/fac);
			/* >>chng 01 mar 24, use MAX2 to protect against roundoff error, PvH */
			gv.bin[nd]->phiTilde[i] += rate*MAX2(yhi-ylo,0.);
			ylo = yhi;
		}
	}

#	if 0
	{
		long i;
		double integral = 0.;
		for( i=0; i < gv.qnflux; i++ )
		{
			integral += gv.bin[nd]->phiTilde[i]*gv.bin[nd]->cnv_H_pCM3*
				rfield.anu[i]*EN1RYD;
		}
		printf(" integral test 3: integral %.6e\n",integral);
	}
#	endif

#	ifdef DEBUG_FUN
	fputs( " <->GrainCollHeating()\n", debug_fp );
#	endif
	return;
}


/*GrainDrift compute grains drift velocity */
static void GrainDrift(void)
{
	long int i, 
	  loop, 
	  nd;

	double alam, 
	  corr, 
	  dmomen, 
	  fac, 
	  fdrag, 
	  g0, 
	  g2, 
	  phi2lm, 
	  psi, 
	  rdust, 
	  si, 
	  vdold, 
	  volmom;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainDrift()\n", debug_fp );
#	endif

	for( nd=0; nd < gv.nBin; nd++ )
	{
		/* find momentum absorbed by grain */
		dmomen = 0.;
		for( i=0; i < rfield.nflux; i++ )
		{
			dmomen += (rfield.flux[i]+rfield.ConInterOut[i]+rfield.outlin[i])*
			  rfield.anu[i]*(gv.bin[nd]->dstab1[i] + gv.bin[nd]->dstsc1[i]);
		}
		assert( dmomen >= 0. );
		dmomen *= EN1RYD*4./gv.bin[nd]->IntArea;

		/* now find force on grain, and drift velocity */
		fac = 2*BOLTZMANN*phycon.te;

		/* now PSI defined by 
		 * >>refer Draine and Salpeter 79 Ap.J. 231, 77 (1979) */
		psi = gv.bin[nd]->dstpot*TE1RYD/phycon.te;
		if( psi > 0. )
		{
			rdust = 1.e-6;
			alam = log(20.702/rdust/psi*phycon.sqrte/phycon.SqrtEden);
		}
		else
		{
			alam = 0.;
		}

		phi2lm = POW2(psi)*alam;
		corr = 2.;
		for( loop = 0; loop < 10 && fabs(corr-1.) > 0.03; loop++ )
		{
			vdold = gv.bin[nd]->DustDftVel;

			/* interactions with protons */
			si = gv.bin[nd]->DustDftVel/phycon.sqrte*7.755e-5;
			g0 = 1.5045*si*sqrt(1.+0.4418*si*si);
			g2 = si/(1.329 + POW3(si));

			/* drag force due to protons, both linear and square in velocity
			 * equation 4 from D+S Ap.J. 231, p77. */
			fdrag = fac*xIonFracs[ipHYDROGEN][2]*(g0 + phi2lm*g2);

			/* drag force due to interactions with electrons */
			si = gv.bin[nd]->DustDftVel/phycon.sqrte*1.816e-6;
			g0 = 1.5045*si*sqrt(1.+0.4418*si*si);
			g2 = si/(1.329 + POW3(si));
			fdrag += fac*phycon.eden*(g0 + phi2lm*g2);

			/* drag force due to collisions with hydrogen and helium atoms */
			si = gv.bin[nd]->DustDftVel/phycon.sqrte*7.755e-5;
			g0 = 1.5045*si*sqrt(1.+0.4418*si*si);
			fdrag += fac*(xIonFracs[ipHYDROGEN][1] + 1.1*xIonFracs[ipHELIUM][1])*g0;

			/* drag force due to interactions with helium ions */
			si = gv.bin[nd]->DustDftVel/phycon.sqrte*1.551e-4;
			g0 = 1.5045*si*sqrt(1.+0.4418*si*si);
			g2 = si/(1.329 + POW3(si));
			fdrag += fac*xIonFracs[ipHELIUM][2]*(g0 + phi2lm*g2);

			/* this term does not work
			 *  2      HEIII*(G0+4.*PSI**2*(ALAM-0.693)*G2) )
			 * this is total momentum absorbed by dust per unit vol */
			volmom = dmomen/SPEEDLIGHT;

			if( fdrag > 0. )
			{
				corr = sqrt(volmom/fdrag);
				gv.bin[nd]->DustDftVel = (float)(vdold*corr);
			}
			else
			{
				corr = 1.;
				negdrg.lgNegGrnDrg = TRUE;
			}

			if( trace.lgTrace && trace.lgDustBug )
			{
				fprintf( ioQQQ, "     %2ld new drift velocity:%10.2e momentum absorbed:%10.2e\n", 
				  loop, gv.bin[nd]->DustDftVel, volmom );
			}
		}
	}


#	ifdef DEBUG_FUN
	fputs( " <->GrainDrift()\n", debug_fp );
#	endif
	return;
}


/*GrnVryDpth set grains abundance as a function of depth into cloud*/
static double GrnVryDpth(
	/* pointer to grain type */
	long int nd)
{
	float GrnVryDpth_v;

#	ifdef DEBUG_FUN
	fputs( "<+>GrnVryDpth()\n", debug_fp );
#	endif

	/*set grains abundance as a function of depth into cloud
	 *NB most quantities are undefined for first calls to this sub */
	/* nd is the pointer to the grain species.  This routine must return
	 * a scale factor for the abundance at this position.
	 * */

	if( gv.bin[nd]->lgDustVary )
	{
		/* sample code - the scale factor will be the hydrogen ionization fraction
		 * following can be used for any other element - first dimesion
		 * is atomic weight and second is stage of ionization */
		GrnVryDpth_v = xIonFracs[ipHYDROGEN][1]/(xIonFracs[ipHYDROGEN][2] + 
		  xIonFracs[ipHYDROGEN][1]);
	}
	else
	{
		GrnVryDpth_v = 1.;
	}


#	ifdef DEBUG_FUN
	fputs( " <->GrnVryDpth()\n", debug_fp );
#	endif
	return( GrnVryDpth_v );
}
