/*Hydrogenic main routine to call HydroLevel and determine model hydrogen atom level balance */
#include "cddefines.h"
#include "taulines.h"
#include "hydrogenic.h"
#include "iso.h"
#include "trace.h"
#include "ionfracs.h"
#include "secondaries.h"
#include "phycon.h"
#include "hmi.h"
#include "chargtran.h"
#include "hcolrec.h"
#include "colion.h"
#include "abundances.h"
#include "ionrange.h"
#include "hcaseb.h"
#include "hmole.h"

void Hydrogenic(void)
{
	long int ipLo ,
		ipHi , 
		ipZ;
	double  coltot, 
	  gamtot;
	float hmovh1 = 0.; 

#	ifdef DEBUG_FUN
	fputs( "<+>Hydrogenic()\n", debug_fp );
#	endif

	for( ipZ=0; ipZ < LIMELM; ipZ++ )
	{
		/* do not consider elements that have been turned off */
		if( abundances.lgElmtOn[ipZ] )
		{
			/* note that ipZ scale is totally on c not physical scale, so 0 is h */
			/* evaluate hydrogenic balance if ionization reaches this high */
			if( (IonRange.IonHigh[ipZ] == ipZ + 1)  )
			{
				/* evaluate collisional rates */
				HydroCollid(ipZ);

				/* evaluate photoionization rates  */
				iso_Photo( ipHYDROGEN , ipZ );

				/* evaluate recombination rates */
				HydroRecom(ipZ);

				/* solve for the level populations */
				HydroLevel(ipZ);

				if( trace.lgTrace )
				{
					fprintf( ioQQQ, "     Hydrogenic Z=%2ld H2OVH1=",ipZ);
					fprintf( ioQQQ,PrintEfmt("%10.3e", iso.xIonRatio[ipHYDROGEN][ipZ]));
					fprintf( ioQQQ, " simple=");
					fprintf( ioQQQ,PrintEfmt("%10.3e", iso.xIonRatio[ipHYDROGEN][ipZ]));
					fprintf( ioQQQ, " H-ovH1:");
					fprintf( ioQQQ,PrintEfmt("%10.2e", hmovh1 ) );
					fprintf( ioQQQ, "\n"); 
				}

			}
			else
			{
				iso.xIonRatio[ipHYDROGEN][ipZ] = 0.;
				iso.xIonRatio[ipHYDROGEN][ipZ] = 0.;

				/* zero it out since no population*/
				for( ipHi=ipH2s; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
				{
					for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
					{
						/* population of lower level rel to ion */
						EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].PopLo = 0.;

						/* population of upper level rel to ion */
						EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].PopHi =  0.;

						/* population of lower level rel to ion, corrected for stim em */
						EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].PopOpc =  0.;
					}
				}
			}
		}
	}

	/* rest is for hydrogen only */

	/* do molecular balance 
	 * hmovh1 will be ratio of molecular to atomic hydrogen
	 * HIonFrac is fration of H that is ionized, ratio of ion to atom */
	hmole(&hmovh1,iso.xIonRatio[ipHYDROGEN][0]);

	/* >>chng 01 may 09, add option to force abundance, with element name ioniz cmmnd */
	if( abundances.lgSetIoniz[ipHYDROGEN] )
	{
		xIonFracs[ipHYDROGEN][2] = abundances.SetIoniz[ipHYDROGEN][1]*xIonFracs[ipHYDROGEN][0];
		xIonFracs[ipHYDROGEN][1] = abundances.SetIoniz[ipHYDROGEN][0]*xIonFracs[ipHYDROGEN][0];
	}
	else
	{
		/* get ionization balance */
		xIonFracs[ipHYDROGEN][2] = (float)(phycon.hden*iso.xIonRatio[ipHYDROGEN][0]/
			(1. + iso.xIonRatio[ipHYDROGEN][0] + hmovh1));

		xIonFracs[ipHYDROGEN][1] = (float)(phycon.hden/(1. + iso.xIonRatio[ipHYDROGEN][0] + hmovh1));
	}

	/* is hydrogen atom in condition of collisional rec? if so then
	 * only very small dT's can be tolerated in ConvTempEdenIoniz
	 * HRecEffec(ipZ) is sum of totcap(i)
	 * RadRec_effec is sum of radiative recombination */
	if( hcaseb.HRecCol[ipHYDROGEN]/iso.RadRec_effec[ipHYDROGEN][ipHYDROGEN] > 100. )
	{
		HColRec.lgHColRec = TRUE;
	}
	else
	{
		HColRec.lgHColRec = FALSE;
	}

	/*fprintf(ioQQQ,"ratio %.1e\n", iso.gamnc[ipHYDROGEN][ipHELIUM][ipH2s]/EmisLines[ipHYDROGEN][ipHELIUM][ipH2s][ipH1s].Aul);*/

	/* this is test whether collisions are important first define ratio of exits
	 * from ground due to coll, rel to everthing else, then flag is large */
	if( iso.PopLTE[ipHYDROGEN][0][ipH1s] > 1e-30 )
	{
		coltot = 
			EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].ColUL*
			iso.PopLTE[ipHYDROGEN][ipHYDROGEN][ipH2p]/iso.PopLTE[ipHYDROGEN][ipHYDROGEN][ipH1s]*
		  phycon.eden*(iso.gamnc[ipHYDROGEN][ipHYDROGEN][ipH2p] + iso.ColIoniz[ipHYDROGEN][ipHYDROGEN][ipH2p] + 
		  EmisLines[ipHYDROGEN][ipHYDROGEN][3][ipH2p].ColUL*iso.PopLTE[ipHYDROGEN][ipHYDROGEN][3]/iso.PopLTE[ipHYDROGEN][ipHYDROGEN][ipH2p])/
		  (EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].ColUL*phycon.eden + 
		  EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].Aul*
		  (EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].Pesc + EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].Pdest) );

		/* caution that collisions are important (and so only small changes in
		 * temperature should happen) if more than 20% of the total */
		if( coltot > 0.2 )
		{
			colion.lgHColionImp = TRUE;
		}
	}
	else
	{
		colion.lgHColionImp = FALSE;
	}

	/* remember the ratio of pops of 2p to 1s for possible printout in prtComment
	 * and to obtain Lya excitation temps.  the pop of ground is not defined if
	 * NO ionization at all since these pops are relative to ion */
	/* >>chng 99 jun 03, added MAX2 to protect against totally neutral gas */
	if( iso.Pop2Ion[ipHYDROGEN][0][ipH2p]/MAX2(SMALLFLOAT,iso.Pop2Ion[ipHYDROGEN][0][ipH1s]) > 0.1 )
	{
		hydro.lgHiPop2 = TRUE;
		hydro.pop2mx = (float)MAX2(iso.Pop2Ion[ipHYDROGEN][ipHYDROGEN][ipH2p]/iso.Pop2Ion[ipHYDROGEN][ipHYDROGEN][ipH1s],
		  hydro.pop2mx);
	}
	/*fprintf(ioQQQ,"hdebuggg %li %.2e %.2e\n" ,
		nzone,
		iso.Pop2Ion[ipHYDROGEN][0][ipH2p]/MAX2(SMALLFLOAT,iso.Pop2Ion[ipHYDROGEN][0][ipH1s]) , 
		iso.gamnc[ipHYDROGEN][0][ipH2p] );*/

	gamtot = iso.gamnc[ipHYDROGEN][ipHYDROGEN][ipH1s] + Secondaries.csupra;

	coltot = iso.ColIoniz[ipHYDROGEN][ipHYDROGEN][ipH1s] + 
		EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].ColUL*4.*
	  iso.Boltzmann[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s];

	/* if ground state destruction rate is significant, recall different dest procceses */
	if( hydro.DestRate[ipHYDROGEN][ipH1s] > 0. )
	{
		hydro.photn2 = 
			(float)(iso.gamnc[ipHYDROGEN][ipHYDROGEN][ipH1s]/hydro.DestRate[ipHYDROGEN][ipH1s] );

		/* fraction of ionizations of H from ground, due to thermal collisions */
		hydro.HCollIonFrac = 
			(float)(iso.ColIoniz[ipHYDROGEN][ipHYDROGEN][ipH1s]*phycon.eden/hydro.DestRate[ipHYDROGEN][ipH1s]);

		/* this flag is used in ConvIonizeOpacityDo to decide whether we
		 * really need to converge the secondary ionization rates */
		Secondaries.sec2total = 
			(float)(Secondaries.csupra / hydro.DestRate[ipHYDROGEN][ipH1s]) ;

		/* frac of ionizations due to ct */
		ChargTran.HIonFrac = ChargTran.HCharExcIonTotal / hydro.DestRate[ipHYDROGEN][ipH1s];
	}
	else
	{
		hydro.HCollIonFrac = 0.;
		hydro.photn2 = 0.;
		Secondaries.sec2total = 0. ;
		ChargTran.HIonFrac = 0.;
	}

	/* now do 21 cm, will define populations here */
	TauLines[ipH21cm].PopHi = xIonFracs[ipHYDROGEN][2]*iso.Pop2Ion[ipHYDROGEN][ipHYDROGEN][0]*0.75;	
	TauLines[ipH21cm].PopLo = xIonFracs[ipHYDROGEN][2]*iso.Pop2Ion[ipHYDROGEN][ipHYDROGEN][0]*0.25;
	/* assume spin temperature is gas kinetic temperature */
	TauLines[ipH21cm].PopOpc = xIonFracs[ipHYDROGEN][2]*iso.Pop2Ion[ipHYDROGEN][ipHYDROGEN][0] * 
		0.25f * TauLines[ipH21cm].EnergyK / phycon.te;

	/* now do 3 cm, the HeII equivalent of H 21 cm will define populations here */
	TauLines[ipHe3cm].PopHi = EmisLines[ipHYDROGEN][ipHELIUM][ipH2p][ipH1s].PopLo*0.75;	
	TauLines[ipHe3cm].PopLo = EmisLines[ipHYDROGEN][ipHELIUM][ipH2p][ipH1s].PopLo*0.25;
	/* assume spin temperature is gas kinetic temperature */
	TauLines[ipHe3cm].PopOpc = EmisLines[ipHYDROGEN][ipHELIUM][ipH2p][ipH1s].PopLo * 
		0.25f * TauLines[ipHe3cm].EnergyK / phycon.te;


	if( trace.lgTrace )
	{
		fprintf( ioQQQ, "       Hydrogenic retrn H,2,NE=");
		fprintf(ioQQQ,"%.2e ", xIonFracs[ipHYDROGEN][1]);
		fprintf(ioQQQ,"%.2e ", xIonFracs[ipHYDROGEN][2]);
		fprintf(ioQQQ,"%.2e ", phycon.eden);
		fprintf( ioQQQ, " REC, COL, GAMT= ");
		/* recomb rate coef, cm^3 s-1 */
		fprintf(ioQQQ,"%.2e ", iso.RadRec_effec[ipHYDROGEN][0] );
		fprintf(ioQQQ,"%.2e ", coltot);
		fprintf(ioQQQ,"%.2e ", gamtot);
		fprintf( ioQQQ, " CSUP=");
		PrintE82( ioQQQ, Secondaries.csupra);
		fprintf( ioQQQ, " Htwo=");
		fprintf(ioQQQ,"%.2e ", hmi.htwo);
		fprintf( ioQQQ, " Te=");
		fprintf(ioQQQ,"%.2e ", phycon.te );
		fprintf( ioQQQ, "\n");
	}

#	ifdef DEBUG_FUN
	fputs( " <->Hydrogenic()\n", debug_fp );
#	endif
	return;
}
