/*HydroLevel solve for ionization balance level populations of model hydrogen atom
 * calls HydroLevelPop or HydroT2Low */
#include "cddefines.h"
#include "taulines.h"
#include "recom.h"
#include "iso.h"
#include "secondaries.h"
#include "trace.h"
#include "elementnames.h"
#include "phycon.h"
#include "opacity.h"
#include "hcaseb.h"
#include "destcrt.h"
#include "ionrange.h"
#include "negcon.h"
#include "hydrogenic.h"

void HydroLevel(long int ipZ)
{
	char chType[6];
	long int i, 
	  ipHi, 
	  ipLo, 
	  level,
	  level_error;
	double BigError ,
	  bottom,
	  colfrc, 
	  HRecNet ,
	  phtfrc, 
	  Ratio_Ionization, 
	  secfrc;

	/* this block of variables will be obtained and freed here */
	long int *ipiv ; /* MALLOC out to [iso.nLevels[ipHYDROGEN][ipZ]+1] */
	double *totcap;/* MALLOC out to [iso.nLevels[ipHYDROGEN][ipZ]+1]*/
	double **SaveZ/*[iso.nLevels[ipHYDROGEN][ipZ]+2][iso.nLevels[ipHYDROGEN]+2]*/, 
	  *bvec/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
	  *error/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
	  *work/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
	  **z/*[iso.nLevels[ipHYDROGEN][ipZ]+2][iso.nLevels[ipHYDROGEN]+2]*/;

#	ifdef DEBUG_FUN
	fputs( "<+>HydroLevel()\n", debug_fp );
#	endif

	/* check that we were called with valid charge */
	assert( ipZ >= 0);
	assert( ipZ < LIMELM );

	/* 
	 * following electron density has approximate correction for neutrals
	 * corr of hi*1.7e-4 accounts for col ion by HI; Drawin Zs Phys 225, 483.
	 * used EdenHCorr instead
	 * edhi = eden + hi * 1.7e-4
	 */

	/* now MALLOC some scratch space */
	ipiv = (long int *)MALLOC(sizeof(long int)*(unsigned)(iso.nLevels[ipHYDROGEN][ipZ]) ) ;
	totcap = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHYDROGEN][ipZ]) ) ;
	bvec = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHYDROGEN][ipZ]+1) ) ;
	error = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHYDROGEN][ipZ]+1) ) ;
	work = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHYDROGEN][ipZ]+1) ) ;
	if( ipiv==NULL || totcap==NULL || bvec==NULL || error==NULL || work==NULL )
	{
		fprintf( ioQQQ, " could not MALLOC 1D arrays\n" );
		puts( "[Stop in HydroLevel]" );
		cdEXIT(1);
	}

	/* now do the 2D arrays */
	if( (SaveZ = (double **)MALLOC(sizeof(double *)*(unsigned)(iso.nLevels[ipHYDROGEN][ipZ]+1) ) ) ==NULL )
	{
		fprintf( ioQQQ, " could not MALLOC SaveZ array in 1D\n" );
		puts( "[Stop in HydroLevel]" );
		cdEXIT(1);
	}

	if(  (z = (double **)MALLOC(sizeof(double *)*(unsigned)(iso.nLevels[ipHYDROGEN][ipZ]+1) ))==NULL  )
	{
		fprintf( ioQQQ, " could not MALLOC z arrays in 1D\n" );
		puts( "[Stop in HydroLevel]" );
		cdEXIT(1);
	}

	/* now do the second dimension */
	for( i=0; i<(iso.nLevels[ipHYDROGEN][ipZ]+1); ++i )
	{
		SaveZ[i] = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHYDROGEN][ipZ]+1) ) ;
		if( SaveZ[i]==NULL )
		{
			fprintf( ioQQQ, " could not MALLOC 2D SaveZ\n" );
			puts( "[Stop in HydroLevel]" );
			cdEXIT(1);
		}

		z[i] = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHYDROGEN][ipZ]+1) ) ;
		if( z[i]==NULL )
		{
			fprintf( ioQQQ, " could not MALLOC 2D z\n" );
			puts( "[Stop in HydroLevel]" );
			cdEXIT(1);
		}
	}

	/* ============================================================================== */
	/* option to print some rates */
	if( (trace.lgTrace && trace.lgHBugFull) && (ipZ == trace.ipZTrace) )
	{
		fprintf( ioQQQ, 
			"       HydroLevel%3ld finds arrays, with optical depths defined? %1c induced 2ph=%12.3e\n", 
		  ipZ, TorF(opac.lgTauOutOn), EmisLines[ipHYDROGEN][ipZ][ipH2s][ipH1s].pump );
		for( ipHi=ipH2s; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
		{
			fprintf( ioQQQ, "up:%2ld", ipHi );
			fprintf( ioQQQ, "lo" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ, "%9ld", ipLo );
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " A*esc" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].Aul*
				  EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].Pesc ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " tauin" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].TauIn ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " t tot" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e", EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].TauTot ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " Esc  " );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e", EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].Pesc ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " Dest " );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].Pdest) );
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " A*dst" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].Aul*
				  EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].Pdest ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " StrkE" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  hydro.pestrk[ipLo][ipHi] ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " EsctE" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  hydro.esesc[ipZ][ipLo][ipHi] ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " B(ul)" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e", EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].pump*
				  iso.stat[ipHYDROGEN][ipLo]/iso.stat[ipHYDROGEN][ipHi] ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " tcont" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].TauCon ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " C(ul)" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e", 
					EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].ColUL*phycon.eden ));
			}
			fprintf( ioQQQ, "\n" );

			if( ipHi == 2 )
			{
				fprintf( ioQQQ, "    FeIIo");
				fprintf( ioQQQ,PrintEfmt("%9.2e", 
					hydro.dstfe2lya* EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].Aul ));
				fprintf( ioQQQ, "\n");
			}
		}

		fprintf( ioQQQ, "         " );
		for( i=1; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf( ioQQQ, "%9ld", i );
		}
		fprintf( ioQQQ, "\n" );
	}
	/* ============================================================================== */

	/* remember creation and destruction rates */
	DestCrt.xIonize[ipZ][ipZ] = iso.gamnc[ipHYDROGEN][ipZ][ipH1s] + 
	  Secondaries.csupra + iso.ColIoniz[ipHYDROGEN][ipZ][ipH1s]*phycon.EdenHCorr;

	DestCrt.Recombine[ipZ][ipZ] = phycon.eden*iso.RadRec_effec[ipHYDROGEN][ipZ];

	/* check how low temperature is; departure coef tend to infinity
	 * as TE goes to zero; matrix formalism won't work on a 32-bit machine
	 * if EXP(H NU/KT) > 1E38 for 13.6EV.
	 * the recombination coefficient is effect radiative (only) rec
	 * coef (optical depths in) */

	/* get simple estimate of ionization balance, first get denominator,
	 * which can be zero during search phase */
	bottom = phycon.eden*(iso.RadRec_effec[ipHYDROGEN][ipZ] + 
	  recom.CotaRate[ipZ]) ;

	if( bottom > 0. )
	{
		iso.xIonRatio[ipHYDROGEN][ipZ] = (iso.gamnc[ipHYDROGEN][ipZ][ipH1s] + 
		  Secondaries.csupra + iso.ColIoniz[ipHYDROGEN][ipZ][ipH1s]*phycon.EdenHCorr )/bottom;
	}
	else
	{
		iso.xIonRatio[ipHYDROGEN][ipZ] = 0.;
	}

	if( trace.lgHBug && trace.lgTrace )
	{
		/* identify how atom is ionized for full trace */
		if( iso.xIonRatio[ipHYDROGEN][ipZ] > 0. )
		{
			/* fraction of ionization due to photoionization */
			phtfrc = iso.gamnc[ipHYDROGEN][ipZ][ipH1s]/((phycon.eden*(iso.RadRec_effec[ipHYDROGEN][ipZ] + 
			  recom.CotaRate[ipZ]) )*
			  iso.xIonRatio[ipHYDROGEN][ipZ]);

			/* fraction of ionization due to collisional ionization */
			colfrc = (iso.ColIoniz[ipHYDROGEN][ipZ][ipH1s]*phycon.EdenHCorr )/
				((phycon.eden*(iso.RadRec_effec[ipHYDROGEN][ipZ] + 
			  recom.CotaRate[0]) )*
			  iso.xIonRatio[ipHYDROGEN][ipZ]);

			/* fraction of ionization due to secondary ionization */
			secfrc = Secondaries.csupra/((phycon.eden*(iso.RadRec_effec[ipHYDROGEN][ipZ] + 
			  recom.CotaRate[0]) )*
			  iso.xIonRatio[ipHYDROGEN][ipZ]);
		}
		else
		{
			phtfrc = 0.;
			colfrc = 0.;
			secfrc = 0.;
		}

		fprintf( ioQQQ, "     HydroLevel Z=%2ld called, simple II/I=",ipZ);
		PrintE93( ioQQQ, iso.xIonRatio[ipHYDROGEN][ipZ]);
		fprintf( ioQQQ," PhotFrc:");
		PrintE93( ioQQQ,phtfrc);
		fprintf(ioQQQ," ColFrc:");
		PrintE93( ioQQQ,colfrc);
		fprintf( ioQQQ," SecFrc");
		PrintE93( ioQQQ, secfrc);
		fprintf( ioQQQ,"  Te:");
		PrintE93( ioQQQ,phycon.te);
		fprintf( ioQQQ," eden:");
		PrintE93( ioQQQ,phycon.eden);
		fprintf( ioQQQ,"\n"); 
	}

	/* for dense models close to lte all ionizations can be from
	 * excited states, and simple and actual pops are very different.
	 * code used simple all the time, caused catastrophe for Kingdon nova model.
	 * now use actual pops if we are into model */
	if( nzone > 2 )
	{
		/* HIonFrac saves ion to neutral, used in bidiag to set hydrogenic ratio */
		Ratio_Ionization = iso.xIonRatio[ipHYDROGEN][ipZ];
	}
	else
	{
		Ratio_Ionization = iso.xIonRatio[ipHYDROGEN][ipZ];
	}

	/* end of prelimaries - start model atom */

	/* which case atom to solve??? */
	if( iso.xIonRatio[ipHYDROGEN][ipZ] < 1e-30 )
	{
		/* don't bother if no ionizing radiation */
		strcpy( chType, "zero " );
		if( trace.lgHBug && trace.lgTrace )
		{
			fprintf( ioQQQ, "     HydroLevel Z=%2ld simple II/I=%10.2e so not doing equilibrium.\n", 
			  ipZ, iso.xIonRatio[ipHYDROGEN][ipZ] );
		}

		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			iso.DepartCoef[ipHYDROGEN][ipZ][i] = 1.;
			iso.Pop2Ion[ipHYDROGEN][ipZ][i] = 0.;
			hydro.DestRate[ipZ][i] = 0.;
			error[i] = 0.;
		}

		hcaseb.HRecEffec[ipZ] = 0.;
		iso.xIonRatio[ipHYDROGEN][ipZ] = 0.;
		iso.xIonRatio[ipHYDROGEN][ipZ] = 0.;

	}
	/* >>chng 99 nov 23, very dense model close to lte had very low simple
	 * ionization fraction but moderate ionization since all pops in
	 * excited states - added test for density 
	 * second change - for all cases use this routine if ionization is
	 * very low indeed */
	/* >>chng 00 aug 26, added matrix force option,
	 * logic is to override default options, "none was set in zero */
	/* NB - this test is mostly bogus since chTypeAtom is "POPU" in zero */
	else if( (strcmp( hydro.chTypeAtom , "LOWT" )==0) ||
		( (strcmp( hydro.chTypeAtom , "none" )==0) &&
		( (iso.xIonRatio[ipHYDROGEN][ipZ] < 1e-15) ||
		(iso.xIonRatio[ipHYDROGEN][ipZ] < 1e-7 && phycon.hden < 1e6 ) ) ) )
	{
		strcpy( chType, "TLow " );
		/* this avoids departure coefficients, 
		 * by doing simple cascade, should always work, 
		 * but not very well at high densities */
		HydroT2Low(ipZ,Ratio_Ionization);

		/* error not set in HydroT2Low so set to zero here */
		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			error[i] = 0.;
		}
	}
	/* >>chng 99 jun 08 Ratio_Ionization from 1e-7 to 1e-5 due to orion nebula crash using
	 * costar stellar atmosphere, in He+ which was barely present */

	 /* >>chng 00 aug 26, added matrix force option,
	 * logic is to override default options, "none was set in zero */
	/* >>chng 01 may 09, remove all logic - always dumps to HydroLevelPop if gotten this far */
#	if 0
	if( (strcmp( hydro.chTypeAtom , "POPU" )==0) ||
		( (strcmp( hydro.chTypeAtom , "none" )==0) &&
		((phycon.te <= thlo.HydTempLimit* (ipZ+1)) || 
		(Ratio_Ionization < 1e-5) || 
	   !iso.lgPopLTE_OK[ipHYDROGEN][ipZ] ) ) )
#	endif
	else
	{
		/* this is the branch that uses the matrix to get level populations */
		strcpy( chType, "Popul" );
		HydroLevelPop(ipZ ,
		  SaveZ/*[iso.nLevels[ipHYDROGEN][ipZ]+2][iso.nLevels[ipHYDROGEN]+2]*/, 
		  bvec/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
		  error/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
		  work/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
		  z/*[iso.nLevels[ipHYDROGEN][ipZ]+2][iso.nLevels[ipHYDROGEN]+2]*/ ,
		  ipiv , /* MALLOC out to [iso.nLevels[ipHYDROGEN][ipZ]+1] */
		  totcap/* MALLOC out to [iso.nLevels[ipHYDROGEN][ipZ]+1]*/);
	}
	/* >>chng 01 may 09, totally remove HydroLevelDep */
#	if 0
	else
	{
		/* departure coefficients, the main solver for most conditions */
		strcpy( chType, "Depar" );
		/* this is the high temperature branch, with dep coef */
		HydroLevelDep(ipZ ,
		  SaveZ/*[iso.nLevels[ipHYDROGEN][ipZ]+2][iso.nLevels[ipHYDROGEN]+2]*/, 
		  bvec/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
		  error/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
		  work/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
		  z/*[iso.nLevels[ipHYDROGEN][ipZ]+2][iso.nLevels[ipHYDROGEN]+2]*/ ,
		  ipiv , /* MALLOC out to [iso.nLevels[ipHYDROGEN][ipZ]+1] */
		  totcap/* MALLOC out to [iso.nLevels[ipHYDROGEN][ipZ]+1]*/);
	}
#	endif
	/* end branch for main solve */
	/*fprintf(ioQQQ,"%li %s\n", nzone,chType);*/

	/* remember largest residual in matrix inversion */
	BigError = 0.;
	level_error = -1;
	/* all three cases end up down here */
	for( ipLo=ipH1s; ipLo < (iso.nLevels[ipHYDROGEN][ipZ] - 1); ipLo++ )
	{
		double abserror;
		abserror = fabs( error[ipLo]) ;
		/* this will be the largest residual in the matrix inversion */
		if( abserror > BigError )
		{
			BigError = abserror ;
			level_error = ipLo;
		}

		for( ipHi=ipLo + 1; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
		{
			/* population of lower level rel to ion */
			EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].PopLo = 
				iso.Pop2Ion[ipHYDROGEN][ipZ][ipLo];

			/* population of upper level rel to ion */
			EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].PopHi = 
				iso.Pop2Ion[ipHYDROGEN][ipZ][ipHi];

			/* population of lower level rel to ion, corrected for stim em */
			EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].PopOpc = 
				(iso.Pop2Ion[ipHYDROGEN][ipZ][ipLo] - 
				EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].PopHi*
				iso.stat[ipHYDROGEN][ipLo]/iso.stat[ipHYDROGEN][ipHi] );
		}
	}
	/* >>chng 00 dec 18, do not let lines going to highest level mase */
	/* higest level can have artifical maser */
	/* >>chng 01 feb 24, this had been in the wrong place in hydrogenic.c, having not effect.
	 * moved here */
	for( ipLo=ipH1s; ipLo < iso.nLevels[ipHYDROGEN][ipZ]-1; ipLo++ )
	{
		/* population of lower level rel to ion, corrected for stim em */
		EmisLines[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]-1][ipLo].PopOpc =  
			MAX2(SMALLFLOAT,EmisLines[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]-1][ipLo].PopOpc);
	}
	/* >>chng 01 feb 24, 2s and 2p are degerneate, so make PopOpc reflect total population. 
	 * this had major effect on radiation pressure due to n=3 to 2p transition, since 
	 * pop of 2p alone is small due to rapid escape route, so radiation pressure was large */
	for( ipHi=ipH2p+1; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
	{
		/* sum of populations in 2s and 2p */
		double sum = EmisLines[ipHYDROGEN][ipZ][ipHi][ipH2s].PopOpc + 
			EmisLines[ipHYDROGEN][ipZ][ipHi][ipH2p].PopOpc;

		/* population of lower level rel to ion, corrected for stim em */
		EmisLines[ipHYDROGEN][ipZ][ipHi][ipH2s].PopOpc =  sum;
		EmisLines[ipHYDROGEN][ipZ][ipHi][ipH2p].PopOpc =  sum;
	}

	/* >>chng 00 dec 17, nova.in had intense maser when H atom made very small,
	 * cap major of highest line */
	/* >>chng 01 feb 24, remove following since not needed, with above */
	/*EmisLines[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]][iso.nLevels[ipHYDROGEN][ipZ]-1].PopOpc = MAX2(SMALLFLOAT ,
		EmisLines[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]][iso.nLevels[ipHYDROGEN][ipZ]-1].PopOpc);*/

	/* matrix inversion should be nearly as good as the accuracy of a double,
	 * but demand that it is better than epsilon for a float */
	if( BigError > FLT_EPSILON ) 
	{
		fprintf(ioQQQ,
			"HydroLevel: warning zone %li - largest residual in hydrogenic %s ipZ=%li matrix inversion is %g "
			"matrix type was %s, level was %li \n", 
			nzone,
			elementnames.chElementName[ipZ],
			ipZ , 
			BigError , 
			chType , 
			level_error);
		ShowMe();
		puts( "[Stop in HydroLevel]" );
		cdEXIT(1);
	}

	/* now get real effective rec coef */
	HRecNet = 0.;
	hcaseb.HRecCol[ipZ] = 0.;

	/* this loop is mainly to establish HRecNet, which is used in a debug print, and
	 * to establish HRecCol to determine whether collision recombination is important
	 * this is only possible after departure coef are known */
	for( i=ipH2s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
	{
		/* induced recombination */
		double HRecInd = iso.RecomInducRate[ipHYDROGEN][ipZ][i]*iso.PopLTE[ipHYDROGEN][ipZ][i];

		/* three body, collisional, recombination */
		hcaseb.HRecCol[ipZ] += (float)(iso.ColIoniz[ipHYDROGEN][ipZ][i]*phycon.EdenHCorr*
		  iso.PopLTE[ipHYDROGEN][ipZ][i]*(1. - iso.DepartCoef[ipHYDROGEN][ipZ][i]));

		/* net total recombination */
		HRecNet += (float)(iso.RadRecomb[ipHYDROGEN][ipZ][i][ipRecRad]*
		  iso.RadRecomb[ipHYDROGEN][ipZ][i][ipRecNetEsc] + 
		  HRecInd + 
		  iso.ColIoniz[ipHYDROGEN][ipZ][i]*phycon.EdenHCorr*iso.PopLTE[ipHYDROGEN][ipZ][i]*
		  (1. - iso.DepartCoef[ipHYDROGEN][ipZ][i]));
	}

	/* this will eventually become the ratio of ionized to neutral hydrogenic 
	 * species, create sum of level pops per ion first */
	iso.xIonRatio[ipHYDROGEN][ipZ] = 0.;
	for( level=ipH1s; level < iso.nLevels[ipHYDROGEN][ipZ]; level++ )
	{
		iso.xIonRatio[ipHYDROGEN][ipZ] += iso.Pop2Ion[ipHYDROGEN][ipZ][level];
	}

	if( iso.xIonRatio[ipHYDROGEN][ipZ] < 0. )
	{
		fprintf( ioQQQ, 
			" HydroLevel finds negative H-like ion fraction for ipZ=%2ld %s using routine %s, val= %10.3e simple=%10.3e TE=%10.3e ZONE=%4ld\n", 
		  ipZ, 
		  elementnames.chElementSym[ipZ],
		  chType , 
		  iso.xIonRatio[ipHYDROGEN][ipZ], 
		  iso.xIonRatio[ipHYDROGEN][ipZ], 
		  phycon.te, 
		  nzone );
		fprintf( ioQQQ, " level pop are:" );
		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf( ioQQQ,PrintEfmt("%8.1e", iso.Pop2Ion[ipHYDROGEN][ipZ][i] ));
		}
		fprintf( ioQQQ, "\n" );
		negcon();
		ShowMe();
		puts( "[Stop in HydroLevel]" );
		cdEXIT(1);
	}

	/* get level populations, two cases, 
	 * first, may be zero since all cases drop down to here, 
	 * this branch of if is for trivial abundance, so zero out species*/
	else if( iso.xIonRatio[ipHYDROGEN][ipZ] >= 0. && iso.xIonRatio[ipHYDROGEN][ipZ] < 1e-30 )
	{
		iso.xIonRatio[ipHYDROGEN][ipZ] = 0.;

		/* reset pointer to one lower stage of ionization so this not
		 * considered again, hydrogenic considered if IonHigh is ipZ+2 */
		IonRange.IonHigh[ipZ] = ipZ;

		/* now zero this out */
		for( ipLo=ipH1s; ipLo < (iso.nLevels[ipHYDROGEN][ipZ] - 1); ipLo++ )
		{
			for( ipHi=ipLo + 1; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
			{
				/* population of lower level rel to ion */
				EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].PopLo = 0.;

				/* population of upper level rel to ion */
				EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].PopHi =  0.;

				/* population of lower level rel to ion, corrected for stim em */
				EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].PopOpc =  0.;

				/* local ots destruction, cm^-3 s^-1 */
				EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].ots =  0.;
			}
		}
	}
	/* this is the main branch that applies when we have non-trivial abundance */
	else
	{
		/* level populations 
		 * this brance we have significant population as sum of levels per ion,
		 * store inverse, ion per atom, here.  For non-H species this is ratio
		 * of highest to next highest stage of ionization */
		iso.xIonRatio[ipHYDROGEN][ipZ] = 1./iso.xIonRatio[ipHYDROGEN][ipZ];
	}

	if( (trace.lgHBugFull && trace.lgTrace) && (ipZ == trace.ipZTrace) )
	{
		fprintf( ioQQQ, "       HLEV HGAMNC" );
		PrintE93( ioQQQ, iso.gamnc[ipHYDROGEN][ipZ][ipH1s] );
		for( i=ipH2s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", iso.gamnc[ipHYDROGEN][ipZ][i] ));
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "       HLEV TOTCAP" );
		for( i=1; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", totcap[i] ));
		}
		fprintf( ioQQQ,PrintEfmt("%10.2e", HRecNet ) );
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "       HLEV IND Rc" );
		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", iso.RecomInducRate[ipHYDROGEN][ipZ][i]*iso.PopLTE[ipHYDROGEN][ipZ][i] ));
		}
		fprintf( ioQQQ, "\n" );

		/* incuded recombination rate coefficients */
		fprintf( ioQQQ, "       IND Rc LTE " );
		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e",
				iso.gamnc[ipHYDROGEN][ipZ][i]*iso.PopLTE[ipHYDROGEN][ipZ][i] ));
		}
		fprintf( ioQQQ, "\n" );

		/* LTE level populations */
		fprintf( ioQQQ, "       HLEV   HLTE" );
		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", iso.PopLTE[ipHYDROGEN][ipZ][i] ));
		}
		fprintf( ioQQQ, "\n" );

		/* fraction of total ionization due to collisions from given level */
		fprintf( ioQQQ, "       HLEVfr cion" );
		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", 
				iso.ColIoniz[ipHYDROGEN][ipZ][i]*
			  iso.Pop2Ion[ipHYDROGEN][ipZ][i]*phycon.EdenHCorr/MAX2(1e-30,hydro.DestRate[ipZ][i]) ) );
		}
		fprintf( ioQQQ, "\n" );

		/* fraction of total ionization due to photoionization from given level */
		if( hcaseb.HRecEffec[ipZ]> 0. )
		{
			fprintf( ioQQQ, "       HLEVfrPhIon" );
			for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf(ioQQQ,PrintEfmt("%9.2e", 
					iso.gamnc[ipHYDROGEN][ipZ][i]*iso.Pop2Ion[ipHYDROGEN][0][i]/MAX2(1e-30,hydro.DestRate[ipZ][i]) ) );
			}
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, "       HLEV     HN" );
		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", iso.Pop2Ion[ipHYDROGEN][ipZ][i] ));
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "       HLEV   b(n)" );
		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", iso.DepartCoef[ipHYDROGEN][ipZ][i] ));
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "       HLEV   X12tot");
		fprintf(ioQQQ,PrintEfmt("%9.2e", Secondaries.x12tot ));
		fprintf( ioQQQ," Grn dest:");
		fprintf(ioQQQ,PrintEfmt("%9.2e",
		  DestCrt.xIonize[ipZ][ipZ] ));
		fprintf(ioQQQ, "\n"); 
	}

	/* check for non-positive level populations */
	if( iso.xIonRatio[ipHYDROGEN][ipZ] > 0. )
	{
		for( level=ipH1s; level < iso.nLevels[ipHYDROGEN][ipZ]; level++ )
		{
			if( iso.Pop2Ion[ipHYDROGEN][ipZ][level] <= 0. )
			{
				fprintf( ioQQQ, 
					" HydroLevel finds negative hydrogen level population for %s ipZ=%ld level %ld value=%10.3e simple=%10.3e TE=%10.3e ZONE=%4ld\n", 
				  elementnames.chElementName[ipZ], ipZ, level, iso.Pop2Ion[ipHYDROGEN][ipZ][level], 
				  iso.xIonRatio[ipHYDROGEN][ipZ], 
				  phycon.te, nzone );
				fprintf( ioQQQ, " level pop are:" );
				for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
				{
					fprintf( ioQQQ,PrintEfmt("%8.1e", iso.Pop2Ion[ipHYDROGEN][ipZ][i] ));
				}
				fprintf( ioQQQ, "\n" );
				negcon();
				ShowMe();
				puts( "[Stop in HydroLevel]" );
				cdEXIT(1);
			}
		}
	}
	else if( iso.xIonRatio[ipHYDROGEN][ipZ] > 0 && 
		iso.xIonRatio[ipHYDROGEN][ipZ] <= 0.)
	{
		/* this is case where we expected ionization, but found none, or negative */
		fprintf( ioQQQ, 
			" HydroLevel finds non-positive hydrogen ion frac for ipZ=%ld (%s) value=%.3e simple= %.3e TE=%.3e ZONE=%ld\n", 
		  ipZ, 
		  elementnames.chElementSym[ipZ],
		  iso.Pop2Ion[ipHYDROGEN][ipZ][level], 
		  iso.xIonRatio[ipHYDROGEN][ipZ], 
		  phycon.te, nzone );
		fprintf( ioQQQ, " level pop are:" );
		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			fprintf( ioQQQ,PrintEfmt("%8.1e", iso.Pop2Ion[ipHYDROGEN][ipZ][i] ));
		}
		fprintf( ioQQQ, "\n" );
		negcon();
		ShowMe();
		puts( "[Stop in HydroLevel]" );
		cdEXIT(1);
	}

	if( trace.lgTrace )
	{
		/* HRecEffec(ipZ) is gross rec coef, computed here while filling in matrix
		 * elements, all physical processes included. 
		 * RadRec_effec is total effective radiative only */
		fprintf( ioQQQ, "       HydroLevel%3ld retrn %s te=",
			ipZ, 
			chType );
		PrintE93( ioQQQ,phycon.te);
		fprintf( ioQQQ," HII/HI=");
		PrintE93( ioQQQ,iso.xIonRatio[ipHYDROGEN][ipZ]);

		fprintf( ioQQQ," simple=");
		PrintE93( ioQQQ,iso.xIonRatio[ipHYDROGEN][ipZ]);

		fprintf( ioQQQ," b1=");
		PrintE82( ioQQQ,iso.DepartCoef[ipHYDROGEN][ipZ][ipH1s]);

		fprintf( ioQQQ," ion rate=");
		PrintE82( ioQQQ,iso.gamnc[ipHYDROGEN][ipZ][ipH1s] + Secondaries.csupra);

		fprintf( ioQQQ," TotRec");
		PrintE82( ioQQQ,hcaseb.HRecEffec[ipZ]);

		fprintf( ioQQQ," RadRec");
		PrintE82( ioQQQ,iso.RadRec_effec[ipHYDROGEN][ipZ]);
		fprintf( ioQQQ, "\n");
	}

	/* update destruction rate */
	DestCrt.xIonize[ipZ][ipZ] = iso.gamnc[ipHYDROGEN][ipZ][ipH1s] + 
	  Secondaries.csupra + iso.ColIoniz[ipHYDROGEN][ipZ][ipH1s]*phycon.EdenHCorr;
	DestCrt.Recombine[ipZ][ipZ] = phycon.eden*iso.RadRec_effec[ipHYDROGEN][ipZ];

	/* now free up the arrays */
	free( ipiv );
	free( totcap);
	free( bvec );
	free( error );
	free( work );

	/* now free up the 2D arrays */
	for( i=0; i<iso.nLevels[ipHYDROGEN][ipZ]+1; ++i )
	{
		free( SaveZ[i] ) ;
		free( z[i] ) ;
	}
	free( SaveZ ) ;
	free( z ) ;

#	ifdef DEBUG_FUN
	fputs( " <->HydroLevel()\n", debug_fp );
#	endif
	return;
}

