/*MeanInc increment mean ionization fractions and temperatures over computed structure, in RadInc */
/*MeanZero zero mean of ionization fractions array */
/*RadMean do radius mean ionization fractions or temperature over radius for any element */
/*VolMean do volume mean ionization fractions or temperature over volume for any element */
#include "cddefines.h"
#include "ionfracs.h"
#include "filfac.h"
#include "radius.h"
#include "nomole.h"
#include "hmi.h"
#include "phycon.h"
#include "mean.h"

void MeanInc(void)
{
	long int i, 
	  nelem;
	double dc, 
	  dcv, 
	  sumrad, 
	  sumvol,
	  sumEdenrad, 
	  sumEdenvol;

	/* 
	 * MeanZero is called at start of calculation by zero, and by
	 * startenditer to initialize the variables 
	 */

	/*
	 * this routine is called by radinc during the calculation to update the 
	 * sums that will become the vol and rad weighted mean ionizations 
	 */

#	ifdef DEBUG_FUN
	fputs( "<+>MeanInc()\n", debug_fp );
#	endif

	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		/* this is mean ionization */
		sumrad = 0.;
		sumvol = 0.;
		sumEdenrad = 0.;
		sumEdenvol = 0.;
		/* this is ionization on the IonFracs scale, offset 1 up since
		 * zero is total abundance.  This works well since the mean
		 * ionization array xIonMeans is also offset up one, since 
		 * [0] is the normalization */
		for( i=1; i <= (nelem + 2); i++ )
		{
			/* xIonMeans[0][i][n] is radial integration */
			dc = xIonFracs[nelem][i]*radius.drad*filfac.FillFac;
			IonMeans.xIonMeans[0][nelem][i] += dc;
			sumrad += dc;

			/* xIonEdenMeans[0][i][n] is radial integration PLUS electron density*/
			dc = xIonFracs[nelem][i]*radius.drad*filfac.FillFac*phycon.eden;
			IonMeans.xIonEdenMeans[0][nelem][i] += dc;
			sumEdenrad += dc;

			/* xIonMeans[1][n][i] is vol integration */
			/* >>chng 00 mar 28, r1r0sq had been dVeff,
			 * bug discovered by Valentina Luridiana */
			dcv = xIonFracs[nelem][i]*radius.drad*filfac.FillFac*radius.r1r0sq;
			IonMeans.xIonMeans[1][nelem][i] += dcv;
			sumvol += dcv;

			/* xIonMeans[1][i][n] is vol integration PLUS electron density */
			/* >>chng 00 mar 28, r1r0sq had been dVeff,
			 * bug discovered by Valentina Luridiana */
			dcv = xIonFracs[nelem][i]*radius.drad*filfac.FillFac*radius.r1r0sq*phycon.eden;
			IonMeans.xIonEdenMeans[1][nelem][i] += dcv;
			sumEdenvol += dcv;
		}
		IonMeans.xIonMeans[0][nelem][0] += sumrad;
		IonMeans.xIonMeans[1][nelem][0] += sumvol;
		IonMeans.xIonEdenMeans[0][nelem][0] += sumEdenrad;
		IonMeans.xIonEdenMeans[1][nelem][0] += sumEdenvol;

		/* this is mean temperature */
		sumrad = 0.;
		sumvol = 0.;
		for( i=1; i <= (nelem + 2); i++ )
		{
			/* TempMeans[0][i][n] is radial integration */
			dc = xIonFracs[nelem][i]*radius.drad*filfac.FillFac;
			IonMeans.TempMeans[0][nelem][i] += dc*phycon.te;
			IonMeans.TempMeans[0+2][nelem][i] += dc;

			/* TempMeans[0][i][n] is radial integration PLUS electron density*/
			dc = xIonFracs[nelem][i]*radius.drad*filfac.FillFac*phycon.eden;
			IonMeans.TempEdenMeans[0][nelem][i] += dc*phycon.te;
			IonMeans.TempEdenMeans[0+2][nelem][i] += dc;

			/* TempMeans[1][i+1][n] is vol integration */
			/*>>chng 00 dec 18, following had dVeff rather than r1r0sq */
			dcv = xIonFracs[nelem][i]*radius.drad*filfac.FillFac*radius.r1r0sq;
			IonMeans.TempMeans[1][nelem][i] += dcv*phycon.te;
			IonMeans.TempMeans[1+2][nelem][i] += dcv;

			/* TempMeans[1][i+1][n] is vol integration PLUS electron density */
			dcv = xIonFracs[nelem][i]*radius.drad*filfac.FillFac*radius.r1r0sq*phycon.eden;
			IonMeans.TempEdenMeans[1][nelem][i] += dcv*phycon.te;
			IonMeans.TempEdenMeans[1+2][nelem][i] += dcv;
		}
	}

	/* =============================================================
	 *
	 * these are some special quanties for the mean 
	 */
	
	/* used to get harmonic mean temperature with respect to H over radius,
	 * for comparison with 21cm temperature */
	/* >>chng 99 jul 7, no h2 option, include h2 as it were two H's */
	if( nomole.lgNoH2Mole )
	{
		dc = (xIonFracs[ipHYDROGEN][1]+2.*hmi.htwo)*radius.drad*filfac.FillFac;
	}
	else
	{
		dc = xIonFracs[ipHYDROGEN][1]*radius.drad*filfac.FillFac;
	}

	/* mean over radius, for comp with 21 cm obs */
	IonMeans.HarMeanTempRadius[0] += dc/phycon.te;
	IonMeans.HarMeanTempRadius[1] += dc;

	/* mean over volume, for symmetry */
	IonMeans.HarMeanTempVolume[0] += dc/phycon.te*radius.r1r0sq;
	IonMeans.HarMeanTempVolume[1] += dc*radius.r1r0sq;

	/* mean over radius, for comp with 21 cm obs */
	dc = hmi.htwo*radius.drad*filfac.FillFac;
	IonMeans.H2MeanTempRadius[0] += dc*phycon.te;
	IonMeans.H2MeanTempRadius[1] += dc;

	/* mean over volume, for symmetry */
	dcv = hmi.htwo*radius.drad*filfac.FillFac*radius.r1r0sq;
	IonMeans.H2MeanTempVolume[0] += dc*phycon.te;
	IonMeans.H2MeanTempVolume[1] += dc;

#	ifdef DEBUG_FUN
	fputs( " <->MeanInc()\n", debug_fp );
#	endif
	return;
}

/*MeanZero zero mean of ionization fractions array */
void MeanZero(void)
{
	long int i, 
	  j, 
	  nelem;

#	ifdef DEBUG_FUN
	fputs( "<+>MeanZero()\n", debug_fp );
#	endif

	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		for( i=0; i <= (nelem + 2); i++ )
		{
			for( j=0; j < 2; j++ )
			{
				/* these are used to save info on temperature and ionization means */
				IonMeans.xIonMeans[j][nelem][i] = 0.;
				IonMeans.xIonEdenMeans[j][nelem][i] = 0.;
				/* double here since [2] and [3] have norm for average */
				IonMeans.TempMeans[j][nelem][i] = 0.;
				IonMeans.TempMeans[j+2][nelem][i] = 0.;
				IonMeans.TempEdenMeans[j][nelem][i] = 0.;
				IonMeans.TempEdenMeans[j+2][nelem][i] = 0.;
			}
		}
	}

	/* mean over radius, for comp with 21 cm obs */
	IonMeans.HarMeanTempRadius[0] = 0.;
	IonMeans.HarMeanTempRadius[1] = 0.;
	/* mean over volume, for symmetry */
	IonMeans.HarMeanTempVolume[0] = 0.;
	IonMeans.HarMeanTempVolume[1] = 0.;

	/* H2 mean temp over radius */
	IonMeans.H2MeanTempRadius[0] = 0.;
	IonMeans.H2MeanTempRadius[1] = 0.;
	/* H2 mean temp over volume */
	IonMeans.H2MeanTempVolume[0] = 0.;
	IonMeans.H2MeanTempVolume[1] = 0.;

#	ifdef DEBUG_FUN
	fputs( " <->MeanZero()\n", debug_fp );
#	endif
	return;
}

/*RadMean derive mean ionization fractions over radius for some element */
void RadMean(
	/* either 'i' or 't' for ionization or temperature */
	char chType,
	/* atomic number on physical, no c, scale */
	long int nelem, 
	/* this will say how many of arlog have non-zero values */
	long int *n, 
	/* array of values, log both cases */
	float arlog[],
	/* TRUE, include electron density, FALSE do not*/
	int lgDensity )
{
	long int i;
	double abund, 
	  normalize;

#	ifdef DEBUG_FUN
	fputs( "<+>RadMean()\n", debug_fp );
#	endif

	assert( chType=='i' || chType=='t' );

	/* fills in array arlog with log of ionization fractions
	 * N is set to highest stage with non-zero abundance
	 * N set to 0 if element turned off
	 *
	 * first call MeanZero to sero out save arrays
	 * next call MeanInc in zone calc to enter ionziation fractions or temperature
	 * finally this routine computes actual means
	 * */
	if( IonMeans.xIonMeans[0][nelem-1][0] <= 0. )
	{
		/* no ionization for this element */
		for( i=0; i < (nelem + 1); i++ )
		{
			arlog[i] = -30.f;
		}
		*n = 0;
		
#		ifdef DEBUG_FUN
		fputs( " <->RadMean()\n", debug_fp );
#		endif
		return;
	}

	*n = nelem + 1;
	while( *n > 1 && IonMeans.xIonMeans[0][nelem-1][*n] <= 0. )
	{
		arlog[*n-1] = -30.f;
		*n -= 1;
	}

	if( chType=='i' && lgDensity)
	{
		/* mean ionization with density*/
		normalize = IonMeans.xIonEdenMeans[0][nelem-1][0];
		for( i=0; i < *n; i++ )
		{
			if( IonMeans.xIonEdenMeans[0][nelem-1][i+1] > 0. )
			{
				abund = IonMeans.xIonEdenMeans[0][nelem-1][i+1];
				arlog[i] = (float)log10(MAX2(1e-30,abund/normalize));
			}
			else
			{
				arlog[i] = -30.f;
			}
		}
	}
	else if( chType=='i' )
	{
		/* mean ionization no density*/
		normalize = IonMeans.xIonMeans[0][nelem-1][0];
		for( i=0; i < *n; i++ )
		{
			if( IonMeans.xIonMeans[0][nelem-1][i+1] > 0. )
			{
				abund = IonMeans.xIonMeans[0][nelem-1][i+1];
				arlog[i] = (float)log10(MAX2(1e-30,abund/normalize));
			}
			else
			{
				arlog[i] = -30.f;
			}
		}
	}
	else if( chType=='t' && lgDensity )
	{
		/* mean temperature */
		for( i=0; i < *n; i++ )
		{
			normalize = IonMeans.TempEdenMeans[0+2][nelem-1][i+1];
			if( normalize > SMALLFLOAT )
			{
				abund = IonMeans.TempEdenMeans[0][nelem-1][i+1];
				arlog[i] = (float)log10(MAX2(1e-30,abund/normalize));
			}
			else
			{
				arlog[i] = -30.f;
			}
		}
	}
	else if( chType=='t'  )
	{
		/* mean temperature no  density*/
		for( i=0; i < *n; i++ )
		{
			normalize = IonMeans.TempMeans[0+2][nelem-1][i+1];
			if( normalize > SMALLFLOAT )
			{
				abund = IonMeans.TempMeans[0][nelem-1][i+1];
				arlog[i] = (float)log10(MAX2(1e-30,abund/normalize));
			}
			else
			{
				arlog[i] = -30.f;
			}
		}
	}
	else
	{
		fprintf(ioQQQ," RadMean called with insane job \n");
	}

#	ifdef DEBUG_FUN
	fputs( " <->RadMean()\n", debug_fp );
#	endif
	return;
}

/*VolMean do volume mean of ionization fractions over volumn of any element */
void VolMean(
	/* either 'i' or 't' for ionization or temperature */
	char chType,
	/* atomic number on physical, no c, scale */
	long int nelem, 
	/* this will say how many of arlog have non-zero values */
	long int *n, 
	/* array of values, log both cases */
	float arlog[],
	/* TRUE, include electron density, FALSE do not*/
	int lgDensity )
{
	long int i;
	double abund, 
	  normalize;

#	ifdef DEBUG_FUN
	fputs( "<+>VolMean()\n", debug_fp );
#	endif

	assert( chType=='i' || chType=='t' );

	/* 
	 * fills in array arlog with log of ionization fractions
	 * N is set to highest stage with non-zero abundance
	 * N set to 0 if element turned off
	 *
	 * first call MeanZero to sero out save arrays
	 * next call MeanInc in zone calc to enter ionziation fractions
	 * finally this routine computes actual means
	 * 
	 */
	if( IonMeans.xIonMeans[1][nelem-1][0] <= 0. )
	{
		/* no ionization for this element */
		for( i=0; i < (nelem + 1); i++ )
		{
			arlog[i] = -30.f;
		}
		*n = 0;
		
#		ifdef DEBUG_FUN
		fputs( " <->VolMean()\n", debug_fp );
#		endif
		return;
	}

	/* this is number of stages of ionization */
	*n = nelem + 1;
	/* fill in higher stages with zero if non-existent, 
	 * also decrement n, the number with non-zero abundances */
	while( *n > 1 && IonMeans.xIonMeans[1][nelem-1][*n] <= 0. )
	{
		arlog[*n-1] = -30.f;
		*n -= 1;
	}
	/* n is now the limit to the number with positive abundances */

	if( chType=='i' && lgDensity)
	{
		/* mean ionization with electron density*/
		/* this is denominator for forming a mean */
		normalize = IonMeans.xIonEdenMeans[1][nelem-1][0];
		/* return log of means */
		for( i=0; i < *n; i++ )
		{
			if( IonMeans.xIonEdenMeans[1][nelem-1][i+1] > 0. )
			{
				abund = IonMeans.xIonEdenMeans[1][nelem-1][i+1];
				arlog[i] = (float)log10(MAX2(1e-30,abund)/normalize);
			}
			else
			{
				arlog[i] = -30.f;
			}
		}
	}
	else if( chType=='i' )
	{
		/* mean ionization */
		/* this is denominator for forming a mean */
		normalize = IonMeans.xIonMeans[1][nelem-1][0];
		/* return log of means */
		for( i=0; i < *n; i++ )
		{
			if( IonMeans.xIonMeans[1][nelem-1][i+1] > 0. )
			{
				abund = IonMeans.xIonMeans[1][nelem-1][i+1];
				arlog[i] = (float)log10(MAX2(1e-30,abund)/normalize);
			}
			else
			{
				arlog[i] = -30.f;
			}
		}
	}
	else if( chType=='t' && lgDensity )
	{
		/* mean temperature with density*/
		/* this is denominator for forming a mean */
		/* return log of means */
		for( i=0; i < *n; i++ )
		{
			normalize = IonMeans.TempEdenMeans[1+2][nelem-1][i+1];
			if( normalize > SMALLFLOAT )
			{
				abund = IonMeans.TempEdenMeans[1][nelem-1][i+1];
				arlog[i] = (float)log10(MAX2(1e-30,abund)/normalize);
			}
			else
			{
				arlog[i] = -30.f;
			}
		}
	}
	else if( chType=='t' )
	{
		/* mean temperature with NO density*/
		/* this is denominator for forming a mean */
		/* return log of means */
		for( i=0; i < *n; i++ )
		{
			normalize = IonMeans.TempMeans[1+2][nelem-1][i+1];
			if( normalize > SMALLFLOAT )
			{
				abund = IonMeans.TempMeans[1][nelem-1][i+1];
				arlog[i] = (float)log10(MAX2(1e-30,abund)/normalize);
			}
			else
			{
				arlog[i] = -30.f;
			}
		}
	}
	else
	{
		fprintf(ioQQQ," VolMean called with insane job\n");
	}

#	ifdef DEBUG_FUN
	fputs( " <->VolMean()\n", debug_fp );
#	endif
	return;
}

