/*PunchDo produce punch output during calculation,
 * chTime is 'MIDL' during calculation, 'LAST' at the end */
/*PunchNewContinuum produce the 'punch new continuum' output */
/*PunchLineStuff punch optical depths or source functions for all transferred lines */
/*pun1Line called by PunchLineStuff to produce output for one line */
/*PunchNewContinuum produce the 'punch new continuum' output */
/*PunLineIntensity produce the 'punch lines intensity' output */
/* punch h emiss, for chapt 4, routine is below */
#include "cddefines.h"
#include "cddrive.h"
#include "physconst.h"
#include "taulines.h"
#include "iso.h"
#include "rt.h"
#include "hydrogenic.h"
#include "nhe1lvl.h"
#include "secondaries.h"
#include "he3lines.h"
#include "prtcolumns.h"
#include "grainvar.h"
#include "linesave.h"
#include "he1rb.h"
#include "makerecomb.h"
#include "mappar.h"
#include "dynamics.h"
#include "colden.h"
#include "photrate.h"
#include "yield.h"
#include "prt.h"
#include "itercnt.h"
#include "hevmolec.h"
#include "heavy.h"
#include "converge.h"
#include "sphere.h"
#include "called.h"
#include "helike.h"
#include "abundances.h"
#include "tfidle.h"
#include "opacity.h"
#include "rfield.h"
#include "phycon.h"
#include "timesc.h"
#include "radius.h"
#include "pop371.h"
#include "assertresults.h"
#include "he.h"
#include "heat.h"
#include "cooling.h"
#include "phe1lv.h"
#include "heopfr.h"
#include "wind.h"
#include "hmi.h"
#include "ipvfil.h"
#include "punchskip.h"
#include "heatcool.h"
#include "hcaseb.h"
#include "ionfracs.h"
#include "pressure.h"
#include "he1rate.h"
#include "herec.h"
#include "nhe1.h"
#include "ph1com.h"
#include "elementnames.h"
#include "plwidth.h"
#include "ipoint.h"
#include "pfeii.h"
#include "puncool.h"
#include "pgaunt.h"
#include "gammas.h"
#include "punspec.h"
#include "chargtran.h"
#include "punt.h"
#include "prtmeanion.h"
#include "plankf.h"
#include "readar.h"
#include "rdinit.h"
#include "punch.h"

static void PunchLineStuff(FILE * io,char *chJob , float xLimit);

/* punch h emiss, for chapt 4, routine is below */
static void AGN_Hemis(FILE *io );

/*PunchNewContinuum produce the 'punch new continuum' output */
static void PunchNewContinuum(FILE * io , long ipCon );

/*PunLineIntensity produce the 'punch lines intensity' output */
static void PunLineIntensity(FILE * io);

void PunchDo(
	char *chTime) /* chTime is null terminated 4 char string, either "MIDL" or "LAST" */
{
	char chEner[LIMELM][7];
	int lgOK, 
	  lgTlkSav;
	FILE * isav;
	long int
	  i, 
	  i1, 
	  ion, 
	  ipConMax, 
	  ipLinMax, 
	  ips, 
	  j, 
	  jj, 
	  limit, 
	  n, 
	  nd, 
	  nelem;
	double ConMax, 
	  RateInter, 
	  av, 
	  conem, 
	  damp, 
	  eps, 
	  flxatt, 
	  flxcor, 
	  flxin, 
	  flxref, 
	  flxtrn, 
	  fout, 
	  fref, 
	  fsum, 
	  opConSum, 
	  opLinSum, 
	  pop, 
	  stage, 
	  sum, 
	  texc, 
	  xLinMax;

#	ifdef DEBUG_FUN
	fputs( "<+>PunchDo()\n", debug_fp );
#	endif

	/* 
	 * the "last" option on punch command, to punch on last iteration,
	 * is parsed at the top of the loop in only one place.  
	 * no further action is needed at all for punch last to work
	 * ok throughout this routine 
	 */

	/* 
	 * each branch can have a test whether chTime is or is not "LAST"
	 * if( strcmp(chTime,"LAST") == 0 )
	 * if "LAST" then this is last call to routine after calc complete
	 * punch only if "LAST" when results at end of cal are needed
	 * test for .not."LAST" is for every zone result, where you do not
	 * want to punch last zone twice
	 */

	/* punch results on unit ipPnunit  */
	if( punch.npunch < 1 )
	{ 
#		ifdef DEBUG_FUN
		fputs( " <->PunchDo()\n", debug_fp );
#		endif
		return;
	}

	for( i=0; i < punch.npunch; i++ )
	{
		/* this global variable to remember where in the punch stack we are */
		punch.ipConPun = i;
		/* 
		 * lgPunLstIter set true if 'last' key occured on punch command
		 * normally is false.  This will skip punching if last set and
		 * this is not last iteration
		 */

		if( IterCnt.lgLastIt || (!punch.lgPunLstIter[i]) )
		{

			if( strcmp(punch.chPunch[i],"ABUN") == 0 )
			{
				/* punch abundances vs depth */
				if( strcmp(chTime,"LAST") != 0 )
				{
					for( j=0; j < LIMELM; j++ )
					{
						fprintf( punch.ipPnunit[i], "%6.2f", 
						  log10(xIonFracs[j][0]) );
					}
					fprintf( punch.ipPnunit[i], "\n" );
				}
			}

			else if( strcmp(punch.chPunch[i],"ADVE") == 0 )
			{
				/* punch advection terms */
				if( strcmp(chTime,"LAST") != 0 && dynamics.lgAdvection )
				{
					if( nzone>0 )
					{
						fprintf( punch.ipPnunit[i], "%.5e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\n",
							radius.depth,
							heat.htot , 
							dynamics.CoolHeat , 
							dynamics.dCoolHeatdT ,
							dynamics.Recomb[ipHYDROGEN][ipHYDROGEN],
							dynamics.Photo,
							dynamics.DynTimestep
							);
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"AGES") == 0 )
			{
				/* punch timescales vs depth */
				if( strcmp(chTime,"LAST") != 0 )
				{
					fprintf( punch.ipPnunit[i], "%10.2e%10.2e%10.2e%10.2e%10.2e\n", 
					  /* depth, cm */
					  radius.depth,
					  /* cooling timescale */
					  phycon.pden*BOLTZMANN*1.5*phycon.te/ heat.htot, 
					  /* H2 formation timescale */
					  timesc.AgeH2MoleDest, 
					  /* CO formation timescale */
					  timesc.AgeCOMoleDest, 
					  /* H recombination timescale */
					  1./(phycon.eden*2.90e-10/(phycon.te70*phycon.te10/phycon.te03)) );
				}
			}

			else if( strcmp(punch.chPunch[i]," AGN") == 0 )
			{
				if( strcmp(chTime,"LAST") == 0 )
				{
					if( strcmp( punch.chPunchArgs[i], "HECS" ) == 0 )
					{
						/* this routine is in helike.c */
						AGN_He1_CS(punch.ipPnunit[i]);
					}
					if( strcmp( punch.chPunchArgs[i], "HEMI" ) == 0 )
					{
						/* punch h emiss, for chapt 4, routine is below */
						AGN_Hemis(punch.ipPnunit[i]);
					}
					else
					{
						fprintf( ioQQQ, " PunchDo does not recognize flag %4.4s set for AGN punch.  This is impossible.\n", 
						  punch.chPunch[i] );
						ShowMe();
						puts( "[Stop in PunchDo]" );
						cdEXIT(1);
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"ASSE") == 0 )
			{
				if( strcmp(chTime,"LAST") == 0 )
				{
					/* punch the assert output */
					/*lint -e534 ignore return value of following  - nothing to do with it */
					lgCheckAsserts(punch.ipPnunit[i]);
					/*lint +e534 ignore return value of following  - nothing to do with it */
				}
			}

			else if( strncmp(punch.chPunch[i],"CHA",3) == 0 )
			{
				if( strcmp(chTime,"LAST") == 0 )
				{
					/* one of the charge transfer options, all in chargtran.c */
					ChargTranPun( punch.ipPnunit[i] , punch.chPunch[i] );
				}
			}

			else if( strcmp(punch.chPunch[i],"COOL") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
					PunCool(punch.ipPnunit[i]);
			}

			else if( strcmp(punch.chPunch[i],"COLU") == 0 )
			{
				/* punch column densities */
				if( strcmp(chTime,"LAST") == 0 )
					PrtColumns(punch.ipPnunit[i]);
			}

			else if( strcmp(punch.chPunch[i],"COMP") == 0 )
			{
				/* compton energy exchange coefficients */
				if( strcmp(chTime,"LAST") != 0 )
				{
					for( jj=0; jj<rfield.nflux; jj = jj + PunchSkip.ncPunchSkip)
					{
						fprintf( punch.ipPnunit[i], "%10.2e%10.2e%10.2e\n", 
						  rfield.anu[jj], rfield.comup[jj]/rfield.widflx[jj], 
						  rfield.comdn[jj]/rfield.widflx[jj] );
					}
				}
			}

			/* punch continuum commands */
			else if( strcmp(punch.chPunch[i],"CON ") == 0 )
			{
				/* this branch is the usual "punch continuum" case */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */
				if( strcmp(chTime,"LAST") == 0 )
				{
					for( j=0; j<rfield.nflux ; j = j+PunchSkip.ncPunchSkip)
					{
						/* four continua predicted here;
						 * incident, attenuated incident, emitted,
						 * then attenuated incident + emitted, last reflected
						 * reflected continuum is stored relative to inner radius
						 * others are stored for this radius */

						/* NB this code also used in punch emitted,
						 * transmitted continuum commands */

						/* the incident continuum */
						flxin = rfield.FluxSave[j]*rfield.anu2[j]*
						  EN1RYD/rfield.widflx[j];

						/* the reflected continuum */
						flxref = (rfield.anu2[j]*(rfield.ConRefIncid[j]+rfield.ConEmitReflec[j])/rfield.widflx[j] +
							rfield.anu[j]*PLWidth.PunchLWidth*rfield.reflin[j])*EN1RYD;

						/* the attenuated incident continuum */
						flxatt = rfield.flux[j]*rfield.anu2[j]*EN1RYD/
						  rfield.widflx[j]*radius.r1r0sq;

						/* the outward emitted continuum */
						conem = ((rfield.ConEmitOut[j])/
						  rfield.widflx[j]*rfield.anu2[j] + PLWidth.PunchLWidth*
						  rfield.outlin[j]*rfield.anu[j])*radius.r1r0sq*
						  EN1RYD*sphere.covgeo;

						/* sum of emitted and transmitted continua */
						flxtrn = conem + flxatt;

						/* photon energy in appropriate energy or wavelength units */
						fprintf( punch.ipPnunit[i],"%.3e\t", AnuUnit(rfield.AnuOrg[j]) );
						/* indicent continuum */
						fprintf( punch.ipPnunit[i],"%.3e\t", flxin ); 
						/* trans cont */
						fprintf( punch.ipPnunit[i],"%.3e\t", flxatt ); 
						/* DiffOut cont */
						fprintf( punch.ipPnunit[i],"%.3e\t", conem ); 
						/* net trans cont */
						fprintf( punch.ipPnunit[i],"%.3e\t", flxtrn ); 
						/* reflc cont */
						fprintf( punch.ipPnunit[i],"%.3e\t", flxref ); 
						/* total cont */
						fprintf( punch.ipPnunit[i],"%.3e\t", flxref + flxtrn ); 
						fprintf( punch.ipPnunit[i], "%4.4s\t%4.4s\n", 
						/* line	label */
						  rfield.chLineLabel[j] ,
						/* cont label*/
						  rfield.chContLabel[j] );
					}
				}
			}

			/* this is the punch spectrum command, 
			 * the new continuum command that will replace the previous one */
			else if( strcmp(punch.chPunch[i],"CONN") == 0 )
			{
				if( strcmp(chTime,"LAST") == 0 )
					/* io unit and which new continuum this is (was set when punch read in */
					PunchNewContinuum( punch.ipPnunit[i] , (long)punch.punarg[0][i]);
			}

			else if( strcmp(punch.chPunch[i],"CONC") == 0 )
			{
				/* punch incident continuum */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */
				if( strcmp(chTime,"LAST") == 0 )
				{
					/* incident continuum */
					for( j=0; j<rfield.nflux; j = j + PunchSkip.ncPunchSkip)
					{
						flxin = rfield.FluxSave[j]*rfield.anu2[j]*
						  EN1RYD/rfield.widflx[j];
						/* >>chng 96 oct 22, format of anu to 11.5 to resolve grid near 1 ryd */
						fprintf( punch.ipPnunit[i], "%.5e\t%.3e\n", 
						  AnuUnit(rfield.AnuOrg[j]), flxin );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"CONG") == 0 )
			{
				/* punch emitted grain continuum */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */
				if( strcmp(chTime,"LAST") == 0 )
				{
					/* tried to remember emission from the two grain types in rtdiffuse */
					for( j=0; j<rfield.nflux; j = j + PunchSkip.ncPunchSkip)
					{
						fsum = gv.GraphiteEmission[j]*rfield.anu2[j]*
						  EN1RYD/rfield.widflx[j] *radius.r1r0sq*sphere.covgeo;

						fout = gv.SilicateEmission[j]*rfield.anu2[j]*
						  EN1RYD/rfield.widflx[j] *radius.r1r0sq*sphere.covgeo;

						/* >>chng 96 oct 22, format of anu to 11.5 to resolve grid near 1 ryd */
						fprintf( punch.ipPnunit[i], "%12.4e %10.3e %10.3e\n", 
						  AnuUnit(rfield.AnuOrg[j]) , fsum , fout );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"CONR") == 0 )
			{
				/* punch reflected continuum */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */
				if( strcmp(chTime,"LAST") == 0 )
				{
					if( sphere.lgSphere )
					{
						fprintf( punch.ipPnunit[i], " Reflected continuum not predicted when SPHERE is set.\n" );
						fprintf( ioQQQ , 
							"\n\n>>>>>>>>>>>>>\n Reflected continuum not predicted when SPHERE is set.\n" );
						puts( "[Stop in PunchDo]" );
						cdEXIT(1);
					}

					for( j=0; j<rfield.nflux; j = j + PunchSkip.ncPunchSkip)
					{
						/* reflected continuum */
						flxref = rfield.anu2[j]*(rfield.ConRefIncid[j]+rfield.ConEmitReflec[j])/
						  rfield.widflx[j]*EN1RYD;
						/* reflected lines */
						fref = rfield.anu[j]*PLWidth.PunchLWidth*
						  rfield.reflin[j]*EN1RYD;
						/* ratio of reflected to incident continuum, the albedo */
						if( rfield.FluxSave[j] > 1e-25 )
						{
							av = rfield.ConRefIncid[j]/rfield.FluxSave[j];
						}
						else
						{
							av = 0.;
						}
						/* >>chng 96 oct 22, format of anu to 12.5 to resolve grid near 1 ryd */
						fprintf( punch.ipPnunit[i], "%12.5e%12.4e%12.4e%12.4e%12.4e %4.4s\n", 
						  AnuUnit(rfield.AnuOrg[j]), flxref, fref, flxref + fref, 
						  av, rfield.chContLabel[j] );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"CNVE") == 0 )
			{
				/* the punch convergence error command */
				if( strcmp(chTime,"LAST") != 0 )
				{
					fprintf( punch.ipPnunit[i], 
						"%.4e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
						radius.depth, 
						pressure.PressureCorrect, 
						pressure.PressureCurrent, 
						(pressure.PressureCorrect - pressure.PressureCurrent)*100./pressure.PressureCorrect, 
						phycon.EdenTrue,
						phycon.eden,
						(phycon.EdenTrue - phycon.eden)*100./phycon.EdenTrue,
						heat.htot,
						cooling.ctot,
						(heat.htot - cooling.ctot)*100./heat.htot );
				}
			}

			else if( strcmp(punch.chPunch[i],"CONB") == 0 )
			{
				/* punch continuum bins binning */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */
				if( strcmp(chTime,"LAST") != 0 )
				{
					for( j=0; j<rfield.nupper; j = j + PunchSkip.ncPunchSkip)
					{
						fprintf( punch.ipPnunit[i], "%14.5e %14.5e %14.5e\n", 
						  AnuUnit(rfield.AnuOrg[j]), rfield.anu[j], rfield.widflx[j] );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"COND") == 0 )
			{
				/* punch diffuse continuum the local thermal emission */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */
				if( strcmp(chTime,"LAST") == 0 )
				{
					/* this option to punch diffuse continuum */
					for( j=0; j<rfield.nflux; j = j+PunchSkip.ncPunchSkip)
					{
						/* >>chng 96 oct 22, format of anu to 12.5 to resolve grid near 1 ryd */
						fprintf( punch.ipPnunit[i], "%12.5e\t%12.5e\n", 
						  AnuUnit(rfield.AnuOrg[j]), 
						  rfield.ConEmitLocal[j]*rfield.anu2[j]*EN1RYD/rfield.widflx[j]);
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"CONE") == 0 )
			{
				/* punch emitted continuum */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */
				if( strcmp(chTime,"LAST") == 0 )
				{
					/* punch emitted continuum */
					for( j=0; j<rfield.nflux;j = j +PunchSkip.ncPunchSkip)
					{
						/* this is the reflected component */
						flxref = (rfield.anu2[j]*(rfield.ConRefIncid[j]+rfield.ConEmitReflec[j])/
						  rfield.widflx[j] + rfield.anu[j]*PLWidth.PunchLWidth*
						  rfield.reflin[j])*EN1RYD;

						/* this is the total emission in the outward direction */
						conem = (rfield.ConEmitOut[j])/rfield.widflx[j]*rfield.anu2[j] + 
							PLWidth.PunchLWidth*rfield.outlin[j]*rfield.anu[j];

						conem *= radius.r1r0sq*EN1RYD*sphere.covgeo;

						/* output: photon energy, reflected, outward, total emission
						 *  >>chng 96 oct 22, format of anu to 11.5 to resolve grid near 1 ryd */
						fprintf( punch.ipPnunit[i], "%.5e\t%.3e\t%.3e\t%.3e\t%4.4s\t%4.4s\n", 
						  AnuUnit(rfield.AnuOrg[j]), 
						  flxref, 
						  conem, 
						  flxref + conem, 
						  rfield.chLineLabel[j], 
						  rfield.chContLabel[j]
						   );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"CONF") == 0 )
			{
				/* FeII continuuum, check that FeII is turned on */
				if( FeII.lgFeIION && ( strcmp(chTime,"LAST") == 0 ))
				{
					for( j=0; j < nFeIIConBins; j++ )
					{
						/* [x][0] is center wavelength, [1] and [2] are upper and
						 * lower bounds in Angstroms.  
						 * these are set in FeIIZero */
						/* emission from large FeII atom, integrated over band */
						fprintf( punch.ipPnunit[i], "%.2f\t%e \n", 
						(FeII_Cont[j][1]+FeII_Cont[j][2])/2.,
						FeII_Cont[j][0] );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"CONi") == 0 )
			{
				/* punch continuum interactions */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */

				/* continuum interactions */
				if( strcmp(chTime,"LAST") != 0 )
				{
					/* this is option to set lowest energy */
					if( punch.punarg[0][i] <= 0. )
					{
						i1 = 1;
					}
					else if( punch.punarg[0][i] < 100. )
					{
						i1 = ipoint(punch.punarg[0][i]);
					}
					else
					{
						i1 = (long int)punch.punarg[0][i];
					}

					fref = 0.;
					fout = 0.;
					fsum = 0.;
					sum = 0.;
					flxin = 0.;

					for( j=i1-1; j < rfield.nflux; j++ )
					{
						fref += rfield.flux[j]*opac.opac[j];
						fout += rfield.otslin[j]*opac.opac[j];
						fsum += rfield.otscon[j]*opac.opac[j];
						sum += rfield.ConInterOut[j]*opac.opac[j];
						flxin += rfield.outlin[j]*opac.opac[j];
					}
					fprintf( punch.ipPnunit[i], "%10.2e%10.2e%10.2e%10.2e%10.2e\n", 
					  fref, fout, fsum, sum, flxin );
				}
			}

			else if( strcmp(punch.chPunch[i],"CONI") == 0 )
			{
				/* punch ionizing continuum */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */

				if( strcmp(chTime,"LAST") == 0 )
				{
					/* this flag will remember whether we have ever printed anything */
					int lgPrt=FALSE;

					/* punch ionizing continuum command
					 * this is option to set lowest energy */
					if( punch.punarg[0][i] <= 0. )
					{
						i1 = 1;
					}
					else if( punch.punarg[0][i] < 100. )
					{
						i1 = ipoint(punch.punarg[0][i]);
					}
					else
					{
						i1 = (long int)punch.punarg[0][i];
					}

					sum = 0.;
					for( j=i1-1; j < rfield.nflux; j++ )
					{
						flxcor = rfield.flux[j] + 
						  rfield.otslin[j] + 
						  rfield.otscon[j] + 
						  rfield.ConInterOut[j] +
						  rfield.outlin[j];

						sum += flxcor*opac.opac[j];
					}

					if( sum > 0. )
					{
						sum = 1./sum;
					}
					else
					{
						sum = 1.;
					}

					fsum = 0.;

					for( j=i1-1; j<rfield.nflux ; ++j)
					{
						flxcor = rfield.flux[j] + 
						  rfield.otslin[j] + 
						  rfield.otscon[j] + 
						  rfield.ConInterOut[j]+
						  rfield.outlin[j];

						fsum += flxcor*opac.opac[j];

						/* punched quantities are freq, flux, flux*cross sec,
						 * fraction of total, integral fraction of total */
						RateInter = flxcor*opac.opac[j]*sum;

						/* punage(i,2) is lowest interaction rate to consider, def=0.01 (1 percnt) */
						/* >>chng 01 nov 22, format to c-friendly */
						if( (RateInter >= punch.punarg[1][i]) && (flxcor > SMALLFLOAT) )
						{
							lgPrt = TRUE;
							/* >>chng 96 oct 22, format of anu to 11.5 to resolve grid near 1 ryd */
							fprintf( punch.ipPnunit[i], 
								"%li\t%.5e\t%.2e\t%.2e\t%.2f\t%.2f\t%.2f\t%.2f\t%.2e\t%.2e\t%.4s\t%.4s\n", 
							  j,
							  AnuUnit(rfield.AnuOrg[j]), 
							  flxcor, 
							  flxcor*opac.opac[j], 
							  rfield.flux[j]/flxcor, 
							  rfield.otslin[j]/flxcor, 
							  rfield.otscon[j]/flxcor, 
							  (rfield.ConInterOut[j]+ rfield.outlin[j])/flxcor, 
							  RateInter, 
							  fsum*sum, 
							  rfield.chLineLabel[j], 
							  rfield.chContLabel[j] );
						}
					}
					if( !lgPrt )
					{
						/* entered logical block but did not print anything */
						fprintf(punch.ipPnunit[i],
							" punchdo, the PUNCH IONIZING CONTINUUM command did not find a strong point, sum fsum were %.2e %.2e\n",sum,fsum);
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"CORA") == 0 )
			{
				/* punch raw continuum */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */

				if( strcmp(chTime,"LAST") == 0 )
				{
					/* this option to punch all raw ionizing continuum */
					for( j=0;j<rfield.nflux;j = j + PunchSkip.ncPunchSkip)
					{
						fprintf( punch.ipPnunit[i], 
							"%.5e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%4.4s\t%4.4s\n", 
						  AnuUnit(rfield.AnuOrg[j]), 
						  rfield.flux[j], 
						  rfield.otslin[j], 
						  rfield.otscon[j], 
						  rfield.ConRefIncid[j],
						  rfield.ConEmitReflec[j], 
						  rfield.ConInterOut[j],
						  rfield.outlin[j], 
						  rfield.ThrowOut[j] ,
						  rfield.ConEmitOut[j] ,
						  rfield.chLineLabel[j], 
						  rfield.chContLabel[j] );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"CONo") == 0 )
			{
				/* punch outward local continuum */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */

				if( strcmp(chTime,"LAST") == 0 )
				{
					/* option to punch out outward continuum here */
					for( j=0;j<rfield.nflux; j = j + PunchSkip.ncPunchSkip)
					{
						fprintf( punch.ipPnunit[i], "%11.5e%10.2e%10.2e%10.2e%10.2e\n", 
						  AnuUnit(rfield.AnuOrg[j]), 
						  rfield.SavOutCon[j], 
						  rfield.ConEmitOut[j] + rfield.outlin[j], 
						  rfield.SavOutCon[j]* opac.opac[j]*rfield.anu[j], 
						  (rfield.flux[j] + rfield.otscon[j] + rfield.otslin[j] + 
						  rfield.ConInterOut[j])*opac.opac[j]*
						  rfield.anu[j] );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"CONO") == 0 )
			{
				/* punch outward continuum */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */

				if( strcmp(chTime,"LAST") == 0 )
				{
					/* option to punch out outward continuum */
					for( j=0; j<rfield.nflux; j = j + PunchSkip.ncPunchSkip)
					{
						fprintf( punch.ipPnunit[i], "%11.5e%10.2e%10.2e%10.2e%10.2e\n", 
						  AnuUnit(rfield.AnuOrg[j]), 
						  rfield.flux[j]*rfield.anu2[j]* EN1RYD/rfield.widflx[j]*radius.r1r0sq, 
						  rfield.ConInterOut[j]/rfield.widflx[j]*rfield.anu2[j]*radius.r1r0sq*EN1RYD, 
						  PLWidth.PunchLWidth* rfield.outlin[j]*rfield.anu[j]*radius.r1r0sq*EN1RYD, 
						  (rfield.ConInterOut[j]/rfield.widflx[j]*
						  rfield.anu2[j] + PLWidth.PunchLWidth*rfield.outlin[j]*
						  rfield.anu[j])*radius.r1r0sq*EN1RYD );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"CONT") == 0 )
			{
				/* punch transmitted continuum */
				/* set pointer for possible change in units of energy in continuum
				 * AnuUnit will give anu in whatever units were set with punch units */

				if( strcmp(chTime,"LAST") == 0 )
				{
					/* this option to punch transmitted continuum */
					for( j=0;j<rfield.nflux; j = j + PunchSkip.ncPunchSkip)
					{
						/* attenuated incident continuum
						 * >>chng 97 jul 10, remove PunchLWidth from this one only since
						 * we must conserve energy even in lines */
						flxatt = rfield.flux[j]*rfield.anu2[j]*EN1RYD/
						  rfield.widflx[j]*radius.r1r0sq;

						/*conem = (rfield.ConOutNoInter[j] + rfield.ConInterOut[j]+rfield.outlin[j])*
						  rfield.anu2[j] ;
						conem *= radius.r1r0sq*EN1RYD*sphere.covgeo;*/
						/* >>chng 00 jan 03, above did not include all contributors.  
						 * Pasted in below from usual
						 * punch continuum command */
						conem = (rfield.ConEmitOut[j]/
						  rfield.widflx[j]*rfield.anu2[j] + PLWidth.PunchLWidth*
						  rfield.outlin[j]*rfield.anu[j])*radius.r1r0sq*
						  EN1RYD*sphere.covgeo;

						flxtrn = conem + flxatt;

						/* use AnuOrg here instead of anu since probably
						 * going to be used by table read
						 * and we want good anu array for sanity check*/
						fprintf( punch.ipPnunit[i],"%10.3e\t", AnuUnit(rfield.AnuOrg[j]) );
						fprintf( punch.ipPnunit[i],"%10.3e\n", flxtrn );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"DUSO") == 0 )
			{
				/* punch grain opacity */
				if( strcmp(chTime,"LAST") == 0 )
				{
					for( j=1; j <= rfield.nflux; j++ )
					{
						fprintf( punch.ipPnunit[i], "%12.4e%10.2e%10.2e%10.2e\n", 
						  rfield.anu[j-1], gv.dstab[j-1] + gv.dstsc[j-1], 
						  gv.dstab[j-1], gv.dstsc[j-1] );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"DUSP") == 0 )
			{
				fprintf( punch.ipPnunit[i], "%.4e\t", 
				  radius.depth );
				/* punch grain physical conditions */
				for( nd=0; nd < gv.nBin; nd++ )
				{
					fprintf( punch.ipPnunit[i], "%.2e\t", gv.bin[nd]->tedust );
				}

				for( nd=0; nd < gv.nBin; nd++ )
				{
					fprintf( punch.ipPnunit[i], "%.2e\t", gv.bin[nd]->dstpot*EVRYD );
				}

				for( nd=0; nd < gv.nBin; nd++ )
				{
					fprintf( punch.ipPnunit[i], "%.2e\t", gv.bin[nd]->DustDftVel );
				}

				fprintf( punch.ipPnunit[i], "%.2e\t%.2e\n", 
				  heat.heating[0][13]/heat.htot, 
				  gv.GasCoolColl/heat.htot );
			}

			else if( strcmp(punch.chPunch[i],"DUSQ") == 0 )
			{
				/* punch grain Qs */
				if( strcmp(chTime,"LAST") == 0 )
				{
					for( j=0; j < rfield.nflux; j++ )
					{
						fprintf( punch.ipPnunit[i], "%11.4e", 
						  rfield.anu[j] );
						for( nd=0; nd < gv.nBin; nd++ )
						{
							fprintf( punch.ipPnunit[i], "%9.1e%9.1e", 
							   gv.bin[nd]->dstab1[j]*4./gv.bin[nd]->IntArea,
							   gv.bin[nd]->dstsc1[j]*4./gv.bin[nd]->IntArea );
						}
						fprintf( punch.ipPnunit[i], "\n" );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"ELEM") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
				{
					/* this is the index for the atomic number on the physical scale */
					jj = (long int)punch.punarg[0][i];

					fprintf( punch.ipPnunit[i], " %.4e", 
					  radius.depth );

					for( j=1; j <= (jj + 1); j++ )
					{
						assert( jj > 0 );
						fprintf( punch.ipPnunit[i], "\t%.2e", 
						  xIonFracs[jj-1][j]/xIonFracs[jj-1][0] );
					}
					fprintf( punch.ipPnunit[i], "\n" );
				}
			}

			else if( strcmp(punch.chPunch[i],"RECA") == 0 )
			{
				/* this will create table for agn then exit, 
				 * routine is in makerecom.c */
				MakeRecombAGN( punch.ipPnunit[i] );
				cdEXIT(1);
			}

			else if( strcmp(punch.chPunch[i],"RECE") == 0 )
			{
				/* punch recombination efficiencies, 
				 * option turned on with the  "punch recombination efficiencies" command
				 * output for the punch recombination coefficients command is actually
				 * produced by a series of routines, as they generate the recombination
				 * coefficients.  these include 
				 * dielsupres, helium, hydrorecom, iibod, and makerecomb*/
				fprintf( punch.ipPnunit[i], 
					"%12.4e %12.4e %12.4e %12.4e %12.4e %12.4e %12.4e\n", 
				  iso.RadRecomb[ipHYDROGEN][0][0][ipRecRad], 
				  iso.RadRecomb[ipHYDROGEN][0][0][ipRecNetEsc] ,
				  iso.RadRecomb[ipHYDROGEN][0][2][ipRecRad],
				  iso.RadRecomb[ipHYDROGEN][0][2][ipRecNetEsc],
				  phe1lv.he1rec[ipRecNetEsc][0], 
				  phe1lv.he1rec[ipRecEsc][0], 
				  heopfr.ophe1f[0] );

			}
			else if( strcmp(punch.chPunch[i],"FE2f") == 0 )
			{
				/* set with punch feii fred command, this punches some stuff from
				* fred hamann's feii atom */
				if( strcmp(chTime,"LAST") != 0 )
					pfeii(punch.ipPnunit[i]);

			}
			else if( strcmp(punch.chPunch[i],"FE2d") == 0 )
			{
				/* punch some departure cefficients for large feii atom */
				if( strcmp(chTime,"LAST") != 0 )
					FeIIPunDepart(punch.ipPnunit[i],FALSE);

			}
			else if( strcmp(punch.chPunch[i],"FE2D") == 0 )
			{
				/* punch all departure cefficients for large feii atom */
				if( strcmp(chTime,"LAST") != 0 )
					FeIIPunDepart(punch.ipPnunit[i],TRUE);

			}
			else if( strcmp(punch.chPunch[i],"FE2p") == 0 )
			{
				/* punch some level populations for large feii atom */
				if( strcmp(chTime,"LAST") != 0 )
					FeIIPunPop(punch.ipPnunit[i],FALSE);

			}
			else if( strcmp(punch.chPunch[i],"FE2P") == 0 )
			{
				/* punch all level populations for large feii atom */
				if( strcmp(chTime,"LAST") != 0 )
					FeIIPunPop(punch.ipPnunit[i],TRUE);

			}
			else if( strcmp(punch.chPunch[i],"GAMM") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
				{
					long nion , ns;
					/* punch photoionization rates, with the PUNCH GAMMAS command */
					for( nelem=0; nelem < LIMELM; nelem++ )
					{
						for( nion=0; nion <= nelem; nion++ )
						{
							for( ns=0; ns < Heavy.nsShells[nion][nelem]; ns++ )
							{
								fprintf( punch.ipPnunit[i], "%3ld%3ld%3ld%10.2e%10.2e%10.2e", 
									nelem+1, nion+1, ns+1, 
									PhotRate.PhotoRate[0][ns][nion][nelem], 
									PhotRate.PhotoRate[1][ns][nion][nelem] ,
									PhotRate.PhotoRate[2][ns][nion][nelem] );

								for( i=0; i < yield.nyield[ns][nion][nelem]; i++ )
								{
									fprintf( punch.ipPnunit[i], "%5.2f", yield.vyield[i][ns][nion][nelem] );
								}
								fprintf( punch.ipPnunit[i], "\n" );
							}
						}
					}
				}
					/*PunGamma(punch.ipPnunit[i]);*/

			}
			else if( strcmp(punch.chPunch[i],"GAUN") == 0 )
			{
				/* punch gaunt factors */
				if( strcmp(chTime,"LAST") != 0 )
					pgaunt(punch.ipPnunit[i]);

			}
			else if( strcmp(punch.chPunch[i],"HEAT") == 0 )
			{
				/* punch heating */
				if( strcmp(chTime,"LAST") != 0 )
					HeatPunch(punch.ipPnunit[i]);
			}

			/*  punch helium */
			/* informatin ou the helium singlets */
			else if( strcmp(punch.chPunch[i],"HEL1") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
				{
					if( he1rate.smhe2ov1 > 1e-30 )
					{
						/* this is simple estimate of singlet to total he */
						fref = (xIonFracs[ipHELIUM][2]/he1rate.smhe2ov1)/xIonFracs[ipHELIUM][0];
					}
					else
					{
						fref = 0.;
					}

					/* second number is ratio of predicted HeII/AheII to simple
					 *  clhe2ov1 is computed He1/He1 */
					fprintf( punch.ipPnunit[i], 
						"he1%10.2e%10.2e%10.2e%10.2e%10.2e%10.2e%10.2e\n", 
					  xIonFracs[ipHELIUM][1]/xIonFracs[ipHELIUM][0], 
					  fref, he1rate.clhe2ov1, he1rate.smhe2ov1, 
					  herecCom.reci*phycon.eden, 
					  phe1lv.he1gam[0], 
					  he.hei3/xIonFracs[ipHELIUM][0] );

					/* print out photoionization rates */
					GammaPrt(nhe1Com.nhe1[0],rfield.nflux,opac.iophe1[0],
					  punch.ipPnunit[i],phe1lv.he1gam[0],phe1lv.he1gam[0]*
					  0.05);
				}
			}

			/* information about the helium triplets - actually special he new debug
			 * get it with punch helium triplets */
			else if( strcmp(punch.chPunch[i],"HEL3") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
				{
					fprintf( punch.ipPnunit[i], 
						"%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n",
						/* ionization fracs */
						xIonFracs[ipHELIUM][2]/he.hei1, 
						iso.xIonRatio[ipHELIUM][1],

						/* 23S pops */
						he3lines.p2s, 
						EmisLines[ipHELIUM][1][ipHe2p3P0][ipHe2s3S].PopLo*xIonFracs[ipHELIUM][2],

						/* recom coef */
						herecCom.rectri+he1rb.he1rtt,
						iso.RadRec_effec[ipHELIUM][ipHELIUM],

						/* ground state photo rates */
						phe1lv.he1gam[0],
						iso.gamnc[ipHELIUM][ipHELIUM][ipHe1s1S]
						);

#					if 0
					/* this is simple estimate of triplet to total he */
					if( he3rate.she2ov3 > 0. )
					{
						fref = (xIonFracs[ipHELIUM][2]/he3rate.she2ov3)/xIonFracs[ipHELIUM][0];
					}
					else
					{
						fref = 0.;
					}
					fprintf( punch.ipPnunit[i], "HeTrip%10.2e%10.2e%10.2e%10.2e%10.2e\n", 
					  he.hei3/xIonFracs[ipHELIUM][0], fref, herecCom.rectri*
					  phycon.eden, he3gams.he3gam[0], he3rate.collhe3 );
					/* print out photoionization rates */
					GammaPrt(he.nhei3,nhe1Com.nhe1[0],opac.ioptri,
					  punch.ipPnunit[i] , he3gams.he3gam[0] , he3gams.he3gam[0]*0.05);
#					endif
				}
			}

			/* punch hummer, results needed for Lya transport, to feed into David's routine */
			else if( strcmp(punch.chPunch[i],"HUMM") == 0 )
			{
				damp = EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].damp;
				eps = EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].Aul/
				  (EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].ColUL *phycon.eden);
				fprintf( punch.ipPnunit[i], 
					" %10.3e %10.3e %10.3e %10.3e %10.3e %10.3e %10.3e\n", 
				  radius.depth, 
				  EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].TauIn, 
				  iso.Pop2Ion[ipHYDROGEN][0][ipH1s]*xIonFracs[ipHYDROGEN][2], 
				  iso.Pop2Ion[ipHYDROGEN][0][ipH2p]*xIonFracs[ipHYDROGEN][2], 
				  phycon.te, damp, eps );
			}

			else if( strcmp(punch.chPunch[i],"HYDc") == 0 )
			{
				/* punch hydrogen physical conditions */
				fprintf( punch.ipPnunit[i], 
					" %.3e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
				  radius.depth, phycon.te, phycon.hden, phycon.eden, 
				  xIonFracs[ipHYDROGEN][1]/phycon.hden, xIonFracs[ipHYDROGEN][2]/phycon.hden, hmi.htwo/phycon.hden, 
				  hmi.h2plus/phycon.hden, hmi.h3plus/phycon.hden, 
				  hmi.hminus/phycon.hden );
			}

			else if( strcmp(punch.chPunch[i],"HYDi") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
				{
					/* punch hydrogen ionization
					 * this will be total decays to ground */
					RateInter = 0.;
					stage = iso.ColIoniz[ipHYDROGEN][0][0]*phycon.eden*iso.Pop2Ion[ipHYDROGEN][0][ipH1s];
					fref = iso.gamnc[ipHYDROGEN][0][ipH1s]*iso.Pop2Ion[ipHYDROGEN][0][ipH1s];
					fout = iso.Pop2Ion[ipHYDROGEN][0][ipH1s];
					for( ion=ipH2s; ion < iso.nLevels[ipHYDROGEN][ipHYDROGEN]; ion++ )
					{
						/* this is total decays to ground */
						RateInter += 
							EmisLines[ipHYDROGEN][ipHYDROGEN][ion][ipH1s].Aul*
						  (EmisLines[ipHYDROGEN][ipHYDROGEN][ion][ipH1s].Pesc + 
						  EmisLines[ipHYDROGEN][ipHYDROGEN][ion][ipH1s].Pdest);
						/* total photo from all levels */
						fref += iso.gamnc[ipHYDROGEN][0][ion]*iso.Pop2Ion[ipHYDROGEN][0][ion];
						/* total col ion from all levels */
						stage += iso.ColIoniz[ipHYDROGEN][0][ion]*phycon.eden*
						  iso.Pop2Ion[ipHYDROGEN][0][ion];
						fout += iso.Pop2Ion[ipHYDROGEN][0][ion];
					}
					fprintf( punch.ipPnunit[i], "hion\t%4ld\t%10.2e\t%10.2e", 
					  nzone, 
					  iso.gamnc[ipHYDROGEN][0][ipH1s], 
					  hcaseb.HRecEffec[0] );
					for(j=0; j < LIMELM; j++)
					{
						fprintf( punch.ipPnunit[i], "\t%10.2e", 
						  iso.RadRec_caseB[ipHYDROGEN][j] );
					}

					fprintf( punch.ipPnunit[i], 
						"\t%10.2e\t%10.2e\t%10.2e\t%10.2e\t%10.2e\t%10.2e\t%10.2e\t%10.2e\n", 
					  xIonFracs[ipHYDROGEN][2]/xIonFracs[ipHYDROGEN][1], 
					  iso.gamnc[ipHYDROGEN][0][ipH1s]/(phycon.eden*hcaseb.HRecEffec[0]), 
					  iso.RadRecomb[ipHYDROGEN][0][1][ipRecEsc], 
					  RateInter, 
					  fref/MAX2(1e-37,fout), 
					  stage/MAX2(1e-37,fout), 
					  iso.gamnc[ipHYDROGEN][0][ipH1s]*xIonFracs[ipHYDROGEN][1]/(phycon.eden* xIonFracs[ipHYDROGEN][2]) ,
					  Secondaries.csupra);

					GammaPrt(iso.ipIsoLevNIonCon[ipHYDROGEN][0][0],rfield.nflux,iso.ipOpac[ipHYDROGEN][0][ipH1s],
					  punch.ipPnunit[i],iso.gamnc[ipHYDROGEN][0][ipH1s],iso.gamnc[ipHYDROGEN][0][ipH1s]*
					  0.05);
				}
			}

			else if( strcmp(punch.chPunch[i],"HYDp") == 0 )
			{
				/* punch hydrogen populations */
				fprintf( punch.ipPnunit[i], " %10.2e%10.2e%10.2e", 
				  radius.depth, xIonFracs[ipHYDROGEN][1], xIonFracs[ipHYDROGEN][2] );
				for( j=ipH1s; j <= 6; j++ )
				{
					fprintf( punch.ipPnunit[i], "%10.2e", iso.Pop2Ion[ipHYDROGEN][0][j] );
				}
				fprintf( punch.ipPnunit[i], "%10.2e%10.2e\n", 
				  iso.Pop2Ion[ipHYDROGEN][0][ipH2s], iso.Pop2Ion[ipHYDROGEN][0][ipH2p] );
			}

			else if( strcmp(punch.chPunch[i],"HYD2") == 0 )
			{
				/* punch hydrogen 21 cm optical depth and spin temperature */
				fprintf( punch.ipPnunit[i], "%.2e\t%.2e\t%.2e\n", 
				radius.depth, TauLines[ipH21cm].TauIn , phycon.te );
			}

			else if( strcmp(punch.chPunch[i],"HYDr") == 0 )
			{
				/* punch hydrogen recomb cooling for agn */
				phycon.te = 2500.;
				while( phycon.te <= 20000. )
				{
					double r1;
					double ThinCoolingCaseB; 

					tfidle(FALSE);

					r1 = HydroRecCool(1,0);
					ThinCoolingCaseB = pow(10.,((-25.859117 + 
					  0.16229407*phycon.telogn[0] + 
					  0.34912863*phycon.telogn[1] - 
					  0.10615964*phycon.telogn[2])/(1. + 
					  0.050866793*phycon.telogn[0] - 
					  0.014118924*phycon.telogn[1] + 
					  0.0044980897*phycon.telogn[2] + 
					  6.0969594e-5*phycon.telogn[3])))/phycon.te;

					fprintf( punch.ipPnunit[i], " %10.2e\t", 
						phycon.te);
					fprintf( punch.ipPnunit[i], " %10.2e\t", 
						(r1+ThinCoolingCaseB)/(BOLTZMANN*phycon.te) );

					fprintf( punch.ipPnunit[i], " %10.2e\t", 
						r1/(BOLTZMANN*phycon.te));

					fprintf( punch.ipPnunit[i], " %10.2e\n", 
						ThinCoolingCaseB/(BOLTZMANN*phycon.te));

					phycon.te *= 2.;
				}
			}

			else if( strcmp(punch.chPunch[i],"IONI") == 0 )
			{
				if( strcmp(chTime,"LAST") == 0 )
				{
					/* punch mean ionization distribution */
					PrtMeanIon( 'i', FALSE , punch.ipPnunit[i] );
				}
			}

			else if( strcmp(punch.chPunch[i]," IP ") == 0 )
			{
				if( strcmp(chTime,"LAST") == 0 )
				{
					/* punch valence shell ip's */
					for( nelem=0; nelem < LIMELM; nelem++ )
					{
						for( n=0; n <= 15; n += 15 )
						{
							if( n + 1 <= nelem+1 )
							{
								fprintf( punch.ipPnunit[i], 
								  " \n" );
								limit = MIN2(n+15,nelem+1);
								fprintf( punch.ipPnunit[i], 
								  "%2.2s", elementnames.chElementSym[nelem]);
								for( ion=n + 1; ion <= limit; ion++ )
								{
									fprintf( punch.ipPnunit[i], 
									  "%4ld", ion );
								}
								fprintf( punch.ipPnunit[i], 
								  "\n" );
							}

							if( n + 1 <= nelem+1 )
							{
								for( ips=0; ips < Heavy.nsShells[n][MIN2(n+15,nelem+1)-1]; ips++ )
								{
									limit = MIN2(n+15,nelem+1);
									for( j=n; j < limit; j++ )
									{
										sum = PH1COM.PH1[ips][nelem+1-j][nelem][0];
										if( sum < 2. )
										{
											strcpy( chEner[j], "      " );
										}
										else if( sum < 10. )
										{
											sprintf( chEner[j], "%6.3f", sum );
										}
										else if( sum < 100. )
										{
											sprintf( chEner[j], "%6.2f", sum );
										}
										else if( sum < 1000. )
										{
											sprintf( chEner[j], "%6.1f", sum );
										}
										else
										{
											sprintf( chEner[j], "%6ld",  (long)(sum) );
										}
									}

									limit = MIN2(n+15,nelem+1);
									fprintf( punch.ipPnunit[i], 
									  "%2.2s", Heavy.chShell[ips]
									   );

									for( j=n; j < limit; j++ )
									{
										fprintf( punch.ipPnunit[i], 
										  "%6.6s", chEner[j] );
									}

									fprintf( punch.ipPnunit[i], 
									  "\n" );
								}
							}
						}
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"LINC") == 0 )
			{
				/* punch line cumulative */
				/* lgOK not used, placdholder */
				if( strcmp(chTime,"LAST") != 0 )
				{
					/* not used for anything here, but should not pass baloney*/
					lgOK = TRUE;
					punlin(punch.ipPnunit[i],"PUNC",lgOK); 
				}
			}

			else if( strcmp(punch.chPunch[i],"LINL") == 0 )
			{
				/* punch line labels, then stop */
				prt_LineLabels(punch.ipPnunit[i]);
				puts( "[Stop in PunchDo]" );
				fprintf( ioQQQ, " Cloudy ends:\n" );
				cdEXIT(1);
			}

			else if( strcmp(punch.chPunch[i],"LINO") == 0 )
			{
				if( strcmp(chTime,"LAST") == 0 )
				{
					/* punch line optical depths, routine is below, file static */
					PunchLineStuff(punch.ipPnunit[i],"optical" , punch.punarg[0][i]);
				}
			}

			else if( strcmp(punch.chPunch[i],"LINP") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
				{
					static int lgFirst=TRUE;
					/* punch line populations, need to do this twice if very first
					 * call since first call to PunchLineStuff generates atomic parameters
					 * rather than level pops, routine is below, file static */
					PunchLineStuff(punch.ipPnunit[i],"populat" , punch.punarg[0][i]);
					if( lgFirst )
					{
						lgFirst = FALSE;
						PunchLineStuff(punch.ipPnunit[i],"populat" , punch.punarg[0][i]);
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"LIND") == 0 )
			{
				/* punch line data, then stop */
				PunchLineData(punch.ipPnunit[i]);
				puts( "[Normal stop in PunchDo]" );
				fprintf( ioQQQ, " Cloudy ends:\n" );
				cdEXIT(1);
			}

			else if( strcmp(punch.chPunch[i],"LINS") == 0 )
			{
				/* punch line structure */
				if( strcmp(chTime,"LAST") != 0 )
				{
					/* not acutally used, but should not pass baloney*/
					lgOK = TRUE;
					punlin(punch.ipPnunit[i],"PUNS",lgOK);
				}
			}

			else if( strcmp(punch.chPunch[i],"LINA") == 0 )
			{
				if( strcmp(chTime,"LAST") == 0 )
				{
					/* punch out all lines with energies */
					for( j=0; j < LineSave.nsum; j++ )
					{
						/* xLineEnergy is sentinel saying to add to continuum only if positive */
						if( LineSv[j].xLineEnergy > 0. && 
							LineSv[j].sumlin > 0. )
						{
							fprintf( punch.ipPnunit[i], "%12.5e\t%8.3f\t%4.4s ", 
							  AnuUnit(LineSv[j].xLineEnergy), 
							  log10(MAX2(SMALLFLOAT,LineSv[j].sumlin) ) + radius.Conv2PrtInten, 
							  LineSv[j].chALab );

							prt_wl( punch.ipPnunit[i], LineSv[j].wavelength );

							fprintf( punch.ipPnunit[i], " \t%c\n", 
							  LineSv[j].chSumTyp);
						}
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"LINI") == 0 )
			{
				if( strcmp(chTime,"LAST") == 0 && 
					(nzone/punch.LinEvery)*punch.LinEvery != nzone )
				{
					/* this is the last zone
					 * punch line intensities - but do not do last zone twice */
					PunLineIntensity(punch.ipPnunit[i]);
				}
				else if( strcmp(chTime,"LAST") != 0 )
				{
					/* following so we only punch first zone if LinEvery reset */
					if( (punch.lgLinEvery && nzone == 1) || 
					  (nzone/punch.LinEvery)*punch.LinEvery == 
					  nzone )
					{
						/* this is middle of calculation
						 * punch line intensities */
						PunLineIntensity(punch.ipPnunit[i]);
					}
				}
			}

			/* punch Lya - some details about Lya */
			else if( strcmp(punch.chPunch[i],"LYMA") == 0 )
			{
				pop = (iso.Pop2Ion[ipHYDROGEN][0][ipH2p]/6.)/(iso.Pop2Ion[ipHYDROGEN][0][ipH1s]/2.);
				if( pop > 0. )
				{
					texc = -1.183e5/log(pop);
				}
				else
				{
					texc = 0.;
				}
				fprintf( punch.ipPnunit[i], 
					"%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
					radius.depth,
				  EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].TauIn, 
				  pop, 
				  texc, 
				  phycon.te, 
				  texc/phycon.te ,
				  EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].Pdest, 
				  opac.opac[EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].ipCont-1],
				  opac.albedo[EmisLines[ipHYDROGEN][ipHYDROGEN][ipH2p][ipH1s].ipCont-1] );
			}

			else if( strcmp(punch.chPunch[i],"MAP ") == 0 )
			{
				/* do the map now if we are at the zone, or if this
				 * is the LAST call to this routine and map not done yet */
				if(  (MapPar.lgMapDone==FALSE ) &&
					(nzone == MapPar.MapZone  ||  strcmp(chTime,"LAST") == 0 ) )
				{
					lgTlkSav = called.lgTalk;
					called.lgTalk = TRUE;
					isav = ioQQQ;
					ioQQQ = punch.ipPnunit[i];
					punt(" map");
					ioQQQ = isav;
					called.lgTalk = lgTlkSav;
				}
			}

			else if( strcmp(punch.chPunch[i],"MOLE") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
				{
					/* molecules, especially for pdr */
					fprintf( punch.ipPnunit[i], 
						"%.3e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
					  radius.depth, 
					  2.*hmi.htwo/phycon.hden, 
					  2.*hmi.htwo_star/phycon.hden, 
					  hevmolec.hevmol[ipCO]  /xIonFracs[ipCARBON][0], 
					  hevmolec.hevmol[ipH2O] /xIonFracs[ipOXYGEN][0], 
					  hevmolec.hevmol[ipOH]  /xIonFracs[ipOXYGEN][0], 
					  hevmolec.hevmol[ipCH]  /xIonFracs[ipCARBON][0], 
					  2.*hevmolec.hevmol[ipOTWO]/xIonFracs[ipOXYGEN][0] );
				}
			}

			else if( strcmp(punch.chPunch[i],"OPAC") == 0 )
			{
				/* punch opacity - routine will parse which type of opacity punch to do */
				if( strcmp(chTime,"LAST") == 0 )
					PunOpac(punch.ipPnunit[i],i);
			}

			/* punch optical depths command */
			else if( strcmp(punch.chPunch[i],"OPTI") == 0 )
			{
				if( strcmp(chTime,"LAST") == 0 )
				{
					for( j=0; j < rfield.nflux; j++ )
					{
						fprintf( punch.ipPnunit[i], 
							"%12.4e\t%.3e\t%12.4e\t%.3e\n", 
						  AnuUnit(rfield.AnuOrg[j]), 
						  opac.TauAbsFace[j]+opac.TauScatFace[j], 
						  opac.TauAbsFace[j], 
						  opac.TauScatFace[j] );
					}
				}
			}

			else if( strcmp(punch.chPunch[i]," OTS") == 0 )
			{
				ConMax = 0.;
				xLinMax = 0.;
				opConSum = 0.;
				opLinSum = 0.;
				ipLinMax = 1;
				ipConMax = 1;

				for( j=0; j < rfield.nflux; j++ )
				{
					opConSum += rfield.otscon[j]*opac.opac[j];
					opLinSum += rfield.otslin[j]*opac.opac[j];
					if( rfield.otslin[j]*opac.opac[j] > xLinMax )
					{
						xLinMax = rfield.otslin[j]*opac.opac[j];
						ipLinMax = j+1;
					}
					if( rfield.otscon[j]*opac.opac[j] > ConMax )
					{
						ConMax = rfield.otscon[j]*opac.opac[j];
						ipConMax = j+1;
					}
				}
				fprintf( punch.ipPnunit[i], 
					"tot con lin=%.2e%.2e lin=%.4s%.4e%.2e con=%.4s%.4e%.2e\n", 
				  opConSum, opLinSum, rfield.chLineLabel[ipLinMax-1]
				  , rfield.anu[ipLinMax-1], xLinMax, rfield.chContLabel[ipConMax-1]
				  , rfield.anu[ipConMax-1], ConMax );
			}

			else if( strcmp(punch.chPunch[i],"OVER") == 0 )
			{
				/* this is the floor for the smallest ionization fractions printed */
				double toosmall = SMALLFLOAT;

				/* overview of model results,
				 * depth, te, hden, eden, ion fracs H, He, c, O */
				if( strcmp(chTime,"LAST") != 0 )
				{
					/* print the depth */
					fprintf( punch.ipPnunit[i], "%.5e\t", radius.depth );

					/* temperature, heating */
					fprintf( punch.ipPnunit[i], "%.3f\t%.3f", 
					  log10(phycon.te), log10(heat.htot) ); 

					/* hydrogen and electron densities */
					fprintf( punch.ipPnunit[i], "\t%.3f\t%.3f", 
					  log10(phycon.hden), log10(phycon.eden) );
					  
					/* molecular fraction of hydrogen */
					fprintf( punch.ipPnunit[i], "\t%.3f", 
					  log10(MAX2(toosmall,2.*hmi.htwo/phycon.hden)) );
					  
					/* ionization fractions of hydrogen */
					fprintf( punch.ipPnunit[i], "\t%.3f\t%.3f", 
					  -log10(MAX2(toosmall,xIonFracs[ipHYDROGEN][1]/phycon.hden)), 
					  -log10(MAX2(toosmall,xIonFracs[ipHYDROGEN][2]/phycon.hden)) );

					/* ionization fractions of helium */
					for( j=1; j <= 3; j++ )
					{
						fprintf( punch.ipPnunit[i], "\t%.3f", 
						  -log10(MAX2(toosmall,xIonFracs[ipHELIUM][j]/
						  xIonFracs[ipHELIUM][0])) );
					}

					/* carbon monoxide molecular fraction of CO */
					fprintf( punch.ipPnunit[i], "\t%.3f", 
					  log10(MAX2(toosmall, hevmolec.hevmol[ipCO]/xIonFracs[ipCARBON][0])) );

					/* ionization fractions of carbon */
					for( j=1; j <= 4; j++ )
					{
						fprintf( punch.ipPnunit[i], "\t%.3f", 
						  -log10(MAX2(toosmall,xIonFracs[ipCARBON][j]/
						  xIonFracs[ipCARBON][0])) );
					}

					/* ionization fractions of carbon */
					for( j=1; j <= 6; j++ )
					{
						fprintf( punch.ipPnunit[i], "\t%6.3f", 
						  -log10(MAX2(toosmall,xIonFracs[ipOXYGEN][j]/
						  xIonFracs[ipOXYGEN][0])) );
					}
					fprintf( punch.ipPnunit[i], "\n" );
				}
			}

			else if( strcmp(punch.chPunch[i]," PDR") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
				{
					av = opac.TauTotalGeo[0][ipvfil.ipVFilter-1]*1.08574;
					fprintf( punch.ipPnunit[i], 
						"%.3e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
					  radius.depth, 
					  coldenCom.colden[ipCOLUMNDENSITY], 
					  av, 
					  phycon.te, 
					  xIonFracs[ipHYDROGEN][1]/phycon.hden, 
					  hmi.htwo/phycon.hden, 
					  hmi.htwo_star/phycon.hden, 
					  xIonFracs[ipCARBON][1]/xIonFracs[ipCARBON][0], 
					  hevmolec.hevmol[ipCO]/xIonFracs[ipCARBON][0], 
					  hevmolec.hevmol[ipH2O]/xIonFracs[ipOXYGEN][0] );
				}
			}

			else if( strcmp(punch.chPunch[i],"PHYS") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
				{
					/* punch physical conditions */
					fprintf( punch.ipPnunit[i], "%14.6e%12.4e%11.3e%11.3e%11.3e%11.3e\n", 
					  radius.depth, phycon.te, phycon.hden, 
					  phycon.eden, heat.htot, wind.AccelTot );
				}
			}

			else if( strcmp(punch.chPunch[i],"PRES") == 0 )
			{
				/* the punch pressure command */
				if( strcmp(chTime,"LAST") != 0 )
				{
					fprintf( punch.ipPnunit[i], 
						"%.4e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%c\n", 
					  radius.depth, 
					  pressure.PressureCorrect, 
					  pressure.PressureCurrent, 
					  pressure.PressureInit + pressure.PresInteg, 
					  pressure.PressureInit, 
					  pressure.PressureGas, 
					  pressure.PressureRam, 
					  pressure.PressureRadiation, 
					  /* subtract continuum rad pres which has already been added on */
					  pressure.PresInteg - pressure.pinzon, 
					  wind.windv/1e5,
					  sqrt(5.*pressure.PressureGas/3./phycon.xMassDensity)/1e5,
					  TorF(conv.lgConvPres) );
				}
			}

			else if( strcmp(punch.chPunch[i],"RADI") == 0 )
			{
				if( strcmp(chTime,"LAST") != 0 )
				{
					fprintf( punch.ipPnunit[i], " %4ld%12.4e%12.4e%12.4e\n", 
					  nzone, radius.Radius, radius.depth, 
					  radius.drad );
				}
			}

			else if( strcmp(punch.chPunch[i],"RESU") == 0 )
			{
				/*  punch results of the calculation */
				if( strcmp(chTime,"LAST") == 0 )
					punResults(punch.ipPnunit[i]);
			}

			else if( strcmp(punch.chPunch[i],"SOUS") == 0 )
			{
				/* full spectrum of continuum source function at 1 depth
				 *  command was "punch source spectrum" */
				if( strcmp(chTime,"LAST") != 0 )
				{
					limit = MIN2(rfield.ipMaxBolt,rfield.nflux);
					for( j=0; j < limit; j++ )
					{
						fprintf( punch.ipPnunit[i], 
							"%12.4e%12.4e%12.4e%12.4e%12.4e\n", 
						  rfield.anu[j], 
						  rfield.ConEmitLocal[j]/rfield.widflx[j], 
						  opac.opac[j], 
						  rfield.ConEmitLocal[j]/rfield.widflx[j]/MAX2(1e-35,opac.opac[j]), 
						  rfield.ConEmitLocal[j]/rfield.widflx[j]/MAX2(1e-35,opac.opac[j])/plankf(j) );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"SOUD") == 0 )
			{
				/* parts of continuum source function vs depth
				 * command was source depth */
				j = iso.ipIsoLevNIonCon[ipHYDROGEN][0][ipH1s] + 2;
				fprintf( punch.ipPnunit[i], 
					"%12.4e%12.4e%12.4e%12.4e\n", 
				  opac.TauAbsFace[j-1], 
				  rfield.ConEmitLocal[j-1]/rfield.widflx[j-1]/MAX2(1e-35,opac.opac[j-1]), 
				  rfield.otscon[iso.ipIsoLevNIonCon[ipHYDROGEN][0][ipH1s]-1], 
				  rfield.otscon[iso.ipIsoLevNIonCon[ipHYDROGEN][0][0]-1]/opac.opac[iso.ipIsoLevNIonCon[ipHYDROGEN][0][ipH1s]-1] );
			}

			/* this is punch special option */
			else if( strcmp(punch.chPunch[i],"SPEC") == 0 )
			{
				PunSpec(punch.ipPnunit[i],chTime);
			}

			else if( strcmp(punch.chPunch[i],"TEGR") == 0 )
			{
				/* punch history of heating and cooling */
				if( strcmp(chTime,"LAST") != 0 )
				{
					fprintf( punch.ipPnunit[i], " " );
					for( j=0; j < NGRID; j++ )
					{
						fprintf( punch.ipPnunit[i], 
							"%4ld%11.3e%11.3e%11.3e\n", 
						  HeatCool.nZonGrid[j], 
						  HeatCool.TeGrid[j], 
						  HeatCool.HtGrid[j], 
						  HeatCool.ClGrid[j] );
					}
				}
			}

			else if( strcmp(punch.chPunch[i],"TEMP") == 0 )
			{
				/* temperature and its derivative */
				fprintf( punch.ipPnunit[i], " %4ld%12.4e%12.4e\n", 
				  nzone, phycon.te, (phycon.te - phycon.tlast)/
				  radius.drad );
			}

			else if( strcmp(punch.chPunch[i],"TIME") == 0 )
			{
				static double ElapsedTime , ZoneTime;
				if( nzone<=1 )
				{
					ElapsedTime = cdExecTime();
					ZoneTime = 0.;
				}
				else
				{
					double t = cdExecTime();
					ZoneTime = t - ElapsedTime;
					ElapsedTime = t;
				}

				/* zone, time for this zone, elapsed time */
				fprintf( punch.ipPnunit[i], " %ld\t%.3f\t%.2f\n", 
				  nzone,  ZoneTime , ElapsedTime );
			}

			else if( strcmp(punch.chPunch[i],"TPRE") == 0 )
			{
				/* temperature and its predictors, turned on with punch tprid */
				fprintf( punch.ipPnunit[i], "%5ld %11.4e %11.4e %11.4e %g\n", 
				  nzone, phycon.TeInit, phycon.TeProp, phycon.te, 
				  (phycon.TeProp- phycon.te)/phycon.te );
			}

			else if( strcmp(punch.chPunch[i],"VERN") == 0 )
			{
				if( (strcmp(chTime,"LAST") == 0) && FeII.lgFeIION)
				{
					/*FeIILines("pun",norm.ScaleNormLine/LineSv[norm.ipNormWavL].sumlin);*/
					/*FeIIPunchLines(FeII.ioVerner);*/
					FeIIPunchLines( punch.ipPnunit[i] );
				}
			}

			else if( strcmp(punch.chPunch[i],"WIND") == 0 )
			{
				/* wind velocity, radiative accel, and ratio total to e- accel */
				fprintf( punch.ipPnunit[i], 
					"%.5e\t%.5e\t%.4e\t%.4e\t%.4e\t%.4e\t%.4e\n", 
				  radius.Radius, 
				  radius.depth, 
				  wind.windv, 
				  wind.AccelTot, 
				  wind.AccelLine,
				  wind.AccelCont ,
				  wind.fmul );
			}

			else
			{
				/* this is insanity, internal flag set in getpunch not seen here */
				fprintf( ioQQQ, " PunchDo does not recognize flag %4.4s set by GETPUNCH.  This is impossible.\n", 
				  punch.chPunch[i] );
				ShowMe();
				puts( "[Stop in PunchDo]" );
				cdEXIT(1);
			}
			/* at start of loop, broke to here if not last iteration,
			 * and punch last was set
			 * throw special line if punching every iteration */
			if( strcmp(chTime,"LAST") == 0 && !IterCnt.lgLastIt && punch.lgHashEndIter )
			{
				fprintf( punch.ipPnunit[i], " ###########################\n" );
			}
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->PunchDo()\n", debug_fp );
#	endif
	return;
}

/*PunLineIntensity produce the 'punch lines intensity' output */
static void PunLineIntensity(FILE * io)
{
	char chCard[81];
	int lgEOF;
	long int i;

#	ifdef DEBUG_FUN
	fputs( "<+>PunLineIntensity()\n", debug_fp );
#	endif

	/* used to punch out all the emission line intensities
	 * first initialize the line image reader */

	rdinit();

	fprintf( io, "**********************************************************************************************************************************\n" );
	
	lgEOF = FALSE;
	while( !lgEOF )
	{
		readar(chCard,&lgEOF);
		if( !lgEOF )
		{
				fprintf( io, "%s\n", chCard );
		}
	}

	/* first print any cautions or warnings */
	cdWarnings( io);
	cdCautions( io);
	fprintf( io, "zone=%5ld\n", nzone );
	fprintf( io, "**********************************************************************************************************************************\n" );
	fprintf( io, "begin emission lines\n" );

	/* >>chng 96 jul 03, only punch zero intensities */
	PunResults1Line(io,"    ",0,0.,'o',"Start");
	for( i=0; i < LineSave.nsum; i++ )
	{
		if( LineSv[i].sumlin > 0. )
		{
			PunResults1Line(io,(char*)LineSv[i].chALab,LineSv[i].wavelength,
			  LineSv[i].sumlin, LineSv[i].chSumTyp, "Line ");
		}
	}

	PunResults1Line(io,"    ",0,0.,'o',"Flush");

	fprintf( io, "     \n" );
	fprintf( io, "**********************************************************************************************************************************\n" );


#	ifdef DEBUG_FUN
	fputs( " <->PunLineIntensity()\n", debug_fp );
#	endif
	return;
}

static int lgPopsFirst , lgOptical ;

/*PunchLineStuff punch optical depths or source functions for all transferred lines */
static void PunchLineStuff(FILE * io,char *chJob , float xLimit )
{

	long int ipZ , ipLo , ipHi , i;
	long index = 0;
	static int lgFirst=TRUE;

#	ifdef DEBUG_FUN
	fputs( "<+>PunchLineStuff()\n", debug_fp );
#	endif

	/*find out which job this is and set a flag to use later */
	if( strcmp( &*chJob , "optical" ) == 0 )
	{
		/* punch line optical depths */
		lgOptical = TRUE;
		lgPopsFirst = FALSE;
	}
	else if( strcmp( &*chJob , "populat" ) == 0 )
	{
		lgOptical = FALSE;
		/* level population information */
		if( lgFirst )
		{
			lgPopsFirst = TRUE;
			fprintf(io,"index\tAn.ion\tgLo\tgUp\tE(wn)\tgf\n");
			lgFirst = FALSE;
		}
		else
		{
			lgPopsFirst = FALSE;
		}
	}
	else
	{
		fprintf( ioQQQ, " insane job in PunchLineStuff =%s\n", 
		  &*chJob );
		puts( "[Stop in PunchLineStuff]" );
		cdEXIT(1);
	}

	/* loop over all lines, calling put1Line to create info (routine located below) */
	/*hydrogen like lines */
	for( ipZ=0; ipZ < LIMELM; ipZ++ )
	{
		if( abundances.lgElmtOn[ipZ]  )
		{
			for( ipHi=ipH2s; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
			{
				for( ipLo=ipH1s; ipLo <= (ipHi - 1); ipLo++ )
				{
					++index;
					pun1Line( &EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo] , io , xLimit  , index);
				}
			}
			/* also do lyman lines if optical depths are to be done */
			if( lgOptical )
			{
				for( ipHi=iso.nLevels[ipHYDROGEN][ipZ]; ipHi < iso.nLyman[ipHYDROGEN]; ipHi++ )
				{
					++index;
					pun1Line( &iso.ExtraLymanLines[ipHYDROGEN][ipZ][ipHi] , io , xLimit  , index);
				}
			}
		}
	}

	/*helium like lines */
	for( ipZ=1; ipZ < LIMELM; ipZ++ )
	{
		if( ipZ < 2 || abundances.lgElmtOn[ipZ] )
		{
			/* arrays are dim'd ipHe2p1P */
			for( ipLo=ipHe1s1S; ipLo < iso.nLevels[ipHELIUM][ipZ]-1; ipLo++ )
			{
				/* set ENTIRE array to impossible values, in case of bad pointer */
				for( ipHi=ipLo+1; ipHi < iso.nLevels[ipHELIUM][ipZ]; ipHi++ )
				{
					++index;
					pun1Line( &EmisLines[ipHELIUM][ipZ][ipHi][ipLo] , io, xLimit  , index);
				}
			}
			/* also do lyman lines if optical depths are to be done */
			if( lgOptical )
			{
				for( ipHi=iso.nHighest[ipHELIUM][ipZ]; ipHi < iso.nLyman[ipHELIUM]; ipHi++ )
				{
					++index;
					pun1Line( &iso.ExtraLymanLines[ipHELIUM][ipZ][ipHi] , io , xLimit  , index);
				}
			}
		}
	}

	/* index from 1 to skip over dummy line */
	for( i=1; i < nLevel1; i++ )
	{
		++index;
		pun1Line( &TauLines[i] , io , xLimit  , index);
	}

	for( i=0; i < nWindLine; i++ )
	{
		if( TauLine2[i].nelem != TauLine2[i].IonStg )
		{
			++index;
			pun1Line( &TauLine2[i] , io , xLimit  , index);
		}
	}

	if( FeII.lgFeIION )
	{
		/* do optical depths of FeII lines, if large atom is turned on */
		Fe2PunchLineStuff( io , xLimit  , index);
	}

	/* C12O16 */
	for( i=0; i < nCORotate; i++ )
	{
		++index;
		pun1Line( &C12O16Rotate[i] , io , xLimit  , index);
	}

	/* C13O16 */
	for( i=0; i < nCORotate; i++ )
	{
		++index;
		pun1Line( &C13O16Rotate[i] , io , xLimit  , index);
	}
	fprintf(io, "##################################\n"); 

#	ifdef DEBUG_FUN
	fputs( " <->PunchLineStuff()\n", debug_fp );
#	endif
	return;
}

/*pun1Line called by PunchLineStuff to produce output for one line */
void pun1Line( EmLine * t , FILE * io , float xLimit  , long index )
{

	if( lgOptical )
	{
		/* optical depths, no special first time, only print them */
		if( t->TauIn >= xLimit )
		{
			/* label like "C  4" or "H  1" */
			fprintf( io , "%2.2s%2.2s\t", 
			  elementnames.chElementSym[t->nelem-1] ,
			  elementnames.chIonStage[t->IonStg-1]  );

			/* print wavelengths, either line in main printout labels, 
			 * or in various units in exponential notation */
			if( strcmp( punch.chConPunEnr[punch.ipConPun], "labl" )== 0 )
			{
				prt_wl( io , t->WLAng );
			}
			else
			{
				/* this converts energy in rydbergs into any of the other units */
				fprintf( io , "%.7e", AnuUnit(t->EnergyRyd) );
			}
			fprintf( io , "\t%.3f\n", log10(MAX2(SMALLFLOAT,t->TauIn))  );
		}
	}
	else if( lgPopsFirst )
	{
		char chLbl[11];
		/* first call to line populations, print atomic parameters and indices */
		strcpy( chLbl, chLineLbl(t) );
		fprintf(io, "%li\t%s\t" , index , chLbl );
		/* stat weights */
		fprintf(io, "%.0f\t%.0f\t", 
			t->gLo ,t->gHi);
		/* energy difference, gf */
		fprintf(io, "%.2f\t%.3e\t", 
			t->EnergyWN ,t->gf);
		fprintf(io, "\n");
	}
	else
	{
		/* not first call, so do level populations and indices defined above */
		if( t->PopHi > xLimit )
		{
			fprintf(io,"%li\t%.2e\t%.2e\n", index,
				t->PopLo , 
				t->PopHi );
		}
	}
}

/*PunchNewContinuum produce the 'punch new continuum' output */
static void PunchNewContinuum(FILE * io, long ipCon )
{
	long int ipLo, ipHi,
		j ,
		ncells;

	double 
		wllo=3500. , 
		wlhi=7000. , 
		resolution = 2.;

	double *energy, 
		*cont_incid,
		*cont_atten,
		*diffuse_in,
		*diffuse_out;

	/* get the low limit, cp_range_min is zero if not set */
	wllo = MAX2( rfield.anu[0] , punch.cp_range_min[ipCon] );
	if( punch.cp_range_max[ipCon] > 0. )
	{
		/* set to smaller of these two  */
		wlhi = MIN2( rfield.anu[rfield.nflux-1] , punch.cp_range_max[ipCon]  );
	}
	else
	{
		/* use high energy limit since not set */
		wlhi = rfield.anu[rfield.nflux-1] ;
	}

	if( punch.cp_resolving_power[ipCon] != 0. )
	{
		/* next two bogus to fool lint - they cannot happen */
		ipLo = LONG_MAX;
		ipHi = LONG_MAX;
		/* wwe will set a new continuum mesh */
		ncells = (long)((wlhi-wllo)/resolution + 1);
	}
	else
	{
		/* use native continuum mesh */
		ipLo = ipoint(wllo)-1;
		ipHi = ipoint(wlhi)-1;
		ncells = ipHi - ipLo + 1;
	}

	/* now allocate the space */
	if( (energy = (double *)MALLOC( (size_t)(ncells+1)*sizeof(double) ) )==NULL )
		bad_malloc();
	if( (cont_incid = (double *)MALLOC( (size_t)(ncells+1)*sizeof(double) ) )==NULL )
		bad_malloc();
	if( (cont_atten = (double *)MALLOC( (size_t)(ncells+1)*sizeof(double) ) )==NULL )
		bad_malloc();
	if( (diffuse_in = (double *)MALLOC( (size_t)(ncells+1)*sizeof(double) ) )==NULL )
		bad_malloc();
	if( (diffuse_out = (double *)MALLOC( (size_t)(ncells+1)*sizeof(double) ) )==NULL )
		bad_malloc();

	/* was the resolution changed from the default native resolution ? */
	if( punch.cp_resolving_power[ipCon] != 0. )
	{
		/* now create energy grid */
		energy[0] = wlhi;
		j = 0;
		while( energy[j]-resolution > wllo )
		{
			++j;
			assert( j< ncells+1 );
			energy[j] = energy[j-1] - resolution;
		}

		ncells = j;
		/* now convert to rydbergs */
		for( j=0; j<ncells; ++j )
		{
			energy[j] = RYDLAM / energy[j];
		}
	}
	else 
	{
		/* now convert to rydbergs */
		for( j=0; j<ncells; ++j )
		{
			energy[j] = rfield.AnuOrg[j+ipLo] - rfield.widflx[j+ipLo]/2.;
		}
	}

	/* for cdSPEC the energy vector is the lower edge of the energy cell */
	/* get incident continuum */
	cdSPEC( 1 , energy , ncells , cont_incid );
	/* get attenuated incident continuum */
	cdSPEC( 2 , energy , ncells , cont_atten );
	/* get attenuated incident continuum */
	cdSPEC( 5 , energy , ncells , diffuse_in );
	/* get attenuated incident continuum */
	cdSPEC( 4 , energy , ncells , diffuse_out );

	/* for this example we will do a wavelength range */
	for( j=0; j<ncells-1 ; ++j )
	{
		/* photon energy in appropriate energy or wavelength units */
		fprintf( io,"%.3e\t", AnuUnit((float)(energy[j]+rfield.widflx[j+ipLo]/2.) ) );
		fprintf( io,"%.3e\t", cont_incid[j] );
		fprintf( io,"%.3e\t", cont_atten[j] );
		fprintf( io,"%.3e", diffuse_in[j]+diffuse_out[j] );
		fprintf( io,"\n" );
	}

	free(energy);
	free(cont_incid);
	free(diffuse_in);
	free(diffuse_out);
	free(cont_atten);
}

/* punch agn hemiss, for chapt 4, routine is below */
static void AGN_Hemis(FILE *io )
{
#	define NTE 4
	float te[NTE] = {5000., 10000., 15000., 20000.};
	float *continuum[NTE];
	long i , j;

	/* make table of continuous emission at various temperatuers */
	/* first allocate space */
	for( i=0;i<NTE; ++i)
	{
		if( (continuum[i] = (float *)MALLOC((unsigned)rfield.nflux*sizeof(float) ))==NULL)
			bad_malloc();

		/* set the next temperature */
		phycon.te = te[i];
		/* recompute everything at this new temp */
		tfidle(TRUE);
		/* converge the pressure-temperature-ionization solution for this zone */
		ConvPresTempEdenIoniz();

		/* now get the thermal emission */
		RTDiffuse();
		for(j=0;j<rfield.nflux; ++j )
		{
			continuum[i][j] = rfield.ConEmitLocal[j]/(float)phycon.eden/
				(xIonFracs[ipHYDROGEN][2] + xIonFracs[ipHELIUM][2] + xIonFracs[ipHELIUM][3] );
		}
	}

	/* print title for line */
	fprintf(io,"wl");
	for( i=0;i<NTE; ++i)
	{
		fprintf(io,"\t%.0f",te[i]);
	}
	fprintf( io , "\tcont\n"); 

	/* not print all n temperatures across a line */
	for(j=0;j<rfield.nflux; ++j )
	{
		fprintf( io , "%12.5e", 
		  AnuUnit(rfield.AnuOrg[j]) );
		/* loop over the temperatures, and for each, calculate a continuum */
		for( i=0;i<NTE; ++i)
		{
			fprintf(io,"\t%.3e",continuum[i][j]*rfield.anu2[j]*EN1RYD/rfield.widflx[j]);
		}
		/* cont label and end of line*/
		fprintf( io , "\t%s\n" , rfield.chContLabel[j]); 
	}

	/* now free the continua */
	for( i=0;i<NTE; ++i)
	{
		free( continuum[i] );
	}

	fprintf( ioQQQ, " I have left the code in a disturbed state, and will now exit.\n");
	puts( "[Stop in PunchDo]" );
	cdEXIT(1);
}
