/*ffun evaluate total flux for sum of all continuum sources */
/*ffun1 derive flux at a specific energy, for one continuum */
/*ReadTable called by TABLE READ to read in continuum from PUNCH TRANSMITTED CONTINUUM */
#include "cddefines.h"
#include "rfield.h"
#include "physconst.h"
#include "ipoint.h"
#include "bit32.h"
#include "opacity.h"
#include "ffun.h"

/*ReadTable called by TABLE READ to read in continuum from PUNCH TRANSMITTED CONTINUUM */
static void ReadTable(FILE * io);

double ffun(double anu)
{
	double ffun_v;
	static int lgWarn = FALSE;

#	ifdef DEBUG_FUN
	fputs( "<+>ffun()\n", debug_fp );
#	endif

	/* FFUN returns the sum of photons per unit time, area, energy,
	 * for all continuua in the calculation
	 * we loop over all nspec continuum sources - ipspec points to each
	 * */
	ffun_v = 0.;
	for( rfield.ipspec=0; rfield.ipspec < rfield.nspec; rfield.ipspec++ )
	{
		ffun_v += ffun1(anu)*rfield.spfac[rfield.ipspec];
	}

	if( (ffun_v > 1e35 && (!lgWarn)) && bit32.lgBit32 )
	{
		lgWarn = TRUE;
		fprintf( ioQQQ, " FFUN:  The net continuum is very intense for a 32 bit cpu.\n" );
		fprintf( ioQQQ, " I will try to press on, but may have problems.\n" );
	}


#	ifdef DEBUG_FUN
	fputs( " <->ffun()\n", debug_fp );
#	endif
	return( ffun_v );
}

/*ffun1 derive flux at a specific energy, for one continuum */
double ffun1(double xnu)
{
	char chKey[6];
	long int i;
	double fac, 
	  ffun1_v, 
	  y;
	static int lgWarn = FALSE;

#	ifdef DEBUG_FUN
	fputs( "<+>ffun1()\n", debug_fp );
#	endif


	/* confirm that pointer is within range */
	ASSERT( rfield.ipspec >= 0);
	ASSERT( rfield.ipspec < rfield.nspec );

	/* FFUN1 returns photons per unit time, area, energy, for one continuum
	 * ipspec is the pointer to the continuum source, in the order
	 * entered on the command lines */

	/*begin sanity check */
	ASSERT( xnu >= rfield.emm*0.99 );
	ASSERT( xnu <= rfield.egamry*1.01 );
	/*end sanity check */

	strcpy( chKey, rfield.chSpType[rfield.ipspec] );

	if( strcmp(chKey,"AGN  ") == 0 )
	{
		/* power law with cutoff at both ends
		 * nomenclature all screwed up - slope is cutoff energy in Ryd,
		 * cutoff[1][i] is ratio of two continua from alpha ox
		 * cutoff[2][i] is slope of bb temp */
		ffun1_v = pow(xnu,-1. + rfield.cutoff[1][rfield.ipspec])*
		  sexp(xnu/rfield.slope[rfield.ipspec])*sexp(0.01/
		  xnu);
		/* only add on x-ray for energies above 0.1 Ryd */
		if( xnu > 0.1 )
		{
			if( xnu < 7350. )
			{
				/* cutoff is actually normalization constant
				 * below 100keV continuum is nu-1 */
				ffun1_v += pow(xnu,rfield.cutoff[2][rfield.ipspec] - 
				  1.)*rfield.cutoff[0][rfield.ipspec]*sexp(1./
				  xnu);
			}
			else
			{
				ffun1_v += pow(7350.,rfield.cutoff[2][rfield.ipspec] - 
				  1.)*rfield.cutoff[0][rfield.ipspec]/
				  POW3(xnu/7350.);
			}
		}

	}
	else if( strcmp(chKey,"POWER") == 0 )
	{
		/* power law with cutoff at both ends */
		ffun1_v = pow(xnu,-1. + rfield.slope[rfield.ipspec])*
		  sexp(xnu/rfield.cutoff[0][rfield.ipspec]+rfield.cutoff[1][rfield.ipspec]/
		  xnu);

	}
	else if( strcmp(chKey,"BLACK") == 0 )
	{
		const double db_log = log(DBL_MAX);

		/* black body */
		fac = TE1RYD*xnu/rfield.slope[rfield.ipspec];
		/* >>>chng 00 apr 13 from 80 to log(dbl_max) */
		if( fac > db_log )
		{
			ffun1_v = 0.;
		}
		else if( fac > 1.e-5 )
		{
			ffun1_v = xnu*xnu/(exp(fac) - 1.);
		}
		else
		{
			ffun1_v = xnu*xnu/(fac*(1. + fac/2.));
		}

	}
	else if( strcmp(chKey,"INTER") == 0 )
	{
		/* interpolate on input spectrum */
		if( xnu >= rfield.tNuRyd[0][rfield.ipspec] )
		{
			/* loop starts at second array element, [1], since want to 
			 * find next continuum energy greater than desired point */
			for( i=1; i < rfield.nupper; i++ )
			{
				if( xnu < rfield.tNuRyd[i][rfield.ipspec] )
				{
					/* points to interpolate upon are in f_nu */
					y = rfield.tFluxLog[i-1][rfield.ipspec] + 
					  rfield.tslop[i-1][rfield.ipspec]*
					  log10(xnu/rfield.tNuRyd[i-1][rfield.ipspec]);
					ASSERT( y >= MIN2( rfield.tFluxLog[i-1][rfield.ipspec] , rfield.tFluxLog[i][rfield.ipspec] )) ;
					ASSERT( y <= MAX2( rfield.tFluxLog[i-1][rfield.ipspec] , rfield.tFluxLog[i][rfield.ipspec] )) ;

					/* return value is photon density, div by energy */
					ffun1_v = pow(10.,y)/xnu ;
					
#					ifdef DEBUG_FUN
					fputs( " <->ffun1()\n", debug_fp );
#					endif
					return( ffun1_v );
				}
			}
			/* energy above highest in table */
			ffun1_v = 0.;
		}
		else
		{
			/* energy below lowest on table */
			ffun1_v = 0.;
		}

	}
	else if( strcmp(chKey,"BREMS") == 0 )
	{
		/* brems continuum, rough gaunt factor */
		fac = TE1RYD*xnu/rfield.slope[rfield.ipspec];
		ffun1_v = sexp(fac)/pow(xnu,1.2);

	}
	else if( strcmp(chKey,"LASER") == 0 )
	{
#		ifdef BIG
#			undef BIG
#		endif
#		define	BIG	1e10
#		ifdef SMALL
#			undef SMALL
#		endif
#		define	SMALL	1e-25
		/* a laser, mostly one frequency */
		/* >>chng 01 jul 01, was hard-wired 0.05 rel frac, change to optional
		 * second parameter, with default of 0.05 */
		/*if( xnu > 0.95*rfield.slope[rfield.ipspec] && xnu < 
		  1.05*rfield.slope[rfield.ipspec] )*/
		if( xnu > (1.-rfield.cutoff[0][rfield.ipspec])*rfield.slope[rfield.ipspec] && 
			xnu < (1.+rfield.cutoff[0][rfield.ipspec])*rfield.slope[rfield.ipspec] )
		{
			ffun1_v = BIG;
		}
		else
		{
			ffun1_v = SMALL;
		}

	}
	else if( strcmp(chKey,"READ ") == 0 )
	{
		/* if this is first time called then we need to read in file */
		if( rfield.tslop[0][rfield.ipspec] == 0. )
		{
			/* >>chng 01 nov 01, array index for below was bogus, only worked for a single continuum */
			ReadTable(rfield.ioTableRead[rfield.ipspec]);
			rfield.tslop[0][rfield.ipspec] = 1.;
		}
		/* use array of values read in on TABLE READ command */
		if( xnu >= rfield.egamry )
		{
			ffun1_v = 0.;
		}
		else
		{
			i = ipoint(xnu);
			if( i > rfield.nupper || i < 1 )
			{
				ffun1_v = 0.;
			}
			else
			{
				ffun1_v = rfield.ConTabRead[i-1]/rfield.anu[i-1]/rfield.anu[i-1];
			}
		}

	}
	else if( strcmp(chKey,"VOLK ") == 0 )
	{
		/* use array of values read in from Kevin Volk's rebinning of
		 * large atlas grids */
		if( xnu >= rfield.egamry )
		{
			ffun1_v = 0.;
		}
		else
		{
			i = ipoint(xnu);
			if( i > rfield.nupper )
			{
				fprintf( ioQQQ, " ffun1: Too many points - increase ncell\n" );
				fprintf( ioQQQ, " cell needed=%4ld ncell=%4ld\n", 
				  i, rfield.nupper );
				puts( "[Stop in ffun1]" );
				cdEXIT(EXIT_FAILURE);
			}
			if( i > rfield.nupper || i < 1 )
			{
				ffun1_v = 0.;
			}
			else
			{
				/* bug fixed Jul 9 93: FFUN1 = TSLOP(IPSPEC,I) / ANU(I) / ANU(I)
				 *   i has value 939 */
				ffun1_v = rfield.tslop[i-1][rfield.ipspec]/
				  rfield.anu[i-1];
			}
		}
	}
	else
	{
		fprintf( ioQQQ, " ffun1: I do not understand continuum label %5.5s\n", 
		  chKey );
		puts( "[Stop in ffun1]" );
		cdEXIT(EXIT_FAILURE);
	}

	if( (ffun1_v > 1e35 && (!lgWarn)) && bit32.lgBit32 )
	{
		lgWarn = TRUE;
		fprintf( ioQQQ, " FFUN1:  Continuum%4ld is very intense for a 32 bit cpu.\n", 
		  rfield.ipspec );
		fprintf( ioQQQ, " I will try to press on, but may have problems.\n" );
	}

#	ifdef DEBUG_FUN
	fputs( " <->ffun1()\n", debug_fp );
#	endif
	return( ffun1_v );
}

/*ReadTable called by TABLE READ to read in continuum from PUNCH TRANSMITTED CONTINUUM */
static void ReadTable(FILE * io)
{
	char chLine[INPUT_LINE_LENGTH];
	long int i, 
	  nPoints;
	double Differ, 
	  EnerLast;

#	ifdef DEBUG_FUN
	fputs( "<+>ReadTable()\n", debug_fp );
#	endif

	/* make sure the file handle points somewhere */
	ASSERT( io != NULL );

	/* read in first line of header */
	if( NULL==fgets( chLine , (int)sizeof(chLine) , io ) )
	{
		fprintf( ioQQQ, " error 1 reading input continuum file.\n" );
		puts( "[Stop in readtable]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* now read in the file of numbers */
	i = 0;
	/* keep reading until we hit eol or run out of room in the array */
	while( (NULL!=fgets(chLine, (int)sizeof(chLine),io)) && (i<rfield.nupper) )
	{
		sscanf( chLine, "%f%f ", &opac.tmn[i], &rfield.ConTabRead[i] );
		++i;
	}
	/* put pointer at last good value */
	nPoints = i - 1;

	/* check that sane number of values entered */
	if( nPoints < 1 )
	{
		fprintf( ioQQQ, " ReadTable, file for TABLE READ has too few points, =%5ld\n", 
		  nPoints );
		puts( "[Stop in readtable]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* check on units of energy scale, convert to Rydbergs if necessary
	 * tmn is scale read in from punch file, anu is correct scale
	 * EnerLast is energy of last point in rydbergs */
	EnerLast = opac.tmn[nPoints];
	if( fabs(opac.tmn[0]-rfield.anu[0])/rfield.anu[0] > 1e-3 )
	{
		/* first energy does not appear to have been in Rydbergs */
		if( opac.tmn[0] < 0. )
		{
			fprintf( ioQQQ, " ReadTable:  the first energy in table read file is not positive.  Something is wrong.\n" );
			puts( "[Stop in readtable]" );
			cdEXIT(EXIT_FAILURE);
		}
		else if( fabs(opac.tmn[0]-RYDLAM/rfield.anu[0]*1e-4)/opac.tmn[0] < 
		  1e-3 )
		{
			/* wavelength in microns */
			EnerLast = RYDLAM/opac.tmn[nPoints]/1e4;
		}
		else if( fabs(opac.tmn[0]-RYDLAM/rfield.anu[0])/opac.tmn[0] < 
		  1e-3 )
		{
			/* wavelength in Angstroms */
			EnerLast = RYDLAM/opac.tmn[nPoints];
		}
		else if( fabs(opac.tmn[0]-rfield.anu[0]*EVRYD*1e-3)/opac.tmn[0] < 
		  1e-3 )
		{
			/* wavelength in keV */
			EnerLast = opac.tmn[nPoints]/EVRYD/1e-3;
		}
		else if( fabs(opac.tmn[0]-rfield.anu[0]*EVRYD)/opac.tmn[0] < 
		  1e-3 )
		{
			/* wavelength in eV */
			EnerLast = opac.tmn[nPoints]/EVRYD;
		}
		else
		{
			fprintf( ioQQQ, " ReadTable:  the energy scale in the table read file makes no sense to me.\n" );
			puts( "[Stop in readtable]" );
			cdEXIT(EXIT_FAILURE);
		}
	}

	/* now check now the energies of the highest points agree */
	Differ = fabs(EnerLast-rfield.anu[nPoints])/rfield.anu[nPoints];
	if( Differ > 0.001 )
	{
		fprintf( ioQQQ, " ReadTable: The energy mesh of the file read in by the TABLE READ command does not agree with this version of Cloudy.\n" );
		fprintf( ioQQQ, " ReadTable: Was the file generated by an older version of the code?\n" );
		fprintf( ioQQQ, " ReadTable: Number of points read in=%5ld\n", 
		  nPoints );
		fprintf( ioQQQ, " ReadTable: input, internal energies=%12.4e%12.4e\n", 
		  opac.tmn[nPoints], rfield.anu[nPoints] );
		puts( "[Stop in readtable]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* make sure rest of array has valid zeros */
	for( i=nPoints + 1; i < rfield.nupper; i++ )
	{
		rfield.ConTabRead[i] = 0.;
	}

	fclose(io);

#	ifdef DEBUG_FUN
	fputs( " <->ReadTable()\n", debug_fp );
#	endif
	return;
}
