/*H2_Create create H2 molecules, called by ContCreatePointers */
/*H2_ReadEnergies read energies for all electronic levels */
/*H2_ReadTransprob read transition probabilities */
/*H2_ContPoint set the ipCont struc element for the H2 molecule, called by ContCreatePointers */
/*H2_Accel radiative acceleration due to H2 */
/*H2_RadPress rad pre due to h2 lines called in PressureTotalDo */
/*H2_InterEnergy internal energy of H2 called in PressureTotalDo */
/*H2_PunchLineStuff include H2 lines in punched optical depths, etc, called from PunchLineStuff */
/*H2_EmitOut do emission from H2 - called from RTDiffuse */
/*H2_RTMake do RT for H2 - called from RTMake */
/*H2_TauInc increment optical depth for the H2 molecule, called from RTOptDepthIncre */
/*H2_TauInit initialize optical depths in H2, called from RTOptDepthInit */
/*H2_TauAver the large H2 molecule, called from RTOptDepthReset */
/*H2_LevelPops do level populations for H2, called by Hydrogenic */
/*H2_AddLines add in explicit lines from the large H2 molecule, called by lines_molecules */
/*H2_ReadDissprob read dissociation probabilities and kinetic energies for all electronic levels */
/*TODO put in excited molecular dissociation from v >=4 as in hazy 2 */
#include "cddefines.h" 
#include "path.h" 
#include "taulines.h" 
#include "abscf.h" 
#include "putline.h" 
#include "punch.h" 
#include "wind.h" 
#include "hmi.h" 
#include "rt.h" 
#include "rtescprob.h" 
#include "ipoint.h" 
#include "phycon.h" 
#include "rfield.h" 
#include "getgf.h" 
#include "physconst.h" 
#include "opacity.h" 
#include "h2.h"

/*TODO heating and cooling */
/*TODO destruction rate */

/* if this is set true then code will print energies and stop */
/*@-redef@*/
enum {DEBUG=FALSE};
/*@+redef@*/

/* when true this says for h2_create to immediately quit, and no try to do anything at all. */
#define lgQUIT FALSE

/* this is the number of electronic levels - in h2.h */
/* #define N_H2_ELEC	7*/

/* this will become energies[elec][vib][rot] */
static double ***dissprob;
static double ***disske;
static double ***energies;
static double ***Boltzmann;
static double ***Populations_LTE;
static double ***populations;
static double ***rate_in;
static double ***rate_out;
static float ***stat;

/* this will be the number of vibration levels within each elec */
/* fix this! */
/* number of vib states within electronic states from
 * >>refer	H2	energies	Abgrall, */
static long int nVib_hi[N_H2_ELEC] = {14 , 37 , 13 , 13, 9, 2 , 2};

/* dissociation energies (cm-1) for each electronic state, from Field */
/*TODO still need last three dissociation energies */
static double DissocEnergies[N_H2_ELEC] = { 36113., 118372., 118372., 118372., 118372.,0.,0. };

/* this gives the first rotational state for each electronic state - J=0 does
 * not exist when Lambda = 1 */
static long int Jlowest[N_H2_ELEC] = {0 , 0 , 1  , 1 , 0 , 1 , 1 };

/* number of rotation levels within each elec - vib */
/*lint -e785 too few init for aggregate */
long int nRot_hi[N_H2_ELEC][50]=
	/* ground, X */
	{ {31, 30, 28, 27, 25, 
		23, 22, 20, 18, 16, 
		14, 12, 10,  7,  3 } ,
	/* B */
	{25,25,25,25,25,25,25,25, 25,25,
	 25,25,25,25,25,25,25,25, 25,25,
	 25,25,25,25,25,25,25,25, 23,21,
	 19,17,15,15,11,9,7, 7},
	/* C plus */
	{ 25, 25, 25, 25, 24, 23, 21, 19, 17, 14, 12, 10, 6, 2 },
	/* C minus (the same) */
	{ 25, 25, 25, 25, 24, 23, 21, 19, 17, 15, 13, 10, 7, 2 },
	/* B primed */
	{19,17, 14, 12, 9, 8, 7, 7, 4, 1 },
	/* D plus */
	/*{2, 2, 2, 2, 2, 2, 1, 0, 1, 1, 1},*/
	{13, 10, 5},
	/* D minus */
	/*{2, 2,2, 2, 2, 2, 1, 0, 1, 1, 1} }*/
	{25 , 25 ,25 } }
	;
/*lint +e785 too few init for aggregate */

/* read energies for all electronic levels */
static void H2_ReadEnergies( long int nelec )
{
	char cdDATAFILE[N_H2_ELEC][FILENAME_PATH_LENGTH_2] = 
	{
		"H2_energy_X.dat" ,
		"H2_energy_B.dat" , 
		"H2_energy_C_plus.dat" ,
		"H2_energy_C_minus.dat" , 
		"H2_energy_B_primed.dat" , 
		"H2_energy_D_plus.dat" ,
		"H2_energy_D_minus.dat" 
	};
	FILE *ioDATA;
	char chLine[FILENAME_PATH_LENGTH_2] , 
		chFilename[FILENAME_PATH_LENGTH_2] ;
	long int i, n1, n2, n3, iVib , iRot;
	int lgEOL;

	/* check on path is file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , cdDATAFILE[nelec] );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , cdDATAFILE[nelec] );
	}

	/* now open the data file */
	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadEnergies could not open %s\n", cdDATAFILE[nelec] );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " H2_ReadEnergies could not open %s\n", cdDATAFILE[nelec]);
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in H2_ReadEnergies]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read the first line and check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadEnergies could not read first line of %s\n", cdDATAFILE[nelec]);
		puts( "[Stop in H2_ReadEnergies]" );
		cdEXIT(EXIT_FAILURE);
	}
	i = 1;
	/* level 1 magic number */
	n1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* magic number
	 * the following is the set of numbers that appear at the start of level1.dat 01 08 10 */
	if( ( n1 != 2 ) || ( n2 != 4 ) || ( n3 != 29 ) )
	{
		fprintf( ioQQQ, 
			" H2_ReadEnergies: the version of %s is not the current version.\n", cdDATAFILE[nelec] );
		fprintf( ioQQQ, 
			" I expected to find the number 2 4 29 and got %li %li %li instead.\n" ,
			n1 , n2 , n3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in H2_ReadEnergies]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read until not a comment */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
		bad_read();

	while( chLine[0]=='#' )
	{
		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			bad_read();
	}

	for( iVib=0; iVib<=nVib_hi[nelec]; ++iVib )
	{
		for( iRot=Jlowest[nelec]; iRot<=nRot_hi[nelec][iVib]; ++iRot )
		{
			i = 1;
			sscanf(chLine,"%li\t%li\t%le", &n1 , &n2 , &energies[nelec][iVib][iRot] );
			ASSERT( n1 == iVib );
			ASSERT( n2 == iRot );
			/* in atomic units, or 1 Hartree, or two rydbergs */
			if( nelec == 0 )
			{
				/* only do this for Phil Stancil's file */
				/* corrections are to get lowest rotation level to have energy of zero */
				/*energies[0][iVib][iRot] = -( energies[0][iVib][iRot]- 1.6456827E-01 ) * 219474.62;*/
				/* >>chng 02 may 17, to new Stancil file */
				energies[0][iVib][iRot] = -( energies[0][iVib][iRot]- 3.6118114E+04 );
				/*energies[0][iVib][iRot] = -energies[nelec][iVib][iRot];*/
			}
			ASSERT( energies[nelec][iVib][iRot]> 0. || (nelec==0 && iVib==0 && iRot==0 ) );
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
				bad_read();
			while( chLine[0]=='#' )
			{
				if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
					bad_read();
			}
		}
	}
	fclose( ioDATA );
}
/* read dissociation probabilities and kinetic energies for all electronic levels */
static void H2_ReadDissprob( long int nelec )
{
	char cdDATAFILE[N_H2_ELEC][FILENAME_PATH_LENGTH_2] = 
	{
		"H2_dissprob_X.dat" ,/* this does not exist and nelec == 0 is not valid */
		"H2_dissprob_B.dat" , 
		"H2_dissprob_C_plus.dat" ,
		"H2_dissprob_C_minus.dat" , 
		"H2_dissprob_B_primed.dat" , 
		"H2_dissprob_D_plus.dat" ,
		"H2_dissprob_D_minus.dat" 
	};
	FILE *ioDATA;
	char chLine[FILENAME_PATH_LENGTH_2] , 
		chFilename[FILENAME_PATH_LENGTH_2] ;
	long int i, n1, n2, n3, iVib , iRot;
	int lgEOL;

	ASSERT( nelec > 0 );

	/* check on path is file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , cdDATAFILE[nelec] );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , cdDATAFILE[nelec] );
	}

	/* now open the data file */
	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadDissprob could not open %s\n", cdDATAFILE[nelec] );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " H2_ReadDissprob could not open %s\n", cdDATAFILE[nelec]);
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in H2_ReadDissprob]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read the first line and check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadDissprob could not read first line of %s\n", cdDATAFILE[nelec]);
		puts( "[Stop in H2_ReadDissprob]" );
		cdEXIT(EXIT_FAILURE);
	}
	i = 1;
	/* level 1 magic number */
	n1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* magic number
	 * the following is the set of numbers that appear at the start of level1.dat 01 08 10 */
	if( ( n1 != 2 ) || ( n2 != 4 ) || ( n3 != 29 ) )
	{
		fprintf( ioQQQ, 
			" H2_ReadDissprob: the version of %s is not the current version.\n", cdDATAFILE[nelec] );
		fprintf( ioQQQ, 
			" I expected to find the number 2 4 29 and got %li %li %li instead.\n" ,
			n1 , n2 , n3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in H2_ReadDissprob]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read until not a comment */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
		bad_read();

	while( chLine[0]=='#' )
	{
		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			bad_read();
	}

	for( iVib=0; iVib<=nVib_hi[nelec]; ++iVib )
	{
		for( iRot=Jlowest[nelec]; iRot<=nRot_hi[nelec][iVib]; ++iRot )
		{
			i = 1;
			sscanf(chLine,"%li\t%li\t%le\t%le", 
				&n1 , &n2 , 
				&dissprob[nelec][iVib][iRot] , 
				&disske[nelec][iVib][iRot]);

			ASSERT( n1 == iVib );
			ASSERT( n2 == iRot );
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
				bad_read();
			while( chLine[0]=='#' )
			{
				if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
					bad_read();
			}
		}
	}
	fclose( ioDATA );
}


/* read transition probabilities */
static void H2_ReadTransprob( long int nelec )
{
	char cdDATAFILE[N_H2_ELEC][FILENAME_PATH_LENGTH_2] = 
	{
		"H2_transprob_X.dat" ,
		"H2_transprob_B.dat" , 
		"H2_transprob_C_plus.dat" ,
		"H2_transprob_C_minus.dat" , 
		"H2_transprob_B_primed.dat" , 
		"H2_transprob_D_plus.dat" ,
		"H2_transprob_D_minus.dat" 
	};
	FILE *ioDATA;
	char chLine[FILENAME_PATH_LENGTH_2] , 
		chFilename[FILENAME_PATH_LENGTH_2] ;
	long int i, n1, n2, n3;
	long int iVibHi , iVibLo , iRotHi , iRotLo , iElecHi , iElecLo;
	int lgEOL;

	/* check on path is file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , cdDATAFILE[nelec] );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , cdDATAFILE[nelec] );
	}

	/* now open the data file */
	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadTransprob could not open %s\n", cdDATAFILE[nelec] );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " H2_ReadTransprob could not open %s\n", cdDATAFILE[nelec]);
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in H2_ReadTransprob]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read the first line and check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadTransprob could not read first line of %s\n", cdDATAFILE[nelec]);
		puts( "[Stop in H2_ReadTransprob]" );
		cdEXIT(EXIT_FAILURE);
	}
	i = 1;
	/* level 1 magic number */
	n1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* magic number
	 * the following is the set of numbers that appear at the start of level1.dat 01 08 10 */
	if( ( n1 != 2 ) || ( n2 != 4 ) || ( n3 != 29 ) )
	{
		fprintf( ioQQQ, 
			" H2_ReadTransprob: the version of %s is not the current version.\n", cdDATAFILE[nelec] );
		fprintf( ioQQQ, 
			" I expected to find the number 2 4 29 and got %li %li %li instead.\n" ,
			n1 , n2 , n3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in H2_ReadTransprob]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read until not a comment */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
		bad_read();

	while( chLine[0]=='#' )
	{
		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			bad_read();
	}
	iVibHi = 1;
	while( iVibHi >= 0 )
	{
		double Aul;
		sscanf(chLine,"%li\t%li\t%li\t%li\t%li\t%li\t%le", 
			&iElecHi , &iVibHi ,&iRotHi , &iElecLo , &iVibLo , &iRotLo , &Aul );
		ASSERT( iElecHi == nelec );
		/* negative iVibHi says end of data */
		if( iVibHi < 0 )
			continue;

		/* check that we actually included the levels in the model representation */
		if( iVibHi <= nVib_hi[iElecHi] && 
		    iVibLo <= nVib_hi[iElecLo] && 
			iRotHi <= nRot_hi[iElecHi][iVibHi] && 
			iRotLo <= nRot_hi[iElecLo][iVibLo])
		{
			double ener = energies[iElecHi][iVibHi][iRotHi] - energies[iElecLo][iVibLo][iRotLo];

			H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul = (float)Aul;
			/* prints transitions with negative energies  -  should not happen */
			if( ener <= 0. )
			{
				fprintf(ioQQQ,"negative energy H2 transition\t%li\t%li\t%li\t%li\t%.2e\t%.2e\n", 
					iVibHi,iVibLo,iRotHi,iRotLo,Aul,
					H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN);
				broken();
			}
		}
#		if 0
		/* this prints all levels with As but without energies */
		else
		{
			fprintf(ioQQQ,"no\t%li\t%li\t%li\t%li\t%.2e\n", 
				iVibHi,iVibLo,iRotHi,iRotLo,Aul);
		}
#		endif

		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			bad_read();
		while( chLine[0]=='#' )
		{
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
				bad_read();
		}
	}
	fclose( ioDATA );
}


/*H2_Create create H2 molecules, called by ContCreatePointers */
void H2_Create(void)
{
	long int iElecHi , iElecLo ;
	long int iVibHi , iVibLo ;
	long int iRotHi , iRotLo ;
	long int iElec, iVib , iRot;

	/* this is flag set above - when true h2 code is not executed - this is way to
	 * avoid this code when it is not working */
	/* only malloc vectors one time per coreload */

	if( lgQUIT || lgH2_READ_DATA || !h2.lgH2ON )
		return;

	/* this var is in h2.h and prevents h2 from being change once committed here */
	lgH2_READ_DATA = TRUE;

	/* create space for the electronic levels */
	if( (energies = (double***)MALLOC(sizeof(double**)*(unsigned)h2.nelec ) )==NULL )
		bad_malloc();
	if( (Boltzmann = (double***)MALLOC(sizeof(double**)*(unsigned)h2.nelec ) )==NULL )
		bad_malloc();
	if( (dissprob = (double***)MALLOC(sizeof(double**)*(unsigned)h2.nelec ) )==NULL )
		bad_malloc();
	if( (disske = (double***)MALLOC(sizeof(double**)*(unsigned)h2.nelec ) )==NULL )
		bad_malloc();
	if( (Populations_LTE = (double***)MALLOC(sizeof(double**)*(unsigned)h2.nelec ) )==NULL )
		bad_malloc();
	if( (populations = (double***)MALLOC(sizeof(double**)*(unsigned)h2.nelec ) )==NULL )
		bad_malloc();
	if( (rate_in = (double***)MALLOC(sizeof(double**)*(unsigned)h2.nelec ) )==NULL )
		bad_malloc();
	if( (rate_out = (double***)MALLOC(sizeof(double**)*(unsigned)h2.nelec ) )==NULL )
		bad_malloc();
	if( (stat = (float***)MALLOC(sizeof(float**)*(unsigned)h2.nelec ) )==NULL )
		bad_malloc();

	for( iElec = 0; iElec<h2.nelec; ++iElec )
	{

		if(DEBUG)
			fprintf(ioQQQ,"elec %li highest vib= %li\n", iElec , nVib_hi[iElec] );

		ASSERT( nVib_hi[iElec] > 0 );

		/* nVib_hi is now the highest vib level before dissociation,
		 * now allocate space to hold the number of rotation levels */
		if( (energies[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			bad_malloc();
		if( (Boltzmann[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			bad_malloc();
		/* the ground electronic state is stable - these do not exist */
		if( iElec > 0 )
		{
			if( (dissprob[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
				bad_malloc();
			if( (disske[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
				bad_malloc();
		}
		if( (Populations_LTE[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			bad_malloc();
		if( (populations[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			bad_malloc();
		if( (rate_in[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			bad_malloc();
		if( (rate_out[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			bad_malloc();
		if( (stat[iElec] = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			bad_malloc();

		/* now loop over all vib levels, and find out how many rotation levels there are */
		/* ground is special since use tabulated data - there are 14 vib states,
			* ivib=14 is highest */
		for( iVib = 0; iVib <= nVib_hi[iElec] ; ++iVib )
		{
			/* lastly create the space for the rotation quantum number */
			if( (energies[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				bad_malloc();
			if( (Boltzmann[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				bad_malloc();
			if( iElec > 0 )
			{
				if( (dissprob[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
					bad_malloc();
				if( (disske[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
					bad_malloc();
			}
			if( (Populations_LTE[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				bad_malloc();
			if( (populations[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				bad_malloc();
			if( (rate_in[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				bad_malloc();
			if( (rate_out[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				bad_malloc();
			if( (stat[iElec][iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				bad_malloc();
		}
	}

	/* space for the energy vector is now malloced, must fill it in */
	for( iElec=0; iElec<h2.nelec; ++iElec )
	{
		/* get energies out of files */
		H2_ReadEnergies(iElec);

		/* get dissociation probabilities and energies - ground state is stable */
		if( iElec > 0 )
			H2_ReadDissprob(iElec);
	}

	/* ============================================= */
	/* create the main array of lines */
	if( (H2Lines = (EmLine******)MALLOC(sizeof(EmLine *****)*(unsigned)h2.nelec ) )==NULL )
		bad_malloc();

	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		if( (H2Lines[iElecHi] = (EmLine*****)MALLOC(sizeof(EmLine ****)*(unsigned)(nVib_hi[iElecHi]+1) ) )==NULL )
			bad_malloc();

		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			if( (H2Lines[iElecHi][iVibHi] = 
				(EmLine****)MALLOC(sizeof(EmLine ***)*(unsigned)(nRot_hi[iElecHi][iVibHi]+1) ) )==NULL )
				bad_malloc();

			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				 * concerned with excited electronic levels as a photodissociation process
				 * code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				if( (H2Lines[iElecHi][iVibHi][iRotHi] = 
					(EmLine***)MALLOC(sizeof(EmLine **)*(unsigned)(1) ) )==NULL )
					bad_malloc();

				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					 * but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;

					if( (H2Lines[iElecHi][iVibHi][iRotHi][iElecLo] = 
						(EmLine**)MALLOC(sizeof(EmLine *)*(unsigned)(nv+1) ) )==NULL )
						bad_malloc();

					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							/* max because cannot malloc 0 bytes */
							nr = MAX2(1,iRotHi-1);
						if( (H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo] = 
							(EmLine*)MALLOC(sizeof(EmLine)*(unsigned)(nr+1) ) )==NULL )
							bad_malloc();
					}
				}
			}
		}
	}

	/* now zero out the transition probabilities */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* NB - X is the only lower level considered here, since we are only 
				 * concerned with excited electronic levels as a photodissociation process
				 * code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					 * but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;
						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/* set inital values for each line structure */
							EmLineZero( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] );
						}
					}
				}
			}
		}
	}

	/* space for the energy vector is now malloced, must read trans probs from table */
	for( iElec=0; iElec<h2.nelec; ++iElec )
	{
		/* ground state has tabulated data */
		H2_ReadTransprob(iElec);
	}

	/* now free unused transitions to excited electronic states */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				 * concerned with excited electronic levels as a photodissociation process
				 * code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					 * but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];

						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;
						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{

							/* this is special flag for H2 - these are used in velset (in tfidle.c) to 
							* set doppler velocities for species */
							/* NB this must be kept parallel with nelem and ionstag in H2Lines EmLine struc,
							* since that struc expects to find the abundances here - abund set in hmole.c */
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].nelem = LIMELM+3;
							/* this does not mean anything for a molecule */
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].IonStg = 1;

							/* statistical weights of lower and upper levels */
							/* unlike atoms, for H2 nuclear spin is taken into account - so the
								* statistical weight of even and odd J states differ by factor of 3 - see page 166, sec par
								* >>>refer	H2	stat wght	Shull, J.M., & Beckwith, S., 1982, ARAA, 20, 163-188 */
							if( IS_ODD(iRotLo) )
							{
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo = 3.f*(2.f*iRotHi+1.f);
							}
							else
							{
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo = 2.f*iRotHi+1.f;
							}
							if( IS_ODD(iRotHi) )
							{
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi = 3.f*(2.f*iRotHi+1.f);
							}
							else
							{
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi = 2.f*iRotHi+1.f;
							}
							stat[iElecHi][iVibHi][iRotHi] = H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi;
							stat[iElecLo][iVibLo][iRotLo] = H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo;

							/* energy of the transition in wavenumbers */
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN = 
								(float)(energies[iElecHi][iVibHi][iRotHi] - energies[iElecLo][iVibLo][iRotLo]);
							/*ASSERT( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN > 0. );*/

							/*wavelength of transition in Angstroms */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN > SMALLFLOAT)
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].WLAng = 
								(float)(1.e8f/H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN);

							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyK = 
								(float)(T1CM)*H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN;

							/* energy of photon in ergs */
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyErg = 
								(float)(ERG1CM)*H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN;

							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{
								/* line redistribution function - will use complete redistribution */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].iRedisFun = ipCRD;

								/* line optical depths in direction towards source of ionizing radiation */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauIn = opac.taumin;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauCon = opac.taumin;
								/* outward optical depth */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauTot = 1e20f;


								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].damprel = 
									(float)(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul*
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].WLAng*1e-8/PI4);

								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].damp = 
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].damprel / 1e5f;

								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gf = 
									(float)(GetGF(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul,
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN,
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi ) );

								/* derive the abs coef, call to function is gf, wl (A), g_low */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].opacity = (float)(
									abscf(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gf,
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN,
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo));
							}
						}
					}
				}
			}
		}
	}

	/* at this stage the full electronic, vibration, and rotation energies have been defined,
	 * this is an option to print the energies */
	{
		/* set following TRUE to get printout, FALSE to not print energies */
		if( DEBUG )
		{
			/* print title for quantum numbers and energies */
			/*fprintf(ioQQQ,"elec\tvib\trot\tenergy\n");*/
			for( iElec=0; iElec<h2.nelec; ++iElec )
			{
				/* now must specify the number of rotation levels within the vib levels */
				for( iVib=0; iVib<=nVib_hi[iElec]; ++iVib )
				{
					for( iRot=0; iRot<=nRot_hi[iElec][iVib]; ++iRot )
					{
						fprintf(ioQQQ,"%li\t%li\t%li\t%.5e\n",
							iElec, iVib, iRot ,
							energies[iElec][iVib][iRot]);
					}
				}
			}
			/* this will exit the program after printing the level energies */
			cdEXIT(EXIT_SUCCESS);
		}
	}
	return;
}

/* set the ipCont struc element for the H2 molecule, called by ContCreatePointers */
void H2_ContPoint( void )
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;

	if( !h2.lgH2ON )
		return;

	/* set array index for line energy within continuum array */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				 * concerned with excited electronic levels as a photodissociation process
				 * code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					 * but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ipCont = 
								(int)ipLineEnergy(
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN * WAVNRYD , 
								"H2  " , 0 );
						}
					}
				}
			}
		}
	}
	return;
}

/* ===================================================================== */
/* radiative acceleration due to H2 called in forlin */
double H2_Accel(void)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;
	double h2_drive;

	if( !h2.lgH2ON )
		return(0.);

	/* this routine computes the line driven radiative acceleration
	 * due to H2 molecule*/

	h2_drive = 0.;
	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{
								h2_drive += H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].pump*
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyErg*
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc;
							}
						}
					}
				}
			}
		}
	}
	return h2_drive;
}

/* ===================================================================== */
/* rad pre due to h2 lines called in PressureTotalDo */
double H2_RadPress(void)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;
	double press;

	/* will be used to check on size of opacity, was capped at this value */
	float smallfloat=SMALLFLOAT*10.f;

	if( !h2.lgH2ON )
		return(0.);

	press = 0.;
	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{
								if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi > smallfloat &&
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc > smallfloat )
								{
									double RadPres1 = 5.551e-2*(
										powi(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN/1.e6,4))*
										(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi/
										H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi)/
										(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo/
										H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo)*
										RT_LineWidth(&H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo])*2.;

										press += RadPres1;
								}
							}
						}
					}
				}
			}
		}
	}
	return press;
}

/* ===================================================================== */
/* internal energy of H2 called in PressureTotalDo */
double H2_InterEnergy(void)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;
	double energy;

	if( !h2.lgH2ON )
		return(0.);

	energy = 0.;
	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{
								energy += 
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi* 
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyErg;
							}
						}
					}
				}
			}
		}
	}
	return energy;
}

/* ===================================================================== */
/* punch line data for H2 molecule */
void H2_PunData(
	/* io unit for punch */
	FILE* ioPUN ,
	/* punch all levels if true, only subset if false */
	int lgDoAll )
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;

#	ifdef DEBUG_FUN
	fputs( "<+>H2_PunData()\n", debug_fp );
#	endif

	if( !h2.lgH2ON )
		return;

	if( lgDoAll )
	{
		fprintf( ioQQQ, 
			" H2_PunData ALL option not implemented in H2_PunData yet 1\n" );
		puts( "[Stop in H2_PunData]" );
		cdEXIT(EXIT_FAILURE);
	}
	else
	{

		/* loop over all possible lines */
		for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
		{
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					/* now the lower levels */
					/* NB - X is the only lower level considered here, since we are only 
					* concerned with excited electronic levels as a photodissociation process
					* code exists to relax this assumption - simply change following to iElecHi */
					long int lim_elec_lo = 0;
					for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
					{
						/* want to include all vib states in lower level if different elec level,
						* but only lower vib levels if same elec level */
						long int nv = nVib_hi[iElecLo];
						if( iElecLo==iElecHi )
							nv = iVibHi;
						for( iVibLo=0; iVibLo<=nv; ++iVibLo )
						{
							long nr = nRot_hi[iElecLo][iVibLo];
							if( iElecLo==iElecHi && iVibHi==iVibLo )
								nr = iRotHi-1;

							for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
							{
								if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
								{
									/* print quantum indices */
									fprintf(ioPUN,"%2li %2li %2li %2li %2li %2li ",
										iElecHi,iVibHi,iRotHi,iElecLo,iVibLo,iRotLo );
									Punch1LineData( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] , ioPUN);
								}
							}
						}
					}
				}
			}
		}
		fprintf( ioPUN , "\n");
	}

#	ifdef DEBUG_FUN
	fputs( " <->H2_PunData()\n", debug_fp );
#	endif
	return;
}

/* include H2 lines in punched optical depths, etc, called from PunchLineStuff */
void H2_PunchLineStuff( FILE * io , float xLimit  , long index)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;

	if( !h2.lgH2ON )
		return;

	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{
								pun1Line( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] , io , xLimit  , index);
							}
						}
					}
				}
			}
		}
	}
}

/* do emission from H2 - called from RTDiffuse */
void H2_EmitOut(double VolFac, 
				double ref)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo , ip;

#	ifdef DEBUG_FUN
	fputs( "<+>H2_EmitOut()\n", debug_fp );
#	endif

	if( !h2.lgH2ON )
		return;

	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{
								/* pointer to line energy in continuum array */
								ip = H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ipCont;

								rfield.outlin[ip-1] += (float)(
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].phots*
									VolFac*opac.tmn[ip-1]*
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ColOvTot);

								rfield.reflin[ip-1] += (float)(
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].phots * ref);
							}
						}
					}
				}
			}
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->H2_EmitOut()\n", debug_fp );
#	endif
	return;
}

/* do RT for H2 - called from RTMake */
void H2_RTMake( int lgDoEsc )
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;

#	ifdef DEBUG_FUN
	fputs( "<+>H2_RTMake()\n", debug_fp );
#	endif

	if( !h2.lgH2ON )
		return;

	/* this routine drives calls to make RT relations for H2 molecule */
	if( wind.windv == 0. )
	{
		/* static soluiton */
		/* loop over all possible lines */
		for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
		{
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					/* now the lower levels */
					/* NB - X is the only lower level considered here, since we are only 
					* concerned with excited electronic levels as a photodissociation process
					* code exists to relax this assumption - simply change following to iElecHi */
					long int lim_elec_lo = 0;
					for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
					{
						/* want to include all vib states in lower level if different elec level,
						* but only lower vib levels if same elec level */
						long int nv = nVib_hi[iElecLo];
						if( iElecLo==iElecHi )
							nv = iVibHi;
						for( iVibLo=0; iVibLo<=nv; ++iVibLo )
						{
							long nr = nRot_hi[iElecLo][iVibLo];
							if( iElecLo==iElecHi && iVibHi==iVibLo )
								nr = iRotHi-1;

							for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
							{
								if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
								{
									RTMakeStat( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] , lgDoEsc );
								}
							}
						}
					}
				}
			}
		}
	}
	else
	{
		/* loop over all possible lines */
		for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
		{
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					/* now the lower levels */
					/* NB - X is the only lower level considered here, since we are only 
					* concerned with excited electronic levels as a photodissociation process
					* code exists to relax this assumption - simply change following to iElecHi */
					long int lim_elec_lo = 0;
					for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
					{
						/* want to include all vib states in lower level if different elec level,
						* but only lower vib levels if same elec level */
						long int nv = nVib_hi[iElecLo];
						if( iElecLo==iElecHi )
							nv = iVibHi;
						for( iVibLo=0; iVibLo<=nv; ++iVibLo )
						{
							long nr = nRot_hi[iElecLo][iVibLo];
							if( iElecLo==iElecHi && iVibHi==iVibLo )
								nr = iRotHi-1;

							for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
							{
								if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
								{
									RTMakeWind( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] , lgDoEsc );
								}
							}
						}
					}
				}
			}
		}
	}
}

/* increment optical depth for the H2 molecule, called from RTOptDepthIncre */
void H2_TauInc(void)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo;

	if( !h2.lgH2ON )
		return;

	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{
								RTLineTauInc( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] );
							}
						}
					}
				}
			}
		}
	}
}


/* initialize optical depths in H2, called from RTOptDepthInit */
void H2_TauInit( void )
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo;

	if( !h2.lgH2ON )
		return;

	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{
								/* inward optical depth */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauIn = opac.taumin;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauCon = opac.taumin;

								/* outward optical depth */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauTot = 1e20f;

								/* escape probability */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pesc = 1.;

								/* inward part of line */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].FracInwd = 1.;

								/* destruction probability */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pdest = 0.;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pelec_esc = 0.;

								/* line pumping rate */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].pump = 0.;

								/* population of lower level with correction for stim emission */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc = 0.;

								/* population of lower level */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo = 0.;

								/* population of upper level */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi = 0.;

								/* following two heat exchange excitation, deexcitation */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].cool = 0.;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].heat = 0.;

								/* intensity of line */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].xIntensity = 0.;

								/* opacity in line */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].dTau = 0.;
							}
						}
					}
				}
			}
		}
	}
}

/* the large H2 molecule, called from RTOptDepthReset */
void H2_TauAver( void )
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo;

	if( !h2.lgH2ON )
		return;

	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{
								/* inward optical depth */
								RTTauUpdate( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] , 0.5);
							}
						}
					}
				}
			}
		}
	}
}

/* do level populations for H2, called by Hydrogenic */
void H2_LevelPops( void )
{
	static float TeUsed=-1.f;
	static double part_fun = 0.;
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo;

	/* H2 not on, so space not allocated */
	if( !h2.lgH2ON )
		return;

	if( hmi.htwo/phycon.hden < h2.H2_to_H_limit )
	{
		/* zero everything out then quit */
		/* loop over all possible lines */
		for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
		{
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					/* now the lower levels */
					/* NB - X is the only lower level considered here, since we are only 
					* concerned with excited electronic levels as a photodissociation process
					* code exists to relax this assumption - simply change following to iElecHi */
					long int lim_elec_lo = 0;
					for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
					{
						/* want to include all vib states in lower level if different elec level,
						* but only lower vib levels if same elec level */
						long int nv = nVib_hi[iElecLo];
						if( iElecLo==iElecHi )
							nv = iVibHi;
						for( iVibLo=0; iVibLo<=nv; ++iVibLo )
						{
							long nr = nRot_hi[iElecLo][iVibLo];
							if( iElecLo==iElecHi && iVibHi==iVibLo )
								nr = iRotHi-1;

							for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
							{
								if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
								{
									/* population of lower level with correction for stim emission */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc = 0.;

									/* population of lower level */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo = 0.;

									/* population of upper level */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi = 0.;

									/* following two heat exchange excitation, deexcitation */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].cool = 0.;
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].heat = 0.;

									/* intensity of line */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].xIntensity = 0.;

									/* opacity in line */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].dTau = 0.;

									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].phots = 0.;
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ots = 0.;
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].AovTot = 0.;
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ColOvTot = 0.;
								}
							}
						}
					}
				}
			}
		}
		/* end of zero abundance branch */
		return;
	}

	/* check whether we need to update the Boltzmann factors */
	/*lint -e777 float test equality */
	if( phycon.te != TeUsed )
	/*lint +e777 float test equality */
	{
		/* loop over all remaining levels */
		for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
		{
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				/* NB this loop is unlike all the others in this file since it does not include
				 * the lowest upper rotation state - rest are RotHi=Jlowest */
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					Boltzmann[iElecHi][iVibHi][iRotHi] = 
						sexp( energies[iElecHi][iVibHi][iRotHi] / phycon.te_wn );
					/* sum the partition funciton */
					part_fun += Boltzmann[iElecHi][iVibHi][iRotHi] * stat[iElecHi][iVibHi][iRotHi];
					ASSERT( part_fun > 0 );
				}
			}
		}
		/* loop over all possible lines and set LTE populations, which only depend on Te*/
		for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
		{
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					/* now the lower levels */
					/* NB - X is the only lower level considered here, since we are only 
					* concerned with excited electronic levels as a photodissociation process
					* code exists to relax this assumption - simply change following to iElecHi */
					long int lim_elec_lo = 0;

					/* upper level populations in LTE */
					double PopHi = Boltzmann[iElecHi][iVibHi][iRotHi] * stat[iElecHi][iVibHi][iRotHi] *hmi.htwo; 
					ASSERT( PopHi >= 0.);
					Populations_LTE[iElecHi][iVibHi][iRotHi] = PopHi;

					/* now loop over all lower levels */
					for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
					{
						/* want to include all vib states in lower level if different elec level,
						* but only lower vib levels if same elec level */
						long int nv = nVib_hi[iElecLo];
						if( iElecLo==iElecHi )
							nv = iVibHi;
						for( iVibLo=0; iVibLo<=nv; ++iVibLo )
						{
							long nr = nRot_hi[iElecLo][iVibLo];
							if( iElecLo==iElecHi && iVibHi==iVibLo )
								nr = iRotHi-1;

							/* zero out these rates */
							rate_in[iElecHi][iVibHi][iRotHi] = 0.;
							rate_out[iElecHi][iVibHi][iRotHi] = 0.;

							for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
							{
								double PopLo = Boltzmann[iElecLo][iVibLo][iRotLo] * stat[iElecLo][iVibLo][iRotLo]/part_fun *hmi.htwo; 

								/* the LTE populations of the levels */
								Populations_LTE[iElecLo][iVibLo][iRotLo] = PopLo;

								/* LTE population of lower level with correction for stim emission */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc = PopLo - PopHi*
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo / 
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi;

								/* LTE population of lower level */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo = PopLo;

								/* LTE population of upper level */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi = PopHi;

								/* very important - collision strength */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].cs = 0.;
							}
						}
					}
				}
			}
		}
	}

	/* loop over all possible pumping routes to get radiative excitation rate */
	/* for now assume that levels within X are in LTE, and just use Bolzmann factors and
	 * statistical weights */
	for( iElecHi=1; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				 * concerned with excited electronic levels as a photodissociation process
				 * code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{

								/* this is the rate [cm-3 s-1] electrons move into the upper level */
								rate_in[iElecHi][iVibHi][iRotHi] +=
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].pump *
									Populations_LTE[iElecLo][iVibLo][iRotLo] * 
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo/
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi;

								/* this is the rate [s-1] electrons move out of the upper level */
								rate_out[iElecHi][iVibHi][iRotHi] +=
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].pump +
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul*
									(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pesc + 
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pelec_esc + 
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pdest);

								/* this term is spontaneous dissociation of excited elec states into 
								 * the X continuum */
								if( iElecHi> 0 )
									rate_out[iElecHi][iVibHi][iRotHi] += dissprob[iElecHi][iVibHi][iRotHi];
							}
						}
					}
				}
				/* above was the last of the lower level populations loops - 
				 * so all routes into and out of the upper elec levels have been defined 
				 * we can now get the population [cm-3] */
				populations[iElecHi][iVibHi][iRotHi] = rate_in[iElecHi][iVibHi][iRotHi] /
					MAX2( SMALLFLOAT , rate_out[iElecHi][iVibHi][iRotHi]);
				ASSERT( populations[iElecHi][iVibHi][iRotHi] >= 0. );
			}
		}
	}	

	/* for now assume that X has populations in LTE */
	iElecHi = 0;
	for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
	{
		for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
		{
			populations[iElecHi][iVibHi][iRotHi] = Populations_LTE[iElecHi][iVibHi][iRotHi];
		}
	}

	/* loop over all possible lines and set populations, and quantities that depend on escape prob, dest, etc */
	for( iElecHi=0; iElecHi<h2.nelec; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi = 
								populations[iElecHi][iVibHi][iRotHi]; 
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo = 
								populations[iElecLo][iVibLo][iRotLo]; 
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc = 
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo - 
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi*
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo / 
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi;
							ASSERT(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi >= 0. &&
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo >=  0.);

							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )
							{

								/* following two heat exchange excitation, deexcitation */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].cool = 0.;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].heat = 0.;

								/* number of photons in the line */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].phots = 
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul * 
									(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pesc + 
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pelec_esc) * 
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi; 

								/* intensity of line */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].xIntensity = 
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].phots *
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyErg;

								/* ots destruction rate */fixit();/* remove these asserts */
								ASSERT( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi >= 0.);
								ASSERT( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pdest >= 0. &&
								        H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pdest <= 1. );
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ots = (float)(
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul *
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi *
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pdest);

								/* dump the ots rate into the stack */
								RT_OTS_AddLine(
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ots,
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ipCont );


								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].AovTot = 1.;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ColOvTot = 0.;
							}
						}
					}
				}
			}
		}
	}	

	return;
}

/* add in explicit lines from the large H2 molecule, called by lines_molecules */
void H2_AddLines(void)
{
	/* these are the quantum designations of the lines we will output */
#	define N_H2_LINES_OUTPUT	6
	long int ip_H2_lines_output[N_H2_LINES_OUTPUT*6] = {
	/* R branch - J decreases by 1 in emission line */
	/* P branch - J increases by 1 in emission line */
	/* Q branch - J changes   by 0 in emission line */
	/* S branch - J decreases by 2 in emission line */
	0,0,2, 0,0,0 , /* S(0) */
	0,0,3, 0,0,1 , /* S(1) */
	0,0,4, 0,0,2 , /* S(2) */
	0,0,5, 0,0,3 , /* S(3) */
	0,0,6, 0,0,4 , /* S(4) */
	0,1,3, 0,0,1 } /* 1-0 S(1) */;

	long int i;

	/* H2 not on, so space not allocated */
	if( !h2.lgH2ON )
		return;

	for( i=0; i<N_H2_LINES_OUTPUT; ++i )
	{
		/* the upper state */
		long int iElecHi = ip_H2_lines_output[i*6],
			iVibHi = ip_H2_lines_output[i*6+1],
			iRotHi = ip_H2_lines_output[i*6+2];
		/* the lower state */
		long int iElecLo = ip_H2_lines_output[i*6+3],
			iVibLo = ip_H2_lines_output[i*6+4],
			iRotLo = ip_H2_lines_output[i*6+5];

		/* dump the line */
		PutLine(&H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo]);
	}
}

