/*HeatSum evaluate heating and secondary ionization for current conditions */
/*HeatZero is called by ConvIonizeOpacityDo */
#include "cddefines.h"
#include "physconst.h"
#include "ionrange.h"
#include "cooling.h"
#include "heat.h"
#include "heavy.h"
#include "trace.h"
#include "secondaries.h"
#include "coolheavy.h"
#include "iso.h"
#include "hevmolec.h"
#include "hmi.h"
#include "hydrogenic.h"
#include "abundances.h"
#include "ionrec.h"
#include "ionfracs.h"
#include "phycon.h"
#include "numderiv.h"
#include "holod.h"
#include "chargtran.h"
#include "heatsum.h"
/* this is the faintest relative heating we will print */
#define	FAINT_HEAT	0.02

void HeatSum( void )
{
	/* use to dim some vectors */
#	ifdef NDIM
#		undef NDIM
#	endif
#	define NDIM 40

	/* secondary ionization and exciation due to compton scattering */
	float seccmp , scmpla;

	/* ionization and excitation rates from hydrogen itself */
	double SeconIoniz_iso[NISO] , 
		SeconExcit_iso[NISO] ;

	long int i, 
	  ion,
	  ipnt, 
	  ipsave[NDIM], 
	  j, 
	  jpsave[NDIM], 
	  limit ,
	  nelem ;
	double HeatingLo ,
		HeatingHi ,
		secmet ,
		smetla;
	long ipISO,
		ns;
	static long int nhit=0, 
	  nzSave=0;
	double Photo_Heat,
		OtherHeat , 
		deriv, 
		oldfac, 
		save[NDIM];
	static double oldheat=0., 
	  oldtemp=0.;

	float SaveOxygen1 , SaveCarbon1 ;

	/* routine to sum up total heating, and print agents if needed
	 * it also computes the true deriv, dH/dT */
	double xe;

#	ifdef DEBUG_FUN
	fputs( "<+>HeatSum()\n", debug_fp );
#	endif

	/*******************************************************************
	 *
	 * reevaluate the secondary ionization and excitation rates 
	 *
	 *******************************************************************/

	/* phycon.ElecFrac is electron fraction */
	/* XE = EDEN/(total neutrals, roughly H)
	 * analytic fits to Shull and Van Steenberg (1985; Ap.J. 298, 268).
	 * lgSecOFF turns off secondary ionizations, sets heating effic to 100% */

	if( Secondaries.lgSecOFF || phycon.ElecFrac > 0.95 )
	{
		Secondaries.heatef = 1.;
		Secondaries.efionz = 0.;
		Secondaries.exctef = 0.;
	}
	else
	{

		/* at very low ionization - as per 
		 * >>>refer	sec	ioniz	Xu and McCray 1991, Ap.J. 375, 190.
		 * everything goes to asymptote that is not present in Shull and
		 * Van Steenberg - do this by not letting phycon.ElecFrac fall below 1e-4 */
		xe = MAX2(phycon.ElecFrac,1e-4);

		/* this is heating efficiency for high-energy photoejections.  It is the ratio
		 * of the final heat added to the energy of the photo electron.  Fully
		 * ionized, this is unity, and less than unity for neutral media.  
		 * It is a scale factor that multiplies the
		 * high energy heating rate */
		Secondaries.heatef = (float)(0.9971*(1. - pow(1. - pow(xe,0.2663),1.3163)));

		/* secondary ionizations - this is the number of secondary ionizations
		 * produced for each ryd of photo electron energy.  It only multiplies the high-energy
		 * heating rate.  It is zero for a fully ionized gas */
		Secondaries.sec2prim = (float)(0.3908*pow(1. - pow(xe,0.4092),1.7592));

		/* by dividing by the energy of one ryd, we can use this factor to multiply
		 * the heating rate in ergs */
		Secondaries.efionz = (float)(Secondaries.sec2prim/EN1RYD);

		/* secondary excitation of L-alpha, actually all Lyman lines */
		Secondaries.exctef = (float)(0.4766*pow(1. - pow(xe,0.2735),1.5221));
		Secondaries.exctef /= (float)EN1RYD;
	}

	/*******************************************************************
	 *
	 * get total heating from all species
	 *
	 *******************************************************************/

	/* get total heating */
	Photo_Heat = 0.;

	/* add in effects of high-energy opacity of CO using C and O atomic opacities
	 * k-shell and valence photo of C and O in CO is not explicitly counted elsewhere
	 * this trick roughly accounts for it*/
	SaveOxygen1 = xIonFracs[ipOXYGEN][0];
	SaveCarbon1 = xIonFracs[ipCARBON][0];

	/* atomic C and O will include CO during the heating sum loop */
	xIonFracs[ipOXYGEN][0] += hevmolec.hevmol[ipCO];
	xIonFracs[ipCARBON][0] += hevmolec.hevmol[ipCO];

	/* this will hold cooling due to metal collisional ionization */
	CoolHeavy.colmet = 0.;
	/* metals secondary ionization, Lya excitation */
	secmet = 0.;
	smetla = 0.;
	for( ipISO=0; ipISO<NISO; ++ipISO )
	{
		SeconIoniz_iso[ipISO] = 0.;
		SeconExcit_iso[ipISO] = 0.;
	}

	/* this loop includes hydrogenic, which is special case due to presence
	 * of substantial excited state populations */
	for( nelem=0; nelem<LIMELM; ++nelem)
	{
		if( abundances.lgElmtOn[nelem] )
		{
			/* sum heat over all stages of ionization that exist */
			/* h-like and he-like are special because full atom always done, 
			 * can be substantial,  pops in excited states, with little in ground 
			 * (true near lte), these are done in following loop */

			/* >>chng 02 mar 27 change logic to do any number of iso seq */
			/*limit = MIN2( IonRange.IonHigh[nelem] , nelem );*/
			limit = MIN2( IonRange.IonHigh[nelem] , nelem+1-NISO );
			/* >>chng 99 may 08, this had included hydrogenic, but without correction
			 * for fraction in excited states, so overestimated heating in dense
			 * models near lte */
			for( ion=IonRange.IonLow[nelem]; ion < limit; ion++ )
			{
				/* this will be heating for a single stage of ionization */
				HeatingLo = 0.;
				HeatingHi = 0.;
				for( ns=0; ns < Heavy.nsShells[nelem][ion]; ns++ )
				{
					/* heating by various sub-shells */
					HeatingLo += ionrec.PhotoRate_Ground[nelem][ion][ns][1];
					HeatingHi += ionrec.PhotoRate_Ground[nelem][ion][ns][2];
				}

				/* total heating, all shells, for this stage of ionization */
				heat.heating[nelem][ion] = xIonFracs[nelem][ion]* 
					(HeatingLo + HeatingHi*Secondaries.heatef);

				/* add to total heating */
				Photo_Heat += heat.heating[nelem][ion];

				/* Cooling due to collisional ionization of heavy elements
				 * CollidRate[nelem][ion][1] is cooling, erg/s/atom, eval in CollidIonize */
				CoolHeavy.colmet += xIonFracs[nelem][ion]*ionrec.CollIonRate_Ground[nelem][ion][1];

				/* secondary ionization rate,  */
				secmet += HeatingHi*Secondaries.efionz* xIonFracs[nelem][ion];

				/* LA excit rate, =0 if ionized */
				smetla += HeatingHi*Secondaries.exctef* xIonFracs[nelem][ion];
			}

			/* heating - cooling due to excited states is in iso_cool */
			/* need max here since limit is negative for h he */
			for( ion=MAX2(0,limit); ion < IonRange.IonHigh[nelem] ; ion++ )
			{
				/* this is the iso sequence */
				ipISO = nelem-ion;
				/* this will be heating for a single stage of ionization */
				HeatingLo = 0.;
				HeatingHi = 0.;
				/* the outer shell contains the compton recoil part */
				for( ns=0; ns < Heavy.nsShells[nelem][ion]; ns++ )
				{
					/* heating by low energy, and then high energy, light */
					HeatingLo += ionrec.PhotoRate_Ground[nelem][ion][ns][1];
					HeatingHi += ionrec.PhotoRate_Ground[nelem][ion][ns][2];
				}

				/* net heating */
				heat.heating[nelem][ion] = xIonFracs[nelem][ion+1]*
				  iso.Pop2Ion[ipISO][nelem][0]*(HeatingLo + HeatingHi*Secondaries.heatef);

				/* add to total heating */
				Photo_Heat += heat.heating[nelem][ion];

				/* Cooling due to collisional ionization of h-like and he-like species
				 * CollidRate[nelem][ion][1] is cooling, erg/s/atom, eval in CollidIonize
				CollIonCool_iso[ipISO] += xIonFracs[nelem][ion+1]*iso.Pop2Ion[ipISO][nelem][0]*
					ionrec.CollIonRate_Ground[nelem][ion][1]; */

				/* secondary ionization rate,  */
				SeconIoniz_iso[ipISO] += HeatingHi*Secondaries.efionz* 
					iso.Pop2Ion[ipISO][nelem][0]*xIonFracs[nelem][ion+1]/phycon.CollidDensity;

				/* LA excit rate, =0 if ionized */
				SeconExcit_iso[ipISO] += HeatingHi*Secondaries.exctef* 
					iso.Pop2Ion[ipISO][nelem][0]*xIonFracs[nelem][ion+1]/phycon.CollidDensity;

				ASSERT( SeconIoniz_iso[ipISO]>=0. && 
					SeconExcit_iso[ipISO]>=0.);
			}

			/* make sure stages of ionization with no abundances,
			 * also has no heating */
			for( ion=0; ion<IonRange.IonLow[nelem]; ion++ )
			{
				ASSERT( heat.heating[nelem][ion] == 0. );
			}
			for( ion=IonRange.IonHigh[nelem]+1; ion<nelem+1; ion++ )
			{
				ASSERT( heat.heating[nelem][ion] == 0. );
			}
		}
	}

	/* now reset the abundances */
	xIonFracs[ipOXYGEN][0] = SaveOxygen1;
	xIonFracs[ipCARBON][0] = SaveCarbon1;

	/* convert secmet to proper final units */
	secmet /= phycon.CollidDensity;
	smetla /= phycon.CollidDensity;

	/* >>chng 01 dec 20, do full some over all secondaries */
	/* bound compton recoil heating */
	/* >>chng 02 mar 28, save heating in this var rather than heating[0][18] 
	 * since now saved in photo heat 
	 * this is only used for a printout and in lines, not as heat since already counted*/
	ionrec.CompRecoilHeatLocal = 0.;
	for( nelem=0; nelem<LIMELM; ++nelem )
	{
		for( ion=0; ion<nelem+1; ++ion )
		{
			ionrec.CompRecoilHeatLocal += 
				ionrec.CompRecoilHeatRate[nelem][ion]*Secondaries.heatef*xIonFracs[nelem][ion];
		}
	}

	/* find heating due to charge transfer */
	heat.heating[0][24] = ChargTranSumHeat();

	/* heating due to pair production */
	heat.heating[0][21] = 
		ionrec.PairProducPhotoRate[2]*Secondaries.heatef*(phycon.hden + 4.F*abundances.gas_phase[ipHELIUM]);
	/* last term above is number of nuclei in helium */

	/* this is heating due to fast neutrons, assumed to secondary ionize */
	heat.heating[0][20] = 
		ionrec.xNeutronHeatRate*Secondaries.heatef*phycon.hden;

	/* turbulent heating, assumed to be a non-ionizing heat agent, added here */
	heat.heating[0][20] += ionrec.ExtraHeatRate;

	/* cosmic ray heating, counts as non-ionizing heat since already result of cascade,
	 * this was per secondary ionization, so must multiply by number of sec that
	 * occur per primary */
	heat.heating[0][20] += 
		ionrec.CosRayHeatRate*
		(xIonFracs[ipHYDROGEN][0] + xIonFracs[ipHELIUM][0] + hmi.htwo)*Secondaries.sec2prim;

	/* now sum up all heating agents not included in sum over normal bound levels above */
	OtherHeat = 0.;
	for( nelem=0; nelem<LIMELM; ++nelem)
	{
		/* we now have ionization solution for this element,sum heat over
		 * all stages of ionization that exist */
		/* >>>chng 99 may 08, following loop had started at nelem+3, and so missed [1][0],
		 * which is excited states of hydrogenic species.  increase this limit */
		for( i=nelem+1; i < LIMELM; i++ )
		{
			OtherHeat += heat.heating[nelem][i];
		}
	}

	heat.htot = OtherHeat + Photo_Heat;

	/* check whether Lya heating is important
	 * if so then only very small temp changes can be made */
	if( heat.heating[0][10]/heat.htot > 0.15 )
	{
		static long int nPrinted=0;
		hydro.lgHCoolng = TRUE;
		if(nPrinted==0)
			fprintf(ioQQQ," lya heating hittt\n");
		++nPrinted;
	}
	else
	{
		hydro.lgHCoolng = FALSE;
	}

	/* add on line heating to this array, heatl was evaluated in sumcool
	 * not zero, because of roundoff error */
	if( cooling.heatl/cooling.ctot < -1e-15 )
	{
		fprintf( ioQQQ, " HeatSum gets negative line heating,%10.2e%10.2e this is insane.\n", 
		  cooling.heatl, cooling.ctot );

		fprintf( ioQQQ, " this is zone%4ld\n", nzone );
		ShowMe();
		puts( "[Stop in sumheat]" );
		cdEXIT(EXIT_FAILURE);
	}

	/*******************************************************************
	 *
	 * secondary ionization and excitation rates 
	 *
	 *******************************************************************/

	/* secondary rates due to hydrogen */
#	if 0
	HyIon = (float)(
		ionrec.PhotoRate_Ground[ipHYDROGEN][0][0][2]*Secondaries.efionz*xIonFracs[ipHYDROGEN][0]/phycon.CollidDensity);
	HyExc =(float)(
		ionrec.PhotoRate_Ground[ipHYDROGEN][0][0][2]*Secondaries.exctef*1.3333F*xIonFracs[ipHYDROGEN][0]/phycon.CollidDensity);

	/* secondary ioinzation due to helium, the contents of the PhotoRate array is the
	 * heating per particle due to high-energy light, */
	SeconIoniz_iso[ipHE_LIKE] =(float)(
		(ionrec.PhotoRate_Ground[ipHELIUM][0][0][2]*Secondaries.efionz*xIonFracs[ipHELIUM][0] +
		ionrec.PhotoRate_Ground[ipHELIUM][1][0][2]*Secondaries.efionz*xIonFracs[ipHELIUM][1])/phycon.CollidDensity);

	/* total secondary excitation of Lya due to helium, in all following, fac of 1.33 is
	 * asymptotic limit for all hydrogen lines above Lya */
	SeconExcit_iso[ipHE_LIKE] =(float)(
		(ionrec.PhotoRate_Ground[ipHELIUM][0][0][2]*Secondaries.exctef*1.3333F*xIonFracs[ipHELIUM][0] +
		ionrec.PhotoRate_Ground[ipHELIUM][1][0][2]*Secondaries.exctef*1.3333F*xIonFracs[ipHELIUM][1])/
		phycon.CollidDensity);
#	endif

	/* the terms seccmp & scmpla contain energies added in highen.  
	 * The only significant one is usually bound compton heating except when 
	 * cosmic rays are present */

	/* add on heating due to pair production, ionization then excitation */
	seccmp = (float)(
		ionrec.PairProducPhotoRate[2]*Secondaries.efionz*
		(phycon.hden + 4.F*abundances.gas_phase[ipHELIUM])/phycon.CollidDensity);

	scmpla = (float)(
		ionrec.PairProducPhotoRate[2]*Secondaries.exctef*1.3333F*
		(phycon.hden + 4.F*abundances.gas_phase[ipHELIUM])/phycon.CollidDensity);

	/* this is heating due to fast neutrons, assumed to secondary ionize */
	seccmp += (float)(
		ionrec.xNeutronHeatRate*Secondaries.efionz*phycon.hden/phycon.CollidDensity);

	scmpla += (float)(
		ionrec.xNeutronHeatRate*Secondaries.exctef*1.3333F*phycon.hden/
		phycon.CollidDensity);

	/* cosmic ray ionization */
	/* >>>chng 99 apr 29, term in PhotoRate was not present */
	seccmp += (float)(
		/* this term is cosmic ray ionization, set in highen, did not multiply by
		 * collider density in highen, so do not divide by it here */
		/* >>chng 99 jun 29, added efionz to cosmic ray rate, also multiply
		 * by number of secondaries per primary*/
		ionrec.CosRayIonRate*Secondaries.sec2prim +
		/* this is the cosmic ray heating rate */
		ionrec.CosRayHeatRate*Secondaries.efionz);

	/* cosmic ray Lya excitation rate */
	scmpla += (float)(
		ionrec.CosRayHeatRate*Secondaries.exctef*1.3333f*
		/* multiply by atomic H and He densities */
		(xIonFracs[ipHYDROGEN][0] + xIonFracs[ipHELIUM][0])/phycon.CollidDensity);

#	if 0
	/* >>chng 02 mar 31, compton recoil added to valence photo term,
	 * so already has been counted */
	/* >>chng 01 dec 20 add full range of atoms */
	/* bound compton heating */
	for( nelem=0; nelem<LIMELM; ++nelem )
	{
		for( ion=0; ion<nelem+1; ++ion )
		{
			seccmp += 
				(float)(ionrec.CompRecoilHeatRate[nelem][ion]*Secondaries.efionz*xIonFracs[nelem][ion]/
						phycon.CollidDensity);

			/* recoil ionization production of Lya */
			scmpla += 
				(float)(ionrec.CompRecoilHeatRate[nelem][ion]*Secondaries.exctef*1.3333F*xIonFracs[nelem][ion]/
						phycon.CollidDensity);
		}
	}
#	endif

	/* find total suprathermal collisional ionization rate
	 * CSUPRA is col ionize rate from all processes (inv sec)
	 * X12tot is LA excit rate, inv sec
	 * SECCMP evaluated in HIGHEN, =ioniz rate for cosmic rays, sec bound */

	/* option to force secondary ionization rate, normally false */
	if( Secondaries.lgCSetOn )
	{
		Secondaries.csupra = Secondaries.SetCsupra;
		Secondaries.x12tot = Secondaries.SetCsupra;
	}
	else
	{
		double facold , facnew;
		/* CollidDensity is total neutral particle density */
		/* >>chng 02 may 30, dense neutral model had eden - csupra oscil -
		 * try taking average of old and new */
		if( Secondaries.csupra / MAX2(SMALLFLOAT , hydro.DestRate[ipHYDROGEN][ipH1s] ) > 0.1 )
		{
			/* supra are dominant H destruction - make small changes */
			facold = 0.9;
		}
		else
		{
			/* secondaries are not important - only use new */
			facold = 0.;
		}
		facnew = 1. - facold;
		Secondaries.csupra = (float)(Secondaries.csupra* facold + facnew*
			(seccmp + 
			SeconIoniz_iso[ipH_LIKE] + 
			SeconIoniz_iso[ipHE_LIKE] + 
			secmet ));

		/* now do same for Ly-alpha */
		Secondaries.x12tot = (float)( Secondaries.x12tot*facold + facnew*
			(scmpla + 
			SeconExcit_iso[ipH_LIKE] + 
			SeconExcit_iso[ipHE_LIKE] + 
			smetla));
	}

	if( trace.lgTrace && Secondaries.csupra > 0. )/**/
	{
		fprintf( ioQQQ, 
			"   HeatSum retrn CSUPRA %9.2e SECCMP %6.3f SecHI %6.3f SECHE %6.3f SECMET %6.3f efrac= %9.2e \n", 
		  Secondaries.csupra, 
		  seccmp/Secondaries.csupra, 
		  SeconIoniz_iso[ipH_LIKE] / Secondaries.csupra, 
		  SeconIoniz_iso[ipHE_LIKE]/Secondaries.csupra, 
		  secmet/Secondaries.csupra ,
		  phycon.ElecFrac );
	}

	/*******************************************************************
	 *
	 * get derivative of total heating
	 *
	 *******************************************************************/

	/* now get derivative of heating due to photoionization, 
	 * >>chng 96 jan 14
	 *>>>>>NB heating going down with iincreasing temp is negative */
	heat.dHTotDT = -0.7*Photo_Heat/phycon.te;

	/* oldfac was factor used in old implimentation
	 * any term using it should be rethought */
	oldfac = -0.7/phycon.te;

	/* carbon monoxide */
	heat.dHTotDT += heat.heating[0][9]*oldfac;

	/* ly alpha collisoinal heating */
	heat.dHTotDT += heat.heating[0][10]*oldfac;

	/* line heating */
	heat.dHTotDT += heat.heating[0][22]*oldfac;

	/* free free heating, propto T^-0.5
	 * >>chng 96 oct 30, from heating(1,12) to FreeFreeHeat,
	 * do cooling separately assume FreeFreeHeat goes as T^-3/2
	 * dHTotDT = dHTotDT + heating(1,12) * (-0.5/te) */
	heat.dHTotDT += hydro.FreeFreeHeat*(-1.5/phycon.te);

	/* grain heating by photo
	 * >>chng 97 jul 02, change deriv in following, better fit to what came out */
	heat.dHTotDT += heat.heating[0][13]*0.1/phycon.te;

	/* grain heating by collsional */
	heat.dHTotDT += heat.heating[0][14]*oldfac;

	/* helium triplets heating */
	heat.dHTotDT += heat.heating[1][2]*oldfac;

	/* hydrogen molecule dissociation */
	heat.dHTotDT += heat.heating[0][17]*oldfac;

	/* >>chng 96 nov 15, had been oldfac below, wrong sign
	 * H- heating, goes up with temp since rad assoc does */
	heat.dHTotDT += heat.heating[0][15]*0.7/phycon.te;

	/* H2+ heating */
	heat.dHTotDT += heat.heating[0][16]*oldfac;

	/* Balmer continuum and all other excited states
	 * - T goes up so does pop and heating
	 * but this all screwed up by change in eden */
	heat.dHTotDT += heat.heating[0][1]*oldfac;

	/* all three body heating, opposite of coll ion cooling */
	heat.dHTotDT += heat.heating[0][3]*oldfac;

	/* bound electron recoil heating
	heat.dHTotDT += ionrec.CompRecoilHeatLocal*oldfac; */

	/* compton heating */
	heat.dHTotDT += heat.heating[0][19]*oldfac;

	/* extra heating source, usually zero */
	heat.dHTotDT += heat.heating[0][20]*oldfac;

	/* pair production */
	heat.dHTotDT += heat.heating[0][21]*oldfac;

	/* deriv of heating due to collisions of H lines, heating(1,24) */
	for( ipISO=ipH_LIKE; ipISO<NISO; ++ipISO )
	{
		for( nelem=0; nelem<LIMELM; ++nelem)
		{
			heat.dHTotDT += iso.dLTot[ipISO][nelem];
		}
	}

	/* heating due to Katya's FeII atom */
	if( heat.heating[25][27] > 0. )
	{
		heat.dHTotDT += holod.dfcool;
	}

	/* possibility of getting emperical heating derivative
	 * normally false, set true with 'set numerical derivatives' command */
	if( NumDeriv.lgNumDeriv )
	{
		/*lint -e777 test floats for equality */
		if( ((nzone > 2 && nzone == nzSave) && 
			oldtemp != phycon.te) && nhit > 4 )
		/*lint +e777 test floats for equality */
		{
			/* hnit is number of trys on this zone - use to stop numerical problems
			 * do not evaluate numerical deriv until well into solution */
			deriv = (oldheat - heat.htot)/(oldtemp - phycon.te);
			heat.dHTotDT = deriv;
		}
		else
		{
			deriv = heat.dHTotDT;
		}

		/* this happens when new zone starts */
		if( nzone != nzSave )
		{
			nhit = 0;
		}
		nzSave = nzone;
		nhit += 1;
		oldheat = heat.htot;
		oldtemp = phycon.te;
	}

	if( trace.lgTrace && trace.lgHeatBug )
	{
		ipnt = 0;
		/* this loops through the 2D array that contains all agents counted in htot */
		for( i=0; i < LIMELM; i++ )
		{
			for( j=0; j < LIMELM; j++ )
			{
				if( heat.heating[i][j]/heat.htot > FAINT_HEAT )
				{
					ipsave[ipnt] = i;
					jpsave[ipnt] = j;
					save[ipnt] = heat.heating[i][j]/heat.htot;
					/* increment ipnt, but do not let it get too big */
					ipnt = MIN2((long)NDIM-1,ipnt+1);
				}
			}
		}

		/* now check for possible line heating not counted in 1,23
		 * there should not be any significant heating source heree
		 * since they would not be counted in deriv correctly */
		for( i=0; i < cooling.ncltot; i++ )
		{
			if( cooling.heatnt[i]/heat.htot > FAINT_HEAT )
			{
				ipsave[ipnt] = -1;
				jpsave[ipnt] = i;
				save[ipnt] = cooling.heatnt[i]/heat.htot;
				ipnt = MIN2((long)NDIM-1,ipnt+1);
			}
		}

		fprintf( ioQQQ, "    HeatSum HTOT%11.3e Te:%10.3e dH/dT%10.2e other %10.2e photo %10.2e\n", 
		  heat.htot, phycon.te, heat.dHTotDT ,	OtherHeat , Photo_Heat);

		fprintf( ioQQQ, "  " );
		for( i=0; i < ipnt; i++ )
		{
			/*lint -e644 these three are initialized above */
			fprintf( ioQQQ, "   [%ld][%ld]%6.3f",
				ipsave[i], 
				jpsave[i],
				save[i] );
			/*lint +e644 these three are initialized above */
			/* throw a cr every n numbers */
			if( !(i%8) && i>0 && i!=(ipnt-1) )
			{
				fprintf( ioQQQ, "\n  " );
			}
		}
		fprintf( ioQQQ, "\n" );
	}

#	ifdef DEBUG_FUN
	fputs( " <->HeatSum()\n", debug_fp );
#	endif
	return;
}

/* =============================================================================*/
/* HeatZero is called by ConvIonizeOpacityDo */
void HeatZero( void )
{
	long int i , j;

	for( i=0; i < LIMELM; i++ )
	{
		for( j=0; j < LIMELM; j++ )
		{
			heat.heating[i][j] = 0.;
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->HeatZero()\n", debug_fp );
#	endif
	return;
}
