/*HydroLevelPop solve for ionization balance level populations of model hydrogen atom */
#include "cddefines.h"
#include "taulines.h"
#include "iso.h"
#include "secondaries.h"
#include "dynamics.h"
#include "chargtran.h"
#include "trace.h"
#include "phycon.h"
#include "ionfracs.h"
#include "typmatrx.h"
#include "linpack.h"
#include "veclib.h"
#include "hydrogenic.h"

void HydroLevelPop(long int nelem ,
	double **SaveZ/*[iso.numLevels[ipH_LIKE][nelem]+2][iso.numLevels[ipH_LIKE]+2]*/, 
	double *bvec/*[iso.numLevels[ipH_LIKE][nelem]+2]*/, 
	double *error/*[iso.numLevels[ipH_LIKE][nelem]+2]*/, 
	double *work/*[iso.numLevels[ipH_LIKE][nelem]+2]*/, 
	double **z/*[iso.numLevels[ipH_LIKE][nelem]+2][iso.numLevels[ipH_LIKE]+2]*/ ,
	long int *ipiv , /* MALLOC out to [iso.numLevels[ipH_LIKE][nelem]+1] */
	double *totcap/* MALLOC out to [iso.numLevels[ipH_LIKE][nelem]+1]*/)
{
	long int n, 
	  ipHi, 
	  ipLo, 
	  j, 
	  job, 
	  level, 
	  nerror1,
	  nerror2;

	double HRateDestGnd[8];

#	ifdef DEBUG_FUN
	fputs( "<+>HydroLevelPop()\n", debug_fp );
#	endif

	/* check that we were called with valid charge */
	ASSERT( nelem >= 0);
	ASSERT( nelem < LIMELM );

	/* 
	 * following electron density has approximate correction for neutrals
	 * corr of hi*1.7e-4 accounts for col ion by HI; Drawin Zs Phys 225, 483.
	 * used EdenHCorr instead
	 * edhi = eden + hi * 1.7e-4
	 */

	iso.RecomTotal[ipH_LIKE][nelem] = 0.;
	for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
	{

		/* total rate level n is populated from the continuum */
		/* PopLTE(n,nelem) is only LTE pop when mult by Ne Nh */
		totcap[n] = 
			/* radiative recombination */
			(double)(iso.RadRecomb[ipH_LIKE][nelem][n][ipRecRad]*
			iso.RadRecomb[ipH_LIKE][nelem][n][ipRecNetEsc]) + 

			/* induced recombination */
			iso.RecomInducRate[ipH_LIKE][nelem][n]*iso.PopLTE[ipH_LIKE][nelem][n] + 

			/* collisional recombination */
			iso.ColIoniz[ipH_LIKE][nelem][n]*phycon.EdenHCorr*iso.PopLTE[ipH_LIKE][nelem][n] ;

		z[iso.numLevels[ipH_LIKE][nelem]][n] = totcap[n] * phycon.eden;

		iso.RecomTotal[ipH_LIKE][nelem] += (float)totcap[n];
	}

	/* master balance equation */
	for( level=ipH1s; level < iso.numLevels[ipH_LIKE][nelem]; level++ )
	{

		/* all process depopulating level */
		z[level][level] = iso.gamnc[ipH_LIKE][nelem][level] + 
		  iso.ColIoniz[ipH_LIKE][nelem][level]* phycon.EdenHCorr + 
		  Secondaries.csupra;

		/* all processes populating level from below */
		for( ipLo=ipH1s; ipLo < level; ipLo++ )
		{
			double CollRate = EmisLines[ipH_LIKE][nelem][level][ipLo].ColUL* phycon.EdenHCorr;

			z[ipLo][level] = 
				-CollRate * 
			  (double)iso.stat[ipH_LIKE][nelem][level]/(double)iso.stat[ipH_LIKE][nelem][ipLo]*
			  iso.Boltzmann[ipH_LIKE][nelem][level][ipLo]  -
			  EmisLines[ipH_LIKE][nelem][level][ipLo].pump ;

			/* pumping out of here to lower level */
			/* fixit(); for 1s-2s two photon induced processes, pump contains the
			 * 1s - 2s pump rate, while Aul has the 2s-1s induced rate added in,
			 * and so is the total.  This should have a branch testing for 1s-2s
			 * and included the reverse pump in that case.  should not matter since
			 * induced 2s-1s should be slow compated to spontaneous A */
			z[level][level] += EmisLines[ipH_LIKE][nelem][level][ipLo].pump *
			  iso.stat[ipH_LIKE][nelem][ipLo]/iso.stat[ipH_LIKE][nelem][level];

			/* collisions out of here to lower level */
			z[level][level] += CollRate;

			/* radiative decays out of here to lower level */
			z[level][level] += 
				EmisLines[ipH_LIKE][nelem][level][ipLo].Aul*
			  (EmisLines[ipH_LIKE][nelem][level][ipLo].Pesc + 
			  EmisLines[ipH_LIKE][nelem][level][ipLo].Pelec_esc +
			  EmisLines[ipH_LIKE][nelem][level][ipLo].Pdest);
		}

		/* all processes populating level from above */
		for( ipHi=level + 1; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
		{
			double RadDecay , CollRate ;
			RadDecay =
				EmisLines[ipH_LIKE][nelem][ipHi][level].Aul*
			  (EmisLines[ipH_LIKE][nelem][ipHi][level].Pesc + 
			  EmisLines[ipH_LIKE][nelem][ipHi][level].Pelec_esc +
			  EmisLines[ipH_LIKE][nelem][ipHi][level].Pdest);

			/* >>chng 02 feb 06, define this and use in each direction below -
			 * attempt at solving roundoff problems on alphas */
			CollRate = EmisLines[ipH_LIKE][nelem][ipHi][level].ColUL *phycon.EdenHCorr;

			z[ipHi][level] = 
				-(RadDecay + 
			  EmisLines[ipH_LIKE][nelem][ipHi][level].pump*
			  iso.stat[ipH_LIKE][nelem][level]/iso.stat[ipH_LIKE][nelem][ipHi] + 
			  CollRate );

			/* pumping out of here to upper level */
			z[level][level] += EmisLines[ipH_LIKE][nelem][ipHi][level].pump ;

			/* collisions out of here to upper level */
			z[level][level] += (double)iso.stat[ipH_LIKE][nelem][ipHi] / (double)iso.stat[ipH_LIKE][nelem][level] *
				iso.Boltzmann[ipH_LIKE][nelem][ipHi][level]*CollRate;
		}
	}

	if( Secondaries.Hx12[0][1] > 0. )
	{
		/* now add on supra thermal excitation */
		for( level=ipH2s; level < iso.numLevels[ipH_LIKE][nelem]; level++ )
		{
#			if 0
			/* stuff in min after Hx12 evaluates to 0 for atom, or 1 for atom */
			z[ipH1s][ipH1s] += Secondaries.Hx12[MIN2(nelem,1)][level];

			z[level][ipH1s] -= 
				(Secondaries.Hx12[MIN2(nelem,1)][level]*
				iso.stat[ipH_LIKE][nelem][ipH1s]/ iso.stat[ipH_LIKE][nelem][level]);

			z[ipH1s][level] -= 
				Secondaries.Hx12[MIN2(nelem,1)][level];

			z[level][level] += Secondaries.Hx12[MIN2(nelem,1)][level]*
				iso.stat[ipH_LIKE][nelem][ipH1s]/ iso.stat[ipH_LIKE][nelem][level];
#			endif
			double RateUp , RateDown;

			RateUp = Secondaries.Hx12[MIN2(nelem,1)][level];

			RateDown = RateUp * (double)iso.stat[ipH_LIKE][nelem][ipH1s] /
				(double)iso.stat[ipH_LIKE][nelem][level];

			/* stuff in min after Hx12 evaluates to 0 for atom, or 1 for ion */
			/* total rate out of lower level */
			z[ipH1s][ipH1s] += RateUp;

			/* rate from the upper level to ground */
			z[level][ipH1s] -= RateDown ;

			/* rate from ground to upper level */
			z[ipH1s][level] -= RateUp ;

			z[level][level] += RateDown;  
		}
	}

	/* charge transfer, hydrogen onto everything else */
	if( nelem == 0 )
	{
		/* >>chng 01 apr 28, add this in */
		int ipCollide , ion;

		/* add on charge transfer ionization of hydrogen,
		 * recombination for other element is ionization of hydrogen */
		ChargTran.HCharExcIonTotal = 0.;
		for( ipCollide=1; ipCollide<LIMELM; ++ipCollide)
		{
			double hold_one = 0.;
			/* this is ion on the abundances scale, 1 is atom, so goes up to ipCollide+1,
			 * for helium ipCollide=1, ion must go up to 3 */
			/* check that array bounds not exceeded */
			ASSERT( (ipCollide+2-2) < (LIMELM+1) );
			for( ion=2; ion<=ipCollide+2; ++ion )
			{
				/* we intentionally skip CT with O+ since this would overwhelm every
				 * other process, but does not actually amount to much since is in
				 * balance with opposite CT reaction */
				if( ipCollide == ipOXYGEN && ion == 2 ) continue;

				/* charge transfer ionization of H, recombination for other species */
				hold_one = ChargTran.HCharExcRec[ipCollide][ion-2]*xIonFracs[ipCollide][ion-1];
				ChargTran.HCharExcIonTotal += hold_one;
			}
		}

		z[ipH1s][ipH1s] += ChargTran.HCharExcIonTotal;

		/* >>chng 01 may 07,  add this in */
		/* charge transfer recombination of hydrogen,
		 * which is ionization of the heavy element */
		ChargTran.HCharExcRecTotal = 0.;
		for( ipCollide=1; ipCollide<LIMELM; ++ipCollide)
		{
			/* this is ion on the abundances scale, 1 is atom, so goes up to ipCollide+1,
			 * for helium ipCollide=1, ion must go up to 3 */
			/* check that array bounds not exceeded */
			ASSERT( (ipCollide) < (LIMELM+1) );
			for( ion=1; ion<=ipCollide+1; ++ion )
			{
				/* skip Oo => O+ */
				if( ipCollide == ipOXYGEN && ion == 1 ) continue;
				/* charge transfer ionization of H, recombination for other species */
				ChargTran.HCharExcRecTotal += 
					ChargTran.HCharExcIon[ipCollide][ion-1]*xIonFracs[ipCollide][ion-1];
			}
		}
		/*fprintf(ioQQQ," %.2e %.2e %.2e \n", z[iso.numLevels[ipH_LIKE][nelem]+1][ipH1s] , ChargTran.HCharExcRecTotal, 
			z[iso.numLevels[ipH_LIKE][nelem]+1][ipH2p]);*/
		z[iso.numLevels[ipH_LIKE][nelem]][ipH1s] += ChargTran.HCharExcRecTotal;

#		if 0
		fprintf(ioQQQ," h net ion %.2e %.2e %.2e\n",
			ChargTran.HCharExcIonTotal,ChargTran.HCharExcRecTotal,
			ChargTran.HCharExcIonTotal-ChargTran.HCharExcRecTotal);
#		endif

		totcap[ipH1s] = z[iso.numLevels[ipH_LIKE][nelem]][ipH1s]/phycon.eden;
		/* nb, iso.RecomTotal[ipH_LIKE] should be updated too */

		{
			/*@-redef@*/
			enum {DEBUG=FALSE};
			/*@+redef@*/
			if( DEBUG )
			{
				long int maxnelem=-1, maxion=-1;
				double ctmax = 0.;
				for( ipCollide=1; ipCollide<LIMELM; ++ipCollide)
				{
					/* this is ion on the abundances scale, 1 is atom, so goes up to ipCollide+1,
					 * for helium ipCollide=1, ion must go up to 3 */
					for( ion=1; ion<=ipCollide+1; ++ion )
					{
						/* charge transfer ionization of H, recombination for other species */
						/*z[iso.numLevels[ipH_LIKE][nelem]+1][ipH1s] += ChargTran.HCharExcIon[ipCollide][ion-1]*xIonFracs[ipCollide][ion-1];*/
						if(	ChargTran.HCharExcIon[ipCollide][ion-1]*xIonFracs[ipCollide][ion-1] > ctmax )
						{
							ctmax = ChargTran.HCharExcIon[ipCollide][ion-1]*xIonFracs[ipCollide][ion-1];
							maxnelem = ipCollide;
							maxion = ion;
						}
					}
				}
				fprintf(ioQQQ," max H ct recom rate%.2e frac tot %.2e ipCollide %li ion %li\n", 
					ctmax,
					ctmax/MAX2(SMALLFLOAT,ChargTran.HCharExcRecTotal),
					maxnelem,
					maxion);
			}
		}
	}

	/* add in advection - these terms normally zero */
	z[iso.numLevels[ipH_LIKE][nelem]][ipH1s] += dynamics.Recomb[nelem][nelem];
	z[ipH1s][ipH1s] += dynamics.Photo;

	/* =================================================================== 
	 *
	 * at this point all matrix elements have been established 
	 *
	 * ==================================================================== */

	/* save total ground state destruction rate for possible later use */
	for( ipLo=ipH1s; ipLo<iso.numLevels[ipH_LIKE][nelem]; ++ipLo )
	{
		hydro.DestRate[nelem][ipLo] = z[ipLo][ipLo];
	}

	if( (trace.lgTrace && trace.lgIsoTraceFull[ipH_LIKE]) && (nelem == trace.ipIsoTrace[ipH_LIKE]) )
	{
		if( nelem == 0 )
		{
			/* remember ground state destruction rate */
			HRateDestGnd[0] = z[ipH1s][ipH1s];
			HRateDestGnd[1] = iso.gamnc[ipH_LIKE][nelem][ipH1s]/
			  z[ipH1s][ipH1s];
			HRateDestGnd[2] = iso.ColIoniz[ipH_LIKE][nelem][ipH1s]*
			  phycon.EdenHCorr/z[ipH1s][ipH1s];
			HRateDestGnd[3] = Secondaries.csupra/z[ipH1s][ipH1s];
			HRateDestGnd[4] = Secondaries.Hx12[MIN2(nelem,1)][ipH2p]*
			  9./z[ipH1s][ipH1s];

			fprintf(ioQQQ," grnd dest fracs:");
			for( j=ipH1s; j < 5; j++ )
			{
				fprintf( ioQQQ," ");
				fprintf( ioQQQ,PrintEfmt("%8.1e", HRateDestGnd[j] ));
			}
			fprintf(ioQQQ," hcoldc %e\n", phycon.EdenHCorr );
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, "  pop level     others => (HydroLevelPop)\n" );
		for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
		{
			fprintf( ioQQQ, "       HII%2ld", n );
			for( j=ipH1s; j < (iso.numLevels[ipH_LIKE][nelem] + 1); j++ )
			{
				fprintf( ioQQQ," ");
				/*fprintf( ioQQQ,PrintEfmt("%8.1e", z[j][n] ) );*/
				fprintf( ioQQQ,"%.9e\t", z[j][n] );
			}
			fprintf( ioQQQ, "\n" );
		}
	}

	/* save matrix */
	for( j=ipH1s; j < iso.numLevels[ipH_LIKE][nelem]; j++ )
	{
		for( n=ipH1s; n < (iso.numLevels[ipH_LIKE][nelem] + 1); n++ )
		{
			SaveZ[n][j] = z[n][j];
		}
	}

	/* which matrix solver? */
	if( strcmp(TypMatrx.chMatrix,"matin1 ") == 0 )
	{
		/* this is the usual matrix inversion method, slightly faster */
		/*matin1();*/
		nerror1 = 0;
		if( nerror1 != 0 )
		{
			fprintf( ioQQQ, " hydrogen matrix error, stop.\n" );
			puts( "[Stop in HydroLevelPop]" );
			cdEXIT(EXIT_FAILURE);
		}
	}

	/* this is the default */
	else if( strcmp(TypMatrx.chMatrix,"linpack") == 0 )
	{
		/*double amat[LMHLVL+2][LMHLVL+2];*/
		double *amat;
	  /* iso.numLevels[ipH_LIKE][nelem]+1 is right dimension of matrix */
#		ifdef AMAT
#			undef AMAT
#		endif
#		define AMAT(I_,J_)	(*(amat+(I_)*(iso.numLevels[ipH_LIKE][nelem])+(J_)))
		/* MALLOC space for the  1-d array */
		if( (amat=(double*)MALLOC( (sizeof(double)*(unsigned)((iso.numLevels[ipH_LIKE][nelem])*(iso.numLevels[ipH_LIKE][nelem])) ))) == NULL )
		{
			fprintf( ioQQQ, " HydroLevelPop MALLOC amat error\n" );
			puts( "[Stop in HydroLevelPop]" );
			cdEXIT(EXIT_FAILURE);
		}

		/* this one may be more robust */
		for( j=ipH1s; j < iso.numLevels[ipH_LIKE][nelem]; j++ )
		{
			for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
			{
				/*amat[n][j] = z[n][j];*/
				AMAT(n,j) = z[n][j];
			}
			bvec[j] = z[iso.numLevels[ipH_LIKE][nelem]][j];
		}
		/*DGETRF(iso.numLevels[ipH_LIKE][nelem]+1,iso.numLevels[ipH_LIKE]+1,
		  (double*)amat,LMHLVL+2,ipiv,&nerror1);
		DGETRS('N',iso.numLevels[ipH_LIKE][nelem]+1,1,(double*)amat,LMHLVL+2,ipiv,
		  bvec,LMHLVL+1,&nerror2);*/

		DGETRF(iso.numLevels[ipH_LIKE][nelem],iso.numLevels[ipH_LIKE][nelem],
		  amat,(iso.numLevels[ipH_LIKE][nelem]),ipiv,&nerror1);

		DGETRS('N',iso.numLevels[ipH_LIKE][nelem],1,amat,(iso.numLevels[ipH_LIKE][nelem]),ipiv,
		  bvec,(iso.numLevels[ipH_LIKE][nelem]),&nerror2);

		if( nerror1 != 0 || nerror2 != 0 )
		{
			fprintf( ioQQQ, " HydroLevelPop dgetrs error\n" );
			puts( "[Stop in HydroLevelPop]" );
			cdEXIT(EXIT_FAILURE);
		}
		/* now put results back into z so rest of code treates only
		 * one case - as if matin1 had been used */
		for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
		{
			z[iso.numLevels[ipH_LIKE][nelem]][n] = bvec[n];
		}
		free( amat);
	}
	else if( strcmp(TypMatrx.chMatrix,"veclib ") == 0 )
	{
		double /*amat[LMHLVL+2][LMHLVL+2], */ rcond;
		/* Jason found this one on the Exemplar, distributed source just stops */
		fprintf( ioQQQ, " this has not been checked since H atom conv\n" );
		for( j=ipH1s; j < iso.numLevels[ipH_LIKE][nelem]; j++ )
		{
			/*for( n=ipH1s; n <= iso.numLevels[ipH_LIKE][nelem]; n++ )
			{
				amat[n][j] = z[n][j];
			}*/
			bvec[j] = z[iso.numLevels[ipH_LIKE][nelem]][j];
		}
		job = 0;
		rcond = 0.;
		/*dgeco((double*)amat,iso.numLevels[ipH_LIKE][nelem]+2,iso.numLevels[ipH_LIKE]+1,ipiv,rcond, work);
		dgesl((double*)amat,iso.numLevels[ipH_LIKE][nelem]+2,iso.numLevels[ipH_LIKE]+1,ipiv,bvec, job);*/
		dgeco((double*)z,iso.numLevels[ipH_LIKE][nelem],iso.numLevels[ipH_LIKE][nelem],ipiv,rcond, work);
		dgesl((double*)z,iso.numLevels[ipH_LIKE][nelem],iso.numLevels[ipH_LIKE][nelem],ipiv,bvec, job);
		/* now put results back into z so rest of code treates only
		 * one case - as if matin1 had been used */
		for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
		{
			z[iso.numLevels[ipH_LIKE][nelem]][n] = bvec[n];
		}
		puts( "[Stop in HydroLevelPop]" );
		cdEXIT(EXIT_FAILURE);
	}
	else
	{
		fprintf( ioQQQ, " chMatrix type insane in HydroLevelPop, was%7.7s\n", 
		  TypMatrx.chMatrix );
		puts( "[Stop in HydroLevelPop]" );
		cdEXIT(EXIT_FAILURE);
	}
	/* end of branches for which matrix solution, now check if valid */

	/* check whether solution is valid */
	for( level=ipH1s; level < iso.numLevels[ipH_LIKE][nelem]; level++ )
	{
		double BigSoln= 0.;
		error[level] = 0.;
		for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
		{
			/* remember the largest value of the soln matrix to div by below */
			if( fabs(SaveZ[n][level] ) > BigSoln )
				BigSoln = fabs(SaveZ[n][level]);

			error[level] += SaveZ[n][level]*z[iso.numLevels[ipH_LIKE][nelem]][n];
		}

		if( BigSoln > 0. )
		{
			error[level] = (error[level] - SaveZ[iso.numLevels[ipH_LIKE][nelem]][level])/ BigSoln;
		}
		else
		{
			error[level] = 0.;
		}
	}

	/* convert from departure coef in Z(I) to level pop rel to HII */

	/* put departure coefficients and level populations into master array */
	for( ipLo=ipH1s; ipLo < iso.numLevels[ipH_LIKE][nelem]; ipLo++ )
	{
		iso.Pop2Ion[ipH_LIKE][nelem][ipLo] = (float)z[iso.numLevels[ipH_LIKE][nelem]][ipLo];
		if( iso.PopLTE[ipH_LIKE][nelem][ipLo] > 0. )
		{
			iso.DepartCoef[ipH_LIKE][nelem][ipLo] = 
				(iso.Pop2Ion[ipH_LIKE][nelem][ipLo]/
				(iso.PopLTE[ipH_LIKE][nelem][ipLo]* phycon.eden) );
		}
		else
		{
			iso.DepartCoef[ipH_LIKE][nelem][ipLo] = 0.;
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->HydroLevelPop()\n", debug_fp );
#	endif
	return;
}

