/*HydroPesc evaluate escape and destruction probabilities for hydrogen lines,
 * called by RTMake */
#include "cddefines.h"
#include "physconst.h"
#include "taulines.h"
#include "iso.h"
#include "hydrogenic.h"
#include "strk.h"
#include "phycon.h"
#include "converge.h"
#include "opacity.h"
#include "doppvel.h"
#include "twophoton.h"
#include "trace.h"
#include "rfield.h"
#include "ionfracs.h"
#include "wind.h"
#include "atom_oi.h"
#include "rt.h"
#include "rtescprob.h"
/* needed in excluded debug print statement */
#include "fe2ovr.h"

void HydroPesc(
	/* nelem is on C scale, 0 for H, 1 for He, etc */
	long int nelem ,
	/* flag saying whether we should do escape prob  and reevaluate A and opac (TRUE) or
	 * only dest probs (FALSE).  This is called with TRUE one time per zone,
	 * when optical depths are updated.  It is called FALSE many times, within ionzie,
	 * each time the opacity changes */
	int lgDoEsc )
{
	static int lgTOnSave;
	long int i,
	  ipHi, 
	  ipLo, 
	  limit;
	double abund=0., 
	  coloi, 
	  factor, 
	  tout,
	  z4;/* physical scale z to the 4th power, used often below */
	float dest=0.f, 
	  esin;
	float FracNew ;
	/* this will be used to save old vals of destruction prob in case
	 * we need to use mean of old and new, as is necessary when ots oscillations
	 * take place */
	float DespSave[NHYDRO_MAX_LEVEL];
	float eEscSave[NHYDRO_MAX_LEVEL];
	float ri2s1s;
	float ri1s2s;

#	ifdef DEBUG_FUN
	fputs( "<+>HydroPesc()\n", debug_fp );
#	endif

	/* check that we were called with valid charge */
	ASSERT( nelem >= 0);
	ASSERT( nelem < LIMELM );
	for( i=0;i<NHYDRO_MAX_LEVEL; ++i)
	{
		DespSave[i] = -FLT_MAX;
		eEscSave[i] = -FLT_MAX;
	}

	/* will need this for some scaling laws - physical Z to the 4th power*/
	z4 = POW2(nelem+1.);
	z4 *= z4;

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, 
			"          HydroPesc nelem=%ld called\n", 
		  nelem );
	}

	if( lgDoEsc )
	{
		/* hydrogen lines opacity is function of A's
		 * above 15 just use actual A's for high density limit
		 * only change opacities for levels between 4 and 15 */
		/* NB! must never set opacity for 2s-1s two-photon transition here, since
		 * the "line" can become optically thick if treated as a line.  It is a 
		 * continuum and the opacity is not related to the A by the usual expression */
		limit = MIN2(16,iso.numLevels[ipH_LIKE][nelem]);
		/* do Paschen and higher lines first, 
		 * do balmer below since must separate 2s and 2p */
		for( ipHi=4; ipHi < limit; ipHi++ )
		{
			for( ipLo=3; ipLo < ipHi; ipLo++ )
			{
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul = 
					(float)(hydro.HyLife[ipHi]*
					HydroBranch(ipHi,ipLo,nelem+1)*z4);
				ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul > 0.);

				/* make self-consistent opacity, convert new As back into opacities */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].opacity = 
					(float)(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul*
				  2.2448e-26*iso.stat[ipH_LIKE][nelem][ipHi]/
				  iso.stat[ipH_LIKE][nelem][ipLo]*
				  POW3(RYDLAM/(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyWN * WAVNRYD)));

				/* check that results are ok */
				ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].opacity > 0.);
			}
		}

		/* the actual branching ratio from high levels down to
		 * 2s and 2p depend on the density.  the code goes to
		 * the correct low and high density limits - I know of
		 * nothing better than can be done. */
		factor = MAX2( 0.25 , 0.32 - 0.07*phycon.eden/(phycon.eden + 1e7) );

		/* treat 2s = to 2p for HydroBranch, which returns total to 2s+2p */
		for( ipHi=4; ipHi < limit; ipHi++ )
		{
			/* get new effective A for this density and temperature */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].Aul = 
				(float)(hydro.HyLife[ipHi]*
				factor*HydroBranch(ipHi,2,nelem+1)*z4);

				/* check that results are ok */
			ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].Aul > 0.);

			/* do 2p by scaling relative to 2s */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].Aul = (float)(
				EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].Aul / factor *( 1. - factor )); 

				/* check that results are ok */
			ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].Aul > 0.);

			/* make self-consistent opaciity for 2s, from A */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].opacity = 
				(float)(EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].Aul*
			  2.2448e-26*iso.stat[ipH_LIKE][nelem][ipHi]/
			  iso.stat[ipH_LIKE][nelem][ipH2s]*
			  POW3(RYDLAM/(EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].EnergyWN * WAVNRYD)));

				/* check that results are ok */
			ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].opacity > 0.);

			/* make self-consistent opaciity for 2p, from A */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].opacity = 
				(float)(EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].Aul*
			  2.2448e-26*iso.stat[ipH_LIKE][nelem][ipHi]/
			  iso.stat[ipH_LIKE][nelem][ipH2p]*
			  POW3(RYDLAM/ (EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].EnergyWN * WAVNRYD)));

				/* check that results are ok */
			ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].opacity > 0.);
		}
	}
	/* end test branch lgDoEsc */

	/* now update escape and destruction prob */

	if( xIonFracs[nelem][nelem+1] > 1e-30 )
	{
		factor = xIonFracs[nelem][nelem+1];
	}
	else
	{
		/* case where almost no parent ion - this will make
		 * very large line opacity, so background dest small */
		factor = 1.;
	}

	/* save destruction probs for Lyman lines in case ots rates oscillate */
	for( ipHi=ipH2p; ipHi< iso.numLevels[ipH_LIKE][nelem]; ++ipHi )
	{
		DespSave[ipHi] = EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].Pdest;
		eEscSave[ipHi] = EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].Pelec_esc;
	}

	/* first will be static solution */
	if( wind.windv == 0. )
	{
		/* hydrogenic lyman lines special since outward optical depths always set,
		 * trick routines for Lyman lines only */
		lgTOnSave = opac.lgTauOutOn;
		opac.lgTauOutOn = TRUE;

		/* first do lyman alpha lya la, but only if optical depths not overrun */
		/* >>chng 01 apr 01, from 0.9 to 0.99 since outer edge better defined */
		tout = EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauTot*0.99 - 
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauIn;

		/* must temporarily make ipLnPopOpc physical */
		EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].PopOpc *= 
			(float)factor;

		/* generate escape prob, pumping rate, destruction prob, 
		 * inward outward fracs  */
		RTMakeStat(&EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s] , lgDoEsc );

		/* go back to original units so that final correction ok */
		EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].PopOpc /= 
			(float)factor;

		/* >>>chng 99 dec 18, repair dest prob that got clobbered in call to
		 * RTMakeStat, since will not be evaluated when tout not positive */
		EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest = DespSave[ipH2p];
		ASSERT( EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest>= 0. );
		EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pelec_esc = eEscSave[ipH2p];

		/* only update La dest prob if we have good outward optical depths */
		if( tout > 0. )
		{
			double opac_line , eesc;

			tout = EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauTot - 
			  EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauIn;

			abund = EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].PopOpc*xIonFracs[nelem][nelem+1];

			/* the descruction prob comes back as dest */
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pesc = 
				(float)(RTesc_lya(&esin, &dest,abund,nelem));

			/* this is current destruction rate */
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest = dest;
			ASSERT( dest >= 0. );

			/* elec scat escape prob */
			opac_line = abund * EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].opacity/DoppVel.doppler[nelem];

			if( opac_line > SMALLFLOAT )
			{
				double es = phycon.eden*6.65e-25;
				/* this is equation 5 of 
				*>>refer	line	desp	Netzer, H., Elitzur, M., & Ferland, G. J. 1985, ApJ, 299, 752*/
				eesc = es/(es+opac_line) * MAX2(0.,1.-EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pesc - EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest);;
			}
			else
				eesc = 0.;

			EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pelec_esc = (float)eesc;

			if( nelem == 0 )
			{
				/* add on destruction of hydrogen Lya by FeII
				 * now add in FeII deexcitation via overlap,
				 * but only as loss, not photoionization, source
				 * dstfe2lya is Ly-alpha deexcit by FeII overlap - conversion into FeII em */
				/* find FeII overlap destruction rate, 
				 * this does NOTHING when large FeII atom is turned on */
				fe2ovr();

				/* this was introduced in the fort - c conversion, and is a bug since
				 * dest feii added to ots dest */
				/*EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pdest = MIN2(1.f,
				  EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pdest + hydro.dstfe2lya);*/
				/* >>chng 00 jan 06, let dest be large than 1 to desaturate the atom */
				/* >>chng 01 apr 01, add test for tout >= 0., 
				 * must not add to Pdest when it has not been refreshed here */
				if( tout >= 0. )
					EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest += hydro.dstfe2lya;
			}

			/* >>>chng 99 apr 16, average of old and new rates to damp opacity - Lya ots
			 * rates when opacity is large, as in HeI trip opac in Balmer continuum.
			 * the pphheonly.in test case exposes this problem */
			{
				/* Lya debugging */
				/*@-redef@*/
				enum {DEBUG=FALSE};
				/*@+redef@*/
				if( DEBUG && nelem==0 && nzone > 180)
				{
					fprintf(ioQQQ,
						"z%3li Lya eval Pdest popl\t%g\tconopac\t%g\tPdest\t%g\tPesc\t%g\t tot in\t%g\t%g\n",
						nzone ,
						abund , 
						opac.opac[EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].ipCont-1],
						EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest, 
						EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pesc, 
						EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauTot,
						EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauIn);
				}
			}

			/*  this is fraction of line which is inward escaping */
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].FracInwd = rt.fracin;
			/*if(nelem==1)
				fprintf(ioQQQ," desp=%g\n", dest );*/
		}

		/* now do remainder of Lyman lines, skipping 2s */
		ipLo=ipH1s;
		for( ipHi=3; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
		{
			/* must temporarily make ipLnPopOpc physical */
			EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc *= 
				(float)factor;

			/* generate escape prob, pumping rate, destruction prob, 
			 * inward outward fracs  */
			RTMakeStat(&EmisLines[ipH_LIKE][nelem][ipHi][ipLo] , lgDoEsc );

			{
				/* Lyman line debugging */
				/*@-redef@*/
				enum {DEBUG=FALSE};
				/*@+redef@*/
				if( DEBUG && nelem==0 && iteration == 2)
				{
					fprintf(ioQQQ,
						"z%3li Ly3 eval Pdest popl\t%g\tconopac\t%g\tPdest\t%g\tPesc\t%g\t tot in\t%g\t%g\t ots\t%g\t conv\t%i\n",
						nzone ,
						abund , 
						opac.opac[EmisLines[ipH_LIKE][nelem][3][ipH1s].ipCont-1],
						EmisLines[ipH_LIKE][nelem][3][ipH1s].Pdest, 
						EmisLines[ipH_LIKE][nelem][3][ipH1s].Pesc, 
						EmisLines[ipH_LIKE][nelem][3][ipH1s].TauTot,
						EmisLines[ipH_LIKE][nelem][3][ipH1s].TauIn,
						EmisLines[ipH_LIKE][nelem][3][ipH1s].ots,
						conv.lgSearch
						);
				}
			}

			/* go back to original units so that final correction ok */
			EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc /= 
				(float)factor;
		}

		/* reset the flag, so only Lyman lines forced to include outward */
		opac.lgTauOutOn = lgTOnSave;

		/* this loop for Balmer lines which are special,
		 * because must bring 2s and 2p together */
		/* now do 2s and 2p, must bring optical depths together */
		for( ipHi=3; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
		{
			/* these are used for saving current state of two lines */
			float tauin2s , tauin2p , tauout2s , tauout2p;
			double opac2s , opac2p;

			/* 2s inward and total optical depths */
			tauin2s =  EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauIn;
			tauout2s = EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauTot;

			/* 2p inward and total optical depths */
			tauin2p =  EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauIn;
			tauout2p = EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauTot;

			/* add inward optical depths together */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauIn += 
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauIn;

			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauIn = 
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauIn;

			/* add both total optical depths together */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauTot += 
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauTot;

			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauTot = 
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauTot;

			/* current 2s and 2p opacities */
			opac2s = EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc;
			opac2p = EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc;

			/* add opacities together */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc += 
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc;

			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc = 
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc;

			/* must temporarily make ipLnPopOpc physical */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc *= 
				(float)factor;
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc *= 
				(float)factor;

			/* generate escape prob, pumping rate, destruction prob, 
			 * inward outward fracs  */
			RTMakeStat(&EmisLines[ipH_LIKE][nelem][ipHi][ipH2s] , lgDoEsc );
			RTMakeStat(&EmisLines[ipH_LIKE][nelem][ipHi][ipH2p] , lgDoEsc );

			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc =   opac2s;
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauIn =   tauin2s;
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauTot = tauout2s;

			/* 2p inward and total optical depths */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc =   opac2p;
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauIn =   tauin2p;
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauTot = tauout2p;
		}

		/* now do Paschen and higher lines */
		for( ipLo=3; ipLo < (iso.numLevels[ipH_LIKE][nelem] - 1); ipLo++ )
		{
			for( ipHi=ipLo+1; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
			{
				/* must temporarily make ipLnPopOpc physical */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc *= 
					(float)factor;

				/* generate escape prob, pumping rate, destruction prob, 
				 * inward outward fracs  */
				RTMakeStat(&EmisLines[ipH_LIKE][nelem][ipHi][ipLo] , lgDoEsc );

				/* go back to original units so that final correction ok */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc /= 
					(float)factor;
			}
		}

		/* 666    C90 had the following code and warning about oscillations
		 *  highest level is a mess, set escape prob to one, actually a blend
		 *  if this is removed, then oscillations in totally optically thick
		 *  blr occur, for cases where all lines are optically thick
		 *  see model badbugs/bug0.in */
		/* >>chng 02 may 08, only update Pesc if it was updated in RTMakeStat */
		if( lgDoEsc )
			EmisLines[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem]-1][ipH1s].Pesc = (float)(
			MAX2( EmisLines[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem]-1][ipH1s].Pesc , 0.1));

		for( ipLo=ipH2s; ipLo < (iso.numLevels[ipH_LIKE][nelem] - 1); ipLo++ )
		{
			EmisLines[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem]-1][ipLo].Pesc = 1.;
		}
	}
	else
	{
		/* this is wind solution branch */
		/* hydrogenic lyman lines special since outward optical depths always set,
		 * trick routines for Lyman lines only */
		lgTOnSave = opac.lgTauOutOn;
		opac.lgTauOutOn = TRUE;

		/* say that no outward optical depths, but we are still ok */
		tout = 0.;

		/* windy model */
		for( ipLo=ipH1s; ipLo < (iso.numLevels[ipH_LIKE][nelem] - 1); ipLo++ )
		{
			for( ipHi=MAX2((long)ipH2p,ipLo+1); ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
			{
				/* some lines don't really exist */
				if( EmisLines[ipH_LIKE][nelem][ipHi][ipLo].ipCont <1 ) 
					continue;

				/* must temporarily make ipLnPopOpc physical */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc *= 
					(float)factor;

				RTMakeWind(&EmisLines[ipH_LIKE][nelem][ipHi][ipLo] , lgDoEsc );

				/* go back to original units so that final correction ok */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc /= 
					(float)factor;
			}
			/* reset the flag, so only Lyman lines forced to include outward */
			opac.lgTauOutOn = lgTOnSave;
		}
	}


	/* this is option to damp out Lyman line dest probs if ots rates oscillating */
	if( conv.lgOscilOTS )
	{
		/* this is damper used to stop oscillations when now present*/
		FracNew = 0.2f;
	}
	else
	{
		/* this is damper used to stop oscillations even when not detected */
		FracNew = 0.5f;
	}
	{
		/* following should be set true to print ots contributors */
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG && nelem==0 )
		{
			fprintf(ioQQQ,"Lya aver DespSave\t%g\tPdest\t%g\n",
				DespSave[ipH2p],
				EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest);
		}
	}
	/* >>chng 01 apr 01, add test for tout > 0, only in this case will
	 * new solution even be attempted */
	if( tout >= 0. )
	{
		for( ipHi=ipH2p; ipHi< iso.numLevels[ipH_LIKE][nelem]; ++ipHi )
		{
			/*lint -e771 DespSave possibly not initialized */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].Pdest = (1.f-FracNew)*DespSave[ipHi] +
				FracNew * EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].Pdest;
			ASSERT( EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].Pdest>= 0. );
			EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].Pelec_esc = (1.f-FracNew)*eEscSave[ipHi] +
				FracNew * EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].Pelec_esc;
			/*lint +e771 DespSave possibly not initialized */
		}
	}

	{
		/* following should be set true to print ots contributors */
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG )
		{
			if( nelem<0 && iteration == 2)
			{
				fprintf(ioQQQ,
					"z%3li Bala popopc\t%g\tconopac\t%g\tPdest\t%g\tPup\t%g\tPlo\t%g\t%g pesc%g\n",
					nzone ,
					EmisLines[ipH_LIKE][nelem][3][4].PopOpc , 
					opac.opac[EmisLines[ipH_LIKE][nelem][3][2].ipCont-1],
					EmisLines[ipH_LIKE][nelem][3][2].Pdest, 
					EmisLines[ipH_LIKE][nelem][3][2].PopHi, 
					EmisLines[ipH_LIKE][nelem][3][2].PopLo,
					EmisLines[ipH_LIKE][nelem][3][2].TauIn,
					EmisLines[ipH_LIKE][nelem][3][2].Pesc
					);
			}
			else if (nelem==1 && iteration == 2 )
			{
				fprintf(ioQQQ,
					"z%3li he62a popopc\t%g\tconopac\t%g\tPdest\t%g\tPup\t%g\tPlo\t%g\t%g pesc%g\n",
					nzone ,
					EmisLines[ipH_LIKE][nelem][5][4].PopOpc , 
					opac.opac[EmisLines[ipH_LIKE][nelem][5][4].ipCont-1],
					EmisLines[ipH_LIKE][nelem][5][4].Pdest, 
					EmisLines[ipH_LIKE][nelem][5][4].PopHi, 
					EmisLines[ipH_LIKE][nelem][5][4].PopLo,
					EmisLines[ipH_LIKE][nelem][5][4].TauIn,
					EmisLines[ipH_LIKE][nelem][5][4].Pesc
					);
			}
		}
	}

	/* reset the flag */
	opac.lgTauOutOn = lgTOnSave;

	/* find the rate of total and induced two photon */
	twopht(nelem, &ri2s1s , &ri1s2s , ipHYDROGEN );

	EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].pump = ri1s2s;
	/* >>chng 01 jan 18, us derived two-photon, the sum of spontaneous and induced */
	EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].Aul = ri2s1s;

	if( nelem == ipHYDROGEN )
	{
		/* do the 8446 problem */
		atom_oi_calc(&coloi);
		EmisLines[ipH_LIKE][ipHYDROGEN][3][ipH1s].Pesc = atom_oi.pmph31/
		  EmisLines[ipH_LIKE][ipHYDROGEN][3][ipH1s].Aul;
	}

	/* hydrogen-only solutions */
	/* >>chng 02 may 08, add test on lgDoEsc since adding to Pesc, and this is only
	 * evaulated above if lgDoEsc is true */
	if( nelem == ipHYDROGEN && lgDoEsc )
	{
		/* find Stark escape probabilities */
		strk();
		for( ipLo=ipH1s; ipLo < (iso.numLevels[ipH_LIKE][nelem] - 1); ipLo++ )
		{
			for( ipHi=MAX2((long)ipH2p,ipLo+1); ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
			{
				/* >>chng 02 may 08, moved elec scat esc to rtmakestat */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Pesc = (float)(MIN2(1.,
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Pesc+
					hydro.pestrk[ipLo][ipHi]));
			}
		}

		if( trace.lgTrace && trace.lgIsoTraceFull[ipH_LIKE] )
		{
			fprintf( ioQQQ, "       HydroPesc calls P8446 who found pmph31=%10.2e\n", 
			  atom_oi.pmph31 );
		}

		{
			/*@-redef@*/
			enum {DEBUG=FALSE};
			/*@+redef@*/
			if( DEBUG )
			{
				fprintf(ioQQQ,"hydropescdebugg\t%li\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
					nzone, 
					EmisLines[ipH_LIKE][nelem][2][0].pump,
					EmisLines[ipH_LIKE][nelem][4][3].TauIn,
					EmisLines[ipH_LIKE][nelem][4][3].TauCon,
					EmisLines[ipH_LIKE][nelem][4][3].pump,
					rfield.flux[EmisLines[ipH_LIKE][nelem][4][3].ipCont-1],
					rfield.OccNumbIncidCont[EmisLines[ipH_LIKE][nelem][4][3].ipCont-1]
					);
			}
		}
	}

	{
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG )
		{
			if( nelem==0&& nzone>164 )
			{
				fprintf(ioQQQ,"hydropescdebugg\t%li\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
					nzone, 
					EmisLines[ipH_LIKE][nelem][2][0].TauIn,
					EmisLines[ipH_LIKE][nelem][2][0].TauTot,
					EmisLines[ipH_LIKE][nelem][2][0].Pdest,
					EmisLines[ipH_LIKE][nelem][2][0].Pesc,
					tout);
			}
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->HydroPesc()\n", debug_fp );
#	endif
	return;
}

