/*LevelN compute an arbitrary N level atom */
#include "cddefines.h"
#include "cooling.h"
#include "physconst.h"
#include "phycon.h"
#include "trace.h"
#include "typmatrx.h"
#include "veclib.h"
#include "rt.h"
#include "linpack.h"
#include "leveln.h"


void LevelN(
	/* nlev is the number of levels to compute*/ 
	long int nlev, 
	/* ABUND is total abundance of species, used for nth equation */
	float abund, 
	/* G(nlev) is stat weight of levels */
	double g[], 
	/* EX(nlev) is excitation potential of levels, deg K
	 * 0 for first one, NOT d(ENER), but energy rel to ground */
	double ex[], 
	/* populations of each level as deduced here */
	double pops[], 
	/* departure coefficient, derived below */
	double depart[],
	/* data(ihi,ilo) is transit rate, from up to low, A * esc prob
	 * data(ilo,ihi) is col str from up to low */
	double ***data, 
	/* dest(ilo,ihi) is destruction rate, from up to low, A * dest prob,
	 * asserts confirm that ihi,lo is zero */
	double ***dest, 
	/* pump(lo, hi) is pumping rate from lower to upper level (s^-1), (hi,lo) must be zero  */
	double ***pump, 
	/* collision rates, evaluated here and returned for cooling by calling function,
	 * unless following flag is true.  If true then calling function has already filled
	 * in these rates.  CollRate[i][j] is rate from i to j */
	double ***CollRate,
	/* flag saying whether CollRate already done, or we need to do it here */
	int lgCollRateDone,
	/* 2D array of integer indices for line energy in continuum array, used
	 * for setting ots rates.  ignored if zero, indices are [low][high],
	 * indices are on f not c scale */
	long int ***ipdest, 
	/* total cooling and its derivative, set here but nothing done with it*/
	double *cooltl, 
	double *coolder, 
	/* string used to identify calling program in case of error */
	char *chLabel, 
	/* lgNegPop flag indicating what we have done
	 * positive if negative populations occurred
	 * zero if normal calculation done
	 * negative if too cold (for some atoms other routine will be called in this case) */
	int *lgNegPop,
	/* option to print debug information */
	int lgDeBug )
{
	/* side-effects of this routine:
	 * lines added to ots array 
	 */

	/*
	 * deduced variables
	 * EXCIT(IHI,ILO) excitation Boltzman factor
	 * CRATE(IHI,ILO) collis deexcit rate
	 * CRATE(ILO,IHI) collis excit rate
	 *
	 */

	/* LIMLEVELN is upper limit to number of levels allowed by present vectors - 
	 * it is defined in poplevls.h and is currently 20 */

	long int level, 
	  ihi, 
	  ilo, 
	  i,
	  j, 
	  job, 
	  ner;

	double cool1, 
	  rcond, 
	  ots;
#	if 0
	double 
	  amat[LIMLEVELN - 1][LIMLEVELN - 1], 
	  excit[LIMLEVELN][LIMLEVELN], 
	  zz[LIMLEVELN][LIMLEVELN],
	  bvec[LIMLEVELN - 1], 
	  work[LIMLEVELN - 1];
	long int ipiv[LIMLEVELN - 1], 
#	endif
	double 
	  *bvec, 
	  *work, 
	  *amat, 
	  **excit, 
	  **zz;
	long int *ipiv;

#	ifdef DEBUG_FUN
	fputs( "<+>LevelN()\n", debug_fp );
#	endif

	/* >>chng 00 oct 13, MALLOC and free space for arrays */
	if( (bvec = (double *)MALLOC( sizeof(double)*(size_t)(nlev+1) )) == NULL )
	{ 
		printf( " not enough memory to allocate bvec in LevelN\n" );
		puts( "[Stop in LevelN]" );
		cdEXIT(EXIT_FAILURE);
	}
	if( (work = (double *)MALLOC( sizeof(double)*(size_t)(nlev) )) == NULL )
	{ 
		printf( " not enough memory to allocate work in LevelN\n" );
		puts( "[Stop in LevelN]" );
		cdEXIT(EXIT_FAILURE);
	}
	if( (ipiv = (long int *)MALLOC( sizeof(long int)*(size_t)(nlev) )) == NULL )
	{ 
		printf( " not enough memory to allocate ipiv in LevelN\n" );
		puts( "[Stop in LevelN]" );
		cdEXIT(EXIT_FAILURE);
	}
	/* amat converted to 1d array */
#	ifdef AMAT
#		undef AMAT
#	endif
#	define AMAT(I_,J_)	(*(amat+(I_)*(nlev)+(J_)))
	/* MALLOC space for the  1-d array */
	if( (amat=(double*)MALLOC(sizeof(double)*(size_t)(nlev*nlev))) == NULL )
	{
		fprintf( ioQQQ, " LevelN MALLOC amat error\n" );
		puts( "[Stop in LevelN]" );
		cdEXIT(EXIT_FAILURE);
	}
	ASSERT(amat != NULL);

	/* create space for the 2D arrays */
	if( (excit = ((double **)MALLOC((size_t)(nlev+1)*sizeof(double *)))) == NULL )
	{
		fprintf(ioQQQ," LevelN could not malloc1 excit\n");
		puts( "[Stop in LevelN]" );
		cdEXIT(EXIT_FAILURE);
	}
	/* create space for the 2D arrays */
	if( (zz = ((double **)MALLOC((size_t)(nlev+1)*sizeof(double *)))) == NULL )
	{
		fprintf(ioQQQ," LevelN could not malloc1 zz\n");
		puts( "[Stop in LevelN]" );
		cdEXIT(EXIT_FAILURE);
	}
	for( i=0; i<(nlev+1); ++i )
	{
		if( (excit[i] = ((double *)MALLOC((size_t)(nlev+1)*sizeof(double )))) == NULL )
		{
			fprintf(ioQQQ," LevelN could not malloc2 excit\n");
			puts( "[Stop in LevelN]" );
			cdEXIT(EXIT_FAILURE);
		}
		if( (zz[i] = ((double *)MALLOC((size_t)(nlev+1)*sizeof(double )))) == NULL )
		{
			fprintf(ioQQQ," LevelN could not malloc2 zz\n");
			puts( "[Stop in LevelN]" );
			cdEXIT(EXIT_FAILURE);
		}
	}

	/*begin sanity checks
	 * excitation temperature of lowest level must be zero */
	ASSERT( ex[0] == 0. );

	/* both dest and pump (low, hi) are not used, should be zero */
	for( ihi=1; ihi < nlev; ihi++ )
	{
		for( ilo=0; ilo < ihi; ilo++ )
		{
			/* zero ots destruction rate */
			ASSERT( (*dest)[ihi][ilo] == 0. );
			ASSERT( (*pump)[ihi][ilo] == 0. );
			ASSERT( (*ipdest)[ihi][ilo] == 0 );
			/* these are collision strenghts and escape rates */
			ASSERT( (*data)[ihi][ilo] >= 0 );
			ASSERT( (*data)[ilo][ihi] >= 0 );
			ASSERT( (*dest)[ilo][ihi] >= 0 );
		}
	}

	/* data(level,level) should be zero */
	for( ihi=0; ihi < nlev; ihi++ )
	{
		ASSERT( (*data)[ihi][ihi] == 0 );
	}
	/*end sanity checks */

	if( lgDeBug || (trace.lgTrace && trace.lgTrLevN) )
	{
		fprintf( ioQQQ, " LevelN trace printout for atom=%s \n", chLabel);
		fprintf( ioQQQ, " dest\n" );

		fprintf( ioQQQ, "  hi  lo" );

		for( ilo=0; ilo < nlev-1; ilo++ )
		{
			fprintf( ioQQQ, "%4ld      ", ilo );
		}
		fprintf( ioQQQ, "      \n" );

		for( ihi=1; ihi < nlev; ihi++ )
		{
			fprintf( ioQQQ, "%3ld", ihi );
			for( ilo=0; ilo < ihi; ilo++ )
			{
				fprintf( ioQQQ, "%10.2e", (*dest)[ilo][ihi] );
			}
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, " A*esc\n" );
		fprintf( ioQQQ, "  hi  lo" );
		for( ilo=0; ilo < nlev-1; ilo++ )
		{
			fprintf( ioQQQ, "%4ld      ", ilo );
		}
		fprintf( ioQQQ, "      \n" );

		for( ihi=1; ihi < nlev; ihi++ )
		{
			fprintf( ioQQQ, "%3ld", ihi );
			for( ilo=0; ilo < ihi; ilo++ )
			{
				fprintf( ioQQQ, "%10.2e", (*data)[ilo][ihi] );
			}
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, " pump\n" );

		fprintf( ioQQQ, "  hi  lo" );
		for( ilo=0; ilo < nlev-1; ilo++ )
		{
			fprintf( ioQQQ, "%4ld      ", ilo );
		}
		fprintf( ioQQQ, "      \n" );

		for( ihi=1; ihi < nlev; ihi++ )
		{
			fprintf( ioQQQ, "%3ld", ihi );
			for( ilo=0; ilo < ihi; ilo++ )
			{
				fprintf( ioQQQ, "%10.2e", (*pump)[ilo][ihi] );
			}
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, " coll str\n" );
		fprintf( ioQQQ, "  hi  lo" );
		for( ilo=0; ilo < nlev-1; ilo++ )
		{
			fprintf( ioQQQ, "%4ld      ", ilo );
		}
		fprintf( ioQQQ, "      \n" );

		for( ihi=1; ihi < nlev; ihi++ )
		{
			fprintf( ioQQQ, "%3ld", ihi );
			for( ilo=0; ilo < ihi; ilo++ )
			{
				fprintf( ioQQQ, "%10.2e", (*data)[ihi][ilo] );
			}
			fprintf( ioQQQ, "\n" );
		}
	}

	/* check for zero abundance and exit if so */
	if( abund <= 0. )
	{
		*cooltl = 0.;
		*coolder = 0.;
		/* says calc was ok */
		*lgNegPop = FALSE;

		for( level=0; level < nlev; level++ )
		{
			pops[level] = 0.;
			depart[level] = 0.;
		}

		depart[0] = 1.;
		free( bvec );
		free( work );
		free( ipiv );
		free( amat );
		for( i=0; i < (nlev+1); ++i )
		{
			free( excit[i] );
			free( zz[i] );
		}
		free( excit );
		free( zz );

		/* there are TWO abort returns in this sub */
		
#		ifdef DEBUG_FUN
		fputs( " <->LevelN()\n", debug_fp );
#		endif
		return;
	}

	for( ilo=0; ilo < (nlev - 1); ilo++ )
	{
		for( ihi=ilo + 1; ihi < nlev; ihi++ )
		{
			excit[ilo][ihi] = sexp((ex[ihi]-ex[ilo])/phycon.te);
		}
	}

	if( trace.lgTrace && trace.lgTrLevN )
	{
		fprintf( ioQQQ, " excit, te=%10.2e\n", phycon.te );
		fprintf( ioQQQ, "  hi  lo" );

		for( ilo=1; ilo <= nlev; ilo++ )
		{
			fprintf( ioQQQ, "%4ld      ", ilo );
		}
		fprintf( ioQQQ, "      \n" );

		for( ihi=0; ihi < nlev; ihi++ )
		{
			fprintf( ioQQQ, "%3ld", ihi );
			for( ilo=0; ilo < nlev; ilo++ )
			{
				fprintf( ioQQQ, "%10.2e", excit[ilo][ihi] );
			}
			fprintf( ioQQQ, "\n" );
		}
	}

	/* punt if total excitation rate is zero */
	if( excit[0][nlev-1] + (*pump)[0][nlev-1] < 1e-13 )
	{
		*cooltl = 0.;
		*coolder = 0.;
		/* special flag saying too cool for highest level to be computed.
		 * some routines will call another routine for lower levels in this case */
		*lgNegPop = -1;

		for( level=1; level < nlev; level++ )
		{
			pops[level] = 0.;
			/* these are off by one - lowest index is zero */
			depart[level] = 0.;
		}

		/* everything in ground */
		pops[0] = abund;
		depart[0] = 1.;

		free( bvec );
		free( work );
		free( ipiv );
		free( amat );
		for( i=0; i < (nlev+1); ++i )
		{
			free( excit[i] );
			free( zz[i] );
		}
		free( excit );
		free( zz );
		
#		ifdef DEBUG_FUN
		fputs( " <->LevelN()\n", debug_fp );
#		endif
		return;
	}

	/* already have excitation pumping, now get deexcitation */
	for( ilo=0; ilo < (nlev - 1); ilo++ )
	{
		for( ihi=ilo + 1; ihi < nlev; ihi++ )
		{
			/* (*pump)[low][ihi] is excitation rate due to external continuum,
			 * so derive rate from upper to lower */
			(*pump)[ihi][ilo] = (*pump)[ilo][ihi]*g[ilo]/g[ihi];
		}
	}

	/* evaluate collision rates from collision strengths, but only if calling
	 * routine has not already done this */
	if( !lgCollRateDone )
	{
		/* factor is 8.629E-6 / SQRTE * EDEN */
		for( ilo=0; ilo < (nlev - 1); ilo++ )
		{
			for( ihi=ilo + 1; ihi < nlev; ihi++ )
			{
				/* this should be a collision strength */
				ASSERT( (*data)[ihi][ilo]>= 0. );
				/* this is deexcitation rate
				 * factor is 8.629E-6 / SQRTE * EDEN */
				(*CollRate)[ihi][ilo] = (*data)[ihi][ilo]/g[ihi]*phycon.cdsqte;
				/* this is excitation rate */
				(*CollRate)[ilo][ihi] = (*CollRate)[ihi][ilo]*g[ihi]/g[ilo]*
				  excit[ilo][ihi];
			}
		}
	}

	/* rate equations equal zero */
	for( level=0; level < nlev; level++ )
	{
		for( j=1; j < nlev; j++ )
		{
			zz[level][j] = 0.;
		}
		zz[level][0] = 1.0;
	}

	/* following is colum of vector */
	for( level=0; level < nlev; level++ )
	{
		zz[nlev][level] = 0.;
	}

	/* zz(nlev,nlev+1) = abund
	 * solve the problem with everything normalzied to pop of unity
	 * for numerical stbility
	 * zz(nlev,nlev+1) = 1. */
	zz[nlev][0] = 1.;

	/* eqns for destruction of level
	 * data(hi,lo) are A's, CollRate is coll excit in direction
	 * pump[low][high] is excitation rate from low to high */
	for( level=1; level < nlev; level++ )
	{
		zz[level][level] = 0.;

		/* leaving level to below */
		for( ilo=0; ilo < level; ilo++ )
		{
			zz[level][level] += (*CollRate)[level][ilo] + (*data)[ilo][level] + (*dest)[ilo][level] + 
			  (*pump)[level][ilo];
			/* >>chng 97 jul 31, added pumping down
			 * coming to level I from below */
			zz[ilo][level] = -(*CollRate)[ilo][level] - (*pump)[ilo][level];
		}

		/* leaving level to above */
		for( ihi=level + 1; ihi < nlev; ihi++ )
		{
			zz[level][level] += (*CollRate)[level][ihi] + (*pump)[level][ihi];
			/* coming to level from above */
			zz[ihi][level] = -(*CollRate)[ihi][level] - (*data)[level][ihi] - (*dest)[level][ihi] - 
			  (*pump)[ihi][level];
			/* >>chng 97 jul 31, added pumping down */
		}
	}

	/* which matrix solver? */
	if( strcmp(TypMatrx.chMatrix,"matin1 ") == 0 )
	{
		/*matin1();*/
		ner = 1;
		if( ner != 0 )
		{
			fprintf( ioQQQ, " LevelN MATRIX ERROR.\n" );
			puts( "[Stop in leveln]" );
			cdEXIT(EXIT_FAILURE);
		}
	}

	else if( strcmp(TypMatrx.chMatrix,"linpack") == 0 )
	{
		/* this one may be more robust, and is the default */
		for( j=0; j < nlev; j++ )
		{
			for( level=0; level < nlev; level++ )
			{
				/*amat[level][j] = zz[level][j];*/
				AMAT(level,j) = zz[level][j];
			}
			bvec[j] = zz[nlev][j];
		}

		/* >>chng 00 oct 13, for MALLOC'd space*/
		/*DGETRF(nlev,nlev,(double*)amat,LIMLEVELN-1,ipiv,&ner);*/
		DGETRF(nlev,nlev,amat,nlev,ipiv,&ner);
		/* usage DGETRS, 'N' = no transpose
		 * order of matrix,
		 * number of cols in bvec, =1
		 * array
		 * leading dim of array */
		/*DGETRS('N',nlev,1,(double*)amat,LIMLEVELN-1,ipiv,bvec,LIMLEVELN-1,&ner);*/
		/* >>chng 00 oct 13, for MALLOC'd space*/
		DGETRS('N',nlev,1,amat,nlev,ipiv,bvec,nlev,&ner);

		if( ner != 0 )
		{
			fprintf( ioQQQ, " fe2 ir dgetrs error\n" );
			puts( "[Stop in leveln]" );
			cdEXIT(EXIT_FAILURE);
		}

		/* now put results back into z so rest of code treates only
		 * one case - as if matin1 had been used */
		for( level=0; level < nlev; level++ )
		{
			zz[nlev][level] = bvec[level];
		}
	}

	else if( strcmp(TypMatrx.chMatrix,"veclib ") == 0 )
	{
		/* Jason found this one on the Exemplar, distributed source just stops */
		fprintf( ioQQQ, " this has not been checked since H atom conv\n" );
		for( j=0; j < nlev; j++ )
		{
			for( level=0; level < nlev; level++ )
			{
				AMAT(level,j) = zz[level][j];
			}
			bvec[j] = zz[nlev][j];
		}

		job = 0;
		rcond = 0.;
		/*TODO in following one is dimension, second is number of levels */
		dgeco(amat,5,5,ipiv,rcond,work);
		dgesl(amat,5,5,ipiv,bvec,job);
		/* now put results back into z so rest of code treates only
		 * one case - as if matin1 had been used */
		for( level=0; level < nlev; level++ )
		{
			zz[nlev][level] = bvec[level];
		}

		puts( "[Stop in leveln]" );
		cdEXIT(EXIT_FAILURE);
	}

	else
	{
		fprintf( ioQQQ, " chMatrix type insane in hydrogen, was%7.7s\n", 
		  TypMatrx.chMatrix );
		puts( "[Stop in leveln]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* set populations */
	for( level=0; level < nlev; level++ )
	{
		/* convert to real abundances */
		pops[level] = (double)(zz[nlev][level]*abund);
	}

	/* now find total cooling and its derivative */
	*cooltl = 0.;
	*coolder = 0.;
	for( ihi=1; ihi < nlev; ihi++ )
	{
		for( ilo=0; ilo < ihi; ilo++ )
		{
			/* this is net cooling in oK */
			cool1 = (pops[ilo]*(*CollRate)[ilo][ihi] - pops[ihi]*(*CollRate)[ihi][ilo])*
			  (ex[ihi] - ex[ilo]);
			*cooltl += cool1;

			/* simplistic way to handle line heating - better would
			 * be to go  over to method in level2 and level3 */
			*coolder += (float)(MAX2(0.,cool1)*((ex[ihi] - ex[0])*cooling.tsq1 - 
			  cooling.halfte));
		}
	}
	/* convert to ergs */
	*cooltl *= BOLTZMANN;
	*coolder *= BOLTZMANN;

	/* fill in departure coefficients */
	if( pops[0] > 1e-20 && excit[0][nlev-1] > 1e-20 )
	{
		/* >>chng 00 aug 10, loop had been from 1 and 0 was set to total abundance */
		depart[0] = 1.;
		for( ihi=1; ihi < nlev; ihi++ )
		{
			/* these are off by one - lowest index is zero */
			depart[ihi] = (pops[ihi]/pops[0])*(g[0]/g[ihi])/
			  excit[0][ihi];
		}
	}

	else
	{
		/* >>chng 00 aug 10, loop had been from 1 and 0 was set to total abundance */
		for( ihi=0; ihi < nlev; ihi++ )
		{
			/* these are off by one - lowest index is zero */
			depart[ihi] = 0.;
		}
		depart[0] = 1.;
	}

	/* in case atom has been trimed down */
	for( level=nlev; level < nlev; level++ )
	{
		pops[level] = 0.;
		depart[level] = 0.;
	}

	/* sanity check for stability of inversion */
	*lgNegPop = FALSE;
	for( level=0; level < nlev; level++ )
	{
		if( pops[level] < 0. )
		{
			*lgNegPop = TRUE;
		}
	}

	if( *lgNegPop )
	{
		fprintf( ioQQQ, " LevelN found negative population, atom=%4.4s\n", 
		  chLabel );
		fprintf( ioQQQ, " absolute population=" );

		for( level=0; level < nlev; level++ )
		{
			fprintf( ioQQQ, "%10.2e", pops[level] );
		}

		fprintf( ioQQQ, "\n" );
		for( level=0; level < nlev; level++ )
		{
			pops[level] = (double)MAX2(0.,pops[level]);
		}
	}

	if(  lgDeBug || (trace.lgTrace && trace.lgTrLevN) )
	{
		fprintf( ioQQQ, "\n absolute population   " );
		for( level=0; level < nlev; level++ )
		{
			fprintf( ioQQQ, " %10.2e", pops[level] );
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, " departure coefficients" );
		for( level=0; level < nlev; level++ )
		{
			fprintf( ioQQQ, " %10.2e", depart[level] );
		}
		fprintf( ioQQQ, "\n\n" );
	}

	for( ihi=1; ihi < nlev; ihi++ )
	{
		/* >>chng 00 aug 04, loop had been to ilo < (ihi - 1) so skipped
		 * all alpha transitions */
		/*for( ilo=0; ilo < (ihi - 1); ilo++ )*/
		for( ilo=0; ilo < ihi; ilo++ )
		{
			/* set ots destruction rate */
			if( (*ipdest)[ilo][ihi] > 0 )
			{
				ots = pops[ihi]*(*dest)[ilo][ihi];
				RT_OTS_AddLine((float)ots,(*ipdest)[ilo][ihi]);
			}
		}
	}

	free( bvec );
	free( work );
	free( ipiv );
	free( amat );
	for( i=0; i < (nlev+1); ++i )
	{
		free( excit[i] );
		free( zz[i] );
	}
	free( excit );
	free( zz );

#	ifdef DEBUG_FUN
	fputs( " <->LevelN()\n", debug_fp );
#	endif
	return;

}
