/* PressureChange called by ConvPresTempEdenIoniz
 * evaluate the current pressure, change needed to get it to converge,
 * aplies this correction factor to all gas constituents,
 * sets conv.lgConvPres true if good pressure, false if pressure change capped
 */
#include "cddefines.h"
#include "physconst.h"
#include "abundances.h"
#include "hmi.h"
#include "trace.h"
#include "wind.h"
#include "timefc.h"
#include "fluct.h"
#include "filfac.h"
#include "denpow.h"
#include "radius.h"
#include "phycon.h"
#include "he.h"
#include "ionfracs.h"
#include "dynamics.h"
#include "pressure.h"
#include "colden.h"
#include "dndt.h"
#include "hevmolec.h"
#include "tfidle.h"
#include "radacl.h"
#include "tabden.h"
#include "fabden.h"
#include "tababun.h"
#include "converge.h"

void PressureChange( 
	/* this is true if change in pressure has changed sign, showing near solution */
	int lgPresOscil )
{
	long int ion, 
	  nelem;

	double abun, 
	  dnew, 
	  edensave, 
	  hold, 
	  term;

	/* biggest multiplicative change in the pressure that we will allow */
	/* allowed error in the pressure is phycon.PressureError*/
	float pdelta;

	static double FacAbun, 
	  FacAbunSav, 
	  OldAbun;

	double windnw;

	static double factor = 1.;
	static double pold = 0.;

#	ifdef DEBUG_FUN
	fputs( "<+>PressureChange()\n", debug_fp );
#	endif

	edensave = phycon.eden;

	/* first evaluate total pressure for this location, and current conditions
	 * CurrentPressure is just sum of gas plus local line radiation pressure */
	pressure.PressureCurrent = PressureTotalDo();

	/* make sure this is set by one of the following branches */
	pressure.PressureCorrect = 0.;

	/* this is special case where we are working on first zone, at
	 * illuminated face of the cloud.  Here we want to remember the
	 * initial pressure in case constant pressure is needed */
	if( nzone == 1 )
	{
		/* this is first zone, lock onto pressure */
		pressure.PressureInit = pressure.PressureCurrent;
		pressure.PressureRamInit = RamPressureDo();
		if( trace.lgTrace )
		{
			fprintf( ioQQQ, 
			  " PressureChange called, this is first zone, so reseting pressure to%10.3e\n", 
			  pressure.PressureInit );
		}
	}

	/* remember old hydrogen density */
	hold = phycon.hden;

	/* inside out globule */
	if( strcmp(pressure.chCPres,"GLOB") == 0 )
	{
		/* GLBDST is distance from globule, or glbrad-DEPTH */
		if( radius.glbdst < 0. )
		{
			fprintf( ioQQQ, " Globule distance is negative, internal overflow has occured,  sorry.\n" );
			fprintf( ioQQQ, " This is routine PressureChange, GLBDST is%10.2e\n", 
			  radius.glbdst );
			puts( "[Stop in PressureChange]" );
			cdEXIT(EXIT_FAILURE);
		}
		factor = (radius.glbden*pow(radius.glbrad/(radius.glbdst),radius.glbpow))/
		  phycon.hden;
		pressure.PressureCorrect = pressure.PressureCurrent*factor;
	}

	/* this is positive wind velocity the outflowing wind beyond sonic point */
	else if( (strcmp(pressure.chCPres,"WIND") == 0) && wind.windv > 0. )
	{

		static double rsave = 0.;
		/* this is logic for "old" wind solution,
		 * which assumes positive velocity, well above sonic point.
		 * following makes sure wind v only updated once per zone */
		if( radius.depth != rsave )
		{
			rsave = radius.depth;
			/* evaluate radiative acceleration */
			radacl();
			/* Wind model
			 * G, COMASS = mass of star in solar masses */
			wind.agrav = (float)((6.67e-8*wind.comass)*(SOLAR_MASS/radius.Radius)/
			  radius.Radius);

			/* is this the first zone? */
			if( nzone > 2 )
			{
				/* acceleration due to grad P(rad), xMassDensity is gm per cc */
				wind.AccelPres = (float)(-(pressure.PressureRadiation - pold)/radius.drad/
				  phycon.xMassDensity);
			}
			else
			{
				wind.AccelPres = 0.;
			}

			/* this is numerically unstable */
			wind.AccelPres = 0.;

			/* AccelXXX is computed in radinc, is continuum and line acceleration
			 * AccelCont is continuum accel 
			 * AccelLine is line accel 
			 * AccelPres is gradient of local gas pressure */
			wind.AccelTot = wind.AccelCont + wind.AccelLine + wind.AccelPres;

			/* remember largest radiative acceleration */
			wind.AccelMax = (float)MAX2(wind.AccelMax,wind.AccelTot);

			/* keep track of average acceleration */
			wind.AccelAver += wind.AccelTot*(float)radius.dReff;
			wind.acldr += (float)radius.dReff;

			/* following is form of energy equation, only used to check if neg vel */
			term = POW2(wind.windv) + 2.*(wind.AccelTot - wind.agrav)*
			  radius.drad;

			/* remember the current wind velocity
			vold = wind.windv; */
			if( term <= 1e3 )
			{
				/* wind vel is way below sonic point, give up, do not change vel */
				wind.lgVelPos = FALSE;
			}
			else
			{
				/* square of new wind velocity for outer edge of this zone */
				windnw = (double)(wind.windv*wind.windv) + (double)(2.*
				  (wind.AccelTot-wind.agrav))*radius.drad;

				/* WINDV is velocity at OUTER edge of this zone */
				wind.windv = (float)sqrt(windnw);
				wind.lgVelPos = TRUE;
			}

			/* following needed for expansion cooling */
			dndt.dDensityDT = (float)(wind.AccelTot/wind.windv + 2.*wind.windv/
			  radius.Radius);

			/* following used to control new DRAD
			wind.dVeldRad = (float)(fabs(wind.windv-vold)/wind.windv/radius.drad); */

		}
		else
		{
			factor = 1.;
		}

		factor = wind.emdot/(wind.windv*phycon.hden)/radius.r1r0sq;
		pold = pressure.PressureRadiation;
		pressure.PressureCorrect = pressure.PressureCurrent * factor;
	}

	/* this is negative wind velocity the new dynamics */
	else if( (strcmp(pressure.chCPres,"WIND") == 0) && wind.windv < 0. )
	{
		/* sets pressure.PressureCorrect  */
		factor = DynaChangeDensity();
	}

	else if( strcmp(pressure.chCPres,"TIME") == 0 )
	{
		/* called by TIMER for N = N(T) */
		fprintf( ioQQQ, " TIME does not now work.  Sorry.\n" );
		factor = timefcCom.timefc;
		puts( "[Stop in PressureChange]" );
		cdEXIT(EXIT_FAILURE);
	}

	else if( strcmp(pressure.chCPres,"SINE") == 0 )
	{
		/* rapid density fluctuation */
		factor = (fluct.cfirst*cos(radius.depth*fluct.flong+fluct.flcPhase) + 
		  fluct.csecnd)/phycon.hden;
		pressure.PressureCorrect = pressure.PressureCurrent*factor;
	}

	else if( strcmp(pressure.chCPres,"POWR") == 0 )
	{
		/* power law function of radius */
		dnew = denpow.den0*pow(radius.Radius/radius.rinner,(double)denpow.DensityPower);
		factor = dnew/phycon.hden;
		pressure.PressureCorrect = pressure.PressureCurrent*factor;
	}

	else if( strcmp(pressure.chCPres,"POWD") == 0 )
	{
		/* power law function of depth */
		dnew = denpow.den0*pow(1. + radius.depth/denpow.rscale,(double)denpow.DensityPower);
		factor = dnew/phycon.hden;
		pressure.PressureCorrect = pressure.PressureCurrent*factor;
	}

	else if( strcmp(pressure.chCPres,"POWC") == 0 )
	{
		/* power law function of column density */
		dnew = denpow.den0*pow(1.f + coldenCom.colden[ipCOLUMNDENSITY]/
		  denpow.rscale,denpow.DensityPower);
		factor = dnew/phycon.hden;
		pressure.PressureCorrect = pressure.PressureCurrent*factor;
	}

	else if( strcmp(pressure.chCPres,"CPRE") == 0 )
	{
		/* constant pressure */
		if( pressure.lgConPres )
		{
			/* >>chng 01 oct 31, replace pneed with CorretPressure */
			/* this has pressure due to incident continuum */
			pressure.PressureCorrect = pressure.PressureInit + pressure.PresInteg;
		}
		else
		{
			/* this does not have pressure due to incident continuum*/
			pressure.PressureCorrect = pressure.PressureInit*
				/* following term normally unity, power law set with option par on cmmnd*/
				pow(radius.Radius/radius.rinner,(double)pressure.PresPowerlaw);
		}

		/* ratio of correct to current pressures */
		factor = pressure.PressureCorrect / pressure.PressureCurrent;

		/*pressure.PressureCorrect = pressure.PressureCurrent*factor;*/
		/* >>chng 01 oct 31, this was not used, comment out */
		/*pold = pressure.PressureCurrent;*/
	}

	else if( strncmp( pressure.chCPres ,"DLW" , 3) == 0 )
	{
		if( strcmp(pressure.chCPres,"DLW1") == 0 )
		{
			/* call ACF sub */
			factor = fabden(radius.Radius,radius.depth)/phycon.hden;
		}
		else if( strcmp(pressure.chCPres,"DLW2") == 0 )
		{
			/* call table interpolation subroutine
			 * >>chng 96 nov 29, added tabden */
			factor = tabden(radius.Radius,radius.depth)/phycon.hden;
		}
		else
		{
			fprintf( ioQQQ, " Insanity, PressureChange gets chCPres=%4.4s\n", 
			  pressure.chCPres );
			puts( "[Stop in PressureChange]" );
			cdEXIT(EXIT_FAILURE);
		}
		pressure.PressureCorrect = pressure.PressureCurrent*factor;
	}

	else if( strcmp(pressure.chCPres,"CDEN") == 0 )
	{
		/* this is the default, constant density */
		factor = 1.;
		pressure.PressureCorrect = pressure.PressureCurrent*factor;
	}

	else
	{
		fprintf( ioQQQ, " Unknown pressure law=%4.4s STOP.\n", 
		  pressure.chCPres );
		puts( "[Stop in PressureChange]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* now see whether current pressure is within error bounds */
	if( factor > 1. + phycon.PressureError || factor < 1. - phycon.PressureError )
	{
		conv.lgConvPres = FALSE;
	}
	else
	{
		conv.lgConvPres = TRUE;
	}

	/* one of the branches above must have reset this variable,
	 * and it was init to 0 at start.  Confirm that non-zero */
	ASSERT( pressure.PressureCorrect > FLT_MIN );

	/* check whether pressure is within error tolerance,
	 * this check should agree with previous, but uses different set of variables */
	if( fabs(1.-pressure.PressureCurrent/pressure.PressureCorrect) > phycon.PressureError )
	{
		conv.lgConvPres = FALSE;
	}

	/* >>chng 01 nov 08 pass logical var saying whether sign of change in pressure has
	 * changed, indicating wwe are close to soln - use bigger steps if 
	 * change in sign has not occurred */
	if( lgPresOscil )
	{
		/* near solution since change in pressure has changed sign */
		pdelta = 1.;
	}
	else
	{
		pdelta = 4.;
	}

	/* make sure that change is not too extreme */
	factor = MIN2(factor,1.+pdelta*phycon.PressureError);
	factor = MAX2(factor,1.-pdelta*phycon.PressureError);

	{
		/*@-redef@*/
		enum{DEBUG=FALSE};
		static long int nsave=-1;
		/*@+redef@*/
		if( DEBUG /*&& nzone > 150 && iteration > 1*/ )
		{
			if( nsave-nzone ) fprintf(ioQQQ,"\n");
			nsave = nzone;
			fprintf(ioQQQ,"nnzzone\t%li\t%.2f%%\t%.3f\t%.2e\t%.2e\t%.2e\t%.2e\n", 
				nzone,
				factor,
				/* when this is negative we need to raise the density */
				(pressure.PressureCorrect-pressure.PressureCurrent)*100./pressure.PressureCorrect, 
				pressure.PressureCorrect,
			    pressure.PressureCurrent, 
				pressure.PressureGas,
				pressure.PressureRadiation ) ;
		}
	}

	/* >>chng 97 jun 03, added variable abundances for element table command
	 * option to get abundances off a table with element read command */
	if( abundances.lgAbTaON )
	{
		for( nelem=1; nelem < LIMELM; nelem++ )
		{
			if( abundances.lgAbunTabl[nelem] )
			{
				abun = tababun(radius.Radius,radius.depth,nelem+1)*
				  phycon.hden;

				hold = abun/abundances.gas_phase[nelem];
				abundances.gas_phase[nelem] = (float)abun;

				for( ion=0; ion < (nelem + 2); ion++ )
				{
					xIonFracs[nelem][ion] *= (float)hold;
				}
			}
		}
	}

	/* this happens if fluctuations abundances command entered */
	if( !fluct.lgDenFluc )
	{
		if( nzone <= 1 )
		{
			OldAbun = 1.;
			FacAbun = 1.;
			FacAbunSav = fluct.cfirst*cos(radius.depth*fluct.flong+
			  fluct.flcPhase) + fluct.csecnd;
		}

		else
		{
			OldAbun = FacAbunSav;
			/* rapid abundances fluctuation */
			FacAbunSav = fluct.cfirst*cos(radius.depth*fluct.flong+
			  fluct.flcPhase) + fluct.csecnd;
			FacAbun = FacAbunSav/OldAbun;
		}
	}

	else
	{
		FacAbun = 1.;
	}

	if( factor*FacAbun != 1. )
	{
		/* [nelem][0] is abundance of nelem
		 * H, He not affected by abundance fluctuations */
		for( nelem=0; nelem < 2; nelem++ )
		{
			xMolFracs[nelem] *= (float)factor;
			abundances.gas_phase[nelem] *= (float)factor;
			for( ion=0; ion < (nelem + 2); ion++ )
			{
				/* >>chng 96 jul 12 had only multiplied total abun, not ions */
				xIonFracs[nelem][ion] *= (float)factor;
			}
		}

		/* Lit on up is, so with FacAbun */
		for( nelem=2; nelem < LIMELM; nelem++ )
		{
			xMolFracs[nelem] *= (float)(factor*FacAbun);
			abundances.gas_phase[nelem] *= (float)(factor*FacAbun);
			for( ion=0; ion < (nelem + 2); ion++ )
			{
				/* >>chng 96 jul 12 had only multiplied total abun, not ions */
				xIonFracs[nelem][ion] *= (float)(factor*FacAbun);
			}
		}

		phycon.eden *= factor;

		/* >>chng 96 oct 25 update some te-ne variables */
		tfidle(FALSE);
		phycon.hden *= (float)factor;
		he.hei1 *= (float)factor;
		he.hei3 *= (float)factor;

		/* molecules done in hmole */
		hmi.htwo *= (float)factor;
		hmi.h2plus *= (float)factor;
		hmi.hminus *= (float)factor;
		hmi.hehp *= (float)factor;

		/* molecules done in comole */
		for( ion=0; ion < NHEVML; ion++ )
		{
			hevmolec.hevmol[ion] *= (float)FacAbun;
		}
	}

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, 
		  " PressureChange called, changing HDEN from %10.3e to %10.3e Set fill fac to %10.3e\n", 
		  hold, phycon.hden, filfac.FillFac );

		if( trace.lgNeBug )
		{
			fprintf( ioQQQ, " EDEN change PressureChange from to %10.3e %10.3e %10.3e\n", 
			  edensave, phycon.eden, edensave/phycon.eden );
		}
	}

	{
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG && (nzone>215)/**/ )
		{
			fprintf( ioQQQ, 
				"%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%c\n", 
			  radius.depth, 
			  pressure.PressureCurrent, 
			  pressure.PressureInit + pressure.PresInteg, 
			  pressure.PressureInit, 
			  pressure.PressureGas, 
			  pressure.PressureRam, 
			  pressure.PressureRadiation, 
			  /* subtract continuum rad pres which has already been added on */
			  pressure.PresInteg - pressure.pinzon, 
			  wind.windv/1e5,
			  sqrt(5.*pressure.PressureGas/3./phycon.xMassDensity)/1e5,
			  TorF(conv.lgConvPres) );
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->PressureChange()\n", debug_fp );
#	endif
	return;
}

