/*PressureTotalDo determine the gas and line radiation pressures for current conditions,
 * return value is total pressure */
/*RamPressureDo determine the ram pressure for current conditions */
#include "cddefines.h"
#include "taulines.h"
#include "physconst.h"
#include "hydrogenic.h"
#include "iso.h"
#include "abundances.h"
#include "wind.h"
#include "struc.h"
#include "phycon.h"
#include "hmi.h"
#include "h2.h"
#include "hevmolec.h"
#include "pop371.h"
#include "ionfracs.h"
#include "trace.h"
#include "rtescprob.h"
#include "atomcwgt.h"
#include "doppvel.h"
#include "ph1com.h"
#include "heavy.h"
#include "texcline.h"
#include "ionrange.h"
#include "pressure.h"

/* return value is total pressure */
double PressureTotalDo(void)
{
	long int i, 
		ion,
		/* will be used to point to line causing radiation pressure */
	  ip1=LONG_MAX,
	  ip2=LONG_MAX,
	  ip3=LONG_MAX,
	  ip4=LONG_MAX,
	  ipISO ,
	  ipHi, 
	  ipLo, 
	  ipSet , /* used in debug print statement to see which line adds most pressure */
	  nelem;

	/* will be used to check on size of opacity, was capped at this value */
	float smallfloat=SMALLFLOAT*10.f;

	double 
		Energy12,
		Energy13,
		Piso_seq[NISO],
		PLevel1=0.,
		PLevel2=0.,
		PHfs = 0.,
		PCO=0.,
		P_H2=0.,
		P_FeII=0.,
	  RadPres1, 
	  TotalPressure_v, 
	  p2, 
	  pmx;

#	ifdef DEBUG_FUN
	fputs( "<+>PressureTotalDo()\n", debug_fp );
#	endif

	/* PressureTotalDo - evaluate total pressure
	 * this function evaluates the following
	 * PressureTotalDo - gas and radiation pressure, dynes/cm^2
	 * GasPres gas pressure, same units
	 *
	 * HDEN is defined as total hydrogen nuclei phycon; all forms
	 *
	 * phycon.TotalNuclei is total number of nuclei and molecules
	 * nelem,0 is abundance of nelem */

	/* find total number of nuclei, and total ionization energy of gas */
	phycon.TotalNuclei = 0.;
	phycon.EnergyIonization = 0.;
	phycon.EnergyBinding = 0.;
	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		/* sum of all gas phase elements - [0] is the sum of the densities of all stages
		 * of ionization of the atom nelem */
		phycon.TotalNuclei += abundances.gas_phase[nelem];
		for( ion=IonRange.IonLow[nelem]; ion<=IonRange.IonHigh[nelem]; ++ion )
		{
			for( i=1; i<=ion; ++i )
			{
				long int nelec = nelem+1 - i ;
				/* this is the sum of all the energy needed to bring the atom up
				 * to the ion+1 level of ionization */
				phycon.EnergyIonization += xIonFracs[nelem][ion] * 
					PH1COM.PH1[Heavy.nsShells[nelem][i-1]-1][nelec][nelem][0]/EVRYD* 0.9998787;
			}
		}
	}

	/* convert to ergs/cm^3 */
	phycon.EnergyIonization *= EN1RYD;

	/* add on molecules */
	for( i=0; i < NHEVML; i++ )
	{
		phycon.TotalNuclei += hevmolec.hevmol[i];
	}

	/* do not double count C+, which is in this network as ipCP */
	phycon.TotalNuclei -= hevmolec.hevmol[ipCP];

	/* add on hydrogen molecules */
	phycon.TotalNuclei += hmi.htwo + hmi.h2plus + hmi.hminus + hmi.h3plus;

	/* phycon.ElecFrac is electron fraction, used for secondary ionization efficiency */
	phycon.ElecFrac = (float)(phycon.eden/phycon.TotalNuclei);

	/* pden is the total number of particles per unit vol */
	phycon.pden = (float)(phycon.eden + phycon.TotalNuclei);

	/* this is total neutral particle phycon for collisional ionization */
	phycon.CollidDensity = 0.;
	for( i=0; i < LIMELM; i++ )
	{
		phycon.CollidDensity += xIonFracs[i][0];
	}

	/* now add all the heavy molecules */
	for( i=0; i < NHEVML; i++ )
	{
		phycon.CollidDensity += hevmolec.hevmol[i];
	}

	/* do not double count C+, which is in this network as ipCP */
	phycon.CollidDensity -= hevmolec.hevmol[ipCP];

	phycon.CollidDensity += (float)(hmi.h2plus + hmi.hminus + hmi.h3plus + 
	  2.*hmi.htwo);

	/* gas pressure */
	pressure.PressureGas = (float)(phycon.pden*phycon.te*BOLTZMANN);

	/* WMOLE is molecular weight, AtomicMassUnit per particle */
	phycon.wmole = 0.;
	for( i=0; i < LIMELM; i++ )
	{
		phycon.wmole += abundances.gas_phase[i]*(float)AtomcWgt.AtomicWeight[i];
	}

	phycon.wmole += (float)(hevmolec.hevmol[ipCH]*13. + hevmolec.hevmol[ipCHP]*
	  13. + hevmolec.hevmol[ipOH]*17. + hevmolec.hevmol[ipOHP]*
	  17. + hevmolec.hevmol[ipOTWO]*32. + hevmolec.hevmol[ipCTWO]*
	  24. + hevmolec.hevmol[ipCO]*28. + hevmolec.hevmol[ipCOP]*
	  28. + hevmolec.hevmol[ipH2O]*18. + hevmolec.hevmol[ipH2OP]*
	  18. + hevmolec.hevmol[ipO2P]*32. + hevmolec.hevmol[ipC2P]*
	  24. + hevmolec.hevmol[ipH3P]*3. + hevmolec.hevmol[ipH3OP]*
	  19. + hevmolec.hevmol[ipCH2P]*14. + hevmolec.hevmol[ipCH2]*
	  14. + hevmolec.hevmol[ipCH3]*15.);

	phycon.wmole /= phycon.pden;
	ASSERT( phycon.wmole > 0. && phycon.pden > 0. );

	/* this is Lya excitation temperature, has always been evaluated here */
	hydro.TexcLya = (float)TexcLine( &EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s] );

	/* xMassDensity is density in grams per cc */
	phycon.xMassDensity = (float)(phycon.wmole*ATOMIC_MASS_UNIT*phycon.pden);

	/* >>chng 01 nov 02, move to here from dynamics routines */
	/* this is the current ram pressure, at this location,
	 * ratio of masses is to get local velocity from mass conservation */
	if( nzone>1 )
	{
		/* we have a valid solution, are into model with all vars defined */
		ASSERT( struc.DenMass[0] > 0. );
		pressure.PressureRam = phycon.xMassDensity*
			POW2(wind.windv0*struc.DenMass[0]/phycon.xMassDensity);
	}
	else
	{
		/* we are searching for first solution, or working on first zone */
		pressure.PressureRam = RamPressureDo();
	}

	/* RadPres is pressure due to lines, lgRadPresON turns off or on */
	pressure.PressureRadiation = 0.;
	pmx = 0.;

	/* this is the total Lya line width, in velocity units, cm/s
	 * >>chng 00 dec 04, this had been inside if( pressure.lgRadPrsON but
	 * must always be evaluated since needed for pumping of FeII by Lya */
	/* if statement stops evaluation before initial quantities set */
	if( EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].PopOpc > 0. )
	{
		hydro.HLineWidth = (float)(
			RT_LyaWidth(
			EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].TauIn,
		   EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].TauTot,
		   4.72e-2/phycon.sqrte,
			DoppVel.doppler[ipHYDROGEN] ) );
		ASSERT( hydro.HLineWidth >= 0.);
	}
	else
	{
		hydro.HLineWidth = 0.;
	}

	/* find max rad pressure even if capped
	 * lgRadPrsON is turned off by NO RADIATION PRESSURE command */
	if( pressure.lgRadPrsON )
	{
		nelem = ipHYDROGEN;
		ipLo = ipH1s;
		ipHi = ipH2p;
		/* needs to be larger than a small float since this is capped whe
		 * defined*/
		if( EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopHi > smallfloat &&
			EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc > smallfloat )
		{
			/* this is the 2p/1s level population, will be used for excitation
			 * temperature and radiation pressure */
			p2 = EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopHi/EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc/3.;

			/* coefficient assumes that HLineWidth is half width, not total */
			pressure.HLalpha = (float)(p2*5.08e-6*hydro.HLineWidth);
		}
		else
		{
			p2 = 0.;
			pressure.HLalpha = 0.;
		}
		pressure.PressureRadiation = pressure.HLalpha;
		ASSERT( pressure.HLalpha >= 0. );
		/*fprintf(ioQQQ,"presbugg %e\n", pressure.HLalpha);*/

		/* these identify La as the pressure we have so far */
		pmx = pressure.HLalpha;
		ip1 = ipH2p;
		ip2 = ipH1s;
		ip3 = ipHYDROGEN;
		ip4 = ipH_LIKE;
		ipSet = 2;
		for( ipISO=ipHYDROGEN; ipISO<NISO; ++ipISO )
		{
			Piso_seq[ipISO] = 0.;
			/* >>chng 01 sep 23, rewrote for iso sequences */
			for( nelem=ipISO; nelem < LIMELM; nelem++ )
			{
				if( IonRange.IonHigh[nelem] >= nelem + 1 - ipISO  )
				{
					/* >>chng 01 jan 08, do not include highest level since maser can occur,
					* and pops are set to small number in this case */
					for( ipLo=0; ipLo < (iso.numLevels[ipISO][nelem] - 2); ipLo++ ) 
					{
						for( ipHi=ipLo+1; ipHi < iso.numLevels[ipISO][nelem]-1; ipHi++ )
						{
							/* needs to be larger than a small float since this is capped whe
							* defined*/
							if( EmisLines[ipISO][nelem][ipHi][ipLo].PopHi > smallfloat &&
								EmisLines[ipISO][nelem][ipHi][ipLo].PopOpc > smallfloat &&
								/* >>chng 01 nov 01, add test that have not overrun optical depth scale */
								( (EmisLines[ipISO][nelem][ipHi][ipLo].TauTot - EmisLines[ipISO][nelem][ipHi][ipLo].TauIn) > smallfloat ) )
							{
								RadPres1 = 5.551e-2*
								(powi(EmisLines[ipISO][nelem][ipHi][ipLo].EnergyWN/1.e6,4))*
								(EmisLines[ipISO][nelem][ipHi][ipLo].PopHi/
								iso.stat[ipISO][nelem][ipHi])/
								(EmisLines[ipISO][nelem][ipHi][ipLo].PopOpc/
								iso.stat[ipISO][nelem][ipLo])*
								/* RT_LineWidth gets linewidth in terms of RT effects */
								RT_LineWidth(&EmisLines[ipISO][nelem][ipHi][ipLo])* 2.;
								/* H I lya was set above - do not want to reset it here,
								 * which would double count the line */
								if( ipISO==ipH_LIKE && ipLo==ipH1s && ipHi==ipH2p )
								{
									RadPres1 = 0.;
								}
								if( RadPres1 > pmx )
								{
									ipSet = 2;
									pmx = RadPres1;
									ip1 = ipHi;
									ip2 = ipLo;
									ip3 = nelem;
									ip4 = ipISO;
								}
								Piso_seq[ipISO] += RadPres1;
								pressure.PressureRadiation += (float)RadPres1;
							}
						}
					}
				}
			}
			ASSERT( Piso_seq[ipISO] >= 0. );
		}
		{
			/* option to print particulars of some line when called */
			/*@-redef@*/
			enum {DEBUG=FALSE};
			/*@+redef@*/
			if( DEBUG && iteration > 1 && nzone > 197 )
			{
				ip1 = 6;ip2 = 3; ip3 = 0;
				fprintf(ioQQQ,
					"%li\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n",
					nzone,
					pmx,
					EmisLines[ipISO][ip3][ip1][ip2].PopOpc,
					EmisLines[ipISO][ip3][7][0].PopLo,
					EmisLines[ipISO][ip3][7][2].PopLo,
					EmisLines[ipISO][ip3][7][3].PopLo,
					EmisLines[ipISO][ip3][7][4].PopLo,
					EmisLines[ipISO][ip3][7][5].PopLo,
					EmisLines[ipISO][ip3][7][6].PopLo);
			}
#			if 0
			if( DEBUG && iteration > 1 && nzone > 150 )
			{
				ip1 = 6;ip2 = 3; ip3 = 0;
				fprintf(ioQQQ,
					"%li\t%.2e\t%li\t%li\t%li\t%.2e\t%.2e\n",
					nzone,
					pmx,
					ip1,
					ip2,
					ip3,
					EmisLines[ipISO][ip3][ip1][ip2].PopHi,
					EmisLines[ipISO][ip3][ip1][ip2].PopOpc );
			}
#			endif
		}
#		if 0
		HeLike = 0.;
		/* line radiation pressure for he-like lines */
		for( nelem=1; nelem < LIMELM; nelem++ )
		{

			if( IonRange.IonHigh[nelem] >= nelem )
			{
				for( ipHi=1; ipHi <iso.numLevels[ipHE_LIKE][nelem]; ipHi++ )
				{
					for( ipLo=0; ipLo < ipHi; ipLo++ ) 
					{
						/* needs to be larger than a small float since this is capped whe
						 * defined*/
						if( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopHi > smallfloat &&
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopOpc > smallfloat )
						{
							RadPres1 = 5.551e-2*
							  (powi(EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN/1.e6,4))*
							  (EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopHi/
							  iso.stat[ipHE_LIKE][nelem][ipHi])/
							  (EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopOpc/
							  iso.stat[ipHE_LIKE][nelem][ipLo])*
							  /* RT_LineWidth gets linewidth in terms of RT effects,
							   * in rtescprob */
							  RT_LineWidth(&EmisLines[ipHE_LIKE][nelem][ipHi][ipLo])* 2.;
							if( RadPres1 > pmx )
							{
								ipSet = 7;
								pmx = RadPres1;
								ip1 = ipHi;
								ip2 = ipLo;
								ip3 = nelem;
								/*
								if( nzone>215 ) fprintf(ioQQQ,"%li %li %li hi %.2e opc %.2e %.2e %.2e %.2e\n", 
									ip1 , ip2 , ip3 , 
									EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopHi ,
									EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopOpc,
									EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].TauIn,
									EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Pesc,
									EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Pdest);*/
							}
							HeLike += (float)RadPres1;
							pressure.PressureRadiation += (float)RadPres1;
						}
					}
				}
			}
		}
		ASSERT( HeLike >= 0. );
		{
			/* option to print particulars of some line when called */
			/*@-redef@*/
			enum {DEBUG=FALSE};
			/*@+redef@*/
			if( DEBUG )
			{
				if( /*pressure.pbeta >0.2 &&*/ nzone > 215 )
				{
					fprintf(ioQQQ,
						"PressureTotalDo zone %li Prad/Pgas%e , He %e pmx %e ip1 %li ip2 %li ip3 %li\n",
					nzone,
					pressure.pbeta,
					HeLike,
					pmx,
					ip1,
					ip2,
					ip3
					);
				}
			}
		}
#		endif

		PLevel1 = 0.;
		/* line radiation pressure from large set of level 1 lines */
		for( i=1; i <= nLevel1; i++ )
		{
			if( TauLines[i].PopHi > 1e-30 )
			{
				RadPres1 = 5.551e-2*(powi(TauLines[i].EnergyWN/
				  1.e6,4))*(TauLines[i].PopHi/TauLines[i].gHi)/
				  (TauLines[i].PopLo/TauLines[i].gLo)*
				  RT_LineWidth(&TauLines[i])*2.;
				if( RadPres1 > pmx )
				{
					ipSet = 3;
					pmx = RadPres1;
					ip1 = i;
				}
				PLevel1 += (float)RadPres1;
				pressure.PressureRadiation += (float)RadPres1;
			}
		}
		ASSERT( PLevel1 >= 0. );

		/* level 2 lines */
		PLevel2 = 0.;
		for( i=0; i < nWindLine; i++ )
		{
			if( TauLine2[i].PopHi > 1e-30 )
			{
				RadPres1 = 5.551e-2*(powi(TauLine2[i].EnergyWN/
				  1.e6,4))*(TauLine2[i].PopHi/TauLine2[i].gHi)/
				  (TauLine2[i].PopLo/TauLine2[i].gLo)*
				  RT_LineWidth(&TauLine2[i])*2.;
				pressure.PressureRadiation += (float)RadPres1;
				PLevel2 += (float)RadPres1;
				if( RadPres1 > pmx )
				{
					ipSet = 4;
					pmx = RadPres1;
					ip1 = i;
				}
			}
		}
		ASSERT( PLevel2 >= 0. );

		/* level 2 lines */
		PHfs = 0.;
		for( i=0; i < nHFLines; i++ )
		{
			if( HFLines[i].PopHi > 1e-30 )
			{
				RadPres1 = 5.551e-2*(powi(HFLines[i].EnergyWN/
				  1.e6,4))*(HFLines[i].PopHi/HFLines[i].gHi)/
				  (HFLines[i].PopLo/HFLines[i].gLo)*
				  RT_LineWidth(&HFLines[i])*2.;
				pressure.PressureRadiation += (float)RadPres1;
				PHfs += (float)RadPres1;
				if( RadPres1 > pmx )
				{
					ipSet = 8;
					pmx = RadPres1;
					ip1 = i;
				}
			}
		}
		ASSERT( PHfs >= 0. );

		/* radiation pressure due to H2 lines */
		P_H2 = H2_RadPress();
		pressure.PressureRadiation += (float)P_H2;

		/* radiation pressure due to FeII lines and large atom */
		P_FeII = FeIIRadPress();
		pressure.PressureRadiation += (float)P_FeII;

		PCO = 0.;
		/* co carbon monoxide lines */
		for( i=0; i < nCORotate; i++ )
		{
			if( C12O16Rotate[i].PopHi > 1e-30 )
			{
				RadPres1 = 5.551e-2*(powi(C12O16Rotate[i].EnergyWN/
				  1.e6,4))*(C12O16Rotate[i].PopHi/C12O16Rotate[i].gHi)/
				  (C12O16Rotate[i].PopLo/C12O16Rotate[i].gLo)*
				  RT_LineWidth(&C12O16Rotate[i])*2.;
				pressure.PressureRadiation += (float)RadPres1;
				PCO += (float)RadPres1;
				if( RadPres1 > pmx )
				{
					ipSet = 5;
					pmx = RadPres1;
					ip1 = i;
				}
			}
			if( C13O16Rotate[i].PopHi > 1e-30 )
			{
				RadPres1 = 5.551e-2*(powi(C13O16Rotate[i].EnergyWN/
				  1.e6,4))*(C13O16Rotate[i].PopHi/C13O16Rotate[i].gHi)/
				  (C13O16Rotate[i].PopLo/C13O16Rotate[i].gLo)*
				  RT_LineWidth(&C13O16Rotate[i])*2.;
				pressure.PressureRadiation += (float)RadPres1;
				PCO += (float)RadPres1;
				if( RadPres1 > pmx )
				{
					ipSet = 6;
					pmx = RadPres1;
					ip1 = i;
				}
			}
		}
		ASSERT( PCO >= 0. );

	}
	else
	{
		/* case where radiation pressure is not turned on */
		ip1 = 0;
		ipSet = 0;
	}

	/* PBETA is ratio of radiation to gas + continuum pressure */
	if( pressure.lgConPres )
	{
		/* this includes integrated incident continuum */
		pressure.pbeta = (float)(pressure.PressureRadiation/(pressure.PressureGas + pressure.PresInteg));
	}
	else
	{
		/* this does not */
		pressure.pbeta = (float)(pressure.PressureRadiation/pressure.PressureGas);
	}

	{
		/* option to print particulars of some line when called */
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		/*lint -e644 Piso_seq not initialized */
		if( DEBUG && iteration > 1 && nzone > 150 )
		{
			fprintf(ioQQQ,
				"PressureTotalDo zone %li Prad/Pgas%.2e , constituents follow H%e He%e L1%.2e L2%.2e CO%.2e fs%.2e h2%.2e\n",
			nzone,
			pressure.pbeta,
			Piso_seq[ipH_LIKE],
			Piso_seq[ipHE_LIKE],
			PLevel1,
			PLevel2,
			PCO,
			PHfs,
			P_H2);
		}
		/*lint +e644 Piso_seq not initialized */
	}

	/* this would be a major logical error .. .. . */
	if( pressure.PressureRadiation < 0. )
	{
		fprintf(ioQQQ,
			"PressureTotalDo: negative pressure, constituents follow %e %e %e %e %e \n",
		Piso_seq[ipH_LIKE],
		Piso_seq[ipHE_LIKE],
		PLevel1,
		PLevel2,
		PCO);
		ShowMe();
		puts( "[Stop in PressureTotalDo]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* following test will never suceed, here to trick lint, ipSet is only used
	 * when needed for debug */
	if( trace.lgTrace && ipSet <0 )
	{
		fprintf(ioQQQ,
			"      PressureTotalDo, pressure.pbeta = %e, ipSet%li ip1=%li \n", 
			pressure.pbeta,ipSet,ip1 );
	}

	/* this is the total internal energy of the gas, the amount of energy needed
	 * to produce the current level populations, relative to ground */
	phycon.EnergyExcitation = 0.;
	ipLo = 0;
	for( ipISO=ipHYDROGEN; ipISO<NISO; ++ipISO )
	{
		for( nelem=ipISO; nelem < LIMELM; nelem++ )
		{
			if( IonRange.IonHigh[nelem] == nelem + 1 - ipISO )
			{
				for( ipHi=1; ipHi < iso.numLevels[ipISO][nelem]; ipHi++ ) 
				{
					phycon.EnergyExcitation += 
						EmisLines[ipISO][nelem][ipHi][ipLo].PopHi * 
						EmisLines[ipISO][nelem][ipHi][ipLo].EnergyErg*
						xIonFracs[nelem][nelem+2-ipISO-1];
				}
			}
		}
	}
#	if 0
	/* internal energy due to he-like lyman lines */
	for( nelem=1; nelem < LIMELM; nelem++ )
	{
		if( IonRange.IonHigh[nelem] >= nelem )
		{
			for( ipHi=ipLo+1; ipHi <iso.numLevels[ipHE_LIKE][nelem]; ipHi++ )
			{
				phycon.EnergyExcitation += 
					EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopHi* 
					EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyErg*
					xIonFracs[nelem][nelem];
			}
		}
	}
#	endif

	for( i=1; i <= nLevel1; i++ )
	{
		phycon.EnergyExcitation += 
			TauLines[i].PopHi* TauLines[i].EnergyErg;
	}
	for( i=0; i < nWindLine; i++ )
	{
		phycon.EnergyExcitation += 
			TauLine2[i].PopHi* TauLine2[i].EnergyErg;
	}
	for( i=0; i < nHFLines; i++ )
	{
		phycon.EnergyExcitation += 
			HFLines[i].PopHi* HFLines[i].EnergyErg;
	}
	Energy12 = 0.;
	Energy13 = 0.;
	for( i=0; i < nCORotate; i++ )
	{
		Energy12 += C12O16Rotate[i].EnergyErg;
		phycon.EnergyExcitation += 
			C12O16Rotate[i].PopHi* Energy12;

		Energy13 += C13O16Rotate[i].EnergyErg;
		phycon.EnergyExcitation += 
			C13O16Rotate[i].PopHi* Energy13;
	}

	/* internal energy of H2 */
	phycon.EnergyExcitation += H2_InterEnergy();

	/* internal energy of large FeII atom */
	phycon.EnergyExcitation += FeII_InterEnergy();

	/* ==================================================================
	 * end internal energy of atoms and molecules */

	/* stop model from running away on first iteration, when line optical
	 * depths are not defined correctly anyway.
	 * if( iter.le.itermx .and. RadPres.ge.GasPres ) then
	 * >>chng 97 jul 23, only cap radiation pressure on first iteration */
	if( iteration <= 1 && pressure.PressureRadiation >= pressure.PressureGas )
	{
		pressure.RadBetaMax = (float)MAX2(pressure.RadBetaMax,pressure.pbeta);
		/* stop RadPres from exceeding the gas pressure on first iteration */
		pressure.PressureRadiation = (float)MIN2(pressure.PressureRadiation,pressure.PressureGas);
		pressure.lgPradCap = TRUE;
	}

	/* remember the globally most important line, in the entire model */
	if( pressure.pbeta > pressure.RadBetaMax )
	{
		pressure.RadBetaMax = pressure.pbeta;
		pressure.ipPradMax = ip1;
		if( ipSet == 1 )
		{
			/* special first hydrogenic set */
			strcpy( pressure.chLineRadPres, "First 20" );
		}
		else if( ipSet == 2 )
		{
			/* hydrogenic */
			strcpy( pressure.chLineRadPres, "Hydro ");
			strcat( pressure.chLineRadPres, chLineLbl(&EmisLines[ip4][ip3][ip1][ip2]) );
		}
		else if( ipSet == 3 )
		{
			/* level 1 lines */
			strcpy( pressure.chLineRadPres, "Level1 ");
			strcat( pressure.chLineRadPres, chLineLbl(&TauLines[ip1]) );
		}
		else if( ipSet == 4 )
		{
			/* level 2 lines */
			strcpy( pressure.chLineRadPres, "Level2 ");
			strcat( pressure.chLineRadPres, chLineLbl(&TauLine2[ip1]) );
		}
		else if( ipSet == 5 )
		{
			/* c12o16 carbon monoxide lines */
			strcpy( pressure.chLineRadPres, "12CO ");
			strcat( pressure.chLineRadPres, chLineLbl(&C12O16Rotate[ip1]) );
		}
		else if( ipSet == 6 )
		{
			/* c13o16 carbon monoxide lines */
			strcpy( pressure.chLineRadPres, "13CO ");
			strcat( pressure.chLineRadPres, chLineLbl(&C13O16Rotate[ip1]) );
		}
		else if( ipSet == 7 )
		{
			/* helium lines */
			strcpy( pressure.chLineRadPres, "HeLike ");
			strcat( pressure.chLineRadPres, chLineLbl(&EmisLines[ipHE_LIKE][ip3][ip1][ip2]) );
		}
		else if( ipSet == 8 )
		{
			/* helium lines */
			strcpy( pressure.chLineRadPres, "hyperf ");
			strcat( pressure.chLineRadPres, chLineLbl(&HFLines[ip1]) );
		}
		else
		{
			fprintf(ioQQQ," PressureTotalDo ipSet insanity.  this is impossible.\n");
			ShowMe();
			puts( "[Stop in PressureTotalDo]" );
			cdEXIT(EXIT_FAILURE);
		}
	}

	/* radiation pressure only included in total eqn of state when this flag is
	 * true, set with constant pressure command */
	/* >>chng 01 nov 02, add ram pressure here */
	if( pressure.lgRadPresON )
	{
		TotalPressure_v = pressure.PressureGas + pressure.PressureRadiation + pressure.PressureRam;
	}
	else
	{
		TotalPressure_v = pressure.PressureGas + pressure.PressureRam;
	}

	{
		/*@-redef@*/
		enum{DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG && pressure.PressureCurrent > SMALLFLOAT /*&& iteration > 1*/ )
		{
			fprintf(ioQQQ,"pressureee%li\t%.2e\t%.2e\t%.2e\t%.3f\t%.3f\t%.3f\n", 
				nzone,
				pressure.PressureCorrect,
			    pressure.PressureCurrent, 
				TotalPressure_v ,
				pressure.PressureGas/pressure.PressureCurrent,
				pressure.PressureRadiation/pressure.PressureCurrent ,
				pressure.PressureRam/pressure.PressureCurrent
				);
		}
	}

	ASSERT( TotalPressure_v > 0. );

	/* remember highest pressure anywhere */
	pressure.PresMax = MAX2(pressure.PresMax,(float)TotalPressure_v);

#	ifdef DEBUG_FUN
	fputs( " <->PressureTotalDo()\n", debug_fp );
#	endif
	return( TotalPressure_v );
}
/* calculates the ram pressure using current density and velocity,
 * this is not used through most of calculation, where mass flux
 * conservation relates density and velocity.  That is explicitly
 * written above 
 */
double RamPressureDo(void)
{
	double RamPressure_v;

#	ifdef DEBUG_FUN
	fputs( "<+>RamPressureDo()\n", debug_fp );
#	endif

	ASSERT( phycon.xMassDensity > 0. );

	/* this is ram pressure - xMassDensity is density in gm/cm3,
	 * wind.windv is current wind velocity in cm/s */
	RamPressure_v = phycon.xMassDensity * POW2( wind.windv );

#	ifdef DEBUG_FUN
	fputs( " <->RamPressureDo()\n", debug_fp );
#	endif
	return( RamPressure_v );
}

