/*SanityCheck, check that various parts of the code still work, called by Cloudy after continuum
 * and optical depth arrays are set up, but before initial temperature and ionization */
#include "cddefines.h"
#include "expnf.h"
#include "abundances.h"
#include "linpack.h"
#include "helike.h"
#include "rfield.h"
#include "veclib.h"
#include "taulines.h"
#include "ionfracs.h"
#include "hypho.h"
#include "iso.h"
#include "phycon.h"
#include "opacity.h"
#include "bevington.h"
#include "hydro_bauman.h"
#include "hydrogenic.h"
#include "typmatrx.h"
#include "heavy.h"
#include "kshllenr.h"
#include "trace.h"
#include "sanitycheck.h"

/* NB - this routine must not change any global variables - any that are changed as part of
 * a test must be reset, so that the code retains state */

static void SanityCheckBegin(void );
static void SanityCheckFinal(void );
/* chJob is either "begin" or "final" 
 * "begin is before code starts up
 * "final is after model is complete */
void SanityCheck(char *chJob )
{
	if( strcmp(chJob,"begin") == 0 )
	{
		SanityCheckBegin();
	}
	else if( strcmp(chJob,"final") == 0 )
	{
		SanityCheckFinal();
	}
	else
	{
		fprintf(ioQQQ,"SanityCheck called with insane argument.\n");
		puts( "[Stop in SanityCheck]" );
		cdEXIT(EXIT_FAILURE);
	}
}

static void SanityCheckFinal(void )
{
	/* PrtComment also has some ending checks on sanity */
}

static void SanityCheckBegin(void )
{
	int lgOK=TRUE;
	int lgFlag;
	long ner , ipiv[3] , 
		i , 
		j , 
		job , 
		nelem , 
		ion ,
		nshells;

	/* this will be charge to the 4th power */
	double Aul ,
		error,
		Z4;

	float temp;
	long n;
	long ipISO;

#	define NDIM 10
	double x , ans1 , ans2  , xMatrix[NDIM][NDIM] , yVector[NDIM] ,
		rcond, 
	  work[NDIM];
	float *fvector;
	long int *ipvector;

#	ifdef DEBUG_FUN
	fputs( "<+>SanityCheck()\n", debug_fp );
#	endif

	/*********************************************************
	 *                                                       *
	 * confirm that various part of cloudy still work        *
	 *                                                       *
	 *********************************************************/

	/* if this is no longer true at end, we have a problem */
	lgOK = TRUE;

	/*********************************************************
	 *                                                       *
	 * check that all the Lyas As are ok                     *
	 *                                                       *
	 *********************************************************/
	for( nelem=0; nelem<LIMELM; ++nelem )
	{
		/* this element may be turned off */
		if( abundances.lgElmtOn[nelem] )
		{ 
			/* H_Einstein_A( n, l, np, lp, iz ) - all are on physics scale */
			Aul = H_Einstein_A( 2, 1, 1, 0, nelem+1 );
			/*fprintf(ioQQQ,"%li\t%.4e\n", nelem+1, EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Aul );*/
			if( fabs( Aul - EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Aul ) /Aul > 0.01 )
			{
				fprintf(ioQQQ," SanityCheck found insane H-like As.\n");
				lgOK = FALSE;
			}
		}
	}

	/*********************************************************
	 *                                                       *
	 * check some transition probabililties for he-like ions *
	 *                                                       *
	 *********************************************************/
	for( nelem=1; nelem<LIMELM; ++nelem )
	{
		/* the helike 9-1 transition, A(3^3P to 2^3S) */
		double as[]={
#		if 0
		   0.       , 9.40E+06 , 2.59E+08 , 1.59E+09 , 5.40E+09 , 1.36E+10 
		 , 2.86E+10 , 5.34E+10 , 9.25E+10 , 1.46E+11 , 2.27E+11 , 3.27E+11 
		 , 4.73E+11 , 6.41E+11 , 8.77E+11 , 1.14E+12 , 1.50E+12 , 1.91E+12 
		 , 2.40E+12 , 2.92E+12 , 3.66E+12 , 4.45E+12 , 5.36E+12 , 6.40E+12 
		 , 7.59E+12 , 8.80E+12 , 1.05E+13 , 1.22E+13 , 1.41E+13 , 1.62E+13
#		endif
		 /* updated with Johnson values */
		 0.       ,9.47e+006 ,3.44e+008 ,1.74e+009 ,5.51e+009 ,1.34e+010 ,
		2.79e+010 ,5.16e+010 ,8.81e+010 ,1.41e+011 ,2.15e+011 ,3.15e+011 ,
		4.46e+011 ,6.14e+011 ,8.26e+011 ,1.09e+012 ,1.41e+012 ,1.80e+012 ,
		2.26e+012 ,2.80e+012 ,3.44e+012 ,4.18e+012 ,5.04e+012 ,6.02e+012 ,
		7.14e+012 ,8.40e+012 ,9.83e+012 ,1.14e+013 ,1.32e+013 ,1.52e+013
		};

		if( iso.numLevels[ipHE_LIKE][nelem] > 8 )
		{	
			/* following used to print current values of As */
			/*fprintf(ioQQQ,"%.2e\n", EmisLines[ipHE_LIKE][nelem][9][1].Aul );*/
			if( fabs( as[nelem] - EmisLines[ipHE_LIKE][nelem][9][1].Aul ) /as[nelem] > 0.025 )
			{
				fprintf(ioQQQ,
					" SanityCheck found insane He-like As: expected, nelem=%li found: %.2e %.2e.\n",
					nelem,
					as[nelem] , 
					EmisLines[ipHE_LIKE][nelem][9][1].Aul );
				lgOK = FALSE;
			}
		}
	}

	/*********************************************************
	 *                                                       *
	 * check the threshold photoionization cs for He I       *
	 *                                                       *
	 *********************************************************/
	/* Don't check for sanity in these if running the Benjamin test...	*/
	if(  abundances.lgElmtOn[ipHELIUM] && !helike.lgSetBenjamin )
	{
		/* HeI photoionization cross sections at threshold for lowest 20 levels */
#		define NHE1CS	20
		double he1cs[NHE1CS] = 
		{
			5.480E-18 , 9.253E-18 , 1.598E-17 , 1.598E-17 , 1.598E-17 , 1.348E-17 , 
			8.025E-18 , 1.449E-17 , 2.852E-17 , 1.848E-17 , 1.813E-17 , 2.699E-17 , 
			1.077E-17 , 2.038E-17 , 4.159E-17 , 3.670E-17 , 3.575E-17 , 1.900E-17 , 
			1.900E-17 , 4.175E-17 
			/*		            , 1.33E-17 , 2.81E-17 , 5.79E-17 , 5.31E-17 , 
			5.31E-17 , 4.08E-17 , 4.07E-17 , 1.74E-17 , 1.74E-17 , 5.81E-17	*/
		};
		/* loop over levels and check on photo cross section */
		j = MIN2( NHE1CS+1 , iso.numLevels[ipHE_LIKE][ipHELIUM] );
		for( n=1; n<j; ++n )
		{
			/* above list of levels does not include the ground */
			i = iso.ipOpac[ipHE_LIKE][ipHELIUM][n];
			ASSERT( i>0 );
			/*fprintf(ioQQQ,"%li\t%lin", n , i );*/
			/* >>chng 02 apr 10, from 0.01 to 0.02, values stored
			 * where taken from calc at low contin resolution, when continuum
			 * resolution changed this changes too */
			/*fprintf(ioQQQ,"%li %.2e\n", n,( he1cs[n-1] - opac.OpacStack[i - 1] ) /he1cs[n-1] );*/
			if( fabs( he1cs[n-1] - opac.OpacStack[i - 1] ) /he1cs[n-1] > 0.02 )
			{
				fprintf(ioQQQ,
					" SanityCheck found insane HeI photo cs: expected, n=%li found: %.3e %.3e.\n",
					n,
					he1cs[n-1] , 
					opac.OpacStack[i - 1]);
				fprintf(ioQQQ,
					" n, l, s=%li %li  %lifound:\n",
					iso_quant_desig[ipHE_LIKE][ipHELIUM][n].n ,
					iso_quant_desig[ipHE_LIKE][ipHELIUM][n].l ,
					iso_quant_desig[ipHE_LIKE][ipHELIUM][n].s);
				lgOK = FALSE;
			}
		}


		/* check the recombination coefficients for he ground */
		error = fabs( HelikeCheckRecomb( ipHELIUM , 0 , 7500. ) );
		if( error > 0.01 )
		{
			fprintf(ioQQQ,
				" SanityCheck found insane1 HeI recom coef: expected, n=%i error: %.2e \n",
				0,
				error );
			lgOK = FALSE;
		}


		/* check the recombination coefficients for he 23P */
		error = fabs( HelikeCheckRecomb( ipHELIUM , 1 , 12500. ) );
		if( error > 0.01 )
		{
			fprintf(ioQQQ,
				" SanityCheck found insane2 HeI recom coef: expected, n=%i error: %.2e \n",
				1,
				error );
			lgOK = FALSE;
		}

	}

	/*********************************************************
	 *                                                       *
	 * check out the sorting routine                         *
	 *                                                       *
	 *********************************************************/

#	define NSORT 100 

	if( (fvector = (float *)MALLOC((NSORT)*sizeof(float) )) ==NULL )
		bad_malloc();

	if( (ipvector = (long *)MALLOC((NSORT)*sizeof(long int) )) ==NULL )
		bad_malloc();

	nelem = 1;
	/* make up some unsorted values */
	for( i=0; i<NSORT; ++i )
	{
		nelem *= -1;
		fvector[i] = (float)(nelem * (NSORT-i));
	}

	/*spsort netlib routine to sort array returning sorted indices */
	spsort(fvector, 
		   NSORT, 
		  ipvector, 
		  /* flag saying what to do - 1 sorts into increasing order, not changing
		   * the original routine */
		  1, 
		  &lgFlag);

	if( lgFlag ) lgOK = FALSE;

	for( i=1; i<NSORT; ++i )
	{
		/*fprintf(ioQQQ," %li %li %.0f\n", 
			i, ipvector[i],fvector[ipvector[i]] );*/
		if( fvector[ipvector[i]] <= fvector[ipvector[i-1]] )
		{
			fprintf(ioQQQ," SanityCheck found insane sort\n");
			lgOK = FALSE;
		}
	}

	free( fvector );
	free( ipvector);
#	undef NSORT 

	temp = (float)sqrt(phycon.te);
	/* check that the temperatures make sense */
	if( fabs(temp - phycon.sqrte )/temp > 1e-5 )
	{
		fprintf(ioQQQ , "SanityCheck finds insane te %e sqrt te %e sqrte %e dif %e\n",
			phycon.te , 
			sqrt(phycon.te) , 
			phycon.sqrte , 
			fabs(sqrt(phycon.te) - phycon.sqrte ) );
		lgOK = FALSE;
	}

	/*********************************************************
	 *                                                       *
	 * confirm that widflx and anu arrays correspond         *
	 * to one another                                        *
	 *                                                       *
	 *********************************************************/

#	if 0
	/* this check on widflx can't be used since some sharpling curved continua, like laser,
	 * totally fail due to non-linear nature of widflx and anu relationship */
#	if !defined(NDEBUG)
	x = 0.;
	for( i=1; i<rfield.nupper-1; ++i )
	{
		if( fabs( ((rfield.anu[i+1]-rfield.anu[i]) + (rfield.anu[i]-rfield.anu[i-1])) /rfield.widflx[i] /2.-1.) > 0.02 )
		{
			ans1 = fabs( ((rfield.anu[i+1]-rfield.anu[i]) + (rfield.anu[i]-rfield.anu[i-1])) /rfield.widflx[i] /2.-1.) ;
			fprintf(ioQQQ," SanityCheck found insane widflx anu[i+1]=%e anu[i]=%e widflx=%e delta=%e rel err %e\n",
			rfield.anu[i+1] , rfield.anu[i] , rfield.widflx[i] , rfield.anu[i+1] -rfield.anu[i] , ans1 );
			lgOK = FALSE;
			x = MAX2( ans1 , x);
		}
		/* problems when at energy where resolution of grid changes dramatically */
		/* this is resolution at current energy */
		ans1 = rfield.widflx[i] / rfield.anu[i];
		if( (rfield.anu[i]+rfield.widflx[i]/2.)*(1.-ans1/10.) > rfield.anu[i+1] - rfield.widflx[i+1]/2.) 
		{
			fprintf(ioQQQ," SanityCheck found insane overlap1 widflx %e %e %e %e %e %e\n",
			rfield.anu[i] , rfield.widflx[i], rfield.anu[i] + rfield.widflx[i]/2. , rfield.anu[i+1], 
			rfield.widflx[i+1], rfield.anu[i+1] -rfield.widflx[i+1]/2. );
			lgOK = FALSE;
		}
		if( !lgOK )
		{
			fprintf(ioQQQ," big error was %e\n", x);
		}
	}
#	endif
#	endif


	/*********************************************************
	 *                                                       *
	 * confirm that hydrogen einstein As are still valid     *
	 *                                                       *
	 *********************************************************/
	for( nelem=0; nelem<2; ++nelem )
	{
		/* this element may be turned off */
		if( abundances.lgElmtOn[nelem] )
		{ 
			/*Z4 = (double)(POW2(nelem+1)*POW2(nelem+1));*/
			/* form charge to the 4th power */
			Z4 = (double)(nelem+1);
			Z4 *= Z4;
			Z4 *= Z4;
			/* H Lya */
			ans1 = EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Aul;
			ans2 = 6.265e8*Z4;
			if( fabs(ans1-ans2)/ans2 > 1e-3 )
			{
				fprintf(ioQQQ , "SanityCheck finds insane A for H Lya %g %g nelem=%li\n",
					ans1 , ans2 , nelem );
				lgOK = FALSE;
			}

#			if 0
			/*must disable since, at this time, induced is included in Aul */
			/* H two photon */
			ans1 = EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].Aul;
			ans2 = 8.226*powi( (1.+nelem) , 6 );
			if( fabs(ans1-ans2)/ans2 > 1e-3 )
			{
				fprintf(ioQQQ , "SanityCheck finds insane A for H 2-phot %g %g nelem=%li\n",
					ans1 , ans2 , nelem );
				lgOK = FALSE;
			}
#			endif

#			if 0
			/* Balmer alpha - only way out of 3 for case b*/
			ans1 = EmisLines[ipH_LIKE][nelem][3][ipH2s].Aul;
			ans2 = 1.10318e7*Z4;
			if( fabs(ans1-ans2)/ans2 > 1e-3 )
			{
				fprintf(ioQQQ , "SanityCheck finds insane A for H 2s %g %g nelem=%li\n",
					ans1 , ans2 , nelem );
				lgOK = FALSE;
			}
			ans1 = EmisLines[ipH_LIKE][nelem][3][ipH2p].Aul;
			ans2 = 3.3095e7*Z4;
			if( fabs(ans1-ans2)/ans2 > 1e-3 )
			{
				fprintf(ioQQQ , "SanityCheck finds insane A for H 2s %g %g nelem=%li\n",
					ans1 , ans2 , nelem );
				lgOK = FALSE;
			}

			/* for higher line the branching ratio depends on l-mixing and so no close
			 * test is possible.  However all decays under case b should add up to
			 * hlife for that level */
			/* Balmer beta 4 - 2 plus Paschen alpha, 4-3*/
			ans1 = EmisLines[ipH_LIKE][nelem][4][ipH2s].Aul + 
				EmisLines[ipH_LIKE][nelem][4][ipH2p].Aul + EmisLines[ipH_LIKE][nelem][4][3].Aul;
			ans2 = (8.424e6+ 8.991e6)*Z4;
			if( fabs(ans1-ans2)/ans2 > 1e-2 )
			{
				fprintf(ioQQQ , 
					"SanityCheck finds insane A for summed H=4 decays found=%g correct=%g nelem=%li\n",
					ans1 , ans2 , nelem );
				lgOK = FALSE;
			}

#			endif
			/* Balmer gamma 5 - 2 + 5-4 + 5-3*/
			ans1 = EmisLines[ipH_LIKE][nelem][5][ipH2s].Aul + 
				EmisLines[ipH_LIKE][nelem][5][ipH2p].Aul + 
				EmisLines[ipH_LIKE][nelem][5][3].Aul + EmisLines[ipH_LIKE][nelem][5][4].Aul;
			ans2 = (2.532e6+2.20e6+2.70e6)*Z4;
			if( fabs(ans1-ans2)/ans2 > 1e-2 )
			{
				fprintf(ioQQQ , 
					"SanityCheck finds insane A for H5-2 found=%g correct=%g nelem=%li\n",
					ans1 , ans2 , nelem );
				lgOK = FALSE;
			}

		}
	}

	/* check that hydrogenic branching ratios add up to unity */
	for( nelem=0; nelem<LIMELM; ++nelem )
	{
		int ipHi, ipLo;
		for( ipHi=4; ipHi<=15; ++ipHi )
		{
			double sum = 0.;
			for( ipLo=2; ipLo<ipHi;++ipLo )
			{
				sum += HydroBranch(ipHi,ipLo,nelem+1);
			}
			if( fabs(sum-1.)>0.01 ) 
			{
				fprintf(ioQQQ , 
					"SanityCheck H branching ratio sum not unity for nelem=%li upper n=%i sum=%.3e\n",
					nelem, ipHi, sum );
				lgOK = FALSE;
			}
		}
	}

	/* check photo cross sections for H */
	ipISO = 0;
	nelem = 0;
	/* loop starts at 3, the first level with n = n and full l */
	for( n=3; n<MIN2(100., iso.numLevels[ipISO][nelem] ) ; ++n )
	{
		float anu[1]={1.} , cs[1] ;
		double energy;

		/* photon energy where cross section will be evaluated,
		 * this is in Ryd */
		energy = iso.xIsoLevNIonRyd[ipISO][nelem][n];
		anu[0] = (float)(energy*1.01);

		hypho(
			/* Z-Nelec, the residual charge, 1 for hydrogen, 2 for helium */
			(double)(nelem+1), 
			/* principle quantum number */
		  n, 
		  /* lowest angular momentum */
		  0, 
		  /* highest angular momentum */
		  n-1, 
		  /* scaled lowest photon energy, 
		   * => incorrect?? in units of zed^2/n^2, 
		   * at which the cs will be done */
		  energy, 
		  /* number of points to do */
		  1, 
		  /* array of energies (in units given above) */
		  anu , 
		  /* calculated photoionization cross section in cm^-2 */
		  cs);

		error = fabs(cs[0] - opac.OpacStack[iso.ipOpac[ipISO][nelem][n]-1] )/
			(        (cs[0] + opac.OpacStack[iso.ipOpac[ipISO][nelem][n]-1] )/2.);
		/*fprintf(ioQQQ,"z=%ld n=%ld error %g old %e new %e\n",nelem,n, error,
			opac.OpacStack[iso.ipOpac[ipISO][nelem][n]-1] ,cs[0] );*/
		if( error > 0.05 )
		{
			fprintf(ioQQQ , "SanityCheck finds insane H photo cs\n");
			lgOK = FALSE;
		}
	}
						
	/*********************************************************
	 *                                                       *
	 * confirm that exponential integral routines still work *
	 *                                                       *
	 *********************************************************/

	/* check that first and second exponential integrals are ok,
	 * step through range of values, beginning with following */
	x = 1e-3;
	do
	{
		/* check that fast e1 routine is ok */
		ans1 = ee1(x);
		ans2 = expnf( 1 , x );
		if( fabs(ans1-ans2)/(ans1+ans2) > 1e-2 )
		{
			fprintf(ioQQQ , "SanityCheck finds insane E1 %g %g %g\n",
				x , ans1 , ans2 );
			lgOK = FALSE;
		}

		/* check that e2 is ok */
		ans1 = e2(x,exp(-x));
		ans2 = expnf( 2 , x );
		if( fabs(ans1-ans2)/(ans1+ans2) > 1e-2 )
		{
			fprintf(ioQQQ , "SanityCheck finds insane E2 %g %g %g\n",
				x , ans1 , ans2 );
			lgOK = FALSE;
		}

		/* now increment x */
		x *= 2.;
		/* following limit set by sexp returning zero, used in ee1 */
	} while( x < 64. );


	/*********************************************************
	 *                                                       *
	 * confirm that matrix inversion routine still works     *
	 *                                                       *
	 *********************************************************/

	/* these are the answer, chosen to get xvec 1,2,3 */
	yVector[0] = 1.;
	yVector[1] = 3.;
	yVector[2] = 3.;

	/* zero out the main matrix */
	for(i=0;i<3;++i)
	{
		for( j=0;j<3;++j )
		{
			xMatrix[i][j] = 0.;
		}
	}

	/* remember that order is column, row, alphabetical order, rc */
	xMatrix[0][0] = 1.;
	xMatrix[0][1] = 1.;
	xMatrix[1][1] = 1.;
	xMatrix[2][2] = 1.;

	/* which matrix solver? */
	if( strcmp(TypMatrx.chMatrix,"matin1 ") == 0 )
	{
		/*matin1();*/
		ner = 1;
		if( ner != 0 )
		{
			fprintf( ioQQQ, " SanityCheck MATRIX ERROR.\n" );
			puts( "[Stop in SanityCheck]" );
			cdEXIT(EXIT_FAILURE);
		}
	}

	/* this is the default matrix solver */
	/* this test is the 1-d matrix with 2-d macro simulation */
	else if( strcmp(TypMatrx.chMatrix,"linpack") == 0 )
	{
		double *A;
	  /* LDA is right dimension of matrix */
#		define AA(I_,J_)	(*(A+(I_)*(NDIM)+(J_)))
		/* MALLOC space for the  1-d array */
		if( (A=(double*) MALLOC( (sizeof(double)*NDIM*NDIM ))) == NULL )
		{
			fprintf( ioQQQ, " SanityCheck MALLOC A error\n" );
			puts( "[Stop in SanityCheck]" );
			cdEXIT(EXIT_FAILURE);
		}
		/* copy over the main matrix */
		for(i=0;i<3;++i)
		{
			for( j=0;j<3;++j )
			{
				AA(i,j) = xMatrix[i][j];
			}
		}

		/*void DGETRF(long,long,double*,long,long[],long*);*/
		DGETRF(3,3,A,NDIM,ipiv,&ner);
		if( ner != 0 )
		{
			fprintf( ioQQQ, " SanityCheck DGETRF error\n" );
			puts( "[Stop in SanityCheck]" );
			cdEXIT(EXIT_FAILURE);
		}

		/* usage DGETRS, 'N' = no transpose
		 * order of matrix,
		 * number of cols in bvec, =1
		 * array
		 * leading dim of array */
		DGETRS('N',3,1,A,NDIM,ipiv,yVector,3,&ner);

		if( ner != 0 )
		{
			fprintf( ioQQQ, " SanityCheck DGETRS error\n" );
			puts( "[Stop in SanityCheck]" );
			cdEXIT(EXIT_FAILURE);
		}
		/* release the vector */
		free( A );
	}

	else if( strcmp(TypMatrx.chMatrix,"veclib ") == 0 )
	{
		/* Jason found this one on the Exemplar, distributed source just stops */
		fprintf( ioQQQ, " this has not been checked since H atom conv\n" );
		/*TODO in following one is dimension, second is number of levels */
		rcond = 0.;
		job = 0;
		dgeco((double*)xMatrix,NDIM,NDIM,ipiv,rcond,work);
		dgesl((double*)xMatrix,NDIM,NDIM,ipiv,yVector,job);
		/* now put results back into z so rest of code treates only
		 * one case - as if matin1 had been used */

		puts( "[Stop in SanityCheck]" );
		cdEXIT(EXIT_FAILURE);
	}

	else if( strcmp(TypMatrx.chMatrix,"Bevingt") == 0 )
	{
		/* this came from Bevington */
		matinv(xMatrix,3,&rcond);
		for( j=0; j < 3; j++ )
		{
			work[j] = 0.;
			for( i=0; i < 3; i++ )
			{
				work[j] += yVector[i]*xMatrix[i][j];
			}
		}
		for( j=0; j < 3; j++ )
		{
			yVector[j] = work[j];
		}
	}

	else
	{
		fprintf( ioQQQ, " chMatrix type insane in SanityCheck, was%7.7s\n", 
		  TypMatrx.chMatrix );
		puts( "[Stop in SanityCheck]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* now check on validity of the solution, demand that this
	 * simple problem have come within a few epsilons of the
	 * correct answer */

	/* find largest deviation */
	rcond = 0.;
	for(i=0;i<3;++i)
	{
		x = fabs( yVector[i]-i-1.);
		rcond = MAX2( rcond, x );
		/*printf(" %g ", yVector[i]);*/
	}

	if( rcond>DBL_EPSILON)
	{
		fprintf(ioQQQ,
			"SanityCheck found too large a deviation in matrix solver = %g \n", 
			rcond);
		/* set flag saying that things are not ok */
		lgOK = FALSE;
	}
	/* end matrix inversion check */


	/* these pointers were set to INT_MIN in ContCreatePointers,
	 * then set to valid numbers in ipShells and OpacityCreate1Element
	 * this checks that all values have been properly filled */
	for( nelem=0; nelem<LIMELM; ++nelem )
	{
		/* must reset state of code after tests performed, remember state here */
		float xIonF[NISO][LIMELM] , hn[NISO][LIMELM];
		double hbn[NISO][LIMELM];

		if( abundances.lgElmtOn[nelem] )
		{
			/* set these abundances so that opacities can be checked below */
			hbn[ipHYDROGEN][nelem] = iso.DepartCoef[ipH_LIKE][nelem][ipH1s];
			hn[ipHYDROGEN][nelem] = iso.Pop2Ion[ipH_LIKE][nelem][ipH1s];
			xIonF[ipHYDROGEN][nelem] = xIonFracs[nelem][nelem+1];

			iso.Pop2Ion[ipH_LIKE][nelem][ipH1s] = 1.;
			iso.DepartCoef[ipH_LIKE][nelem][ipH1s] = 0.;
			xIonFracs[nelem][nelem+1] = 1.;

			if( nelem > ipHYDROGEN )
			{

				hbn[ipHELIUM][nelem] = iso.DepartCoef[ipHE_LIKE][nelem][0];
				hn[ipHELIUM][nelem] = iso.Pop2Ion[ipHE_LIKE][nelem][0];
				xIonF[ipHELIUM][nelem] = xIonFracs[nelem][nelem];

				/* this does not exist for hydrogen itself */
				iso.Pop2Ion[ipHE_LIKE][nelem][ipH1s] = 1.;
				iso.DepartCoef[ipHE_LIKE][nelem][ipH1s] = 0.;
				xIonFracs[nelem][nelem] = 1.;
			}

			for( ion=0; ion<=nelem; ++ion )
			{
				/* loop over all shells that are defined */
				for( nshells=0; nshells<Heavy.nsShells[nelem][ion]; ++nshells )
				{
					for( j=0; j<3; ++j )
					{
						/* >>chng 00 apr 05, array index is on fortran scale so must be
						 * >= 1.  This test had been <0, correct for C.  Caught by Peter van Hoof */
						if( opac.ipElement[nelem][ion][nshells][j] <=0 )
						{
							/* this is not possible */
							fprintf(ioQQQ,
								"SanityCheck found insane ipElement for nelem=%li ion=%li nshells=%li j=%li \n", 
								nelem , ion , nshells, j );
							fprintf(ioQQQ,
								"value was %li  \n", opac.ipElement[nelem][ion][nshells][j] );
							/* set flag saying that things are not ok */
							lgOK = FALSE;
						}
					}
				}

				if( nelem > 1 )
				{
					float saveion[LIMELM+3];
					/* check that photoionization cross sections are ok */
					for( j=1; j <= (nelem + 2); j++ )
					{
						saveion[j] = xIonFracs[nelem][j-1];
						xIonFracs[nelem][j-1] = 0.;
					}

					xIonFracs[nelem][ion] = 1.;

					OpacityZero();
					opac.lgRedoStatic = TRUE;

					/* generate opacity with standard routine - this is the one
					 * called in OpacityAddTotal to make opacities in usual calculations */
					OpacityAdd1Element(nelem);

					/* this starts one beyond energy of threshold since cs may be zero there */
					for( j=Heavy.ipHeavy[nelem][ion]; j < MIN2(rfield.nflux,KshllEnr.KshellLimit) ; j++ )
					{
						if( opac.opac[j]+opac.OpacStatic[j] < FLT_MIN )
						{
							/* this is not possible */
							fprintf(ioQQQ,
								"SanityCheck found non-positive photo cs for nelem=%li ion=%li \n", 
								nelem , ion );
							fprintf(ioQQQ,
								"value was %.2e + %.2e nelem %li ion %li at energy %.2e\n", 
								opac.opac[j] ,
								opac.OpacStatic[j] ,
								nelem , 
								ion , 
								rfield.anu[j]);
							/* set flag saying that things are not ok */
							lgOK = FALSE;
							break;/**/
						}
					}
					/* reset the ionization distribution */
					for( j=1; j <= (nelem + 2); j++ )
					{
						xIonFracs[nelem][j-1] = saveion[j];
					}

				}
			}
			iso.DepartCoef[ipH_LIKE][nelem][ipH1s] = hbn[ipHYDROGEN][nelem];
			iso.Pop2Ion[ipH_LIKE][nelem][ipH1s] = hn[ipHYDROGEN][nelem];
			xIonFracs[nelem][nelem+1] = xIonF[ipHYDROGEN][nelem];

			if( nelem > ipHYDROGEN )
			{
				iso.DepartCoef[ipHE_LIKE][nelem][ipH1s] = hbn[ipHELIUM][nelem];
				iso.Pop2Ion[ipH_LIKE][nelem][ipH1s] = hn[ipHELIUM][nelem];
				xIonFracs[nelem][nelem] = xIonF[ipHELIUM][nelem];
			}
		}
	}


	/*********************************************************
	 *                                                       *
	 * everything is done, all checks make, did we pass them?*
	 *                                                       *
	 *********************************************************/

	if( lgOK )
	{
		/*return if ok */
		if( trace.lgTrace )
		{
			fprintf( ioQQQ, " SanityCheck returns OK\n");
		}

#		ifdef DEBUG_FUN
		fputs( " <->SanityCheck()\n", debug_fp );
#		endif
		return;
	}

	else
	{
		/* stop since problem encountered, lgEOF set FALSE */
		fprintf(ioQQQ , "SanityCheck finds insanity so exiting\n");
		ShowMe();
		cdEXIT(EXIT_FAILURE);
	}
}

