/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*BiDiag solve the bi-diagonal matrix for ionization balance */
#include "cddefines.h"
#include "yield.h"
#include "ionrange.h"
#include "prt.h"
#include "ionrec.h"
#include "iso.h"
#include "dynamics.h"
#include "heat.h"
#include "linpack.h"
#include "phycon.h"
#include "chargtran.h"
#include "heavy.h"
#include "elementnames.h"
#include "dense.h"
#include "converge.h"
#include "negcon.h"
#include "radius.h"
#include "bidiag.h"
#include "hmole.h"
#include "nomole.h"

void tridiag(double *a, double *b, long int n);
void solveions(double *ion, double *rec, double *snk, double *src,
							 long int nlev, long int nmax);

void BiDiag(
	/* this is element on the c scale, H is 0 */
	long int nelem, 
	/* option to print this element when called */
	int lgPrintIt)
{
	int lgDone, 
	  lgNegPop,
		lgTriDiag;
	long int LoopBig, 
	  ion, 
	  limit, 
	  MaxVal, 
	  nej, 
	  nelec, 
	  ns,
		jmax=-1;
	float abundOld;
	double pop_ion_ov_neut[LIMELM+2], 
	  ratio, 
	  sum;
	double AbundNew[LIMELM+1];
	double total_abund, 
		norm, 
		den;
	int lgSingular = TRUE;

#	ifdef DEBUG_FUN
	fputs( "<+>BiDiag()\n", debug_fp );
#	endif

	/* this is on the c scale, so H is 0 */
	ASSERT( nelem >= 0);
	ASSERT( IonRange.IonLow[nelem] >= 0 );
	ASSERT( IonRange.IonHigh[nelem] >= 0 );

	if( prt.lgPrtArry || lgPrintIt )
	{
		/* say who we are, what we are doing .... */
		fprintf( ioQQQ, "\n %s BiDiag: ionization/recombination rates [s-1] for %s\n", 
			elementnames.chElementSym[nelem],
			elementnames.chElementName[nelem] );
	}

	/* H is special because its abundance spills into three routines -
	 * the ion/atom solver (this routine), the H-mole solvers (hmole), and
	 * the heavy molecule solver.  xmolecules only includes the heavy mole
	 * part for H only.  So the difference between gas_phase and xmolecules
	 * includes the H2 part of the molecular network.  This branch does
	 * this special H case, then the general case (heavy elements are
	 * not part of the H2 molecular network) */

	/* >>chng 01 dec 07, define total_abund, total atom and ion abundance here, removing molecules */
	if( nelem == ipHYDROGEN )
	{
		/* for the time being, hydrogen is a special case since hmole does not yet include the
		 * atom/molecules - this will eventually be removed, since this routine will not be called
		 * for hydrogen - hmole will do this */
		/*TODO remove this block when hmole is fully functional */
		total_abund = dense.xIonDense[nelem][0] + dense.xIonDense[nelem][1];
		/*if( iteration>1 ) fprintf(ioQQQ,"adveccc%li %.3e %.3e \n",
			nzone , dynamics.Rate ,
			dynamics.Source[0][0] );*/
	}
	else
	{
		total_abund = MAX2( SMALLFLOAT , dense.gas_phase[nelem] -  dense.xMolecules[nelem] );
	}

	/* >>chng 00 dec 12, return if IonHigh is zero, since no ionization at all */
	if( IonRange.IonHigh[nelem] == 0 )
	{
		/* set the atom to the total gas phase abundance */
		dense.xIonDense[nelem][0] = (float)total_abund ;
#		ifdef DEBUG_FUN
		fputs( " <->BiDiag()\n", debug_fp );
#		endif
		return;
	}

	/* >>chng 01 may 09, add option to force ionization distribution with element name ioniz */
	if( dense.lgSetIoniz[nelem] )
	{
		for( ion=0; ion<nelem+2; ++ion )
		{
			dense.xIonDense[nelem][ion] = dense.SetIoniz[nelem][ion]*(float)total_abund;
		}
#		ifdef DEBUG_FUN
		fputs( " <->BiDiag()\n", debug_fp );
#		endif
		return;
	}

	/* impossible for HIonFrac[nelem] to be zero if IonHigh(nelem)=nelem+1
	 * HIonFrac(nelem) is stripped to hydrogen */
	/* >>chng 01 oct 30, to assert */
	ASSERT( (IonRange.IonHigh[nelem] < nelem + 1) || iso.pop_ion_ov_neut[ipH_LIKE][nelem] > 0. );

#	if !defined(NDEBUG)
	/* set to negative value to make sure (with assert) reset before used below */
	for( ion=0; ion<nelem+2; ++ion )
	{
		pop_ion_ov_neut[ion] = -1.;
	}
#	endif

	/* this will be a sanity check */
	lgNegPop = FALSE;

	LoopBig = 0;
	lgDone = FALSE;
	while( LoopBig < 5 && (!lgDone) )
	{
		/* zero out the ionization and recombination rates that we will modify here,
		 * but not he h-like and he-like stages which are done elsewhere,
		 * the nelem stage of ionization is he-like,
		 * the nelem+1 stage of ionization is h-like */

		/* sanity check */
		ASSERT( IonRange.IonLow[nelem] < IonRange.IonHigh[nelem] );

		/* loop over stages of ionization that we solve for here, 
		 * up through and including one less than nelem-NISO,
		 * never actually do highest NISO stages of ionization since they
		 * come from the ionization ratio from the next lower stage */
		/* >>chng 02 nov 04, from 2 to NISO */
		/*limit = MIN2(nelem-2,IonRange.IonHigh[nelem]-1);*/
		limit = MIN2(nelem-NISO,IonRange.IonHigh[nelem]-1);

		/* zero-out loop comes before main loop since there are off-diagonal
		 * elements in the main ionization loop, due to multi-electron processes,
		 * TotIonizRate and TotRecom were already set in h-like and he-like solvers */
		for( ion=0; ion <= limit; ion++ )
		{
			ionrec.TotIonizRate[nelem][ion] = 0.;
		}

		/* now get actual arrays of ionization and recombination processes,
		 * but only for the ions that are done as two-level systems */
		for( ion=IonRange.IonLow[nelem]; ion <= limit; ion++ )
		{
			/* get total destruction rates */
			/* collisional ionization */
			ionrec.TotIonizRate[nelem][ion] += ionrec.CollIonRate_Ground[nelem][ion][0];

			/* >>chng 02 aug 20, add this ionization process */
			/* inner shell ionization by lines, UTA, etc */
			ionrec.TotIonizRate[nelem][ion] += ionrec.xInnerShellIonize[nelem][ion];

			/* loop over all shells to include photoionization */
			for( ns=0; ns < Heavy.nsShells[nelem][ion]; ns++ )
			{
				ionrec.TotIonizRate[nelem][ion] += ionrec.PhotoRate_Shell[nelem][ion][ns][0];

				/* following is most number of electrons freed */
				nelec = yield.nyield[nelem][ion][ns];

				/* following loop is over number of electrons that come out of shell
				 * with multiple electron ejection */
				for( nej=2; nej <= nelec; nej++ )
				{
					/* this is highest possible stage of ionization -
					 * do not want to ignore ionization that go beyond this */

					MaxVal = MIN2(ion+nej,IonRange.IonHigh[nelem]);
					/* if( dense.xIonDense(nelem,ion+nej-1).gt.1e-30 ) then */
					if( dense.xIonDense[nelem][MaxVal-1] > 1e-30 )
					{
						/* ion points to current stage of ionization */
						ratio = (double)(dense.xIonDense[nelem][ion])/
						  (double)(dense.xIonDense[nelem][MaxVal-1]);
					}
					else
					{
						ratio = 1.;
					}
					/* yield here is fraction removing ion electrons */
					ionrec.TotIonizRate[nelem][MaxVal-1] += ratio *
						ionrec.PhotoRate_Shell[nelem][ion][ns][0]*
					  yield.vyield[nelem][ion][ns][nej-1] *
					  /* >>chng 01 dec 18, number of electrons */
					  (double)nej;
				}
			}
			/* this is charge transfer ionization of this specieds by hydrogen and helium */
			ionrec.TotIonizRate[nelem][ion] += ChargTran.HeCharExcIon[nelem][ion]*dense.xIonDense[ipHELIUM][1]+ 
			  ChargTran.HCharExcIon[nelem][ion]*dense.xIonDense[ipHYDROGEN][1];

			/* recombination rates were already set by routines responsible for them */
		}
		/* after this loop, ionrec.TotIonizRate and ionrec.TotRecomRate have been defined for the
		 * stages of ionization that are done with simple */

		/* >> chng 03 jan 13 rjrw, always solve using full matrix solution to allow for sources
		 * from molecular netork */
		/* last test - only do advection if we have not overrun the radius scale */
#define RJRW 1
		if(RJRW || (iteration >= 2 && dynamics.lgAdvection && radius.depth < dynamics.oldFullDepth) )
		{
			double *amat,*achk,*b,*snk;
			long *ipiv;
			long ilow, range , i , j , nerror1 , nerror2 ;
			ilow = IonRange.IonLow[nelem];

			/* range will be from 0 through highest stage of ionization
			range = limit-ilow+2;   */
			/* do full range of ionization - this is number of ionization stages */
			range = IonRange.IonHigh[nelem]-IonRange.IonLow[nelem]+1;

#			ifdef MAT
#				undef MAT
#				endif
#			define MAT(M_,I_,J_)	(*((M_)+(I_)*(range)+(J_)))
			/* MALLOC space for the  1-d array */
			if( (amat=(double*)MALLOC( (sizeof(double)*(unsigned)((range)*(range)) ))) == NULL )
				BadMalloc();
			if( (achk=(double*)MALLOC( (sizeof(double)*(unsigned)((range)*(range)) ))) == NULL )
				BadMalloc();
			if( (b=(double*)MALLOC( (sizeof(double)*(unsigned)((range)) ))) == NULL )
				BadMalloc();
			if( (snk=(double*)MALLOC( (sizeof(double)*(unsigned)((range)) ))) == NULL )
				BadMalloc();
			if( (ipiv=(long*)MALLOC( (sizeof(long)*(unsigned)((range)) ))) == NULL )
				BadMalloc();

			for( i=0; i<range;i++ )
			{
				for( j=0;j<range;j++ )
				{
					MAT(amat,i,j) = 0.;
				}
				b[i] = 0.;
			}

			/* these are the main balance equations, equation 9 in the advection writeup */

			lgTriDiag = TRUE; /* Set lgTriDiag = FALSE if non-tridiagonal terms are ever added to amat */

			for( i=0; i<range;i++ )
			{
				ion = i+ilow;
				/* fprintf(ioQQQ," %li %li %.3e\n",i,i,MAT(amat,i,i)); */
				if( i != range-1 )
				{
					MAT(amat,i,i) += ionrec.TotIonizRate[nelem][ion];
					MAT(amat,i+1,i) = -ionrec.TotRecomRate[nelem][ion];
					/* fprintf(ioQQQ," %li %li %.3e\n",i,i+1,MAT(amat,i,i+1)); */
				}
				if( i != 0 )
				{
					MAT(amat,i,i) += ionrec.TotRecomRate[nelem][ion-1];
					MAT(amat,i-1,i) = -ionrec.TotIonizRate[nelem][ion-1];
					/* fprintf(ioQQQ," %li %li %.3e\n",i,i-1,MAT(amat,i,i-1)); */
				}

			}

			if (nelem == ipHYDROGEN && !nomole.lgNoH2Mole)
			{
				double totsrc = 0., totsnk = 0.;
				for (i=0;i<range;i++) {
					ion = i+ilow;
					b[i] += ionmole.source[ion];
					totsrc += ionmole.source[ion];
					MAT(amat,i,i) += ionmole.sink[ion];
					totsnk += ionmole.sink[ion];
				}
				if (totsrc != 0. && totsnk != 0.)
					lgSingular = FALSE;

				if (0)
				{
					fprintf(ioQQQ,"Mol sourc %g %g\n", ionmole.source[0], ionmole.source[1]);
					fprintf(ioQQQ,"Mol sinks %g %g\n", ionmole.sink[0], ionmole.sink[1]); 
				}
			}

			/* chng 03 jan 13 rjrw, add in dynamics if required here,
				 last test - only do advection if we have not overrun the radius scale */
			if( iteration >= 2 && dynamics.lgAdvection && radius.depth < dynamics.oldFullDepth)
			{
				for( i=0; i<range;i++ )
				{			 
					ion = i+ilow;
					MAT(amat,i,i) += dynamics.Rate;
					b[i] += dynamics.Source[nelem][ion];
					/* fprintf(ioQQQ," %li %li %.3e (%.3e %.3e)\n",i,i,MAT(amat,i,i),dynamics.Rate, dynamics.Source[nelem][ion]);*/
				}
				lgSingular = FALSE;
			}

			if (lgSingular)
			{
				double ionprod=1., recomb, scale = 0., value;
				/* Simple estimate of most abundant ion */
				jmax = 0;
				for (i=0; i<range-1;i++)
				{ 
					ion = i+ilow;
					ionprod *= ionrec.TotIonizRate[nelem][ion];
					recomb = ionrec.TotRecomRate[nelem][ion];
					if (ionprod == 0)  /* All the gas will be more neutral than this */
						break;
					if (recomb <= 0.) /*  */
						break;

					ionprod /= recomb;
					if (ionprod > 1.) 
					{
						jmax = i;
						ionprod = 1.;
					}
				}
				
				/* Matrix will be singular, so replace most abundant ion row with sum rule,
				 * scaled to maintain matrix condition. */
				scale = 0.;
				for (i=0; i<range;i++)
				{
					value = MAT(amat,i,i);
					if (value > scale)
					{
						scale = value;
					}
				}
				scale *= 1e-6;
				
#if 1
				for( i=0; i<range;i++ )
				{
					MAT(amat,i,jmax) = 0.;
				}
				MAT(amat,jmax,jmax) = scale;
				b[jmax] = scale;
#else
				for( i=0; i<range;i++ )
				{
					MAT(amat,i,jmax) = scale;
				}
				b[jmax] = total_abund*scale;
#endif
			}

			if (0 && nelem == ipHELIUM && dynamics.lgAdvection) 
			{
				fprintf(ioQQQ," %li %.3e \n",nzone,dynamics.Rate);
				fprintf(ioQQQ," %.3e %.3e\n", ionrec.TotIonizRate[nelem][0], ionrec.TotIonizRate[nelem][1]);
				fprintf(ioQQQ," %.3e %.3e\n", ionrec.TotRecomRate[nelem][0], ionrec.TotRecomRate[nelem][1]);
				fprintf(ioQQQ," %.3e %.3e %.3e\n", dynamics.Source[nelem][0], dynamics.Source[nelem][1], dynamics.Source[nelem][2]);
			}

			if (RJRW && 0)
			{
				for( i=0; i<range;i++ )
				{
						for( j=0;j<range;j++ )
						{
							fprintf(ioQQQ,"%e\t",MAT(amat,j,i));
							MAT(achk,j,i) = MAT(amat,j,i);
						}
						fprintf(ioQQQ,"\n");
				}
				for( i=0; i<range;i++ )
				{
					fprintf(ioQQQ,"%e\t",b[i]);
				}
				fprintf(ioQQQ,"\n");
			}

			if (lgTriDiag) 
			{
				if (0) 
				{
					/* Use Lapack tridiagonal solver */
					double *dl, *d, *du;
					int i1, i2, ne;
					
					d = (double *) MALLOC((unsigned)range*sizeof(double));
					du = (double *) MALLOC((unsigned)(range-1)*sizeof(double));
					dl = (double *) MALLOC((unsigned)(range-1)*sizeof(double));

					for (i=0;i<range-1;i++) 
					{
						du[i] = MAT(amat,i+1,i);
						d[i] = MAT(amat,i,i);
						dl[i] = MAT(amat,i,i+1);
					}
					d[i] = MAT(amat,i,i);

					i1 = range;
					i2 = 1;
					ne = 0;
					dgtsv(&i1, &i2, dl, d, du, b, &i2, &ne);
					
					free(dl);free(du);free(d);
				} 
				else if (0) 
				{

					/* Use tridiagonal solver */
					tridiag(amat,b,range);

				}
				else if (1)
				{
					/* Use tridiagonal solver re-coded to avoid rounding errors
					 * on diagonal -- uses determination of jmax for the
					 * singular case, but is otherwise independent of the array
					 * filling code above */
					
					for (i=0;i<range;i++) 
					{
						b[i] = snk[i] = 0.;
					}
					if (nelem == ipHYDROGEN && !nomole.lgNoH2Mole)
					{
						for (i=0;i<range;i++) 
						{
							ion = i+ilow;
							b[i] += ionmole.source[ion];
							snk[i] += ionmole.sink[ion];
						}
					}
					if ( iteration >= 2 && dynamics.lgAdvection && radius.depth < dynamics.oldFullDepth )
					{
						for (i=0;i<range;i++) 
						{
							ion = i+ilow;
							b[i] += dynamics.Source[nelem][ion];
							snk[i] += dynamics.Rate;
						}
					}
					
					solveions(ionrec.TotIonizRate[nelem]+ilow,ionrec.TotRecomRate[nelem]+ilow,
										snk,b,range,jmax);
				}
			} 
			else 
			{
				/* Use general matrix solver */
				DGETRF(range,range,amat,range,ipiv,&nerror1);
				DGETRS('N',range,1,amat,range,ipiv,b,range,&nerror2);
			}

			{
				/*@-redef@*/
				/* this is to debug following failed assert */
				enum {DEBUG_LOC=FALSE};
				/*@+redef@*/
				if( DEBUG_LOC && (nzone >380 ) && nelem == ipHYDROGEN )
				{
					fprintf(ioQQQ,"debuggg\t%li\t%.4e\t%.4e\tIon\t%.3e\tRec\t%.3e\n", 
						nzone,
						phycon.te,
						dense.eden,
						ionrec.TotIonizRate[nelem][0] , 
						ionrec.TotRecomRate[nelem][0]);
					fprintf(ioQQQ," Msrc %.3e %.3e\n", ionmole.source[0], ionmole.source[1]);
					fprintf(ioQQQ," Msnk %.3e %.3e\n", ionmole.sink[0], ionmole.sink[1]);
					fprintf(ioQQQ," Poprat %.3e nomol %.3e\n",b[1]/b[0],
					  ionrec.TotIonizRate[nelem][0]/ionrec.TotRecomRate[nelem][0]);
				}
			}

			if (RJRW && 0)
			{
				double test;
				for (i=0; i<range; i++) {
					test = 0.;
					for (j=0; j<range; j++) {
						test = test+b[j]*MAT(achk,j,i);
					}
					fprintf(ioQQQ,"%e\t",test);
				}
				fprintf(ioQQQ,"\n");

				test = 0.;
				fprintf(ioQQQ," ion %li abundance %.3e\n",nelem,total_abund);
				for( ion=IonRange.IonLow[nelem]; ion < IonRange.IonHigh[nelem]; ion++ )
				{
					if ( ionrec.TotRecomRate[nelem][ion] != 0 && b[ion-ilow] != 0 )
						fprintf(ioQQQ," %li %.3e %.3e : %.3e\n",ion,b[ion-ilow],
										b[ion-ilow+1]/b[ion-ilow],
										ionrec.TotIonizRate[nelem][ion]/ionrec.TotRecomRate[nelem][ion]);
					else
						fprintf(ioQQQ," %li %.3e [One ratio infinity]\n",ion,b[ion-ilow]);
					test += b[ion-ilow];
				}
				test += b[ion-ilow];
				fprintf(ioQQQ," %li %.3e\t>> total %.3e\n",ion,b[ion-ilow],test);
			}

			/* 
			 * >> chng 03 jan 15 rjrw:- terms are now included for
			 * molecular sources and sinks of H and H+.
			 *
			 * When the network is not in equilibrium, this will lead to a
			 * change in the derived abundance of H and H+ when after the
			 * matrix solution -- the difference between `norm' and 1. is a
			 * measure of the quality of the solution (it will be 1. if the
			 * rate of transfer into Ho/H+ balances the rate of transfer
			 * out, for the consistent relative abundances).
			 *
			 * We therefore renormalize to keep the total H abundance
			 * correct -- only the molecular network is allowed to change
			 * this.
			 *
			 * To do this, only the ion abundances are corrected, as the
			 * molecular abundances may depend on several different
			 * conserved species.
			 *
			 * */
			if (lgSingular)
			{
				fixit(); 				/* dense.xIonDense[nelem][] should be set to
													 default values before this ... */
				dense.xIonDense[nelem][ilow] = (float)total_abund;
				for ( i=1;i < range; i++ )
				{
					dense.xIonDense[nelem][i+ilow] = 0.;
				}
			}

			if (iteration >= 2 && dynamics.lgAdvection && radius.depth < dynamics.oldFullDepth 
					&& nelem == ipHYDROGEN && nomole.lgNoH2Mole)
			{
				/* The normalization out of the matrix solution is correct and
				 * should be retained if: dynamics is on and the total
				 * abundance of HI & H+ isn't being controlled by the
				 * molecular network */
				norm = 1.;
			}
			else
			{
				norm = 0.;
				den = 0.;
				
				for ( i=0;i < range; i++ )
				{
					ion = i+ilow;
					norm += dense.xIonDense[nelem][ion];
					den += b[i];
				} 
				
				if (den > 0.)
				{
					norm /= den;
					fixit(); /* norm should == 1 when the molecules and
										* ionization are in equilibrium.  Should monitor
										* this figure of merit in calling routine.
										* */
					if (0 && !lgSingular)
						fprintf(ioQQQ,"Zone %li elem %ld normalization error %g\n",nzone,nelem,norm-1.);
				}
				else
				{
					norm = 1.;
				}
			}

			for( i=0; i < range; i++ )
			{
				ion = i+ilow;

				/*fprintf(ioQQQ," %li %li %.3e %.3e\n",nelem,ion,b[ion-ilow+1],b[ion-ilow]);
				pop_ion_ov_neut[ion] = b[ion-ilow+1]/b[ion-ilow];*/

				dense.xIonDense[nelem][ion] = (float)(b[i]*norm);
				/* fprintf(ioQQQ,"%ld %g [%g]\n",ion,dense.xIonDense[nelem][ion],b[i]*norm); */
				ASSERT( dense.xIonDense[nelem][ion]>= 0. );
			}
			
			/*fprintf(ioQQQ," Sums %.3e %.3e %.3e\n",sum1,sum2,dynamics.Rate);*/
			free(ipiv);
			free(b);
			free(achk);
			free(snk);
			free(amat);

			/* Zero levels with abundances < 1e-25 which which will suffer numerical noise */
			while( IonRange.IonHigh[nelem] > IonRange.IonLow[nelem] && 
						 dense.xIonDense[nelem][IonRange.IonHigh[nelem]] < 1e-25*total_abund )
			{
				ASSERT( dense.xIonDense[nelem][IonRange.IonHigh[nelem]] >= 0. );
				/* zero out abundance and heating due to stage of ionization we are about to zero out */
				/* fprintf(ioQQQ,"Zone %ld: elem %ld losing %ld (%g %g)\n",nzone,nelem,IonRange.IonHigh[nelem],
					 dense.xIonDense[nelem][IonRange.IonHigh[nelem]],total_abund); */
				dense.xIonDense[nelem][IonRange.IonHigh[nelem]] = 0.;
				heat.heating[nelem][IonRange.IonHigh[nelem]-1] = 0.;
				/* decrement counter */
				--IonRange.IonHigh[nelem];
			}

			/* sanity check, either offset stages of low and high ionization,
			 * or no ionization at all */
			ASSERT( (IonRange.IonLow[nelem] < IonRange.IonHigh[nelem]) ||
							(IonRange.IonLow[nelem]==0 && IonRange.IonHigh[nelem]==0 ) );

#			ifdef DEBUG_FUN
			fputs( " <->BiDiag()\n", debug_fp );
#			endif
			return;
		}

		/* >>chng 03 jan 13 rjrw, always does matrix solution so the rest is now dead code */

		/* invert and solve bidiagonal matrix */
		/* >>chng 99 may 1, this had been set to 1 for all previous versions of the code,
		 * and range of ionization was limited by logic dealing with ionization parameter.
		 * that logic was removed (caused other problems) and now comphi.in crashes with
		 * overflow in loop below.  dur to pop_ion_ov_neut getting larger than biggest double.
		 * change 1 to 1e-200 to get more range, could cause problems with very low
		 * ionization solns??? */

		/* get sum of ratios for stages higher than this lowest stage of ionization,
		 * limit is the highest stage of ionization actually done with simple 2-level atom */
		/* >>chng 02 aug 28, go up to highest stage of ionization, even if he- or h- like */
		for( ion=IonRange.IonLow[nelem]; ion <= IonRange.IonHigh[nelem]-1; ion++ )
		{
			ASSERT( ionrec.TotIonizRate[nelem][ion] >= 0. );
			ASSERT( ionrec.TotRecomRate[nelem][ion] >= 0. );
			pop_ion_ov_neut[ion] = ionrec.TotIonizRate[nelem][ion]/MAX2(SMALLFLOAT,ionrec.TotRecomRate[nelem][ion]);
		}

		/* at this point we have the sum of ratios for ionization stages that are done
		 * with simple two-level atom.  Now do he-like and h-like stages of ionization */

		/* test for h-like stages of ionization  */
		if( IonRange.IonHigh[nelem] == nelem + 1 )
		{
			if( pop_ion_ov_neut[nelem] == 0. )
			{
				/* zero out stage we won't consider any more */
				dense.xIonDense[nelem][nelem+1] = 0.;
				/* this is zero abundance, drop IonHigh */
				IonRange.IonHigh[nelem] = nelem;
			}
			else
			{
				ASSERT( dynamics.lgAdvection ||
					(fabs(pop_ion_ov_neut[nelem]-iso.pop_ion_ov_neut[ipH_LIKE][nelem])/
					iso.pop_ion_ov_neut[ipH_LIKE][nelem]<1e-3) ) ;
			}

		}

		/* test for he-like stages of ionization */
		if( (IonRange.IonHigh[nelem] >= nelem) && (IonRange.IonLow[nelem]<nelem)  )
		{
			if( pop_ion_ov_neut[nelem-1] == 0. )
			{
				/* zero out stage we won't consider any more */
				dense.xIonDense[nelem][nelem] = 0.;
				/* this is zero abundance, drop IonHigh */
				IonRange.IonHigh[nelem] = nelem-1;
			}
			else
			{
				/* these two currently do not agree when dynamics enabled */
				ASSERT( dynamics.lgAdvection ||
					(fabs(pop_ion_ov_neut[nelem-1]-iso.pop_ion_ov_neut[ipHE_LIKE][nelem])/
					iso.pop_ion_ov_neut[ipHE_LIKE][nelem]<1e-3) ) ;
			}
		}

		/* limit is the highest stage that has an ionization balance to the
		 * next higher stage.  If limit is not equal to IonHigh-1 
		 * then because IonHigh is too high, and will be trimed down later */
		/*ASSERT( limit == (IonRange.IonHigh[nelem]-1) );*/

		/* at this stage it is possible that some elements of pop_ion_ov_neut will be
		 * zero, especially during search phase when ionizing continuum does
		 * not extend through range of ionization */
		/* >>chng 01 apr 21, first comp had been against 0, change to low+2 */
		/* >>chng 02 may 24, change limit on pop_ion_ov_neut from SMALLFLOAT to 1e-20 */
		while( IonRange.IonHigh[nelem] > IonRange.IonLow[nelem] && 
			pop_ion_ov_neut[IonRange.IonHigh[nelem]-1] < 1e-20 )
			/*pop_ion_ov_neut[IonRange.IonHigh[nelem]-1] < SMALLFLOAT )*/
		{
			/* >>chng 01 oct 30, add this assert to go with preset to -1 above */
			ASSERT( pop_ion_ov_neut[IonRange.IonHigh[nelem]-1]>= 0. );
			/* zero out abundance and heating due to stage of ionization we are about to zero out */
			dense.xIonDense[nelem][IonRange.IonHigh[nelem]] = 0.;
			heat.heating[nelem][IonRange.IonHigh[nelem]-1] = 0.;
			/* decrement counter */
			--IonRange.IonHigh[nelem];
		}

		/* sanity check, either offset stages of low and high ionization,
		 * or no ionization at all */
		ASSERT( (IonRange.IonLow[nelem] < IonRange.IonHigh[nelem]) ||
			(IonRange.IonLow[nelem]==0 && IonRange.IonHigh[nelem]==0 ) );

		/* >>chng 00 dec 12, return if IonHigh is zero, since no ionization at all */
		if( IonRange.IonHigh[nelem] == 0 )
		{
			/* set the atom to the total gas phase abundance */
			dense.xIonDense[nelem][0] = (float)total_abund;

#			ifdef DEBUG_FUN
			fputs( " <->BiDiag()\n", debug_fp );
#			endif
			return;
		}

		/* find dynamic range of abundances - necessary to make sure
		 * sum of abundances will not under or over flow*/
		sum = 0.;
		for( ion=IonRange.IonLow[nelem]; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			sum += log10( pop_ion_ov_neut[ion] );
		}

		/* form inital sum of abundances such that they are evenly
		 * divided on either side of unity - this will be scale factor
		 * for this initial set of abundances */
		ratio = pow(10. , sum/2 );

		/* define set of new ion fractions, on arbitrary scale for now */
		AbundNew[IonRange.IonLow[nelem]] = 1./ratio;
		sum = AbundNew[IonRange.IonLow[nelem]];

		/* define next higher abundance in terms of previous one, still on
		 * free-floating scale, but keeping sum of all these abundances,
		 * which will be renomalized to the total */
		for( ion=IonRange.IonLow[nelem]; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			AbundNew[ion+1] = pop_ion_ov_neut[ion] * AbundNew[ion];
			sum += AbundNew[ion+1];
		}

		/* now renormalize so that abundances are correct */
		ratio = total_abund/sum;
		sum = 0.;
		for( ion=IonRange.IonLow[nelem]; ion <= IonRange.IonHigh[nelem]; ion++ )
		{
			AbundNew[ion] *= ratio;
			sum += AbundNew[ion];
		}
		{
			/*@-redef@*/
			/* this is to debug following failed assert */
			enum {DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC && lgPrintIt )
			{
				fprintf(ioQQQ,"sum was %.2e nelem is %li total abundance is %.2e\n", 
					sum , nelem , total_abund );
			}
		}
		/* check that total abundance is positive */
		ASSERT( sum > 0.);
		/* and that it adds up to the correct abundance */
		ASSERT( fabs( sum-total_abund)/total_abund < 1e-7 );

		/* zero out abundances of ions below lowest stage we will consider */
		for( ion=0; ion < IonRange.IonLow[nelem]; ion++ )
		{
			/* recall that dense.xIonDense ion scale offset by 1 since [0] is total gas phase abundance */
			dense.xIonDense[nelem][ion] = 0.;
		}

		/* zero out abundances of ions above highest stage we will consider */
		for( ion=IonRange.IonHigh[nelem]+1; ion < (nelem + 2); ion++ )
		{
			/* recall that dense.xIonDense ion scale offset by 1 since [0] is total gas phase abundance */
			dense.xIonDense[nelem][ion] = 0.;
		}

		lgDone = TRUE;
		/* we now have a guess at the new ionization fractions - now do something
		 * with this, keeping solution stable */
		for( ion=IonRange.IonLow[nelem]; ion <= IonRange.IonHigh[nelem]; ion++ )
		{
			/* special logic for search phase when don't have good soln */
			if( conv.lgSearch && (dense.xIonDense[nelem][ion] > 0.) && (AbundNew[ion] > 0.) &&
				nelem > ipHELIUM )
			{
				/* this branch used during search phase */
				/* >>chng 96 oct 27, sulphur and above oscillated, corrected by
				 * using log mean during search phase */
				abundOld = dense.xIonDense[nelem][ion];

				/* >>chng 99 Jul 02, the log in the following is protected against 
				 * non-pos value by if branch above */
				dense.xIonDense[nelem][ion] = 
					(float)sqrt(dense.xIonDense[nelem][ion]*AbundNew[ion]);

				/* >>chng 97 jul 29, set flag to redo if changes happen */
				if( dense.xIonDense[nelem][ion] > 1e-15 )
				{
					if( fabs(abundOld/dense.xIonDense[nelem][ion]- 1.) > 0.2 )
					{
						lgDone = FALSE;

						/* these are printed as the old and new values that forced 
						 * declaration of no convergence */
						conv.BadConvIoniz[0] = abundOld;
						conv.BadConvIoniz[1] = dense.xIonDense[nelem][ion];
					}
				}
			}
			else
			{
				/* this branch used for usual search for soln deep in model */
				abundOld = dense.xIonDense[nelem][ion];
				dense.xIonDense[nelem][ion] = (float)(AbundNew[ion]);
				if( dense.xIonDense[nelem][ion]/total_abund > 1e-10 )
				{
					if( fabs(abundOld/dense.xIonDense[nelem][ion]-1.) > 0.2 )
					{
						lgDone = FALSE;

						/* these are printed as the old and new values that forced 
						 * declaration of no convergence */
						conv.BadConvIoniz[0] = abundOld;
						conv.BadConvIoniz[1] = dense.xIonDense[nelem][ion];
					}
				}
			}
			/* check for negative populations */
			if( dense.xIonDense[nelem][ion] < 0. )
				lgNegPop = TRUE;
		}

		if( prt.lgPrtArry || lgPrintIt )
		{
			/* this is the array that was inverted to produce abundances */
			fprintf( ioQQQ, " IonRat   " );
			for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
			{
				fprintf( ioQQQ, " %8.1e", pop_ion_ov_neut[ion] );
			}
			fprintf( ioQQQ, "\n" );
			/* these are the predicted abundances */
			fprintf( ioQQQ, " AbundX   " );
			for( ion=1; ion <= IonRange.IonHigh[nelem]+1; ion++ )
			{
				fprintf( ioQQQ, " %8.1e", dense.xIonDense[nelem][ion-1] );
			}
			fprintf( ioQQQ, "\n" );
		}
		++LoopBig;
	}

	if( !lgDone )
	{
		conv.lgConvIoniz = FALSE;
		strcpy( conv.chConvIoniz,  "BiDiag:" );

		/* this is two-char short form of element name */
		strcat( conv.chConvIoniz, elementnames.chElementSym[nelem] );
	}

	/* this can`t possibly happen */
	if( lgNegPop )
	{
		fprintf( ioQQQ, " Negative population found for abundance of ionization stage of element %4.4s, ZONE=%4ld\n", 
		  elementnames.chElementNameShort[nelem], nzone );

		fprintf( ioQQQ, " Populations were" );
		for( ion=1; ion <= IonRange.IonHigh[nelem]+1; ion++ )
		{
			fprintf( ioQQQ, "%9.1e", dense.xIonDense[nelem][ion-1] );
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, " destroy vector =" );
		for( ion=1; ion <= IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, "%9.1e", ionrec.TotIonizRate[nelem][ion-1] );
		}
		fprintf( ioQQQ, "\n" );

		/* check whether any negative level populations occured */
		lgNegPop = FALSE;
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			if( ionrec.TotIonizRate[nelem][ion] < 0. )
				lgNegPop = TRUE;
		}

		/* print some extra stuff if destroy was negative */
		if( lgNegPop )
		{
			fprintf( ioQQQ, " CTHeavy  vector =" );
			for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
			{
				fprintf( ioQQQ, "%9.1e", ChargTran.HeCharExcIon[nelem][ion] );
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, " HCharExcIon vtr=" );
			for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
			{
				fprintf( ioQQQ, "%9.1e", ChargTran.HCharExcIon[nelem][ion] );
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, " CollidRate  vtr=" );
			for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
			{
				fprintf( ioQQQ, "%9.1e", ionrec.CollIonRate_Ground[nelem][ion][0] );
			}
			fprintf( ioQQQ, "\n" );

			/* photo rates per subshell */
			fprintf( ioQQQ, " photo rates per subshell, ion\n" );
			for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
			{
				fprintf( ioQQQ, "%3ld", ion );
				for( ns=0; ns < Heavy.nsShells[nelem][ion]; ns++ )
				{
					fprintf( ioQQQ, "%9.1e", ionrec.PhotoRate_Shell[nelem][ion][ns][0] );
				}
				fprintf( ioQQQ, "\n" );
			}
		}

		/* now check out creation vector */
		fprintf( ioQQQ, " create  vector =" );
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, "%9.1e", ionrec.TotRecomRate[nelem][ion] );
		}
		fprintf( ioQQQ, "\n" );

		negcon();
		ShowMe();
		puts( "[Stop in bidiag]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* option to print ionization and recombination arrays
	 * prt flag set with "print array" command */
	if( prt.lgPrtArry || lgPrintIt )
	{
		/* total ionization rate, all processes */
		fprintf( ioQQQ, " %s Ioniz total " ,elementnames.chElementSym[nelem]);
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %9.2e", ionrec.TotIonizRate[nelem][ion] );
		}
		fprintf( ioQQQ, "\n" );

		/* sum of all creation processes */
		fprintf( ioQQQ, " %s Recom total " ,elementnames.chElementSym[nelem]);
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %9.2e", ionrec.TotRecomRate[nelem][ion] );
		}
		fprintf( ioQQQ, "\n" );

		/* collisional ionization */
		fprintf( ioQQQ, " %s Coll ioniz  " ,elementnames.chElementSym[nelem] );
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %9.2e", ionrec.CollIonRate_Ground[nelem][ion][0] );
		}
		fprintf( ioQQQ, "\n" );

		/* photo ionization */
		fprintf( ioQQQ, " %s Phot ioniz  " ,elementnames.chElementSym[nelem]);
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %9.2e", 
				ionrec.PhotoRate_Shell[nelem][ion][Heavy.nsShells[nelem][ion]-1][0] );
		}
		fprintf( ioQQQ, "\n" );

		/* charge exchange ionization */
		fprintf( ioQQQ, " %s chr trn ion " ,elementnames.chElementSym[nelem] );
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %9.2e", ChargTran.HCharExcIon[nelem][ion]*
			  dense.xIonDense[ipHYDROGEN][1] );
		}
		fprintf( ioQQQ, "\n" );

		/* charge exchange recombination */
		fprintf( ioQQQ, " %s chr trn rec "  ,elementnames.chElementSym[nelem]);
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %9.2e", ChargTran.HCharExcRec[nelem][ion]*
			  dense.xIonDense[ipHYDROGEN][0] );
		}
		fprintf( ioQQQ, "\n" );

		/* the "new" abundances the resulted from the previous ratio */
		fprintf( ioQQQ, " %s Abun [cm-3] " ,elementnames.chElementSym[nelem] );
		for( ion=0; ion <= IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %9.2e", AbundNew[ion] );
		}
		fprintf( ioQQQ, "\n" );
	}

#	ifdef DEBUG_FUN
	fputs( " <->BiDiag()\n", debug_fp );
#	endif
	return;
}


/* 

	 Solve an ionization level system with specified ionization and
	 recombination rates between neighbouring ions, and additional sink
	 and source terms.  The sink array is overwritten, and the results
	 appear in the source array.

	 Written in matrix form, the algorithm is equivalent to the
	 tridiagonal algorithm in Numerical Recipes applied to:

	 / i_0+a_0     -r_0          .           .    .  \ / x_0 \   / s_0 \
	 |  -i_0    i_1+a_1+r_0    -r_1          .    .  | | x_1 |   | s_1 |
	 |    .        -i_1      i_2+a_2+r_1   -r_2   .  | | x_2 |   | s_2 |
   |    .          .       (etc....)               | | ... | = | ... |
   \    .          .          .                    / \     /   \     /

	 where i, r are the ionization and recombination rates, s is the
	 source rate and a is the sink rate.

	 This matrix is diagonally dominant only when the sink terms are
	 large -- the alternative method coded here prevents rounding error
	 in the diagonal terms disturbing the solution when this is not the
	 case.

*/

void solveions(double *ion, double *rec, double *snk, double *src,
	       long int nlev, long int nmax)
{
  double kap, bet;
  long int i;

  if (nmax != -1) 
  {
    /* Singular case */
    src[nmax] = 1.;
    for (i=nmax;i<nlev-1;i++)
      src[i+1] = src[i]*ion[i]/rec[i];
    for (i=nmax-1;i>=0;i--)
      src[i] = src[i+1]*rec[i]/ion[i];
  } 
  else 
  {
    i = 0;
    kap = snk[0];    
    for (i=0;i<nlev-1;i++) 
    {
      bet = ion[i]+kap;
      if (bet == 0.)
      {
				fprintf(ioQQQ,"Ionization solver error\n");
				puts("[Stop in solveions]");
				cdEXIT(EXIT_FAILURE);
      }
      bet = 1./bet;
      src[i] *= bet;
      src[i+1] += ion[i]*src[i];
      snk[i] = bet*rec[i];
      kap = kap*snk[i]+snk[i+1];
    }
	bet = kap;
	if (bet == 0.)
	{
		fprintf(ioQQQ,"Ionization solver error\n");
		puts("[Stop in solveions]");
		cdEXIT(EXIT_FAILURE);
	}
	src[i] /= bet;
    
    for (i=nlev-2;i>=0;i--)
    {
      src[i] += snk[i]*src[i+1];
    }
  }
}

#			ifdef MAT
#				undef MAT
#				endif
#			define MAT(M_,I_,J_)	(*((M_)+(I_)*(n)+(J_)))
void tridiag(double *a, double *b, long int n)
{
	unsigned long j;
	double c,*g,t;
	
	g = (double *) MALLOC(n*sizeof(double));
	if (MAT(a,0,0) == 0.) 
	{
		fprintf(ioQQQ,"Error 1 in tridiag\n");
		exit(-1);
	}
	c = MAT(a,0,0);
	b[0] /= c;
	for (j=1; j<(unsigned)n; j++) 
	{
		g[j] = MAT(a,j,j-1)/c;
		t = MAT(a,j-(unsigned)1,j);
		c = MAT(a,j,j)-t*g[j];
		if (c == 0.) {
			fprintf(ioQQQ,"Error 2 in tridiag\n");
			exit(-1);
		}
		b[j] = (b[j]-t*b[j-1])/c;
	}
	for (j=(unsigned)n-1;j>0;j--)
		b[j-1] -= g[j]*b[j];
	free(g);
}

