/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*H2_Create create H2 molecules, called by ContCreatePointers after continuum mesh is set up */
/*H2_ReadEnergies read energies for all electronic levels */
/*H2_ReadTransprob read transition probabilities */
/*H2_ReadCollRates read transition probabilities */
/*H2_ContPoint set the ipCont struc element for the H2 molecule, called by ContCreatePointers */
/*H2_Accel radiative acceleration due to H2 */
/*H2_RadPress rad pre due to h2 lines called in PresTotCurrent */
/*H2_Zero zero out vars in the large H2 molecule, called from zero */
/*H2_InterEnergy internal energy of H2 called in PresTotCurrent */
/*H2_PunchLineStuff include H2 lines in punched optical depths, etc, called from PunchLineStuff */
/*H2_RTDiffuse do emission from H2 - called from RTDiffuse */
/*H2_RTMake do RT for H2 - called from RTMake */
/*H2_TauInc increment optical depth for the H2 molecule, called from RTOptDepthIncre */
/*H2_TauInit initialize optical depths in H2, called from RTOptDepthInit */
/*H2_ParsePunch parse the punch h2 command */
/*H2_TauAver the large H2 molecule, called from RTOptDepthReset */
/*H2_Reset called to reset variables that are needed after an iteration */
/*H2_colden maintain H2 column densities within X */
/*H2_LevelPops do level populations for H2, called by Hydrogenic */
/*H2_LinesAdd add in explicit lines from the large H2 molecule, called by lines_molecules */
/*H2_ReadDissprob read dissociation probabilities and kinetic energies for all electronic levels */
/*H2_Punch punch some properties of the large H2 molecule */
/*H2_prt_line_tau print line optical depths, called from premet in response to print line optical depths command*/
/*H2_prt_column_density print H2 column density, called from prtcolumns */
/*H2_prt_Zone print H2 info into zone results, called from prtzone for each printed zone */
/*H2_Level_lowJ evaluate CO rotation cooling */
/*H2_cooling evaluate cooling and heating due to H2 molecule */
/*H2_ParseAtom parse information from the rotor command line */
/*H2_Read_hminus_distribution read distribution function for H2 population following formation from H minus */
/*H2_Read_Cosmicray_distribution read distribution function for H2 population following cosmic ray collisional excitation */  
/*H2_vib_dist evaluates the vibration distribution for H2 formed on grains */
/*TODO put in excited molecular dissociation from v >=4 as in hazy 2 */
#include "cddefines.h" 
#include "physconst.h" 
#include "path.h" 
#include "taulines.h" 
#include "linesave.h" 
#include "punch.h" 
#include "leveln.h" 
#include "converge.h" 
#include "secondaries.h" 
#include "hmi.h" 
#include "rt.h" 
#include "norm.h" 
#include "prt.h" 
#include "rtescprob.h" 
#include "radius.h" 
#include "grainvar.h" 
#include "prtmet.h"
#include "ipoint.h" 
#include "phycon.h" 
#include "thermal.h" 
#include "colden.h" 
#include "dense.h" 
#include "rfield.h" 
#include "lines_service.h" 
#include "opacity.h" 
#include "h2.h"

/* if this is set true then code will print energies and stop */
/*@-redef@*/
enum {DEBUG_ENER=FALSE};
/*@+redef@*/

/* when true this says for h2_create to immediately quit, and no try to do anything at all. */
#define lgQUIT FALSE

/* this is the number of electronic levels - in h2.h */
/* #define N_H2_ELEC	7*/

/* the number of times the H2 molecules has been called in this iteration.  For the
 * very first call we will use lte for the level populations, for later calls
 * use the last solution */
static long int nCallH2_this_iteration;

/* these will mostly become xxx[elec][vib][rot] */
static float ***dissprob;
static float ***disske;
static double ***energy_wn;
static double ***Boltzmann;
static double ***populations_LTE;
static float ***old_populations;
static float ***populations;
/*static double ***rate_in;*/
/*static double ***rate_out;*/
/* this is total statistical weight, including nuclear spin */
static float ***stat;
/* this is true if state is para, false if ortho */
static int ***lgOrtho;
/* this will say (with char - chlgPara[lgOrtho] ) whether ortho or para */
char chlgPara[2]={'P','O'};

/* this is array of accumulated line intensities, used for punch he lines command */
static double ****SaveLine;

/* this says whether line exists */
int ****lgLineExist;

static int **ipPhoto;
static float **col_rate_in;
static float **col_rate_out;
static float **rad_rate_in;
static float **rad_rate_out;

#define CR_PRINT	FALSE
#define CR_X	1
#define CR_VIB	15
#define CR_J	10
#define	CR_EXIT	3
/* this will hold the cr rates */
static float cr_rate[CR_X][CR_VIB][CR_J][CR_EXIT];

/* save rate coef (cm3 s-1) for collisional dissociation */
static float **coll_dissoc_rate_coef;

/* column density within X only vib and rot */
static float **H2_X_colden;

/* formation into specific states witin X only vib and rot */
static float **H2_X_formation;
/* vib, rot, last dim is grain type */
static float ***H2_X_grain_formation_distribution;

/* the number of temperature points in the data file */
#define	nTE_HMINUS	7
/* vib, rot, last dim is temperature */
static float ***H2_X_hminus_formation_distribution;
static float te_hminus[nTE_HMINUS] = {10.,30.,100.,300.,1000.,3000.,10000.};

/* these are energies and indices for levels within X */
static float *Xenergies;
static long int *ipX_ener_sort;
static long int *ipVib_energies;
static long int *ipRot_energies;

/* this is energy difference between bottom of potential well and 0,0
 * the Takahashi energy scale is from the bottom,
 * 2201.9 wavenumbers  */
static const double energy_off = 0.273*FREQ_1EV/SPEEDLIGHT ;

/* the density (cm-3) of ortho H2 */
static double ortho_colden,
/* the density (cm-3) of para H2 */
	para_colden;

/* the number of ro-vib levels in each elec state */
static long int nLevels_per_elec[N_H2_ELEC];

/* the total population in each elec state */
static float pops_per_elec[N_H2_ELEC];

/* total population in each vib state */
static float **pops_per_vib;

/* this will contain a vector for collisions within the X ground elec state,
 * CollRateFit[coll_type][vib_up][rot_up][vib_lo][rot_lo][3] */
/* the number of different types of colliders */
#define	N_X_COLLIDER	5
static float collider_density[N_X_COLLIDER];
static float collider_density_total;
/* this is the highest vib state that has collision data */
#define	VIB_COLLID	3
/* this is the actual rate, cm^3 s^-1, for each collider
 * CollRate[coll_type][vib_up][rot_up][vib_lo][rot_lo] */
static float ******CollRateFit;
static float *****CollRate;

/* the order of the electronic states is
 * X, B, C+, C-, B', D+, and D- */
/* this will be the number of vibration levels within each elec */
/* number of vib states within electronic states from
 * >>refer	H2	energies	Abgrall, */
static long int nVib_hi[N_H2_ELEC] = {14 , 37 , 13 , 13, 9, 2 , 2};

/* this integer is added to rotation quantum number J for the test of whether
 * a particular J state is ortho or para - the state is ortho if J+below is odd,
 * and para if J+below is even */
static int nRot_add_ortho_para[N_H2_ELEC] = {0 , 1 , 1 , 0, 1, 1 , 0};

/* this gives the first rotational state for each electronic state - J=0 does
 * not exist when Lambda = 1 */
static long int Jlowest[N_H2_ELEC] = {0 , 0 , 1  , 1 , 0 , 1 , 1 };

/* intensity, relative to normalization line, for faintest line to punch */
static float thresh_punline_h2;

/* limit to the ratio H2/Htot - if ratio is below this, large atom is not called */
static double H2_to_H_limit;

/* the number of electronic quantum states to include.
 * To do both Lyman and Werner bands want nelec = 3 */
static long int n_elec_states=N_H2_ELEC;

/* the number of vibration and rotation levels used in the matrix solution
 * of the level populations */
static long int matrix_vib , matrix_rot;

/* this is option to use estimates of the collision rates from g-bar approximations */
/* turn lgColl_gbar on/off with atom h2 gbar on off */
static int lgColl_gbar;

/* this is option to turn off the calculated collision rates */
static int lgColl_deexec_Calc;
/* this is option to turn off gueses of collisional dissociation rates */
static int lgColl_dissoc_coll;

/* turn on trace information */
static int lgH2_TRACE;

/* dissociation energies (cm-1) for each electronic state, from 
 * >>refer	H2	energies	Sharp, T. E., 1971, Atomic Data, 2, 119 */
/* >>chng 02 oct 08, improved energies */
static double DissocEnergies[N_H2_ELEC] = 
{ 36118.11, 118375.6, 118375.6, 118375.6, 118375.6,133608.6,133608.6 };
/* original values
 { 36113., 118372., 118372., 118372., 118372.,0.,0. };*/

/* number of rotation levels within each elec - vib */
/*lint -e785 too few init for aggregate */
long int nRot_hi[N_H2_ELEC][50]=
	/* ground, X */
	{ {31, 30, 28, 27, 25, 
		23, 22, 20, 18, 16, 
		14, 12, 10,  7,  3 } ,
	/* B */
	{25,25,25,25,25,25,25,25, 25,25,
	 25,25,25,25,25,25,25,25, 25,25,
	 25,25,25,25,25,25,25,25, 23,21,
	 19,17,15,15,11,9,7, 7},
	/* C plus */
	{ 25, 25, 25, 25, 24, 23, 21, 19, 17, 14, 12, 10, 6, 2 },
	/* C minus (the same) */
	{ 25, 25, 25, 25, 24, 23, 21, 19, 17, 15, 13, 10, 7, 2 },
	/* B primed */
	{19,17, 14, 12, 9, 8, 7, 7, 4, 1 },
	/* D plus */
	{13, 10, 5},
	/* D minus */
	{25 , 25 ,25 } }
	;
/*lint +e785 too few init for aggregate */

/*H2_Level_lowJ evaluate CO rotation cooling */
static void H2_Level_lowJ(
	long int nVibrate ,
	long int nRotate ,
	float abundan );

/*H2_Read_hminus_distribution read distribution function for H2 population following formation from H minus */
static void H2_Read_hminus_distribution(void)
{
	FILE *ioDATA;
	char chLine[FILENAME_PATH_LENGTH_2] , 
		chFilename[FILENAME_PATH_LENGTH_2] ;
	long int i, n1, n2, n3, iVib , iRot;
	int lgEOL;
	double sumrate[nTE_HMINUS];
	/* set true for lots of printout */
#	define H2HMINUS_PRT	FALSE

	/* check on path is file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , "H2_hminus_deposit.dat" );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , "H2_hminus_deposit.dat" );
	}

	/* now open the data file */
	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " H2_Read_hminus_distribution could not open %s\n", "H2_hminus_deposit.dat" );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " H2_Read_hminus_distribution could not open %s\n", "H2_hminus_deposit.dat");
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in H2_Read_hminus_distribution]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read the first line and check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " H2_Read_hminus_distribution could not read first line of %s\n", "H2_hminus_deposit.dat");
		puts( "[Stop in H2_Read_hminus_distribution]" );
		cdEXIT(EXIT_FAILURE);
	}

	i = 1;
	/* level 1 magic number */
	n1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* magic number
	 * the following is the set of numbers that appear at the start of H2_hminus_deposit.dat 01 08 10 */
	if( ( n1 != 2 ) || ( n2 != 10 ) || ( n3 != 17 ) )
	{
		fprintf( ioQQQ, 
			" H2_Read_hminus_distribution: the version of %s is not the current version.\n", "H2_hminus_deposit.dat" );
		fprintf( ioQQQ, 
			" I expected to find the number 2 10 17 and got %li %li %li instead.\n" ,
			n1 , n2 , n3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in H2_Read_hminus_distribution]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read until not a comment */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
		BadRead();

	while( chLine[0]=='#' )
	{
		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			BadRead();
	}

	/* convert temps to log */
	for(i=0; i<nTE_HMINUS; ++i )
	{
		te_hminus[i] = (float)log10(te_hminus[i]);
		sumrate[i] = 0.;
	}

	iRot = 1;
	iVib = 1;
	while( iVib >= 0 )
	{
		/* set true to print rates */
		
		double a[nTE_HMINUS] , ener;
		sscanf(chLine,"%li\t%li\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf", 
			&iVib ,&iRot , &ener, &a[0],&a[1],&a[2] , &a[3],&a[4],&a[5] ,&a[6] 
			);
		/* negative iVib says end of data */
		if( iVib < 0 )
			continue;

		/* check that we actually included the levels in the model representation */
		ASSERT( iVib <= nVib_hi[0] && 
			iRot <= nRot_hi[0][iVib] );

		if( H2HMINUS_PRT )
			fprintf(ioQQQ,"hminusss\t%li\t%li", iVib , iRot );
		for( i=0; i<nTE_HMINUS; ++i )
		{
			H2_X_hminus_formation_distribution[i][iVib][iRot] = (float)pow(10.,-a[i]);
			sumrate[i] += H2_X_hminus_formation_distribution[i][iVib][iRot];
			if( H2HMINUS_PRT )
				fprintf(ioQQQ,"\t%.3e", H2_X_hminus_formation_distribution[i][iVib][iRot] );
		}
		if( H2HMINUS_PRT )
			fprintf(ioQQQ,"\n" );

		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			BadRead();
		while( chLine[0]=='#' )
		{
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
				BadRead();
		}
	}
	fclose( ioDATA );

	if( H2HMINUS_PRT )
	{
		/* print total rate */
		fprintf(ioQQQ," total H- formation rate ");
		/* convert temps to log */
		for(i=0; i<nTE_HMINUS; ++i )
		{
			fprintf(ioQQQ,"\t%.3e" , sumrate[i]);
		}
		fprintf(ioQQQ,"\n" );
	}

	/* convert to dimensionless factors that add to unity */
	for( iVib=0; iVib<=nVib_hi[0]; ++iVib )
	{
		for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
		{
			for(i=0; i<nTE_HMINUS; ++i )
			{
				H2_X_hminus_formation_distribution[i][iVib][iRot] /= (float)sumrate[i];
			}
		}
	}

	if( H2HMINUS_PRT )
	{
		/* print total rate */
		fprintf(ioQQQ,"  H- distribution function ");
		for( iVib=0; iVib<=nVib_hi[0]; ++iVib )
		{
			for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
			{
				fprintf(ioQQQ,"%li\t%li", iVib , iRot );
				for(i=0; i<nTE_HMINUS; ++i )
				{
					fprintf(ioQQQ,"\t%.3e", H2_X_hminus_formation_distribution[i][iVib][iRot] );
				}
				fprintf(ioQQQ,"\n" );
			}
		}
	}

	return;
}

/*H2_Read_Cosmicray_distribution read distribution function for H2 population following cosmic ray collisional excitation */
static void H2_Read_Cosmicray_distribution(void)
{
	FILE *ioDATA;
	char chLine[FILENAME_PATH_LENGTH_2] , 
		chFilename[FILENAME_PATH_LENGTH_2] ;
	long int i, n1, n2, n3, iVib , iRot;
	long neut_frac;
	int lgEOL;

	/* check on path is file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , "H2_CosmicRay_collision.dat" );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , "H2_CosmicRay_collision.dat" );
	}

	/* now open the data file */
	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " H2_Read_CosmicRay_distribution could not open %s\n", chFilename );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " H2_Read_CosmicRay_distribution could not open %s\n", chFilename );
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in H2_Read_Cosmicray_distribution]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read the first line and check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " H2_Read_Cosmicray_distribution could not read first line of %s\n", "H2_Cosmic_collision.dat");
		puts( "[Stop in H2_Read_Cosmicray_distribution]" );
		cdEXIT(EXIT_FAILURE);
	}

	i = 1;
	/* level 1 magic number */
	n1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* magic number
	 * the following is the set of numbers that appear at the start of H2_Cosmic_collision.dat 01 21 03 */
	if( ( n1 != 1 ) || ( n2 != 21 ) || ( n3 != 3 ) )
	{
		fprintf( ioQQQ, 
			" H2_Read_Cosmicray_distribution: the version of %s is not the current version.\n", "H2_Cosmic_collision.dat" );
		fprintf( ioQQQ, 
			" I expected to find the number 1 21 3 and got %li %li %li instead.\n" ,
			n1 , n2 , n3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in H2_Read_Cosmicray_distribution]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read until not a comment */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
		BadRead();

	while( chLine[0]=='#' )
	{
		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			BadRead();
	}

	iRot = 1;
	iVib = 1;
	neut_frac = 0;
	while( iVib >= 0 )
	{
		long int j_minus_ji ;
		double a[10] ;
	
		sscanf(chLine,"%li\t%li\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf", 
			&iVib ,&j_minus_ji , &a[0],&a[1],&a[2],&a[3],&a[4],&a[5],&a[6],&a[7],&a[8],&a[9] 
			);
		/* negative iVib says end of data */
		if( iVib < 0 )
			continue;

		/* cr_rate[CR_X][CR_VIB][CR_J][CR_EXIT];*/
		/* check that we actually included the levels in the model representation */
		ASSERT( iVib < CR_VIB );
		ASSERT( j_minus_ji == -2 || j_minus_ji == +2 || j_minus_ji == 0 );
		ASSERT( neut_frac < CR_X );

		/* now make i_minus_ji an array index */
		j_minus_ji = 1 + j_minus_ji/2;
		ASSERT( j_minus_ji>=0 && j_minus_ji<=2 );
		
		/*lint -e662 possible creation of out of bounds pointer */
		for( iRot=0; iRot<CR_J; ++iRot )
		{
			cr_rate[neut_frac][iVib][iRot][j_minus_ji] = (float)a[iRot];
		}
		/*lint +e662 possible creation of out of bounds pointer */

		if( CR_PRINT )
		{
			fprintf(ioQQQ,"cr rate\t%li\t%li", iVib , j_minus_ji ); 
			for( iRot=0; iRot<CR_J; ++iRot )
			{ 
				fprintf(ioQQQ,"\t%.3e", cr_rate[neut_frac][iVib][iRot][j_minus_ji] );
			} 
			fprintf(ioQQQ,"\n" );
		}

		/* now get next line */
		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			BadRead();
		while( chLine[0]=='#' )
		{
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
				BadRead();
		}
	}
	fclose( ioDATA );

	return;
}
/* read energies for all electronic levels */
static void H2_ReadEnergies( long int nelec )
{
	char cdDATAFILE[N_H2_ELEC][FILENAME_PATH_LENGTH_2] = 
	{
		"H2_energy_X.dat" ,
		"H2_energy_B.dat" , 
		"H2_energy_C_plus.dat" ,
		"H2_energy_C_minus.dat" , 
		"H2_energy_B_primed.dat" , 
		"H2_energy_D_plus.dat" ,
		"H2_energy_D_minus.dat" 
	};
	FILE *ioDATA;
	char chLine[FILENAME_PATH_LENGTH_2] , 
		chFilename[FILENAME_PATH_LENGTH_2] ;
	long int i, n1, n2, n3, iVib , iRot;
	int lgEOL;

	/* check on path is file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , cdDATAFILE[nelec] );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , cdDATAFILE[nelec] );
	}

	/* now open the data file */
	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadEnergies could not open %s\n", cdDATAFILE[nelec] );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " H2_ReadEnergies could not open %s\n", cdDATAFILE[nelec]);
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in H2_ReadEnergies]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read the first line and check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadEnergies could not read first line of %s\n", cdDATAFILE[nelec]);
		puts( "[Stop in H2_ReadEnergies]" );
		cdEXIT(EXIT_FAILURE);
	}
	i = 1;
	/* level 1 magic number */
	n1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* magic number
	 * the following is the set of numbers that appear at the start of level1.dat 01 08 10 */
	if( ( n1 != 2 ) || ( n2 != 4 ) || ( n3 != 29 ) )
	{
		fprintf( ioQQQ, 
			" H2_ReadEnergies: the version of %s is not the current version.\n", cdDATAFILE[nelec] );
		fprintf( ioQQQ, 
			" I expected to find the number 2 4 29 and got %li %li %li instead.\n" ,
			n1 , n2 , n3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in H2_ReadEnergies]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read until not a comment */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
		BadRead();

	while( chLine[0]=='#' )
	{
		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			BadRead();
	}

	/* this will count the number of levels within each elec state */
	nLevels_per_elec[nelec] = 0;

	for( iVib=0; iVib<=nVib_hi[nelec]; ++iVib )
	{
		for( iRot=Jlowest[nelec]; iRot<=nRot_hi[nelec][iVib]; ++iRot )
		{
			i = 1;
			sscanf(chLine,"%li\t%li\t%le", &n1 , &n2 , &energy_wn[nelec][iVib][iRot] );
			ASSERT( n1 == iVib );
			ASSERT( n2 == iRot );
			/* in atomic units, or 1 Hartree, or two rydbergs */
			if( nelec == 0 )
			{
				/* only do this for Phillip Stancil's file */
				/* corrections are to get lowest rotation level to have energy of zero */
				energy_wn[0][iVib][iRot] = -( energy_wn[0][iVib][iRot]- 3.6118114E+04 );
			}
			ASSERT( energy_wn[nelec][iVib][iRot]> 0. || (nelec==0 && iVib==0 && iRot==0 ) );
			/* increment number of levels within this elec state */
			++nLevels_per_elec[nelec];

			/* now start reading next line */
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
				BadRead();
			while( chLine[0]=='#' )
			{
				if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
					BadRead();
			}
		}
	}
	fclose( ioDATA );
}

/* read dissociation probabilities and kinetic energies for all electronic levels */
static void H2_ReadDissprob( long int nelec )
{
	char cdDATAFILE[N_H2_ELEC][FILENAME_PATH_LENGTH_2] = 
	{
		"H2_dissprob_X.dat" ,/* this does not exist and nelec == 0 is not valid */
		"H2_dissprob_B.dat" , 
		"H2_dissprob_C_plus.dat" ,
		"H2_dissprob_C_minus.dat" , 
		"H2_dissprob_B_primed.dat" , 
		"H2_dissprob_D_plus.dat" ,
		"H2_dissprob_D_minus.dat" 
	};
	FILE *ioDATA;
	char chLine[FILENAME_PATH_LENGTH_2] , 
		chFilename[FILENAME_PATH_LENGTH_2] ;
	long int i, n1, n2, n3, iVib , iRot;
	int lgEOL;

	ASSERT( nelec > 0 );

	/* check on path is file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , cdDATAFILE[nelec] );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , cdDATAFILE[nelec] );
	}

	/* now open the data file */
	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadDissprob could not open %s\n", cdDATAFILE[nelec] );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " H2_ReadDissprob could not open %s\n", cdDATAFILE[nelec]);
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in H2_ReadDissprob]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read the first line and check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadDissprob could not read first line of %s\n", cdDATAFILE[nelec]);
		puts( "[Stop in H2_ReadDissprob]" );
		cdEXIT(EXIT_FAILURE);
	}
	i = 1;
	/* level 1 magic number */
	n1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* magic number
	 * the following is the set of numbers that appear at the start of level1.dat 01 08 10 */
	if( ( n1 != 3 ) || ( n2 != 2 ) || ( n3 != 11 ) )
	{
		fprintf( ioQQQ, 
			" H2_ReadDissprob: the version of %s is not the current version.\n", cdDATAFILE[nelec] );
		fprintf( ioQQQ, 
			" I expected to find the number 3 2 11 and got %li %li %li instead.\n" ,
			n1 , n2 , n3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in H2_ReadDissprob]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read until not a comment */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
		BadRead();

	while( chLine[0]=='#' )
	{
		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			BadRead();
	}

	for( iVib=0; iVib<=nVib_hi[nelec]; ++iVib )
	{
		for( iRot=Jlowest[nelec]; iRot<=nRot_hi[nelec][iVib]; ++iRot )
		{
			double a, b;
			i = 1;
			sscanf(chLine,"%li\t%li\t%le\t%le", 
				&n1 , &n2 , 
				/* dissociation probability */
				&a ,
				/* dissociation kinetic energy - eV not ergs */
				&b);

			/* these have to agree if data file is valid */
			ASSERT( n1 == iVib );
			ASSERT( n2 == iRot );

			/* dissociation probability */
			dissprob[nelec][iVib][iRot] = (float)a;
			/* dissociation kinetic energy - eV not ergs */
			disske[nelec][iVib][iRot] = (float)b;

			/* now get next line */
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
				BadRead();
			while( chLine[0]=='#' )
			{
				if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
					BadRead();
			}
		}
	}
	fclose( ioDATA );
}

/* read collision rates */
static void H2_ReadCollRates( long int nColl )
{
	/* the colliders are H, He, H2 ortho, H2 para, H+ */
	char cdDATAFILE[N_X_COLLIDER][FILENAME_PATH_LENGTH_2] = 
	{
		"H2_coll_H.dat" ,
		"H2_coll_He.dat" , 
		"H2_coll_H2ortho.dat" ,
		"H2_coll_H2para.dat",
		"H2_coll_Hp.dat"  
	};
	FILE *ioDATA;
	char chLine[FILENAME_PATH_LENGTH_2] , 
		chFilename[FILENAME_PATH_LENGTH_2] ;
	long int i, n1, n2, n3;
	long int iVibHi , iVibLo , iRotHi , iRotLo;
	int lgEOL;

	/* check on path is file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , cdDATAFILE[nColl] );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , cdDATAFILE[nColl] );
	}

	/* now open the data file */
	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadCollRates could not open %s\n", cdDATAFILE[nColl] );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " H2_ReadCollRates could not open %s\n", cdDATAFILE[nColl]);
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in H2_ReadCollRates]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read the first line and check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadCollRates could not read first line of %s\n", cdDATAFILE[nColl]);
		puts( "[Stop in H2_ReadCollRates]" );
		cdEXIT(EXIT_FAILURE);
	}
	i = 1;
	/* level 1 magic number */
	n1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* magic number
	 * the following is the set of numbers that appear at the start of level1.dat 01 08 10 */
	if( ( n1 != 2 ) || ( n2 != 4 ) || ( n3 != 29 ) )
	{
		fprintf( ioQQQ, 
			" H2_ReadCollRates: the version of %s is not the current version.\n", cdDATAFILE[nColl] );
		fprintf( ioQQQ, 
			" I expected to find the number 2 4 29 and got %li %li %li instead.\n" ,
			n1 , n2 , n3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in H2_ReadCollRates]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read until not a comment */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
		BadRead();

	while( chLine[0]=='#' )
	{
		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			BadRead();
	}
	iRotHi = 1;
	while( iRotHi >= 0 )
	{
		double a[3];
		sscanf(chLine,"%li\t%li\t%li\t%li\t%le\t%le\t%le", 
			&iVibHi ,&iRotHi , &iVibLo , &iRotLo , &a[0],&a[1],&a[2] );
		/* negative iRotHi says end of data */
		if( iRotHi < 0 )
			continue;

		/* check that we actually included the levels in the model representation */
		ASSERT( iVibHi <= VIB_COLLID && 
		    iVibLo <= VIB_COLLID && 
			iRotHi <= nRot_hi[0][iVibHi] && 
			iRotLo <= nRot_hi[0][iVibLo]);

		/* some H collision rates have the same upper and lower indices - skip them */
		if( !( (iVibHi == iVibLo) && (iRotHi == iRotLo  )) )
		{
			/* this is downward transition - make sure that the energy difference is positive */
			ASSERT( (energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo] ) > 0. );
			for( i=0; i<3; ++i )
			{
				CollRateFit[nColl][iVibHi][iRotHi][iVibLo][iRotLo][i] = (float)a[i];
			}

			/* this prints all levels with rates 
			fprintf(ioQQQ,"no\t%li\t%li\t%li\t%li\t%.2e\t%.2e\t%.2e\n", 
				iVibHi,iRotHi,iVibLo,iRotLo,a[0],a[1],a[2]);*/
		}

		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			BadRead();
		while( chLine[0]=='#' )
		{
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
				BadRead();
		}
	}
	fclose( ioDATA );
}
/* end of H2_ReadCollRates */

/* read transition probabilities */
static void H2_ReadTransprob( long int nelec )
{
	char cdDATAFILE[N_H2_ELEC][FILENAME_PATH_LENGTH_2] = 
	{
		"H2_transprob_X.dat" ,
		"H2_transprob_B.dat" , 
		"H2_transprob_C_plus.dat" ,
		"H2_transprob_C_minus.dat" , 
		"H2_transprob_B_primed.dat" , 
		"H2_transprob_D_plus.dat" ,
		"H2_transprob_D_minus.dat" 
	};
	FILE *ioDATA;
	char chLine[FILENAME_PATH_LENGTH_2] , 
		chFilename[FILENAME_PATH_LENGTH_2] ;
	long int i, n1, n2, n3;
	long int iVibHi , iVibLo , iRotHi , iRotLo , iElecHi , iElecLo;
	int lgEOL;

	/* check on path is file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , cdDATAFILE[nelec] );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , cdDATAFILE[nelec] );
	}

	/* now open the data file */
	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadTransprob could not open %s\n", cdDATAFILE[nelec] );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " H2_ReadTransprob could not open %s\n", cdDATAFILE[nelec]);
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in H2_ReadTransprob]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read the first line and check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " H2_ReadTransprob could not read first line of %s\n", cdDATAFILE[nelec]);
		puts( "[Stop in H2_ReadTransprob]" );
		cdEXIT(EXIT_FAILURE);
	}
	i = 1;
	/* level 1 magic number */
	n1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	n3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* magic number
	 * the following is the set of numbers that appear at the start of level1.dat 01 08 10 */
	if( ( n1 != 2 ) || ( n2 != 4 ) || ( n3 != 29 ) )
	{
		fprintf( ioQQQ, 
			" H2_ReadTransprob: the version of %s is not the current version.\n", cdDATAFILE[nelec] );
		fprintf( ioQQQ, 
			" I expected to find the number 2 4 29 and got %li %li %li instead.\n" ,
			n1 , n2 , n3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in H2_ReadTransprob]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* read until not a comment */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
		BadRead();

	while( chLine[0]=='#' )
	{
		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			BadRead();
	}
	iVibHi = 1;
	while( iVibHi >= 0 )
	{
		double Aul;
		sscanf(chLine,"%li\t%li\t%li\t%li\t%li\t%li\t%le", 
			&iElecHi , &iVibHi ,&iRotHi , &iElecLo , &iVibLo , &iRotLo , &Aul );
		ASSERT( iElecHi == nelec );
		/* negative iVibHi says end of data */
		if( iVibHi < 0 )
			continue;

		/* check that we actually included the levels in the model representation */
		if( iVibHi <= nVib_hi[iElecHi] && 
		    iVibLo <= nVib_hi[iElecLo] && 
			iRotHi <= nRot_hi[iElecHi][iVibHi] && 
			iRotLo <= nRot_hi[iElecLo][iVibLo])
		{
			double ener = energy_wn[iElecHi][iVibHi][iRotHi] - energy_wn[iElecLo][iVibLo][iRotLo];

			H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul = (float)Aul;
			/* prints transitions with negative energies  -  should not happen */
			if( ener <= 0. )
			{
				fprintf(ioQQQ,"negative energy H2 transition\t%li\t%li\t%li\t%li\t%.2e\t%.2e\n", 
					iVibHi,iVibLo,iRotHi,iRotLo,Aul,
					H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN);
				ShowMe();
			}
		}
#		if 0
		/* this prints all levels with As but without energies */
		else
		{
			fprintf(ioQQQ,"no\t%li\t%li\t%li\t%li\t%.2e\n", 
				iVibHi,iVibLo,iRotHi,iRotLo,Aul);
		}
#		endif

		if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			BadRead();
		while( chLine[0]=='#' )
		{
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
				BadRead();
		}
	}
	fclose( ioDATA );
}

/* this is equation 8 of Takahashi 2001, clearer definition is given in
 * equation 5 and following discussion of
 * >>refer	H2	formation	Takahashi, J., & Uehara, H., 2001, ApJ, 561, 843-857
 * 0.27eV, convert into wavenumbers */
static double XVIB[H2_TOP] = { 0.70 , 0.60 , 0.20 };
static double Xdust[H2_TOP] = { 0.04 , 0.10 , 0.40 };
double EH2_eval(  long int iVib , H2_type ipH2 )
{
	double EH2_here;
	double Evm = DissocEnergies[0]* XVIB[ipH2] + energy_off;

	double Ev = (energy_wn[0][iVib][0]+energy_off);
	/* equation 9 of Takahashi 2001 which is only an approximation
	double EH2 = DissocEnergies[0] * (1. - Xdust[ipH2] ); */
	/* equation 1, 2 of 
	 * Takahashi, Junko, & Uehara, Hideya, 2001, ApJ, 561, 843-857,
	 * this is heat deposited on grain by H2 formation in this state */
	double Edust = DissocEnergies[0] * Xdust[ipH2] *
		( 1. - ( (Ev - Evm) / (DissocEnergies[0]+energy_off-Evm)) *
		( (1.-Xdust[ipH2])/2.) );
	ASSERT( Edust >= 0. );

	/* energy is total binding energy less energy lost on grain surface and energy offset */
	EH2_here = DissocEnergies[0]+energy_off - Edust;
	ASSERT( EH2_here >= 0.);

	return EH2_here;
}

/*H2_vib_dist evaluates the vibration distribution for H2 formed on grains */
double H2_vib_dist( long int iVib , H2_type ipH2 , double EH2)
{
	double G1[H2_TOP] = { 0.3 , 0.4 , 0.9 };
	double G2[H2_TOP] = { 0.6 , 0.6 , 0.4 };
	double Evm = DissocEnergies[0]* XVIB[ipH2] + energy_off;
	double Fv;
	if( (energy_wn[0][iVib][0]+energy_off) <= Evm )
	{
		/* equation 4 of Takahashi 2001 */
		Fv = sexp( POW2( (energy_wn[0][iVib][0]+energy_off - Evm)/(G1[ipH2]* Evm ) ) );
	}
	else
	{
		/* equation 5 of Takahashi 2001 */
		Fv = sexp( POW2( (energy_wn[0][iVib][0]+energy_off - Evm)/(G2[ipH2]*(EH2 - Evm ) ) ) );
	}
	return Fv;
}

/*H2_Create create variables for the H2 molecule, called by ContCreatePointers after continuum
 * mesh has been set up */
void H2_Create(void)
{
	long int i , iElecHi , iElecLo ;
	long int iVibHi , iVibLo ;
	long int iRotHi , iRotLo ;
	long int iElec, iVib , iRot;
	long int nColl,
		nlines;
	int ier;
	int nEner;
	H2_type ipH2;
	float sum , sumj , sumv , sumo , sump;

	/* this is flag set above - when true h2 code is not executed - this is way to
	 * avoid this code when it is not working */
	/* only malloc vectors one time per coreload */

	if( lgQUIT || lgH2_READ_DATA || !h2.lgH2ON )
		return;

	/* this var is in h2.h and prevents h2 from being change once committed here */
	lgH2_READ_DATA = TRUE;

	/* create special vector that saves collision rates within ground */
	/* this will contain a vector for collisions within the X ground elec state,
	* CollRateFit[coll_type][vib_up][rot_up][vib_lo][rot_lo][3] */
	/* N_X_COLLIDER is number of different species that collide */
	if( (CollRateFit = (float******)MALLOC(sizeof(float*****)*(unsigned)N_X_COLLIDER ) )==NULL )
		BadMalloc();
	if( (CollRate = (float*****)MALLOC(sizeof(float****)*(unsigned)N_X_COLLIDER ) )==NULL )
		BadMalloc();
	iElecHi = 0;
	for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
	{
		/* the current data set is limited to vib hi <= 3 */
		/* VIB_COLLID is highest vib state with collision data */
		/*if( (CollRateFit[nColl] = (float*****)MALLOC(sizeof(float****)*(unsigned)(nVib_hi[iElecHi]+1) ) )==NULL )*/
		if( (CollRateFit[nColl] = (float*****)MALLOC(sizeof(float****)*(unsigned)(VIB_COLLID+1) ) )==NULL )
			BadMalloc();
		if( (CollRate[nColl] = (float****)MALLOC(sizeof(float***)*(unsigned)(VIB_COLLID+1) ) )==NULL )
			BadMalloc();
		/*for( iVibHi = 0; iVibHi <= nVib_hi[iElecHi] ; ++iVibHi )*/
		for( iVibHi = 0; iVibHi <= VIB_COLLID ; ++iVibHi )
		{
			if( (CollRateFit[nColl][iVibHi] = (float****)MALLOC(sizeof(float***)*(unsigned)(nRot_hi[iElecHi][iVibHi]+1) ) )==NULL )
				BadMalloc();
			if( (CollRate[nColl][iVibHi] = (float***)MALLOC(sizeof(float**)*(unsigned)(nRot_hi[iElecHi][iVibHi]+1) ) )==NULL )
				BadMalloc();
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/*if( (CollRateFit[nColl][iVibHi][iRotHi] = (float***)MALLOC(sizeof(float**)*(unsigned)(nVib_hi[iElecHi]+1) ) )==NULL )*/
				if( (CollRateFit[nColl][iVibHi][iRotHi] = (float***)MALLOC(sizeof(float**)*(unsigned)(VIB_COLLID+1) ) )==NULL )
					BadMalloc();
				if( (CollRate[nColl][iVibHi][iRotHi] = (float**)MALLOC(sizeof(float*)*(unsigned)(VIB_COLLID+1) ) )==NULL )
					BadMalloc();
				/*for( iVibLo=0; iVibLo<=iVibHi; ++iVibLo )*/
				for( iVibLo=0; iVibLo<(VIB_COLLID+1); ++iVibLo )
				{
					if( (CollRateFit[nColl][iVibHi][iRotHi][iVibLo] = (float**)MALLOC(sizeof(float*)*(unsigned)(nRot_hi[iElecHi][iVibLo]+1) ) )==NULL )
						BadMalloc();
					if( (CollRate[nColl][iVibHi][iRotHi][iVibLo] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[iElecHi][iVibLo]+1) ) )==NULL )
						BadMalloc();
					for( iRotLo=0; iRotLo<=nRot_hi[iElecHi][iVibLo]; ++iRotLo )
					{
						/* the last one - the three coefficients */
						if( (CollRateFit[nColl][iVibHi][iRotHi][iVibLo][iRotLo] = (float*)MALLOC(sizeof(float)*(unsigned)(3) ) )==NULL )
							BadMalloc();

						/* zero out the collisional rates since only a minority of them will are known*/
						for( i=0; i<3; ++i )
						{
							CollRateFit[nColl][iVibHi][iRotHi][iVibLo][iRotLo][i] = 0.;
						}
						CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo] = 0.;
					}
				}

			}
		}
	}

	/* create space for the electronic levels */
	if( (energy_wn = (double***)MALLOC(sizeof(double**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();
	if( (Boltzmann = (double***)MALLOC(sizeof(double**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();
	if( (dissprob = (float***)MALLOC(sizeof(float**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();
	if( (disske = (float***)MALLOC(sizeof(float**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();
	if( (populations_LTE = (double***)MALLOC(sizeof(double**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();
	if( (populations = (float***)MALLOC(sizeof(float**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();
	if( (old_populations = (float***)MALLOC(sizeof(float**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();
	/*if( (rate_in = (double***)MALLOC(sizeof(double**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();
	if( (rate_out = (double***)MALLOC(sizeof(double**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();*/
	if( (stat = (float***)MALLOC(sizeof(float**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();
	if( (lgOrtho = (int***)MALLOC(sizeof(int**)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();
	if( (pops_per_vib = (float**)MALLOC(sizeof(float*)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();

	for( iElec = 0; iElec<n_elec_states; ++iElec )
	{

		if( lgH2_TRACE )
			fprintf(ioQQQ,"elec %li highest vib= %li\n", iElec , nVib_hi[iElec] );

		ASSERT( nVib_hi[iElec] > 0 );

		/* nVib_hi is now the highest vib level before dissociation,
		 * now allocate space to hold the number of rotation levels */
		if( (energy_wn[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			BadMalloc();
		if( (Boltzmann[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			BadMalloc();
		/* the ground electronic state is stable - these do not exist */
		if( iElec > 0 )
		{
			if( (dissprob[iElec] = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
				BadMalloc();
			if( (disske[iElec] = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
				BadMalloc();
		}
		if( (populations_LTE[iElec] = (double**)MALLOC(sizeof(double*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			BadMalloc();
		if( (populations[iElec] = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			BadMalloc();
		if( (old_populations[iElec] = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			BadMalloc();
		if( (stat[iElec] = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			BadMalloc();
		if( (lgOrtho[iElec] = (int**)MALLOC(sizeof(int*)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			BadMalloc();
		if( (pops_per_vib[iElec] = (float*)MALLOC(sizeof(float)*(unsigned)(nVib_hi[iElec]+1) ) )==NULL )
			BadMalloc();

		/* now loop over all vib levels, and find out how many rotation levels there are */
		/* ground is special since use tabulated data - there are 14 vib states,
		 * ivib=14 is highest */
		for( iVib = 0; iVib <= nVib_hi[iElec] ; ++iVib )
		{
			/* lastly create the space for the rotation quantum number */
			if( (energy_wn[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				BadMalloc();
			if( (Boltzmann[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				BadMalloc();
			if( iElec > 0 )
			{
				if( (dissprob[iElec][iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
					BadMalloc();
				if( (disske[iElec][iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
					BadMalloc();
			}
			if( (populations_LTE[iElec][iVib] = (double*)MALLOC(sizeof(double)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				BadMalloc();
			if( (populations[iElec][iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				BadMalloc();
			if( (old_populations[iElec][iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				BadMalloc();
			if( (stat[iElec][iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				BadMalloc();
			if( (lgOrtho[iElec][iVib] = (int*)MALLOC(sizeof(int)*(unsigned)(nRot_hi[iElec][iVib]+1) ) )==NULL )
				BadMalloc();
		}
	}

	/* these do not have electronic levels - all within X */
	if( (ipPhoto = (int**)MALLOC(sizeof(int*)*(unsigned)(nVib_hi[0]+1) ) )==NULL )
		BadMalloc();
	if( (col_rate_in = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[0]+1) ) )==NULL )
		BadMalloc();
	if( (col_rate_out = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[0]+1) ) )==NULL )
		BadMalloc();
	if( (rad_rate_in = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[0]+1) ) )==NULL )
		BadMalloc();
	if( (rad_rate_out = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[0]+1) ) )==NULL )
		BadMalloc();
	if( (coll_dissoc_rate_coef = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[0]+1) ) )==NULL )
		BadMalloc();
	if( (H2_X_colden = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[0]+1) ) )==NULL )
		BadMalloc();
	if( (H2_X_formation = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[0]+1) ) )==NULL )
		BadMalloc();
	/* space for the vibration levels */
	for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
	{
		/* space for the rotation quantum number */
		if( (ipPhoto[iVib] = (int*)MALLOC(sizeof(int)*(unsigned)(nRot_hi[0][iVib]+1) ) )==NULL )
			BadMalloc();
		if( (col_rate_in[iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[0][iVib]+1) ) )==NULL )
			BadMalloc();
		if( (col_rate_out[iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[0][iVib]+1) ) )==NULL )
			BadMalloc();
		if( (rad_rate_in[iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[0][iVib]+1) ) )==NULL )
			BadMalloc();
		if( (rad_rate_out[iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[0][iVib]+1) ) )==NULL )
			BadMalloc();
		if( (coll_dissoc_rate_coef[iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[0][iVib]+1) ) )==NULL )
			BadMalloc();
		if( (H2_X_colden[iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[0][iVib]+1) ) )==NULL )
			BadMalloc();
		if( (H2_X_formation[iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[0][iVib]+1) ) )==NULL )
			BadMalloc();
		
		for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
		{
			/* zero out the matrices */
			H2_X_colden[iVib][iRot] = 0.;
			H2_X_formation[iVib][iRot] = 0.;
		}
	}

	/* distribution function for populations following formation from H minus H- */
	if( (H2_X_hminus_formation_distribution = (float***)MALLOC(sizeof(float**)*(unsigned)(nTE_HMINUS) ) )==NULL )
		BadMalloc();
	for( i=0; i<nTE_HMINUS; ++i )
	{
		if( (H2_X_hminus_formation_distribution[i] = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[0]+1) ) )==NULL )
			BadMalloc();
		/* space for the vibration levels */
		for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
		{
			if( (H2_X_hminus_formation_distribution[i][iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[0][iVib]+1) ) )==NULL )
				BadMalloc();
			for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
			{
				H2_X_hminus_formation_distribution[i][iVib][iRot] = 0.;
			}
		}
	}
	H2_Read_hminus_distribution();

	/* read in cosmic ray distribution inforamtion */
	H2_Read_Cosmicray_distribution();

	/* >>chng 02 oct 08, resolved grain types so increased dim of var*/
	/* grain formation matrix */
	if( (H2_X_grain_formation_distribution = (float***)MALLOC(sizeof(float**)*(unsigned)(H2_TOP) ) )==NULL )
		BadMalloc();
	for( ipH2=0; ipH2<H2_TOP; ++ipH2 )
	{
		if( (H2_X_grain_formation_distribution[ipH2] = (float**)MALLOC(sizeof(float*)*(unsigned)(nVib_hi[0]+1) ) )==NULL )
			BadMalloc();
		
		/* space for the vibration levels */
		for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
		{
			if( (H2_X_grain_formation_distribution[ipH2][iVib] = (float*)MALLOC(sizeof(float)*(unsigned)(nRot_hi[0][iVib]+1) ) )==NULL )
				BadMalloc();
			for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
			{
				H2_X_grain_formation_distribution[ipH2][iVib][iRot] = 0.;
			}
		}
	}

	/* space for the energy vector is now malloced, must fill it in,
	 * defines array energy_wn[nelec][iVib][iRot] */
	for( iElec=0; iElec<n_elec_states; ++iElec )
	{
		/* get energies out of files into array energy_wn[nelec][iVib][iRot] */
		H2_ReadEnergies(iElec);

		/* get dissociation probabilities and energies - ground state is stable */
		if( iElec > 0 )
			H2_ReadDissprob(iElec);
	}

	/* >>02 oct 18, add photodissociation, H2 + hnu => 2H + KE */
	/* we now have ro-vib energies, now set up threshold array offsets
	 * for photodissociation */
	for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
	{
		for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
		{
			/* this is energy needed to get up to n=3 electronic continuum */
			double thresh = (DissocEnergies[1] - energy_wn[0][iVib][iRot])*WAVNRYD;
			ASSERT( thresh > 0. );
			ipPhoto[iVib][iRot] = ipoint(thresh);
		}
	}

	if(lgH2_TRACE) 
	{
		long int nlevels;
		fprintf(ioQQQ,
			" H2_Create: there are %li electronic levels, in each level there are",
			n_elec_states);
		nlevels = 0;
		for( iElec=0; iElec<n_elec_states; ++iElec)
		{
			/* print the number of levels within iElec */
			fprintf(ioQQQ,"\t(%li %li)", iElec ,  nLevels_per_elec[iElec] );
			nlevels += nLevels_per_elec[iElec] ;
		}
		fprintf(ioQQQ,
			" for a total of %li levels.\n", nlevels );
	}

	/* now read in the various sets of collision data */
	for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
	{
		/* ground state has tabulated data */
		H2_ReadCollRates(nColl);
	}

	if( (Xenergies = (float*)MALLOC(sizeof(float)*(unsigned)nLevels_per_elec[0] ) )==NULL )
		BadMalloc();
	if( (ipX_ener_sort = (long int*)MALLOC(sizeof(long int)*(unsigned)nLevels_per_elec[0] ) )==NULL )
		BadMalloc();
	if( (ipVib_energies = (long int*)MALLOC(sizeof(long int)*(unsigned)nLevels_per_elec[0] ) )==NULL )
		BadMalloc();
	if( (ipRot_energies = (long int*)MALLOC(sizeof(long int)*(unsigned)nLevels_per_elec[0] ) )==NULL )
		BadMalloc();
	/* we need to create a vector of sorted energies for X */

	nEner = 0;
	iElecHi = 0;
	/* get set of energies */
	for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
	{
		for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
		{
			Xenergies[nEner] = (float)energy_wn[iElecHi][iVibHi][iRotHi];
			ipVib_energies[nEner] = iVibHi;
			ipRot_energies[nEner] = iRotHi;
			++nEner;
		}
	}
	ASSERT( nLevels_per_elec[0]==nEner);

	/* sort the energy levels so that we can do top-down trickle of states */
	/*spsort netlib routine to sort array returning sorted indices */
	spsort(
		/* input array to be sorted */
		Xenergies, 
		/* number of values in x */
		nLevels_per_elec[0], 
		/* permutation output array */
		ipX_ener_sort, 
		/* flag saying what to do - 1 sorts into increasing order, not changing
		* the original vector, -1 sorts into decreasing order. 2, -2 change vector */
		1, 
		/* error condition, should be 0 */
		&ier);
	/* now loop over the energies confirming the order */
	for( nEner=0; nEner<nLevels_per_elec[0]-1; ++nEner )
	{
		ASSERT( Xenergies[ipX_ener_sort[nEner]] < Xenergies[ipX_ener_sort[nEner+1]] );
		/* following will print quantum indices and energies */
		/*fprintf(ioQQQ,"%li\t%li\t%.3e\n",
			ipVib_energies[ipX_ener_sort[nEner]],
			ipRot_energies[ipX_ener_sort[nEner]],
			Xenergies[ipX_ener_sort[nEner]]);*/
	}

	iElecHi = 0;
	if( (SaveLine = 
		(double****)MALLOC(sizeof(double***)*(unsigned)(nVib_hi[iElecHi]+1) ) )==NULL )
		BadMalloc();
	if( (lgLineExist = 
		(int****)MALLOC(sizeof(int***)*(unsigned)(nVib_hi[iElecHi]+1) ) )==NULL )
		BadMalloc();
	for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
	{
		if( (SaveLine[iVibHi] = 
			(double***)MALLOC(sizeof(double**)*(unsigned)(nRot_hi[iElecHi][iVibHi]+1) ) )==NULL )
			BadMalloc();
		if( (lgLineExist[iVibHi] = 
			(int***)MALLOC(sizeof(int**)*(unsigned)(nRot_hi[iElecHi][iVibHi]+1) ) )==NULL )
			BadMalloc();
		iElecLo = 0;
		for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
		{
			/* now the lower levels */
			if( (SaveLine[iVibHi][iRotHi] = 
				(double**)MALLOC(sizeof(double *)*(unsigned)(iVibHi+1) ) )==NULL )
				BadMalloc();
			if( (lgLineExist[iVibHi][iRotHi] = 
				(int**)MALLOC(sizeof(int *)*(unsigned)(iVibHi+1) ) )==NULL )
				BadMalloc();
			for( iVibLo=0; iVibLo<=iVibHi; ++iVibLo )
			{
				long nr = nRot_hi[iElecLo][iVibLo];
				if( iVibHi==iVibLo )
					/* max because cannot malloc 0 bytes */
					nr = MAX2(1,iRotHi-1);
				if( (SaveLine[iVibHi][iRotHi][iVibLo] = 
					(double*)MALLOC(sizeof(double )*(unsigned)(nr+1) ) )==NULL )
					BadMalloc();
				if( (lgLineExist[iVibHi][iRotHi][iVibLo] = 
					(int*)MALLOC(sizeof(int )*(unsigned)(nr+1) ) )==NULL )
					BadMalloc();
				for(iRotLo=0; iRotLo<=nr; ++iRotLo )
				{
					lgLineExist[iVibHi][iRotHi][iVibLo][iRotLo] = FALSE;
					SaveLine[iVibHi][iRotHi][iVibLo][iRotLo] = 0.;
				}
			}
		}
	}

	/* ============================================= */
	/* create the main array of lines */
	if( (H2Lines = (EmLine******)MALLOC(sizeof(EmLine *****)*(unsigned)n_elec_states ) )==NULL )
		BadMalloc();

	nlines = 0;
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{

		if( (H2Lines[iElecHi] = 
			(EmLine*****)MALLOC(sizeof(EmLine ****)*(unsigned)(nVib_hi[iElecHi]+1) ) )==NULL )
			BadMalloc();

		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			if( (H2Lines[iElecHi][iVibHi] = 
				(EmLine****)MALLOC(sizeof(EmLine ***)*(unsigned)(nRot_hi[iElecHi][iVibHi]+1) ) )==NULL )
				BadMalloc();

			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				 * concerned with excited electronic levels as a photodissociation process
				 * code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				if( (H2Lines[iElecHi][iVibHi][iRotHi] = 
					(EmLine***)MALLOC(sizeof(EmLine **)*(unsigned)(1) ) )==NULL )
					BadMalloc();

				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					 * but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					/* within X, no transitions v_hi < v_lo transitions exist */ 
					if( iElecLo==iElecHi )
						nv = iVibHi;

					if( (H2Lines[iElecHi][iVibHi][iRotHi][iElecLo] = 
						(EmLine**)MALLOC(sizeof(EmLine *)*(unsigned)(nv+1) ) )==NULL )
						BadMalloc();

					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							/* max because cannot malloc 0 bytes */
							nr = MAX2(1,iRotHi-1);
						if( (H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo] = 
							(EmLine*)MALLOC(sizeof(EmLine)*(unsigned)(nr+1) ) )==NULL )
							BadMalloc();
						/* set flag saying that space exists */
						if( iElecLo==0 && iElecHi==0 )
						{
							for( iRotLo=0; iRotLo<=nr; ++iRotLo )
							{
								lgLineExist[iVibHi][iRotHi][iVibLo][iRotLo] = TRUE;
							}
						}
						nlines += nr+1;
					}
				}
			}
		}
	}
	if( lgH2_TRACE )
		fprintf(ioQQQ," There are a total of %li lines in the entire H2 molecule.\n", nlines );

	/* now zero out the transition probabilities */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* NB - X is the only lower level considered here, since we are only 
				 * concerned with excited electronic levels as a photodissociation process
				 * code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					 * but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;
						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/* set inital values for each line structure */
							EmLineZero( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] );
							/* some of these may not be initialized */
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul = 0.;
						}
					}
				}
			}
		}
	}

	/* space for the energy vector is now malloced, must read trans probs from table */
	for( iElec=0; iElec<n_elec_states; ++iElec )
	{
		/* ground state has tabulated data */
		H2_ReadTransprob(iElec);
	}

	/* set all statistical weights - ours is total statistical weight - 
	 * including nuclear spin */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* unlike atoms, for H2 nuclear spin is taken into account - so the
				 * statistical weight of even and odd J states differ by factor of 3 - see page 166, sec par
				 * >>>refer	H2	stat wght	Shull, J.M., & Beckwith, S., 1982, ARAA, 20, 163-188 */
				if( IS_ODD(iRotHi+nRot_add_ortho_para[iElecHi]) )
				{
					/* ortho */
					lgOrtho[iElecHi][iVibHi][iRotHi] = TRUE;
					stat[iElecHi][iVibHi][iRotHi] = 3.f*(2.f*iRotHi+1.f);
				}
				else
				{
					/* para */
					lgOrtho[iElecHi][iVibHi][iRotHi] = FALSE;
					stat[iElecHi][iVibHi][iRotHi] = (2.f*iRotHi+1.f);
				}
			}
		}
	}

	/* now free unused transitions to excited electronic states */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				 * concerned with excited electronic levels as a photodissociation process
				 * code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					 * but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];

						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;
						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{

							/* this is special flag for H2 - these are used in velset (in tfidle.c) to 
							* set doppler velocities for species */
							/* NB this must be kept parallel with nelem and ionstag in H2Lines EmLine struc,
							* since that struc expects to find the abundances here - abund set in hmole.c */
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].nelem = LIMELM+3;
							/* this does not mean anything for a molecule */
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].IonStg = 1;

							/* statistical weights of lower and upper levels */
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo = stat[iElecLo][iVibLo][iRotLo];
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi = stat[iElecHi][iVibHi][iRotHi];

							/* energy of the transition in wavenumbers */
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN = 
								(float)(energy_wn[iElecHi][iVibHi][iRotHi] - energy_wn[iElecLo][iVibLo][iRotLo]);

							/*wavelength of transition in Angstroms */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN > SMALLFLOAT)
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].WLAng = 
								(float)(1.e8f/H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN /
								RefIndex(&H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo]));

							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyK = 
								(float)(T1CM)*H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN;

							/* energy of photon in ergs */
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyErg = 
								(float)(ERG1CM)*H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN;

							/*if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )*/
							/* >>chng 03 feb 14, change to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								/* line redistribution function - will use complete redistribution */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].iRedisFun = ipCRD;

								/* line optical depths in direction towards source of ionizing radiation */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauIn = opac.taumin;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauCon = opac.taumin;
								/* outward optical depth */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauTot = 1e20f;


								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].dampXvel = 
									(float)(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul/
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN/PI4);

								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gf = 
									(float)(GetGF(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul,
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN,
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi ) );

								/* derive the abs coef, call to function is gf, wl (A), g_low */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].opacity = (float)(
									abscf(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gf,
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN,
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo));
							}
						}
					}
				}
			}
		}
	}

	/* define branching ratios for deposition of H2 formed on grain surfaces,
	 * set true to use Takahashi distrution, false to use Draine & Bertoldi */
#	define TAKAHASHI_DIST	TRUE

	for( ipH2=0; ipH2<H2_TOP; ++ipH2 )
	{
		sum = 0.;
		sumj = 0.;
		sumv = 0.;
		sumo = 0.;
		sump = 0.;
		iElec = 0;
		if( !TAKAHASHI_DIST )
		{
			/* H2 formation temperature, for equation 19, page 271, of
			* >>refer	H2	formation destribution	Draine, B.T., & Bertoldi, F., 1996, ApJ, 468, 269-289
			*/
#			define T_H2_FORM 50000.
			for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
			{
				for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
				{
					/* no distinction between grain surface composition */
					H2_X_grain_formation_distribution[ipH2][iVib][iRot] = 
						/* first term is nuclear stat weight */
						(1.f+2.f*lgOrtho[iElec][iVib][iRot]) * (1.f+iVib) *
						(float)sexp( energy_wn[iElec][iVib][iRot]*T1CM/T_H2_FORM );

					sum += H2_X_grain_formation_distribution[ipH2][iVib][iRot];
					sumj += iRot * H2_X_grain_formation_distribution[ipH2][iVib][iRot];
					sumv += iVib * H2_X_grain_formation_distribution[ipH2][iVib][iRot];
					if( lgOrtho[iElec][iVib][iRot] )
					{
						sumo += H2_X_grain_formation_distribution[ipH2][iVib][iRot];
					}
					else
					{
						/* >>chng 02 nov 14, [0][iVib][iRot] -> [ipH2][iVib][iRot], PvH */
						sump += H2_X_grain_formation_distribution[ipH2][iVib][iRot];
					}
				}
			}
		}
		else
		{
			/* >>chng 02 oct 08, resolved grain types */
			/* number of different grain types H2_TOP is set in grainvar.h,
			* types are ice, silicate, graphite */
			double Xrot[H2_TOP] = { 0.14 , 0.15 , 0.15 };
			double Xtrans[H2_TOP] = { 0.12 , 0.15 , 0.25 };
			/* first normalize the vibration distribution function */
			double sumvib = 0.;
			double EH2;

			for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
			{
				double vibdist;
				EH2 = EH2_eval( iVib , ipH2 );
				vibdist = H2_vib_dist( iVib , ipH2 , EH2);
				sumvib += vibdist;
			}
			/* this branch, use distribution function from
			* >>refer	Takahashi, Junko, 2001, ApJ, 561, 254-263 */
			for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
			{
				double Ev = (energy_wn[iElec][iVib][0]+energy_off);
				double Fv;
				/* equation 10 of Takahashi 2001, extra term is energy offset between bottom of potential
				 * the 0,0 level */
				double Erot;
				/*fprintf(ioQQQ," Evvvv\t%i\t%li\t%.3e\n", ipH2 ,iVib , Ev*WAVNRYD*EVRYD);*/

				EH2 = EH2_eval( iVib , ipH2 );

				/* equation 3 of Taktahashi & Uehara */
				Erot = (EH2 - Ev) * Xrot[ipH2] / (Xrot[ipH2] + Xtrans[ipH2]);

				/* email exchange with Junko Takahashi - 
				Thank you for your E-mail.
				I did not intend to generate negative Erot.
				I cut off the populations if their energy levels are negative, and made the total
				population be unity by using normalization factors (see, e.g., Eq. 12).

				I hope that my answer is of help to you and your work is going well.
				With best wishes,
				Junko

				>Thanks for the reply.  By cutting off the population, should we set the
				>population to zero when Erot becomes negative, or should we set Erot to
				>a small positive number? 

				I just set the population to zero when Erot becomes negative.
				Our model is still a rough one for the vibration-rotation distribution function
				of H2 newly formed on dust, because we have not yet had any exact
				experimental or theoretical data about it.
				With best wishes,
				Junko

				 */

				if( Erot > 0. )
				{
					/* the vibrational distribution */
					Fv = H2_vib_dist( iVib , ipH2 , EH2) / sumvib;
					/*fprintf(ioQQQ," vibbb\t%li\t%.3e\n", iVib , Fv );*/

					for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
					{
						/* equation 6 of Takahashi 2001 */
						double gaussian = 
							sexp( POW2( (energy_wn[iElec][iVib][iRot] - energy_wn[iElec][iVib][0] - Erot) /
							(0.5 * Erot) ) );
						/* equation 7 of Takahashi 2001 */
						double thermal = 
							sexp( (energy_wn[iElec][iVib][iRot] - energy_wn[iElec][iVib][0]) /
							Erot );

						/* take the mean of the two */
						double aver = ( gaussian + thermal ) / 2.;
						/*fprintf(ioQQQ,"rottt\t%i\t%li\t%li\t%.3e\t%.3e\t%.3e\t%.3e\n",
							ipH2,iVib,iRot,
							(energy_wn[iElec][iVib][iRot]+energy_off)*WAVNRYD*EVRYD,
							gaussian, thermal , aver );*/

						/* thermal does become > 1 since Erot can become negative */
						ASSERT( gaussian <= 1. /*&& thermal <= 10.*/ );

						H2_X_grain_formation_distribution[ipH2][iVib][iRot] = (float)(
							/* first term is nuclear stat weight */
							(1.f+2.f*lgOrtho[iElec][iVib][iRot]) * Fv * (2.*iRot+1.) * aver );

						sum += H2_X_grain_formation_distribution[ipH2][iVib][iRot];
						sumj += iRot * H2_X_grain_formation_distribution[ipH2][iVib][iRot];
						sumv += iVib * H2_X_grain_formation_distribution[ipH2][iVib][iRot];
						if( lgOrtho[iElec][iVib][iRot] )
						{
							sumo += H2_X_grain_formation_distribution[ipH2][iVib][iRot];
						}
						else
						{
							sump += H2_X_grain_formation_distribution[ipH2][iVib][iRot];
						}
					
					}
				}
				else
				{
					/* this branch Erot is non-positive, so no distribution */
					for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
					{
						H2_X_grain_formation_distribution[ipH2][iVib][iRot] = 0.;
					}
				}
			}
		}
		if( lgH2_TRACE )
			fprintf(ioQQQ, "H2 form grains mean J= %.3f mean v = %.3f ortho/para= %.3f\n", 
				sumj/sum , sumv/sum , sumo/sump );

		/* now rescale so that integral is unity */
		for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
		{
			for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
			{
				H2_X_grain_formation_distribution[ipH2][iVib][iRot] /= sum;
				/* print the distribution function */
				/*fprintf(ioQQQ,"disttt\t%i\t%li\t%li\t%.4e\t%.4e\n",
					ipH2, iVib , iRot, 
					energy_wn[iElec][iVib][iRot] , 
					H2_X_grain_formation_distribution[ipH2][iVib][iRot] );*/
			}
		}
	}
	/*exit(123);*/

	/* at this stage the full electronic, vibration, and rotation energies have been defined,
	 * this is an option to print the energies */
	{
		/* set following TRUE to get printout, FALSE to not print energies */
		if( DEBUG_ENER )
		{
			/* print title for quantum numbers and energies */
			/*fprintf(ioQQQ,"elec\tvib\trot\tenergy\n");*/
			for( iElec=0; iElec<n_elec_states; ++iElec )
			{
				/* now must specify the number of rotation levels within the vib levels */
				for( iVib=0; iVib<=nVib_hi[iElec]; ++iVib )
				{
					for( iRot=0; iRot<=nRot_hi[iElec][iVib]; ++iRot )
					{
						fprintf(ioQQQ,"%li\t%li\t%li\t%.5e\n",
							iElec, iVib, iRot ,
							energy_wn[iElec][iVib][iRot]);
					}
				}
			}
			/* this will exit the program after printing the level energies */
			cdEXIT(EXIT_SUCCESS);
		}
	}

	return;
}

/*H2_Reset called to reset variables that are needed after an iteration */
void H2_Reset( void )
{
	/* number of times large molecules evaluated in this iteration,
	 * is FALSE if never evaluated */
	nCallH2_this_iteration = 0;

	/* these remember the largest and smallest factors needed to
	 * renormalize the H2 chemistry */
	h2.renorm_max = 1.;
	h2.renorm_min = 1.;
}

/*H2_Zero zero out vars in the large H2 molecule, called from zero 
 * NB - this routine is called before space allocated - must not zero out
 * any allocated arrays */
void H2_Zero( void )
{
	/* this is the smallest ratio of H2/H where we will bother with the large H2 molecule
	 * this value was chosen so that large mole used at very start of TH85 standard pdr,
	 * NB - this appears in headinfo and must be updated there is changed here */
	H2_to_H_limit = 1e-6;

	h2.lgH2ON = FALSE;
	lgH2_TRACE = FALSE;

	/* option to turn off or on gbar collisions of the collision rate,
	 * default is to have it on */
	/* turn lgColl_gbar on/off with atom h2 gbar on off */
	lgColl_gbar = TRUE;

	/* include collision rates that come from real calculations,
	 * off with atom h2 collisions off command */
	lgColl_deexec_Calc = TRUE;
	lgColl_dissoc_coll = TRUE;

	/* these remember the largest and smallest factors needed to
	 * renormalize the H2 chemistry */
	h2.renorm_max = 1.;
	h2.renorm_min = 1.;

	h2.frac_abund = 0.;
	nCallH2_this_iteration = 0;
	h2.ortho_density = 0.;
	h2.para_density = 0.;
	hmi.H2_Solomon_rate_BigH2 = 0.;
	hmi.H2_photodissoc_BigH2 = 0.;

	/* say that H2 has never been computed */
	hmi.lgBigH2_evaluated = FALSE;

	hmi.lgH2_Thermal_BigH2 = TRUE;
	hmi.lgH2_Chemistry_BigH2 = TRUE;

	if( !lgH2_READ_DATA )
	{
		/* the number of electronic levels in the H2 molecule,
		 * to just do the Lyman and Werner bands set to 3 -
		 * reset with atom h2 levels command,
		 * default is all levels with data */
		n_elec_states = N_H2_ELEC;
	}

	/* these are the default number of vib and rot states that are done with a matrix inversionn
	 * change with atom h2 matrix nVIB nROT */
#	define	MATRIX_nVIB 0
#	define	MATRIX_nROT 0

	/* the number of vibration and rotation levels used in the matrix solution
	 * of the level populations - set with atom h2 matrix nVib nRot */
	matrix_vib = MATRIX_nVIB;
	matrix_rot = MATRIX_nROT;

	return;

}

/* set the ipCont struc element for the H2 molecule, called by ContCreatePointers */
void H2_ContPoint( void )
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;

	if( !h2.lgH2ON )
		return;

	/* set array index for line energy within continuum array */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				 * concerned with excited electronic levels as a photodissociation process
				 * code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					 * but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/*if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )*/
							/* >>chng 03 feb 14, change to > 0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ipCont = 
									(int)ipLineEnergy(
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN * WAVNRYD , 
									"H2  " , 0 );
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ipFine = 
									ipFineCont(
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN * WAVNRYD );
							}
						}
					}
				}
			}
		}
	}
	return;
}

/* ===================================================================== */
/* radiative acceleration due to H2 called in forlin */
double H2_Accel(void)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;
	double h2_drive;

	if( !h2.lgH2ON || !h2.nCallH2_this_zone )
		return(0.);

	/* this routine computes the line driven radiative acceleration
	 * due to H2 molecule*/

	h2_drive = 0.;
	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/* >>chng 03 feb 14, from !=0 to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								h2_drive += H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].pump*
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyErg*
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc;
							}
						}
					}
				}
			}
		}
	}
	return h2_drive;
}

/* ===================================================================== */
/* rad pre due to h2 lines called in PresTotCurrent */
double H2_RadPress(void)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;
	double press;

	/* will be used to check on size of opacity, was capped at this value */
	float smallfloat=SMALLFLOAT*10.f;

	if( !h2.lgH2ON || !h2.nCallH2_this_zone )
		return(0.);

	press = 0.;
	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/* >>chng 03 feb 14, from !=0 to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi > smallfloat &&
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc > smallfloat )
								{
									double RadPres1 = 5.551e-2*(
										powi(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyWN/1.e6,4))*
										(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi/
										H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi)/
										(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo/
										H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo)*
										RT_LineWidth(&H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo])*2.;

										press += RadPres1;
								}
							}
						}
					}
				}
			}
		}
	}
	return press;
}

/* ===================================================================== */
/* internal energy of H2 called in PresTotCurrent */
double H2_InterEnergy(void)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;
	double energy;

	if( !h2.lgH2ON || !h2.nCallH2_this_zone )
		return(0.);

	energy = 0.;
	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/* >>chng 03 feb 14, from !=0 to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								energy += 
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi* 
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyErg;
							}
						}
					}
				}
			}
		}
	}
	return energy;
}

/* ===================================================================== */
/* punch line data for H2 molecule */
void H2_Punch_line_data(
	/* io unit for punch */
	FILE* ioPUN ,
	/* punch all levels if true, only subset if false */
	int lgDoAll )
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;

#	ifdef DEBUG_FUN
	fputs( "<+>H2_Punch_line_data()\n", debug_fp );
#	endif

	if( !h2.lgH2ON )
		return;

	if( lgDoAll )
	{
		fprintf( ioQQQ, 
			" H2_Punch_line_data ALL option not implemented in H2_Punch_line_data yet 1\n" );
		puts( "[Stop in H2_Punch_line_data]" );
		cdEXIT(EXIT_FAILURE);
	}
	else
	{

		/* loop over all possible lines */
		for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
		{
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					/* now the lower levels */
					/* NB - X is the only lower level considered here, since we are only 
					* concerned with excited electronic levels as a photodissociation process
					* code exists to relax this assumption - simply change following to iElecHi */
					long int lim_elec_lo = 0;
					for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
					{
						/* want to include all vib states in lower level if different elec level,
						* but only lower vib levels if same elec level */
						long int nv = nVib_hi[iElecLo];
						if( iElecLo==iElecHi )
							nv = iVibHi;
						for( iVibLo=0; iVibLo<=nv; ++iVibLo )
						{
							long nr = nRot_hi[iElecLo][iVibLo];
							if( iElecLo==iElecHi && iVibHi==iVibLo )
								nr = iRotHi-1;

							for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
							{
								/* >>chng 03 feb 14, from !=0 to >0 */
								if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
								{
									/* print quantum indices */
									fprintf(ioPUN,"%2li %2li %2li %2li %2li %2li ",
										iElecHi,iVibHi,iRotHi,iElecLo,iVibLo,iRotLo );
									Punch1LineData( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] , ioPUN);
								}
							}
						}
					}
				}
			}
		}
		fprintf( ioPUN , "\n");
	}

#	ifdef DEBUG_FUN
	fputs( " <->H2_Punch_line_data()\n", debug_fp );
#	endif
	return;
}

/* include H2 lines in punched optical depths, etc, called from PunchLineStuff */
void H2_PunchLineStuff( FILE * io , float xLimit  , long index)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;

	if( !h2.lgH2ON )
		return;

	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/*if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )*/
							/* >>chng 03 feb 14, from !=0 to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								pun1Line( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] , io , xLimit  , index);
							}
						}
					}
				}
			}
		}
	}
}

/* do emission from H2 - called from RTDiffuse */
void H2_RTDiffuse(double VolFac, 
				double ref)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo , ip;

#	ifdef DEBUG_FUN
	fputs( "<+>H2_RTDiffuse()\n", debug_fp );
#	endif

	if( !h2.lgH2ON || !h2.nCallH2_this_zone )
		return;

	/* loop over all possible lines */
	/* NB - this loop does not include the electronic lines */
	for( iElecHi=0; iElecHi<1; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/*if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )*/
							/* >>chng 03 feb 14, from !=0 to > 0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								/* pointer to line energy in continuum array */
								ip = H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ipCont;
								rfield.outlin[ip-1] += (float)(
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].phots*
									VolFac*opac.tmn[ip-1]*
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ColOvTot);
								rfield.reflin[ip-1] += (float)(
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].phots * ref);
							}
						}
					}
				}
			}
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->H2_RTDiffuse()\n", debug_fp );
#	endif
	return;
}

/* do RT for H2 - called from RTMake */
void H2_RTMake( int lgDoEsc )
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;

#	ifdef DEBUG_FUN
	fputs( "<+>H2_RTMake()\n", debug_fp );
#	endif

	if( !h2.lgH2ON )
		return;

	/* this routine drives calls to make RT relations for H2 molecule */
	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/* >>chng 03 feb 14, change test from !=0 to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								RTMakeLine( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] , lgDoEsc );
							}
						}
					}
				}
			}
		}
	}
	/* this is to take mean of upward transition probability */
	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			double sumpop = 0.;
			double sumpopA = 0.;
			for( iElecHi=1; iElecHi<n_elec_states; ++iElecHi )
			{
				for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
				{
					for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
					{
						/* now the lower levels */
						/* NB - X is the only lower level considered here, since we are only 
						* concerned with excited electronic levels as a photodissociation process
						* code exists to relax this assumption - simply change following to iElecHi */
						iElecLo = 0;
						/* want to include all vib states in lower level if different elec level,
						* but only lower vib levels if same elec level */
						for( iVibLo=0; iVibLo<=nVib_hi[iElecLo]; ++iVibLo )
						{
							long nr = nRot_hi[iElecLo][iVibLo];
							for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
							{
								/* >>chng 03 feb 14, change test from !=0 to >0 */
								if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
								{
									sumpop += H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo;
									sumpopA += H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo*
										H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul;
								}
							}
						}
					}
				}
			}
			fprintf(ioQQQ,"sumpop = %.3e sumpopA= %.3e A=%.3e\n", sumpop, sumpopA, 
				sumpopA/MAX2(SMALLFLOAT,sumpop) );
		}
	}
	return;
}

/* increment optical depth for the H2 molecule, called from RTOptDepthIncre */
void H2_TauInc(void)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo;

	if( !h2.lgH2ON || !h2.nCallH2_this_zone )
		return;

	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/*if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )*/
							/* >>chng 03 feb 14, from !=0 to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								RTLineTauInc( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] );
							}
						}
					}
				}
			}
		}
	}
}


/* initialize optical depths in H2, called from RTOptDepthInit */
void H2_TauInit( void )
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo;

	if( !h2.lgH2ON )
		return;

	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/*if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul != 0. )*/
							/* >>chng 03 feb 14, from !=0 to > 0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								/* >>chng 03 feb 14, use EmLineZero rather than explicit sets */
								EmLineZero( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] );

								/* outward optical depth */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauTot = 1e20f;

#								if 0
								/* outward optical depth */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauTot = 1e20f;

								/* inward optical depth */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauIn = opac.taumin;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].TauCon = opac.taumin;

								/* escape probability */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pesc = 1.;

								/* inward part of line */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].FracInwd = 1.;

								/* destruction probability */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pdest = 0.;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pelec_esc = 0.;

								/* line pumping rate */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].pump = 0.;

								/* population of lower level with correction for stim emission */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc = 0.;

								/* population of lower level */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo = 0.;

								/* population of upper level */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi = 0.;

								/* following two heat exchange excitation, deexcitation */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].cool = 0.;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].heat = 0.;

								/* intensity of line */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].xIntensity = 0.;

								/* opacity in line */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].dTau = 0.;
#								endif
							}
						}
					}
				}
			}
		}
	}
}

/* the large H2 molecule, called from RTOptDepthReset */
void H2_TauAver( void )
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo;

	if( !h2.lgH2ON )
		return;

	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/* >>chng 03 feb 14, change test from !=0 to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								/* inward optical depth */
								RTTauUpdate( &H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] , 0.5);
							}
						}
					}
				}
			}
		}
	}
}

/* this is fraction of population that is within levels done with matrix */
static double frac_matrix;

/* do level populations for H2, called by Hydrogenic after ionization and H chemistry
 * has been recomputed */
void H2_LevelPops( void )
{
	static float TeUsedBoltz=-1.f;
	static float TeUsedColl=-1.f;
	static double part_fun = 0.;
	long int loop_h2_pops;
	double rate;
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo;
	long int nColl;
	int lgConv_h2_pops;
	long int nEner,
		ipHi, ipLo;
	long int iElec , iVib , iRot, nd;
	float renorm,
		sum_pops_matrix;
	float colldown,
		collup;
	/* flags indicating whether non-positive level populations occurred
	int lgNegPop , lgZeroPop; */
	/* keep track of changes in population */
	float PopChgMax;
	long int iRotMaxChng , iVibMaxChng;

	/* H2 not on, so space not allocated */
	if( !h2.lgH2ON )
		return;

	if(lgH2_TRACE) 
		fprintf(ioQQQ,
		"\n***************H2_LevelPops call %li zone %li\n", 
		nCallH2_this_iteration,
		nzone);

	/* if H2 fraction is small then just zero out populations and cooling, and return,
	 * but, if H2 has ever been done, redo irregarless of abundance -
	 * if large H2 is ever evaluated then H2_to_H_limit is ignored */
	if( !hmi.lgBigH2_evaluated && hmi.htwo_total/dense.gas_phase[ipHYDROGEN] < H2_to_H_limit )
	{
		if(lgH2_TRACE) fprintf(ioQQQ," zero and return\n");
		/* zero everything out - loop over all possible lines */
		for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
		{
			pops_per_elec[iElecHi] = 0.;
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				pops_per_vib[iElecHi][iVibHi] = 0.;
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					/* now the lower levels */
					/* NB - X is the only lower level considered here, since we are only 
					* concerned with excited electronic levels as a photodissociation process
					* code exists to relax this assumption - simply change following to iElecHi */
					long int lim_elec_lo = 0;
					for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
					{
						/* want to include all vib states in lower level if different elec level,
						* but only lower vib levels if same elec level */
						long int nv = nVib_hi[iElecLo];
						if( iElecLo==iElecHi )
							nv = iVibHi;
						for( iVibLo=0; iVibLo<=nv; ++iVibLo )
						{
							long nr = nRot_hi[iElecLo][iVibLo];
							if( iElecLo==iElecHi && iVibHi==iVibLo )
								nr = iRotHi-1;

							for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
							{
								/* >>chng 03 feb 14, change !=0 to >0 */
								if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
								{
									/* population of lower level with correction for stim emission */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc = 0.;

									/* population of lower level */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo = 0.;

									/* population of upper level */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi = 0.;

									/* following two heat exchange excitation, deexcitation */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].cool = 0.;
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].heat = 0.;

									/* intensity of line */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].xIntensity = 0.;

									/* opacity in line */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].dTau = 0.;

									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].phots = 0.;
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ots = 0.;
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ColOvTot = 0.;
								}
							}
						}
					}
				}
			}
		}
		hmi.H2_Solomon_rate_BigH2= 0.;
		/* end of zero abundance branch */
		return;
	}

	/* check whether we need to update the Boltzmann factors, LTE level populations,
	 * and partition function.  lte level pops normalized by partition function,
	 * so sum of pops is unity */

	/* say that H2 has been computed, ignore previous limit to abund
	 * in future - this is to prevert oscillations as model is engaged */
	hmi.lgBigH2_evaluated = TRUE;

	/* do we need to update the boltzmann factors? */
	/*lint -e777 float test equality */
	if( phycon.te != TeUsedBoltz )
	/*lint +e777 float test equality */
	{
		part_fun = 0.;
		TeUsedBoltz = phycon.te;
		/* loop over all levels setting Boltzmann and deriving partition function */
		for( iElec=0; iElec<n_elec_states; ++iElec )
		{
			for( iVib=0; iVib<=nVib_hi[iElec]; ++iVib )
			{
				for( iRot=Jlowest[iElec]; iRot<=nRot_hi[iElec][iVib]; ++iRot )
				{
					Boltzmann[iElec][iVib][iRot] = 
						sexp( energy_wn[iElec][iVib][iRot] / phycon.te_wn );
					/* sum the partition funciton */
					part_fun += Boltzmann[iElec][iVib][iRot] * stat[iElec][iVib][iRot];
					ASSERT( part_fun > 0 );
				}
			}
		}
		/* have partition function, set populations_LTE */
		for( iElec=0; iElec<n_elec_states; ++iElec )
		{
			for( iVib=0; iVib<=nVib_hi[iElec]; ++iVib )
			{
				for( iRot=Jlowest[iElec]; iRot<=nRot_hi[iElec][iVib]; ++iRot )
				{
					populations_LTE[iElec][iVib][iRot] = 
						Boltzmann[iElec][iVib][iRot] * 
						stat[iElec][iVib][iRot] / part_fun;
				}
			}
		}
		if(lgH2_TRACE) 
			fprintf(ioQQQ,
			"H2 set Boltzmann factors, T=%.2f, partition function is %.2f\n",
			phycon.te,
			part_fun);
	}
	/* end loop setting Boltzmann factors, partition function, and lte populations */

	/* check whether we need to update the collision rates */
	if( fabs(1. - TeUsedColl / phycon.te ) > 0.05  )
	{
		long int numb_coll_trans = 0;
		double excit;
		double t = phycon.te/1000. + 1.;
		double t2 = POW2(t);
		TeUsedColl = phycon.te;
		iElecHi = 0;
		iElecLo = 0;
		if(lgH2_TRACE) 
			fprintf(ioQQQ,"H2 set collision rates\n");
		/* set true to print all collision rates then quit */
#		define PRT_COLL	FALSE
		/* loop over all possible bound-bound collisional changes within X 
		 * and set collision rates, which only depend on Te
		 * will go through array in energy order since coll trans do not
		 * correspond to a line */
		coll_dissoc_rate_coef[0][0] = 0.;
		for( ipHi=1; ipHi<nLevels_per_elec[0]; ++ipHi )
		{
			double energy;

			/* obtain the proper indices for the upper level */
			long int ip = ipX_ener_sort[ipHi];
			iVibHi = ipVib_energies[ip];
			iRotHi = ipRot_energies[ip];

			/* this is a guess of the collisional dissociation rate coeffient -
			 * will be multiplied by the sum of all colliders */
			energy = DissocEnergies[0] - energy_wn[0][iVibHi][iRotHi];
			ASSERT( energy > 0. );
			/* we made this up - boltzmann factor times rough coefficient */
			coll_dissoc_rate_coef[iVibHi][iRotHi] = 
				1e-14f * (float)sexp(energy/phycon.te_wn) * lgColl_dissoc_coll;
			/*fprintf(ioQQQ,"coll_dissoc_rateee\t%li\t%li\t%.3e\t%.3e\n",
				iVibHi,iRotHi,energy, coll_dissoc_rate_coef[iVibHi][iRotHi] );*/

			for( ipLo=0; ipLo<ipHi; ++ipLo )
			{
				/* these are fits to the existing collision data */
				double gbarcoll[N_X_COLLIDER][3] = 
				{   
					{-9.9265 , -0.1048 , 0.456  },
					{-8.281  , -0.1303 , 0.4931 },
					{-10.0357, -0.0243 , 0.67   },
					{-8.6213 , -0.1004 , 0.5291 },
					{-9.2719 , -0.0001 , 1.0391 } };

				ip = ipX_ener_sort[ipLo];
				iVibLo = ipVib_energies[ip];
				iRotLo = ipRot_energies[ip];

				/* do not do the very large v levels - space not even allocated */
				if( iVibLo>VIB_COLLID || iVibHi>VIB_COLLID )
					continue;

				ASSERT( energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo] > 0.);

				/* in following the colliders are H, He, H2(ortho), H2(para), and H+ */
				/* fits were read in from the following files: "H2_coll_H.dat" ,
				 * "H2_coll_He.dat" , "H2_coll_H2ortho.dat" ,"H2_coll_H2para.dat",
				 * "H2_coll_Hp.dat" */

				/* keep track of number of different collision routes */
				++numb_coll_trans;
				/* this is sum over all different colliders */
				for( nColl=0; nColl<N_X_COLLIDER-1; ++nColl )
				{
					/* this branch - real collision rate coefficients exist, use them */
					if( CollRateFit[nColl][iVibHi][iRotHi][iVibLo][iRotLo][0]!= 0 )
					{
						double r = CollRateFit[nColl][iVibHi][iRotHi][iVibLo][iRotLo][0] + 
							       CollRateFit[nColl][iVibHi][iRotHi][iVibLo][iRotLo][1]/t + 
							       CollRateFit[nColl][iVibHi][iRotHi][iVibLo][iRotLo][2]/t2;
						CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo] = 
							(float)pow(10.,r)*lgColl_deexec_Calc;
						if( PRT_COLL )
							fprintf(ioQQQ,"col fit\t%li\t%li\t%li\t%li\t%li\t%.4e\t%.4e\n",
								nColl,
								iVibHi,iRotHi,iVibLo,iRotLo,
								energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo],
								CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo] );
					}
					/* this is option to use guess of collision rate coefficient - but only if this is 
					 * a downward transition that does not mix ortho and para */
					/* turn lgColl_gbar on/off with atom h2 gbar on off */
					else if( lgColl_gbar  && 
						(lgOrtho[0][iVibHi][iRotHi]-lgOrtho[0][iVibLo][iRotLo]==0) )
					{
						/* the fit is log(K)=y_0+a*((x)^b), where K is the rate coefficient,
						* and x is the energy in wavenumbers */
						double ediff = energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo];
						/* do not let energy diff be smaller than 100 wn, the smallest
						 * diff for which we fit the rate coefficients */
						ediff = MAX2(100., ediff );
						CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo] = 
							(float)pow(10. ,
							gbarcoll[nColl][0] + gbarcoll[nColl][1] * 
							pow(ediff,gbarcoll[nColl][2]) )*lgColl_deexec_Calc;

						if( PRT_COLL )
							fprintf(ioQQQ,"col gbr\t%li\t%li\t%li\t%li\t%li\t%.4e\t%.4e\n",
								nColl+10,
								iVibHi,iRotHi,iVibLo,iRotLo,
								energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo],
								CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo] );
					}
				}
				/* last one is special, with a different fit - this is the collision of H2 with
				 * protons - of this group, cause ortho - para conversion */
				/* >>refer	H2	coll Hp	Gerlich, D., 1990, J. Chem. Phys., 92, 2377-2388 */
				if( CollRateFit[N_X_COLLIDER-1][iVibHi][iRotHi][iVibLo][iRotLo][1] != 0 )
				{
					CollRate[N_X_COLLIDER-1][iVibHi][iRotHi][iVibLo][iRotLo] = 
						CollRateFit[N_X_COLLIDER-1][iVibHi][iRotHi][iVibLo][iRotLo][0] * 1e-10f *
						/* sec fit coef was dE in milli eV */
						(float)sexp( CollRateFit[N_X_COLLIDER-1][iVibHi][iRotHi][iVibLo][iRotLo][1]/1000./phycon.te_eV)*lgColl_deexec_Calc;
					if( PRT_COLL )
						fprintf(ioQQQ,"col fit\t%i\t%li\t%li\t%li\t%li\t%.4e\t%.4e\n",
							N_X_COLLIDER-1,
							iVibHi,iRotHi,iVibLo,iRotLo,
							energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo],
							CollRate[N_X_COLLIDER-1][iVibHi][iRotHi][iVibLo][iRotLo] );
				}
				/* this is option to use guess of rate coefficient for ortho-para
				 * conversion by collision with protons */
				/* turn lgColl_gbar on/off with atom h2 gbar on off */
				else if( lgColl_gbar )
				{
					/* the fit is log(K)=y_0+a*((x)^b), where K is the rate coefficient,
					 * and x is the energy in wavenumbers */
					double ediff = energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo];
					ediff = MAX2(100., ediff );
					CollRate[N_X_COLLIDER-1][iVibHi][iRotHi][iVibLo][iRotLo] = 
						(float)pow(10. ,
						gbarcoll[N_X_COLLIDER-1][0] + gbarcoll[N_X_COLLIDER-1][1] * 
						pow(ediff ,gbarcoll[N_X_COLLIDER-1][2])	)*lgColl_deexec_Calc;

					if( PRT_COLL )
						fprintf(ioQQQ,"col gbr\t%i\t%li\t%li\t%li\t%li\t%.4e\t%.4e\n",
							N_X_COLLIDER-1+10,
							iVibHi,iRotHi,iVibLo,iRotLo,
							energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo],
							CollRate[N_X_COLLIDER-1][iVibHi][iRotHi][iVibLo][iRotLo] );
				}
				{
					/*@-redef@*/
					enum {DEBUG_LOC=FALSE};
					/*@+redef@*/
					if( DEBUG_LOC )
					{
						fprintf(ioQQQ,"bugcoll\tiVibHi\t%li\tiRotHi\t%li\tiVibLo\t%li\tiRotLo\t%li\tcoll\t%.2e\n",
							iVibHi,iRotHi,iVibLo,iRotLo,
							CollRate[N_X_COLLIDER-1][iVibHi][iRotHi][iVibLo][iRotLo] );
					}
				}
			}
		}

		/* at this stage the collision rates that came in from the large data files
		 * have been entered into the CollRate array.  Now add on three extra collision
		 * terms, the ortho para atomic H collision rates from
		 * >>>refer	H2	collision	Sun, Y., & Dalgarno, A., 1994, ApJ, 427, 1053-1056 
		 */
		nColl = 0;
		iElecHi = 0;
		iElecLo = 0;
		iVibHi = 0;
		iVibLo = 0;

		/* >>chng 02 nov 13, the sun and dalgarno rates diverge to + inf below this temp */
		if( phycon.te >= 100. )
		{
			double excit1;
			/* this is the J=1-0 downward collision rate */
			iRotLo = 0;
			iRotHi = 1;
			excit1 = sexp( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyK/phycon.te);
			excit = sexp( -(POW2(5.30-460./phycon.te)-21.2) )*1e-13;

			CollRate[0][iVibHi][iRotHi][iVibLo][iRotLo] = (float)(
				excit*H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo/
				H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi / 
				/* >>chng 02 nov 13, from 2nd to first */
				MAX2(SMALLFLOAT,excit1) )*lgColl_deexec_Calc;
				/*sexp( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyK/phycon.te))*lgColl_deexec_Calc;*/

			if( PRT_COLL )
				fprintf(ioQQQ,"col o-p\t%li\t%li\t%li\t%li\t%li\t%.4e\t%.4e\n",
					nColl,
					iVibHi,iRotHi,iVibLo,iRotLo,
					energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo],
					CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo] );

			/* this is the J=3-0 downward collision rate */
			iRotLo = 0;
			iRotHi = 3;
			excit = sexp( -(POW2(6.36-373./phycon.te)-34.5) )*1e-13;
			excit1 = sexp( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyK/phycon.te);
			CollRate[0][iVibHi][iRotHi][iVibLo][iRotLo] = (float)(
				excit*H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo/
				H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi / 
				MAX2(SMALLFLOAT,excit1) )*lgColl_deexec_Calc;

			if( PRT_COLL )
				fprintf(ioQQQ,"col o-p\t%li\t%li\t%li\t%li\t%li\t%.4e\t%.4e\n",
					nColl,
					iVibHi,iRotHi,iVibLo,iRotLo,
					energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo],
					CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo] );

			/* this is the downward J=2-1 collision rate */
			iRotLo = 1;
			iRotHi = 2;
			excit = sexp( -(POW2(5.35-454./phycon.te)-23.1 ) )*1e-13;
			excit1 = sexp( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyK/phycon.te);
			CollRate[0][iVibHi][iRotHi][iVibLo][iRotLo] = (float)(
				excit*H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo/
				H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi / 
				MAX2(SMALLFLOAT,excit1) )*lgColl_deexec_Calc;

			if( PRT_COLL )
				fprintf(ioQQQ,"col o-p\t%li\t%li\t%li\t%li\t%li\t%.4e\t%.4e\n",
					nColl,
					iVibHi,iRotHi,iVibLo,iRotLo,
					energy_wn[0][iVibHi][iRotHi] - energy_wn[0][iVibLo][iRotLo],
					CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo] );
		}
		else
		{
			CollRate[0][iVibHi][1][iVibLo][0] = 0.;
			CollRate[0][iVibHi][3][iVibLo][0] = 0.;
			CollRate[0][iVibHi][2][iVibLo][1] = 0.;
		}

		if( lgH2_TRACE )
			fprintf(ioQQQ,
			" collision rates updated for new temp, number of trans is %li\n",
			numb_coll_trans);
	}
	if( PRT_COLL )
		exit(98);
	/* end loop setting collision rates */

	/* set the populations when this is the first call to this routine on 
	 * current iteration- will use LTE populations  */
	if( nCallH2_this_iteration==0 )
	{
		/* use lte populations */
		if(lgH2_TRACE) fprintf(ioQQQ,"H2 1st call - using lte level pops\n");
		for( iElec=0; iElec<n_elec_states; ++iElec )
		{
			pops_per_elec[iElec] = 0.;
			for( iVib=0; iVib<=nVib_hi[iElec]; ++iVib )
			{
				pops_per_vib[iElec][iVib] = 0.;
				for( iRot=Jlowest[iElec]; iRot<=nRot_hi[iElec][iVib]; ++iRot )
				{
					old_populations[iElec][iVib][iRot] = 
						(float)populations_LTE[iElec][iVib][iRot]*hmi.htwo_total;
					populations[iElec][iVib][iRot] = old_populations[iElec][iVib][iRot];
				}
			}
		}
		/* first guess at ortho and para densities */
		h2.ortho_density = 0.75*hmi.htwo_total;
		h2.para_density = 0.25*hmi.htwo_total;
		/* this is the fraction of the H2 pops that are within the levels done with a matrix */
		frac_matrix = 1.;
	}
	/* end loop setting inital populations */

#	define	FRAC_OLD	(0.5f)
	/* find sum of all populations in X */
	iElec = 0;
	pops_per_elec[0] = 0.;
	for( iVib=0; iVib<=nVib_hi[iElec]; ++iVib )
	{
		pops_per_vib[0][iVib] = 0.;
		for( iRot=Jlowest[iElec]; iRot<=nRot_hi[iElec][iVib]; ++iRot )
		{
			pops_per_elec[0] += populations[iElec][iVib][iRot];
			pops_per_vib[0][iVib] += populations[iElec][iVib][iRot];
		}
	}
	/* now renorm to correct current htwo population */
	renorm = hmi.htwo_total/pops_per_elec[0];
	for( iVib=0; iVib<=nVib_hi[iElec]; ++iVib )
	{
		/* NB this loop is unlike all the others in this file since it does not include
			* the lowest upper rotation state - rest are RotHi=Jlowest */
		for( iRot=Jlowest[iElec]; iRot<=nRot_hi[iElec][iVib]; ++iRot )
		{
			populations[iElec][iVib][iRot] *= renorm;
			old_populations[iElec][iVib][iRot] = 
				FRAC_OLD * old_populations[iElec][iVib][iRot] + 
				(1.f-FRAC_OLD) * populations[iElec][iVib][iRot];
		}
	}
	if(lgH2_TRACE)
		fprintf(ioQQQ,
		" H2 entry, old pops sumed to %.3e, renorm to htwo den of %.3e\n",
		pops_per_elec[0],
		hmi.htwo_total);

	/* these are the colliders that will be considered as depopulating agents */
	/* the colliders are H, He, H2 ortho, H2 para, H+ */
	/* atomic hydrogen */
	collider_density[0] = dense.xIonDense[ipHYDROGEN][0];
	/* atomic helium */
	collider_density[1] = dense.xIonDense[ipHELIUM][0];
	/* all ortho h2 */
	collider_density[2] = (float)h2.ortho_density;
	/* all para H2 */
	collider_density[3] = (float)h2.para_density;
	/* protons - ionized hydrogen */
	collider_density[4] = dense.xIonDense[ipHYDROGEN][1];
	/* 02 oct 13 add this */
	/* assume that H3+ has precisely same effects as proton */
	collider_density[4] += hmi.Molec[ipMH3p];

	/* this is total density of all collideres, is only used for collisional dissociation */
	collider_density_total = 0.;
	for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
	{
		collider_density_total += collider_density[nColl];
	}
	/* >>chng 02 oct 24, add electrons since have roughly same rate as others
	 * for collisional dissociation */
	collider_density_total += (float)dense.eden;

	if(lgH2_TRACE)
	{
		fprintf(ioQQQ," Collider densities are:");
		for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
		{
			fprintf(ioQQQ,"\t%.3e", collider_density[nColl]);
		}
		fprintf(ioQQQ,"\n");
	}

	/* rate of entry into X from H- and formation on grain surfaces */
	/* >>refer	H2	excitation	Bertoldi, F., & Draine, B., 1996, ApJ, 468, 269, 
	 * section 6.3, 2.4 */
	/* they describe J v levels populated when H2 forms on grains.  
	 * Section 2.4 says that <v> = 5.3, <J> = 8.7, and an ortho to para ratio of 2.78.  
	 * We place 1/3.78 of newly formed H2 into J = 8, and 1  1/3.78 into J = 9.  
	 * All were placed into v = 5. */
	rate = gv.rate_h2_form_grains_used_total;

	for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
	{
		for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
		{
			H2_X_formation[iVib][iRot] = 0.;
			col_rate_in[iVib][iRot] = 0.;
		}
	}

	/* >>chng 02 oct 08, resolved grain types */
	for( nd=0; nd<gv.nBin; ++nd )
	{
		H2_type ipH2 = gv.which_H2distr[gv.bin[nd]->matType];
		for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
		{
			for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
			{
				/* >>chng 02 nov 14, changed indexing into H2_X_grain_formation_distribution and gv.bin, PvH */
				H2_X_formation[iVib][iRot] += 
					H2_X_grain_formation_distribution[ipH2][iVib][iRot] * 
					/* units of following are s-1 */
					(float)gv.bin[nd]->rate_h2_form_grains_used;
			}
		}
	}

	/* formation of H2 in excited states from H- H minus */
	/* >>chng 02 oct 17, temp dependent fit to rate, updated reference,
	 * about 40% larger than before */
	rate = hmi.Molec[ipMHm]*hmi.assoc_detach;
	/*rate = hmi.hminus*1.35e-9f;*/
	/* place 50-50 into 0,0 and 0,2, according to
	 * >>refer	H2	excitation	Abel, T., Anninos, P., Zhang, Y., & Norman, M., 1997, 
	 * >>refercon	New Astronomy, 2, 181-207*/
	/* convert to dimensionless factors that add to unity */
	/* >>chng 02 oct 17, use proper distribution function */
	for( iVib=0; iVib<=nVib_hi[0]; ++iVib )
	{
		for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
		{
			int i=1;
			for(i=0; i<nTE_HMINUS; ++i )
			{
				H2_X_formation[iVib][iRot] +=
					(float)(H2_X_hminus_formation_distribution[i][iVib][iRot] * rate);
			}
		}
	}

	/* >>chng 03 feb 10, add this population process */
	/* H2+ + H => H2 + H+,
	 * >>refer	H2	population	Krstic, P.S., preprint 
	 * all goes into v=4 but no J information, assume into J = 0 */
	rate = hmi.bh2h2p * hmi.Molec[ipMH2p];
	iVib = 4;
	iRot = 0;
	H2_X_formation[iVib][iRot] += (float)rate;

	/* this flag will say whether H2 populations have converged,
	 * by comparing old and new values */
	lgConv_h2_pops = FALSE;
	/* this will count number of passes around following loop */
	loop_h2_pops = 0;
	/*TODO what happens if this loop does not include excited elec states? */
	while( loop_h2_pops < 10 && !lgConv_h2_pops )
	{
		float rate_in , rate_out;
		++loop_h2_pops;
		/* loop over all possible pumping routes to excited electronic states
		* to get radiative excitation and dissociation rates */
		for( iElecHi=1; iElecHi<n_elec_states; ++iElecHi )
		{
			/* this will be total population in each electronic state */
			pops_per_elec[iElecHi] = 0.;

			if(lgH2_TRACE) fprintf(ioQQQ," Pop(e=%li):",iElecHi);
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				pops_per_vib[iElecHi][iVibHi] = 0.;
				/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					/* sum of all rates into and out of these upper levels */
					rate_in = 0.;
					/* this term is spontaneous dissociation of excited elec states into 
					* the X continuum */
					rate_out = dissprob[iElecHi][iVibHi][iRotHi];

					/* now loop over all levels within X */
					iElecLo=0;
					for( iVibLo=0; iVibLo<=nVib_hi[iElecLo]; ++iVibLo )
					{
						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nRot_hi[iElecLo][iVibLo]; ++iRotLo )
						{
							/* >>chng 03 feb 14, change test from !=0 to > 0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								/* this is the rate [cm-3 s-1] electrons move into the upper level */
								rate_in +=
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].pump *
									old_populations[iElecLo][iVibLo][iRotLo];

								/* this is the rate [s-1] electrons move out of the upper level */
								rate_out +=
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul*
									/* escape and destruction */
									(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pesc + 
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pelec_esc + 
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pdest +
									 /* induced emission down */
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].pump *
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo/
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi);

								ASSERT( (rate_in >=0.) && rate_out >= 0. );
							}
						}
					}

					/*  get the population [cm-3] */
					populations[iElecHi][iVibHi][iRotHi] = rate_in /
						MAX2( SMALLFLOAT , rate_out);
					ASSERT( populations[iElecHi][iVibHi][iRotHi] >= 0. && 
						populations[iElecHi][iVibHi][iRotHi] <= hmi.htwo_total );

					/* this is partial sum - total pop in this vib state */
					pops_per_vib[iElecHi][iVibHi] += populations[iElecHi][iVibHi][iRotHi];

					/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
					/*fprintf(ioQQQ,"rot %li in %.2e out %.2e rel pop %.2e\n",
						iRotHi, rate_in ,
						rate_out,
						populations[iElecHi][iVibHi][iRotHi] );*/
				}
				/* end excit vib pops loop */
				if(lgH2_TRACE) fprintf(ioQQQ,"\t%.2e",pops_per_vib[iElecHi][iVibHi]/hmi.htwo_total);

				/* total pop in each elec state */
				pops_per_elec[iElecHi] += pops_per_vib[iElecHi][iVibHi];
			}
			/* end excited elec pops loop */
			if(lgH2_TRACE) fprintf(ioQQQ,"\n");
		}

		/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
		/* now do lowest levels populations with matrix, 
		 * these should be collisionally dominated */
		/*H2_Level_lowJ evaluate CO rotation cooling */
		H2_Level_lowJ(
			/* the number of rotation levels */
			matrix_rot ,
			/* the number of vibration levels */
			matrix_vib ,
			/* the total abundance - frac_matrix is fraction of pop that was in these
			* levels the last time this was done */
			hmi.htwo_total * (float)frac_matrix );

		/* above set pops of excited electronic levels and low levels within X - 
		* now do something about excited levels within the X state */
		/* nLevels_per_elec is number of levels within elec 0 - so nEner is one
		* beyond end of array here - but will be decremented at start of loop */
		nEner = nLevels_per_elec[0];
		iElec = 0;

		/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
		/* this is the main loop that determines populations within large J states in X */
		while( (--nEner) >= matrix_rot )
		{
			long int ip = ipX_ener_sort[nEner];
			iVib = ipVib_energies[ip];
			iRot = ipRot_energies[ip];

			/* do not update populations if they were done in matrix */
			if( (iVib < matrix_vib) && iRot<matrix_rot )
				continue;

			/* count formation from grains and H- as a collisional formation process */
			col_rate_in[iVib][iRot] = H2_X_formation[iVib][iRot];
			rad_rate_in[iVib][iRot] = 0.;
			/* this represents collisional dissociation into continuum of X,
			 * rates are just guesses */
			col_rate_out[iVib][iRot] = collider_density_total *
				coll_dissoc_rate_coef[iVib][iRot];

			/* >>chng 02 oct 18 add this */
			/* this is photodissociation into 2H */
			rad_rate_out[iVib][iRot] = rfield.flux_accum[ipPhoto[iVib][iRot]-1]*0.25e-18f;

			/* 02 oct 13, add this */
			/* ortho para conversion, and deexcitation, while on grain surface,
			 * H2 + grain => H2 + grain, do not want to add rate to 0,0 */
			if( iVib!= 0 || iRot!=0 )
			{
				/* this is the ground v J = 1->0 transition */
				if( iVib == 0 && iRot == 1 )
				{
					col_rate_out[iVib][iRot] += 
						/* H2 grain interactions
						* H2 ortho - para conversion on grain surface,
						* rate (s-1) all v,J levels go to 0 or 1, preserving nuclear spin */
						(float)hmi.rate_h2_allX_2_J0_grains;
				}
				/* add both rates if not 0,0 or 0,1 */
				else if( iVib >0  || iRot > 1 )
				{
					col_rate_out[iVib][iRot] += 
						/* H2 grain interactions
						* H2 ortho - para conversion on grain surface,
						* rate (s-1) all v,J levels go to 0 or 1, preserving nuclear spin */
						(float)hmi.rate_h2_allX_2_J0_grains + 
						/* rate (s-1) all v,J levels go to 0, regardless of nuclear spin */
						(float)hmi.rate_h2_ortho_para_conserve;

				}
			}

			/* >>chng 03 jan 22, add excitation due to cosmic rays,
			 * csupra is total ionization rate */
			if( iVib < CR_VIB )
			{
				/* cr_rate[CR_X][CR_VIB][CR_J][CR_EXIT];*/
				if( iRot> 1 && iRot < CR_J )
				{
					/* collision J to J-2 */
					col_rate_out[iVib][iRot] += 
						Secondaries.csupra * cr_rate[0][iVib][iRot][0];
					col_rate_in[iVib][iRot-2] += old_populations[0][iVib][iRot]*
						Secondaries.csupra * cr_rate[0][iVib][iRot][0];
				}

				if( iRot<CR_J-2 && iRot+2 <= nRot_hi[iElec][iVib])
				{
					col_rate_out[iVib][iRot+2] += 
						Secondaries.csupra * cr_rate[0][iVib][iRot][2];
					col_rate_in[iVib][iRot+2] += old_populations[0][iVib][iRot]*
						Secondaries.csupra * cr_rate[0][iVib][iRot][2];
				}
			}

			/* this sum is for X going into all electronic excited states */
			for( iElecHi=1; iElecHi<n_elec_states; ++iElecHi )
			{
				for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
				{
					for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
					{
						/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
						/* the rate electrons enter this state from excited elec states */
						/* >>chng 03 feb 14, change test from != 0 to > 0 */
						if( H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Aul > 0. )
						{
							float rateone;
							/* the spontaneous rate down, escape and destruction, this
							 * ignores stilulated emission - very good approximation */
							rateone =
								old_populations[iElecHi][iVibHi][iRotHi]*
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Aul*
								/* escape and destruction */
								(H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Pesc + 
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Pelec_esc + 
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Pdest + 
								/* induced emission down */
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].pump *
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].gLo/
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].gHi);
							ASSERT( rateone >= 0. );
							rad_rate_in[iVib][iRot] += rateone;

							/* pump up */
							rad_rate_out[iVib][iRot] +=
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].pump;
							{
								/*@-redef@*/
								enum {DEBUG_LOC=FALSE};
								/*@+redef@*/
								if( DEBUG_LOC && iRot==1 && iVib==0 )
								{
									fprintf(ioQQQ,"bug E vibhi\t%li\trothi\t%li\trate_in\t%.2e\tAul\t%.2e\tpop\t%.2e\n",
										iVibHi,iRotHi, rateone ,
										H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Aul,
										old_populations[iElecHi][iVibHi][iRotHi] );
								}
							}

							/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
						}
					}
				}
			}

			{
				/*@-redef@*/
				enum {DEBUG_LOC=FALSE};
				/*@+redef@*/
				/* this debug gives total rates in and out after the excited elec states
				 * have been dealt with */
				if( DEBUG_LOC && iVib == 4 && iRot==0 )
				{
					fprintf(ioQQQ,"eleccc vib\t%li\trot\t%li\tin\t%.2e\tout\t%.2e\trel pop\t%.2e\n",
						iVib,iRot, rad_rate_in[iVib][iRot] ,
						rad_rate_out[iVib][iRot],
						old_populations[iElec][iVib][iRot] );
				}
			}

			/* now sum over states within X which are higher than current state */
			iElecHi = 0;
			for( ipHi = nEner+1; ipHi<nLevels_per_elec[0]; ++ipHi )
			{
				ip = ipX_ener_sort[ipHi];
				iVibHi = ipVib_energies[ip];
				iRotHi = ipRot_energies[ip];
				/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
				/* the rate electrons enter this state from excited states within X
				 * by radiative decays, which have delta J = 0 or 2 */
				/* note test on vib is needed - iVibHi<iVib, energy order ok and space not allocated */
				if( (fabs(iRotHi-iRot)==2 || iRotHi==iRot )&& (iVib<=iVibHi) )
				/* >>chng 03 feb 14, change to test on Aul */
				/*if( H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Aul>0. )*/
				{
					float rateone;
					rateone =
						old_populations[iElecHi][iVibHi][iRotHi]*
						 H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Aul*
						(H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Pesc + 
						 H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Pelec_esc + 
						 H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Pdest);
					if( rateone <0 )
					{
						fprintf(ioQQQ,"hit\n");
						rateone = 0.;
					}
					/*ASSERT( rateone >= 0. );*/
					rad_rate_in[iVib][iRot] += rateone;

					{
						/*@-redef@*/
						enum {DEBUG_LOC=FALSE};
						/*@+redef@*/
						if( DEBUG_LOC && iRot==7 && iVib==3 )
						{
							fprintf(ioQQQ,"bug X vibhi\t%li\trothi\t%li\trate_in\t%.2e\tAul\t%.2e\tpop\t%.2e\tsum\n",
								iVibHi,iRotHi, rateone ,
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Aul,
								old_populations[iElecHi][iVibHi][iRotHi] );
						}
					}
				}
				/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
				/* collisional interactions with upper levels within X */
				/* VIB_COLLID is the highest vib level that has coll data */
				if( iVibHi <= VIB_COLLID && iVib <= VIB_COLLID)
				{
					colldown = 0.;
					for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
					{
						/* downward collision rate, units s-1 */
						float colldown1 = 
							CollRate[nColl][iVibHi][iRotHi][iVib][iRot]*collider_density[nColl];
						colldown += colldown1;
#						if 0
						if( iRotHi==3 && iVibHi==0 && iRot==1 && iVib==0 )
						{
							fprintf(ioQQQ,
								"bug C1 vibhi\t%li\trothi\t%li\trate_dn\t%.2e\tpopu\t%.2e\trate\t%.2e\tden%.2e\n",
								iVibHi,iRotHi, colldown1 ,
								old_populations[iElecHi][iVibHi][iRotHi],
								CollRate[nColl][iVibHi][iRotHi][iVib][iRot],
								collider_density[nColl]);
						}
#						endif
					}
					/* rate in from upper level, units cm-3 s-1 */
					col_rate_in[iVib][iRot] += old_populations[iElecHi][iVibHi][iRotHi]*colldown;
					/* convert real rate back into electron cs in case other parts of code
					 * need this for reference */
					if( iVibHi >= iVib )
					{
						H2Lines[iElec][iVibHi][iRotHi][iElec][iVib][iRot].ColUL = colldown;
						LineConvRate2CS( &H2Lines[iElec][iVibHi][iRotHi][iElec][iVib][iRot] );
					}
					/* ortho para conversion on grain surfaces, also ortho-ortho and para-para */
					/* 02 oct 13, add this */
					/*TODO is this right - two rates that add together?  or is
					 * J=1 to 0 only added for 0,1 to 0,0? */
					if( (iVib ==0) && (iRot <=1) &&
						(lgOrtho[0][iVibHi][iRotHi]-lgOrtho[0][iVib][iRot])==0 )
					{
						/* this is same nuclear spin */
						col_rate_in[iVib][iRot] += old_populations[iElecHi][iVibHi][iRotHi]*
							(float)hmi.rate_h2_ortho_para_conserve;
					}
					else if( iVib==0 && iRot== 0 )
					{
						/* this is everything down to v=0 && J=0 */
						col_rate_in[iVib][iRot] += old_populations[iElecHi][iVibHi][iRotHi]*
							(float)hmi.rate_h2_allX_2_J0_grains;
					}

					/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
					/* upward collisions out of this level - probably very small
					 * since dE >> kT - Boltzmann is e(- nu/kT ) */
					collup = (float)(colldown *	
						stat[iElecHi][iVibHi][iRotHi] / stat[iElec][iVib][iRot] *
						Boltzmann[iElecHi][iVibHi][iRotHi] /
						MAX2(SMALLFLOAT , Boltzmann[iElec][iVib][iRot] ) );
					col_rate_out[iVib][iRot] += collup;
					{
						/*@-redef@*/
						enum {DEBUG_LOC=FALSE};
						/*@+redef@*/
						if( DEBUG_LOC && iRot==1 && iVib==0 && iRotHi==3 && iVibHi==0 )
						{
							fprintf(ioQQQ,"bug C1 vibhi\t%li\trothi\t%li\trate_in\t%.2e\tpop\t%.2e\tsum\t%.2e\n",
								iVibHi,iRotHi, colldown , collup ,
								col_rate_in[iVib][iRot] );
						}
						if( DEBUG_LOC && iRotHi==1 && iVibHi==0 && iRot==0 && iVib==0 )
						{
							fprintf(ioQQQ,"bug C1 vibhi\t%li\trothi\t%li\trate_in\t%.2e\tpop\t%.2e\tsum\t%.2e\n",
								iVibHi,iRotHi, 
								colldown , 
								collup ,
								col_rate_in[iVib][iRot] );
						}
					}
				}

				/* this branch no coll data, but will make some up on the fly,
				* buy only if g-bar is on and not spin changing */
				/* turn lgColl_gbar on/off with atom h2 gbar on off */
				else if( lgColl_gbar && 
					(lgOrtho[0][iVibHi][iRotHi]-lgOrtho[0][iVib][iRot]==0) )
				{
					/* "typical" rates for H, He, H2ortho, H2para, H+ */
					float csguess[N_X_COLLIDER]={1e-15f,1e-17f,1e-18f,1e-18f,3e-10f};

					colldown = 0.;
					for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
					{
						colldown +=
							csguess[nColl]*collider_density[nColl];
					}
					col_rate_in[iVib][iRot] += old_populations[iElecHi][iVibHi][iRotHi]*colldown;
					{
						/*@-redef@*/
						enum {DEBUG_LOC=FALSE};
						/*@+redef@*/
						if( DEBUG_LOC && iRot==1 && iVib==0 )
						{
							fprintf(ioQQQ,"bug Cf vibhi\t%li\trothi\t%li\trate_in\t%.2e\tpop\t%.2e\tsum\t%.2e\n",
								iVibHi,iRotHi, colldown ,
								old_populations[iElecHi][iVibHi][iRotHi] ,
								col_rate_in[iVib][iRot]);
						}
					}
				}
			}
			/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
			{
				/*@-redef@*/
				enum {DEBUG_LOC=FALSE};
				/*@+redef@*/
				/* this debug gives total rates after interactions with 
				 * higher levels within X have been included */
				if( DEBUG_LOC&&iRot==1 && iVib==0 )
				{
					fprintf(ioQQQ,"aboveee vib\t%li\trot\t%li\tin\t%.2e\tout\t%.2e\trel pop\t%.2e\n",
						iVib,iRot, col_rate_in[iVib][iRot] ,
						col_rate_out[iVib][iRot],
						old_populations[iElec][iVib][iRot]/hmi.htwo_total );
				}
			}

			/* we now have total rate this state is populated from above, now get rate
			 * this state intereacts with levels that are below */
			iElecLo = 0;
			for( ipLo = 0; ipLo<nEner; ++ipLo )
			{
				ip = ipX_ener_sort[ipLo];
				iVibLo = ipVib_energies[ip];
				iRotLo = ipRot_energies[ip];
				/* radiative interactions between this level and lower levels */
				/* the test on vib is needed - the energies are ok but the space does not exist */
				if( ((fabs(iRotLo-iRot) == 2)||(fabs(iRotLo-iRot) == 0))  && (iVibLo<=iVib)/**/)
				{
					rad_rate_out[iVib][iRot] +=
						H2Lines[iElec][iVib][iRot][iElecLo][iVibLo][iRotLo].Aul*
						(H2Lines[iElec][iVib][iRot][iElecLo][iVibLo][iRotLo].Pesc + 
						H2Lines[iElec][iVib][iRot][iElecLo][iVibLo][iRotLo].Pelec_esc + 
						H2Lines[iElec][iVib][iRot][iElecLo][iVibLo][iRotLo].Pdest);
				}
				/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
				/* collisions to level below us - 
				* VIB_COLLID is the highest vib level that has coll data */
				if( iVib <= VIB_COLLID && iVibLo <= VIB_COLLID)
				{
					colldown = 0.;
					for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
					{
						colldown +=
							CollRate[nColl][iVib][iRot][iVibLo][iRotLo]*collider_density[nColl];
					}
					col_rate_out[iVib][iRot] += colldown;
					/* rate up into this level from below */
					collup = (float)(colldown* old_populations[iElecLo][iVibLo][iRotLo] *	
						stat[iElec][iVib][iRot] / stat[iElecLo][iVibLo][iRotLo] *
						Boltzmann[iElec][iVib][iRot] /
						MAX2(SMALLFLOAT , Boltzmann[iElecLo][iVibLo][iRotLo] ));
					col_rate_in[iVib][iRot] += collup ;

					{
						/*@-redef@*/
						enum {DEBUG_LOC=FALSE};
						/*@+redef@*/
						if( DEBUG_LOC && iRot==3 && iVib==0 && iRotLo==1 && iVibLo==0 )
						{
							fprintf(ioQQQ,"bug C2 vibhi\t%li\trothi\t%li\trate_in\t%.2e\tpop\t%.2e\tsum\t%.2e\n",
								iVib,iRot, colldown , collup ,
								col_rate_in[iVib][iRot] );
						}
						/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
						if( DEBUG_LOC && iRot==1 && iVib==0 && iRotLo==0 && iVibLo==0 )
						{
							fprintf(ioQQQ,"bug C2 vibhi\t%li\trothi\t%li\trate_in\t%.2e\tpop\t%.2e\tsum\t%.2e\n",
								iVib,iRot, 
								colldown , 
								collup ,
								col_rate_in[iVib][iRot] );
						}
					}
				}
				/* >>chng 02 jul 29, add wild guess if lower level is close but delta v is large */
				/* turn lgColl_gbar on/off with atom h2 gbar on off */
				/*TODO - did all gbar coll include test on spin changing?  Also, this
				 * does not have the back collision term */
				else if( lgColl_gbar && 
					(lgOrtho[0][iVib][iRot]-lgOrtho[0][iVibLo][iRotLo]==0) &&
					((energy_wn[iElec][iVib][iRot]-energy_wn[0][iVibLo][iRotLo])<2.*phycon.te)
					)
				{
					/* "typical" rates for H, He, H2ortho, H2para, H+ */
					float csguess[N_X_COLLIDER]={1e-15f,1e-17f,1e-18f,1e-18f,3e-10f};

					colldown = 0.;
					for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
					{
						colldown +=
							csguess[nColl]*collider_density[nColl];
					}
					col_rate_out[iVib][iRot] += colldown;
					/* rate up into this level from below */
					collup = (float)(colldown* old_populations[iElecLo][iVibLo][iRotLo] *	
						stat[iElec][iVib][iRot] / stat[iElecLo][iVibLo][iRotLo] *
						Boltzmann[iElec][iVib][iRot] /
						MAX2(SMALLFLOAT , Boltzmann[iElecLo][iVibLo][iRotLo] ));
					col_rate_in[iVib][iRot] += collup ;
				}
			}
			/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/

			/* we now have the total rates into and out of this level, get its population */
			if( (col_rate_out[iVib][iRot]+rad_rate_out[iVib][iRot])> SMALLFLOAT )
			{
				populations[iElec][iVib][iRot] = 
					(col_rate_in[iVib][iRot]+ rad_rate_in[iVib][iRot]) / 
					(col_rate_out[iVib][iRot]+rad_rate_out[iVib][iRot]) ;
#				if 0
				if( iVib==0 && iRot==0 )
					fprintf(ioQQQ," bugggpop\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\n",

						populations[iElec][iVib][iRot]/hmi.htwo_total ,

						(col_rate_in[iVib][iRot]+rad_rate_in[iVib][iRot])/
						(col_rate_out[iVib][iRot]+rad_rate_out[iVib][iRot])/hmi.htwo_total ,

						(col_rate_in[iVib][iRot]+rad_rate_in[iVib][iRot]),
						col_rate_out[iVib][iRot]+rad_rate_out[iVib][iRot] ,

						populations[iElec][iVib][1]/hmi.htwo_total ,
						(col_rate_in[iVib][1]+rad_rate_in[iVib][1])/
						(col_rate_out[iVib][1]+rad_rate_out[iVib][1])/hmi.htwo_total ,

						(col_rate_in[iVib][1]+rad_rate_in[iVib][1]),
						col_rate_out[iVib][1]+rad_rate_out[iVib][1] );
#				endif

			}
			else
			{
				populations[iElec][iVib][iRot] = 0.;
			}
			{
				/*@-redef@*/
				enum {DEBUG_LOC=FALSE};
				/*@+redef@*/
				/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
				if( DEBUG_LOC && iRot==1 && iVib==0 )
				{
					fprintf(ioQQQ,"debugggg vib\t%li\trot\t%li\tin\t%.2e\tout\t%.2e\trel pop\t%.2e\n",
						iVib,iRot, col_rate_in[iVib][iRot] ,
						col_rate_out[iVib][iRot],
						populations[iElec][iVib][iRot]/hmi.htwo_total );
				}
			}

			ASSERT( populations[iElec][iVib][iRot] >= 0.  );
		}
		/* find ortho and para densites, sum of pops in each vib */
		iElecHi = 0;
		if(lgH2_TRACE) 
		{
			fprintf(ioQQQ," Pop(e=%li):",iElecHi);
		}

		/* this will become total pop is X, which will be renormed to equal hmi.htwo_total */
		pops_per_elec[0] = 0.;
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			double sumv;
			sumv = 0.;
			pops_per_vib[0][iVibHi] = 0.;

			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				pops_per_elec[0] += populations[iElecHi][iVibHi][iRotHi];
				sumv += populations[iElecHi][iVibHi][iRotHi];
				pops_per_vib[0][iVibHi] += populations[iElecHi][iVibHi][iRotHi];
			}
			/* print sum of populations in each vib if trace on */
			if(lgH2_TRACE) fprintf(ioQQQ,"\t%.2e",sumv/hmi.htwo_total);
		}
		/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
		if(lgH2_TRACE) 
		{
			fprintf(ioQQQ,"\n");
			/* print the ground vib state */
			fprintf(ioQQQ," Pop(0,J):");
			iElecHi = 0;
			iVibHi = 0;
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				fprintf(ioQQQ,"\t%.2e",populations[iElecHi][iVibHi][iRotHi]/hmi.htwo_total);
			}
			fprintf(ioQQQ,"\n");
		}

		/* now find population in states done with matrix - this is only used to pass
		* to matrix solver */
		iElec = 0;
		sum_pops_matrix = 0.;
		for( iVib=0; iVib<matrix_vib; ++iVib )
		{
			for( iRot=0; iRot<matrix_rot; ++iRot )
			{
				sum_pops_matrix += populations[iElec][iVib][iRot];
			}
		}
		/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
		/* this is self consistent since pops_per_elec[0] came from current soln,
		* as did the matrix.  pops will be renomalized by results from the chemistry
		* a few lines down */
		frac_matrix = sum_pops_matrix / pops_per_elec[0];

		/* assuming that all H2 population is in X, this is the
		* ratio of H2 that came out of the chemistry network to what we just obtained -
		* we need to multiply the pops by renorm to agree with the chemistry */
		renorm = hmi.htwo_total/pops_per_elec[0];
		pops_per_elec[0] = hmi.htwo_total;

		{
			/*@-redef@*/
			enum {DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC )
			{
				iElec = 0;
				iVib = 0;
				fprintf(ioQQQ,"loop pops");
				for(iRot=0; iRot<3; ++iRot )
				{
					fprintf(ioQQQ,"\t%.3e\t%.3e\t%.3e\t%.3e", 
						populations[iElec][iVib][iRot]/hmi.htwo_total,

						(col_rate_in[iVib][iRot]+rad_rate_in[iVib][iRot])/
						MAX2(SMALLFLOAT,col_rate_out[iVib][iRot]+rad_rate_out[iVib][iRot])/hmi.htwo_total,

						(col_rate_in[iVib][iRot]+rad_rate_in[iVib][iRot]),
						MAX2(SMALLFLOAT,col_rate_out[iVib][iRot]+rad_rate_out[iVib][iRot]) );
				}
				fprintf(ioQQQ,"\n");
			}
		}

		/* at the moment nothing will be done with these */
		/*lgNegPop = FALSE;
		lgZeroPop = FALSE;*/
		/* renormalize populations, then sum ortho and para */
		for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
		{
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					/* make sure population is sane, then renormalize it */
					if( populations[iElecHi][iVibHi][iRotHi]<0. )
					{
						/*lgNegPop = TRUE;*/
						/* only print flag if not first call */
						/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
						if( conv.nPres2Ioniz>0)
							fprintf(ioQQQ,
								"PROBLEM: negative H2 level population, E=%li v=%li J=%li pop=%.2e H2=%.2e\n",
								iElecHi , iVibHi , iRotHi , 
								populations[iElecHi][iVibHi][iRotHi] ,
								hmi.htwo_total );
						populations[iElecHi][iVibHi][iRotHi] = 0.;
					}
					else if( populations[iElecHi][iVibHi][iRotHi]== 0. )
					{
						/*fprintf(ioQQQ,
							"PROBLEM: zero H2 level population, E=%li v=%li J=%li H2=%.2e\n",
							iElecHi , iVibHi , iRotHi , 
							hmi.htwo_total );*/
						/*lgZeroPop = TRUE;*/
					}
					else
					{
						populations[iElecHi][iVibHi][iRotHi] *= renorm;
					}
					/* keep density of ortho and para separate */
					/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
#					if 0
					if( lgOrtho[iElecHi][iVibHi][iRotHi] )
					{
						h2.ortho_density += populations[iElecHi][iVibHi][iRotHi];
					}
					else
					{
						h2.para_density += populations[iElecHi][iVibHi][iRotHi];
					}

		/* update densities used for collision rates */
		/* all ortho h2 */
		collider_density[2] = (float)h2.ortho_density;
		/* all para H2 */
		collider_density[3] = (float)h2.para_density;
		/* end find ortho and para densites */
#					endif
				}
			}
		}

		/* convergence check */
		PopChgMax = 0.;
		iElecHi = 0;
		iRotMaxChng =-1;
		iVibMaxChng = -1;
		/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
		for( iVibHi=0; iVibHi<MIN2(10,nVib_hi[iElecHi]); ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<MIN2(10,nRot_hi[iElecHi][iVibHi]); ++iRotHi )
			{
				/* keep track of largest relative change in populations */
				if( fabs(populations[iElecHi][iVibHi][iRotHi] - 
					old_populations[iElecHi][iVibHi][iRotHi])/
					/* on first call some very high J states can have zero pop */
					MAX2(SMALLFLOAT,populations[iElecHi][iVibHi][iRotHi]) > fabs(PopChgMax) &&
					populations[iElecHi][iVibHi][iRotHi]>SMALLFLOAT )
				{
					PopChgMax = (float)(
						(populations[iElecHi][iVibHi][iRotHi] - 
						old_populations[iElecHi][iVibHi][iRotHi])/
						populations[iElecHi][iVibHi][iRotHi]);
					iRotMaxChng = iRotHi;
					iVibMaxChng = iVibHi;
				}
			}
		}
		/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
		/* is change too large? */
		if( fabs(PopChgMax)>0.1 || 
			(fabs(hmi.HeatH2Dexc_BigH2)/thermal.ctot > 0.1 && fabs(PopChgMax)>0.03) )
		{
			/* this branch 
			 * molecular populations are not converged */
			conv.lgConvIoniz = FALSE;
			lgConv_h2_pops = FALSE;
			strcpy( conv.chConvIoniz, "H2 pop cnv" );
		}
		else
		{
			lgConv_h2_pops = TRUE;
		}
		/* end convergence criteria */

		/* remember largest and smallest chemistry renorm factor -
		 * if both networks are parallel will be unity */
		h2.renorm_max = MAX2( renorm , h2.renorm_max );
		h2.renorm_min = MIN2( renorm , h2.renorm_min );

		if(lgH2_TRACE) 
		{
			fprintf(ioQQQ,
			"H2 chem renorm fac: %.4e, ortho/para ratio: %.3e, frac of pop in matrix: %.3f\n",
			renorm,
			h2.ortho_density / h2.para_density ,
			frac_matrix);

			/* =======================INSIDE POPULATIONS CONVERGE LOOP =====================*/
			fprintf(ioQQQ,
			"end loop %li H2 max rel chng=%.3e from %.3e to %.3e at v=%li J=%li\n\n",
			loop_h2_pops,
			PopChgMax , 
			old_populations[0][iVibMaxChng][iRotMaxChng],
			populations[0][iVibMaxChng][iRotMaxChng],
			iVibMaxChng , iRotMaxChng
			);
		}

		/* now save this solution to use for next old pop */
		for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
		{
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				/* NB this loop is unlike all the others in this file since it does not include
				 * the lowest upper rotation state - rest are RotHi=Jlowest */
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					old_populations[iElecHi][iVibHi][iRotHi] = 
						FRAC_OLD*old_populations[iElecHi][iVibHi][iRotHi] +
						(1.f-FRAC_OLD)*populations[iElecHi][iVibHi][iRotHi];
				}
			}
		}
	}
	/* =======================END POPULATIONS CONVERGE LOOP =====================*/

	h2.ortho_density = 0.;
	h2.para_density = 0.;
	/* loop over all possible lines and set populations, and quantities that depend on escape prob, dest, etc */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				long int lim_elec_lo = 0;
				/* find total ortho and para densities */
				if( lgOrtho[iElecHi][iVibHi][iRotHi] )
				{
					h2.ortho_density += populations[iElecHi][iVibHi][iRotHi];
				}
				else
				{
					h2.para_density += populations[iElecHi][iVibHi][iRotHi];
				}
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi = 
								populations[iElecHi][iVibHi][iRotHi]; 
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo = 
								populations[iElecLo][iVibLo][iRotLo]; 
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopOpc = 
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo - 
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi*
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gLo / 
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].gHi;
							ASSERT(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi >= 0. &&
							H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopLo >=  0.);

							/* >>chng 03 feb 14, from !=0 to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								/* following two heat exchange excitation, deexcitation */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].cool = 0.;
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].heat = 0.;

								/* number of photons in the line */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].phots = 
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul * 
									(H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pesc + 
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pelec_esc) * 
									 H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi; 

								/* intensity of line */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].xIntensity = 
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].phots *
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].EnergyErg;

								/* ots destruction rate */
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ots = (float)(
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul *
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].PopHi *
								H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Pdest);

								if( iElecHi==0 )
								{
									/*TODO - put H2Lines in outward beams in RTDiffuse */
									/* the ground electronic state, most excitations are not direct pumping 
									 * (rather indirect, which does not count for ColOvTot) */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ColOvTot = 1.;
									/* dump the ots rate into the stack - but only for ground elec state*/
									RT_OTS_AddLine(
										H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ots,
										H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ipCont );
								}
								else
								{
									/* these are excited electronic states, entirely pumped */
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].ColOvTot = 0.;
								}
							}
						}
					}
				}
			}
		}
	}	

	/* this is total rate (s-1) h2 dissoc into X continuum by Solomon process,
	 * sum over all excited elec states */
	hmi.H2_Solomon_rate_BigH2 = 0.;
	for( iElec=1; iElec<n_elec_states; ++iElec )
	{
		for( iVib=0; iVib<=nVib_hi[iElec]; ++iVib )
		{
			for( iRot=Jlowest[iElec]; iRot<=nRot_hi[iElec][iVib]; ++iRot )
			{
				/* this is the total rate of dissociation of excited elec states into 
				* the X continuum */
				/* at this point the units are cm-3 s-1 */
				hmi.H2_Solomon_rate_BigH2 += 
					populations[iElec][iVib][iRot]*dissprob[iElec][iVib][iRot];

			}
		}
	}

	/* add up H2 + hnu => 2H, continuum photodissociation,
	 * this is not the Solomon process, true continuum */
	hmi.H2_photodissoc_BigH2 = 0.;
	iElec = 0;
	for( iVib=0; iVib<=nVib_hi[iElec]; ++iVib )
	{
		for( iRot=Jlowest[iElec]; iRot<=nRot_hi[iElec][iVib]; ++iRot )
		{
			/* this is the total rate of direct photo-dissociation of excited elec states into 
			* the X continuum - this is continuum photodissociation, not the Solomon process */
			hmi.H2_photodissoc_BigH2 += 
				populations[iElec][iVib][iRot]*rfield.flux_accum[ipPhoto[iVib][iRot]-1]*0.25e-18f;
		}
	}
	/* at this point units of hmi.H2_Solomon_rate_BigH2 are cm-3 s-1 
	 * since populations are included -
	 * div by pops to get actual dissocation rate, s-1 */
	if( hmi.htwo_total > SMALLFLOAT )
	{
		hmi.H2_Solomon_rate_BigH2 /= hmi.htwo_total;
		hmi.H2_photodissoc_BigH2 /= hmi.htwo_total;
	}
	else
	{
		hmi.H2_Solomon_rate_BigH2 = 0.;
		hmi.H2_photodissoc_BigH2 = 0.;
	}

	/* update number of times we have been called */
	++nCallH2_this_iteration;

	/* this will say how many times the large H2 molecule has been called in this zone -
	 * if not called (due to low H2 abundance) then not need to update its line arrays */
	++h2.nCallH2_this_zone;

	return;
}

/* add in explicit lines from the large H2 molecule, called by lines_molecules */
void H2_LinesAdd(void)
{
	/* these are the quantum designations of the lines we will output */
#	if 0
#	define N_H2_LINES_OUTPUT	1
	long int ip_H2_lines_output[N_H2_LINES_OUTPUT*6] = {
	/* R branch - J decreases by 1 in emission line */
	/* P branch - J increases by 1 in emission line */
	/* Q branch - J changes   by 0 in emission line */
	/* S branch - J decreases by 2 in emission line */
	/*0,0,2, 0,0,0 ,  S(0) */
	/* 0,0,3, 0,0,1 , S(1) */
	/*0,0,4, 0,0,2 ,  S(2) */
	/*0,0,5, 0,0,3 ,  S(3) */
	/*0,0,6, 0,0,4 ,  S(4) */
	0,1,3, 0,0,1 } /* 1-0 S(1) */;

	/* the most commonly observed H2 lines:
	 * 0-0 R(5),
	 * 0-0 S(0), 
	 * 0-0 S(1), 
	 * 0-0 S(2), 
	 * 0-0 S(3), 
	 * 0-0 S(5), 
	 * 0-0 S(19), 
	 * 0-0 S(25), 
	 * 1-0 Q(3), 
	 * 1-0 Q(5), 
	 * 1-0 R(5) 
	 * 1-0 S(1), 
	 * 1-0 S(7), 
	 * 1-0 Q(3), 
	 * 2-1 S(1), 
	 * 6-4 O(3),  
	 */
#	endif
	int iRotHi, iVibHi, iElecHi ,iRotLo, iVibLo, iElecLo,
		ipHi , ipLo;

	/* H2 not on, so space not allocated */
	if( !h2.lgH2ON )
		return;

	iElecHi = 0;
	iElecLo = 0;
#	if 0
	/* print all the v = 0, 1, ro-vib lines */
	for( iVibHi=0; iVibHi<2; ++iVibHi )
	{
		for( iRotHi=2; iRotHi<nRot_hi[0][0]; ++iRotHi )
		{
			/* all ground vib state rotation lines - first is J to J-2 */
			PutLine(&H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotHi-2]);
			/* next is J to J - may not have been defined for some rot,
			 * and certainly not defined for iVibHi = 0 - since J=J and v=v */
			if( iVibHi > 0 && H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotHi].Aul > 0. )
				PutLine(&H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotHi]);
		}
	}
#	endif
	/* print all lines from lowest n levels within X */
	for( ipHi=2; ipHi<nLevels_per_elec[iElecHi]; ++ipHi )
	{
		/* obtain the proper indices for the upper level */
		long int ip = ipX_ener_sort[ipHi];
		iVibHi = ipVib_energies[ip];
		iRotHi = ipRot_energies[ip];
		for( ipLo=0; ipLo<ipHi; ++ipLo )
		{
			ip = ipX_ener_sort[ipLo];
			iVibLo = ipVib_energies[ip];
			iRotLo = ipRot_energies[ip];
			if( iVibHi >= iVibLo && (abs(iRotLo-iRotHi)==2 || (iRotLo-iRotHi)==0) )
			{
				if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul>0. )
				{
					/* all ground vib state rotation lines - first is J to J-2 */
					PutLine(&H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo]);
					if( LineSave.ipass == 0 )
					{
						SaveLine[iVibHi][iRotHi][iVibLo][iRotLo] =  0.;
					}
					else if( LineSave.ipass == 1 )
					{
						SaveLine[iVibHi][iRotHi][iVibLo][iRotLo] += (float)(
							radius.dVeff*H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].xIntensity);
					}
				}
			}
		}
	}
	return;
#	if 0
	for( i=0; i<N_H2_LINES_OUTPUT; ++i )
	{
		/* the lower state */
		iElecLo = ip_H2_lines_output[i*6+3];
		iVibLo = ip_H2_lines_output[i*6+4];
		iRotLo = ip_H2_lines_output[i*6+5];
		/* the upper state */
		iElecHi = ip_H2_lines_output[i*6];
		iRotHi = ip_H2_lines_output[i*6+2];
		iVibHi = ip_H2_lines_output[i*6+1];

		/* dump the line */
		PutLine(&H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo]);
	}

	/* only do this if molecule has been updated in this zone */
	/* this saves all lines for possible punch output */
	iElecLo = 0;
	iElecHi = 0;
	/* first branch is when called after space allocated, but before intensities
	 * known, or first zone computed - job is to zero out storage arrays */
	if( LineSave.ipass == 0 )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				for( iVibLo=0; iVibLo<=iVibHi; ++iVibLo )
				{
					long nr = nRot_hi[iElecLo][iVibLo];
					if( iVibHi==iVibLo )
						/* max because cannot malloc 0 bytes */
						nr = MAX2(1,iRotHi-1);
					for( iRotLo=0; iRotLo<nr; ++iRotLo )
					{
						SaveLine[iVibHi][iRotHi][iVibLo][iRotLo] =  0.;
					}
				}
			}
		}
	}
	else if( LineSave.ipass == 1 )
	{
		/* only add the flux into the save array if the H2 molecule was updated in this zone */
		if( h2.nCallH2_this_zone )
		{
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					for( iVibLo=0; iVibLo<=iVibHi; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iVibHi==iVibLo )
							/* max because cannot malloc 0 bytes */
							nr = MAX2(1,iRotHi-1);
						for( iRotLo=0; iRotLo<nr; ++iRotLo )
						{
							if( (iElecHi==iElecLo) && (iVibHi==iVibLo) && (iRotHi==iRotLo) )
								continue;
							ASSERT( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul >= 0. );
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								ASSERT( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].xIntensity>= 0.);
								ASSERT( SaveLine[iVibHi][iRotHi][iVibLo][iRotLo] >= 0.);
								SaveLine[iVibHi][iRotHi][iVibLo][iRotLo] += (float)(
									radius.dVeff*H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].xIntensity);
							}
						}
					}
				}
			}
		}
	}
#	endif
}

/* returns a char with the spectroscopic branch of a transition */
static char chMolBranch( long iRotHi , long int iRotLo )
{
	/* these are the spectroscopic branches */
	char chBranch[5] = {'O','P','Q','R','S'};
	/* this is the index within the chBranch array */
	int ip = 2 + (iRotHi - iRotLo);
	if( ip<0 || ip>=5 )
	{
		fprintf(ioQQQ," chMolBranch called with insane iRotHi=%li iRotLo=%li ip=%i\n",
			iRotHi , iRotLo , ip );
		ip = 0;
	}
	fixit();/*rm above, uncomment below */
	/*ASSERT( ip>= 0 && ip < 5  );*/
	return( chBranch[ip] );
}

/* punch some properties of the large H2 molecule */
void H2_Punch( FILE* io , char chJOB[] , char chTime[] , long int ipPun )
{
	long int iVibHi , iElecHi , iRotHi , iVibLo , iElecLo , iRotLo;
	long int LimVib , LimRot;

	/* which job are we supposed to do? This routine is active even when H2 is not turned on */

	/* populations in last zone */
	if( (strcmp( chJOB , "H2po" ) == 0) && (strcmp(chTime,"LAST") == 0) &&
		(punch.punarg[2][ipPun] != 0) )
	{
		if( h2.lgH2ON )
		{
			iVibHi= 0;
			iRotHi = 0;
			iElecHi=0;
			/* the limit to the number of vibration levels punched -
			* default is all, but first two numbers on punch h2 pops command
			* reset limit */
			/* this is limit to vibration */
			if( punch.punarg[0][ipPun] > 0 )
			{
				LimVib = (long)punch.punarg[0][ipPun];
			}
			else
			{
				LimVib = nVib_hi[iElecHi];
			}

			/* first punch the current ortho, para, and total H2 density */
			fprintf(io,"%i\t%i\t%.3e\tortho\n", 
				103 , 
				103 ,
				h2.ortho_density );
			fprintf(io,"%i\t%i\t%.3e\tpara\n", 
				101 , 
				101 ,
				h2.para_density );
			fprintf(io,"%i\t%i\t%.3e\ttotal\n", 
				0 , 
				0 ,
				hmi.htwo_total );

			/* now punch the actual populations, first part both matrix and triplets */
			for( iVibHi=0; iVibHi<=LimVib; ++iVibHi )
			{
				/* this is limit to rotation quantum index */
				if( punch.punarg[1][ipPun] > 0 )
				{
					LimRot = (long)punch.punarg[1][ipPun];
				}
				else
				{
					LimRot = nRot_hi[iElecHi][iVibHi];
				}
				if( punch.punarg[2][ipPun] > 0 )
				{
					long int i;
					/* this option punch matrix */
					if( iVibHi == 0 )
					{
						fprintf(io,"vib\\rot");
						/* this is first vib, so make row of rot numbs */
						for( i=0; i<=LimRot; ++i )
						{
							fprintf(io,"\t%li",i);
						}
						fprintf(io,"\n");
					}
					fprintf(io,"%li",iVibHi );
					for( iRotHi=Jlowest[iElecHi]; iRotHi<=LimRot; ++iRotHi )
					{
						fprintf(io,"\t%.3e", 
							populations[iElecHi][iVibHi][iRotHi]/hmi.htwo_total );
					}
					fprintf(io,"\n" );
				}
				else if( punch.punarg[2][ipPun] < 0 )
				{
					/* this option punch triplets - the default */
					for( iRotHi=Jlowest[iElecHi]; iRotHi<=LimRot; ++iRotHi )
					{
						fprintf(io,"%li\t%li\t%c\t%.1f\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\n", 
							/* upper vibration and rotation quantum numbers */
							iVibHi , iRotHi ,
							/* an 'O' or 'P' for ortho or para */
							chlgPara[lgOrtho[iElecHi][iVibHi][iRotHi]],
							/* the energy in wavenumbers */
							energy_wn[iElecHi][iVibHi][iRotHi],
							/* actual population relative to total H2 */
							populations[iElecHi][iVibHi][iRotHi]/hmi.htwo_total ,
							/* old level populations for comparison */
							old_populations[iElecHi][iVibHi][iRotHi]/hmi.htwo_total ,
							/* populations per h2 and per statistical weight */
							populations[iElecHi][iVibHi][iRotHi]/hmi.htwo_total/stat[iElecHi][iVibHi][iRotHi] ,
							/* lte departure coefficient */
							populations[iElecHi][iVibHi][iRotHi]/MAX2(SMALLFLOAT,populations_LTE[iElecHi][iVibHi][iRotHi] ) ,
							/* fraction of exits that were collisional */
							col_rate_out[iVibHi][iRotHi]/MAX2(SMALLFLOAT,col_rate_out[iVibHi][iRotHi]+rad_rate_out[iVibHi][iRotHi]) ,
							/* fraction of entrys that were collisional */
							col_rate_in[iVibHi][iRotHi]/MAX2(SMALLFLOAT,col_rate_in[iVibHi][iRotHi]+rad_rate_in[iVibHi][iRotHi]),
							/* collisions out */
							col_rate_out[iVibHi][iRotHi],
							/* radiation out */
							rad_rate_out[iVibHi][iRotHi] ,
							/* radiation out */
							col_rate_in[iVibHi][iRotHi],
							/* radiation in */
							rad_rate_in[iVibHi][iRotHi]
							
							) ;
					}
				}
			}
		}
	}
	/* PUNCH H2 POPULATIONS ZONE 
	 * populations of v=0 for each zone */
	else if( (strcmp( chJOB , "H2po" ) == 0) && (strcmp(chTime,"LAST") != 0) &&
		(punch.punarg[2][ipPun] == 0) )
	{
		if( h2.lgH2ON )
		{
			iElecHi = 0;
			iVibHi = 0;
			/* this is limit to rotation quantum index */
			if( punch.punarg[1][ipPun] > 0 )
			{
				LimRot = (long)punch.punarg[1][ipPun];
			}
			else
			{
				LimRot = nRot_hi[iElecHi][iVibHi];
			}
			fprintf(io,"%.5e\t%.3e\t%.3e", radius.depth , 
				h2.ortho_density , h2.para_density);
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=LimRot; ++iRotHi )
			{
				fprintf(io,"\t%.3e", 
					populations[iElecHi][iVibHi][iRotHi]/hmi.htwo_total );
			}
			fprintf(io,"\n");
		}
	}

	/* column densities */
	else if( (strcmp( chJOB , "H2cl" ) == 0) && (strcmp(chTime,"LAST") == 0) )
	{
		iVibHi= 0;
		iRotHi = 0;
		iElecHi=0;
		/* the limit to the number of vibration levels punched -
		 * default is all, but first two numbers on punch h2 pops command
		 * reset limit */
		/* this is limit to vibration */
		if( punch.punarg[0][ipPun] > 0 )
		{
			LimVib = (long)punch.punarg[0][ipPun];
		}
		else
		{
			LimVib = nVib_hi[iElecHi];
		}

		/* first punch ortho and para populations */
		fprintf(io,"%i\t%i\t%.3e\tortho\n", 
			103 , 
			103 ,
			ortho_colden );
		fprintf(io,"%i\t%i\t%.3e\tpara\n", 
			101 , 
			101 ,
			para_colden );
		/* total H2 column density */
		fprintf(io,"%i\t%i\t%.3e\ttotal\n", 
			0 , 
			0 ,
			colden.colden[ipCOLH2] );

		/* punch level column densities */
		for( iVibHi=0; iVibHi<=LimVib; ++iVibHi )
		{
		if( h2.lgH2ON )
		{
			/* this is limit to rotation quantum index */
			if( punch.punarg[1][ipPun] > 0 )
			{
				LimRot = (long)punch.punarg[1][ipPun];
			}
			else
			{
				LimRot = nRot_hi[iElecHi][iVibHi];
			}
			if( punch.punarg[2][ipPun] > 0 )
			{
				long int i;
				/* punch matrix */
				if( iVibHi == 0 )
				{
					fprintf(io,"vib\\rot");
					/* this is first vib, so make row of rot numbs */
					for( i=0; i<=LimRot; ++i )
					{
						fprintf(io,"\t%li",i);
					}
					fprintf(io,"\n");
				}
				fprintf(io,"%li",iVibHi );
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=LimRot; ++iRotHi )
				{
					fprintf(io,"\t%.3e", 
						H2_X_colden[iVibHi][iRotHi]/hmi.htwo_total );
				}
				fprintf(io,"\n" );
			}
			else
			{
				/* punch triplets - the default */
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=LimRot; ++iRotHi )
				{
					fprintf(io,"%li\t%li\t%.1f\t%.3e\t%.3e\n", 
						iVibHi , 
						iRotHi ,
						/* energy relative to 0,0, T1CM converts wavenumber to K */
						energy_wn[iElecHi][iVibHi][iRotHi]*T1CM,
						H2_X_colden[iVibHi][iRotHi] ,
						H2_X_colden[iVibHi][iRotHi]/stat[iElecHi][iVibHi][iRotHi]);
				}
			}
		}
		}
	}
	else if( (strcmp(chJOB , "H2pd" ) == 0) && (strcmp(chTime,"LAST") != 0) )
	{
		/* punch pdr 
		 * output some pdr information (densities, rates) for each zone */
		fprintf(io,"%.5e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
			/* depth in cm */
			radius.depth ,
			/* the computed ortho and para densities */
			h2.ortho_density , 
			h2.para_density ,
			/* the Lyman Werner band dissociation, Tielens & Hollenbach */
			hmi.H2_Solomon_rate_TH85 , 
			/* the Lyman Werner band dissociation, Bertoldi & Draine */
			hmi.H2_Solomon_rate_BD96,
			/* the Lyman Werner band dissociation, big H2 mole */
			hmi.H2_Solomon_rate_BigH2);
	}
	else if( (strcmp(chJOB , "H2co" ) == 0) && (strcmp(chTime,"LAST") != 0) )
	{
		/* PUNCH H2 COOLING - do heating cooling for each zone old new H2 */
		fprintf(io,"%.5e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
			/* depth in cm */
			radius.depth ,
			/* total cooling, equal to total heating */
			thermal.ctot , 
			/* H2 destruction by Solomon process, TH85 rate */
			hmi.H2_Solomon_rate_TH85,
			/* H2 destruction by Solomon process, big H2 model rate */
			hmi.H2_Solomon_rate_BigH2,
			/* H2 photodissociation heating, eqn A9 of Tielens & Hollenbach 1985a */
			hmi.HeatH2Dish_TH85,
			/* heating due to dissociation of elec excited states */
			hmi.HeatH2Dish_BigH2 , 
			/* cooling (usually neg and so heating) due to collisions within X */
			hmi.HeatH2Dexc_TH85,
			hmi.HeatH2Dexc_BigH2 
			);

	}
	else if( (strcmp(chJOB , "H2ra" ) == 0) && (strcmp(chTime,"LAST") != 0) )
	{
		/* punch h2 rates - some rates and lifetimes */
		double sumpop = 0. , sumlife = 0.;

		/* this block, find lifetime against photo excitation into excited elec states */
		iElecLo = 0;
		iVibLo = 0;
		if( h2.lgH2ON )
		{
			for( iElecHi=1; iElecHi<n_elec_states; ++iElecHi )
			{
				for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
				{
					for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
					{
						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nRot_hi[iElecLo][iVibLo]; ++iRotLo )
						{
							/* >>chng 03 feb 14, from !=0 to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								sumlife +=
									H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].pump *
									populations[iElecLo][iVibLo][iRotLo];
								sumpop +=
									populations[iElecLo][iVibLo][iRotLo];
							}
						}
					}
				}
			}
		}

		/* find photoexcitation rates from v=0 */
		/* PDR information for each zone */
		fprintf(io,"%.5e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
			/* depth in cm */
			radius.depth ,
			/* total H2 fraction */
			hmi.htwo_total/dense.gas_phase[ipHYDROGEN] ,
			/* rate H2 forms on grains */
			gv.rate_h2_form_grains_used_total , 
			/* rate H2 forms by H minus route */
			hmi.Molec[ipMHm]*1.35e-9,
			/* H2 destruction by Solomon process, TH85 rate */
			hmi.H2_Solomon_rate_TH85,
			/* H2 destruction by Solomon process, Bertoldi & Draine rate */
			hmi.H2_Solomon_rate_BD96,
			/* H2 destruction by Solomon process, big H2 model rate */
			hmi.H2_Solomon_rate_BigH2,
			/* the radiation field relative to the Habing value */
			hmi.GammaHabing,
			/* the column density in H2 */
			colden.colden[ipCOLH2],
			sumlife/MAX2(SMALLFLOAT , sumpop ) );
	}
	else if( (strcmp(chJOB , "H2ln" ) == 0) && (strcmp(chTime,"LAST") == 0) )
	{
		/* the full emission line spectrum */
		double thresh ;
		double renorm;
		if( h2.lgH2ON )
		{
			/* get the normalization line */
			if( LineSv[norm.ipNormWavL].sumlin > 0. )
			{
				renorm = norm.ScaleNormLine/LineSv[norm.ipNormWavL].sumlin;
			}
			else
			{
				renorm = 1.;
			}
			if( renorm > SMALLFLOAT )
			{
				/* this is threshold for faintest line, normally 0, set with 
				* number on punch verner command */
				thresh = thresh_punline_h2/(float)renorm;
			}
			else
			{
				thresh = 0.f;
			}

			iElecLo = 0;
			iElecHi = 0;
			/* punch lines at end of iteration */
			for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
			{
				for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
				{
					for( iVibLo=0; iVibLo<=iVibHi; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iVibHi==iVibLo )
							/* max because cannot malloc 0 bytes */
							nr = MAX2(1,iRotHi-1);
						for( iRotLo=0; iRotLo<nr; ++iRotLo )
						{
							if( SaveLine[iVibHi][iRotHi][iVibLo][iRotLo] > thresh )
							{
								/* air wavelength in microns */
								/* WLAng contains correction for index of refraction of air */
								double wl = H2Lines[0][iVibHi][iRotHi][0][iVibLo][iRotLo].WLAng/1e4;
								fixit();/* rm following test and print */
								if( abs(iRotHi-iRotLo)>2 )
									fprintf(ioQQQ,"insane chng rot %li %li %li %li \n",
									iVibHi , iRotHi , iVibLo , iRotLo );
								fprintf(io, "%li-%li %c(%li)", 
									iVibHi , 
									iVibLo ,
									chMolBranch( iRotHi , iRotLo ) ,
									iRotLo );
								fprintf( io, "\t%ld\t%ld\t%ld\t%ld", 
									iVibHi , iRotHi , iVibLo , iRotLo);
								/* WLAng contains correction for index of refraction of air */
								fprintf( io, "\t%.7f\t", wl );
								/*prt_wl print floating wavelength in Angstroms, in output format */
								prt_wl( io , H2Lines[0][iVibHi][iRotHi][0][iVibLo][iRotLo].WLAng );
								fprintf( io, "\t%.3f\t%.3e\n", 
									log10(MAX2(1e-37,SaveLine[iVibHi][iRotHi][iVibLo][iRotLo])) + radius.Conv2PrtInten, 
									SaveLine[iVibHi][iRotHi][iVibLo][iRotLo]*renorm );
							}
						}
					}
				}
			}
		}
	}

	return;
}


/* print line optical depths, called from premet in response to print line optical depths command*/
void H2_prt_line_tau(void)
{
		long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo ;

#	ifdef DEBUG_FUN
	fputs( "<+>H2_prt_line_tau()\n", debug_fp );
#	endif

	if( !h2.lgH2ON )
		return;

	/* loop over all possible lines */
	for( iElecHi=0; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				/* now the lower levels */
				/* NB - X is the only lower level considered here, since we are only 
				* concerned with excited electronic levels as a photodissociation process
				* code exists to relax this assumption - simply change following to iElecHi */
				long int lim_elec_lo = 0;
				for( iElecLo=0; iElecLo<=lim_elec_lo; ++iElecLo )
				{
					/* want to include all vib states in lower level if different elec level,
					* but only lower vib levels if same elec level */
					long int nv = nVib_hi[iElecLo];
					if( iElecLo==iElecHi )
						nv = iVibHi;
					for( iVibLo=0; iVibLo<=nv; ++iVibLo )
					{
						long nr = nRot_hi[iElecLo][iVibLo];
						if( iElecLo==iElecHi && iVibHi==iVibLo )
							nr = iRotHi-1;

						for( iRotLo=Jlowest[iElecLo]; iRotLo<=nr; ++iRotLo )
						{
							/* >>chng 03 feb 14, from !=0 to >0 */
							if( H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo].Aul > 0. )
							{
								prme(" c",&H2Lines[iElecHi][iVibHi][iRotHi][iElecLo][iVibLo][iRotLo] );
							}
						}
					}
				}
			}
		}
	}

	return;
}

/*H2_Level_lowJ evaluate CO rotation cooling */
static void H2_Level_lowJ(
	/* number of vibrational levels we consider here - not highest vib*/
	long int nVibrate ,
	/* number of rotational levels we consider here - not highest rot */
	long int nRotate ,
	float abundan )
{

	/* will need to MALLOC space for these but only on first call */
	static double **data, 
	  **dest, 
	  /* pump[low][high] is rate (s^-1) from lower to upper level */
	  **pump,
	  **CollRate_levn,
	  *pops,
	  *depart,
	  /* statistical weight */
	  *stat_levn ,
	  /* excitation energies in kelvin */
	  *excit;

	static long int **ipdest;
	static long int *lookup_Rot , *lookup_Vib;

	static int lgFirst=TRUE;
	long int i,
		j,
		ilo , 
		ihi,
		ip,
		iElec,
		iElecHi,
		iVib,
		iRot,
		iVibHi,
		iRotHi;
	long nLevels;
	int lgDeBug,lgNegPop;
	double rot_cooling , dCoolDT ;
	double pop_e, factor, sum_pops_inbound;
	static long int ndimMalloced = 0;

#	ifdef DEBUG_FUN
	fputs( "<+>H2_Level_lowJ()\n", debug_fp );
#	endif

	/* pad the highest rotation level by one, since has edge effects,
	 * this J willa be filled in downstream by main routine */
	++nRotate;

	/* this is the total number of levels we will need - each is number of
	 * level (vib and rot), and add one for the fake continuum */
	nLevels = nVibrate * nRotate +1;

	/* option to not use the matrix */
	if( nLevels <= 1 )
		return;

	if( lgFirst )
	{
		/* will never do this again */
		lgFirst = FALSE;
		/* remember how much space we malloced in case ever called with more needed */
		ndimMalloced = nLevels;
		/* allocate the 1D arrays*/
		if( (excit = (double *)MALLOC( sizeof(double)*(size_t)(nLevels) )) == NULL )
			BadMalloc();
		if( (stat_levn = (double *)MALLOC( sizeof(double)*(size_t)(nLevels) )) == NULL )
			BadMalloc();
		if( (pops = (double *)MALLOC( sizeof(double)*(size_t)(nLevels) )) == NULL )
			BadMalloc();
		if( (depart = (double *)MALLOC( sizeof(double)*(size_t)(nLevels) )) == NULL )
			BadMalloc();
		/* create space for the 2D arrays */
		if( (pump = ((double **)MALLOC((size_t)(nLevels)*sizeof(double *)))) == NULL )
			BadMalloc();
		if( (CollRate_levn = ((double **)MALLOC((size_t)(nLevels)*sizeof(double *)))) == NULL )
			BadMalloc();
		if( (dest = ((double **)MALLOC((size_t)(nLevels)*sizeof(double *)))) == NULL )
			BadMalloc();
		if( (data = ((double **)MALLOC((size_t)(nLevels)*sizeof(double *)))) == NULL )
			BadMalloc();
		if( (ipdest = ((long int **)MALLOC((size_t)(nLevels)*sizeof(long int *)))) == NULL )
			BadMalloc();
		if( (lookup_Rot = ((long int *)MALLOC((size_t)(nLevels)*sizeof(long int)))) == NULL )
			BadMalloc();
		if( (lookup_Vib = ((long int *)MALLOC((size_t)(nLevels)*sizeof(long int)))) == NULL )
			BadMalloc();
		for( i=0; i<(nLevels); ++i )
		{
			if( (pump[i] = ((double *)MALLOC((size_t)(nLevels)*sizeof(double )))) == NULL )
				BadMalloc();
			if( (CollRate_levn[i] = ((double *)MALLOC((size_t)(nLevels)*sizeof(double )))) == NULL )
				BadMalloc();
			if( (dest[i] = ((double *)MALLOC((size_t)(nLevels)*sizeof(double )))) == NULL )
				BadMalloc();
			if( (data[i] = ((double *)MALLOC((size_t)(nLevels)*sizeof(double )))) == NULL )
				BadMalloc();
			if( (ipdest[i] = ((long int *)MALLOC((size_t)(nLevels)*sizeof(double )))) == NULL )
				BadMalloc();
		}

		ip = 0;
		/* generate lookup table to address the real molecule in a rational way */
		for( iVib=0; iVib<nVibrate; ++iVib)
		{
			for( iRot=0; iRot<nRotate; ++iRot )
			{
				lookup_Rot[ip] = iRot;
				lookup_Vib[ip] = iVib;
				++ip;
			}
		}
		/* this has to work - but ip does not include the fake e state */
		ASSERT( ip == nLevels-1 );

		/* the statistical weights of the levels */
		for( j=0; j < nLevels-1; j++ )
		{
			iRot = lookup_Rot[j];
			iVib = lookup_Vib[j];
			/* statistical weights for each level */
			stat_levn[j] = H2Lines[0][iVib][iRot+1][0][iVib][iRot].gLo;
		}
		/* this is the highest level - the electronic excited levels */
		stat_levn[nLevels-1] = 1.;

		/* set up the excitation potentials of each level relative to ground */
		for( j=0; j < nLevels-1; j++ )
		{
			iRot = lookup_Rot[j];
			iVib = lookup_Vib[j];
			/* excitation energy of each level relative to ground, in K */
			excit[j] = energy_wn[0][iVib][iRot]*T1CM;
		}
		excit[nLevels-1] = 1e5;

		for( j=0; j < nLevels-1; j++ )
		{
			/* make sure that the energies are ok */
			ASSERT( excit[j+1] > excit[j] );
		}
	}
	/* this is test for call with too many rotation levels to handle - logic needs
	 * for largest model atom to be called first */
	if( nLevels > ndimMalloced )
	{
		fprintf(ioQQQ," H2_Level_lowJ has been called with the number of rotor levels greater than space allocated.\n");
		puts( "[Stop in H2_Level_lowJ]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* all elements are used, and must be set to zero */
	for( i=0; i < nLevels; i++ )
	{
		for( j=0; j < nLevels; j++ )
		{
			data[j][i] = 0.;
			dest[j][i] = 0.;
			pump[j][i] = 0.;
			ipdest[j][i] = 0;
			CollRate_levn[j][i] = 0.;
		}
	}

	/* this is the total population in all excited electronic states */
	pop_e = 0;
	for( iElecHi=1; iElecHi<n_elec_states; ++iElecHi )
	{
		pop_e += pops_per_elec[iElecHi];
	}

	/* find all radiative interactions within matrix, and between
	 * matrix and upper X and excited electronic states */
	iElec = 0;
	for( i=0; i < nLevels-1; i++ )
	{
		double rateup , ratedown;
		iRot = lookup_Rot[i];
		iVib = lookup_Vib[i];
		/* this loop does radiative decays from upper states within small
		 * matrix region, into this lower level */
		for( ihi=i+1; ihi<nLevels-1; ++ihi )
		{
			iRotHi = lookup_Rot[ihi];
			iVibHi = lookup_Vib[ihi];
			/* NB - the dest prob is included in the total and the dest is set to zero
			 * since we want to only count one ots rate, in main calling routine,
			 * and do not want matrix solver below to include it */
			data[i][ihi] = H2Lines[0][iVibHi][iRotHi][0][iVib][iRot].Aul*(
				H2Lines[0][iVibHi][iRotHi][0][iVib][iRot].Pesc + 
				H2Lines[0][iVibHi][iRotHi][0][iVib][iRot].Pdest +
				H2Lines[0][iVibHi][iRotHi][0][iVib][iRot].Pelec_esc);
			dest[i][ihi] = 0.;
			pump[i][ihi] = H2Lines[0][iVibHi][iRotHi][0][iVib][iRot].pump;
			/* the continuum indices are on the f, not c, scale, and will be passed to 
			* LevelN, which works on f, not c, scale */
			ipdest[i][ihi] = H2Lines[0][iVibHi][iRotHi][0][iVib][iRot].ipCont;
		}
		/* this sum dos all pump and decays from all electronic excited states */
		rateup = 0.;
		ratedown = 0.;
		/* pop_e is zero the first time this is called in an iteration, and might
		 * become small if there is no uv light - the lower level is [0][iVib][iRot] */
		if( pop_e > SMALLFLOAT )
		{
			for( iElecHi=1; iElecHi<n_elec_states; ++iElecHi )
			{
				for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
				{
					for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
					{
						/* the rate electrons enter this state from excited elec states */
						/* >>chng 03 feb 14, from !=0 to >0 */
						if( H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Aul > 0. )
						{
							ratedown +=
								populations[iElecHi][iVibHi][iRotHi] / pop_e *
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Aul*
								(H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Pesc + 
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Pelec_esc + 
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].Pdest);
							rateup +=
								populations[iElecHi][iVibHi][iRotHi] / pop_e *
								H2Lines[iElecHi][iVibHi][iRotHi][iElec][iVib][iRot].pump;
						}
					}
				}
			}
		}
	
		{
			/*@-redef@*/
			enum {DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC && iRot==0 && iVib < 10 )
			{
				fprintf(ioQQQ,"debugh2\t%li\t%li\t%.3e\t%.3e\n",iRot,iVib,rateup, ratedown );
			}
		}
		ihi = nLevels-1;
		data[i][ihi] = ratedown;
		pump[i][ihi] = rateup;
		dest[i][ihi] = 0.;
		/* the continuum indices are on the f, not c, scale, and will be passed to 
		* LevelN, which works on f, not c, scale */
		ipdest[i][ihi] = H2Lines[1][0][0][0][0][0].ipCont;
	}

	/* not used - this can be deleted */
	/* set cs data, which will not be used, to unity */

	/* now evaluate the collision rates */
	lgDeBug = FALSE;
	for( ilo=0; ilo < nLevels-2; ilo++ )
	{
		iRot = lookup_Rot[ilo];
		iVib = lookup_Vib[ilo];
		if(lgDeBug)fprintf(ioQQQ,"%li",ilo);
		for( ihi=ilo+1; ihi < nLevels-1; ihi++ )
		{
			int nColl;
			iRotHi = lookup_Rot[ihi];
			iVibHi = lookup_Vib[ihi];
			/* first do deexcitation rate */
			CollRate_levn[ihi][ilo] = 0.;
			if( iVibHi <= VIB_COLLID && iVib <= VIB_COLLID)
			{
				for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
				{
					CollRate_levn[ihi][ilo] +=
						CollRate[nColl][iVibHi][iRotHi][iVib][iRot]*collider_density[nColl];
				}
			}
			if(lgDeBug)fprintf(ioQQQ,"\t%.1e",CollRate_levn[ihi][ilo]);

			/* now get excitation rate */
			CollRate_levn[ilo][ihi] = CollRate_levn[ihi][ilo]*
				Boltzmann[0][iVibHi][iRotHi]/MAX2(SMALLFLOAT,Boltzmann[0][iVib][iRot])*
				H2Lines[0][iVibHi][iRotHi][0][iVib][iRot].gHi / 
				H2Lines[0][iVibHi][iRotHi][0][iVib][iRot].gLo;

		}
		CollRate_levn[ilo][nLevels-1] = 0.;
		if(lgDeBug)fprintf(ioQQQ,"\n");
	}

	lgDeBug = FALSE;
	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			fprintf(ioQQQ,"excit");
			for(ilo=0; ilo<nLevels; ++ilo )
			{
				fprintf(ioQQQ,"\t%li",ilo );
			}
			fprintf(ioQQQ,"\n");
			for(ihi=0; ihi<nLevels;++ihi)
			{
				fprintf(ioQQQ,"\t%.2e",excit[ihi] );
			}
			fprintf(ioQQQ,"\n");

			fprintf(ioQQQ,"data [n][]\\[][n]\n");
			for(ilo=0; ilo<nLevels; ++ilo )
			{
				fprintf(ioQQQ,"\t%li",ilo );
			}
			fprintf(ioQQQ,"\n");
			for(ihi=0; ihi<nLevels;++ihi)
			{
				fprintf(ioQQQ,"%li", ihi);
				for(ilo=0; ilo<nLevels; ++ilo )
				{
					fprintf(ioQQQ,"\t%.2e",data[ihi][ilo] );
				}
				fprintf(ioQQQ,"\n");
			}

			fprintf(ioQQQ,"pump [n][]\\[][n]\n");
			for(ilo=0; ilo<nLevels; ++ilo )
			{
				fprintf(ioQQQ,"\t%li",ilo );
			}
			fprintf(ioQQQ,"\n");
			for(ihi=0; ihi<nLevels;++ihi)
			{
				fprintf(ioQQQ,"%li", ihi);
				for(ilo=0; ilo<nLevels; ++ilo )
				{
					fprintf(ioQQQ,"\t%.2e",pump[ihi][ilo] );
				}
				fprintf(ioQQQ,"\n");
			}

			fprintf(ioQQQ,"CollRate_levn [n][]\\[][n]\n");
			for(ilo=0; ilo<nLevels; ++ilo )
			{
				fprintf(ioQQQ,"\t%li",ilo );
			}
			fprintf(ioQQQ,"\n");
			for(ihi=0; ihi<nLevels;++ihi)
			{
				fprintf(ioQQQ,"%li", ihi);
				for(ilo=0; ilo<nLevels; ++ilo )
				{
					fprintf(ioQQQ,"\t%.2e",CollRate_levn[ihi][ilo] );
				}
				fprintf(ioQQQ,"\n");
			}
		}
	}

	LevelN(
		/* number of levels */
		nLevels,
		abundan,
		stat_levn,
		excit,
		pops,
		depart,
		&data,
		&dest,
		&pump,
		&CollRate_levn,
		/* say that we have evaluated the collision rates already */
		TRUE,
		&ipdest,
		&rot_cooling,
		&dCoolDT,
		" H2 ",
		/* lgNegPop positive if negative pops occured, negative if too cold */
		&lgNegPop,
	    lgDeBug );/* option to print suff - set to true for debug printout */

	/* this will be sum of populations of levels within correct bounds of matrix,
	 * not increased bounds we set above */
	sum_pops_inbound = 0.;
	/* can only define first LIMLEVELN elements, the vector's length */
	for( i=0; i< nLevels-1 ; ++i )
	{
		iRot = lookup_Rot[i];
		iVib = lookup_Vib[i];
		populations[0][iVib][iRot] = (float)pops[i];
		/* this is sum of pops within proper bounds, and will probably be smaller
		 * than abundan, the correct population - each of following is number
		 * of levels, not quantum of highest levels */
		if( iVib < matrix_vib && iRot < matrix_rot )
			sum_pops_inbound += pops[i];
	}

	/* now correct populations of levels that should have been in the matrix */
	factor =  abundan / MAX2( SMALLFLOAT ,sum_pops_inbound );
	if(lgH2_TRACE) 
		fprintf(ioQQQ," ratio of correct matrix pops to total is %.4e\n", factor );
	for( i=0; i< nLevels-1 ; ++i )
	{
		iRot = lookup_Rot[i];
		iVib = lookup_Vib[i];
		/* correct all populations, since same scale factor is off for all */
		populations[0][iVib][iRot] *= (float)factor;
		pops[i] *= factor;
	}

	if(lgH2_TRACE) 
	{
		/* print pops that came out of matrix */
		fprintf(ioQQQ,"\tH2_Level_lowJ hmi.htwo_total: %.3e matrix rel pops\n",hmi.htwo_total);
		for( iVib=0; iVib<matrix_vib; ++iVib )
		{
			fprintf(ioQQQ,"\tVib%li", iVib);
			for( iRot=0; iRot<matrix_rot; ++iRot )
			{
				fprintf(ioQQQ,"\t%.2e",populations[0][iVib][iRot]/hmi.htwo_total);
			}
			fprintf(ioQQQ,"\n");
		}
		/* factor corrects for amount by which total abundance should have been off,
		 * since abund should have been the specified matrix, and we did a bit extra 
		 * for continity */
		fprintf(ioQQQ,"\tfake e state:%.2e should have been %.3e\n", 
			pops[nLevels-1] * factor/hmi.htwo_total,
			pop_e*frac_matrix);
	}

	if( lgNegPop > 0 )
	{
		fprintf(ioQQQ,"H2_Level_lowJ called LevelN which returned negative populations.\n");
	}

#	ifdef DEBUG_FUN
	fputs( " <->H2_Level_lowJ()\n", debug_fp );
#	endif
	return;
}

#if 0
/* this was attempt at lepp & shull coll diss rate, but formalism can't work */
/*H2_H_CollDissoc H2 collisional dissociation by H */
static double H2_H_CollDissoc( long iVib , long iRot )
{
	double rate;
	static float tused=-1;
	static double rate00=-1;
	/* collisional dissociation rates taken from 
	 * >>refer	H2	coll diss	Lepp, S., & Shull, J.M., 1983, ApJ, 270, 578-582 */

	if( tused == phycon.te )
	{
		tused = phycon.te;
		/* this is the low density rate, k_D(0,0) in Lepp Shull equation 4 */
		if( phycon.te > 7390. )
		{
			rate00 = 6.11e-14 * sexp( 2.93e4/phycon.te);
		}
		else
		{
			rate00 = 2.67e-15 * sexp( POW2(6750./phycon.te) );
		}
	}
	/* find rate for iVib and iRot from above, and eqn 4 of LS */
	rate = rate00 * exp( 13.45

	return rate;
}
#endif

/*H2_cooling evaluate cooling and heating due to H2 molecule */
void H2_Cooling(void)
{
	long int iElecHi , iElecLo , iVibHi , iVibLo , iRotHi , iRotLo;
	double rate_dn_heat, 
		rate_up_cool;
	long int nColl,
		ipHi, ipLo;
	double Big1;
	long int ipVib_big_hi,ipVib_big_lo ,ipRot_big_hi ,ipRot_big_lo;
	static double old_HeatH2Dexc;

#	ifdef DEBUG_FUN
	fputs( "<+>H2_Cooling()\n", debug_fp );
#	endif

	if( !h2.lgH2ON || !h2.nCallH2_this_zone )
	{
		hmi.HeatH2Dexc_BigH2 = 0.;
		hmi.HeatH2Dish_BigH2 = 0.;
		hmi.deriv_HeatH2Dexc_BigH2 = 0.;
		return;
	}

	hmi.HeatH2Dish_BigH2 = 0.;
	/* heating due to dissociation of elec excited states */
	for( iElecHi=1; iElecHi<n_elec_states; ++iElecHi )
	{
		for( iVibHi=0; iVibHi<=nVib_hi[iElecHi]; ++iVibHi )
		{
			for( iRotHi=Jlowest[iElecHi]; iRotHi<=nRot_hi[iElecHi][iVibHi]; ++iRotHi )
			{
				hmi.HeatH2Dish_BigH2 += 
					populations[iElecHi][iVibHi][iRotHi] * 
					dissprob[iElecHi][iVibHi][iRotHi] *
					disske[iElecHi][iVibHi][iRotHi];
			}
		}
	}
	/* disske was in eV - convert to ergs */
	hmi.HeatH2Dish_BigH2 *= (float)EN1EV;

	hmi.HeatH2Dexc_BigH2 = 0.;
	/* these are the colliders that will be considered as depopulating agents */
	/* the colliders are H, He, H2 ortho, H2 para, H+ */
	/* atomic hydrogen */
#	if 0
	collider_density[0] = dense.xIonDense[ipHYDROGEN][0];
	/* atomic helium */
	collider_density[1] = dense.xIonDense[ipHELIUM][0];
	/* all ortho h2 */
	collider_density[2] = (float)h2.ortho_density;
	/* all para H2 */
	collider_density[3] = (float)h2.para_density;
	/* protons - ionized hydrogen */
	collider_density[4] = dense.xIonDense[ipHYDROGEN][1];
#	endif

	/* now make sum of all collisions within X itself */
	iElecHi = 0;
	iElecLo = 0;
	Big1 = 0.;
	ipVib_big_hi = 0;
	ipVib_big_lo = 0;
	ipRot_big_hi = 0;
	ipRot_big_lo = 0;
	/* this will be derivative */
	hmi.deriv_HeatH2Dexc_BigH2 = 0.;
	for( ipHi=1; ipHi<nLevels_per_elec[iElecHi]; ++ipHi )
	{
		long int ip = ipX_ener_sort[ipHi];
		iVibHi = ipVib_energies[ip];
		iRotHi = ipRot_energies[ip];
		if( iVibHi > VIB_COLLID )
			continue;

		for( ipLo=0; ipLo<ipHi; ++ipLo )
		{
			double oneline;
			ip = ipX_ener_sort[ipLo];
			iVibLo = ipVib_energies[ip];
			iRotLo = ipRot_energies[ip];
			if( iVibLo > VIB_COLLID)
				continue;

			rate_dn_heat = 0.;
			rate_up_cool = 0.;

			/* this sum is total downward heating */
			for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
			{
				rate_dn_heat +=
					populations[iElecHi][iVibHi][iRotHi] * 
					CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo]*
					collider_density[nColl];

				/* now get upward collisional cooling by detailed balance */
				rate_up_cool += 
					populations[iElecLo][iVibLo][iRotLo] *
					/* downward collision rate */
					CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo]*
					collider_density[nColl]*
					/* rest converts into upward collision rate */
					stat[iElecHi][iVibHi][iRotHi] / stat[iElecLo][iVibLo][iRotLo] *
					Boltzmann[iElecHi][iVibHi][iRotHi] /
					MAX2(SMALLFLOAT , Boltzmann[iElecLo][iVibLo][iRotLo] );
			}

			/* net heating in ergs - this will usually be heating */
			oneline = (rate_dn_heat - rate_up_cool)*
				(energy_wn[iElecHi][iVibHi][iRotHi] - energy_wn[iElecLo][iVibLo][iRotLo]) *
				ERG1CM;
			hmi.HeatH2Dexc_BigH2 += (float)oneline;

			/* deriv wrt temperature - assume exp wrt ground - this needs to be
			 * divided by square of temperature in wn - 
			 * done at end of loop */
			hmi.deriv_HeatH2Dexc_BigH2 +=  (float)(oneline *energy_wn[iElecHi][iVibHi][iRotHi]);

			/* option to keep track of strongest single agent */
/*#			define H2COOLDEBUG*/
#			if defined(H2COOLDEBUG)
			if( fabs(oneline) > fabs(Big1 ) )
			{
				Big1 = oneline;
				ipVib_big_hi = iVibHi;
				ipVib_big_lo = iVibLo;
				ipRot_big_hi = iRotHi;
				ipRot_big_lo = iRotLo;
			}
#			endif

			/* this would be a major logical error */
			ASSERT( 
				(rate_up_cool==0 && rate_dn_heat==0) || 
				(energy_wn[iElecHi][iVibHi][iRotHi] > energy_wn[iElecLo][iVibLo][iRotLo]) );
		}
	}

	/* this is deriv of collisional heating wrt temperature - needs to be
	 * divided by square of temperature in wn */
	hmi.deriv_HeatH2Dexc_BigH2 /=  POW2(phycon.te_wn) ;

#	if defined(H2COOLDEBUG)
	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && (fabs(hmi.HeatH2Dexc_BigH2) > SMALLFLOAT) )
		{
			int iVib = 0;
			fprintf(ioQQQ," H2_Cooling, total coll %.2e, frac 1 line %.2e %li %li %li %li \n",
				hmi.HeatH2Dexc_BigH2 , Big1/hmi.HeatH2Dexc_BigH2 ,
				ipVib_big_hi , ipRot_big_hi , ipVib_big_lo , ipRot_big_lo );

			/*fprintf(ioQQQ," H2_cooling pops\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\t%.3e\n",
				populations[0][iVib][0]/hmi.htwo_total,
				populations[0][iVib][1]/hmi.htwo_total,
				populations[0][iVib][2]/hmi.htwo_total,
				populations[0][iVib][3]/hmi.htwo_total,
				populations[0][iVib][4]/hmi.htwo_total,
				populations[0][iVib][5]/hmi.htwo_total);*/

			iElecHi = iElecLo = 0;
			iVibHi = iVibLo = 0;
			iRotHi = 3;
			iRotLo = 1;
			rate_dn_heat = rate_up_cool = 0.;
			/* this sum is total downward heating */
			for( nColl=0; nColl<N_X_COLLIDER; ++nColl )
			{
				rate_dn_heat +=
					populations[iElecHi][iVibHi][iRotHi] * 
					CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo]*
					collider_density[nColl];

				/* now get upward collisional cooling by detailed balance */
				rate_up_cool += 
					populations[iElecLo][iVibLo][iRotLo] *
					/* downward collision rate */
					CollRate[nColl][iVibHi][iRotHi][iVibLo][iRotLo]*
					collider_density[nColl]*
					/* rest converts into upward collision rate */
					stat[iElecHi][iVibHi][iRotHi] / stat[iElecLo][iVibLo][iRotLo] *
					Boltzmann[iElecHi][iVibHi][iRotHi] /
					MAX2(SMALLFLOAT , Boltzmann[iElecLo][iVibLo][iRotLo] );
			}

			fprintf(ioQQQ," H2_cooling pop31\t%.3e\tdn up 31\t%.3e\t%.3e\n",
				populations[0][iVib][3]/populations[0][iVib][1],
				rate_dn_heat,rate_up_cool
				);
		}
	}
#	endif
#	undef	H2COOLDEBUG
	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC  )
		{
			static long nzdone=-1 , nzincre;
			if( nzone!=nzdone )
			{
				nzdone = nzone;
				nzincre = -1;
			}
			++nzincre;
			fprintf(ioQQQ," H2 nz\t%.2f\tTe\t%.4e\tH2\t%.3e\tcXH\t%.2e\tdcXH/dt%.2e\tDish\t%.2e \n",
				(double)nzone + (double)nzincre/20., 
				phycon.te,
				hmi.htwo_total ,
				hmi.HeatH2Dexc_BigH2,
				hmi.deriv_HeatH2Dexc_BigH2 ,
				hmi.HeatH2Dish_BigH2);

		}
	}

	/* this can be noisy due to finite accuracy of soln, so take average with
	 * previous value */
	if( nzone <1 )
	{
		old_HeatH2Dexc = hmi.HeatH2Dexc_BigH2;
	}
	else
	{
		hmi.HeatH2Dexc_BigH2 = (float)((hmi.HeatH2Dexc_BigH2+old_HeatH2Dexc)/2.);
		old_HeatH2Dexc = hmi.HeatH2Dexc_BigH2;
	}
	if(lgH2_TRACE) 
		fprintf(ioQQQ,
		" H2_Cooling Ctot\t%.2e\t HeatH2Dish_BigH2 \t%.2e\t HeatH2Dexc_BigH2 \t%.2e\n" ,
		thermal.ctot , 
		hmi.HeatH2Dish_BigH2 , 
		hmi.HeatH2Dexc_BigH2 );

}
/*H2_ParseAtom parse information from the rotor command line */
void H2_ParseAtom(char *chCard )
{
	int lgEOL;
	long int i , j;

#	ifdef DEBUG_FUN
	fputs( "<+>H2_ParseAtom()\n", debug_fp );
#	endif

	/* this command has a 2 in the H2 label - must not parse the two by
	 * accident.  Get the first number off the line image, and confirm that
	 * it is a 2 */
	i = 5;
	j = (long int)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
	if( j != 2 )
	{
		fprintf( ioQQQ, " Something is wrong with the order of the numbers on this line.\n" );
		fprintf( ioQQQ, " The first number I encounter should be a 2.\n Sorry.\n" );
		puts( "[Stop in H2_ParseAtom]" );
		cdEXIT(EXIT_FAILURE);
	}

	if( lgMatch("LEVE",chCard) )
	{
		/* number of electronic levels */

		/* lgH2_READ_DATA is FALSE at start of calculation, set true when space 
		 * allocated for the H lines.  Once done we must ignore all 
		 * future changes in the number of levels */
		if( !lgH2_READ_DATA )
		{
			n_elec_states = (long int)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
			if( lgEOL )
			{
				if( lgMatch("LARG",chCard) )
				{
					/* LARGE is option to use the most number of electronic levels */
					n_elec_states = N_H2_ELEC;
				}
				else
				{
					NoNumb(chCard);
				}
			}

			if( n_elec_states < 2 )
			{
				fprintf( ioQQQ, " This would be too few levels.\n" );
				puts( "[Stop in H2_ParseAtom]" );
				cdEXIT(EXIT_FAILURE);
			}
			/* N_H2_ELEC is in h2.h and is the greatest number of elec lev possible */
			else if( n_elec_states > N_H2_ELEC )
			{
				fprintf( ioQQQ, 
					" This would be too many levels, the limit is %i.\n" , 
					N_H2_ELEC);
				puts( "[Stop in H2_ParseAtom]" );
				cdEXIT(EXIT_FAILURE);
			}
		}
	}

	else if( lgMatch("LIMI",chCard) )
	{
		/* the limit to the H2 / Htot ratio - 
		 * if smaller than this, do not compute large H2 mole */
		H2_to_H_limit = FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		if( lgEOL )
		{
			/* did not find a number, either mistake or key " off" */
			if( lgMatch( " OFF" , chCard ) )
			{
				/* turn off limit */
				H2_to_H_limit = -1.;
			}
			else
			{
				fprintf( ioQQQ," The limit must appear on this line.\n");
				NoNumb( chCard );
			}
		}
		else
		{
			/* got a number, check if negative and so a log */
			/* a number <= 0 is the log of the ratio */
			if( H2_to_H_limit <= 0. )
				H2_to_H_limit = pow(10., H2_to_H_limit);
		}
	}
	else if( lgMatch("FRAC",chCard ) )
	{
		/* this is special option to force H2 abundance to value for testing
		 * this factor will multiply the hydrogen density to become the H2 density
		 * no attempt to conserve particles, or do the rest of the molecular equilibrium
		 * set consistently is made */
		h2.frac_abund = FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		if( lgEOL )
			NoNumb( chCard );

		/* a number <= 0 is the log of the ratio */
		if( h2.frac_abund <= 0. )
			h2.frac_abund = pow(10., h2.frac_abund);
		/* don't let it exceed 0.5 */
		h2.frac_abund = MIN2(0.5 , h2.frac_abund );
	}
	else if( lgMatch("GBAR",chCard ) )
	{
		/* option to either use, or not use, gbar approximation for low X levels with no
		 * collision data - by default it is on */
		if( lgMatch(" OFF",chCard ) )
		{
			lgColl_gbar = FALSE;
		}
		else if( lgMatch(" ON ",chCard ) )
		{
			lgColl_gbar = TRUE;
		}
		else
		{
			fprintf( ioQQQ, 
				" The gbar approximation must be off (\" OFF\") or on (\" ON \").\n");
			puts( "[Stop in H2_ParseAtom]" );
			cdEXIT(EXIT_FAILURE);
		}
	}
	/* option to turn collisional deexcitation off or on */
	else if( lgMatch("COLL",chCard ) && lgMatch("DEEX",chCard ) )
	{
		/* option to turn collisions off */
		if( lgMatch(" ON ",chCard ) )
		{
			/* this is the default, leave collisions off */
			lgColl_deexec_Calc = TRUE;
		}
		else
		{
			/* default (and only reason for this command) is to turn off collisions */
			lgColl_deexec_Calc = FALSE;
		}
	}
	/* option to turn collisional dissociation off or on */
	else if( lgMatch("COLL",chCard ) && lgMatch("DISS",chCard ) )
	{
		/* option to turn collisions off */
		if( lgMatch(" ON ",chCard ) )
		{
			/* this is the default, leave collisions off */
			lgColl_dissoc_coll = TRUE;
		}
		else
		{
			/* default (and only reason for this command) is to turn off collisions */
			lgColl_dissoc_coll = FALSE;
		}
	}
	else if( lgMatch("MATR",chCard ) )
	{
		/* thmatrix option sets the number of vibration and rotation levels that will
		 * be included in the matrix solution */
		matrix_vib = (long)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		matrix_rot = (long)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		if( lgEOL && !lgMatch(" OFF",chCard) )
		{
			/* this branch hit eol but OFF is not on line - this is a mistake */
			fprintf( ioQQQ, 
				" The number of vibration and rotation levels used in the matrix solution must be entered.\n");
			puts( "[Stop in H2_ParseAtom]" );
			cdEXIT(EXIT_FAILURE);
		}
		/* routine does not certify that matrix limits are greater than 1 -
		 * zero or <0 limits just turns if off, as did the off option */
	}
	else if( lgMatch("TRAC",chCard ) )
	{
		/* turns on trace printout */
		lgH2_TRACE = TRUE;
	}
	else if( lgMatch("THER",chCard ) )
	{
		/* change the treatment of the heating - cooling effects of H2,
		 * options are simple (use TH85 expressions) and full (use large molecule)*/
		if( lgMatch("SIMP",chCard ) )
		{
			hmi.lgH2_Thermal_BigH2 = FALSE;
		}
		else if( lgMatch("FULL",chCard ) )
		{
			/* this is the default - use big atom */
			hmi.lgH2_Thermal_BigH2 = TRUE;
		}
	}
	else if( lgMatch("CHEM",chCard ) )
	{
		/* change the treatment of the chemistry - formation and destruction,
		 * options are simple (use TH85 expressions) and full (use large molecule)*/
		if( lgMatch("SIMP",chCard ) )
		{
			hmi.lgH2_Chemistry_BigH2 = FALSE;
		}
		else if( lgMatch("FULL",chCard ) )
		{
			/* this is the default - use big atom */
			hmi.lgH2_Chemistry_BigH2 = TRUE;
		}
	}

	/* there is no final branch - if we do not find a keyword, simply
	 * turn on the H2 molecule */
	/* the mere calling of this routine turns the large H2 molecule on */
	h2.lgH2ON = TRUE;

#	ifdef DEBUG_FUN
	fputs( " <->H2_ParseAtom()\n", debug_fp );
#	endif
	return;
}

/*H2_ParsePunch parse the punch h2 command */
void H2_ParsePunch( char *chCard )
{
	long int i , nelem ;
	int lgEOL;

	i = 5;
	nelem = (long int)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
	if( nelem != 2 )
	{
		fprintf( ioQQQ, " The first number on this line must be the 2 in H2\n Sorry.\n" );
		puts( "[Stop in ParsePunch]" );
		cdEXIT(EXIT_FAILURE);
	}
	/* this provides info on the large H2 molecule */
	if( lgMatch("POPU",chCard) )
	{
		/* punch populations */
		strcpy( punch.chPunch[punch.npunch], "H2po" );

		/* this is an option to scan off highest vib and rot states 
		 * to punch pops - first is limit to vibration, then rotation 
		 * if no number is entered then 0 is set and all levels punched */
		/* now get vib lim */
		punch.punarg[0][punch.npunch] = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

		/* this is limit to rotation quantum index */
		punch.punarg[1][punch.npunch] = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

		/* this says whether to punch triplets or a matrix for output -
		 * default is triplets, so only check for matrix */
		if( lgMatch( "MATR" , chCard ) )
		{
			/* matrix */
			punch.punarg[2][punch.npunch] = 1;
			fprintf( punch.ipPnunit[punch.npunch], "#vib\trot\tpops\n" );
		}
		else if( lgMatch( "ZONE" , chCard ) )
		{
			/* punch v=0 pops for each zone, all along one line */
			punch.punarg[2][punch.npunch] = 0;
			fprintf( punch.ipPnunit[punch.npunch], "#depth\torth\tpar\tv=0 rel pops\n" );
		}
		else
		{
			/* triplets */
			punch.punarg[2][punch.npunch] = -1;
			fprintf( punch.ipPnunit[punch.npunch], "#vib\trot\ts\tenergy(wn)\tpops/H2\told/H2\tpops/g\tdep coef\tFin(Col)\tFout(col)\tRCout\tRRout\tRCin\tRRin\n" );
		}
	}

	else if( lgMatch("COLU",chCard) )
	{
		/* punch column density */
		strcpy( punch.chPunch[punch.npunch], "H2cl" );

		/* this is an option to scan off highest vib and rot states 
		 * to punch pops - first is limit to vibration, then rotation 
		 * if no number is entered then 0 is set and all levels punched */
		/* now get vib lim */
		punch.punarg[0][punch.npunch] = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

		/* highest rotation */
		punch.punarg[1][punch.npunch] = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		/* this says whether to punch triplets or a matrix for output -
		 * default is triplets, so only check for matrix */
		if( lgMatch( "MATR" , chCard ) )
		{
			/* matrix */
			punch.punarg[2][punch.npunch] = 1;
			fprintf( punch.ipPnunit[punch.npunch], "#vib\trot\tcolumn density\n" );
		}
		else
		{
			/* triplets */
			punch.punarg[2][punch.npunch] = -1;
			fprintf( punch.ipPnunit[punch.npunch], "#vib\trot\tEner(K)\tcolden\tc/g\n" );
		}
	}
	else if( lgMatch(" PDR",chCard) )
	{
		/* creation and destruction processes */
		strcpy( punch.chPunch[punch.npunch], "H2pd" );
		fprintf( punch.ipPnunit[punch.npunch], "#H2 creation, destruction. \n" );
	}
	else if( lgMatch("COOL",chCard) )
	{
		/* heating and cooling rates */
		strcpy( punch.chPunch[punch.npunch], "H2co" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#H2 depth\ttot cool\tTH Sol\tBig Sol\tTH pht dis\tpht dis\tTH Xcool\tXcool \n" );
	}			

	else if( lgMatch("LINE",chCard) )
	{
		/* punch H2 lines - all in X */
		strcpy( punch.chPunch[punch.npunch], "H2ln" );
		fprintf( punch.ipPnunit[punch.npunch], "#H2 line\tVhi\tJhi\tVlo\tJlo\twl(mic)\twl(lab)\tlog flux\tI/Inorm \n" );
		/* first optional number changes the theshold of weakest line to print*/
		/* fe2thresh is intensity relative to normalization line,
		* normally Hbeta, and is set to zero in zero.c */

		/* threshold for faintest line to punch, default is 1e-4 of norm line */
		thresh_punline_h2 = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		if( lgEOL )
		{
			/* this will be default relative intensity for faintest line to punch */
			thresh_punline_h2 = 1e-4f;
		}

		/* it is a log if negative */
		if( thresh_punline_h2 < 0. )
		{
			thresh_punline_h2 = (float)pow(10.f,thresh_punline_h2);
		}
	}
	else if( lgMatch("RATE",chCard) )
	{
		/* creation and destruction rates */
		strcpy( punch.chPunch[punch.npunch], "H2ra" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#depth\tH2/Htot\tfrm grn\tfrmH-\tdstTH85\tBD96\tBigH2\tG0\tN(H2)\tEleclife \n" );
	}
	else
	{
		fprintf( ioQQQ, 
			" There must be a second key; they are  RATE, LINE, COOL, COLUMN, _PDR and POPUlations\n" );
		puts( "[Stop in ParsePunch]" );
		cdEXIT(EXIT_FAILURE);
	}
	return;
}

/*H2_colden maintain H2 column densities within X */
void H2_Colden( char *chLabel )
{
	long int iVib , iRot;

	if( !h2.lgH2ON || !h2.nCallH2_this_zone )
		return;

	if( strcmp(chLabel,"ZERO") == 0 )
	{
		/* the column density (cm-2) of ortho and para H2 */
		ortho_colden = 0.;
		para_colden = 0.;
		/* zero out formation rates and column densites */
		for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
		{
			for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
			{
				/* space for the rotation quantum number */
				H2_X_colden[iVib][iRot] = 0.;
			}
		}
	}

	else if( strcmp(chLabel,"ADD ") == 0 )
	{
		ortho_colden += h2.ortho_density*radius.drad_x_fillfac;
		para_colden += h2.para_density*radius.drad_x_fillfac;
		/*  add together column densities */
		for( iVib = 0; iVib <= nVib_hi[0] ; ++iVib )
		{
			for( iRot=Jlowest[0]; iRot<=nRot_hi[0][iVib]; ++iRot )
			{
				/* space for the rotation quantum number */
				H2_X_colden[iVib][iRot] += (float)(populations[0][iVib][iRot]*radius.drad_x_fillfac);
			}
		}
	}

	/* we will not print column densities so skip that - if not print then we have a problem */
	else if( strcmp(chLabel,"PRIN") != 0 )
	{
		fprintf( ioQQQ, " H2_colden does not understand the label %s\n", 
		  chLabel );
		puts( "[Stop in H2_colden]" );
		cdEXIT(EXIT_FAILURE);
	}
}

/*H2_prt_Zone print H2 info into zone results, called from prtzone for each printed zone */
void H2_prt_Zone(void)
{
	int iElecHi , iVibHi ;
#	ifdef DEBUG_FUN
	fputs( "<+>H2_prt_line_tau()\n", debug_fp );
#	endif

	/* no print if H2 not turned on, or not computed for these conditions */
	if( !h2.lgH2ON || !h2.nCallH2_this_zone )
		return;

	fprintf( ioQQQ, " H2 density   ");
	fprintf(ioQQQ,PrintEfmt("%9.2e", hmi.htwo_total));

	fprintf( ioQQQ, " orth/par");
	fprintf(ioQQQ,PrintEfmt("%9.2e", h2.ortho_density / MAX2( SMALLFLOAT , h2.para_density )));

	iElecHi = 0;
	iVibHi = 0;
	fprintf( ioQQQ, " v0 J=0,3");
	fprintf(ioQQQ,PrintEfmt("%9.2e", populations[iElecHi][iVibHi][0] / hmi.htwo_total));
	fprintf(ioQQQ,PrintEfmt("%9.2e", populations[iElecHi][iVibHi][1] / hmi.htwo_total));
	fprintf(ioQQQ,PrintEfmt("%9.2e", populations[iElecHi][iVibHi][2] / hmi.htwo_total));
	fprintf(ioQQQ,PrintEfmt("%9.2e", populations[iElecHi][iVibHi][3] / hmi.htwo_total));

	fprintf( ioQQQ, "    v=0,3");
	fprintf(ioQQQ,PrintEfmt("%9.2e", pops_per_vib[iElecHi][0] / hmi.htwo_total));
	fprintf(ioQQQ,PrintEfmt("%9.2e", pops_per_vib[iElecHi][1] / hmi.htwo_total));
	fprintf(ioQQQ,PrintEfmt("%9.2e", pops_per_vib[iElecHi][2] / hmi.htwo_total));
	fprintf(ioQQQ,PrintEfmt("%9.2e", pops_per_vib[iElecHi][3] / hmi.htwo_total));
	fprintf( ioQQQ, "\n");

	return;
}

/*H2_prt_column_density print H2 info into zone results, called from prtzone for each printed zone */
void H2_prt_column_density(	
	/* this is stream used for io, is stdout when called by final,
	 * is punch unit when punch output generated */
	 FILE *ioMEAN )

{
	int iVibHi ;
#	ifdef DEBUG_FUN
	fputs( "<+>H2_prt_line_tau()\n", debug_fp );
#	endif

	/* no print if H2 not turned on, or not computed for these conditions */
	if( !h2.lgH2ON || !h2.nCallH2_this_zone )
		return;

	fprintf( ioMEAN, " H2 total   ");
	fprintf(ioMEAN,"%7.3f", log10(MAX2(SMALLFLOAT,hmi.htwo_total)));

	fprintf( ioMEAN, " H2 orth   ");
	fprintf(ioMEAN,"%7.3f", log10(MAX2(SMALLFLOAT,ortho_colden)));

	fprintf( ioMEAN, " para");
	fprintf(ioMEAN,"%7.3f", log10(MAX2(SMALLFLOAT,para_colden)));

	iVibHi = 0;
	fprintf( ioMEAN, " v0 J=0,3");
	fprintf(ioMEAN,"%7.3f", log10(MAX2(SMALLFLOAT,H2_X_colden[iVibHi][0])));
	fprintf(ioMEAN,"%7.3f", log10(MAX2(SMALLFLOAT,H2_X_colden[iVibHi][1])));
	fprintf(ioMEAN,"%7.3f", log10(MAX2(SMALLFLOAT,H2_X_colden[iVibHi][2])));
	fprintf(ioMEAN,"%7.3f", log10(MAX2(SMALLFLOAT,H2_X_colden[iVibHi][3])));

#	if 0
	fprintf( ioMEAN, "    v=0,3");
	fprintf(ioMEAN,PrintEfmt("%9.2e", pops_per_vib[iElecHi][0] / hmi.htwo_total));
	fprintf(ioMEAN,PrintEfmt("%9.2e", pops_per_vib[iElecHi][1] / hmi.htwo_total));
	fprintf(ioMEAN,PrintEfmt("%9.2e", pops_per_vib[iElecHi][2] / hmi.htwo_total));
	fprintf(ioMEAN,PrintEfmt("%9.2e", pops_per_vib[iElecHi][3] / hmi.htwo_total));
	fprintf( ioMEAN, "\n");
#	endif

	return;
}
