/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/************************************************************************************************/
/*H_photo_cs_lin returns hydrogenic photoionization cross section in cm-2                       */
/*H_Einstein_A_lin calculates Einstein A for any nlz                                            */
/*hv calculates photon energy in ergs for n -> n' transitions for H and H-like ions             */
/************************************************************************************************/
/***************************  LOG version of h_bauman.c  ****************************************/
/*  In this version, quantites that would normal cause a 64-bit floating point processor        */
/* to either underflow or overflow are evaluated using logs instead of floating point math.     */
/* This allows us to use an upper principle quantum number `n' greater than  which the          */
/* other version begins to fail. The trade-off is, of course, lower accuracy                    */
/* ( or is it precision ). We use LOG_10 for convenience.                                       */
/************************************************************************************************/
#include "cddefines.h"
#include "physconst.h"
#include "hydro_bauman.h"

/*lint -e790 integral to float */
/************************************************************************************************/
/************************************************************************************************/
/*    these routines were written by Robert Bauman                                              */
/*    The main reference for this section of code is                                            */
/*    M. Brocklehurst                                                                           */
/*    Mon. Note. R. astr. Soc. (1971) 153, 471-490                                              */
/*                                                                                              */
/*    The recombination coefficient is obtained from the                                        */
/*    photoionization cross-section (see Burgess 1965).                                         */
/*    We have,                                                                                  */
/*                                                                                              */
/*                  -                           -                l'=l+1                         */
/*                 |  2 pi^(1/2) alpha^4 a_o^2 c |  2 y^(1/2)     ---                           */
/* alpha(n,l,Z,Te)=|-----------------------------|  ---------  Z   >    I(n, l, l', t)          */
/*                 |            3                |     n^2        ---                           */
/*                  -                           -               l'=l-1                          */
/*                                                                                              */
/*      where                         OO                                                        */
/*                                    -                                                         */
/*                                   |                                                          */
/*      I(n, l, l', t) = max(l,l') y | (1 + n^2 K^2)^2 Theta(n,l; K, l') exp( -K^2 y ) d(K^2)   */
/*                                   |                                                          */
/*                                  -                                                           */
/*                                  0                                                           */
/*                                                                                              */
/*      Here K = k / Z                                                                          */
/*                                                                                              */
/*                                                                                              */
/*      and                                                                                     */
/*                                                                                              */
/*                                                                                              */
/*       y =   Z^2 Rhc/(k Te)= 15.778/t                                                         */
/*                                                                                              */
/*       where "t" is the scaled temperature, and "Te" is the electron Temperature              */
/*                                                                                              */
/*       t = Te/(10^4  Z^2)                                                                     */
/*           Te in kelvin                                                                       */
/*                                                                                              */
/*                                           |              |^2                                 */
/*         Theta(n,l; K, l') = (1 + n^2 K^2) | g(n,l; K,l') |                                   */
/*                                           |              |                                   */
/*                                                                                              */
/*                                                                                              */
/*                                                 ---- Not sure if this is K or k              */
/*                                  OO            /     but think it is K                       */
/*         where                    -            v                                              */
/*                         K^2     |                                                            */
/*         g(n,l; K,l') = -----    | R_nl(r) F_(K,l) r dr                                       */
/*                         n^2     |                                                            */
/*                                -                                                             */
/*                                0                                                             */
/*                                                                                              */
/*                                                                                              */
/*                                                                                              */
/*                                                                                              */
/*                                                                                              */
/*                                                                                              */
/*         -                           -                                                        */
/*        |  2 pi^(1/2) alpha^4 a_o^2 c |                                                       */
/*        |-----------------------------|                                                       */
/*        |            3                |                                                       */
/*         -                           -                                                        */
/*                                                                                              */
/*         = 2 * (3.141592654)^1/2 * (7.29735308e-3)^4                                          */
/*                      * (0.529177249e-10)^2 * (2.99792458e8) / 3                              */
/*                                                                                              */
/*         = 2.8129897e-21                                                                      */
/*         Mathematica gives 2.4764282710571237e-21                                             */
/*                                                                                              */
/*    The photoionization cross-section (see also Burgess 1965)                                 */
/*     is given by;                                                                             */
/*                   _               _        l'=l+1                                            */
/*                  |4 PI alpha a_o^2 |  n^2   ---     max(l,l')                                */
/*      a(Z';n,l,k)=|---------------- |  ---    >       --------- Theta(n,l; K, l')             */
/*                  |       3         |  Z^2   ---     (2l + 1)                                 */
/*                   _               _        l'=l-1                                            */
/*                                                                                              */
/*                                                                                              */
/*      where  Theta(n,l; K, l') is defined above                                               */
/************************************************************************************************/
/************************************************************************************************/
/*      For the transformation:                                                                 */
/*                              Z -> rZ = Z'                                                    */
/*                                                                                              */
/*                              k -> rk = k'                                                    */
/*      then                                                                                    */
/*                                                                                              */
/*                              K -> K = K'                                                     */
/*                                                                                              */
/*      and the cross-sections satisfy;                                                         */
/*                                                1                                             */
/*                               a(Z'; n,l,k') = --- a(Z; n,l,k)                                */
/*                                               r^2                                            */
/*                                                                                              */
/*                                                                                              */
/*                                                                                              */
/*      Similiarly, the recombination co-efficient satisfies                                    */
/************************************************************************************************/
/************************************************************************************************/


static double power( double x, long int n );
static double local_product( double K , long int lp );
static double log10_factorial( long int n );
static double   log10_prodxx( long int lp, double Ksqrd );

/****************************************************************************************/
/****************************************************************************************/
/* hv calculates photon energy in ergs for n -> n' transitions for H and H-like ions    */
/*  In the following, we have n > n'                                                    */
/****************************************************************************************/
/****************************************************************************************/
static double hv(
                 /* returns energy in ergs */
                 /* principal quantum number, 1 for ground, upper level     */
                 long int n,
                 /* principal quantum number, 1 for ground, lower level     */
                 long int nprime,
                 long int iz 
                 );

/********************************************************************************/
/*  In the following, we have n > n'                                            */
/********************************************************************************/
/* LOG VERSION ALSO PRESENT */
static double fsff(
                   /* principal quantum number, 1 for ground, upper level       */
                   long int n,
                   /* angular momentum, 0 for s                                 */
                   long int l,
                   /* principal quantum number, 1 for ground, lower level       */
                   long int np
                   );

static double log10_fsff(
                         /* principal quantum number, 1 for ground, upper level       */
                         long int n,
                         /* angular momentum, 0 for s                                 */
                         long int l,
                         /* principal quantum number, 1 for ground, lower level       */
                         long int np
                         );
    

/****************************************************************************/
/*   Calculates the Einstein A's for hydrogen                           */
static double H_Einstein_A_lin(/*  IN THE FOLLOWING WE HAVE  n > n'                        */
                               /* principal quantum number, 1 for ground, upper level      */
                               long int n,
                               /* angular momentum, 0 for s                                */
                               long int l,
                               /* principal quantum number, 1 for ground, lower level      */
                               long int np,
                               /* angular momentum, 0 for s                                */
                               long int lp,
                               /* Nuclear charge, 1 for H+, 2 for He++, etc                */
                               long int iz
                               );

/************************************************************************/
/*  IN THE FOLLOWING WE HAVE  n > n'                                    */
/************************************************************************/
/* returns hydrogenic photoionization cross section in cm-2             */
/* this routine is called by H_photo_cs when n is small */
static double H_photo_cs_lin(
                             /* photon energy relative to threshold energy         */
                             double rel_photon_energy,
                             /* principal quantum number, 1 for ground             */
                             long int n,
                             /* angular momentum, 0 for s                          */
                             long int l,
                             /* charge, 1 for H+, 2 for He++, etc                  */
                             long int iz );

/******************************************************************************
 ******************************************************************************
 *   F21()                                                                
 *   Calculates the Hyper_Spherical_Function 2_F_1(a,b,c;y)                
 *   Here a,b, and c are (long int)                                       
 *   y is of type (double)                                            
 *   A is of type (char) and specifies whether the recursion is over      
 *   a or b. It has values A='a' or A='b'.                                
 ******************************************************************************
 ******************************************************************************/
static double F21(
                  long int a,
                  long int b,
                  long int c,
                  double y,
                  char A
                  );

/********************************************************************************/
/********************************************************************************/
/*   F21_mx()                                                                      
*/
/*   Calculates the Hyper_Spherical_Function 2_F_1(a,b,c;y)                     
*/
/*   Here a,b, and c are (long int)                                             
*/
/*   y is of type (double)                                                      
*/
/*   A is of type (char) and specifies whether the recursion is over            
*/
/*   a or b. It has values A='a' or A='b'.                                      
*/
/********************************************************************************/
/********************************************************************************/
#define yV(I_)        (*(yV+I_))

static double F21i( long int a , long int b , long int c , double y , double *yV );

struct mx
{
  double        m;
  long int      x;
};
                
typedef struct mx mx;


struct mxq
{
  struct mx    mx;
  long int      q;  
};
                    
typedef struct mxq mxq;
                    
                    
static mx normalize_mx( mx target );
static mx add_mx( mx a, mx b );
static mx sub_mx( mx a, mx b );
static mx mxify( double a );
static mx mxify_log10( double log10_a );
static mx mult_mx( mx a, mx b );
                

static double unmxify( mx a_mx );

static mx F21_mx(
              long int a,
              long int b,
              long int c,
              double y,
              char A
              );

/******************************************************************************/
/*  In the following, we have n > n'                                          */
/******************************************************************************/
static double hrii(
                   /* principal quantum number, 1 for ground, upper level     */
                   long int n,
                   /* angular momentum, 0 for s                               */ 
                   long int l,
                   /* principal quantum number, 1 for ground, lower level     */
                   long int np,
                   /* angular momentum, 0 for s                               */
                   long int lp
                   );

static double hrii_log(
                    /* principal quantum number, 1 for ground, upper level       */
                    long int n,
                    /* angular momentum, 0 for s                                 */
                    long int l,
                    /* principal quantum number, 1 for ground, lower level       */
                    long int np,
                    /* angular momentum, 0 for s                                 */
                    long int lp
                    );


/************************************************************************************************/
/************************************************************************************************/
/*  Reentrant version of lgamma i.e. log(gamma(x)).  This function uses the global variable     */
/*  `signgam'.  The reentrant version instead takes a pointer and stores the value through it.  */
/*  extern   long double  lgammal_r(long double  , int *__signgamp)                             */
/************************************************************************************************/
/************************************************************************************************/
# define My_LOG10E      (0.43429448190325182765)          /* log_10 e
*/  
# define My_SQRT2       (1.41421356237309504880)          /* sqrt(2)            
*/
# define My_SQRTPI      (1.77245385090551602729)          /* sqrt(pi)         
*/
# define My_SQRTPIBY2   ((My_SQRTPI) / (My_SQRT2))      /* sqrt(pi)/sqrt(2)     
*/

/*# define ZerO           (0.0)*/
# define OnE            (1.0)
# define TwO            (2.0)
# define ThreE          (3.0)
# define FouR           (4.0)
# define EighT          (8.0)
# define TeN		(10.0)

#define recV(I_)          (*(rcsvV+(I_)))
#define recV_mxq(I_)          (*(rcsvV_mxq+(I_)))


/************************************************************************************************/
/* pre-calculated factorials                                                                    */
/************************************************************************************************/
#define NPRE_FACTORIAL 171

/*static double cs_vs_l[NPRE_FACTORIAL];*/
static const double pre_factorial[NPRE_FACTORIAL]=
{
  1.00000000000000000000e+00,
  1.00000000000000000000e+00,
  2.00000000000000000000e+00,
  6.00000000000000000000e+00,
  2.40000000000000000000e+01,
  1.20000000000000000000e+02,
  7.20000000000000000000e+02,
  5.04000000000000000000e+03,
  4.03200000000000000000e+04,
  3.62880000000000000000e+05,
  3.62880000000000000000e+06,
  3.99168000000000000000e+07,
  4.79001600000000000000e+08,
  6.22702080000000000000e+09,
  8.71782912000000000000e+10,
  1.30767436800000000000e+12,
  2.09227898880000000000e+13,
  3.55687428096000000000e+14,
  6.40237370572800000000e+15,
  1.21645100408832000000e+17,
  2.43290200817664000000e+18,
  5.10909421717094400000e+19,
  1.12400072777760768000e+21,
  2.58520167388849766400e+22,
  6.20448401733239439360e+23,
  1.55112100433309859840e+25,
  4.03291461126605635592e+26,
  1.08888694504183521614e+28,
  3.04888344611713860511e+29,
  8.84176199373970195470e+30,
  2.65252859812191058647e+32,
  8.22283865417792281807e+33,
  2.63130836933693530178e+35,
  8.68331761881188649615e+36,
  2.95232799039604140861e+38,
  1.03331479663861449300e+40,
  3.71993326789901217463e+41,
  1.37637530912263450457e+43,
  5.23022617466601111726e+44,
  2.03978820811974433568e+46,
  8.15915283247897734264e+47,
  3.34525266131638071044e+49,
  1.40500611775287989839e+51,
  6.04152630633738356321e+52,
  2.65827157478844876773e+54,
  1.19622220865480194551e+56,
  5.50262215981208894940e+57,
  2.58623241511168180614e+59,
  1.24139155925360726691e+61,
  6.08281864034267560801e+62,
  3.04140932017133780398e+64,
  1.55111875328738227999e+66,
  8.06581751709438785585e+67,
  4.27488328406002556374e+69,
  2.30843697339241380441e+71,
  1.26964033536582759243e+73,
  7.10998587804863451749e+74,
  4.05269195048772167487e+76,
  2.35056133128287857145e+78,
  1.38683118545689835713e+80,
  8.32098711274139014271e+81,
  5.07580213877224798711e+83,
  3.14699732603879375200e+85,
  1.98260831540444006372e+87,
  1.26886932185884164078e+89,
  8.24765059208247066512e+90,
  5.44344939077443063905e+92,
  3.64711109181886852801e+94,
  2.48003554243683059915e+96,
  1.71122452428141311337e+98,
  1.19785716699698917933e+100,
  8.50478588567862317347e+101,
  6.12344583768860868500e+103,
  4.47011546151268434004e+105,
  3.30788544151938641157e+107,
  2.48091408113953980872e+109,
  1.88549470166605025458e+111,
  1.45183092028285869606e+113,
  1.13242811782062978295e+115,
  8.94618213078297528506e+116,
  7.15694570462638022794e+118,
  5.79712602074736798470e+120,
  4.75364333701284174746e+122,
  3.94552396972065865030e+124,
  3.31424013456535326627e+126,
  2.81710411438055027626e+128,
  2.42270953836727323750e+130,
  2.10775729837952771662e+132,
  1.85482642257398439069e+134,
  1.65079551609084610774e+136,
  1.48571596448176149700e+138,
  1.35200152767840296226e+140,
  1.24384140546413072522e+142,
  1.15677250708164157442e+144,
  1.08736615665674307994e+146,
  1.03299784882390592592e+148,
  9.91677934870949688836e+149,
  9.61927596824821198159e+151,
  9.42689044888324774164e+153,
  9.33262154439441526381e+155,
  9.33262154439441526388e+157,
  9.42594775983835941673e+159,
  9.61446671503512660515e+161,
  9.90290071648618040340e+163,
  1.02990167451456276198e+166,
  1.08139675824029090008e+168,
  1.14628056373470835406e+170,
  1.22652020319613793888e+172,
  1.32464181945182897396e+174,
  1.44385958320249358163e+176,
  1.58824554152274293982e+178,
  1.76295255109024466316e+180,
  1.97450685722107402277e+182,
  2.23119274865981364576e+184,
  2.54355973347218755612e+186,
  2.92509369349301568964e+188,
  3.39310868445189820004e+190,
  3.96993716080872089396e+192,
  4.68452584975429065488e+194,
  5.57458576120760587943e+196,
  6.68950291344912705515e+198,
  8.09429852527344373681e+200,
  9.87504420083360135884e+202,
  1.21463043670253296712e+205,
  1.50614174151114087918e+207,
  1.88267717688892609901e+209,
  2.37217324288004688470e+211,
  3.01266001845765954361e+213,
  3.85620482362580421582e+215,
  4.97450422247728743840e+217,
  6.46685548922047366972e+219,
  8.47158069087882050755e+221,
  1.11824865119600430699e+224,
  1.48727070609068572828e+226,
  1.99294274616151887582e+228,
  2.69047270731805048244e+230,
  3.65904288195254865604e+232,
  5.01288874827499165889e+234,
  6.91778647261948848943e+236,
  9.61572319694108900019e+238,
  1.34620124757175246000e+241,
  1.89814375907617096864e+243,
  2.69536413788816277557e+245,
  3.85437071718007276916e+247,
  5.55029383273930478744e+249,
  8.04792605747199194159e+251,
  1.17499720439091082343e+254,
  1.72724589045463891049e+256,
  2.55632391787286558753e+258,
  3.80892263763056972532e+260,
  5.71338395644585458806e+262,
  8.62720977423324042775e+264,
  1.31133588568345254503e+267,
  2.00634390509568239384e+269,
  3.08976961384735088657e+271,
  4.78914290146339387432e+273,
  7.47106292628289444390e+275,
  1.17295687942641442768e+278,
  1.85327186949373479574e+280,
  2.94670227249503832518e+282,
  4.71472363599206132029e+284,
  7.59070505394721872577e+286,
  1.22969421873944943358e+289,
  2.00440157654530257674e+291,
  3.28721858553429622598e+293,
  5.42391066613158877297e+295,
  9.00369170577843736335e+297,
  1.50361651486499903974e+300,
  2.52607574497319838672e+302,
  4.26906800900470527345e+304,
  7.25741561530799896496e+306
  /*,1.24101807021766782298e+309*/
};

/************************************************************************************************/
/************************************************************************************************/
/*GaussianQuadrature                                                                            */
/*                                                                                              */
/*                                                                                              */
/*  The Mathematica function NIntegrate uses as one of its methods a fairly sophisticated       */
/*  Gauss-Kronrod-based algorithm. The package NumericalMath`GaussianQuadrature' allows         */
/*  you to easily study some of the theory behind ordinary Gaussian quadrature which is a       */
/*  little less sophisticated. The basic idea behind Gaussian quadrature is to approximate      */
/*  the value if an integral as a linear combination of values of the integrand evaluated       */
/*  at specific points:                                                                         */
/*                                                                                              */
/*  Integral(_a,^b){ f(x) }dx = Sum(i = 1 to n) w_i * f(x_i)                                    */
/*                                                                                              */
/*  Since there are 2n free parameters to be chosen (both the abscissas "x_i"  and the          */
/*  weights "w_i" and since both integration and the sum are linear operations, we can          */
/*  expect to be able to make the formula correct for all polynomials of degree less than       */
/*  about 2n. In addition to knowing what the optimal abscissas and weights are, it is          */
/*  often desirable to know how large the error in the approximation will be. This package      */
/*  allows you to answer both of these questions.                                               */
/*                                                                                              */
/*  THE FOLLOWING VAULES ARE FOR THE INTERVAL [0,1] AND NEED TO BE ADJUSTED FOR DIFFERENT       */
/*  INTERVALS.                                                                                  */
/************************************************************************************************/
/************************************************************************************************/
const double GQ32x[32]={
  0.00136806907525921822751,
  0.00719424422736583229991,
  0.01761887220624678461309,
  0.03254696203113015541454,
  0.05183942211697393801735,
  0.07531619313371501493315,
  0.10275810201602879651845,
  0.13390894062985515980629,
  0.16847786653489239951244,
  0.20614212137961883547962,
  0.24655004553388530498812,
  0.28932436193468232731794,
  0.33406569885893617511004,
  0.38035631887393146272769,
  0.42776401920860175325740,
  0.47584616715613084188259,
  0.52415383284386915811740,
  0.57223598079139824674259,
  0.61964368112606853727230,
  0.66593430114106382488995,
  0.71067563806531767268206,
  0.75344995446611469501187,
  0.79385787862038116452037,
  0.83152213346510760048756,
  0.86609105937014484019371,
  0.89724189798397120348155,
  0.92468380686628498506685,
  0.94816057788302606198265,
  0.96745303796886984458546,
  0.98238112779375321538691,
  0.99280575577263416770009,
  0.99863193092474078177249 
};
  
const double GQ32y[32] = {  
  0.00350930500473504830799,
  0.00813719736545283531661,
  0.01269603265463102974400,
  0.01713693145651071657955,
  0.02141794901111334034876,
  0.02549902963118808811936,
  0.02934204673926777360103,
  0.03291111138818092342255,
  0.03617289705442425309822,
  0.03909694789353515322022,
  0.04165596211347337759442,
  0.04382604650220190555384,
  0.04558693934788194233818,
  0.04692219954040228280081,
  0.04781936003963742969040,
  0.04827004425736390026406,
  0.04827004425736390026406,
  0.04781936003963742969040,
  0.04692219954040228280081,
  0.04558693934788194233818,
  0.04382604650220190555384,
  0.04165596211347337759442,
  0.03909694789353515322022,
  0.03617289705442425309822,
  0.03291111138818092342255,
  0.02934204673926777360103,
  0.02549902963118808811936,
  0.02141794901111334034876,
  0.01713693145651071657955,
  0.01269603265463102974400,
  0.00813719736545283531661,
  0.00350930500473504830799 
};


/************************************************************************************************/
/************************************************************************************************/
/*  *** nl2index take the princpal quantum number n (i.e. shell number)                         */
/*  *** and orbital angular momentum number l (i.e s, p, d, f, ...)                             */
/*  *** and generates  an index nl2index                                                        */
/*  ***                      n  - shell number   L-angular momentum                             */
/*  ***  ***  nl2index=0 for 1                   s                                              */
/*  ***  ***          =1     2                   s                                              */
/*  ***  ***          =2     2                   p                                              */
/*  ***  ***          =3     3                   s                                              */
/*  ***  ***          =4     3                   p                                              */
/*  ***  ***          =5     3                   d                                              */
/*  ***  ***          ...... and so on                                                          */
/*************************************************************************************************/
/*************************************************************************************************/


/************************************************************************/
/************************************************************************/
/* (4.0/3.0) * PI * FINE_STRUCTURE_CONSTANT * BORHRADIUS * BORHRADIUS   */
/*                                                                      */
/*                                                                      */
/*      4 PI alpha a_o^2                                                */
/*      ----------------                                                */
/*             3                                                        */
/*                                                                      */
/*      where   alpha = Fine Structure Constant                         */
/*              a_o   = Bohr Radius                                     */
/*                                                                      */
/*      = 3.056708^-02 (au Length)^2                                    */
/*      = 8.56x10^-23 (meters)^2                                        */
/*      = 8.56x10^-19 (cm)^2                                            */
/*      = 8.56x10^+05 (barns)                                           */
/*      = 0.856 (MB or megabarns)                                       */
/*                                                                      */
/*                                                                      */
/*      1 barn = 10^-28 (meter)^2                                       */
/************************************************************************/
/************************************************************************/

static double  bh(
                  double k,
                  long int n,
                  long int l,
                  double *rcsvV
                  );

static double  bh_log(
                   double k,
                   long int n,
                   long int l,
                   mxq *rcsvV_mxq
                   );

static double bhintegrand(
                          double k,
                          long int n,
                          long int l,
                          long int lp,
                          double *rcsvV
                          );

static double bhintegrand_log( 
                           double k,
                           long int n,
                           long int l,
                           long int lp,
                           mxq *rcsvV_mxq
                           );
  
static double bhG(
                  double K,
                  long int n,
                  long int l,
                  long int lp,
                  double *rcsvV
                  );

static mx bhG_mx(
                 double K,  
                 long int n,
                 long int l,
                 long int lp,
                 mxq *rcsvV_mxq
                );
                    
static double bhGp(
                   long int q,
                   double K,  
                   long int n,
                   long int l,
                   long int lp,
                   double *rcsvV,
                   double GK
                   );

static mx bhGp_mx(
                  long int q,
                  double K,
                  long int n,
                  long int l,
                  long int lp,
                  mxq *rcsvV_mxq,
                  mx  GK_mx
                  );
                        
static double bhGm(
                   long int q,
                   double K,
                   long int n,
                   long int l,
                   long int lp,
                   double *rcsvV,
                   double GK   
                   );
                    
static mx bhGm_mx(
                  long int q,
                  double K,
                  long int n,
                  long int l,
                  long int lp,
                  mxq *rcsvV_mxq,
                  mx   GK_mx
                  );


static double bhg( 
                  double K,   
                  long int n, 
                  long int l, 
                  long int lp,
                  double *rcsvV
                  );

static double bhg_log(
                   double K,
                   long int n,
                   long int l,
                   long int lp,
                   mxq *rcsvV_mxq
                   );


/************************Start of program***************************/

double H_photo_cs(
                  /* incident photon energy relative to threshold       */
                  double rel_photon_energy,
                  /* principal quantum number, 1 for ground             */
                  long int n,
                  /* angular momentum, 0 for s                          */
                  long int l,
                  /* charge, 1 for H+, 2 for He++, etc                  */
                  long int iz )
{ 
  double result;
  if( n<= 25 )
    {
      result = H_photo_cs_lin( rel_photon_energy , n , l , iz );
    }
  else
    {
      result = H_photo_cs_log10( rel_photon_energy , n , l , iz );
    } 
  return( result );
}

/************************************************************************/
/*  IN THE FOLLOWING WE HAVE  n > n'                                    */
/************************************************************************/
                   
/* returns hydrogenic photoionization cross section in cm-2             */
/* this routine is called by H_photo_cs when n is small */
static double H_photo_cs_lin(
                             /* photon energy relative to threshold energy         */
                             double rel_photon_energy,
                             /* principal quantum number, 1 for ground             */
                             long int n,
                             /* angular momentum, 0 for s                          */
                             long int l,
                             /* charge, 1 for H+, 2 for He++, etc                  */
                             long int iz )
{    
#       define PHYSICAL_CONSTANT_TWO (8.56e-19)         /*cm^2*/
  long int dim_rcsvV;

  /* >>chng 02 sep 15, make rcsvV always NPRE_FACGTORIAL+3 long */
  /*double *rcsvV;*/
  double rcsvV[NPRE_FACTORIAL+3];
  int i;

  double electron_energy;
  double
    result =  ZerO,
    xn_sqrd  = (double) (n*n),
    z_sqrd = (double) (iz*iz),
    Z      = (double) iz,
    K      = ZerO,                              /* K = k / Z                            */
    k      = ZerO;                             /* k^2 = ejected-electron-energy (Ryd)  */

#if 0
  if ( iz==1 )
    {
      /* for charge of one use the proton mass rydberg constant */
      ryd = HIONPOT;
    }
  else
    {
      /* for all heavier, use infinite mass rydberg constant */
      ryd = 1.; 
    }
  ryd = 1.;
#endif
  
  /* expressions blow up at precisely threshold */
  if ( rel_photon_energy < 1.+FLT_EPSILON )
    {
      /* below or very close to threshold, return zero */
      return( 0. );
    }
    
  if ( n < 1 || l >= n )     
    {
      fprintf(ioQQQ," The quantum numbers are impossible.\n");
      (void) puts( "[Stop in H_photo_cs_lin]" );
      exit(EXIT_FAILURE);
    }
  
  if ( ((2*n) - 1) >= NPRE_FACTORIAL )
    {
      fprintf(ioQQQ," This value of n is too large.\n");
      (void) puts( "[Stop in H_photo_cs_lin]" );
      exit(EXIT_FAILURE);
    }
  
  /* k^2 is the ejected photoelectron energy in ryd */
  /*electron_energy = MAX2( SMALLFLOAT , (photon_energy/ryd) - (z_sqrd/xn_sqrd) );*/
      
  electron_energy = (rel_photon_energy-1.) * (z_sqrd/xn_sqrd);
  k = sqrt( ( electron_energy ) );
    
  K = (k/Z);
     
  dim_rcsvV = (((n * 2) - 1) + 1);
      
  /* >>chng 02 sep 15, make rcsvV always NPRE_FACTORIAL+3 long */
  /* create space 
  if ( (rcsvV = (double*)CALLOC( (size_t)dim_rcsvV, sizeof(double) ) )==NULL )
    exit(EXIT_FAILURE);*/
  for( i=0; i<dim_rcsvV; ++i )
  {
	  rcsvV[i] = 0.;
  }
  
  /* rcsvV contains all results for quantum indices below n, l */
  result = (PHYSICAL_CONSTANT_TWO) * (xn_sqrd/z_sqrd) * bh( K, n, l, rcsvV );
      
  assert( result != ZerO );
  /* >>chng 02 sep 15, make rcsvV always NPRE_FACGTORIAL+3 long */
  /*free( rcsvV );*/
     
  return( result );
}

/*****************************************************************************/
/*H_photo_cs_log10 returns hydrogenic photoionization cross section in cm-2
 * this routine is called by H_photo_cs when n is large */
/*****************************************************************************/
double H_photo_cs_log10(
                        /* photon energy relative to threshold energy         */
                        double rel_photon_energy,
                        /* principal quantum number, 1 for ground            */
                        long int n,
                        /* angular momentum, 0 for s                         */
                        long int l,
                        /* charge, 1 for H+, 2 for He++, etc                 */
                        long int iz
                        )
{
#if 0
#       define PHYSICAL_CONSTANT_TWO (8.56e-19)         /*cm^2*/
#endif
  long int dim_rcsvV_mxq;
  
  mxq *rcsvV_mxq = NULL;
     
  double electron_energy;
  double
    result   =  ZerO,
    xn_sqrd  = (double) (n*n),
    z_sqrd   = (double) (iz*iz),
    Z        = (double) iz,
    K        = ZerO,                       /* K = k / Z                            */
    k        = ZerO;                      /* k^2 = ejected-electron-energy (Ryd)  */

#if 0
  if ( iz==1 )
    {
      /* for charge of one use the proton mass rydberg constant */
      ryd = HIONPOT;
    }
  else
    {
      /* for all heavier, use infinite mass rydberg constant */
      ryd = 1.;
    }
#endif
  /* expressions blow up at precisely threshold */
  if ( rel_photon_energy < 1.+FLT_EPSILON )
    {
      /* below or very close to threshold, return zero */
      return( 0. );
    }
  if ( n < 1 || l >= n )
    {
      fprintf(ioQQQ," The quantum numbers are impossible.\n");
      puts( "[Stop in H_photo_cs_log10]" );
      exit(EXIT_FAILURE);
    }                   
  
     
  /* k^2 is the ejected photoelectron energy in ryd */
  /*electron_energy = MAX2( SMALLFLOAT , (photon_energy/ryd) - (z_sqrd/xn_sqrd) );*/
  electron_energy = (rel_photon_energy-1.) * (z_sqrd/xn_sqrd);
  
  k = sqrt( ( electron_energy ) );
  /* k^2 is the ejected photoelectron energy in ryd */
  /*k = sqrt( ( (photon_energy/ryd) - (z_sqrd/xn_sqrd) ) );*/
     
  K = (k/Z);
  
  dim_rcsvV_mxq = (((n * 2) - 1) + 1);
     
  /* create space */
  if ( (rcsvV_mxq = (mxq*)CALLOC( (size_t)dim_rcsvV_mxq, sizeof(mxq) ) )==NULL )
    exit(EXIT_FAILURE);
    
  result = (PHYSICAL_CONSTANT_TWO) * (xn_sqrd/z_sqrd) * bh_log( K, n, l, rcsvV_mxq );
     
  free( rcsvV_mxq );
      
  return( result );
}

static double bh(
                 /* K = k / Z ::: k^2 = ejected-electron-energy (Ryd)    */
                 double K,
                 /* principle quantum number                             */
                 long int n,
                 /* angular  momentum quantum number                     */
                 long int l,
                 /* Temperary storage for intermediate                   */
                 /*  results of the recursive routine                    */
                 double *rcsvV
                 )
{
  /* l' */
  long int
    lp = 0;
      
  /*      Sum in brocklehurst eq. 3.13 */
  double
    sigma = ZerO; 
                  

  /*      Some precautionary debug stuff */
  assert( n > 0);
  assert( l >= 0);
  assert( n > l);
                 
                 
  if ( l > 0 ) /* no lp=(l-1)  for  l=0 */
    {
      for ( lp = l - 1 ; lp <= l + 1 ; lp = lp + 2 )
        {
          sigma += bhintegrand( K, n, l, lp, rcsvV );
        } 
    }
  else
    { 
      lp = l + 1;
      sigma = bhintegrand( K, n, l, lp, rcsvV );
    }
  assert( sigma != ZerO );

  return( sigma );
}

static double bh_log(
                  /* K = k / Z ::: k^2 = ejected-electron-energy (Ryd)    */
                  double K,
                  /* principle quantum number                             */
                  long int n,
                  /* angular  momentum quantum number                     */
                  long int l,
                  /* Temperary storage for intermediate                   */
                  mxq *rcsvV_mxq
                  /*   results of the recursive routine                   */
                  )
{
  /* l' */
  long int
    lp = 0;
  
        /*      Sum in brocklehurst eq. 3.13 */
  double
    sigma = ZerO;
  
 
        /*      Some precautionary debug stuff */
  assert( n > 0);
  assert( l >= 0);
  assert( n > l);
                  
                  
  if ( l > 0 ) /* no lp=(l-1)  for  l=0 */
    {
      for ( lp = l - 1 ; lp <= l + 1 ; lp = lp + 2 )
        {
          sigma += bhintegrand_log( K, n, l, lp, rcsvV_mxq );
        }
    }
  else
    {
      lp = l + 1;
      sigma = bhintegrand_log( K, n, l, lp, rcsvV_mxq );
    }
  assert( sigma != ZerO );
        
  return( sigma );
}

  
/********************************************************************************/
/********************************************************************************/
/*      Here we calculate the integrand                                         */
/*      (as a function of K, so                                                 */
/*      we need a dK^2 -> 2K dK )                                               */
/*      for  equation 3.14 of reference                                         */
/*                                                                              */
/*      M. Brocklehurst Mon. Note. R. astr. Soc. (1971) 153, 471-490            */
/*                                                                              */
/*      namely:                                                                 */
/*                                                                              */
/*      max(l,l')  (1 + n^2 K^2)^2 Theta(n,l; K, l') exp( -K^2 y ) d(K^2)       */
/*                                                                              */
/*      Note: the "y" is included in the code that called                       */
/*      this function and we include here the n^2 from eq 3.13.                 */
/********************************************************************************/
/********************************************************************************/
  
static double bhintegrand(
                          /* K = k / Z ::: k^2 = ejected-electron-energy (Ryd) */
                          double K,
                          long int n,
                          long int l,
                          long int lp,
                          /* Temperary storage for intermediate         */
                          /*  results of the recursive routine          */
                          double *rcsvV
                          )
{
  double
    Two_L_Plus_One = (double)((2*l) + 1),
    lg = (double)(l > lp ? l : lp);

  double
    n2 = (double)(n*n);

  double
    Ksqrd = (K*K);
                          
                          
  /**********************************************/
  /**********************************************/
  /*                                            */
  /*    l>                                      */
  /*  ------    Theta(nl,Kl')                   */
  /*   2l+2                                     */
  /*                                            */
  /*                                            */
  /* Theta(nl,Kl') =                            */
  /*   (1+n^2K^2) * | g(nl,Kl')|^2              */
  /*                                            */
  /**********************************************/
  /**********************************************/
  double
    d2 = ( OnE + n2 * (Ksqrd) ),
    d5 = bhg( K, n, l, lp, rcsvV ),
    Theta = (d2 * d5 * d5),
    d7 = (lg/Two_L_Plus_One) * Theta;
                          
  assert( Two_L_Plus_One != ZerO );
  assert( Theta != ZerO );
  assert( Ksqrd != ZerO );
  assert( d2 != ZerO );
  assert( d5 != ZerO );
  assert( d7 != ZerO );
  assert( lp >= 0 );
  assert( lg != ZerO );
  assert( n2 != ZerO );
  assert( n  > 0  );
  assert( l  >= 0 );
  assert( K  != ZerO );
 
  return( d7 );
}

/************************************************************************************************/
/************************************************************************************************/
/*    The photoionization cross-section (see also Burgess 1965)                                 */
/*     is given by;                                                                             */
/*                   _               _        l'=l+1                                            */
/*                  |4 PI alpha a_o^2 |  n^2   ---     max(l,l')                                */
/*      a(Z';n,l,k)=|---------------- |  ---    >     ---------- Theta(n,l; K, l')              */
/*                  |       3         |  Z^2   ---     (2l + 1)                                 */
/*                   _               _        l'=l-1                                            */
/*                                                                                              */
/*                                                                                              */
/*      where  Theta(n,l; K, l') is defined                                                     */
/*                                                                                              */
/*                                           |              |^2                                 */
/*         Theta(n,l; K, l') = (1 + n^2 K^2) | g(n,l; K,l') |                                   */
/*                                           |              |                                   */
/*                                                                                              */
/*                                                                                              */
/*                                                 ---- Not sure if this is K or k              */
/*                                  OO            /     but think it is K                       */
/*         where                    -            v                                              */
/*                         K^2     |                                                            */
/*         g(n,l; K,l') = -----    | R_nl(r) F_(K,l) r dr                                       */
/*                         n^2     |                                                            */
/*                                -                                                             */
/*                                0                                                             */
/************************************************************************************************/
/************************************************************************************************/
static double bhintegrand_log(
                           double K,     /* K = k / Z ::: k^2 = ejected-electron-energy (Ryd) */  
                           long int n,
                           long int l,
                           long int lp,
                           /* Temperary storage for intermediate         */
                           /* results of the recursive routine           */
                           mxq *rcsvV_mxq
                           )
{
  double
    d2 = ZerO,
    d5 = ZerO,
    d7 = ZerO,
    Theta = ZerO,
    n2 = (double)(n*n),
    Ksqrd = (K*K),
    Two_L_Plus_One = (double)(2*l + 1),
    lg = (double)(l > lp ? l : lp);

  assert( Ksqrd != ZerO );
  assert( K != ZerO );
  assert( lg != ZerO );
  assert( n2 != ZerO );
  assert( Two_L_Plus_One != ZerO );
                           
                           
  assert( n > 0);
  assert( l >= 0);
  assert( lp >= 0);
                           
  /**********************************************/
  /**********************************************/
  /*                                            */
  /*    l>                                      */
  /*  ------    Theta(nl,Kl')                   */
  /*   2l+2                                     */
  /*                                            */
  /*                                            */
  /* Theta(nl,Kl') =                            */
  /*   (1+n^2K^2) * | g(nl,Kl')|^2              */
  /*                                            */
  /**********************************************/
  /**********************************************/
  d2 = ( OnE + n2 * (Ksqrd) );
  
  assert( d2 != ZerO );
  
  d5 = bhg_log( K, n, l, lp, rcsvV_mxq );
                           
  assert( d5 != ZerO );    
  
  Theta = d2 * d5 * d5;
  assert( Theta != ZerO );
                           
  d7 = (lg/Two_L_Plus_One) * Theta;
  
  assert( d7 != ZerO );
  
  return( d7 );
}

/****************************************************************************************/
/****************************************************************************************/
/*  *** bhG ***                                                                         */
/*    Using various recursion relations                                                 */
/*    (for l'=l+1)                                                                      */
/*    equation: (3.23)                                                                  */
/*    G(n,l-2; K,l-1) = [ 4n^2-4l^2+l(2l-1)(1+(n K)^2) ] G(n,l-1; K,l)                  */
/*                      - 4n^2 (n^2-l^2)[1+(l+1)^2 K^2] G(n,l; K,l+1)                   */
/*                                                                                      */
/*    and (for l'=l-1)                                                                  */
/*    equation: (3.24)                                                                  */
/*    G(n,l-1; K,l-2) = [ 4n^2-4l^2 + l(2l-1)(1+(n K)^2) ] G(n,l; K,l-1)                */
/*                      - 4n^2 (n^2-(l+1)^2)[ 1+(lK)^2 ] G(n,l; K,l+1)                  */
/*                                                                                      */
/*    the starting point for the recursion relations are;                               */
/*    equation: (3.18)                                                                  */
/*                     | pi |(1/2)   8n                                                 */
/*     G(n,n-1; 0,n) = | -- |      ------- (4n)^n exp(-2n)                              */
/*                     | 2  |      (2n-1)!                                              */
/*                                                                                      */
/*     equation: (3.20)                                                                 */
/*                             exp(2n-2/K tan^(-1)(n K)                                 */
/*     G(n,n-1; K,n) =  -----------------------------------------  *  G(n,n-1; 0,n)     */
/*                      sqrt(1 - exp(-2 pi K)) * (1+(n K)^2)^(n+2)                      */
/*                                                                                      */
/*     equation: (3.20)                                                                 */
/*     G(n,n-2; K,n-1) = (2n-2)(1+(n K)^2) n G(n,n-1; K,n)                              */
/*                                                                                      */
/*     equation: (3.21)                                                                 */
/*                       (1+(n K)^2)                                                    */
/*     G(n,n-1; K,n-2) = ----------- G(n,n-1; K,n)                                      */
/*                           2n                                                         */
/*                                                                                      */
/*     equation: (3.22)                                                                 */
/*     G(n,n-2; K,n-3) = (2n-1) (4+(n-1)(1+n^2 K^2)) G(n,n-1; K,n-2)                    */
/****************************************************************************************/

static double bhG(
                  double K,
                  long int n,
                  long int l,
                  long int lp,
                  /* Temperary storage for intermediate         */
                  /* results of the recursive routine           */
                  double *rcsvV
                  )
{
  double
    /* n */
    n1 = (double)n,
    /* n^2 */
    n2 =(double)(n * n),
    /* K^2 */
    Ksqrd = K * K;

  double
    /* (2n-1)! */
    ld1 = pre_factorial[ (2*n) - 1 ],
    /*            (4n)^n                  */
    ld2 = power((double)(4*n), n),
    /*            exp(-2n)                */
    ld3 = exp(-(double)(2 * n));
                  
                  
  /******************************************************************************
   ******************************************************************************
   *    ********G0*******                              
   *                                                                            *
   *            | pi |(1/2)   8n                                                *
   *      G0 =  | -- |      ------- (4n)^n exp(-2n)                             *
   *            | 2  |      (2n-1)!                                             *
   ******************************************************************************
   ******************************************************************************/
  double
    G0 = My_SQRTPIBY2 * (EighT * n1 * ld2 * ld3) / ld1 ;
  
  double
    /*  sqrt(1 - exp(-2 pi/ K))   */ 
    d1 = sqrt( OnE - exp(( -TwO *  PI )/ K )),
    
    /*     (1+(n K)^2)^(n+2)      */
    d2 = power(( OnE + (n2 * Ksqrd)), ( n + 2 )),
                  
    /*       tan^(-1)(n K)        */
    d3 = atan( n1 * K ),
  
    /*     2/K tan^(-1)(n K)      */
    d4 = ((TwO / K) * d3),
  
    /*            2n              */
    d5 = (double) ( 2 * n ),
  
    /*  exp(2n-2/K tan^(-1)(n K)  */
    d6 = exp( d5 - (d4) ),
  
    GK = ( d6 /( d1 * d2 ) ) * G0;
    
    
    
  /* l=l'-1 or l=l'+1 */
  assert( (l == lp - 1) ||  (l == lp + 1) );
  assert( K != ZerO );
  assert( Ksqrd != ZerO );
  assert( n1 != ZerO ); 
  assert( n2 != ZerO );
  assert( ((2*n) - 1) < 1755 );
  assert( ((2*n) - 1) >= 0   );
  assert( ld1 != ZerO );
  assert( (1.0 / ld1) != ZerO );
  assert( ld3 != ZerO );
  
  assert( K != ZerO );
  assert( d1 != ZerO );   
  assert( d2 != ZerO );
  assert( d3 != ZerO );
  assert( d4 != ZerO );
  assert( d5 != ZerO );
  assert( d6 != ZerO );
      
  assert( G0 != ZerO );
  assert( GK != ZerO );
      
  /******************************************************************************/
  /******************************************************************************/
  /*    *****GK*****                                                            */
  /*                                                                            */
  /*                            exp(2n-2/K tan^(-1)(n K)                        */
  /*      G(n,n-1; K,n) =  ----------------------------------------- *  G0      */
  /*                       sqrt(1 - exp(-2 pi/ K)) * (1+(n K))^(n+2)            */
  /******************************************************************************/
  /******************************************************************************/
      
     
  /*  GENERAL CASE: l = l'-1                    */
  if ( l == lp - 1 )   
    {
      double result =  bhGm( l, K, n, l, lp, rcsvV, GK );
      /* Here the m in bhGm() refers            */
      /* to the minus sign(-) in l=l'-1         */
      return( result );
    }
  
  
  /*  GENERAL CASE:     l = l'+1                */
  else if ( l == lp + 1 )
    {
      double result =  bhGp( l, K, n, l, lp, rcsvV, GK );
      /* Here the p in bhGp() refers            */
      /* to the plus sign(+) in l=l'+1          */
      return( result );
    }
  else
    {
      exit(EXIT_FAILURE);
    }
  /* this can't happen but some lint's don't know that */
  /*lint -e527
    exit( EXIT_FAILURE ); */
  /*lint +e527 */
}

/*************log version********************************/
static mx bhG_mx(
                 double K,
                 long int n,
                 long int l,
                 long int lp,
                                /* Temperary storage for intermediate        */
                                /* results of the recursive routine          */
                 mxq *rcsvV_mxq
                 )
{
  double
    log10_GK = ZerO,
    log10_G0 = ZerO;

  double
    d1 = ZerO,
    d2 = ZerO,
    d3 = ZerO,
    d4 = ZerO,
    d5 = ZerO,
    d6 = ZerO;

  double
    ld1 = ZerO,
    ld2 = ZerO,  
    ld3 = ZerO,
    ld4 = ZerO,
    ld5 = ZerO,
    ld6 = ZerO;
                                
                                
  double
    n1 = (double) n,    /* n    */
    n2 = n1 * n1,                /* n^2  */
    Ksqrd = K * K;              /* K^2  */
    
  mx
    GK_mx={0.0,0};
  
  /* l=l'-1 or l=l'+1 */
  assert( (l == lp - 1) ||  (l == lp + 1) );
  assert( K != ZerO );
  assert( n1 != ZerO );
  assert( n2 != ZerO );
  assert( Ksqrd != ZerO );
  assert( ((2*n) - 1) >= 0   );
  
    
  /******************************/
  /*                 n          */
  /*                ---         */
  /*    log( n! ) =  >  log(j)  */
  /*                ---         */
  /*                j=1         */
  /******************************/
  
  /*************************************************************/
  /*                     | pi |(1/2)   8n                      */
  /*     G(n,n-1; 0,n) = | -- |      ------- (4n)^n exp(-2n)   */
  /*                     | 2  |      (2n-1)!                   */
  /*************************************************************/

    
  /******************************/
  /*                            */
  /*                            */
  /*      log10( (2n-1)! )      */
  /*                            */
  /*                            */
  /******************************/
  ld1 = log10_factorial( ( (2 * n) - 1) );
  assert( ld1 >= ZerO );
    
  
  
  /**********************************************/
  /*            (4n)^n                          */
  /**********************************************/
  /*    log10( 4n^n ) = n log10( 4n )           */
  /**********************************************/
  ld2 = n1 * log10( FouR * n1 );
  assert( ld2 >= ZerO );
  
  
  
  /**********************************************/
  /*            exp(-2n)                        */
  /**********************************************/
  /*    log10( exp( -2n ) ) = (-2n) * log10(e)  */
  /**********************************************/
  ld3 = (-(TwO * n1)) * (My_LOG10E);
  assert( ld3 <= ZerO );
  
  
  
  /******************************************************************************/
  /******************************************************************************/
  /*    ********G0*******                                                       */
  /*                                                                            */
  /*            | pi |(1/2)   8n                                                */
  /*      G0 =  | -- |      ------- (4n)^n exp(-2n)                             */
  /*            | 2  |      (2n-1)!                                             */
  /******************************************************************************/
  /******************************************************************************/
  log10_G0 = log10(My_SQRTPIBY2 * EighT * n1) + ( (ld2 + ld3) - ld1);
  
  
  
  /******************************************************************************/
  /******************************************************************************/

  /*    *****GK*****                                                            */
  /*                                                                            */
  /*                            exp(2n- (2/K) tan^(-1)(n K)  )                  */
  /*      G(n,n-1; K,n) =  ----------------------------------------- *  G0      */
  /*                       sqrt(1 - exp(-2 pi/ K)) * (1+(n K))^(n+2)            */
  /******************************************************************************/
  /******************************************************************************/
  
  assert( K != ZerO );
  /**********************************************/
  /*  sqrt(1 - exp(-2 pi/ K))                   */
  /**********************************************/
  /*    log10(sqrt(1 - exp(-2 pi/ K))) =        */
  /*            (1/2) log10(1 - exp(-2 pi/ K))  */
  /**********************************************/
  d1 =  (OnE - exp(-(TwO *  PI )/ K ));
  ld4 = (OnE/TwO) * log10( d1 );
  assert( K != ZerO );
  assert( d1 != ZerO );
  
 
  
  /**************************************/
  /*         (1+(n K)^2)^(n+2)          */
  /**************************************/
  /* log10( (1+(n K)^2)^(n+2) ) =       */
  /*    (n+2) log10( (1 + (n K)^2 ) )   */
  /**************************************/
  
  d2 =  ( OnE + (n2 * Ksqrd));
  ld5 = (n1 + TwO) * log10( d2 );
  assert( d2 != ZerO );
  
  assert( ld5 >= ZerO );
  
  
  /**********************************************/
  /*   exp(2n- (2/K)*tan^(-1)(n K)  )           */
  /**********************************************/
  /* log10( exp(2n- (2/K) tan^(-1)(n K)  ) =    */
  /*     (2n- (2/K)*tan^(-1)(n K) ) * Log10(e)  */
  /**********************************************/
  
  /*    tan^(-1)(n K) )         */
  d3 = atan( n1 * K );
  assert( d3 != ZerO );
  
  
  /*    (2/K)*tan^(-1)(n K) )   */
  d4 = (TwO / K) * d3;
  assert( d4 != ZerO );
  
  
  /*            2n              */
  d5 = (double) ( TwO * n1 );
  assert( d5 != ZerO );
  
  
  /*  (2n-2/K tan^(-1)(n K))    */
  d6 = d5 - d4;
  assert( d6 != ZerO );
  
  
  /* log10( exp(2n- (2/K) tan^(-1)(n K)  )      */
  ld6 = My_LOG10E * d6;
  assert( ld6 != ZerO );
  
  
  /******************************************************************************/
  /******************************************************************************/
  /*    *****GK*****                                                            */
  /*                                                                            */
  /*                            exp(2n- (2/K) tan^(-1)(n K)  )                  */
  /*      G(n,n-1; K,n) =  ----------------------------------------- *  G0      */
  /*                       sqrt(1 - exp(-2 pi/ K)) * (1+(n K))^(n+2)            */
  /******************************************************************************/
  /******************************************************************************/
  
  log10_GK = (ld6 -(ld4 + ld5)) + log10_G0;
  assert( log10_GK != ZerO );
  
  
  
  GK_mx = mxify_log10( log10_GK );
  
  /*  GENERAL CASE: l = l'-1                   */
  if ( l == lp - 1 )
    {
      mx result_mx =  bhGm_mx( l, K, n, l, lp, rcsvV_mxq , GK_mx );
      /* Here the m in bhGm() refers           */
      /* to the minus sign(-) in l=l'-1        */
      return( result_mx );
    }
  /*  GENERAL CASE:     l = l'+1               */
  else if ( l == lp + 1 )
    {
      mx result_mx =  bhGp_mx( l, K, n, l, lp, rcsvV_mxq , GK_mx );
      /* Here the p in bhGp() refers           */
      /* to the plus sign(+) in l=l'+1         */
      return( result_mx );
    }
  else
    {
      exit(EXIT_FAILURE);
    }
  /* this can't happen but some lint's don't know that*/
  /*lint -e527
    exit(EXIT_FAILURE);*/
  /*lint +e527*/
}


/************************************************************************************************/
/************************************************************************************************/
/*  ***   bhGp.c   ***                                                                          */
/*                                                                                              */
/*    Here we calculate G(n,l; K,l') with the  recursive formula                                */
/*    equation: (3.24)                                                                          */
/*                                                                                              */
/*    G(n,l-1; K,l-2) = [ 4n^2-4l^2 + l(2l+1)(1+(n K)^2) ] G(n,l; K,l-1)                        */
/*                                                                                              */
/*                      - 4n^2 (n^2-(l+1)^2)[ 1+(lK)^2 ] G(n,l+1; K,l)                          */
/*                                                                                              */
/*    Under the transformation l -> l + 1 this gives                                            */
/*                                                                                              */
/*    G(n,l+1-1; K,l+1-2) = [ 4n^2-4(l+1)^2 + (l+1)(2(l+1)+1)(1+(n K)^2) ] G(n,l+1; K,l+1-1)    */
/*                                                                                              */
/*                      - 4n^2 (n^2-((l+1)+1)^2)[ 1+((l+1)K)^2 ] G(n,l+1+1; K,l+1)              */
/*                                                                                              */
/*    or                                                                                        */
/*                                                                                              */
/*     G(n,l; K,l-1) = [ 4n^2-4(l+1)^2 + (l+1)(2l+3)(1+(n K)^2) ] G(n,l+1; K,l)                 */
/*                                                                                              */
/*                      - 4n^2 (n^2-(l+2)^2)[ 1+((l+1)K)^2 ] G(n,l+2; K,l+1)                    */
/*                                                                                              */
/*    from the reference                                                                        */
/*    M. BrocKlehurst                                                                           */
/*    Mon. Note. R. astr. Soc. (1971) 153, 471-490                                              */
/*                                                                                              */
/*                                                                                              */
/*  * This is valid for the case l=l'+1  *                                                      */
/*  * CASE:     l = l'+1                 *                                                      */
/*  * Here the p in bhGp() refers        *                                                      */
/*  * to the Plus sign(+) in l=l'+1      *                                                      */
/************************************************************************************************/
/************************************************************************************************/

static double bhGp(
                   long int q,
                   double K,
                   long int n,
                   long int l,
                   long int lp,
                   /* Temperary storage for intermediate        */
                   /*  results of the recursive routine         */
                   double *rcsvV,
                   double GK
                   )
{
  long int
    rindx = 0;

  /* static long int rcsv_Level = 1;
     printf( "bhGp(): recursion level:\t%li\n",rcsv_Level++ ); */

  assert( l == lp + 1 );
  {
    long int
      i = 0;
    if ( l > lp )  
      {
        i = 0 ;
      }
    else
      {
        i = 1;
      }
    rindx = ((2 * q) + i);
  }
                    
  if ( recV(rindx) == ZerO )
    {
      /*  SPECIAL CASE:   n = l+1 = l'+2  */
      if ( q == n - 1 )
        {
          double
            Ksqrd = (K * K),
            n2 = (double) (n*n);
   
          double
            dd1 = (double) (2 * n),
            dd2 = ( OnE + ( n2 * Ksqrd));
       
          /*                    (1+(n K)^2)                */
          /*  G(n,n-1; K,n-2) = ----------- G(n,n-1; K,n)  */
          /*                        2n                     */
          double
            G1 =  ((dd2 * GK) / dd1);
       
          assert( l == lp + 1 );
          assert( Ksqrd != ZerO );
          assert( dd1 != ZerO );
          assert( dd2 != ZerO );
          assert( G1 != ZerO );
      
          recV(rindx) = G1;
         
          return( G1 );
        }
      /*  SPECIAL CASE:   n = l+2 = l'+3  */
      else if ( q == (n - 2) )
        {
          double
            Ksqrd = (K*K),
            n2 = (double)(n*n);
          
          double
            /*                                                               */
            /* G(n,n-2; K,n-3) = (2n-1) (4+(n-1)(1+(n K)^2)) G(n,n-1; K,n-2) */
            /*                                                               */
            dd1 = (double) (2 * n),
            dd2 = ( OnE + ( n2 * Ksqrd)),
            G1 =  ((dd2 * GK) / dd1);
          
          double
            /*                                                               */
            /* G(n,n-2; K,n-3) = (2n-1) (4+(n-1)(1+(n K)^2)) G(n,n-1; K,n-2) */
            /*                                                               */
            dd3 = (double)((2 * n) - 1),
            dd4 = (double)(n - 1),
            dd5 = (FouR + (dd4 * dd2)),
            G2 =  (dd3 * dd5  * G1);
      
          assert( l == lp + 1 );
          assert( Ksqrd != ZerO );
          assert( n2 != ZerO );
          assert( dd1 != ZerO );
          assert( dd2 != ZerO );
          assert( dd3 != ZerO );
          assert( dd4 != ZerO );
          assert( dd5 != ZerO );
          assert( G1 != ZerO );
          assert( G2 != ZerO );
            
          recV(rindx) = G2;
          return( G2 );
        }
      /* The GENERAL CASE n > l + 2 */
      else
        {
          /******************************************************************************/
          /******************************************************************************/
          /*  G(n,l; K,l-1) = [ 4n^2-4(l+1)^2 + (l+1)(2l+3)(1+(n K)^2) ] G(n,l+1; K,l)  */
          /*                                                                            */
          /*                      - 4n^2 (n^2-(l+2)^2)[ 1+((l+1)K)^2 ] G(n,l+2; K,l+1)  */
          /*                                                                            */
          /*            FROM   Eq. 3.24                                                 */
          /*                                                                            */
          /*  G(n,l-1; K,l-2) = [ 4n^2-4l+^2 + l(2l+1)(1+(n K)^2) ] G(n,l; K,l-1)       */
          /*                                                                            */
          /*                      - 4n^2 (n^2-(l+1)^2)[ 1+((lK)^2 ] G(n,l+1; K,l)       */
          /******************************************************************************/
          /******************************************************************************/
          long int
            /* l+1 */
            lp1 = (q + 1),
            /* l+2 */
            lp2 = (q + 2);
         
          double
            /* K^2 */
            Ksqrd = (K*K),
          
            /* n^2*/
            n2 = (double)(n * n),
          
            /* (l+1)^2 */
            lp1s = (double) (lp1 * lp1),
          
            /* (l+2)^2 */
            lp2s = (double)( lp2 * lp2 );
          
          
          
          double
            /* 4n^2       */
            d1 = (FouR * n2),
            
            /* 4(l+1)^2    */
            d2 = (FouR * lp1s),
         
            /* (l+1)(2l+3)    */
            d3 = (double)((lp1)*((2 * q) + 3)),
            
            /* (1+(n K)^2 */
            d4 = (OnE + (n2 * Ksqrd)),
            
            /* [ 4n^2 - 4(l+1)^2 + (l+1)(2l+3)(1+(n K)^2) ] */
            d5 = (d1 - d2 + (d3 * d4)),
            d5_1 = d5 * bhGp( (q+1), K, n, l, lp, rcsvV, GK ),

          
            /*  G(n,l; K,l-1) = [ 4n^2-4(l+1)^2 + (l+1)(2l+3)(1+(n K)^2) ] G(n,l+1; K,l)   */
            /*                                                                             */
            /*                      - 4n^2 (n^2-(l+2)^2)[ 1+((l+1)K)^2 ] G(n,l+2; K,l+1)   */
          
            /* (n^2-(l+2)^2) */
            d6 = (n2 - lp2s),
            
            /* [ 1+((l+1)K)^2 ]  */
            d7 = (OnE + (lp1s * Ksqrd)),
            
            /* { 4n^2 (n^2-(l+1)^2)[ 1+((l+1) K)^2 ] } */
            d8 = (d1 * d6 * d7),
            d8_1 = d8 * bhGp( (q+2), K, n, l, lp, rcsvV, GK ),
            
            d9 = (d5_1 - d8_1);
            
          assert( l == lp + 1 );
          assert( Ksqrd != ZerO );
          assert( n2 != ZerO );
            
          assert( lp1s != ZerO );
          assert( lp2s != ZerO );
            
          assert( d1 != ZerO );
          assert( d2 != ZerO );
          assert( d3 != ZerO );
          assert( d4 != ZerO );
          assert( d5 != ZerO );
          assert( d6 != ZerO );
          assert( d7 != ZerO );
          assert( d8 != ZerO );
          assert( d9 != ZerO );
            
            
          recV(rindx) = d9;
          return( d9 );
        }
    }       
  else
    {
      assert( recV(rindx) != ZerO );
      return( recV(rindx) );
    }
}


/***********************log version*******************************/
static mx bhGp_mx(
                   long int q,
                   double K,
                   long int n,
                   long int l,
                   long int lp,
                   /* Temperary storage for intermediate       */
                   /* results of the recursive routine         */
                   mxq *rcsvV_mxq,
                   mx GK_mx
                   )
{
  long int
    rindx = 0;

  /* static long int rcsv_Level = 1;                                    */
  /*    printf( "bhGp(): recursion level:\t%li\n",rcsv_Level++ );       */

  assert( l == lp + 1 );
  {
    long int
      i = 0;
    if ( l > lp )  
      {
        i = 0 ;
      }
    else
      {
        i = 1;
      }
    rindx = ((2 * q) + i);
  }
                    
  if ( recV_mxq(rindx).q == 0 )
    {
      /*  SPECIAL CASE:   n = l+1 = l'+2 */
      if ( q == n - 1 )
        {
          /******************************************************/
          /******************************************************/
          /*                    (1+(n K)^2)                     */
          /*  G(n,n-1; K,n-2) = ----------- G(n,n-1; K,n)       */
          /*                        2n                          */
          /******************************************************/
          /******************************************************/
          double
            Ksqrd = (K * K),
            n2 = (double) (n*n);
    
          double
            dd1 = (double) (2 * n),
            dd2 = ( OnE + ( n2 * Ksqrd)),
            dd3 = dd2/dd1;
   
          mx        
            dd3_mx = mxify( dd3 ),
            G1_mx = mult_mx( dd3_mx, GK_mx);
      
          G1_mx = normalize_mx( G1_mx );
         
          
          assert( l == lp + 1 );
          assert( Ksqrd != ZerO );
          assert( n2 != ZerO );
          assert( dd1 != ZerO );
          assert( dd2 != ZerO );
          
          recV_mxq(rindx).q  = 1;
          recV_mxq(rindx).mx = G1_mx;
            
          return( G1_mx );
        }
      /*  SPECIAL CASE:   n = l+2 = l'+3 */
      else if ( q == (n - 2) )
        {
          /****************************************************************/
          /****************************************************************/
          /*                                                              */
          /* G(n,n-2; K,n-3) = (2n-1) (4+(n-1)(1+(n K)^2)) G(n,n-1; K,n-2)*/
          /*                                                              */
          /****************************************************************/
          /****************************************************************/
          /*                    (1+(n K)^2)                               */
          /*  G(n,n-1; K,n-2) = ----------- G(n,n-1; K,n)                 */
          /*                        2n                                    */
          /****************************************************************/
          /****************************************************************/
         
          double
            Ksqrd = (K*K),
            n2  = (double) (n*n),
            dd1 = (double) (2 * n),
            dd2 = ( OnE + ( n2 * Ksqrd) ),
            dd3 = (dd2/dd1),
            dd4 = (double) ((2 * n) - 1),  
            dd5 = (double) (n - 1),
            dd6 = (FouR + (dd5 * dd2)),
            dd7 = dd4 * dd6;
          
          /****************************************************************/
          /****************************************************************/
          /*                                                              */
          /* G(n,n-2; K,n-3) = (2n-1) (4+(n-1)(1+(n K)^2)) G(n,n-1; K,n-2)*/
          /*                                                              */
          /****************************************************************/
          /****************************************************************/
          
          mx
            dd3_mx = mxify( dd3 ),

            dd7_mx = mxify( dd7 ),
            G1_mx  = mult_mx( dd3_mx, GK_mx ),
            G2_mx  = mult_mx( dd7_mx, G1_mx );
            
          G2_mx = normalize_mx( G2_mx );
            
            
            
          assert( l == lp + 1 );   
          assert( Ksqrd != ZerO );
          assert( n2 != ZerO );
          assert( dd1 != ZerO );
          assert( dd2 != ZerO );
          assert( dd3 != ZerO );
          assert( dd4 != ZerO );
          assert( dd5 != ZerO );
          assert( dd6 != ZerO );
          assert( dd7 != ZerO );
          
          recV_mxq(rindx).q  = 1;
          recV_mxq(rindx).mx = G2_mx;
          return( G2_mx );
        }
      /* The GENERAL CASE n > l + 2*/
      else  
        {
          /**************************************************************************************/
          /**************************************************************************************/
          /*  G(n,l; K,l-1) = [ 4n^2-4(l+1)^2 + (l+1)(2l+3)(1+(n K)^2) ] G(n,l+1; K,l)          */
          /*                                                                                    */
          /*                      - 4n^2 (n^2-(l+2)^2)[ 1+((l+1)K)^2 ] G(n,l+2; K,l+1)          */
          /*                                                                                    */
          /*            FROM   Eq. 3.24                                                         */
          /*                                                                                    */
          /*  G(n,l-1; K,l-2) = [ 4n^2-4l+^2 + l(2l+1)(1+(n K)^2) ] G(n,l; K,l-1)               */
          /*                                                                                    */
          /*                      - 4n^2 (n^2-(l+1)^2)[ 1+((lK)^2 ] G(n,l+1; K,l)               */
          /**************************************************************************************/
          /**************************************************************************************/
          long int
            /* l+1*/
            lp1 = (q + 1),
            /* l+2*/
            lp2 = (q + 2);
         
          double
            /* K^2*/
            Ksqrd = (K * K),
            /* n^2*/
            n2 = (double)(n * n),
            /* (l+1)^2*/
            lp1s = (double) (lp1 * lp1),
            /* (l+2)^2*/
            lp2s = (double)( lp2 * lp2 );
          
          
          double
            /* 4n^2      */
            d1 = (FouR * n2),
          
            /* 4(l+1)^2*/
            d2 = (FouR * lp1s),
            
            /* (l+1)(2l+3)        */
            d3 = (double)((lp1)*((2 * q) + 3)),
            
            /* (1+(n K)^2         */
            d4 = (OnE + (n2 * Ksqrd)),
            
            /* [ 4n^2 - 4(l+1)^2 + (l+1)(2l+3)(1+(n K)^2) ]       */
            d5 = d1 - d2 + (d3 * d4),
            
            /* (n^2-(l+2)^2)      */
            d6 = (n2 - lp2s),
            
            /* [ 1+((l+1)K)^2 ]   */
            d7 = (OnE + (lp1s * Ksqrd)),
          
            /* { 4n^2 (n^2-(l+1)^2)[ 1+((l+1) K)^2 ] }    */
            d8 = (d1 * d6 * d7);
            
          /**************************************************************************************/
          /**************************************************************************************/
          /*  G(n,l; K,l-1) = [ 4n^2 - 4(l+1)^2 + (l+1)(2l+3)(1+(n K)^2) ] G(n,l+1; K,l)        */
          /*                                                                                    */
          /*                      - 4n^2 (n^2-(l+2)^2)[ 1+((l+1)K)^2 ] G(n,l+2; K,l+1)          */
          /**************************************************************************************/
          /**************************************************************************************/
            
          mx
            d5_mx=mxify( d5 ),
            d8_mx=mxify( d8 );
            
          mx
            d9_mx =mult_mx( d5_mx,  bhGp_mx( (q+1), K, n, l, lp, rcsvV_mxq, GK_mx )),
            d10_mx=mult_mx( d8_mx,  bhGp_mx( (q+2), K, n, l, lp, rcsvV_mxq, GK_mx ));
            
          mx
            result_mx = sub_mx( d9_mx, d10_mx);
          
          result_mx = normalize_mx( result_mx );
            
          assert( d1 != ZerO );
          assert( d2 != ZerO );
          assert( d3 != ZerO );
          assert( d4 != ZerO );
          assert( d5 != ZerO );
          assert( d6 != ZerO );
          assert( d7 != ZerO );
          assert( d8 != ZerO );
            
          assert( l == lp + 1 );
          assert( Ksqrd != ZerO );
          assert( n2 != ZerO );
          assert( lp1s != ZerO );
          assert( lp2s != ZerO );
            
          recV_mxq(rindx).q = 1;
          recV_mxq(rindx).mx = result_mx;
            
          return( result_mx );
        } 
    }
  else      
    {
      assert( recV_mxq(rindx).q != 0 );
      recV_mxq(rindx).q = 1;   
      return( recV_mxq(rindx).mx );
    }
}


/************************************************************************************************/
/************************************************************************************************/
/*  ***   bhGm.c  *** */
/*                                                                                              */
/*    Here we calculate G(n,l; K,l') with the  recursive formula                                */
/*    equation: (3.23)                                                                          */
/*                                                                                              */
/*    G(n,l-2; K,l-1) = [ 4n^2-4l^2 + l(2l-1)(1+(n K)^2) ] G(n,l-1; K,l)                        */
/*                                                                                              */
/*                      - 4n^2 (n^2-l^2)[ 1 + (l+1)^2 K^2 ] G(n,l; K,l+1)                       */
/*                                                                                              */
/*    Under the transformation l -> l + 2 this gives                                            */
/*                                                                                              */
/*    G(n,l+2-2; K,l+2-1) = [ 4n^2-4(l+2)^2 + (l+2)(2(l+2)-1)(1+(n K)^2) ] G(n,l+2-1; K,l+2)    */
/*                                                                                              */
/*                      - 4n^2 (n^2-(l+2)^2)[ 1 + (l+2+1)^2 K^2 ] G(n,l+2; K,l+2+1)             */
/*                                                                                              */
/*                                                                                              */
/*    or                                                                                        */
/*                                                                                              */
/*    G(n,l; K,l+1) = [ 4n^2-4(l+2)^2 + (l+2)(2l+3)(1+(n K)^2) ] G(n,l+1; K,l+2)                */
/*                                                                                              */
/*                      - 4n^2 (n^2-(l+2)^2)[ 1 + (l+3)^2 K^2 ] G(n,l+2; K,l+3)                 */
/*                                                                                              */
/*                                                                                              */
/*    from the reference                                                                        */
/*    M. BrocKlehurst                                                                           */
/*    Mon. Note. R. astr. Soc. (1971) 153, 471-490                                              */
/*                                                                                              */
/*                                                                                              */
/*  * This is valid for the case l=l'-1 *                                                       */
/*  * CASE:     l = l'-1                *                                                       */
/*  * Here the p in bhGm() refers       *                                                       */
/*  * to the Minus sign(-) in l=l'-1    *                                                       */
/************************************************************************************************/
/************************************************************************************************/

static double bhGm(
                   long int q,
                   double K,
                   long int n,
                   long int l,
                   long int lp,
                   double *rcsvV,
                   double GK
                   )
{
  long int
    rindx = 0;


  assert(l == lp - 1);
  assert(l < n);

  {
    long int
      i = 0;

    if ( l > lp )
      {
        i = 0;
      }
    else
      {
        i = 1 ;
      }
    rindx = ((2 * q) + i);  
  }
 
  if ( recV(rindx) == ZerO )
    {
      /*  CASE:     l = n - 1       */
      if ( q == n - 1 )
        {
          assert(l == lp - 1);

          recV( rindx ) = GK;
          return( GK );
        }   
      /*  CASE:     l = n - 2       */
      else if ( q == n - 2 )
        {
          double
            dd1 = ZerO,
            dd2 = ZerO;
       
          double
            G2 = ZerO;
    
          double
            Ksqrd = ZerO,
            n1 = ZerO,
            n2  = ZerO;
      
          assert(l == lp - 1);
         
          /* K^2 */
          Ksqrd = K * K;
          assert( Ksqrd != ZerO );
          
          /* n */
          n1 = (double)n;
          assert( n1 != ZerO );
         
          /* n^2 */
          n2 = (double)(n*n);
          assert( n2 != ZerO );
       
          /*     equation: (3.20)                         */
          /*     G(n,n-2; K,n-1) =                        */
          /*            (2n-1)(1+(n K)^2) n G(n,n-1; K,n) */
          dd1 = (double) ((2 * n) - 1);
          assert( dd1 != ZerO );
            
          dd2 = (OnE + (n2 * Ksqrd));
          assert( dd2 != ZerO );
          
          G2 = dd1 * dd2 * n1 * GK;
          assert( G2 != ZerO );
          
          recV(rindx) = G2;
          assert( G2 != ZerO );
          return( G2 );
        }
      else
        {
          long int 
            /* l+2 */
            lp2 = (q + 2),
            /* l+3 */
            lp3 = (q + 3);
          
          double
            /* (l+2)^2 */
            lp2s = (double) (lp2 * lp2),
            /* (l+3)^2 */
            lp3s = (double) (lp3 * lp3);
          
          /******************************************************************************/
          /******************************************************************************/
          /* G(n,l; K,l+1) = [ 4n^2-4(l+2)^2 + (l+2)(2l+3)(1+(n K)^2) ] G(n,l+1; K,l+2) */
          /*                                                                            */
          /*                    - 4n^2 (n^2-(l+2)^2)[ 1+((l+3)^2 K^2) ] G(n,l+2; K,l+3) */
          /*                                                                            */
          /*                                                                            */
          /*            FROM   Eq. 3.23                                                 */
          /*                                                                            */
          /* G(n,l-2; K,l-1) = [ 4n^2-4l^2 + (l+2)(2l-1)(1+(n K)^2) ] G(n,l-1; K,l)     */
          /*                                                                            */
          /*                   - 4n^2 (n^2-l^2)[ 1 + (l+1)^2 K^2 ] G(n,l; K,l+1)        */
          /******************************************************************************/
          /******************************************************************************/
            
          double
            /* K^2 */
            Ksqrd = (K*K),
            
            /* n^2 */
            n2 = (double)(n*n),
          
            /* 4n^2       */
            d1 = (FouR * n2),
          
            /* 4(l+2)^2    */
            d2 = (FouR * lp2s),
          
            /* (l+2)(2l+3)    */
            d3 = (double)(lp2)*((2*q)+3),
          
            /* (1+(n K)^2 */
            d4 = (OnE + (n2 * Ksqrd)),
          
            /* 4n^2-4(l+2)^2 + (l+2)(2l+3)(1+(n K)^2) */
            d5 = d1 - d2 + (d3 * d4);
            
          
            
          /******************************************************************************/
          /******************************************************************************/
          /* G(n,l; K,l+1) = [ 4n^2-4(l+2)^2 + (l+2)(2l+3)(1+(n K)^2) ] G(n,l+1; K,l+2) */
          /*                                                                            */
          /*                   - 4n^2 (n^2-(l+2)^2)[ 1 + (l+3)^2 K^2 ] G(n,l+2; K,l+3)  */
          /******************************************************************************/
          /******************************************************************************/
          
          double
            /* (n^2-(l+2)^2) */
            d6 = (n2 - lp2s),
            
            /* [ 1+((l+3)K)^2 ]  */
            d7 = (OnE + (lp3s * Ksqrd)),
            
            /* 4n^2 (n^2-(l+2)^2)[ 1 + (l+3)^2 K^2 ] */
            d8 = (d1 * d6 * d7),
            
            d9 = (d5 * bhGm( (q+1), K, n, l, lp, rcsvV, GK )),
            
            d10 = (d8 * bhGm( (q+2), K, n, l, lp, rcsvV, GK )),
            
            d11 = (d9 - d10);
          
          
          assert(l == lp - 1);
          assert( lp2s != ZerO);
          assert( Ksqrd != ZerO);
          assert( n2 != ZerO );
          assert(d1 != ZerO);
          assert(d2 != ZerO);
          assert(d3 != ZerO);  
          assert(d4 != ZerO);
          assert(d5 != ZerO);
          assert(d6 != ZerO);
          assert(d7 != ZerO);
          assert(d8 != ZerO);
          assert(d9 != ZerO);
          assert(d10 != ZerO);  
          assert( lp3s != ZerO);
            
            
          recV(rindx) = d11;
            
          return( d11 );
        } 
    }
  else
    {
      assert(  recV(rindx) != ZerO );
      return( recV(rindx) ); 
    }
}

/************************log version***********************************/
static mx bhGm_mx(
                  long int q,
                  double K,
                  long int n,
                  long int l,
                  long int lp,
                  mxq *rcsvV_mxq,
                  mx GK_mx
                  )
{
  long int
    rindx;

  /*static long int rcsv_Level = 1;                                     */
  /*printf( "bhGm(): recursion level:\t%li\n",rcsv_Level++ );           */

  assert(l == lp - 1);
  assert(l < n);

  {
    long int
      i = 0;

    if ( l > lp )
      {
        i = 0;
      }
    else
      {
        i = 1 ;
      }
    rindx = ((2 * q) + i);
  }
    
  if ( recV_mxq(rindx).q == 0 )
    {
      /*  CASE:     l = n - 1      */
      if ( q == n - 1 )
        {
          mx result_mx=GK_mx;
          result_mx = normalize_mx( result_mx );
   
          recV_mxq( rindx ).q = 1;
          recV_mxq( rindx ).mx = result_mx;

          assert(l == lp - 1);
       
        
          return( result_mx );
        }
      /*  CASE:     l = n - 2      */
      else if ( q == n - 2 )
        {
          double
            Ksqrd = (K * K),                      /* K^2    */
            n1  = (double) n,                     /*  n     */
            n2  = (double) (n*n),                       /* n^2    */
            dd1 = (double) ((2 * n) - 1), /* 2n-1   */
            dd2 = (OnE + (n2 * Ksqrd)), /* (1+(n K)^2) */
            /*(2n-1)(1+(n K)^2) n*/
            dd3 = (dd1*dd2*n1);
          
          /******************************************************/
          /******************************************************/
          /*     G(n,n-2; K,n-1) =                              */
          /*            (2n-1)(1+(n K)^2) n G(n,n-1; K,n)       */
          /******************************************************/
          /******************************************************/
       
          mx
            dd3_mx = mxify( dd3 ),
            G2_mx  = mult_mx( dd3_mx, GK_mx );
      
          G2_mx = normalize_mx( G2_mx );
         
          
            
          assert( l == lp - 1);
          assert( n1 != ZerO );
          assert( n2 != ZerO );
          assert( dd1 != ZerO );
          assert( dd2 != ZerO );   
          assert( dd3 != ZerO );
          assert( Ksqrd != ZerO );
          
          recV_mxq(rindx).q  = 1;
          recV_mxq(rindx).mx = G2_mx;
          
          return( G2_mx );
        }
      else
        {   
          /******************************************************************************/
          /******************************************************************************/
          /* G(n,l; K,l+1) = [ 4n^2-4(l+2)^2 + (l+2)(2l+3)(1+(n K)^2) ] G(n,l+1; K,l+2) */
          /*                                                                            */
          /*                    - 4n^2 (n^2-(l+2)^2)[ 1+((l+3)^2 K^2) ] G(n,l+2; K,l+3) */
          /*                                                                            */
          /*                                                                            */
          /*            FROM   Eq. 3.23                                                 */
          /*                                                                            */
          /* G(n,l-2; K,l-1) = [ 4n^2-4l^2 + (l+2)(2l-1)(1+(n K)^2) ] G(n,l-1; K,l)     */
          /*                                                                            */
          /*                   - 4n^2 (n^2-l^2)[ 1 + (l+1)^2 K^2 ] G(n,l; K,l+1)        */
          /******************************************************************************/
          /******************************************************************************/
          
          long int
            /* l+2      */
            lp2 = (q + 2),
            /* l+3      */
            lp3 = (q + 3);
      
          double
            /* (l+2)^2  */
            lp2s = (double) (lp2 * lp2),
          
            /* (l+3)^2  */
            lp3s = (double) (lp3 * lp3),
          
            /* n^2      */
            n2   = (double)(n*n),
          
            /* K^2      */
            Ksqrd = (K * K);
          
          /*8888888888888888888888888888888888888888888888888888*/
          /*    [ 4n^2-4(l+2)^2 + (l+2)(2l+3)(1+(n K)^2) ]      */
          /*8888888888888888888888888888888888888888888888888888*/
          double  
            /* 4n^2                                             */
            d1 = (FouR * n2),
            
            /* 4(l+2)^2                                         */
            d2 = (FouR * lp2s),
          
            /* (l+2)(2l+3)                                      */
            d3 = (double)(lp2)*((2*q)+3),
          
            /* (1+(n K)^2                                       */
            d4 = (OnE + (n2 * Ksqrd)),  
          
            /* 4n^2-4(l+2)^2 + (l+2)(2l+3)(1+(n K)^2)           */
            d5 = d1 - d2 + (d3 * d4);
          
          mx
            d5_mx=mxify(d5);
          /*8888888888888888888888888888888888888888888888888888*/
          
          
          
          /*8888888888888888888888888888888888888888888888888888*/
          /*        4n^2 (n^2-(l+2)^2)[ 1+((l+3)^2 K^2) ]       */
          /*8888888888888888888888888888888888888888888888888888*/
          double
            /* (n^2-(l+2)^2)                                    */
            d6 = (n2 - lp2s),  
          
            /* [ 1+((l+3)K)^2 ]                                 */
            d7 = (OnE + (lp3s * Ksqrd)), 
          
            /* 4n^2 (n^2-(l+2)^2)[ 1 + (l+3)^2 K^2 ]            */
            d8 = d1 * d6 * d7;
          
          mx
            d8_mx=mxify(d8);
          /*8888888888888888888888888888888888888888888888888888*/
            
            
          /*8888888888888888888888888888888888888888888888888888888888888888888888888888*/
          /* G(n,l; K,l+1) = [ 4n^2-4(l+2)^2 + (l+2)(2l+3)(1+(n K)^2) ] G(n,l+1; K,l+2) */
          /*                                                                            */
          /*                   - 4n^2 (n^2-(l+2)^2)[ 1 + (l+3)^2 K^2 ] G(n,l+2; K,l+3)  */
          /*8888888888888888888888888888888888888888888888888888888888888888888888888888*/
          
          mx
            d9_mx  = bhGm_mx( (q+1), K, n, l, lp, rcsvV_mxq, GK_mx ),
            d10_mx = bhGm_mx( (q+2), K, n, l, lp, rcsvV_mxq, GK_mx ),
            d11_mx = mult_mx( d5_mx, d9_mx ),
            d12_mx = mult_mx( d8_mx, d10_mx),
            result_mx = sub_mx( d11_mx , d12_mx );
          recV_mxq(rindx).q  = 1;
          recV_mxq(rindx).mx = result_mx;
            
          assert(l == lp - 1);
          assert(n2 != ZerO );
          assert(lp2s != ZerO);
          assert( lp3s != ZerO);
          assert(Ksqrd != ZerO);
            
          assert(d1 != ZerO);
          assert(d2 != ZerO);
          assert(d3 != ZerO);
          assert(d4 != ZerO);
          assert(d5 != ZerO);
          assert(d6 != ZerO);
          assert(d7 != ZerO);
          assert(d8 != ZerO);
            
            
          return( result_mx );
        }
    }       
  else
    {
      assert(  recV_mxq(rindx).q != 0 );
      return( recV_mxq(rindx).mx );
    }
}
    

/****************************************************************************************/
/****************************************************************************************/
/*                                                                                      */
/*      bhg.c                                                                           */
/*                                                                                      */
/*      From reference;                                                                 */
/*      M. BrocKlehurst                                                                 */
/*      Mon. Note. R. astr. Soc. (1971) 153, 471-490                                    */
/*                                                                                      */
/*                                                                                      */
/*      We wish to compute the following function,                                      */
/*                                                                                      */
/*    equation: (3.17)                                                                  */
/*                 -           s=l'              - (1/2)                                */
/*                |  (n+l)!   -----               |                                     */
/*    g(nl, Kl) = | --------   | |  (1 + s^2 K^2) |    *  (2n)^(l-n) G(n,l; K,l')       */
/*                | (n-l-1)!   | |                |                                     */
/*                 -           s=0               -                                      */
/*                                                                                      */
/*    Using various recursion relations (for l'=l+1)                                    */
/*                                                                                      */
/*    equation: (3.23)                                                                  */
/*    G(n,l-2; K,l-1) = [ 4n^2-4l^2+l(2l-1)(1+(n K)^2) ] G(n,l-1; K,l)                  */
/*                                                                                      */
/*                      - 4n^2 (n^2-l^2)[1+(l+1)^2 K^2] G(n,l; K,l+1)                   */
/*                                                                                      */
/*    and (for l'=l-1)                                                                  */
/*                                                                                      */
/*    equation: (3.24)                                                                  */
/*    G(n,l-1; K,l-2) = [ 4n^2-4l^2 + l(2l-1)(1+(n K)^2) ] G(n,l; K,l-1)                */
/*                                                                                      */
/*                      - 4n^2 (n^2-(l+1)^2)[ 1+(lK)^2 ] G(n,l; K,l+1)                  */
/*                                                                                      */
/*                                                                                      */
/*    the starting point for the recursion relations are;                               */
/*                                                                                      */
/*                                                                                      */
/*    equation: (3.18)                                                                  */
/*                                                                                      */
/*                     | pi |(1/2)   8n                                                 */
/*     G(n,n-1; 0,n) = | -- |      ------- (4n)^2 exp(-2n)                              */
/*                     | 2  |      (2n-1)!                                              */
/*                                                                                      */
/*     equation: (3.20)                                                                 */
/*                                                                                      */
/*                          exp(2n-2/K tan^(-1)(n K)                                    */
/*     G(n,n-1; K,n) =  ---------------------------------------                         */
/*                      sqrt(1 - exp(-2 pi/ K)) * (1+(n K)^(n+2)                        */
/*                                                                                      */
/*                                                                                      */
/*                                                                                      */
/*     equation: (3.20)                                                                 */
/*     G(n,n-2; K,n-1) = (2n-2)(1+(n K)^2) n G(n,n-1; K,n)                              */
/*                                                                                      */
/*                                                                                      */
/*     equation: (3.21)                                                                 */
/*                                                                                      */
/*                       (1+(n K)^2)                                                    */
/*     G(n,n-1; K,n-2) = ----------- G(n,n-1; K,n)                                      */
/*                           2n                                                         */
/****************************************************************************************/
/****************************************************************************************/
static double bhg(
                  double K,
                  long int n,
                  long int l,
                  long int lp,
                  /* Temperary storage for intermediate      */
                  /*   results of the recursive routine      */
                  double *rcsvV
                  )
{
  /*double
     K^2 
    Ksqrd = (K * K);*/

  double
    /*       (n+l)!           */
    ld1 = pre_factorial[ (n + l) ],

    /*      (n-l-1)!          */
    ld2 = pre_factorial[ ((n - l) - 1) ],

    ld3 = (ld1 / ld2);


  double
    partprod = local_product( K , lp );
  /**************************************************************************************/
  /*    equation: (3.17)                                                                */
  /*                 -           s=l'              - (1/2)                              */
  /*                |  (n+l)!   -----               |                                   */
  /*    g(nl, Kl) = | --------   | |  (1 + s^2 K^2) |    *  (2n)^(l-n) G(n,l; K,l')     */
  /*                | (n-l-1)!   | |                |                                   */
  /*                 -           s=0               -                                    */
  /**************************************************************************************/
    
  /**********************************************/
  /*      -           s=l'              - (1/2) */
  /*     |  (n+l)!   -----               |      */
  /*     | --------   | |  (1 + s^2 K^2) |      */
  /*     | (n-l-1)!   | |                |      */
  /*      -           s=0               -       */
  /**********************************************/
  double
    d2 = sqrt( ld3 * partprod ),
    /* (2n)^(l-n) */
    d3 = power( (2 * n) , (l - n) ),
    d4 = bhG( K, n, l, lp, rcsvV ),
    d5 = (d2 * d3),
    d6 = (d5 * d4);
  
  
  assert(K != ZerO);
  assert( (n+l) >= 1 );
  assert( ((n-l)-1) >= 0 );
  
  assert( partprod != ZerO );
    
  assert( ld1 != ZerO );
  assert( ld2 != ZerO );
  assert( ld3 != ZerO );
  
  assert( d2 != ZerO );
  assert( d3 != ZerO );
  assert( d4 != ZerO );
  assert( d5 != ZerO );
  assert( d6 != ZerO );
    
  return( d6 );
}

/********************log version**************************/
static double bhg_log(
                   double K,
                   long int n,
                   long int l,
                   long int lp,
                   /* Temperary storage for intermediate      */
                   /*   results of the recursive routine      */
                   mxq *rcsvV_mxq
                   )
{
  /**************************************************************************************/
  /*    equation: (3.17)                                                                */
  /*                 -           s=l'              - (1/2)                              */
  /*                |  (n+l)!   -----               |                                   */
  /*    g(nl, Kl) = | --------   | |  (1 + s^2 K^2) |    *  (2n)^(l-n) G(n,l; K,l')     */
  /*                | (n-l-1)!   | |                |                                   */
  /*                 -           s=0               -                                    */
  /**************************************************************************************/
  double
    d1 = (double) (2*n),
    d2 = (double) (l-n),
    Ksqrd  = (K*K);

        /**************************************************************************************/
        /*                                                                                    */
        /*          |  (n+l)!  |                                                              */
        /*    log10 | -------- | = log10((n+1)!) - log10((n-l-1)!)                            */
        /*          | (n-l-1)! |                                                              */
        /*                                                                                    */
        /**************************************************************************************/
                    
  double
    /*       (n+l)!           */
    ld1 = log10_factorial( (n + l) ),
  
    /*      (n-l-1)!          */
    ld2 = log10_factorial( ((n - l) - 1) ),
  
    /**************************/
    /*    s=l'                */
    /*   -----                */
    /*    | |  (1 + s^2 K^2)  */
    /*    | |                 */
    /*    s=0                 */
    /**************************/
    /**********************************************************************/
    /*        |  s=l'                |     s=l'                           */
    /*        | -----                |     ---                            */
    /*  log10 |  | |  (1 + s^2 K^2)  | =    >    log10((1 + s^2 K^2))     */
    /*        |  | |                 |     ---                            */
    /*        |  s=0                 |     s=0                            */
    /**********************************************************************/
    ld3 = log10_prodxx( lp, Ksqrd ),
  
    
    /**********************************************/
    /*      -           s=l'              - (1/2) */
    /*     |  (n+l)!   -----               |      */
    /*     | --------   | |  (1 + s^2 K^2) |      */
    /*     | (n-l-1)!   | |                |      */
    /*      -           s=0               -       */
    /**********************************************/
    
    /***********************************************************************/
    /*                                                                     */
    /*            |  -           s=l'              - (1/2) |               */
    /*            | |  (n+l)!   -----               |      |               */
    /*       log10| | --------   | |  (1 + s^2 K^2) |      | ==            */
    /*            | | (n-l-1)!   | |                |      |               */
    /*            |  -           s=0               -       |               */
    /*                                                                     */
    /*              |                           |  s=l'               |  | */
    /*              |      |  (n+l)!  |         | -----               |  | */
    /*       (1/2)* |log10 | -------- | + log10 |  | |  (1 + s^2 K^2) |  | */
    /*              |      | (n-l-1)! |         |  | |                |  | */
    /*              |                           |  s=0                |  | */
    /*                                                                     */
    /***********************************************************************/
    
    ld4 = (OnE/TwO) * ( ld3 + ld1 - ld2 ),
    
    /**********************************************/
    /*                   (2n)^(l-n)               */
    /**********************************************/
    /*    log10( 2n^(L-n) ) = (L-n) log10( 2n )   */
    /**********************************************/
    ld5 = d2 * log10( d1 ),
    
    /**************************************************************************************/
    /*    equation: (3.17)                                                                */
    /*                 -           s=l'              - (1/2)                              */
    /*                |  (n+l)!   -----               |                                   */
    /*    g(nl, Kl) = | --------   | |  (1 + s^2 K^2) |    *  (2n)^(l-n) * G(n,l; K,l')   */
    /*                | (n-l-1)!   | |                |                                   */
    /*                 -           s=0               -                                    */
    /**************************************************************************************/
    
    /****************************************************/
    /*                                                  */
    /*  -           s=l'              - (1/2)           */
    /* |  (n+l)!   -----               |                */
    /* | --------   | |  (1 + s^2 K^2) |  * (2n)^(L-n)  */
    /* | (n-l-1)!   | |                |                */
    /*  -           s=0               -                 */
    /****************************************************/
    ld6 = (ld5+ld4);
    
  mx
    d6_mx = mxify_log10( ld6 ),
    dd1_mx = bhG_mx( K, n, l, lp, rcsvV_mxq ),
    dd2_mx = mult_mx( d6_mx, dd1_mx ),
    result_mx = normalize_mx( dd2_mx );
    
    
  double
    result = unmxify( result_mx );
    
  assert( result != ZerO );
    
    
  assert(Ksqrd != ZerO);
  assert( ld3 >= ZerO );
    
assert( d1 > ZerO );
  assert( d2 < ZerO );
  
  
  return( result );
}
 

static double local_product( double K , long int lp )
{
  long int
    s = 0;
  
  double
    Ksqrd =(K*K),
    partprod = OnE;

  for ( s = 0; s <= lp; s = s + 1 )
    {
      double
        /* s^2 */
        s2 = (double)(s*s);

      /**************************/
      /*    s=l'                */
      /*   -----                */
      /*    | |  (1 + s^2 K^2)  */
      /*    | |                 */
      /*    s=0                 */
      /**************************/

      partprod *= ( OnE + ( s2  * Ksqrd ) );
    }
  return( partprod );
} 

/************************************************************************/
/************************************************************************/
/*  Find the Einstein A's for hydrogen for a                            */
/*  transition n,l -->  n',l'                                           */
/*                                                                      */
/*  In the following, we will assume n > n'                             */
/************************************************************************/
/************************************************************************/
/*                                                                      */
/*   Einstein A() for the transition from the                           */
/*   initial state n,l to the finial state n',l'                        */
/*   is given by oscillator f()                                         */
/*                                                                      */
/*                     hbar w    max(l,l')  |              | 2          */
/*   f(n,l;n',l') = - -------- ------------ | R(n,l;n',l') |            */
/*                     3 R_oo   ( 2l + 1 )  |              |            */
/*                                                                      */
/*                                                                      */
/*                     E(n,l;n',l')     max(l,l')  |              | 2   */
/*   f(n,l;n',l') = -  ------------   ------------ | R(n,l;n',l') |     */
/*                      3 R_oo         ( 2l + 1 )  |              |     */
/*                                                                      */
/*                                                                      */
/*   See for example Gordan Drake's                                     */
/*     Atomic, Molecular, & Optical Physics Handbook pg.638             */
/*                                                                      */
/*   Here R_oo is the infinite mass Rydberg length                      */
/*                                                                      */
/*                                                                      */
/*        h c                                                           */
/*   R_oo --- = 13.605698 eV                                            */
/*        {e}                                                           */
/*                                                                      */
/*                                                                      */
/*   R_oo =  2.179874e-11 ergs                                          */
/*                                                                      */
/*   w = omega                                                          */
/*     = frequency of transition from n,l to n',l'                      */
/*                                                                      */
/*                                                                      */
/*                                                                      */
/*                                                                      */
/*     here g_k are statistical weights obtained from                   */
/*      the appropriate angular momentum quantum numbers                */
/*                                                                      */
/*                                                                      */
/*                                                                      */
/*                                                                                      */
/*                                                                                      */
/*                                                          -            -  2           */
/*                   64 pi^4 (e a_o)^2    max(l,l')        |              |             */
/*   A(n,l;n',l') = -------------------  -----------  v^3  | R(n,l;n',l') |             */
/*                         3 h c^3         2*l + 1         |              |             */
/*                                                          -            -              */
/*                                                                                      */
/*                                                                                      */
/*  pi             3.141592654                                                          */
/*  plank_hbar     6.5821220         eV sec                                             */
/*  R_oo           2.179874e-11      ergs                                               */
/*  plank_h        6.6260755e-34     J sec                                              */
/*  e_charge       1.60217733e-19    C                                                  */
/*  a_o            0.529177249e-10   m                                                  */
/*  vel_light_c    299792458L        m sec^-1                                           */
/*                                                                                      */
/*                                                                                      */
/*                                                                                      */
/*                                                                                      */
/*  64 pi^4 (e a_o)^2    64 pi^4 (a_o)^2    e^2     1                      1            */
/*  ----------------- = ----------------- -------- ----  = 7.5197711e-38 -----          */
/*     3 h c^3                3  c^2       hbar c  2 pi                   sec           */
/*                                                                                      */
/*                                                                                      */
/*            e^2               1                                                       */
/*  using ---------- = alpha = ----                                                     */
/*          hbar c             137                                                      */
/****************************************************************************************/
/****************************************************************************************/

double H_Einstein_A(/*  IN THE FOLLOWING WE HAVE  n > n'                        */
                    /* principal quantum number, 1 for ground, upper level      */
                    long int n,
                    /* angular momentum, 0 for s                                */
                    long int l,
                    /* principal quantum number, 1 for ground, lower level      */
                    long int np,
                    /* angular momentum, 0 for s                                */
                    long int lp,
                    /* Nuclear charge, 1 for H+, 2 for He++, etc                */
                    long int iz
                    )
{
  double result;
  if( n > 60 || np > 60 )
    {
      result = H_Einstein_A_log10(n,l,np,lp,iz );
    }
  else
    {
      result = H_Einstein_A_lin(n,l,np,lp,iz );
    }
  return( result);
}
/************************************************************************/
/*   Calculates the Einstein A's for hydrogen                           */
/*   for the transition n,l --> n',l'                                   */
/*   units of sec^(-1)                                                  */
/*                                                                      */
/*  In the following, we have n > n'                                    */
/************************************************************************/
static double H_Einstein_A_lin(/*  IN THE FOLLOWING WE HAVE  n > n'                        */
                               /* principal quantum number, 1 for ground, upper level      */
                               long int n,
                               /* angular momentum, 0 for s                                */
                               long int l,
                               /* principal quantum number, 1 for ground, lower level      */
                               long int np,
                               /* angular momentum, 0 for s                                */
                               long int lp,
                               /* Nuclear charge, 1 for H+, 2 for He++, etc                */
                               long int iz
                               )
{
  /****************************************************************************************/
  /*  64 pi^4 (e a_o)^2    64 pi^4 (a_o)^2    e^2     1                      1            */
  /*  ----------------- = ----------------- -------- ----  = 7.5197711e-38 -----          */
  /*     3 h c^3                3  c^2       hbar c  2 pi                   sec           */
  /****************************************************************************************/

  double
    d0 = 7.5197711e-38;

  double
    /********************************************************************************************/
    /*  hv calculates photon energy in ergs for n -> n' transitions for H and H-like ions       */
    /*  simplest case of no "l" or "m" dependence                                               */
    /*  double hv(long int n, long int n', long int Z);                                         */
    /********************************************************************************************/
    d1 = hv( n, np , iz ) * ( 1e-07 ),
    /* NOTE: hv() returns double */
    /* 1 erg = 1e-07 joule */
                               
    /*  plank_h        6.6260755e-34     J sec  */
    d2 = ( d1 /  ( 6.6260755e-34) ),
    /* v = hv / h  */
  
    d3 = (d2 * d2 * d2),
    lg = (double)(l > lp ? l : lp),
    Two_L_Plus_One = (double)(2*l + 1),
    d6 = (lg / Two_L_Plus_One),
    d7 = hri( n, l, np, lp , iz ),
    d8 = (d7 * d7 ),
    result = d0 * d3 * d6 * d8;

  /* validate the incoming data */
  if ( n >= 70 )
    {
      fprintf(ioQQQ,"Principle Quantum Number `n' too large.\n");
      (void) puts( "[Stop in H_Einstein_A_lin]" );
      exit(EXIT_FAILURE);
    }
  if ( iz <1 )
    {
      fprintf(ioQQQ," The charge is impossible.\n");
      (void) puts( "[Stop in H_Einstein_A_lin]" );
      exit(EXIT_FAILURE);
    }
  if ( n <1 || np<1 || l>=n || lp>=np )
    {
      fprintf(ioQQQ," The quantum numbers are impossible.\n");
      (void) puts( "[Stop in H_Einstein_A_lin]" );
      exit(EXIT_FAILURE);
    }
  if ( n <= np  )   
    {
      fprintf(ioQQQ," The principle quantum numbers are such that n <= n'.\n");
      (void) puts( "[Stop in H_Einstein_A_lin]" );
      exit(EXIT_FAILURE);
    }
      
  return( result );
}
     
/**********************log version****************************/
double H_Einstein_A_log10(/* returns Einstein A in units of (sec)^-1                      */
                          long int n,
                          long int l,
                          long int np,
                          long int lp,
                          long int iz
                          )
{
  /****************************************************************************************/
  /*  64 pi^4 (e a_o)^2    64 pi^4 (a_o)^2    e^2     1                      1            */
  /*  ----------------- = ----------------- -------- ----  = 7.5197711e-38 -----          */
  /*     3 h c^3                3  c^2       hbar c  2 pi                   sec           */
  /****************************************************************************************/

  double
    d0 = 7.5197711e-38;

  double
    /********************************************************************************************/
    /*  hv calculates photon energy in ergs for n -> n' transitions for H and H-like ions       */
    /*  simplest case of no "l" or "m" dependence                                               */
    /*  double hv(long int n, long int n', long int Z);                                         */
    /********************************************************************************************/
    d1 = hv( n, np , iz ) * ( 1e-07 ),
    /* NOTE: hv() returns double                        */
    /* 1 erg = 1e-07 joule                              */
                          
    /*  plank_h        6.6260755e-34     J sec          */
    d2 = ( d1 /  ( 6.6260755e-34) ),
    /* v = hv / h                                       */
  
    d3 = (d2 * d2 * d2),
    lg = (double)(l > lp ? l : lp),
    Two_L_Plus_One = (double)(2*l + 1),
    d6 = (lg / Two_L_Plus_One),
    d7 = hri_log10( n, l, np, lp , iz ),
    d8 = (d7 * d7 ),
    result = d0 * d3 * d6 * d8;
  
  /* validate the incoming data                         */
  if ( iz <1 )
    {
      fprintf(ioQQQ," The charge is impossible.\n");
      puts( "[Stop in H_Einstein_A_log10]" );
      exit(EXIT_FAILURE);
    }
  if ( n <1 || np<1 || l>=n || lp>=np )
    {                     
      fprintf(ioQQQ," The quantum numbers are impossible.\n");
      puts( "[Stop in H_Einstein_A_log10]" );
      exit(EXIT_FAILURE);
    }
  if ( n <= np  )
    {
      fprintf(ioQQQ," The principle quantum numbers are such that n <= n'.\n");
      puts( "[Stop in H_Einstein_A_log10]" );
      exit(EXIT_FAILURE);
    }
    
  return( result );
}

/********************************************************************************/
/********************************************************************************/
/* hv calculates photon energy for n -> n' transitions for H and H-like ions    */
/*              simplest case of no "l" or "m" dependence                       */
/*  epsilon_0 = 1 in vacu                                                       */
/*                                                                              */
/*                                                                              */
/*                     R_h                                                      */
/*  Energy(n,Z) = -  -------                                                    */
/*                     n^2                                                      */
/*                                                                              */
/*                                                                              */
/*                                                                              */
/*  Friedrich -- Theoretical Atomic Physics  pg. 60   eq. 2.8                   */
/*                                                                              */
/*         u                                                                    */
/*  R_h = --- R_oo  where                                                       */
/*        m_e                                                                   */
/*                                                                              */
/*        h c                                                                   */
/*  R_oo  --- =  2.179874e-11 ergs                                              */
/*         e                                                                    */
/*                                                                              */
/*  (Harmin Lecture Notes for course phy-651 Spring 1994)                       */
/*  where m_e (m_p) is the mass of and electron (proton)                        */
/*  and u is the reduced electron mass for neutral hydrogen                     */
/*                                                                              */
/*                                                                              */
/*         m_e m_p       m_e                                                    */
/*    u = --------- = -----------                                               */
/*        m_e + m_p   1 + m_e/m_p                                               */
/*                                                                              */
/*        m_e                                                                   */
/*  Now  ----- = 0.000544617013                                                 */
/*        m_p                                                                   */
/*            u                                                                 */
/*  so that  --- =  0.999455679                                                 */
/*           m_e                                                                */
/*                                                                              */
/*                                                                              */
/*  returns energy of photon in ergs                                            */
/*                                                                              */
/*  hv (n,n',Z) is for transitions n -> n'                                      */
/*                                                                              */
/*  1 erg = 1e-07                                                               */
/********************************************************************************/
/********************************************************************************/
/* WARNING: hv() use the electron reduced mass for hydrogen instead of          */
/*      the reduced mass associated with the apropriate ion                     */
/********************************************************************************/
static double hv( long int n, long int nprime, long int iz )
{
  double
    n1  = (double) n,
    n2  = (n1*n1),
    np1 = (double) nprime,
    np2 = (np1*np1),
    rmr = (0.999455679),
    br  = (2.179874e-11),
    izsqrd = (double)(iz*iz);

  double
    d1 = ( OnE / n2  ),
    d2 = ( OnE / np2 ),
    d3 = ( izsqrd * rmr * br ),
    d4 = ( d2 - d1   ),
    result = ( d3 * d4  );



  assert( n > 0 );
  assert( nprime > 0 );
  assert( n > nprime );
  assert( iz > 0 );
  assert( result > ZerO );

  if ( n <= nprime  )
    {
      fprintf(ioQQQ," The principle quantum numbers are such that n <= n'.\n");
      puts( "[Stop in H_Einstein_A_log10]" );
      exit(EXIT_FAILURE);
    }

  return( result );
}

/************************************************************************/
/************************************************************************/
/*   hri()                                                              */
/*   Calculate the hydrogen radial wavefunction intergral               */
/*   for the dipole transition  l'=l-1  or  l'=l+1                      */
/*   for the higher energy state n,l  to the lower energy state n',l'   */
/*   no "m" dependence                                                  */
/************************************************************************/
/*      here we have a transition                                       */
/*      from the higher energy state n,l                                */
/*      to the lower energy state n',l'                                 */
/*      with a dipole selection rule on l and l'                        */
/************************************************************************/
/*                                                                      */
/*   hri() test n,l,n',l'  for domain errors and                        */
/*              swaps n,l <--> n',l' for the case  l'=l+1               */
/*                                                                      */
/*   It then calls hrii()                                               */
/*                                                                      */
/*   Dec. 6, 1999                                                       */
/*   Robert Paul Bauman                                                 */
/************************************************************************/
/************************************************************************/

/************************************************************************/
/*      This routine, hri(), calculates the hydrogen radial intergral,  */
/*      for the transition n,l --> n',l'                                */
/*      It is, of course, dimensionless.                                */
/*                                                                      */
/*  In the following, we have n > n'                                    */
/************************************************************************/
double hri(
           /* principal quantum number, 1 for ground, upper level       */
           long int n,
           /* angular momentum, 0 for s                                 */
           long int l,
           /* principal quantum number, 1 for ground, lower level       */
           long int np,
           /* angular momentum, 0 for s                                 */
           long int lp,
           /* Nuclear charge, 1 for H+, 2 for He++, etc                 */
           long int iz
           )
{
  long int a;
  long int b;
  long int c;
  long int d;
  double ld1 = ZerO;
  double Z   = (double) iz;


  /**********************************************************************/
  /*    from higher energy -> lower energy                              */
  /*    Selection Rule for l and l'                                     */
  /*    dipole process only                                             */
  /**********************************************************************/
           
  assert( n > 0);
  assert( np > 0);
  assert( l >= 0);
  assert( lp >= 0 );
  assert( n > l );
  assert( np > lp );
  assert( n > np || ( n == np && l == lp + 1 ));
  assert( iz > 0 );
  assert( lp == l + 1 || lp == l - 1 );
  
  
  if ( l == lp + 1)
    {
      /*        Keep variable  the same                                 */
      a = n;
      b = l;
      c = np;
      d = lp;
    }
  else if ( l == lp - 1 )
    {
      /* swap n,l with n',l'                                            */
      a = np;
      b = lp;
      c = n;
      d = l;
    }
  else
    {
      printf( "BadMagic: l and l' do NOT satisfy dipole requirements.\n\n" );
      (void) fflush( stdout );
      exit(EXIT_FAILURE);
      /* End of hri() */
    }
  
  /**********************************************/
  /*    Take care of the Z-dependence here.     */
  /**********************************************/
  ld1 =  hrii(a, b, c, d ) / Z;
      
  return( ld1 );
}
  
/************************************************************************/
/************************************************************************/
/*   hri_log10()                                                        */
/*   Calculate the hydrogen radial wavefunction intergral               */
/*   for the dipole transition  l'=l-1  or  l'=l+1                      */
/*   for the higher energy state n,l  to the lower energy state n',l'   */
/*   no "m" dependence                                                  */
/************************************************************************/
/*      here we have a transition                                       */
/*      from the higher energy state n,l                                */
/*      to the lower energy state n',l'                                 */
/*      with a dipole selection rule on l and l'                        */
/************************************************************************/
/*                                                                      */
/*   hri_log10() test n,l,n',l'  for domain errors and                  */
/*              swaps n,l <--> n',l' for the case  l'=l+1               */
/*                                                                      */
/*   It then calls hrii_log()                                              */
/*                                                                      */
/*   Dec. 6, 1999                                                       */
/*   Robert Paul Bauman                                                 */
/************************************************************************/
/************************************************************************/

double hri_log10( long int  n, long int l, long int np, long int lp , long int iz )
{
  /**********************************************************************/
  /*    from higher energy -> lower energy                              */
  /*    Selection Rule for l and l'                                     */
  /*    dipole process only                                             */
  /**********************************************************************/


  long int a;
  long int b;
  long int c;
  long int d;
  double ld1 = ZerO;
  double Z   = (double) iz;

  assert( n > 0);
  assert( np > 0);
  assert( l >= 0);
  assert( lp >= 0 );
  assert( n > l );
  assert( np > lp );
  assert( n > np || ( n == np && l == lp + 1 ));
  assert( iz > 0 );
  assert( lp == l + 1 || lp == l - 1 );
  
  
  if ( l == lp + 1)
    {
      /*        Keep variable  the same                                 */
      a = n; 
      b = l; 
      c = np;
      d = lp;
    }
  else if ( l == lp - 1 )  
    {
      /* swap n,l with n',l'                                            */
      a = np;
      b = lp;
      c = n;
      d = l;
    }
  else
    {
      printf( "BadMagic: l and l' do NOT satisfy dipole requirements.\n\n" );
      fflush( stdout );
      exit(EXIT_FAILURE);
      /* End of hri_log10()                                                   */
    }
  
  /**********************************************/
  /*    Take care of the Z-dependence here.     */
  /**********************************************/
  ld1 =  hrii_log(a, b, c, d ) / Z;
      
      
     
  return( ld1 );
}
     
      
static double hrii( long int  n, long int  l, long int np, long int lp)
{
  /******************************************************************************/
  /******************************************************************************/
  /*      this routine hrii() is internal to the parent routine hri()           */
  /*      this internal routine only considers the case l=l'+1                  */
  /*      the case l=l-1 is done in the parent routine hri()                    */
  /*      by the transformation n <--> n' and l <--> l'                         */
  /*      THUS WE TEST FOR                                                      */
  /*          l=l'-1                                                            */
  /******************************************************************************/
  /******************************************************************************/
  
  long int
    a = 0,
    b = 0,
    c = 0;
      
  long int
    i1 = 0,
    i2 = 0,
    i3 = 0,
    i4 = 0;

  char A='a';
  
  double y = ZerO;
  double fsf = ZerO;
  double
    d1 = ZerO,
    d2 = ZerO,
    d3 = ZerO,
    d4 = ZerO,
    d5 = ZerO,
    d6 = ZerO,
    d7 = ZerO,
    d8 = ZerO,
    d9 = ZerO,
    d10 = ZerO,
    d11 = ZerO,
    d12 = ZerO,
    d13 = ZerO,
    d14 = ZerO;
    
  double   
    d00 = ZerO,
    d01 = ZerO;
  
  assert( l == lp + 1 );
  
  if ( n == np ) /* SPECIAL CASE 1  */
    {   
      /**********************************************************/
      /**********************************************************/
      /* if lp= l + 1 then it has higher energy                 */
      /* i.e.         no photon                                 */
      /* this is the second time we check this, oh well         */
      /**********************************************************/
      /**********************************************************/
      if ( lp != (l - 1) )
        {
          printf( "BadMagic: Energy requirements not meet.\n\n" );
          (void) fflush( stdout );
          exit(EXIT_FAILURE);
        }
    
      d2 = ( ThreE / TwO );
      i1 = n * n;
      i2 = l * l;
      d5 = (double) (i1 - i2);
      d6 = sqrt(d5);
      d7 = (double) n * d6;
      d8 = d2 * d7;
  
        
      return( d8 );
    }
  else if ( l == np && lp == (l - 1) ) /* A Pair of Easy Special Cases */
    {
      if ( l == (n - 1) )
        {
          /**********************************************************************/
          /**********************************************************************/
          /*   R(n,l;n',l') = R(n,n-l;n-1,n-2)                                  */
          /*                                                                    */
          /*                = [(2n-2)(2n-1)]^(1/2)   [4n(n-1)/(2n-1)^2]^n  *    */
          /*                            [(2n-1) - 1/(2n-1)]/4                   */
          /**********************************************************************/
          /**********************************************************************/
      
          d1 = (double) ( (2 * n) - 2 );
          d2 = (double) ( (2 * n) - 1 );
          d3 = d1 * d2;
          d4 = sqrt( d3 );
      
          d5 = (double) ( (4 * n) * ( n - 1 ));
          i1 = ((2 * n) - 1);
          d6 = (double) ( i1 * i1);
          d7 = d5/( d6 );
          d8 = power( d7, n );
  
          d9 = OnE /(d2);
          d10 = d2 - d9; 
          d11 = (d10 /  FouR);
          
          /* Wrap it all up */
          
          d12 = d4 * d8 * d11;
          
          return( d12 );
          
        }
      else
        {
          /******************************************************************************/
          /*   R(n,l;n',l') = R(n,l;l,l-1)                                              */
          /*                                                                            */
          /*                = [(n-l) ... (n+l)/(2l-1)!]^(1/2) [4nl/(n-l)^2]^(l+1) *     */
          /*                            [(n-l)/(n+l)]^(n+l) {1-[(n-l)/(n+l)]^2}/4       */
          /******************************************************************************/
          
          
          
          d2 = OnE;
          for (i1 = -l ; i1 <= l ; i1 = i1 + 1)  /* from n-l to n+l INCLUSIVE */
            {
              d1 = (double)(n - i1);
              d2 = d2 * d1;
            }
          i2 = ((2 * l) - 1);
          d3 = pre_factorial[ i2 ];
          d4 = d2/d3;
          d4 = sqrt( d4 );
          
         
          d5 = (double) ( FouR * n * l );
          i3 = (n - l);
          d6 = (double) ( i3 * i3 );
          d7 = d5 / d6 ;
          d8 = power( d7, (l+1) );
          
          
          i4 = (n + l);
          d9 = (double)( i3 ) / (double)( i4 );
          d10 = power( d9 , i4 );
          
          d11 = (d9 * d9);
          d12 = ( OnE - d11);
          d13 = (d12 / FouR);
              
          /* Wrap it all up */
          d14 = d4 * d8 * d10 * d13;
          return( d14 );
        }
    }     
  /***********************************************************************************************/
  /***********************************************************************************************/
  /*     *                                 THE GENERAL CASE                                      */
  /*     *                 USE RECURSION RELATION FOR HYPERGEOMETRIC FUNCTIONS                   */
  /*     *            REF: D. Hoang-Bing Astron. Astrophys. 238: 449-451 (1990)                  */
  /*     *                      For F(a,b;c;x) we have from eq.4                                 */
  /*     *                                                                                       */
  /*     *   (a-c) F(a-1) = a (1-x) [ F(a) - F(a-1) ] + (a + bx - c) F(a)                        */
  /*     *                                                                                       */
  /*     *                   a (1-x)                       (a + bx - c)                          */
  /*     *         F(a-1) = --------- [ F(a) - F(a-1) ] + -------------- F(a)                    */
  /*     *                  (a-c)                            (a-c)                               */
  /*     *                                                                                       */
  /*     *                                                                                       */
  /*     *     A similiar recusion relation holds for b with a <--> b.                           */
  /*     *                                                                                       */
  /*     *                                                                                       */
  /*     * we have initial conditions                                                            */
  /*     *                                                                                       */
  /*     *                                                                                       */
  /*     *  F(0) = 1              with a = -1                                                    */
  /*     *                                                                                       */
  /*     *               b                                                                       */
  /*     * F(-1) = 1 - (---) x    with a = -1                                                    */
  /*     *               c                                                                       */
  /***********************************************************************************************/
  /***********************************************************************************************/
  
  if (lp == (l - 1)) /* use recursion over "b" */
    {
      A='b';
    }
  else if (lp == (l + 1))                 /* use recursion over "a" */
    {
      A='a';
    }
  else
    {
      printf(" BadMagic: Don't know what to do here.\n\n");
      (void) fflush( stdout );
      exit(EXIT_FAILURE);
    }
  
  
  /********************************************************************/
  /*   Calculate the whole shootin match                              */
  /*                    -                - (1/2)                      */
  /*     (-1)^(n'-1)   | (n+l)! (n'+l-1)! |                           */
  /*     ----------- * | ---------------- |                           */
  /*     [4 (2l-1)!]   | (n-l-1)! (n'-l)! |                           */
  /*                    -                -                            */
  /*        * (4 n n')^(l+1) (n-n')^(n+n'-2l-2) (n+n')^(-n-n')        */
  /*                                                                  */
  /*   This is used in the calculation of hydrogen                    */
  /*    radial wave function integral for dipole transition case      */
  /********************************************************************/
     
  fsf = fsff( n, l, np );
     
  /**************************************************************************************/
  /**************************************************************************************/
  /*      Use a -> a' + 1                                                               */
  /*                                 _                 _                                */
  /*              (a' + 1) (1  - x) |                   |                               */
  /*      F(a') = ----------------- | F(a'+1) - F(a'+2) | + (a' + 1 + bx -c) F(a'+1)    */
  /*                 (a' + 1 -c)    |                   |                               */
  /*                                 -                 -                                */
  /*                                                                                    */
  /*      For the first F() in the solution of the radial integral                      */
  /*                                                                                    */
  /*            a = ( -n + l + 1 )                                                      */
  /*                                                                                    */
  /*      a = -n + l + 1                                                                */
  /*      max(a)  = max(-n)  + max(l)     + 1                                           */
  /*              = -n       + max(n-1)   + 1                                           */
  /*              = -n       + n-1        + 1                                           */
  /*              = 0                                                                   */
  /*                                                                                    */
  /*      similiarly                                                                    */
  /*                                                                                    */
  /*      min(a) = min(-n)   + min(l)   + 1                                             */
  /*             = min(-n)   + 0        + 1                                             */
  /*             = -n        + 1                                                        */
  /*                                                                                    */
  /*      a -> a' + 1   implies                                                         */
  /*                                                                                    */
  /*      max(a') = -1                                                                  */
  /*      min(a') = -n                                                                  */
  /**************************************************************************************/
  /**************************************************************************************/
  
  /* a plus                                                     */
  a = ((-n) + l + 1);
  
  
  /*  for the first 2_F_1 we use b = (-n' + l)                  */
  b = ((-np) + l);
  
  
  /*  c is simple                                               */
  c = (2 * l);
  
  /*             -4 nn'                                         */
  /*  where Y = -------- .                                      */
  /*            (n-n')^2                                        */
  d2 = (double) (n - np) ;
  d3 = d2 * d2 ;
  d4 = OnE / d3 ;
  d5 = (double) (n * np);
  d6 = d5 * FouR ;
  d7 = - d6;
  y = d7 * d4;
  
  
  /*  2_F_1( long int a, long int b, long int c, (double) y, (string) "a" or "b") */
  d00 = F21( a, b, c, y, A );
  
  /**************************************************************/
  /**************************************************************/
  /*  For the second F() in the solution of the radial integral */
  /*                                                            */
  /*        a = ( -n + l - 1 )                                  */
  /*                                                            */
  /*  a = -n + l + 1                                            */
  /*  max(a) = max(-n) + max(l)   - 1                           */
  /*         = -n      + (n - 1)  - 1                           */
  /*         = -2                                               */
  /*                                                            */
  /*  similiarly                                                */
  /*                                                            */
  /*  min(a) = min(-n) + min(l)   - 1                           */
  /*         = (-n)    + 0        - 1                           */
  /*         = -n - 1                                           */
  /*                                                            */
  /*  a -> a' + 1   implies                                     */
  /*                                                            */
  /*  max(a') = -3                                              */
  /*                                                            */
  /*  min(a') = -n - 2                                          */
  /**************************************************************/
  /**************************************************************/
  /*  a minus                                                   */
  a = ((-n) + l - 1);
  
  /*  for the first 2_F_1 we use b = (-n' + l)                  */
  /*    and does not change                                     */
  b = ((-np) + l);
  
        /* c is simple                                                */
  c = (2 * l);
  
  
  /**************************************************************/
  /*             -4 nn'                                         */
  /*  where Y = -------- .                                      */
  /*           (n-n')^2                                         */
  /**************************************************************/
  
        /**************************************************************/
        /*      These are already calculated a few lines up           */
        /*                                                            */
        /*      d2 = (double) (n - np) ;                              */
        /*      d3 = d2 * d2 ;                                        */
        /*      d4 = 1/ d3 ;                                          */
        /*      d5 = (double) (n * np);                               */
        /*      d6 = d5 * 4.0;                                        */
        /*      d7 = - d6;                                            */
        /*      y = d7 * d4;                                          */
        /**************************************************************/
  
  
  d01 = F21(a, b, c, y, A );
  
  /*  Calculate         */
  /*                    */
  /*  (n-n')^2          */
  /*  --------          */
  /*  (n+n')^2          */
  
  i1 = (n - np);
  d1 = power( (double)i1 , (long int) 2 );
  i2 = (n + np);
  d2 = power( (double)i2 , (long int) 2 );
  d3 = (d1 / d2);
        
  d4 = d01 * d3;
  d5 = d00 - d4;
  d6 = fsf * d5;
        
  assert( d6 != ZerO );
        
  return( d6 );
} 


static double hrii_log( long int  n, long int  l, long int np, long int lp)
{
  /******************************************************************************/
  /******************************************************************************/
  /*      this routine hrii_log() is internal to the parent routine hri_log10()    */
  /*      this internal routine only considers the case l=l'+1                  */
  /*      the case l=l-1 is done in the parent routine hri_log10()              */
  /*      by the transformation n <--> n' and l <--> l'                         */
  /*      THUS WE TEST FOR                                                      */
  /*          l=l'-1                                                            */
  /******************************************************************************/
  /******************************************************************************/
        /**************************************************************************************/
        /**************************************************************************************/
        /*   THIS HAS THE GENERAL FORM GIVEN BY (GORDAN 1929):                                */
        /*                                                                                    */
        /*    R(n,l;n',l') = (-1)^(n'-1) [4(2l-1)!]^(-1) *                                    */
        /*                      [(n+l)! (n'+l-1)!/(n-l-1)! (n'-l)!]^(1/2) *                   */
        /*                      (4 n n')^(l+1) (n-n')^(n+n'-2l-2) (n+n')^(-n-n') *            */
        /*                      { F(-n+l+1,-n'+l;2l;-4nn'/[n-n']^2)                           */
        /*                    - (n-n')^2 (n+n')^2 F(-n+l-1,-n'+l;2l; -4nn'/[n-n']^2 ) }       */
        /**************************************************************************************/
        /**************************************************************************************/
  
  
  char A='a';
  
  double y = ZerO;
  double log10_fsf = ZerO;
  
  assert( l == lp + 1 );
  
  if ( n == np ) /* SPECIAL CASE 1                              */
    {
      /**********************************************************/
      /**********************************************************/
      /* if lp= l + 1 then it has higher energy                 */
      /* i.e.         no photon                                 */
      /* this is the second time we check this, oh well         */
      /**********************************************************/
      /**********************************************************/
      if ( lp != (l - 1) )
        {
          printf( "BadMagic: l'= l+1 for n'= n.\n\n" );
          assert( lp == (l + 1) );
          fflush( stdout );
          exit(EXIT_FAILURE);
        }
      else   
        {
          /**********************************************************/
          /**********************************************************/
          /*                        3                               */
          /*   R(nl:n'=n,l'=l+1) = ---  n  sqrt( n^2 - l^2 )        */
          /*                        2                               */
          /**********************************************************/
          /**********************************************************/
      
          long int
            i1 = n * n,
            i2 = l * l;
      
          double
            d1 = ( ThreE / TwO ),
            d2 = (double) n,
            d3 = (double) (i1 - i2),
            d4 = sqrt(d3),
            result = d1 * d2 * d4;
          
         
          assert( d3 >= ZerO );
          return( result );
        }
    }
  else if ( l == np && lp == (l - 1) ) /* A Pair of Easy Special Cases          */
    {
      if ( l == (n - 1) )
        {
          /**********************************************************************/
          /**********************************************************************/
          /*   R(n,l;n',l') = R(n,n-l;n-1,n-2)                                  */
          /*                                                                    */
          /*                = [(2n-2)(2n-1)]^(1/2)   [4n(n-1)/(2n-1)^2]^n  *    */
          /*                            [(2n-1) - 1/(2n-1)]/4                   */
          /**********************************************************************/
          /**********************************************************************/
          double
            d1 = (double) (((2*n)-2)*((2*n)-1)),
            d2 = sqrt( d1 ),
            d3 = (double) ((4*n)*(n-1)),
            d4 = (double) ((2*n)-1),
            d5 = (d4*d4),
            d7 = (d3/d5),
            d8 = power(d7,n),  
            d9 = (OnE/d4), 
            d10 = (d4 - d9),
            d11 = (d10 / FouR),
            result = (d2 * d8 * d11); /* Wrap it all up */
     
          assert( d1 >= ZerO );
         
          
          assert( d3 >= ZerO );
          return( result );
        }
      else
        {
          double
            result = ZerO,
            ld1 = ZerO,
            ld2 = ZerO,
            ld3 = ZerO,
            ld4 = ZerO,
            ld5 = ZerO,
            ld6 = ZerO,  
            ld7 = ZerO;  
            
          /******************************************************************************/
          /******************************************************************************/
          /*   R(n,l;n',l') = R(n,l;l,l-1)                                              */
          /*                                                                            */
          /*                = [(n-l) ... (n+l)/(2l-1)!]^(1/2) [4nl/(n-l)^2]^(l+1) *     */
          /*                            [(n-l)/(n+l)]^(n+l) {1-[(n-l)/(n+l)]^2}/4       */
          /******************************************************************************/
          /******************************************************************************/
          /**************************************/
          /*    [(n-l) ... (n+l)]               */
          /**************************************/
          /*    log10[(n-l) ... (n+l)] =        */
          /*                                    */
          /*        n+l                         */
          /*        ---                         */
          /*         >  log10(j)                */
          /*        ---                         */
          /*      j=n-l                         */
          /**************************************/
            
          ld1 = ZerO;    
          { 
            long int i1 = 0;
            for (i1 = (n-l) ; i1 <= (n+l) ; i1++)  /* from n-l to n+l INCLUSIVE           */
              {
                double
                  d1 = (double)(i1);
          
                ld1 += log10( d1 );
              }
          }
          
          /**************************************/
          /*    (2l-1)!                         */
          /**************************************/
          /*    log10[ (2n-1)! ]                */
          /**************************************/
          ld2 = log10_factorial( ((2 * l) - 1) ); 
          
          assert( ((2*l)+1) >= 0);
          
          /**********************************************/
          /* log10( [(n-l) ... (n+l)/(2l-1)!]^(1/2) ) = */
          /*    (1/2) log10[(n-l) ... (n+l)] -          */
          /*            (1/2) log10[ (2n-1)! ]          */
          /**********************************************/
          ld3 = (OnE/TwO) * (ld1 - ld2);
          
                
          /**********************************************/
          /*            [4nl/(n-l)^2]^(l+1)             */
          /**********************************************/
          /*  log10( [4nl/(n-l)^2]^(l+1) ) =            */
          /*       (l+1) * log10( [4nl/(n-l)^2] )       */
          /*                                            */
          /*    = (l+1)*[ log10(4nl) - 2 log10(n-l) ]   */
          /*                                            */
          /**********************************************/
          {
            double
              d1 = (double) (l+1),
              d2 = (double) (4*n*l),
              d3 = (double) (n-l),
              d4 = log10(d2),
              d5 = log10(d3);
          
            ld4 = d1 * (d4 - (TwO * d5));
          }
          
          
          
          /**********************************************/
          /*             [(n-l)/(n+l)]^(n+l)            */
          /**********************************************/
          /*   log10( [ (n-l)/(n+l) ]^(n+l)  ) =        */
          /*                                            */
          /*    (n+l) * [ log10(n-l) - log10(n+l) ]     */
          /*                                            */
          /**********************************************/
          {
            double
              d1 = (double) (n-l),
              d2 = (double) (n+l),
              d3 = log10( d1 ),   
              d4 = log10( d2 );
              
            ld5 = d2 * (d3 - d4);
          }
          
            
           
          
          /**********************************************/
          /*      {1-[(n-l)/(n+l)]^2}/4                 */
          /**********************************************/
          /*   log10[ {1-[(n-l)/(n+l)]^2}/4 ]           */
          /**********************************************/
          {
            double
              d1 = (double) (n-l),
              d2 = (double) (n+l),
              d3 = (d1/d2),
              d4 = d3*d3,
              d5 = (OnE - d4),
              d6 = (d5/FouR);
              
            ld6 = log10(d6);      
          }
              
            
           
          /******************************************************************************/
          /******************************************************************************/
          /*   R(n,l;n',l') = R(n,l;l,l-1)                                              */
          /*                                                                            */
          /*                = [(n-l) ... (n+l)/(2l-1)!]^(1/2) [4nl/(n-l)^2]^(l+1) *     */
          /*                            [(n-l)/(n+l)]^(n+l) {1-[(n-l)/(n+l)]^2}/4       */
          /******************************************************************************/
          /******************************************************************************/
          
          ld7 = (ld3 + ld4 + ld5 + ld6);
            
          result = pow( TeN, ld7 );
              
              
          assert( result > ZerO );
              
          return( result );  
        }     
    }
  else
    {         
      double
        result = ZerO;
          
      long int
        a = 0,
        b = 0,
        c = 0;
          
      double
        d1 = ZerO,
        d2 = ZerO,
        d3 = ZerO,
        d4 = ZerO,
        d5 = ZerO,
        d6 = ZerO,
        d7 = ZerO;
          
      mx      
        d00={0.0,0},
        d01={0.0,0},
          d02={0.0,0},
            d03={0.0,0};
              
            if (lp == (l - 1)) /* use recursion over "b"  */
              {
                A='b';
              }
            else if (lp == (l + 1))               /* use recursion over "a"  */
              {
                A='a';
              }
            else
              {   
                printf(" BadMagic: Don't know what to do here.\n\n");
                fflush( stdout );
                exit(EXIT_FAILURE);
              }   
        
        
            /**************************************************************************************/
            /**************************************************************************************/
            /*   THIS HAS THE GENERAL FORM GIVEN BY (GORDAN 1929):                                */
            /*                                                                                    */
            /*    R(n,l;n',l') = (-1)^(n'-1) [4(2l-1)!]^(-1) *                                    */
            /*                      [(n+l)! (n'+l-1)!/(n-l-1)! (n'-l)!]^(1/2) *                   */
            /*                      (4 n n')^(l+1) (n-n')^(n+n'-2l-2) (n+n')^(-n-n') *            */
            /*                      { F(-n+l+1,-n'+l;2l;-4nn'/[n-n']^2)                           */
            /*                    - (n-n')^2 (n+n')^2 F(-n+l-1,-n'+l;2l; -4nn'/[n-n']^2 ) }       */
            /**************************************************************************************/
            /**************************************************************************************/
            
           
/***********************************************************************************************************/
            /*   Calculate the whole shootin match                                                         */
            /*                            -                - (1/2)                                         */
            /*             (-1)^(n'-1)   | (n+l)! (n'+l-1)! |                                              */
            /*  fsff() =   ----------- * | ---------------- |      * (4 n n')^(l+1) (n-n')^(n+n'-2l-2)(n+n')^(-n-n')       */
            /*             [4 (2l-1)!]   | (n-l-1)! (n'-l)! |                                              */
            /*                    -                -                                                       */
            /*   This is used in the calculation of hydrogen radial wave function integral for dipole 
transition case       */
            
/**********************************************************************************************************/


            log10_fsf = log10_fsff( n, l, np );


            /******************************************************************************************/
            /******************************************************************************************/
            /******************************************************************************************/
            /******************************************************************************************/
            /*  2_F_1( a, b; c; y )                                                                   */
            /*                                                                                        */
            /*        F21_mx(-n+l+1, -n'+l; 2l; -4nn'/[n-n']^2)                                          */
            /*                                                                                        */
            /*                                                                                        */
            /*      Use a -> a' + 1                                                                   */
            /*                                 _                 _                                    */
            /*              (a' + 1) (1 - x)  |                   |                                   */
            /*      F(a') = ----------------- | F(a'+1) - F(a'+2) | + (a' + 1 + bx -c) F(a'+1)        */
            /*                 (a' + 1 - c)   |                   |                                   */
            /*                                 -                 -                                    */
            /*                                                                                        */
            /*      For the first F() in the solution of the radial integral                          */
            /*                                                                                        */
            /*            a = ( -n + l + 1 )                                                          */
            /*                                                                                        */
            /*      a = -n + l + 1                                                                    */
            /*      max(a)  = max(-n)  + max(l)     + 1                                               */
            /*              = -n       + max(n-1)   + 1                                               */
            /*              = -n       + n-1        + 1                                               */
            /*              = 0                                                                       */
            /*                                                                                        */
            /*      similiarly                                                                        */
            /*                                                                                        */
            /*      min(a) = min(-n)   + min(l)   + 1                                                 */
            /*             = min(-n)   + 0        + 1                                                 */   
            /*             = -n        + 1                                                            */
            /*                                                                                        */
            /*      a -> a' + 1   implies                                                             */
            /*                                                                                        */
            /*      max(a') = -1                                                                      */
            /*      min(a') = -n                                                                      */
            /******************************************************************************************/
            /******************************************************************************************/
            
            /* a plus                                                                         */
            a = ((-n) + l + 1);
            
            /*  for the first 2_F_1 we use b = (-n' + l)                                      */
            b = ((-np) + l);
            
            /*  c is simple                                                                   */
            c = (2 * l);
            
            /**********************************************************************************/
            /**********************************************************************************/
            /*  2_F_1( a, b; c; y )                                                           */
            /*                                                                                */
            /*        F21_mx(-n+l+1, -n'+l; 2l; -4nn'/[n-n']^2)                                  */
            /*                                                                                */
            /*             -4 nn'                                                             */
            /*  where Y = -------- .                                                          */
            /*            (n-n')^2                                                            */
            /*                                                                                */
            /**********************************************************************************/
            /**********************************************************************************/

            d2 = (double) (n - np);
            d3 = (d2 * d2);


            d4 = (OnE / d3);
            d5 = (double) (n * np);
            d6 = (d5 * FouR);


            d7 = -d6;
            y = d7 * d4;



            
/******************************************************************************************************/
            
/******************************************************************************************************/
            /*     *                                 THE GENERAL CASE                                             
*/
            /*     *                 USE RECURSION RELATION FOR HYPERGEOMETRIC FUNCTIONS                          
*/
            /*     *   for F(a,b;c;x) we have from eq.4   D. Hoang-Bing Astron. Astrophys. 238: 449-451 
(1990)    */
            /*     *                                                                                              
*/
            /*     *   (a-c) F(a-1) = a (1-x) [ F(a) - F(a-1) ] + (a + bx - c) F(a)                               
*/
            /*     *                                                                                              
*/

            /*     *                   a (1-x)                       (a + bx - c)                                 
*/
            /*     *         F(a-1) = --------- [ F(a) - F(a-1) ] + -------------- F(a)                           
*/
            /*     *                   (a - c)                          (a - c)                                   
*/
            /*     *                                                                                              
*/
            /*     *                                                                                              
*/
            /*     *     A similiar recusion relation holds for b with a <--> b.                                  
*/
            /*     *                                                                                              
*/
            /*     *                                                                                              
*/
            /*     * we have initial conditions                                                                   
*/
            /*     *                                                                                              
*/
            /*     *                                                                                              
*/
            /*     *  F(0) = 1              with a = -1                                                           
*/
            /*     *                                                                                              
*/
            /*     *               b                                                                              
*/
            /*     * F(-1) = 1 - (---) x    with a = -1                                                           
*/
            /*     *               c                                                                              
*/
            
/******************************************************************************************************/
            
/******************************************************************************************************/
            
            
            /*  2_F_1( long int a, long int b, long int c, (double) y, (string) "a" or "b")   */
            /*        F(-n+l+1,-n'+l;2l;-4nn'/[n-n']^2)                                       */
            d00 = F21_mx( a, b, c, y, A );
            
            
            /**************************************************************/
            /**************************************************************/
            /*  For the second F() in the solution of the radial integral */
            /*                                                            */
            /*        a = ( -n + l - 1 )                                  */
            /*                                                            */
            /*  a = -n + l + 1                                            */
            /*  max(a) = max(-n) + max(l)   - 1                           */
            /*         = -n      + (n - 1)  - 1                           */
            /*         = -2                                               */
            /*                                                            */
            /*  similiarly                                                */
            /*                                                            */
            /*  min(a) = min(-n) + min(l)   - 1                           */
            /*         = (-n)    + 0        - 1                           */
            /*         = -n - 1                                           */
            /*                                                            */
            /*  a -> a' + 1   implies                                     */
            /*                                                            */
            /*  max(a') = -3                                              */
            /*                                                            */
            /*  min(a') = -n - 2                                          */
            /**************************************************************/
            /**************************************************************/
            
            
            /*  a minus                                       */
            a = ((-n) + l - 1);
            
            
            /*  for the first 2_F_1 we use b = (-n' + l)      */
            /*    and does not change                         */
            b = ((-np) + l);
            
            
            /* c is simple                                    */
            c = (2 * l);
            
            
            /**************************************************************/
            /*             -4 nn'                                         */
            /*  where Y = -------- .                                      */
            /*           (n-n')^2                                         */
            /**************************************************************/

            /**************************************************************/
            /*      These are already calculated a few lines up           */
            /*                                                            */
            /*      d2 = (double) (n - np) ;                              */
            /*      d3 = d2 * d2 ;                                        */
            /*      d4 = 1/ d3 ;                                          */
            /*      d5 = (double) (n * np);                               */
            /*      d6 = d5 * 4.0;                                        */
            /*      d7 = - d6;                                            */
            /*      y = d7 * d4;                                          */
            /**************************************************************/

            d01 = F21_mx(a, b, c, y, A );

            /**************************************************************************************/
            /**************************************************************************************/
            /*   THIS HAS THE GENERAL FORM GIVEN BY (GORDAN 1929):                                */
            /*                                                                                    */
            /*    R(n,l;n',l') = (-1)^(n'-1) [4(2l-1)!]^(-1) *                                    */
            /*                      [(n+l)! (n'+l-1)!/(n-l-1)! (n'-l)!]^(1/2) *                   */
            /*                      (4 n n')^(l+1) (n-n')^(n+n'-2l-2) (n+n')^(-n-n') *            */
            /*                      { F(-n+l+1,-n'+l;2l;-4nn'/[n-n']^2)                           */
            /*                    - (n-n')^2 (n+n')^2 F(-n+l-1,-n'+l;2l; -4nn'/[n-n']^2 ) }       */
            /*                                                                                    */
            /*                = fsf * ( F(a,b,c;y)  - d3 * F(a',b',c';y) )                        */
            /*                                                                                    */
            /*        where d3 = (n-n')^2 (n+n')^2                                                */
            /*                                                                                    */
            /**************************************************************************************/
            /**************************************************************************************/
            
            /**************************************************************/
            /**************************************************************/
            /*  Calculate                                                 */
            /*                                                            */
            /*  (n-n')^2                                                  */
            /*  --------                                                  */
            /*  (n+n')^2                                                  */
            /**************************************************************/
            /**************************************************************/
            
            d1 = (double) ((n - np)*(n -np));
            
            d2 = (double) ((n + np)*(n + np));
            
            d3 = (d1 / d2);
            
            d02.x = d01.x;
            d02.m = d01.m * d3;
            
            
            while ( fabs(d02.m) > 1.0e+25 )
              {
                d02.m /= 1.0e+25;
                d02.x += 25;
              }
            
            
            d03.x = d00.x;
            d03.m = d00.m * (OnE - (d02.m/d00.m) * power( TeN , (d02.x - d00.x) ) );
            
            result = pow( TeN, (log10_fsf + d03.x) ) * d03.m;
            
            /********************************************/
            /* The overall sign is given by (-1)^(n'-l) */
            /********************************************/
            /* Calculate (-1)^(n'-l)                    */
            /********************************************/
            if (IS_ODD(np - l))
              {
                result *=-OnE;
              }
            else if ( !(IS_ODD(np - l)) )
              {
                result *= OnE;                  /* Do Nothing */
              }
            else
              {
                printf(" BadMagic: Don't know what to do here.\n\n");
                fflush( stdout );
                exit(EXIT_FAILURE);
              }
            /********************************************/
            /* Another  sign is given by the sign of    */
            /*           (n-n')^(n+n'-2l-2)             */   
            /********************************************/
            if ( (n-np) < 0 )
              {
                if ( IS_ODD( (n + np - (2 * l) - 2) ) )   
                  {
                    result *=-OnE;
                  }
                else
                  {
                    result *=-OnE;
                  }
              }
            else if ( (n-np) > 0 )
              {
                result *= OnE;                  /* Do Nothing */
              }
            else
              {
                printf(" BadMagic: Don't know what to do here.\n\n");
                fflush( stdout );
                exit(EXIT_FAILURE);
              }
            
            
            assert( result != ZerO );
            if ( result < ZerO )
              {
                result *= -OnE;
              }
                   
            return( result );
    }
                    
  /*    Never Get Here !!                                       */
  /* this can't happen but some lint's don't know that          */
  /*lint -e527
    return(EXIT_FAILURE);*/
  /*lint +e527*/
}

static double fsff( long int n, long int l, long int np )
{
  /****************************************************************/
  /*   Calculate the whole shootin match                          */
  /*                    -                - (1/2)                  */
  /*     (-1)^(n'-1)   | (n+l)! (n'+l-1)! |                       */
  /*     ----------- * | ---------------- |                       */
  /*     [4 (2l-1)!]   | (n-l-1)! (n'-l)! |                       */
  /*                    -                -                        */
  /*         * (4 n n')^(l+1) (n-n')^(n+n'-2l-2) (n+n')^(-n-n')   */
  /*                                                              */
  /****************************************************************/
  
  
  long int
    i0 = 0,
    i1 = 0,
    i2 = 0,
    i3 = 0,
    i4 = 0;
  
  double
    d0 = ZerO,
    d1 = ZerO,
    d2 = ZerO,
    d3 = ZerO,
    d4 = ZerO,
    d5 = ZerO;
  
  double
    sigma = OnE;
  
  
  /****************************************************************
   *   Calculate the whole shootin match                          *        
   *     (-1)^(n'-1)   | (n+l)! (n'+l-1)! |                       *
   *     ----------- * | ---------------- |                       *
   *     [4 (2l-1)!]   | (n-l-1)! (n'-l)! |                       *
   *                    -                -                        *
   *         * (4 n n')^(l+1) (n-n')^(n+n'-2l-2) (n+n')^(-n-n')   *
   *                                                              *
   ****************************************************************/
  /* Calculate (-1)^(n'-l) */
  if (IS_ODD(np - l))
    {
      sigma *=-OnE;
    }
  else
    { 
      sigma = OnE;
    }
  assert( sigma != ZerO );
  /*********************/ 
  /* Calculate (2l-1)! */ 
  /*********************/
  i1 = (2 * l - 1);
  if ( i1 < 0 )
    {
      printf( "BadMagic: Relational error amongst n, l, n' and l'\n" );
      exit(EXIT_FAILURE);
    }
  
  /****************************************************************/
  /*   Calculate the whole shootin match                          */
  /*                    -                - (1/2)                  */
  /*     (-1)^(n'-1)   | (n+l)! (n'+l-1)! |                       */
  /*     ----------- * | ---------------- |                       */
  /*     [4 (2l-1)!]   | (n-l-1)! (n'-l)! |                       */
  /*                    -                -                        */
  /*         * (4 n n')^(l+1) (n-n')^(n+n'-2l-2) (n+n')^(-n-n')   */
  /*                                                              */
  /****************************************************************/
      
     
  d0 = pre_factorial[ i1 ];
  d1 = FouR * d0;
  d2 = OnE / d1;
  
  /**********************************************************************/
  /*    We want the (negitive) of this                                  */
  /*    since we really are interested in                               */
  /*    [(2l-1)!]^-1                                                    */
  /**********************************************************************/
     
  sigma = sigma * d2;
  assert( sigma != ZerO );
  
  /**********************************************************************/
  /* Calculate (4 n n')^(l+1)                                           */
  /* power( m , n) calcs m^n                                             */
  /* returns long double with m,n ints                                  */
  /**********************************************************************/
  i0 = (4 * n * np);
  i1 = (l + 1);
  d2 = power( (double)i0 , i1 ) ;
  sigma = sigma * d2;
  assert( sigma != ZerO );
  
  /* Calculate (n-n')^(n+n'-2l-2)                                       */
  i0 = ( n - np );
  i1 = (n + np - (2 * l) - 2);
  d2 = power( (double)i0 , i1 );
  sigma = sigma * d2;
  assert( sigma != ZerO );
  /* Calculate (n+n')^(-n-n')                                           */
  i0 = (n + np);
  i1 = (-n - np);
  d2 = power( (double)i0 , i1 );
  sigma = sigma * d2;
  assert( sigma != ZerO );
  
  /**********************************************************************/ 
  /*                  -                -  (1/2)                         */
  /*                 | (n+l)! (n'+l-1)! |                               */
  /*     Calculate   | ---------------- |                               */
  /*                 | (n-l-1)! (n'-l)! |                               */
  /*                  -                -                                */
  /**********************************************************************/
  
  i1 = (n + l);
  if ( i1 < 0 )
    {
      printf( "BadMagic: Relational error amongst n, l, n' and l'\n" );
      (void) fflush( stdout );  
      exit(EXIT_FAILURE);
    }
  d1 = pre_factorial[ i1 ];
  
  
  i2 = (np + l - 1);
  if ( i2 < 0 )
    {
      printf( "BadMagic: Relational error amongst n, l, n' and l'\n" );
      (void) fflush( stdout );
      exit(EXIT_FAILURE);
    }
  
  d2 = pre_factorial[ i2 ];
  
  i3 = (n - l - 1);
  if ( i3 < 0 )
    {
      printf( "BadMagic: Relational error amongst n, l, n' and l'\n" );
      (void) fflush( stdout );
      exit(EXIT_FAILURE);
    }
      
  d3 = pre_factorial[ i3 ];
  
  i4 = (np - l);
  if ( i4 < 0 )
    {
      printf( "BadMagic: Relational error amongst n, l, n' and l'\n" );
      exit(EXIT_FAILURE);
    }
      
  d4 = pre_factorial[ i4 ];
     
  assert( d3 != ZerO );
  assert( d4 != ZerO );
  
  d5 = (sqrt(d1 *d2));
  d5 /= sqrt(d3);
  d5 /= sqrt(d4);
      
  sigma = sigma * d5;
      
  assert( sigma != ZerO );
      
  return( sigma );
}    

/**************************log version*******************************/
static double log10_fsff( long int n, long int l, long int np )
{ 
  /******************************************************************************************************/
  /******************************************************************************************************/
  /*   Calculate the whole shootin match                                                                */
  /*                    -                - (1/2)                                                        */
  /*         1         | (n+l)! (n'+l-1)! |                                                             */
  /*     ----------- * | ---------------- |      * (4 n n')^(l+1) (n-n')^(n+n'-2l-2) (n+n')^(-n-n')     */
  /*     [4 (2l-1)!]   | (n-l-1)! (n'-l)! |                                                             */
  /*                    -                -                                                              */
  /******************************************************************************************************/
  /******************************************************************************************************/
        
        
        
  double
    d0 = ZerO,
    d1 = ZerO;
        
  double
    ld0 = ZerO,
    ld1 = ZerO,
    ld2 = ZerO,
    ld3 = ZerO,
    ld4 = ZerO;
        
  double
    result = ZerO;
        
        
        
/******************************************************************************************************/
        
/******************************************************************************************************/
        /*    Calculate the log10 of the whole shootin match                                                  
*/
        /*                    -                - (1/2)                                                        
*/
        /*          1        | (n+l)! (n'+l-1)! |                                                             
*/
        /*     ----------- * | ---------------- |      * (4 n n')^(l+1) (n-n')^(n+n'-2l-2) (n+n')^(-n-n')     
*/
        /*     [4 (2l-1)!]   | (n-l-1)! (n'-l)! |                                                             
*/
        /*                    -                -                                                              
*/
        
/******************************************************************************************************/
        
/******************************************************************************************************/
        
        /**********************/
        /* Calculate (2l-1)!  */
        /**********************/
  d0 = (double)(2 * l - 1);
  assert( d0 != ZerO );
    
        
/******************************************************************************************************/
        
/******************************************************************************************************/
        /*    Calculate the whole shootin match                                                               
*/
        /*                    -                - (1/2)                                                        
*/
        /*         1         | (n+l)! (n'+l-1)! |                                                             
*/
        /*     ----------- * | ---------------- |      * (4 n n')^(l+1) |(n-n')^(n+n'-2l-2)| (n+n')^(-n-n')   
*/
        /*     [4 (2l-1)!]   | (n-l-1)! (n'-l)! |                                                             
*/
        /*                    -                -                                                              
*/
        
/******************************************************************************************************/
        
/******************************************************************************************************/
        
  ld0 = log10_factorial( (2 * l - 1) );
  ld1 = log10(FouR);
  result = -(ld0 + ld1);
  assert( result != ZerO );
        
        
        /**********************************************************************/
        /* Calculate (4 n n')^(l+1)                                           */
        /* power( m , n) calcs m^n                                            */
        /* returns long double with m,n ints                                  */
        /**********************************************************************/
  d0 = (double)(4 * n * np);
  d1 = (double)(l + 1);
  result  += d1 * log10(d0) ;
  assert( d0 >= ZerO );
  assert( d1 != ZerO );
        
        
  /**********************************************************************/
  /* Calculate |(n-n')^(n+n'-2l-2)|                                     */
  /*    NOTE: Here we are interested only                               */
  /*             magnitude of (n-n')^(n+n'-2l-2)                        */
  /**********************************************************************/
  d0 = (double)( n - np );
  d1 = (double)(n + np - (2 * l) - 2);
  result  += d1 * log10(fabs(d0)) ;
  assert( fabs(d0) > ZerO );
  assert( d1 != ZerO );
        
        
  /* Calculate (n+n')^(-n-n') */
  d0 = (double) (n + np);
  d1 = (double) (-n - np);
  result  += d1 * log10(d0) ;
  assert( d0 > ZerO );
  assert( d1 != ZerO );
  
  
  /**********************************************************************/
  /*                  -                -  (1/2)                         */
  /*                 | (n+l)! (n'+l-1)! |                               */
  /*     Calculate   | ---------------- |                               */
  /*                 | (n-l-1)! (n'-l)! |                               */
  /*                  -                -                                */
  /**********************************************************************/
  
  
  assert( (n+l) >  ZerO );
  ld0 = log10_factorial( (n + l) );
  
  assert( (np+l-1) > ZerO );
  ld1 = log10_factorial( (np + l - 1) );
        
  assert( (n-l-1) >= ZerO );
  ld2 = log10_factorial( (n - l - 1) );
  
  assert( (np-l) >= ZerO );  
  ld3 = log10_factorial( (np - l) );
  
  ld4 = (OnE/TwO)*((ld0+ld1)-(ld2+ld3));
  
  result += ld4;
  assert( result != ZerO );
  
  return( result );
}


/***************************************************************************/
/*   Find the Oscillator Strength for hydrogen for any                     */
/*   transition n,l -->  n',l'                                             */
/*   returns a double                                                      */
/***************************************************************************/
/***************************************************************************/
#define       R_oo          (2.1784874e-11)             /* ergs   */
/************************************************************************/
/************************************************************************/
/*   Find the Oscillator Strength for hydrogen for any                  */
/*   transition n,l -->  n',l'                                          */
/*   returns a double                                                   */
/*                                                                      */
/*   Einstein A() for the transition from the                           */
/*   initial state n,l to the finial state n',l'                        */
/*   require  the  Oscillator Strength  f()                             */
/*                                                                      */
/*                     hbar w    max(l,l')  |              | 2          */
/*   f(n,l;n',l') = - -------- ------------ | R(n,l;n',l') |            */
/*                     3 R_oo   ( 2l + 1 )  |              |            */
/*                                                                      */
/*                                                                      */
/*                                                                      */   
/*                     E(n,l;n',l')     max(l,l')  |              | 2   */   
/*   f(n,l;n',l') = -  ------------   ------------ | R(n,l;n',l') |     */   
/*                      3 R_oo         ( 2l + 1 )  |              |     */   
/*                                                                      */   
/*                                                                      */   
/*   See for example Gordan Drake's                                     */   
/*      Atomic, Molecular, & Optical Physics Handbook pg.638            */
/*                                                                      */
/*   Here R_oo is the infinite mass Rydberg length                      */
/*                                                                      */
/*                                                                      */
/*        h c                                                           */
/*   R_oo --- = 13.605698 eV                                            */
/*        {e}                                                           */
/*                                                                      */
/*                                                                      */
/*   R_oo =  2.179874e-11 ergs                                          */
/*                                                                      */
/*   w = omega                                                          */
/*     = frequency of transition from n,l to n',l'                      */
/*                                                                      */
/*                                                                      */
/*                                                                      */   
/*     here g_k are statistical weights obtained from                   */   
/*              the appropriate angular momentum quantum numbers        */   
/************************************************************************/   
/************************************************************************/   


/********************************************************************************/
/*   Calc the Oscillator Strength f(*) given by                                 */
/*                                                                              */
/*                     E(n,l;n',l')     max(l,l')  |              | 2           */
/*   f(n,l;n',l') = -  ------------   ------------ | R(n,l;n',l') |             */
/*                      3 R_oo         ( 2l + 1 )  |              |             */
/*                                                                              */
/*   See for example Gordan Drake's                                             */
/*      Atomic, Molecular, & Optical Physics Handbook pg.638                    */
/********************************************************************************/


/************************************************************************/
/*   Calc the Oscillator Strength f(*) given by                         */
/*                                                                      */
/*                     E(n,l;n',l')     max(l,l')  |              | 2   */   
/*   f(n,l;n',l') = -  ------------   ------------ | R(n,l;n',l') |     */   
/*                      3 R_oo         ( 2l + 1 )  |              |     */   
/*                                                                      */   
/*       f(n,l;n',l') is dimensionless.                                 */   
/*                                                                      */   
/*   See for example Gordan Drake's                                     */
/*      Atomic, Molecular, & Optical Physics Handbook pg.638            */
/*                                                                      */
/*  In the following, we have n > n'                                    */
/************************************************************************/
double OscStr_f(
                /*  IN THE FOLLOWING WE HAVE  n > n'                    */
                /* principal quantum number, 1 for ground, upper level  */
                long int n,
                /* angular momentum, 0 for s                            */
                long int l,
                /* principal quantum number, 1 for ground, lower level  */
                long int np,
                /* angular momentum, 0 for s                            */
                long int lp,
                /* Nuclear charge, 1 for H+, 2 for He++, etc            */
                long int iz
                )
{
  double
    d0 = ZerO,
    d1 = ZerO,
    d2 = ZerO,
    d3 = ZerO,
    d4 = ZerO,
    d5 = ZerO,
    d6 = ZerO;


                
  long int
    i1 = 0,
    i2 = 0;
                
  if ( l > lp )
    i1 = l;
  else
    i1 = lp;
                
  i2 = (2 * lp + 1);
  d0 = OnE / ThreE ;
  d1 = (double) i1 / (double) i2;
  /* hv() returns energy in ergs */
  d2 = hv( n, np, iz );
  d3 = d2 / R_oo;
  d4 = hri( n, l, np, lp ,iz );
  d5 = d4 * d4;
    
  d6 = d0 * d1 * d3 * d5;
    
  return( d6 );
}
                
/************************log version ***************************/
double OscStr_f_log10( long int n , long int l , long int np , long int lp , long int iz )
{
  double
    d0 = ZerO,
    d1 = ZerO,
    d2 = ZerO,
    d3 = ZerO,
    d4 = ZerO,
    d5 = ZerO,
    d6 = ZerO;

  long int
    i1 = 0,
    i2 = 0;

  if ( l > lp )
    i1 = l;
  else
    i1 = lp;

  i2 = (2 * lp + 1);
  d0 = OnE / ThreE ;
  d1 = (double) i1 / (double) i2;
  /* hv() returns energy in ergs        */
  d2 = hv( n, np, iz );
  d3 = d2 / R_oo;
  d4 = hri_log10( n, l, np, lp ,iz );
  d5 = d4 * d4;
    
  d6 = d0 * d1 * d3 * d5;
    
  return( d6 );
}

/***********************/
/*                     */
/*      2_F_1.c        */
/*                     */
/***********************/
  
#define F21_Y(I_)        (*(yV+I_))
static double F21( long int a , long int b, long int c, double y, char A )
{ 
  double
    d1 = ZerO; 
      
  long int
    i1 = 0,
    i  = 0;
  
  double *yV;
  
  
  /**************************************************************/
  /*  A must be either 'a' or 'b'                               */
  /*  and is use to determine which                             */
  /*  variable recursion will be over                           */
  /**************************************************************/
      
  assert(  A == 'a' || A == 'b' );
     
  if ( A == 'b' )
    {
      /*        if the recursion is over "b" */
      /*        then make it over "a" by switching these around */
  
      i1 = a  ;
      a  = b  ;
      b  = i1 ;
      A  = 'a';
    }
  /**************************************************************************************/
  /**************************************************************************************/
  /*    malloc space for the  (dynamic) 1-d array                                       */
  /*    2_F_1 works via recursion and needs room to store intermediate results          */
  /*    Here the + 5 is a safety margin                                                 */
  /**************************************************************************************/
  /**************************************************************************************/
  if ( (yV=(double*) /*Must use CALLOC*/ CALLOC( sizeof(double), (size_t)(-a + 5) )) == NULL )
    {
      printf("F21 main: BadMagic: CALLOC yV failed\n");
      (void) fflush( stdout );
      exit(EXIT_FAILURE);
    }
  /**********************************************************************************************/
  /**********************************************************************************************/
  /*  begin sanity check, check order, and that none negative                                   */
  /*                                  THE GENERAL CASE                                          */
  /*                  USE RECURSION RELATION FOR HYPERGEOMETRIC FUNCTIONS                       */
  /*    for F(a,b;c;x) we have from eq.4   D. Hoang-Bing Astron. Astrophys. 238: 449-451 (1990) */
  /*                                                                                            */
  /*    (a-c) F(a-1) = a (1-x) [ F(a) - F(a-1) ] + (a + bx - c) F(a)                            */
  /*                                                                                            */
  /*                    a (1-x)                        (a + bx - c)                             */
  /*           F(a-1) = --------- [ F(a) - F(a-1) ] + -------------- F(a)                       */
  /*                     (a-c)                            (a-c)                                 */
  /*                                                                                            */
  /*                                                                                            */
  /*      A similiar recusion relation holds for b with a <--> b.                               */
  /*                                                                                            */
  /*                                                                                            */
  /*  we have initial conditions                                                                */
  /*                                                                                            */
  /*                                                                                            */
  /*   F(0) = 1              with a = -1                                                        */
  /*                                                                                            */
  /*                b                                                                           */
  /*  F(-1) = 1 - (---) x    with a = -1                                                        */
  /*                c                                                                           */
  /*                                                                                            */
  /*  For the first F() in the solution of the radial integral                                  */
  /*                                                                                            */
  /*        a = ( -n + l + 1 )                                                                  */
  /*                                                                                            */
  /*  a = -n + l + 1                                                                            */
  /*  max(a) = max(-n) + max(l)     + 1                                                         */
  /*         = max(-n) + max(n - 1) + 1                                                         */
  /*         = max(-n + n - 1)      + 1                                                         */
  /*         = max(-1)              + 1                                                         */
  /*         = 0                                                                                */
  /*                                                                                            */
  /*  similiarly                                                                                */
  /*                                                                                            */
  /*  min(a) = min(-n) + min(l)   + 1                                                           */
  /*         = min(-n) + 0        + 1                                                           */
  /*         = (-n)    + 0        + 1                                                           */
  /*         = -n + 1                                                                           */
  /*                                                                                            */
  /*  a -> a' + 1   implies                                                                     */
  /*                                                                                            */
  /*  max(a') = -1                                                                              */
  /*  min(a') = -n                                                                              */
  /*                                                                                            */
  /*  For the second F() in the solution of the radial integral                                 */
  /*                                                                                            */
  /*        a = ( -n + l - 1 )                                                                  */
  /*                                                                                            */
  /*  a = -n + l + 1                                                                            */
  /*  max(a) = max(-n) + max(l)   - 1                                                           */
  /*         = -n      + (n - 1)  - 1                                                           */
  /*         = -2                                                                               */
  /*                                                                                            */
  /*  similiarly                                                                                */
  /*                                                                                            */
  /*  min(a) = min(-n) + min(l)   - 1                                                           */
  /*         = (-n)    + 0        - 1                                                           */
  /*         = -n      - 1                                                                      */
  /*                                                                                            */
  /*  a -> a' + 1   implies                                                                     */
  /*                                                                                            */
  /*  max(a') = -3                                                                              */
  /*  min(a') = -n - 2                                                                          */
  /**********************************************************************************************/
  /**********************************************************************************************/
  
  assert( a <= 0 );
  assert( b <= 0 );
  assert( c >= 0 );
  
  /* ZERO out yV before each call */
  for (i = 0 ; i < -a +5 ; i = i + 1)
    {
      F21_Y(i) = ZerO ;
    }
  
  d1 = F21i(a, b, c, y,  yV );
  free( yV );
  
  return( d1 );
}

static mx F21i_log( long int a , long int b , long int c , double y , mxq *yV );
static mx F21_mx( long int a , long int b, long int c, double y, char A )
{ 
  long int
    i  = 0;
          
  mx
    result_mx = {0.0,0};
      
  mxq *yV = NULL;
          
  /**************************************************************/
  /*  A must be either 'a' or 'b'                               */
  /*  and is use to determine which                             */
  /*  variable recursion will be over                           */
  /**************************************************************/
          
  assert(  A == 'a' || A == 'b' );
         
  if ( A == 'b' )
    {
      /*        if the recursion is over "b"                            */
      /*        then make it over "a" by switching these around         */
      long int
        i1 = a  ;
  
      a  = b  ;
      b  = i1 ;
      A  = 'a';
    }
  /**************************************************************************************/
  /**************************************************************************************/
  /*    malloc space for the  (dynamic) 1-d array                                       */
  /*    2_F_1 works via recursion and needs room to store intermediate results          */
  /*    Here the + 5 is a safety margin                                                 */
  /**************************************************************************************/
  /**************************************************************************************/
  if ( (yV=(mxq *) /*Must use CALLOC*/ CALLOC( sizeof(mxq), (size_t)(-a + 5) )) == NULL )
    {
      printf("F21_mx main: BadMagic: CALLOC yV failed\n");	
      fflush( stdout );
      exit(EXIT_FAILURE);
    }
  /**********************************************************************************************/
  /**********************************************************************************************/
  /*  begin sanity check, check order, and that none negative                                   */
  /*                                  THE GENERAL CASE                                          */
  /*                  USE RECURSION RELATION FOR HYPERGEOMETRIC FUNCTIONS                       */
  /*    for F(a,b;c;x) we have from eq.4   D. Hoang-Bing Astron. Astrophys. 238: 449-451 (1990) */
  /*                                                                                            */
  /*    (a-c) F(a-1) = a (1-x) [ F(a) - F(a-1) ] + (a + bx - c) F(a)                            */
  /*                                                                                            */
  /*                    a (1-x)                        (a + bx - c)                             */
  /*           F(a-1) = --------- [ F(a) - F(a-1) ] + -------------- F(a)                       */
  /*                     (a-c)                            (a-c)                                 */
  /*                                                                                            */
  /*                                                                                            */
  /*      A similiar recusion relation holds for b with a <--> b.                               */
  /*                                                                                            */
  /*                                                                                            */
  /*  we have initial conditions                                                                */
  /*                                                                                            */
  /*                                                                                            */
  /*   F(0) = 1              with a = -1                                                        */
  /*                                                                                            */
  /*                b                                                                          */
  /*  F(-1) = 1 - (---) x    with a = -1                                                       */ 
  /*                c                                                                          */ 
  /*                                                                                           */ 
  /*  For the first F() in the solution of the radial integral                                 */ 
  /*                                                                                           */ 
  /*        a = ( -n + l + 1 )                                                                 */ 
  /*                                                                                           */ 
  /*  a = -n + l + 1                                                                           */ 
  /*  max(a) = max(-n) + max(l)     + 1                                                        */ 
  /*         = max(-n) + max(n - 1) + 1                                                        */ 
  /*         = max(-n + n - 1)      + 1                                                        */ 
  /*         = max(-1)              + 1                                                        */ 
  /*         = 0                                                                               */ 
  /*                                                                                           */ 
  /*  similiarly                                                                               */ 
  /*                                                                                           */ 
  /*  min(a) = min(-n) + min(l)   + 1                                                          */ 
  /*         = min(-n) + 0        + 1                                                          */ 
  /*         = (-n)    + 0        + 1                                                          */ 
  /*         = -n + 1                                                                          */ 
  /*                                                                                           */ 
  /*  a -> a' + 1   implies                                                                    */ 
  /*                                                                                           */
  /*  max(a') = -1                                                                             */ 
  /*  min(a') = -n                                                                             */ 
  /*                                                                                           */ 
  /*  For the second F() in the solution of the radial integral                                */ 
  /*                                                                                           */ 
  /*        a = ( -n + l - 1 )                                                                 */ 
  /*                                                                                           */ 
  /*  a = -n + l + 1                                                                           */ 
  /*  max(a) = max(-n) + max(l)   - 1                                                          */ 
  /*         = -n      + (n - 1)  - 1                                                          */ 
  /*         = -2                                                                              */ 
  /*                                                                                           */ 
  /*  similiarly                                                                               */ 
  /*                                                                                           */ 
  /*  min(a) = min(-n) + min(l)   - 1                                                          */ 
  /*         = (-n)    + 0        - 1                                                          */ 
  /*         = -n      - 1                                                                     */ 
  /*                                                                                           */ 
  /*  a -> a' + 1   implies                                                                    */ 
  /*                                                                                           */ 
  /*  max(a') = -3                                                                             */ 
  /*  min(a') = -n - 2                                                                         */ 
  /**********************************************************************************************/
  /**********************************************************************************************/
  assert( a <= 0 );
  assert( b <= 0 );
  assert( c >= 0 );
  
  
  /* ZERO out yV before each call               */
  for (i = 0 ; i < -a +5 ; i++ )
    {
      ((yV+i) -> mx).m = 0.;
      ((yV+i) -> mx).x = 0;
      (yV+i) -> q  =  0;
    }
  
  result_mx = F21i_log(a, b, c, y,  yV );
  free( yV );
  
  return( result_mx );
}
  
static double F21i(long int a, long int b, long int c, double y, double *yV )
{
  
  double
    d0 = ZerO,
    d1 = ZerO,
    d2 = ZerO,
    d3 = ZerO,
    d4 = ZerO,
    d5 = ZerO,
    d8 = ZerO,
    d9 = ZerO,
    d10 = ZerO,
    d11 = ZerO,
    d12 = ZerO,
    d13 = ZerO,
    d14 = ZerO;
  
  long int
    i1 = 0,
    i2 = 0;
  
  if ( a == 0 )
    {
      return( OnE );
    }
  else if ( a  == -1 )
    {
      assert( c != 0);
      d1 = (double)b;
      d2 = (double)c;
      d3 = ( y * (d1/d2) );
      d4 = OnE - d3 ;
    
      return( d4 );
    }
  /* Check to see if y(-a) != 0 in a very round about way to avoid lclint:error:13 */
  else if ( ( F21_Y(-a) != ZerO ) )
    {
      /* Return the stored result */
      return( F21_Y(-a) );
    }
  else
    {
      /******************************************************************************************/
      /******************************************************************************************/
      /*                             -             -                                            */
      /*             (a)(1 - y)     |               |       (a + bx + c)                        */
      /*   F(a-1) = --------------  | F(a) - F(a+1) | +   --------------- F(a+1)                */
      /*               (a - c)      |               |        (a - c)                            */
      /*                             -             -                                            */
      /*                                                                                        */
      /*                                                                                        */
      /*                                                                                        */
      /*                                                                                        */
      /*                                                                                        */
      /*   with     F(0) = 1                                                                    */
      /*                         b                                                              */
      /*   and     F(-1) = 1 - (---) y                                                          */
      /*                         c                                                              */
      /*                                                                                        */
      /*                                                                                        */
      /*                                                                                        */
      /*   Use a -> a' + 1                                                                      */
      /*                               _                 _                                      */
      /*           (a' + 1) (1  - x)  |                   |    (a' + 1 + bx - c)                */
      /*   F(a') = -----------------  | F(a'+1) - F(a'+2) | +  ----------------- F(a'+1)        */
      /*              (a' + 1 - c)    |                   |      (a' + 1 - c)                   */
      /*                               -                 -                                      */
      /*                                                                                        */
      /*   For the first F() in the solution of the radial integral                             */
      /*                                                                                        */
      /*         a = ( -n + l + 1 )                                                             */
      /*                                                                                        */
      /*   a = -n + l + 1                                                                       */
      /*   max(a) = max(-n) + max(l)     + 1                                                    */
      /*          = -n      + max(n-1)   + 1                                                    */
      /*          = -n      + n-1        + 1                                                    */
      /*          = 0                                                                           */
      /*                                                                                        */
      /*   similiarly                                                                           */
      /*                                                                                        */
      /*   min(a) = min(-n)   + min(l)   + 1                                                    */
      /*          = min(-n)   + 0        + 1                                                    */
      /*          = -n + 1                                                                      */
      /*                                                                                        */
      /*   a -> a' + 1   implies                                                                */
      /*                                                                                        */
      /*   max(a') = -1                                                                         */
      /*   min(a') = -n                                                                         */
      /******************************************************************************************/
      /******************************************************************************************/
      
      i1= (a + 1);
      
      i2= (a + 1 - c);
      d0= (double)i2;
      assert( i2 != 0);
      d1= OnE - y;
      d2= ( (double)i1 * d1);
      d3= (d2 / d0);
      d4= ( (double)b * y);
      d5= ( d0 + d4 );
      
      d8= F21i( (long int)(a + 1), b, c, y, yV );
      
      d9= F21i( (long int)(a + 2), b, c, y, yV );
      
      d10= (d8 - d9);
      d11= (d3 * d10);
      d12= (d5 / d0);
      d13= (d12 * d8);
      d14= (d11 + d13);
      
      /* Store the result for latter use */
      F21_Y(-a) =  d14;
      
      return( d14 );
    }
}


static mx F21i_log( long int a, long int b, long int c, double y, mxq *yV )
{
  mx
    result_mx = {0.0,0};
  
  if ( yV(-a).q != ZerO )
    {
      /* Return the stored result   */
  
      return( yV(-a).mx );
    }
  else if ( a == 0 )
    {
      assert( yV(-a).q == ZerO );
     
      result_mx.m = OnE;
      result_mx.x = 0;
  
      while ( fabs(result_mx.m) > 1.0e+25 )
        {
          result_mx.m /= 1.0e+25;
          result_mx.x += 25;
        }

      assert( ((((yV+ (-a)) -> mx).m) ) == ZerO );
      assert( ((((yV+ (-a)) -> mx).x) ) == 0 );

   
      yV(-a).q = 1;
      yV(-a).mx = result_mx;
     
      return( result_mx );
    }
  else if ( a  == -1 )
    {
      double
        d1 = (double) b,
        d2 = (double) c,
        d3 = ( y * (d1/d2) );
      
      assert( yV(-a).q == ZerO );
  
      assert( c != 0 );
         
      assert( y != ZerO );
          
      result_mx.m = (OnE - d3);
      result_mx.x = 0;
      
      while ( fabs(result_mx.m) > 1.0e+25 )
        {
          result_mx.m /= 1.0e+25;
          result_mx.x += 25;
        }
      
      assert( ((((yV+ (-a)) -> mx).m) ) == ZerO );
      assert( ((((yV+ (-a)) -> mx).x) ) == ZerO );
     
      
      yV(-a).q = 1;
      yV(-a).mx = result_mx;
        
      
      return( result_mx );
    }
  else
    {    
      /******************************************************************************************/
      /******************************************************************************************/
      /*                             -             -                                            */
      /*             (a)(1 - y)     |               |       (a + bx + c)                        */
      /*   F(a-1) = --------------  | F(a) - F(a+1) | +   --------------- F(a+1)                */
      /*               (a - c)      |               |         (a - c)                           */
      /*                                                                                        */
      /*                                                                                        */
      /*                                                                                        */
      /*   with     F(0) = 1                                                                    */
      /*                         b                                                              */
      /*   and     F(-1) = 1 - (---) y                                                          */
      /*                         c                                                              */
      /*                                                                                        */
      /*                                                                                        */
      /*                                                                                        */
      /*   Use a -> a' + 1                                                                      */
      /*                               _                 _                                      */
      /*           (a' + 1) (1  - x)  |                   |    (a' + 1 + bx - c)                */
      /*   F(a') = -----------------  | F(a'+1) - F(a'+2) | +  ----------------- F(a'+1)        */
      /*              (a' + 1 - c)    |                   |      (a' + 1 - c)                   */
      /*                               -                 -                                      */
      /*                                                                                        */
      /*   For the first F() in the solution of the radial integral                             */
      /*                                                                                        */
      /*         a = ( -n + l + 1 )                                                             */
      /*                                                                                        */
      /*   a = -n + l + 1                                                                       */
      /*   max(a) = max(-n) + max(l)     + 1                                                    */
      /*          = -n      + max(n-1)   + 1                                                    */
      /*          = -n      + n-1        + 1                                                    */
      /*          = 0                                                                           */
      /*                                                                                        */
      /*   similiarly                                                                           */
      /*                                                                                        */
      /*   min(a) = min(-n)   + min(l)   + 1                                                    */
      /*          = min(-n)   + 0        + 1                                                    */
      /*          = -n + 1                                                                      */
      /*                                                                                        */
      /*   a -> a' + 1   implies                                                                */
      /*                                                                                        */
      /*   max(a') = -1                                                                         */
      /*   min(a') = -n                                                                         */
      /******************************************************************************************/
      /******************************************************************************************/
      
      mx d8={0.0,0}, d9={0.0,0}, d10={0.0,0}, d11={0.0,0};
      
      double
        db  = (double) b,
        d00 = (double) (a + 1 - c),
        d0  = (double) (a + 1),
        d1  = (OnE - y),
        d2  = (d0  * d1),
        d3  = (d2  / d00),
        d4  = (db  * y),
        d5  = (d00 + d4 ),
        d6  = (d5  / d00);
      
      assert( yV(-a).q == ZerO );
      
      
      /******************************************************************************************/
      /******************************************************************************************/
      /*                               _                 _                                      */
      /*           (a' + 1) (1  - x)  |                   |    (a' + 1 - c) + b*x               */
      /*   F(a') = -----------------  | F(a'+1) - F(a'+2) | +  ------------------ F(a'+1)       */
      /*              (a' + 1 - c)    |                   |      (a' + 1 - c)                   */
      /*                               -                 -                                      */
      /******************************************************************************************/
      /******************************************************************************************/
      d8= F21i_log( (a + 1), b, c, y, yV );
        
      d9= F21i_log( (a + 2), b, c, y, yV );
        
      if ( (d8.m) != ZerO )
        {
          d10.x = d8.x; 
          d10.m = d8.m * (OnE - (d9.m/d8.m) * power( TeN, (d9.x - d8.x)));
        }
      else
        {
          d10.m = -d9.m;
          d10.x =  d9.x;
        }
      
      
      d10.m *= d3;
      
      d11.x = d8.x;
      
      d11.m= (d6 * d8.m);
      
      if (  (d11.m) != ZerO )
        {
          result_mx.x = d11.x;
          result_mx.m = d11.m * (OnE + (d10.m/d11.m) * power( TeN , (d10.x - d11.x) ) );
        }
      else
        {
          result_mx = d10;
        }
         
      
      while ( fabs(result_mx.m) > 1.0e+25 )
        {
          result_mx.m /= 1.0e+25;
          result_mx.x += 25;
        }
      
      /* Store the result for latter use                */
      yV(-a).q = 1;
      yV(-a).mx =  result_mx;
      
      return( result_mx );
    } 
}


#ifndef IS_ODD
#define IS_ODD(j)      (int)((j) & 1 )
#endif
      
static double power( double x, long int n )
     /* returns:        x^n             */
     /* x;              base            */
     /* n;              exponent        */
{
    /* "p" holds partial product */
  double p;
      
  if ( x == 0 )
    return( 0. );
      
  /* test for negative exponent */
  if ( n < 0 )
    {
      n = -n;
      x = 1/x;
    } 

  /* test & set zero power */
  p = IS_ODD(n) ? x : 1;
      
  /* now do the other powers            */ 
  /* sq previous power of x             */
  /* if low order bit set               */
  /* then, multiply partial             */
  /* product by latest power of x       */
  /* this really is a shift, not typo with >= */
  while ( n >>= 1 )
    {
      x *= x;
      if ( IS_ODD(n) )
        p *= x;
    }
      
  return( p );
}

static double log10_factorial( long int n )
{
  /******************************/
  /*                 n          */
  /*                ---         */
  /*    log( n! ) =  >  log(j)  */
  /*                ---         */
  /*                j=1         */
  /******************************/
      
  double
    result = ZerO;
     
  long int
    j = 0;
  
  if ( n < 0 )
    {
      fprintf(ioQQQ,"Domain Error: in function log10_factorial(n) -> n<0 \n");
      fprintf(ioQQQ, "n = %li" , n);
      puts( "[Stop in h_bauman_log.c]" );
      exit(EXIT_FAILURE);
    }

  /* 0! = 1 --> log10(1) = 0    */
  if ( n == 0 ) return( 0.0 );
  
  /* 1! = 1 --> log10(1) = 0    */
  if ( n == 1 ) return( 0.0 );
  
  for ( j = 1; j <= n; j++ )
    { 
      double jj = (double) j;
      result += log10( jj );
     
    }
  return( result );
} 
  
     
#if 0
/* used as parameters in qg32 integration */
static double vJint , zJint;  
        
static double Jint( double theta )
{     
  const double Pi = (double) PI;
  /*                                    */
  /*   [ cos[vx - z sin[x]] ]           */
  /*                                    */
  double
    d0 = ( 1.0 / Pi ),
    d1 = vJint * theta,
    d2 = zJint * sin(theta),
    d3 = (d1 - d2),
    d4 = cos(d3),
    d5 = (d0 * d4);
  
  return( d5 );
}
  
static double AngerJ( double vv, double zz )
{
  const double Pi = (double) PI;

  long int rep , ddiv, divsor;
        
  double y = 0.0;
      
  rep  = 0;
  
  /* Estimate number of peaks in integrand.  */
  /* Divide region of integration by number  */
  /*  peaks in region.                       */
  if( (fabs(vv)) - (int)(fabs(vv)) > 0.5 )
    ddiv = (int)(fabs(vv)) + 1;
  else
    ddiv = (int)(fabs(vv));
    
  divsor  = ((ddiv == 0) ? 1 : ddiv);
  vJint = vv;
  zJint = zz;  
 
  for( rep = 0; rep < divsor; rep++ )
    {
      double
        rl = (((double) rep)/((double) divsor)),
        ru = (((double) (rep+1))/((double) divsor)),
        x_low = (Pi * rl),
        x_up  = (Pi * ru);
  
      y += qg32( x_low, x_up, Jint );
    }
  
  return( y );
}
  
/******************************************************************************/
/******************************************************************************/
/*                                                                            */
/*    Semi-Classical Quantum Defect Radial Integral                           */
/*                                                                            */
/*   See for example                                                          */
/*     Atomic, Molecular & Optical Physics Handbook                           */
/*     Gordon W. F. Drake; Editor                                             */
/*     AIP Press                                                              */
/*     Woddbury, New York.                                                    */
/*     1996                                                                   */
/*                                                                            */
/* NOTE:: we do not include the Bohr Radius a_o in the                        */
/*           definition of of  R(n,L;n'L') as per Drake.                      */
/*                                                                            */
/*                                                                            */
/*                   1  (n_c)^2 | {      D_l max(L,L') }                      */
/*    R(n,L;n'L') = --- ------- | { 1 -  ------------- } J( D_n-1 ; -x )  -   */
/*                   Z   2 D_n  | {          n_c       }                      */
/*                                                                            */
/*                                                                            */
/*                    {      D_L max(L,L') }                                  */
/*                -   { 1 +  ------------- } J( D_n+1 ; -x )                  */
/*                    {          n_c       }                                  */
/*                                                                            */
/*                                                                            */
/*                     2                    |                                 */
/*                  + ---  sin(Pi D_n)(1-e) |                                 */
/*                     Pi                   |                                 */
/*                                                                            */
/*  where                                                                     */
/*        n_c = (2n*'n*)/(n*'+n*)                                             */
/*                                                                            */
/*       Here is the quantity Drake gives...                                  */
/*            n_c = ( 2.0 * nstar * npstar ) / ( nstar + npstar ) ;           */
/*                                                                            */
/*       while V.A. Davidkin uses                                             */
/*            n_c = sqrt(  nstar * npstar  );                                 */
/*                                                                            */
/*        D_n = n*' - n*                                                      */
/*                                                                            */
/*        D_L = L*' - L*                                                      */
/*                                                                            */
/*        x = e D_n                                                           */
/*                                                                            */
/*        Lmx  = max(L',L)                                                    */
/*                                                                            */
/*        e = sqrt( 1 - {Lmx/n_c}^2 )                                         */
/*                                                                            */
/*                                                                            */
/*       Here n* = n - qd where qd is the quantum defect                      */
/*                                                                            */
/******************************************************************************/
/******************************************************************************/

double scqdri(
              double nstar,
              long   int l,
              double npstar,
              long   int lp,
              double iz
              )
{
  const double TwobyPi = (double) TWOBYPI; /* 2/Pi */

  double n_c = ((2.0 * nstar * npstar ) / ( nstar + npstar ));
  double D_n = (nstar - npstar);
  double D_l = (double) ( l - lp );
  double lg  = (double) ( (lp > l) ? lp : l);

  double h = (lg/n_c);
  double g = h*h;
  double f = ( 1.0 - g );
  double e = (( f >= 0.0) ? sqrt( f ) : 0.0 );

  double x  = (e * D_n);
  double z  = (-1.0 * x);
  double v1 = (D_n + 1.0);
  double v2 = (D_n - 1.0);
              
  double d1,d2,d3,d4,d5,d6,d7,d8,d9;
  double d34,d56,d6_1;

  if ( iz == 0.0 )
    iz += 1.0; 
 
  if ( D_n == 0.0 )
    {
      return( -1.0 );
    }
  
  if ( D_n < 0.0 )
    {
      return( -1.0 ); 
    }
  
  if ( f < 0.0 )
    {
      /* This can happen for certain  quantum defects   */
      /* in the lower n=1:l=0 state. In which case you  */
      /* probably should be using some other alogrithm  */
      /* or theory to calculate the dipole moment.      */
      return( -1.0 );
    }
  d1 = ( 1.0 / iz );
  
  d2 = (n_c * n_c)/(2.0 * D_n);
 
  d3 = (1.0 - ((D_l * lg)/n_c));
     
  d4 = AngerJ( v1, z );
     
  d34 = (d3*d4);
  
  d5 = (1.0 + ((D_l * lg)/n_c));
      
  d6 = AngerJ( v2 , z );
  
  d56 = (d5*d6);
     
  d6_1 = PI * D_n;
      
  d7 = TwobyPi * sin( d6_1 ) * (1.0 - e);
      
  d8 = d1 * d2 * ( (d34) - (d56) + d7 );
     
  d9 = d8 * d8;
  
  assert( D_n  > 0.0 );
  assert( l  >= 0  );
  assert( lp >= 0 );
  assert( (l == lp + 1) || ( l == lp - 1) );
  assert( n_c != 0.0 );
  assert( f >= 0.0 );  
  assert( d9  > 0.0 );
  
  return( d9 );
}

#endif
/*lint +e790 integral to float */

/********************************************************************************/
/********************************************************************************/
/*      Here we calculate the integrand                                         */
/*      (as a function of K, so                                                 */
/*      we need a dK^2 -> 2K dK )                                               */
/*      for  equation 3.14 of reference                                         */
/*                                                                              */
/*      M. Brocklehurst Mon. Note. R. astr. Soc. (1971) 153, 471-490            */
/*                                                                              */
/*      namely:                                                                 */
/*                                                                              */
/*      max(l,l')  (1 + n^2 K^2)^2 Theta(n,l; K, l') exp( -K^2 y ) d(K^2)       */
/*                                                                              */
/*      Note: the "y" is included in the code that called                       */
/*      this function and we include here the n^2 from eq 3.13.                 */
/********************************************************************************/
/********************************************************************************/
  
     
static mx normalize_mx( mx target )
{
  while ( fabs(target.m) > 1.0e+25 )
    { 
      target.m /= 1.0e+25;
      target.x += 25;
    }
  while ( fabs(target.m) < 1.0e-25 )
    {
      target.m *= 1.0e+25;
      target.x -= 25;
    }

  return( target );
}



static mx add_mx( mx a, mx b )
{
  mx result = {0.0,0};
     
  if (  (a.m) != ZerO )
    {
      result.x = a.x;
      result.m = a.m * (OnE + (b.m/a.m) * power( TeN , (b.x - a.x) ) );
      result = normalize_mx( result );
    }
  else
    {
      result = normalize_mx( b );   
    }
      
  return( result );  
}

  
 
static mx sub_mx( mx a, mx b )
{
  mx result = {0.0,0};
  b.m = -b.m;
 
  result = add_mx( a, b );
     
  result = normalize_mx( result );
     
  return( result );  
}
      
     
  
static mx mxify( double a )
{
  mx result_mx = {0.0,0};
      
  result_mx.x = 0;   
 
  result_mx.m = a;
  
  result_mx = normalize_mx( result_mx );

  return( result_mx );
}
  
 
  
static double unmxify( mx a_mx )
{
  double d = a_mx.m * power( TeN, a_mx.x );
  
  return( d );
}     
 
static mx mxify_log10( double log10_a )
{     
  mx
    result_mx={0.0,0};
  
  
  while ( log10_a > 25.0 )
    {
      log10_a -= 25.0;
      result_mx.x += 25;
    }
 
  while ( log10_a < -25.0 )
    {
      log10_a += 25.0;
      result_mx.x -= 25;
    }
  
  result_mx.m  = pow(TeN, log10_a);
     
  return( result_mx );
}
      
static mx mult_mx( mx a, mx b )
{
  mx result = {0.0,0};
  
  result.m = (a.m * b.m); 
  result.x = (a.x + b.x);
      
  result = normalize_mx( result );
     
 
  return( result );
}
      
      
static double   log10_prodxx( long int lp, double Ksqrd )
{ 
  long int s=0;
  double
    /* OnE in non_log version */
    partsum = ZerO;
  /*****************************ONE******************************/
  /**************************************************************/
      
  /**************************/ 
  /*    s=l'                */
  /*   -----                */
  /*    | |  (1 + s^2 K^2)  */
  /*    | |                 */
  /*    s=0                 */
  /**************************/
  
  /**********************************************************************/
  /*        |  s=l'                |     s=l'                           */
  /*        | -----                |     ---                            */
  /*  log10 |  | |  (1 + s^2 K^2)  | =    >    log10((1 + s^2 K^2))     */
  /*        |  | |                 |     ---                            */
  /*        |  s=0                 |     s=0                            */
  /**********************************************************************/
  
  
  for ( s = 0; s <= lp; s++ )
    {
      double s2 = (double)(s*s);                           /* s^2 */
  
      /**************************/
      /*    s=l'                */
      /*   -----                */
      /*    | |  (1 + s^2 K^2)  */
      /*    | |                 */
      /*    s=0                 */
      /**************************/
      partsum += log10( ( OnE + ( s2  * Ksqrd ) ) );
  
      assert( partsum >= ZerO );
    }
  assert( partsum >= ZerO );
  
  return( partsum );
}
/*lint +e790 integral to float */


