/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*HydroLevel calls HydroLevelPop or HydroT2Low to solve for ionization balance 
 * and level populations of model hydrogen atom */
#include "cddefines.h"
#include "taulines.h"
#include "iso.h"
#include "dense.h"
#include "secondaries.h"
#include "trace.h"
#include "elementnames.h"
#include "phycon.h"
#include "dynamics.h"
#include "opacity.h"
#include "ionrec.h"
#include "ionrange.h"
#include "negcon.h"
#include "hydrogenic.h"
/*lint -e661 Possible access of out-of-bounds pointer*/
/*lint -e662 Possible access of out-of-bounds pointer*/

void HydroLevel(long int nelem)
{
	char chType[6];
	long int i, 
	  ipHi, 
	  ipLo, 
	  level;
	double bottom,
	  colfrc, 
	  phtfrc, 
	  Ratio_Ionization, 
	  secfrc,
	  sum_popn_ov_ion;

	/* this block of variables will be obtained and freed here */
	double *totcap;/* MALLOC out to [iso.numLevels[ipH_LIKE][nelem]+1]*/

#	ifdef DEBUG_FUN
	fputs( "<+>HydroLevel()\n", debug_fp );
#	endif

	/* check that we were called with valid charge */
	ASSERT( nelem >= 0);
	ASSERT( nelem < LIMELM );

	/* 
	 * following electron density has approximate correction for neutrals
	 * corr of hi*1.7e-4 accounts for col ion by HI; Drawin Zs Phys 225, 483.
	 * used EdenHCorr instead
	 * edhi = eden + hi * 1.7e-4
	 */

	/* now MALLOC some scratch space */
	totcap = (double *)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ) ;
	/* ============================================================================== */
	/* option to print some rates */
	if( (trace.lgTrace && trace.lgIsoTraceFull[ipH_LIKE]) && (nelem == trace.ipIsoTrace[ipH_LIKE]) )
	{
		fprintf( ioQQQ, 
			"       HydroLevel%3ld finds arrays, with optical depths defined? %1c induced 2ph=%12.3e\n", 
		  nelem, TorF(opac.lgTauOutOn), EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].pump );
		for( ipHi=ipH2s; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
		{
			fprintf( ioQQQ, "up:%2ld", ipHi );
			fprintf( ioQQQ, "lo" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ, "%9ld", ipLo );
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " A*esc" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul*
				  EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Pesc ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " A*ees" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul*
				  EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Pelec_esc ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " tauin" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipH_LIKE][nelem][ipHi][ipLo].TauIn ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " t tot" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e", EmisLines[ipH_LIKE][nelem][ipHi][ipLo].TauTot ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " Esc  " );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e", EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Pesc ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " Eesc " );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e", EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Pelec_esc ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " Dest " );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Pdest) );
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " A*dst" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul*
				  EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Pdest ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " StrkE" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  hydro.pestrk[ipLo][ipHi] ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " B(ul)" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e", EmisLines[ipH_LIKE][nelem][ipHi][ipLo].pump*
				  iso.stat[ipH_LIKE][nelem][ipLo]/iso.stat[ipH_LIKE][nelem][ipHi] ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " tcont" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e",  EmisLines[ipH_LIKE][nelem][ipHi][ipLo].TauCon ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "%3ld", ipHi );
			fprintf( ioQQQ, " C(ul)" );
			for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
			{
				fprintf( ioQQQ,PrintEfmt("%9.2e", 
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].ColUL*dense.eden ));
			}
			fprintf( ioQQQ, "\n" );

			if( ipHi == 2 )
			{
				fprintf( ioQQQ, "    FeIIo");
				fprintf( ioQQQ,PrintEfmt("%9.2e", 
					hydro.dstfe2lya* EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Aul ));
				fprintf( ioQQQ, "\n");
			}
		}

		fprintf( ioQQQ, "         " );
		for( i=1; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf( ioQQQ, "%9ld", i );
		}
		fprintf( ioQQQ, "\n" );
	}
	/* ============================================================================== */

	/* check how low temperature is; departure coef tend to infinity
	 * as TE goes to zero; matrix formalism won't work on a 32-bit machine
	 * if EXP(H NU/KT) > 1E38 for 13.6EV.
	 * the recombination coefficient is effect radiative (only) rec
	 * coef (optical depths in) */

	/* get simple estimate of ionization balance, first get denominator,
	 * which can be zero during search phase */
	bottom = dense.eden*(iso.RadRec_effec[ipH_LIKE][nelem] + 
	  ionrec.CotaRate[nelem]) ;

	if( bottom > 0. )
	{
		iso.xIonSimple[ipH_LIKE][nelem] = (iso.gamnc[ipH_LIKE][nelem][ipH1s] + 
		  Secondaries.csupra + iso.ColIoniz[ipH_LIKE][nelem][ipH1s]*phycon.EdenHCorr )/bottom;
	}
	else
	{
		iso.xIonSimple[ipH_LIKE][nelem] = 0.;
	}

	if( trace.lgHBug && trace.lgTrace )
	{
		/* identify how atom is ionized for full trace */
		if( iso.xIonSimple[ipH_LIKE][nelem] > 0. )
		{
			/* fraction of ionization due to photoionization */
			phtfrc = iso.gamnc[ipH_LIKE][nelem][ipH1s]/((dense.eden*(iso.RadRec_effec[ipH_LIKE][nelem] + 
			  ionrec.CotaRate[nelem]) )*
			  iso.xIonSimple[ipH_LIKE][nelem]);

			/* fraction of ionization due to collisional ionization */
			colfrc = (iso.ColIoniz[ipH_LIKE][nelem][ipH1s]*phycon.EdenHCorr )/
				((dense.eden*(iso.RadRec_effec[ipH_LIKE][nelem] + 
			  ionrec.CotaRate[0]) )*
			  iso.xIonSimple[ipH_LIKE][nelem]);

			/* fraction of ionization due to secondary ionization */
			secfrc = Secondaries.csupra/((dense.eden*(iso.RadRec_effec[ipH_LIKE][nelem] + 
			  ionrec.CotaRate[0]) )*
			  iso.xIonSimple[ipH_LIKE][nelem]);
		}
		else
		{
			phtfrc = 0.;
			colfrc = 0.;
			secfrc = 0.;
		}

		fprintf( ioQQQ, "     HydroLevel Z=%2ld called, simple II/I=",nelem);
		PrintE93( ioQQQ, iso.xIonSimple[ipH_LIKE][nelem]);
		fprintf( ioQQQ," PhotFrc:");
		PrintE93( ioQQQ,phtfrc);
		fprintf(ioQQQ," ColFrc:");
		PrintE93( ioQQQ,colfrc);
		fprintf( ioQQQ," SecFrc");
		PrintE93( ioQQQ, secfrc);
		fprintf( ioQQQ,"  Te:");
		PrintE93( ioQQQ,phycon.te);
		fprintf( ioQQQ," eden:");
		PrintE93( ioQQQ,dense.eden);
		fprintf( ioQQQ,"\n"); 
	}

	/* for dense models close to lte all ionizations can be from
	 * excited states, and simple and actual pops are very different.
	 * code used simple all the time, caused catastrophe for Kingdon nova model.
	 * now use actual pops if we are into model */
	/* HIonFrac saves ion to neutral, used in bidiag to set hydrogenic ratio */
	Ratio_Ionization = iso.xIonSimple[ipH_LIKE][nelem];

	/* end of prelimaries - start model atom */

	/* which case atom to solve??? */
	if( iso.xIonSimple[ipH_LIKE][nelem] < 1e-30 )
	{
		/* don't bother if no ionizing radiation */
		strcpy( chType, "zero " );
		if( trace.lgHBug && trace.lgTrace )
		{
			fprintf( ioQQQ, "     HydroLevel Z=%2ld simple II/I=%10.2e so not doing equilibrium.\n", 
			  nelem, iso.xIonSimple[ipH_LIKE][nelem] );
		}

		for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			iso.DepartCoef[ipH_LIKE][nelem][i] = 1.;
			iso.Pop2Ion[ipH_LIKE][nelem][i] = 0.;
			hydro.DestRate[nelem][i] = 0.;
		}

		ionrec.TotRecomRate[nelem][nelem] = 0.;
		iso.xIonSimple[ipH_LIKE][nelem] = 0.;
		iso.pop_ion_ov_neut[ipH_LIKE][nelem] = 0.;

	}
	/* >>chng 99 nov 23, very dense model close to lte had very low simple
	 * ionization fraction but moderate ionization since all pops in
	 * excited states - added test for density 
	 * second change - for all cases use this routine if ionization is
	 * very low indeed */
	/* >>chng 00 aug 26, added matrix force option,
	 * logic is to override default options, "none was set in zero */

	/* NB - this test is mostly bogus since chTypeAtom is "POPU" in zero */

	/* >>chng 02 mar 13, hydro.chTypeAtom had been set to POPU in zero, look for
	 * comment with this date.  This had the effect of killing this following test.
	 * This also meant that the code had been happily inverting these impossible
	 * matrices for quite some time.  This test changed to stringest one, in
	 * light of this */
	/* >>chng 02 jun 29 add element specific test on ionization limit - had just
	 * been any element < 1e-28, but he ii failed at 1e-21 */
	else if( (strcmp( hydro.chTypeAtom , "LOWT" )==0) ||
		     ( (nelem==ipHYDROGEN) && (iso.xIonSimple[ipH_LIKE][nelem] < 1e-28) ) ||
		     ( (nelem>ipHYDROGEN) && (iso.xIonSimple[ipH_LIKE][nelem] < 1e-18) ) )
	{
		strcpy( chType, "TLow " );
		/* this avoids departure coefficients, 
		 * by doing simple cascade, should always work, 
		 * but not very well at high densities */
		HydroT2Low(nelem,Ratio_Ionization);
	}
	/* >>chng 99 jun 08 Ratio_Ionization from 1e-7 to 1e-5 due to orion nebula crash using
	 * costar stellar atmosphere, in He+ which was barely present */

	 /* >>chng 00 aug 26, added matrix force option,
	 * logic is to override default options, "none was set in zero */
	/* >>chng 01 may 09, remove all logic - always dumps to HydroLevelPop if gotten this far */
	else
	{
		/* this is the branch that uses the matrix to get level populations */
		strcpy( chType, "Popul" );
		HydroLevelPop(nelem ,
			totcap/* MALLOC out to [iso.numLevels[ipH_LIKE][nelem]+1]*/);
	}
	/* >>chng 01 may 09, totally remove HydroLevelDep */
	/* end branch for main solve */

	/* all three cases end up down here */
	for( ipLo=ipH1s; ipLo < (iso.numLevels[ipH_LIKE][nelem] - 1); ipLo++ )
	{
		for( ipHi=ipLo + 1; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
		{
			/* population of lower level rel to ion */
			EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopLo = 
				iso.Pop2Ion[ipH_LIKE][nelem][ipLo];

			/* population of upper level rel to ion */
			EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopHi = 
				iso.Pop2Ion[ipH_LIKE][nelem][ipHi];

			/* population of lower level rel to ion, corrected for stim em */
			EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc = 
				(iso.Pop2Ion[ipH_LIKE][nelem][ipLo] - 
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopHi*
				iso.stat[ipH_LIKE][nelem][ipLo]/iso.stat[ipH_LIKE][nelem][ipHi] );
		}
	}
	/* >>chng 00 dec 18, do not let lines going to highest level mase */
	/* higest level can have artifical maser */
	/* >>chng 01 feb 24, this had been in the wrong place in hydrogenic.c, having not effect.
	 * moved here */
	for( ipLo=ipH1s; ipLo < iso.numLevels[ipH_LIKE][nelem]-1; ipLo++ )
	{
		/* population of lower level rel to ion, corrected for stim em */
		EmisLines[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem]-1][ipLo].PopOpc =  
			MAX2(SMALLFLOAT,EmisLines[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem]-1][ipLo].PopOpc);
	}
	/* >>chng 01 feb 24, 2s and 2p are degerneate, so make PopOpc reflect total population. 
	 * this had major effect on radiation pressure due to n=3 to 2p transition, since 
	 * pop of 2p alone is small due to rapid escape route, so radiation pressure was large */
	for( ipHi=ipH2p+1; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
	{
		/* sum of populations in 2s and 2p */
		double sum = EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc + 
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc;

		/* population of lower level rel to ion, corrected for stim em */
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc =  sum;
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc =  sum;
	}

	/* this will eventually become the ratio of ionized to neutral hydrogenic 
	 * species, create sum of level pops per ion first */
	/* >>chng 02 nov 06, have all levels included in TotIonizRate */
	/* >>chng 02 nov 03, add ct to this term */
	/* this is charge transfer ionization of this specieds by hydrogen and helium */
	level = 0;
	ionrec.TotIonizRate[nelem][nelem] = 0.;
		/*iso.Pop2Ion[ipH_LIKE][nelem][level] * (Secondaries.csupra +
		ChargTran.HeCharExcIon[nelem][nelem]*dense.xIonDense[ipHELIUM][1]+ 
		ChargTran.HCharExcIon[nelem][nelem]*dense.xIonDense[ipHYDROGEN][1]);*/
	sum_popn_ov_ion = 0.;
	for( level=ipH1s; level < iso.numLevels[ipH_LIKE][nelem]; level++ )
	{
		/* all ionization processes from each level */
		ionrec.TotIonizRate[nelem][nelem] += 
			iso.xLevel2Cont[ipH_LIKE][nelem][level]*iso.Pop2Ion[ipH_LIKE][nelem][level];
		/* sum of all ionization processes from this atom to ion 
		ionrec.TotIonizRate[nelem][nelem] += 
			iso.Pop2Ion[ipH_LIKE][nelem][level] * (
		iso.gamnc[ipH_LIKE][nelem][level] + 
		iso.ColIoniz[ipH_LIKE][nelem][level]*phycon.EdenHCorr);*/

		sum_popn_ov_ion += iso.Pop2Ion[ipH_LIKE][nelem][level];
	}
	level = 0;
	/* convert back to scaled from ground */
	ionrec.TotIonizRate[nelem][nelem] /= MAX2(SMALLFLOAT , sum_popn_ov_ion);

	if( sum_popn_ov_ion < 0. )
	{
		fprintf( ioQQQ, 
			" HydroLevel finds negative H-like ion fraction for nelem=%2ld %s using routine %s, val= %10.3e simple=%10.3e TE=%10.3e ZONE=%4ld\n", 
		  nelem, 
		  elementnames.chElementSym[nelem],
		  chType , 
		  sum_popn_ov_ion, 
		  iso.xIonSimple[ipH_LIKE][nelem], 
		  phycon.te, 
		  nzone );
		fprintf( ioQQQ, " level pop are:" );
		for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf( ioQQQ,PrintEfmt("%8.1e", iso.Pop2Ion[ipH_LIKE][nelem][i] ));
		}
		fprintf( ioQQQ, "\n" );
		negcon();
		ShowMe();
		puts( "[Stop in HydroLevel]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* get level populations, two cases, 
	 * first, may be zero since all cases drop down to here, 
	 * this branch of if is for trivial abundance, so zero out species*/
	else if( sum_popn_ov_ion >= 0. && sum_popn_ov_ion < 1e-30 )
	{
		iso.pop_ion_ov_neut[ipH_LIKE][nelem] = 0.;

		/* reset pointer to one lower stage of ionization so this not
		 * considered again, hydrogenic considered if IonHigh is nelem+2 */
		IonRange.IonHigh[nelem] = nelem;
		ionrec.TotIonizRate[nelem][nelem] = 0.;

		/* now zero this out */
		for( ipLo=ipH1s; ipLo < (iso.numLevels[ipH_LIKE][nelem] - 1); ipLo++ )
		{
			for( ipHi=ipLo + 1; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
			{
				/* population of lower level rel to ion */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopLo = 0.;

				/* population of upper level rel to ion */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopHi =  0.;

				/* population of lower level rel to ion, corrected for stim em */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc =  0.;

				/* local ots destruction, cm^-3 s^-1 */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].ots =  0.;
			}
		}
	}
	/* this is the main branch that applies when we have non-trivial abundance */
	else
	{
		/* level populations 
		 * this brance we have significant population as sum of levels per ion,
		 * store inverse, ion per atom, here.  For non-H species this is ratio
		 * of highest to next highest stage of ionization */
		iso.pop_ion_ov_neut[ipH_LIKE][nelem] = 1./sum_popn_ov_ion;

		/* >> chng 02ionrec.TotIonizRate[nelem][nelem] Sep 20 rjrw: TotRecomRate from total not ratio */
		/* >>chng 02 oct 01, use recomtot when dynamics enabled, but the ratio of
		 * ionization rate to neutral frac for other cases.  This needs to be sorted out. */
#		ifndef NDEBUG
		{
			double pop_ion_ov_neut = ionrec.TotIonizRate[nelem][nelem] / 
				MAX2(SMALLFLOAT,ionrec.TotRecomRate[nelem][nelem]);

			double ratio_error = fabs(iso.pop_ion_ov_neut[ipH_LIKE][nelem] - pop_ion_ov_neut ) /
				/* >>chng 03 mar 20,
				 * first term below changed from SMALLFLOAT to 1e-8, to prevent
				 * checks on very minor species, as per Robin Williams email.
				 * with old simple bidiagonal solver, each ionization pair had
				 * very high accuracy, not related to the result itself.  Now
				 * ionization is result of soln with multiple ionization stages
				 * being directly coupled, so this level of error exists 
				 * across solution */
				MAX2(1e-7,iso.pop_ion_ov_neut[ipH_LIKE][nelem]);

			ASSERT( dynamics.lgAdvection ||
				/* following needed to pass assert on alpha */
				ionrec.TotIonizRate[nelem][nelem] < SMALLFLOAT || 
				ratio_error < 0.001);
		}
#		endif
	}

	/* this is main trace h-like printout */
	if( (trace.lgIsoTraceFull[ipH_LIKE] && trace.lgTrace) && (nelem == trace.ipIsoTrace[ipH_LIKE]) )
	{
		fprintf( ioQQQ, "       HLEV HGAMNC" );
		PrintE93( ioQQQ, iso.gamnc[ipH_LIKE][nelem][ipH1s] );
		for( i=ipH2s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", iso.gamnc[ipH_LIKE][nelem][i] ));
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "       HLEV TOTCAP" );
		for( i=1; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", totcap[i] ));
		}
		fprintf( ioQQQ,PrintEfmt("%10.2e", ionrec.TotRecomRate[ipH_LIKE][nelem]/dense.eden ) );
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "       HLEV IND Rc" );
		for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", iso.RecomInducRate[ipH_LIKE][nelem][i]*iso.PopLTE[ipH_LIKE][nelem][i] ));
		}
		fprintf( ioQQQ, "\n" );

		/* incuded recombination rate coefficients */
		fprintf( ioQQQ, "       IND Rc LTE " );
		for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e",
				iso.gamnc[ipH_LIKE][nelem][i]*iso.PopLTE[ipH_LIKE][nelem][i] ));
		}
		fprintf( ioQQQ, "\n" );

		/* LTE level populations */
		fprintf( ioQQQ, "       HLEV   HLTE" );
		for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", iso.PopLTE[ipH_LIKE][nelem][i] ));
		}
		fprintf( ioQQQ, "\n" );

		/* fraction of total ionization due to collisions from given level */
		fprintf( ioQQQ, "       HLEVfr cion" );
		for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", 
				iso.ColIoniz[ipH_LIKE][nelem][i]*
			  iso.Pop2Ion[ipH_LIKE][nelem][i]*phycon.EdenHCorr/MAX2(1e-30,hydro.DestRate[nelem][i]) ) );
		}
		fprintf( ioQQQ, "\n" );

		/* fraction of total ionization due to photoionization from given level */
		if( ionrec.TotRecomRate[nelem][nelem]> 0. )
		{
			fprintf( ioQQQ, "       HLEVfrPhIon" );
			for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
			{
				fprintf(ioQQQ,PrintEfmt("%9.2e", 
					iso.gamnc[ipH_LIKE][nelem][i]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][i]/MAX2(1e-30,hydro.DestRate[nelem][i]) ) );
			}
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, "       HLEV     HN" );
		for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", iso.Pop2Ion[ipH_LIKE][nelem][i] ));
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "       HLEV   b(n)" );
		for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf(ioQQQ,PrintEfmt("%9.2e", iso.DepartCoef[ipH_LIKE][nelem][i] ));
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, "       HLEV   X12tot");
		fprintf(ioQQQ,PrintEfmt("%9.2e", Secondaries.x12tot ));
		fprintf( ioQQQ," Grn dest:");
		fprintf(ioQQQ,PrintEfmt("%9.2e",
		  ionrec.TotIonizRate[nelem][nelem] ));
		fprintf(ioQQQ, "\n"); 
	}

	/* find totaql ionization rate while checking for non-positive level populations */
	/* >>chng 02 apr 17, only check on negative pops, alpha turned up some zero pops
	 * in extreme low ionization conditions */
	if( iso.pop_ion_ov_neut[ipH_LIKE][nelem] > 0. )
	{
		for( level=ipH1s; level < iso.numLevels[ipH_LIKE][nelem]; level++ )
		{

			/* >>chng 02 apr 17, only check on negative pops, alpha turned up some zero pops
			* in extreme low ionization conditions */
			/*if( iso.Pop2Ion[ipH_LIKE][nelem][level] <= 0. )*/
			if( iso.Pop2Ion[ipH_LIKE][nelem][level] < 0. )
			{
				fprintf( ioQQQ, 
					" HydroLevel finds negative hydrogen level population for %s nelem=%ld level %ld value=%10.3e simple=%10.3e TE=%10.3e ZONE=%4ld\n", 
				  elementnames.chElementName[nelem], 
				  nelem, level, 
				  iso.Pop2Ion[ipH_LIKE][nelem][level], 
				  iso.xIonSimple[ipH_LIKE][nelem], 
				  phycon.te, nzone );
				fprintf( ioQQQ, " level pop are:" );
				for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
				{
					fprintf( ioQQQ,PrintEfmt("%8.1e", iso.Pop2Ion[ipH_LIKE][nelem][i] ));
				}
				fprintf( ioQQQ, "\n" );
				negcon();
				ShowMe();
				puts( "[Stop in HydroLevel]" );
				cdEXIT(EXIT_FAILURE);
			}
		}
	}
	else if( iso.pop_ion_ov_neut[ipH_LIKE][nelem] < 0.)
	{
		ionrec.TotIonizRate[nelem][nelem] = 0.;
		/* this is case where we expected ionization, but found none, or negative */
		fprintf( ioQQQ, 
			" HydroLevel finds negative hydrogen ion frac for nelem=%ld (%s) value=%.3e simple= %.3e TE=%.3e ZONE=%ld\n", 
		  nelem, 
		  elementnames.chElementSym[nelem],
		  iso.Pop2Ion[ipH_LIKE][nelem][level], 
		  iso.xIonSimple[ipH_LIKE][nelem], 
		  phycon.te, nzone );
		fprintf( ioQQQ, " level pop are:" );
		for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
		{
			fprintf( ioQQQ,PrintEfmt("%8.1e", iso.Pop2Ion[ipH_LIKE][nelem][i] ));
		}
		fprintf( ioQQQ, "\n" );
		negcon();
		ShowMe();
		puts( "[Stop in HydroLevel]" );
		cdEXIT(EXIT_FAILURE);
	}
	/* the case where no ionization is present was covered above, there is no
	 * final else for these tests */

	if( trace.lgTrace )
	{
		/* iso.RecomTotal[nelem] is gross rec coef, computed here while filling in matrix
		 * elements, all physical processes included. 
		 * RadRec_effec is total effective radiative only */
		fprintf( ioQQQ, "       HydroLevel%3ld retrn %s te=",
			nelem, 
			chType );
		PrintE93( ioQQQ,phycon.te);
		fprintf( ioQQQ," HII/HI=");
		PrintE93( ioQQQ,iso.pop_ion_ov_neut[ipH_LIKE][nelem]);

		fprintf( ioQQQ," simple=");
		PrintE93( ioQQQ,iso.xIonSimple[ipH_LIKE][nelem]);

		fprintf( ioQQQ," b1=");
		PrintE82( ioQQQ,iso.DepartCoef[ipH_LIKE][nelem][ipH1s]);

		fprintf( ioQQQ," ion rate=");
		PrintE82( ioQQQ,iso.gamnc[ipH_LIKE][nelem][ipH1s] + Secondaries.csupra);

		fprintf( ioQQQ," TotRec");
		PrintE82( ioQQQ,ionrec.TotRecomRate[nelem][nelem]/dense.eden);

		fprintf( ioQQQ," RadRec");
		PrintE82( ioQQQ,iso.RadRec_effec[ipH_LIKE][nelem]);
		fprintf( ioQQQ, "\n");
	}

	/* now free up the arrays */
	free( totcap );

#	ifdef DEBUG_FUN
	fputs( " <->HydroLevel()\n", debug_fp );
#	endif
	return;
}
/*lint +e661 Possible access of out-of-bounds pointer*/
/*lint +e662 Possible access of out-of-bounds pointer*/

