/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*HydroLevelPop solve for ionization balance level populations of model hydrogen atom */
#include "cddefines.h"
#include "taulines.h"
#include "iso.h"
#include "secondaries.h"
#include "elementnames.h"
#include "chargtran.h"
#include "dense.h"
#include "trace.h"
#include "ionrec.h"
#include "phycon.h"
#include "typmatrx.h"
#include "linpack.h"
#include "hydrogenic.h"
#include "dynamics.h"
/*lint -e662 Possible access of out-of-bounds pointer*/

void HydroLevelPop(long int nelem ,
	double *totcap/* malloc out to [iso.numLevels[ipH_LIKE][nelem]+1]*/)
{
	long int n, 
	  ipHi, 
	  ipLo, 
	  i,
	  j, 
	  level, 
	  nerror1,
	  nerror2,
	  level_error;

	double BigError ,
		Recom3Body ,
	  HRateDestGnd[8];

	long int *ipiv ; /* malloc out to [iso.numLevels[ipH_LIKE][nelem]+1] */
	double 
		*creation,
		*Save_creation,
		**SaveZ/*[iso.numLevels[ipH_LIKE][nelem]+2][iso.numLevels[ipH_LIKE]+2]*/, 
		*work/*[iso.numLevels[ipH_LIKE][nelem]+2]*/, 
		**z/*[iso.numLevels[ipH_LIKE][nelem]+2][iso.numLevels[ipH_LIKE]+2]*/;
	double *error;/*[iso.numLevels[ipH_LIKE][nelem]+2]*/

#	ifdef DEBUG_FUN
	fputs( "<+>HydroLevelPop()\n", debug_fp );
#	endif

	/* check that we were called with valid charge */
	ASSERT( nelem >= 0);
	ASSERT( nelem < LIMELM );
	if( (ipiv = (long int *)MALLOC(sizeof(long int)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ) ) == NULL )
		BadMalloc();
	if( (creation = (double *)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ) ) == NULL )
		BadMalloc();
	if( (Save_creation = (double *)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ) ) == NULL )
		BadMalloc();
	if( (work = (double *)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ) ) == NULL )
		BadMalloc();
	if( (error = (double *)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ) ) == NULL )
		BadMalloc();

	/* now do the 2D arrays */
	if( (SaveZ = (double **)MALLOC(sizeof(double *)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ) ) ==NULL )
		BadMalloc();

	if(  (z = (double **)MALLOC(sizeof(double *)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ))==NULL  )
		BadMalloc();

	/* now do the second dimension */
	for( i=0; i<(iso.numLevels[ipH_LIKE][nelem]); ++i )
	{
		if( (SaveZ[i] = (double *)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ) ) ==NULL )
			BadMalloc();

		if( (z[i] = (double *)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ) ) ==NULL )
			BadMalloc();
	}


	/* 
	 * following electron density has approximate correction for neutrals
	 * corr of hi*1.7e-4 accounts for col ion by HI; Drawin Zs Phys 225, 483.
	 * used EdenHCorr instead
	 * edhi = eden + hi * 1.7e-4
	 */

	ionrec.TotRecomRate[nelem][nelem] = 0.;
	ionrec.RadRecomRateCoef[nelem][nelem] = 0.;
	Recom3Body = 0.;
	for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
	{

		/* total rate level n is populated from the continuum, units at this
		 * stage are cm3 s-1 */
		/* PopLTE(n,nelem) is only LTE pop when mult by Ne Nh */
		totcap[n] = 
			/* radiative recombination */
			(double)(iso.RadRecomb[ipH_LIKE][nelem][n][ipRecRad]*
			iso.RadRecomb[ipH_LIKE][nelem][n][ipRecNetEsc]) + 

			/* induced recombination */
			iso.RecomInducRate[ipH_LIKE][nelem][n]*iso.PopLTE[ipH_LIKE][nelem][n] + 

			/* collisional or three body recombination */
			iso.ColIoniz[ipH_LIKE][nelem][n]*phycon.EdenHCorr*iso.PopLTE[ipH_LIKE][nelem][n];
			{
				/*@-redef@*/
				enum {DEBUG_LOC=FALSE};
				/*@+redef@*/
				if( DEBUG_LOC )
				{
				fprintf(ioQQQ, "bugggg\t%li\t%.4e\t%.3e\t%.3e\n",
				nzone,phycon.te,phycon.EdenHCorr,iso.PopLTE[ipH_LIKE][nelem][n] );
				}
			}

			/* this is three-body recombination rate coef by itself - 
			 * need factor of eden to become rate */
			Recom3Body += iso.ColIoniz[ipH_LIKE][nelem][n]*phycon.EdenHCorr*iso.PopLTE[ipH_LIKE][nelem][n];


		/* this makes units of s-1 */
		creation[n] = totcap[n] * dense.eden;

		/* units of ionrec.TotRecomRate are s-1 */
		ionrec.TotRecomRate[nelem][nelem] += totcap[n] * dense.eden;

		/* just the radiative recombination rate coef */
		ionrec.RadRecomRateCoef[nelem][nelem] += iso.RadRecomb[ipH_LIKE][nelem][n][ipRecRad]*
			iso.RadRecomb[ipH_LIKE][nelem][n][ipRecNetEsc];
	}
	/* fraction of total recombs due to three body - when most are due to this
	 * small changes in temperature can produce large changes in rec coef,
	 * and so in ionization */
	iso.RecomCollisFrac[ipH_LIKE][nelem] = (float)(Recom3Body* dense.eden / ionrec.TotRecomRate[nelem][nelem] );
	/*fprintf(ioQQQ,"threee\t%li\t%.3e\n", nzone , iso.RecomCollisFrac[ipH_LIKE][nelem] );*/

	creation[ipH1s] +=
		/* >>chng 02 may 05, should be just ground state, make it so */
		/*ChargTran.HeCharExcRec[nelem][ion]*dense.xIonDense[ipHELIUM][0] + */
		ChargTran.HeCharExcRec[nelem][nelem]*iso.Pop2Ion[ipHE_LIKE][ipHELIUM][ipHe1s1S]*dense.xIonDense[ipHELIUM][1] + 
		/*ChargTran.HCharExcRec[nelem][ion]*dense.xIonDense[ipHYDROGEN][0];*/
		/* >>chng 01 06 01 should be ground state only, change to do so */
		ChargTran.HCharExcRec[nelem][nelem]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s]*dense.xIonDense[ipHYDROGEN][1];

	/* charge transfer recombination of this species by ionizing hydrogen and helium */
	ionrec.TotRecomRate[nelem][nelem] += 
		/* >>chng 02 may 05, should be just ground state, make it so */
		/*ChargTran.HeCharExcRec[nelem][ion]*dense.xIonDense[ipHELIUM][0] + */
		ChargTran.HeCharExcRec[nelem][nelem]*iso.Pop2Ion[ipHE_LIKE][ipHELIUM][ipHe1s1S]*dense.xIonDense[ipHELIUM][1] + 
		/*ChargTran.HCharExcRec[nelem][ion]*dense.xIonDense[ipHYDROGEN][0];*/
		/* >>chng 01 06 01 should be ground state only, change to do so */
		ChargTran.HCharExcRec[nelem][nelem]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s]*dense.xIonDense[ipHYDROGEN][1];

	/* master balance equation */
	for( level=ipH1s; level < iso.numLevels[ipH_LIKE][nelem]; level++ )
	{
		/* all process depopulating level */
		z[level][level] = (iso.gamnc[ipH_LIKE][nelem][level] + 
		  iso.ColIoniz[ipH_LIKE][nelem][level]* phycon.EdenHCorr + 
		  Secondaries.csupra);

		iso.xLevel2Cont[ipH_LIKE][nelem][level] = (float)z[level][level];

		/* all processes populating level from below */
		for( ipLo=ipH1s; ipLo < level; ipLo++ )
		{
			double CollRate = EmisLines[ipH_LIKE][nelem][level][ipLo].ColUL* phycon.EdenHCorr;

			z[ipLo][level] = 
				-CollRate * 
			  (double)iso.stat[ipH_LIKE][nelem][level]/(double)iso.stat[ipH_LIKE][nelem][ipLo]*
			  iso.Boltzmann[ipH_LIKE][nelem][level][ipLo]  -
			  EmisLines[ipH_LIKE][nelem][level][ipLo].pump ;

			/* pumping out of here to lower level */
			z[level][level] += EmisLines[ipH_LIKE][nelem][level][ipLo].pump *
			  iso.stat[ipH_LIKE][nelem][ipLo]/iso.stat[ipH_LIKE][nelem][level];

			/* collisions out of here to lower level */
			z[level][level] += CollRate;

			/* radiative decays out of here to lower level */
			z[level][level] += 
				EmisLines[ipH_LIKE][nelem][level][ipLo].Aul*
			  (EmisLines[ipH_LIKE][nelem][level][ipLo].Pesc + 
			  EmisLines[ipH_LIKE][nelem][level][ipLo].Pelec_esc +
			  EmisLines[ipH_LIKE][nelem][level][ipLo].Pdest);
		}

		/* all processes populating level from above */
		for( ipHi=level + 1; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
		{
			double RadDecay , CollRate ;
			RadDecay =
				EmisLines[ipH_LIKE][nelem][ipHi][level].Aul*
			  (EmisLines[ipH_LIKE][nelem][ipHi][level].Pesc + 
			  EmisLines[ipH_LIKE][nelem][ipHi][level].Pelec_esc +
			  EmisLines[ipH_LIKE][nelem][ipHi][level].Pdest);

			/* >>chng 02 feb 06, define this and use in each direction below -
			 * attempt at solving roundoff problems on alphas */
			CollRate = EmisLines[ipH_LIKE][nelem][ipHi][level].ColUL *phycon.EdenHCorr;

			z[ipHi][level] = 
				-(RadDecay + 
			  EmisLines[ipH_LIKE][nelem][ipHi][level].pump*
			  iso.stat[ipH_LIKE][nelem][level]/iso.stat[ipH_LIKE][nelem][ipHi] + 
			  CollRate );

			/* pumping out of here to upper level */
			z[level][level] += EmisLines[ipH_LIKE][nelem][ipHi][level].pump ;

			/* collisions out of here to upper level */
			z[level][level] += (double)iso.stat[ipH_LIKE][nelem][ipHi] / (double)iso.stat[ipH_LIKE][nelem][level] *
				iso.Boltzmann[ipH_LIKE][nelem][ipHi][level]*CollRate;
		}
	}

	/* >>chng 02 jul 22, add induced 2-nu to level populations */
	/* induced two photon emission - special because upward and downward are
	 * not related by ratio of statistical weights */
	/* iso.lgInd2nu_On is controlled with SET IND2 ON/OFF command */
	fixit();/* this does not look right - check logic */
	z[ipH2s][ipH1s] -= iso.TwoNu_induc_dn[ipH_LIKE][nelem]*iso.lgInd2nu_On ;
	z[ipH1s][ipH2s] -= iso.TwoNu_induc_up[ipH_LIKE][nelem]*iso.lgInd2nu_On ;

	/* rates out of 1s, and out of 2s */
	z[ipH1s][ipH1s] += iso.TwoNu_induc_up[ipH_LIKE][nelem]*iso.lgInd2nu_On ;
	z[ipH2s][ipH2s] += iso.TwoNu_induc_dn[ipH_LIKE][nelem]*iso.lgInd2nu_On ;

	if( Secondaries.Hx12[0][1] > 0. )
	{
		/* now add on supra thermal excitation */
		for( level=ipH2s; level < iso.numLevels[ipH_LIKE][nelem]; level++ )
		{
			double RateUp , RateDown;

			RateUp = Secondaries.Hx12[MIN2(nelem,1)][level];

			RateDown = RateUp * (double)iso.stat[ipH_LIKE][nelem][ipH1s] /
				(double)iso.stat[ipH_LIKE][nelem][level];

			/* stuff in min after Hx12 evaluates to 0 for atom, or 1 for ion */
			/* total rate out of lower level */
			z[ipH1s][ipH1s] += RateUp;

			/* rate from the upper level to ground */
			z[level][ipH1s] -= RateDown ;

			/* rate from ground to upper level */
			z[ipH1s][level] -= RateUp ;

			z[level][level] += RateDown;  
		}
	}

	/* charge transfer, hydrogen onto everything else */
	if( nelem == ipHYDROGEN )
	{
		/* >>chng 01 apr 28, add this in */
		int ipCollide , ion;

		/* add on charge transfer ionization of hydrogen,
		 * recombination for other element is ionization of hydrogen */
		ChargTran.HCharExcIonTotal = 0.;
		for( ipCollide=1; ipCollide<LIMELM; ++ipCollide)
		{
			double hold_one = 0.;
			/* this is ion on the abundances scale, 1 is atom, so goes up to ipCollide+1,
			 * for helium ipCollide=1, ion must go up to 3 */
			/* check that array bounds not exceeded */
			ASSERT( (ipCollide+2-2) < (LIMELM+1) );
			for( ion=2; ion<=ipCollide+2; ++ion )
			{
				/* we intentionally skip CT with O+ since this would overwhelm every
				 * other process, but does not actually amount to much since is in
				 * balance with opposite CT reaction */
				if( ipCollide == ipOXYGEN && ion == 2 ) continue;

				/* charge transfer ionization of H, recombination for other species */
				hold_one = ChargTran.HCharExcRec[ipCollide][ion-2]*dense.xIonDense[ipCollide][ion-1];
				ChargTran.HCharExcIonTotal += hold_one;
			}
		}

		z[ipH1s][ipH1s] += ChargTran.HCharExcIonTotal;
		iso.xLevel2Cont[ipH_LIKE][nelem][ipH1s] += (float)ChargTran.HCharExcIonTotal;

		/* >>chng 01 may 07,  add this in */
		/* charge transfer recombination of hydrogen,
		 * which is ionization of the heavy element */
		ChargTran.HCharExcRecTotal = 0.;
		for( ipCollide=1; ipCollide<LIMELM; ++ipCollide)
		{
			/* this is ion on the abundances scale, 1 is atom, so goes up to ipCollide+1,
			 * for helium ipCollide=1, ion must go up to 3 */
			/* check that array bounds not exceeded */
			ASSERT( (ipCollide) < (LIMELM+1) );
			for( ion=1; ion<=ipCollide+1; ++ion )
			{
				/* skip Oo => O+ */
				if( ipCollide == ipOXYGEN && ion == 1 ) continue;
				/* charge transfer ionization of H, recombination for other species */
				ChargTran.HCharExcRecTotal += 
					ChargTran.HCharExcIon[ipCollide][ion-1]*dense.xIonDense[ipCollide][ion-1];
			}
		}
		/*fprintf(ioQQQ," %.2e %.2e %.2e \n", z[iso.numLevels[ipH_LIKE][nelem]+1][ipH1s] , ChargTran.HCharExcRecTotal, 
			z[iso.numLevels[ipH_LIKE][nelem]+1][ipH2p]);*/
		creation[ipH1s] += ChargTran.HCharExcRecTotal;
		ionrec.TotRecomRate[nelem][nelem] += ChargTran.HCharExcRecTotal;

#		if 0
		fprintf(ioQQQ," h net ion %.2e %.2e %.2e\n",
			ChargTran.HCharExcIonTotal,ChargTran.HCharExcRecTotal,
			ChargTran.HCharExcIonTotal-ChargTran.HCharExcRecTotal);
#		endif

		totcap[ipH1s] = creation[ipH1s]/dense.eden;

		{
			/*@-redef@*/
			enum {DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC )
			{
				long int maxnelem=-1, maxion=-1;
				double ctmax = 0.;
				for( ipCollide=1; ipCollide<LIMELM; ++ipCollide)
				{
					/* this is ion on the abundances scale, 1 is atom, so goes up to ipCollide+1,
					 * for helium ipCollide=1, ion must go up to 3 */
					for( ion=1; ion<=ipCollide+1; ++ion )
					{
						/* charge transfer ionization of H, recombination for other species */
						/*z[iso.numLevels[ipH_LIKE][nelem]+1][ipH1s] += ChargTran.HCharExcIon[ipCollide][ion-1]*dense.xIonDense[ipCollide][ion-1];*/
						if(	ChargTran.HCharExcIon[ipCollide][ion-1]*dense.xIonDense[ipCollide][ion-1] > ctmax )
						{
							ctmax = ChargTran.HCharExcIon[ipCollide][ion-1]*dense.xIonDense[ipCollide][ion-1];
							maxnelem = ipCollide;
							maxion = ion;
						}
					}
				}
				fprintf(ioQQQ," max H ct recom rate%.2e frac tot %.2e ipCollide %li ion %li\n", 
					ctmax,
					ctmax/MAX2(SMALLFLOAT,ChargTran.HCharExcRecTotal),
					maxnelem,
					maxion);
			}
		}
	}

#if FOO
	{ 
		float mytot,err;
		int myeq;

		for (myeq=ipH1s; myeq<iso.numLevels[ipH_LIKE][nelem]; myeq++ ) {
			mytot = 0.;
			for (level=ipH1s; level<iso.numLevels[ipH_LIKE][nelem]; level++ ) {
				mytot += z[myeq][level];
			}
			err = fabs((mytot-iso.xLevel2Cont[ipH_LIKE][nelem][myeq])/z[myeq][myeq]);
			if (err > 1e-6)
				printf("BalChk: %d %d e %g\n",nelem,myeq,err);
		}
	}
#endif

	/* >>chng 02 Sep 06 rjrw -- all elements have these terms */
	/*>>>chng 02 oct 01, only include if lgAdvection is set */
	if( dynamics.lgAdvection )
	{
		/* add in advection - these terms normally zero */
		/* assume for present that all advection is into ground state */
		creation[ipH1s] += 
			dynamics.Source[nelem][nelem]/MAX2(SMALLFLOAT,dense.xIonDense[nelem][nelem+1])*
			dynamics.lgISO[ipH_LIKE];
		/* >>chng 02 Sep 06 rjrw -- advective term not recombination */
		/* can sink from all components (must do, for conservation) */
		for( ipLo=ipH1s; ipLo<iso.numLevels[ipH_LIKE][nelem]; ++ipLo )
		{
			z[ipLo][ipLo] += dynamics.Rate*dynamics.lgISO[ipH_LIKE];
		}
	}

	/* =================================================================== 
	 *
	 * at this point all matrix elements have been established 
	 *
	 * ==================================================================== */

	/* save total ground state destruction rate for possible later use */
	for( ipLo=ipH1s; ipLo<iso.numLevels[ipH_LIKE][nelem]; ++ipLo )
	{
		hydro.DestRate[nelem][ipLo] = z[ipLo][ipLo];
	}

	if( (trace.lgTrace && trace.lgIsoTraceFull[ipH_LIKE]) && (nelem == trace.ipIsoTrace[ipH_LIKE]) )
	{
		if( nelem == 0 )
		{
			/* remember ground state destruction rate */
			HRateDestGnd[0] = z[ipH1s][ipH1s];
			HRateDestGnd[1] = iso.gamnc[ipH_LIKE][nelem][ipH1s]/
			  z[ipH1s][ipH1s];
			HRateDestGnd[2] = iso.ColIoniz[ipH_LIKE][nelem][ipH1s]*
			  phycon.EdenHCorr/z[ipH1s][ipH1s];
			HRateDestGnd[3] = Secondaries.csupra/z[ipH1s][ipH1s];
			HRateDestGnd[4] = Secondaries.Hx12[MIN2(nelem,1)][ipH2p]*
			  9./z[ipH1s][ipH1s];

			fprintf(ioQQQ," grnd dest fracs:");
			for( j=ipH1s; j < 5; j++ )
			{
				fprintf( ioQQQ," ");
				fprintf( ioQQQ,PrintEfmt("%8.1e", HRateDestGnd[j] ));
			}
			fprintf(ioQQQ," hcoldc %e\n", phycon.EdenHCorr );
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, "  pop level     others => (HydroLevelPop)\n" );
		for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
		{
			fprintf( ioQQQ, "       HII%2ld", n );
			for( j=ipH1s; j < iso.numLevels[ipH_LIKE][nelem]; j++ )
			{
				fprintf( ioQQQ," ");
				/*fprintf( ioQQQ,PrintEfmt("%8.1e", z[j][n] ) );*/
				fprintf( ioQQQ,"%.9e\t", z[j][n] );
			}
			fprintf( ioQQQ, "\n" );
		}
	}

	/* save matrix */
	for( j=ipH1s; j < iso.numLevels[ipH_LIKE][nelem]; j++ )
	{
		for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
		{
			SaveZ[n][j] = z[n][j];
		}
		Save_creation[j] = creation[j];
	}

	/* this is the default */
	if( strcmp(TypMatrx.chMatrix,"linpack") == 0 )
	{
		/*double amat[LMHLVL+2][LMHLVL+2];*/
		double *amat;
	  /* iso.numLevels[ipH_LIKE][nelem]+1 is right dimension of matrix */
#		ifdef AMAT
#			undef AMAT
#		endif
#		define AMAT(I_,J_)	(*(amat+(I_)*iso.numLevels[ipH_LIKE][nelem]+(J_)))
		/* MALLOC space for the  1-d array */
		if( (amat=(double*)MALLOC( (sizeof(double)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]*(iso.numLevels[ipH_LIKE][nelem])) ))) == NULL )
			BadMalloc();

		/* this one may be more robust */
		for( j=ipH1s; j < iso.numLevels[ipH_LIKE][nelem]; j++ )
		{
			for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
			{
				/*amat[n][j] = z[n][j];*/
				AMAT(n,j) = z[n][j];
			}
		}

		/*DGETRF(iso.numLevels[ipH_LIKE][nelem],iso.numLevels[ipH_LIKE][nelem],
		  amat,(iso.numLevels[ipH_LIKE][nelem]),ipiv,&nerror1);

		DGETRS('N',iso.numLevels[ipH_LIKE][nelem],1,amat,(iso.numLevels[ipH_LIKE][nelem]),ipiv,
		  bvec,(iso.numLevels[ipH_LIKE][nelem]),&nerror2);*/

		DGETRF(iso.numLevels[ipH_LIKE][nelem],iso.numLevels[ipH_LIKE][nelem],
		  amat,iso.numLevels[ipH_LIKE][nelem],ipiv,&nerror1);

		DGETRS('N',iso.numLevels[ipH_LIKE][nelem],1,amat,iso.numLevels[ipH_LIKE][nelem],ipiv,
		  creation,iso.numLevels[ipH_LIKE][nelem],&nerror2);

		if( nerror1 != 0 || nerror2 != 0 )
		{
			fprintf( ioQQQ, " HydroLevelPop dgetrs error\n" );
			puts( "[Stop in HydroLevelPop]" );
			cdEXIT(EXIT_FAILURE);
		}

		free( amat);
	}
	else
	{
		fprintf( ioQQQ, " chMatrix type insane in HydroLevelPop, was%7.7s\n", 
		  TypMatrx.chMatrix );
		puts( "[Stop in HydroLevelPop]" );
		cdEXIT(EXIT_FAILURE);
	}
	/* end of branches for which matrix solution, now check if valid */

	/* check whether solution is valid */
	for( level=ipH1s; level < iso.numLevels[ipH_LIKE][nelem]; level++ )
	{
		double BigSoln= 0.;
		error[level] = 0.;
		for( n=ipH1s; n < iso.numLevels[ipH_LIKE][nelem]; n++ )
		{
			/* remember the largest value of the soln matrix to div by below */
			if( fabs(SaveZ[n][level] ) > BigSoln )
				BigSoln = fabs(SaveZ[n][level]);

			error[level] += SaveZ[n][level]*creation[n];
		}

		if( BigSoln > 0. )
		{
			error[level] = (error[level] - Save_creation[level])/ BigSoln;
		}
		else
		{
			error[level] = 0.;
		}
	}

	/* remember largest residual in matrix inversion */
	BigError = -1.;
	level_error = -1;
	for( level=ipH1s; level < iso.numLevels[ipH_LIKE][nelem]; level++ )
	{
		double abserror;
		abserror = fabs( error[level]) ;
		/* this will be the largest residual in the matrix inversion */
		if( abserror > BigError )
		{
			BigError = abserror ;
			level_error = level;
		}
	}

	/* matrix inversion should be nearly as good as the accuracy of a double,
	 * but demand that it is better than epsilon for a float */
	if( BigError > FLT_EPSILON ) 
	{
		fprintf(ioQQQ,
			"HydroLevel: warning zone %li - largest residual in hydrogenic %s nelem=%li matrix inversion is %g "
			"level was %li \n", 
			nzone,
			elementnames.chElementName[nelem],
			nelem , 
			BigError , 
			level_error);
		ShowMe();
		puts( "[Stop in HydroLevelPop]" );
		/* cdEXIT(EXIT_FAILURE); */
	}

#if 0
	if (nelem == 0) 
	{ 
		double tot = 0, rat;
		for( ipLo=ipH1s; ipLo < iso.numLevels[ipH_LIKE][nelem]; ipLo++ ) 
		{
			tot += creation[ipLo];
		}
		rat = (dense.xIonDense[nelem][0]/dense.xIonDense[nelem][1])-tot;
		printf("Ratio: %g %g\n",tot,(dense.xIonDense[nelem][0]/dense.xIonDense[nelem][1])); 
		/* creation[ipH1s] += rat; */
	}
#endif

	/* convert from departure coef in Z(I) to level pop rel to HII */

	/* put departure coefficients and level populations into master array */
	for( ipLo=ipH1s; ipLo < iso.numLevels[ipH_LIKE][nelem]; ipLo++ )
	{
		iso.Pop2Ion[ipH_LIKE][nelem][ipLo] = (float)creation[ipLo];
		if( iso.PopLTE[ipH_LIKE][nelem][ipLo] > 0. )
		{
			iso.DepartCoef[ipH_LIKE][nelem][ipLo] = 
				(iso.Pop2Ion[ipH_LIKE][nelem][ipLo]/
				(iso.PopLTE[ipH_LIKE][nelem][ipLo]* dense.eden) );
		}
		else
		{
			iso.DepartCoef[ipH_LIKE][nelem][ipLo] = 0.;
		}
	}

	free( ipiv );
	free( work );
	free( creation );
	free( Save_creation );
	free( error );

	/* now free up the 2D arrays */
	for( i=0; i<iso.numLevels[ipH_LIKE][nelem]; ++i )
	{
		free( SaveZ[i] ) ;
		free( z[i] ) ;
	}
	free( SaveZ ) ;
	free( z ) ;

#	ifdef DEBUG_FUN
	fputs( " <->HydroLevelPop()\n", debug_fp );
#	endif
	return;
}
/*lint +e662 Possible access of out-of-bounds pointer*/

