/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*lint -e817 pclint generates tons of bogus nelem < 0 warnings*/
/*MakeRecomb generate recombination coefficients for any species */
/*MakeRecombAGN generate recombination coefficients for AGN table */
#include "cddefines.h"
#include "nsbig.h"
#include "phycon.h"
#include "tfidle.h"
#include "heavy.h"
#include "dense.h"
#include "iso.h"
#include "abund.h"
#include "ionrange.h"
#include "chargtran.h"
#include "ionrec.h"
#include "punch.h"
#include "elementnames.h"
#include "drfe.h"
#include "converge.h"
#include "rrfit.h"
#include "makerecomb.h"

void MakeRecomb(double rec[], 
  double pl[], 
  double *dicoef, 
  double *dite, 
  double ditcrt[], 
  double aa[], 
  double bb[], 
  double cc[], 
  double dd[], 
  double ff[], 
  /* nelem is the atomic number on the C scale, 0 for H */
  long int nelem, 
  double tlow[])
{
#define DICOEF(I_,J_)	(*(dicoef+(I_)*(nelem+1)+(J_)))
#define DITE(I_,J_)	(*(dite+(I_)*(nelem+1)+(J_)))
	long int NeBoundIon, 
	  i, 
	  ion, 
	  limit;
	double 
	  fac2, 
	  factor,
	  DielRecomHiT[LIMELM] ,
	  DielRecomLowT[LIMELM] ,
	  RecomRad3BodGrain[LIMELM] ,
	  t4m1, 
	  tefac;
	float radrec;
	/* these are guessed diel rec rates, taken from 
	  >>refer	all	diel	Ali, B., Blum, R. D., Bumgardner, T. E., Cranmer, S. R., Ferland, G. J., 
	  >>refercon Haefner, R. I., & Tiede, G. P. 1991, PASP, 103, 1182 */
	static double cludge[4]={3e-13,3e-12,1.5e-11,2.5e-11};

#	ifdef DEBUG_FUN
	fputs( "<+>MakeRecomb()\n", debug_fp );
#	endif

	/* set up ionization balance matrix, C(I,1)=destruction, 2=creation
	 * heating rates saved in array B(I) in same scrat block
	 * factor is for Aldrovandi+Pequignot fit, FAC2 is for Nuss+Storey
	 * fit for dielectronic recomb
	 * GrnIonRec is rate ions recombine on grain surface, normally zero;
	 * set in hmole, already has factor of hydrogen density
	 * */

	/* routine only used for Li on up */
	ASSERT( nelem < LIMELM);
	ASSERT( nelem > 1 );

	/* check that range of ionization is correct */
	ASSERT( IonRange.IonLow[nelem] >= 0 );
	ASSERT( IonRange.IonLow[nelem] <= nelem+1 );

	nsbigCom.nsbig = MAX2(IonRange.IonHigh[nelem]+1,nsbigCom.nsbig);
	t4m1 = 1e4/phycon.te;
	fac2 = 1e-14*phycon.sqrte;

	/* test putting limit back to h-like */
	/*limit = MIN2(nelem,IonRange.IonHigh[nelem]-1);*/
	limit = MIN2(nelem-NISO,IonRange.IonHigh[nelem]-1);

	/* zero-out loop comes before main loop since there are off-diagonal
	 * elements in the main ionization loop, due to multi-electron processes */
	/* >>chng 00 dec 07, limit changed to identical to bidiag */
	for( ion=0; ion <= limit; ion++ )
	{
		ionrec.TotRecomRate[nelem][ion] = 0.;
		ionrec.RadRecomRateCoef[nelem][ion] = 0.;
		RecomRad3BodGrain[ion] = 0.;
		DielRecomHiT[ion] = 0.;
		DielRecomLowT[ion] = 0.;
	}
	for( ion=limit+1; ion < LIMELM; ion++ )
	{
		/* >>chng 01 dec 18, do not set this to FLT_MAX since it clobbers what
		 * had been set in h-like and he-like routines - that would only affect
		 * the printout */
		RecomRad3BodGrain[ion] = -FLT_MAX;
		DielRecomHiT[ion] = -FLT_MAX;
		DielRecomLowT[ion] = -FLT_MAX;
	}

	/* H-like and He-like stages use results from hydrogenic series done by hydro... routines */
	/* this was evaluated in hydro routines, and must not be changed here
	 * since we want the bidiag routine to get the same answer */
	RecomRad3BodGrain[nelem] = ionrec.TotRecomRate[nelem][nelem]/dense.eden ;
	/* now do he-like */
	RecomRad3BodGrain[nelem-1] = ionrec.TotRecomRate[nelem][nelem-1]/dense.eden ;

	DielRecomHiT[nelem] = 0. ;
	DielRecomHiT[nelem-1] = 0. ;

	DielRecomLowT[nelem] = 0. ;
	DielRecomLowT[nelem-1] = 0. ;

	/* these are counted elsewhere and must not be added here */
	Heavy.xLyaHeavy[nelem][nelem] = 0.;
	Heavy.xLyaHeavy[nelem][nelem-1] = 0.;


	/*--------------------------------------------------------------------- */

	/* ionLow is on physical scale, so is 1 for atom */
	for( ion=IonRange.IonLow[nelem]; ion <= limit; ion++ )
	{
		/* number of bound electrons of the ion, for an atom this is
		 * equal to nelem, the element on the c scale, nelem is 5 for carbon */
		NeBoundIon = nelem - ion;
		if( ionrec.lgVrrFit )
		{
			/* this branch dima's fits */
			radrec = (float)rrfit( nelem+1 , NeBoundIon+1 , phycon.te );
		}
		else
		{
			/* this branch uses old fits to re coef */
			if( (phycon.te > 1e3 || ion > 0) || tlow[0] == 0. )
			{
				radrec = (float)(rec[ion]*pow((double)phycon.te,pl[ion]));
			}
			else
			{
				/* use special Pequignot and Aldrovandi fit for low temp */
				radrec = (float)(tlow[0]*pow((double)phycon.te,tlow[1]));
			}
		}
		ionrec.TotRecomRate[nelem][ion] = dense.eden*(radrec + 
		  ionrec.CotaRate[ion]) ;

		/* recombinatinon on grain surface */
		ionrec.TotRecomRate[nelem][ion] += ionrec.GrnIonRec;

		/* >>chng 02 nov 06 add charge transfer here rather than in bidiag */
		/* charge transfer recombination of this species by ionizing hydrogen and helium */
		ionrec.TotRecomRate[nelem][ion] += 
			/* >>chng 02 may 05, should be just ground state, make it so */
			/*ChargTran.HeCharExcRec[nelem][ion]*dense.xIonDense[ipHELIUM][0] + */
			ChargTran.HeCharExcRec[nelem][ion]*iso.Pop2Ion[ipHE_LIKE][ipHELIUM][ipHe1s1S]*dense.xIonDense[ipHELIUM][1] + 
			/*ChargTran.HCharExcRec[nelem][ion]*dense.xIonDense[ipHYDROGEN][0];*/
			/* >>chng 01 06 01 should be ground state only, change to do so */
			ChargTran.HCharExcRec[nelem][ion]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s]*dense.xIonDense[ipHYDROGEN][1];

		/* rate coefficient for radiative plus three body, and
		 * recombination on grain surfaces */
		RecomRad3BodGrain[ion] = radrec + ionrec.CotaRate[ion] + ionrec.GrnIonRec/dense.eden;

		/* save radiative recombination rate coef since needed for rec cooling,
		 * this should be pure rad rec, and not include density, so rate coef, cm3 s-1 */
		ionrec.RadRecomRateCoef[nelem][ion] = radrec;

		/* Burgess dielectronic recombination */
		DielRecomHiT[ion] = 0.;
		if( phycon.te > (float)(ditcrt[ion]*0.1) )
		{
			DielRecomHiT[ion] = ionrec.DielSupprs[0][ion]/phycon.te32*
			  DICOEF(0,ion)*exp(-DITE(0,ion)/phycon.te)*
			  (1. + DICOEF(1,ion)*
			  sexp(DITE(1,ion)/phycon.te));

			ionrec.TotRecomRate[nelem][ion] += DielRecomHiT[ion]*
			  dense.eden;
		}

		/* Nussbaumer and Storey dielectronic recombination
		 * do not ever do it for rec from
		 * a closed shell, NeBoundIon is number of bound electrons in ion */
		DielRecomLowT[ion] = 0.;
		if( (NeBoundIon != 2 && NeBoundIon != 10) && NeBoundIon != 18 )
		{
			tefac = ff[ion]*t4m1;
			/* fac2 = 1e-14 * sqrte */
			if( ff[ion] != 0. && fabs(tefac) < 30. )
			{
				factor = (((aa[ion]*t4m1+bb[ion])*t4m1+cc[ion])*t4m1+dd[ion])* exp(-tefac);
				DielRecomLowT[ion] = ionrec.DielSupprs[1][ion]*fac2*
					MAX2(0.,factor );
			}
			else if( (phycon.te > 100. && ff[ion] == 0.) && ion <= 3 )
			{
				/* >>chng 01 jun 30, make GuessDiel array */
				DielRecomLowT[ion] = ionrec.DielSupprs[1][ion]*cludge[ion]*
				  ionrec.GuessDiel[ion];
			}
			ionrec.TotRecomRate[nelem][ion] += DielRecomLowT[ion]*
			  dense.eden;
		}

		/* >>chng 01 jun 30, FRAC_LINE was 0.1, not 1, did not include anything except
		 * radiative recombination, the radrec term */
#		define FRAC_LINE 1.
		/* was 0.1 */
		/*Heavy.xLyaHeavy[nelem][ion] = (float)(dense.eden*radrec*FRAC_LINE );*/
		Heavy.xLyaHeavy[nelem][ion] = (float)(dense.eden*
			(radrec+DielRecomLowT[ion]+DielRecomHiT[ion])*FRAC_LINE );

	}

	/* for case of iron, add on Dima's fits to diel part, above was zero */
	if( nelem==ipIRON )
	{
		float DieRate;
		/* add on special dima diel coef */
		limit = MIN2(nelem-NISO,IonRange.IonHigh[nelem]-1);
		for( ion=IonRange.IonLow[nelem]; ion <= limit; ion++ )
		{
			drfe(ion+1,phycon.te,&DieRate);
			DielRecomLowT[ion] = DieRate;
			/*fprintf(ioQQQ,"%li %g %g\n", ion, ionrec.TotRecomRate[NELEM-1][ion], DieRate*dense.eden );*/
			ionrec.TotRecomRate[nelem][ion] += DieRate*dense.eden;
		}
	}

	/* option to punch rec coef */
	if( punch.lgioRecom )
	{
		/* print name of element */
		fprintf( punch.ioRecom, 
			" %s recombination coefficients, te=%.3e \n", 
			elementnames.chElementName[nelem] , phycon.te );

		limit = MIN2(11,IonRange.IonHigh[nelem]);
		for( i=0; i < limit; i++ )
		{
			fprintf( punch.ioRecom, "%10.2e", RecomRad3BodGrain[i] );
		}
		fprintf( punch.ioRecom, " radiative + 3body + grain vs Z\n" );

		for( i=0; i < limit; i++ )
		{
			fprintf( punch.ioRecom, "%10.2e", DielRecomHiT[i] );
		}
		fprintf( punch.ioRecom, " Burgess vs Z\n" );

		for( i=0; i < limit; i++ )
		{
			fprintf( punch.ioRecom, "%10.2e", DielRecomLowT[i] );
		}
		fprintf( punch.ioRecom, " Nussbaumer Storey vs Z\n" );

		/* total recombination rate, with density included - this goes into the matrix */
		for( i=0; i < limit; i++ )
		{
			fprintf( punch.ioRecom, "%10.2e", ionrec.TotRecomRate[nelem][i] );
		}
		fprintf( punch.ioRecom, 
			" total rec rate (with density) for %s\n\n", 
			elementnames.chElementSym[nelem] );

		/* spill over to next line for many stages of ionization */
		if( IonRange.IonHigh[nelem] > 11 )
		{
			limit = MIN2(29,IonRange.IonHigh[nelem]);
			fprintf( punch.ioRecom, " R " );
			for( i=11; i < limit; i++ )
			{
				fprintf( punch.ioRecom, "%10.2e", RecomRad3BodGrain[i] );
			}
			fprintf( punch.ioRecom, "\n" );

			fprintf( punch.ioRecom, " B " );
			for( i=11; i < limit; i++ )
			{
				fprintf( punch.ioRecom, "%10.2e", DielRecomHiT[i] );
			}
			fprintf( punch.ioRecom, "\n" );

			fprintf( punch.ioRecom, " NS" );
			for( i=11; i < limit; i++ )
			{
				fprintf( punch.ioRecom, "%10.2e", DielRecomLowT[i] );
			}
			fprintf( punch.ioRecom, "\n" );

			fprintf( punch.ioRecom, "   " );
			for( i=11; i < limit; i++ )
			{
				fprintf( punch.ioRecom, "%10.2e", ionrec.TotRecomRate[nelem][i] );
			}
			fprintf( punch.ioRecom, "\n\n" );
		}
	}

	/* >>chng 02 nov 09, from -2 to -NISO */
	/*limit = MIN2(nelem-2,IonRange.IonHigh[nelem]-1);*/
	limit = MIN2(nelem-NISO,IonRange.IonHigh[nelem]-1);
	for( i=IonRange.IonLow[nelem]; i <= limit; i++ )
	{
		ASSERT( Heavy.xLyaHeavy[nelem][i] > 0. );
		ASSERT( ionrec.TotRecomRate[nelem][i] > 0. );
	}

#	ifdef DEBUG_FUN
	fputs( " <->MakeRecomb()\n", debug_fp );
#	endif
	return;
#	undef	DITE
#	undef	DICOEF
}

/*MakeRecombAGN generate recombination coefficients for AGN table */
void MakeRecombAGN( FILE * io )
{
#	define N1LIM 3
#	define N2LIM 4
	double te1[N1LIM]={ 5000., 10000., 20000.};
	double te2[N2LIM]={ 20000.,50000.,100000.,1e6};
	/* this is boundary between two tables */
	double BreakEnergy = 100./13.0;
	long int nelem, ion , i;
	/* this will hold element symbol + ionization */
	char chString[100],
		chOutput[100];
	/* save temp here	*/
	float TempSave = phycon.te;
	/* save ne here	*/
	double EdenSave = dense.eden;

#	ifdef DEBUG_FUN
	fputs( "<+>MakeRecomb()\n", debug_fp );
#	endif

	dense.eden = 1.;
	/*CreateData();*/

	/* first put header on file */
	fprintf(io,"X+i\\Te");
	for( i=0; i<N1LIM; ++i )
	{
		phycon.te = (float)te1[i];
		fprintf(io,"\t%.0f K",phycon.te);
	}
	fprintf(io,"\n");

	/* now do loop over temp, but add elements */
	for( nelem=ipLITHIUM; nelem<LIMELM; ++nelem )
	{
		/* this list of elements included in the AGN tables is defined in zeroabun.c */
		if( abund.lgAGN[nelem] )
		{
			for( ion=0; ion<=nelem; ++ion )
			{
				ASSERT( Heavy.Valence_IP_Ryd[nelem][ion] > 0.05 );

				if( Heavy.Valence_IP_Ryd[nelem][ion] > BreakEnergy )
					break;

				/* print chemical symbol */
				sprintf(chOutput,"%s", 
					elementnames.chElementSym[nelem]);
				/* some elements have only one letter - this avoids leaving a space */
				if( chOutput[1]==' ' )
					chOutput[1] = chOutput[2];
				/* now ionization stage */
				if( ion==0 )
				{
					sprintf(chString,"0 ");
				}
				else if( ion==1 )
				{
					sprintf(chString,"+ ");
				}
				else
				{
					sprintf(chString,"+%li ",ion);
				}
				strcat( chOutput , chString );
				fprintf(io,"%5s",chOutput );

				for( i=0; i<N1LIM; ++i )
				{
					phycon.te = (float)te1[i];
					IonRange.IonLow[nelem] = 0;
					IonRange.IonHigh[nelem] = nelem+1;
					tfidle(FALSE);
					ConvIonizeOpacityDo(0);
					fprintf(io,"\t%.2e",ionrec.TotRecomRate[nelem][ion]);
				}
				fprintf(io,"\n");
			}
			fprintf(io,"\n");
		}
	}

	/* second put header on file */
	fprintf(io,"X+i\\Te");
	for( i=0; i<N2LIM; ++i )
	{
		phycon.te = (float)te2[i];
		tfidle(FALSE);
		fprintf(io,"\t%.0f K",phycon.te);
	}
	fprintf(io,"\n");

	/* now do same loop over temp, but add elements */
	for( nelem=ipHYDROGEN; nelem<LIMELM; ++nelem )
	{
		/* this list of elements included in the AGN tables is defined in zeroabun.c */
		if( abund.lgAGN[nelem] )
		{
			for( ion=0; ion<=nelem; ++ion )
			{
				ASSERT( Heavy.Valence_IP_Ryd[nelem][ion] > 0.05 );

				if( Heavy.Valence_IP_Ryd[nelem][ion] <= BreakEnergy )
					continue;

				/* print chemical symbol */
				fprintf(io,"%s", 
					elementnames.chElementSym[nelem]);
				/* now ionization stage */
				if( ion==0 )
				{
					fprintf(io,"0 ");
				}
				else if( ion==1 )
				{
					fprintf(io,"+ ");
				}
				else
				{
					fprintf(io,"+%li",ion);
				}

				for( i=0; i<N2LIM; ++i )
				{
					phycon.te = (float)te2[i];
					IonRange.IonLow[nelem] = 0;
					IonRange.IonHigh[nelem] = nelem+1;
					tfidle(FALSE);
					ConvIonizeOpacityDo(0);
					fprintf(io,"\t%.2e",ionrec.TotRecomRate[nelem][ion]);
				}
				fprintf(io,"\n");
			}
			fprintf(io,"\n");
		}
	}

	phycon.te = TempSave;
	dense.eden = EdenSave;
	tfidle(TRUE);

#	ifdef DEBUG_FUN
	fputs( " <->MakeRecombAGN()\n", debug_fp );
#	endif
	return;
}

/*lint +e817 pclint generates tons of bogus nelem < 0 warnings*/

