/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*NextDR use adaptive logic to find next zone thickness */
/*ContRate called by nextdr to find energy of maximum continuum-gas interaction */
/*GrainRateDr called by nextdr to find grain heating rate dr */
/*ChkRate called by nextdr to check how rates of destruction of various species changes */
/*TODO - this routine is very important since it sets the pace for the calculation,
 * and directly affects the convergence of the code.  Most of the logic is very old and
 * messy.  
 * 1) make sure all test cases have punch dr
 * 2) cat all these reasons together into one file and sort on the reason
 * 3) discover what logic is the main pacesetter for the code
 * 4) which are never triggered and so can be removed
 */
#include "cddefines.h"
#include "lines_service.h"
#include "iso.h"
#include "geometry.h"
#include "opacity.h"
#include "dense.h"
#include "heavy.h"
#include "grainvar.h"
#include "converge.h"
#include "ionrec.h"
#include "rfield.h"
#include "dynamics.h"
#include "thermal.h"
#include "hmi.h"
#include "coolheavy.h"
#include "timesc.h"
#include "fluct.h"
#include "stopcalc.h"
#include "colden.h"
#include "heat.h"
#include "phycon.h"
#include "radius.h"
#include "trace.h"
#include "didz.h"
#include "wind.h"
#include "punch.h"
#include "taulines.h"
#include "pressure.h"
#include "itercnt.h"
#include "dtmase.h"
#include "struc.h"
#include "fndlineht.h"
#include "tabden.h"
#include "fabden.h"
#include "nextdr.h"

/*ChkRate called by nextdr to check how rates of destruction of various species changes */
static void ChkRate(
	  /* element number on physical scale */
	  long int nelem, 
	  /* change in destruction rate */
	  double *dDestRate, 
	  /* old and new destruction rates */
	  double *DestRateOld,
	  double *DestRateNew,
	  /* stage of ionization on the physical scale */
	  long int *istage);

/*ContRate called by nextdr to find energy of maximum continuum-gas interaction */
static void ContRate(double *freqm, 
  double *opacm);

/*GrainRateDr called by nextdr to find grain heating rate dr */
static void GrainRateDr(double *grfreqm, 
  double *gropacm);

/*lint -e777 floating tests for equality are ok and frequent here */
void NextDR(void)
{
	char chDestAtom[9], 
	  chLbl[11];
	int lgDoPun, 
	  lgOscil;

	long int icarstag, 
	  iironstag, 
	  initstag, 
	  ioxystag, 
	  ipStrong, 
	  istage, 
	  k, 
	  level, 
	  limit;

	double ThicknessSet , dThickness , drThickness , DepthToGo ;

	double DrGrainHeat, 
	  GlobDr, 
	  Out2Tot, 
	  SaveOHIIoHI, 
	  SpecDr, 
	  Strong, 
	  TauDTau, 
	  TauInwd, 
	  coleff, 
	  dDRCarbon, 
	  dDRIron, 
	  dDRNitrogen, 
	  dDROxygen, 
	  dDestRate, 
	  dEden, 
	  dHdStep, 
	  dRTaue, 
	  dTdStep, 
	  dnew, 
	  drConPres, 
	  drConvergeIoniz ,
	  drH2_heat_cool = 0. ,
	  dH2_heat_cool = 0.,
	  drH2_abund = 0. ,
	  dH2_abund=0.,
	  drDepth, 
	  drDest, 
	  drEden, 
	  drFail, 
	  drFluc, 
	  drHMase, 
	  drHe1ovHe2,
	  drHion, 
	  drInter, 
	  drLineHeat, 
	  drOutFrac, 
	  drPressure, 
	  drSphere, 
	  drTab, 
	  drdHdStep, 
	  drdTdStep, 
	  drmax, 
	  dt, 
	  dVeldRad,
	  error, 
	  fac2, 
	  factor, 
	  freqm, 
	  grfreqm, 
	  gropacm, 
	  hdnew, 
	  opacm, 
	  OldDR ,
	  OldestEden,
	  winddr, 
	  x;
	double DestOldCarb,DestNewCarb, DestOldNit, DestNewNit , DestOldOxy,DestNewOxy,
		DestOldIron,DestNewIron, DestOld, DestNew;

	static double OHIIoHI, 
	  /*OHe2oHe1 = 0., */
	  OldEden, 
	  OldHeat = 0., 
	  OldTe = 0.,
	  OlddTdStep = 0.,
	  OldH2Abund=0.,
	  OldWindVelocity=0.,
	  Old_He_atom_ov_ion = 0,
	  Old_H2_heat_cool,
	  Old_H2_abund;

	static double BigRadius = 1e30;

#	ifdef DEBUG_FUN
	fputs( "<+>NextDR()\n", debug_fp );
#	endif


	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	 *
	 * changes in logic
	 * 95 oct 19, drSphere now 3% of radius, was 2%, fewer zone
	 * 95 oct 19, subtracted grain opacity from total opacity used
	 * to get thickness in routine ContRate
	 *
	 *>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	 *
	 * free statement labels >= 13
	 *
	 *-----------------------------------------------------------------------
	 *
	 * this sub determines the thickness of the next zone
	 * if is called one time for each zone
	 * flag lgNxtDROff is true if this is initialization of NextDR,
	 * is false if we are to use logic to find dr
	 *
	 *----------------------------------------------------------------------- */

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, "   NextDR called\n" );
	}

	/* save current dr */
	OldDR = radius.drad;

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/*  1    '' NextDR keys from change in H ionization'',e11.3)')
	 * check on change in hydrogen ionizaiton */
	if( nzone <= 1 )
	{
		if( dense.xIonDense[ipHYDROGEN][1] > 0. && dense.xIonDense[ipHYDROGEN][0] > 0. )
		{
			OHIIoHI = dense.xIonDense[ipHYDROGEN][1]/dense.xIonDense[ipHYDROGEN][0];
		}
		else
		{
			OHIIoHI = 0.;
		}
		drHion = BigRadius;
		SaveOHIIoHI = dense.xIonDense[ipHYDROGEN][1]/dense.xIonDense[ipHYDROGEN][0];
		/* else if(hii.gt.0. .and. hi.gt.0. .and. OHIIoHI.gt.0. ) then
		 * >>chng 97 jul 9, for deep in PDR vastly now ionz H slowed down works */
	}

	else if( (dense.xIonDense[ipHYDROGEN][1] > 0. && dense.xIonDense[ipHYDROGEN][0] > 0.) && OHIIoHI > 1e-15 )
	{
		double atomic_frac = (dense.xIonDense[ipHYDROGEN][1]/dense.xIonDense[ipHYDROGEN][0]);
		/* ratio of current HII/HI to old value - < 1 when becoming more neutral */
		/* this is now change in atomic fraction */
		x = 1. - atomic_frac /OHIIoHI;
		if( atomic_frac > 0.05 && atomic_frac < 0.9 )
		{
			/* >>chng 96 feb 23 from 0.7 to 0.3 because punched through H i-front
			 * >>chng 97 may 5, from 0.3 to 0.2 since punched through H i-front */
			/* >>chng 02 jun 05 from 0.2 to 0.05 poorly resolved i-front, also added two-branch logic*/
			drHion = radius.drad*MAX2( 0.2 , 0.05/MAX2(1e-10,x) );
		}
		else if( x > 0. )
		{
			/* >>chng 96 feb 23 from 0.7 to 0.3 because punched through H i-front
			 * >>chng 97 may 5, from 0.3 to 0.2 since punched through H i-front */
			drHion = radius.drad*MAX2( 0.2 , 0.2/MAX2(1e-10,x) );
		}
		else
		{
			drHion = BigRadius;
		}
		SaveOHIIoHI = OHIIoHI;
		OHIIoHI = dense.xIonDense[ipHYDROGEN][1]/dense.xIonDense[ipHYDROGEN][0];
	}

	else
	{
		SaveOHIIoHI = OHIIoHI;
		if( dense.xIonDense[ipHYDROGEN][1] > 0. && dense.xIonDense[ipHYDROGEN][0] > 0. )
		{
			OHIIoHI = dense.xIonDense[ipHYDROGEN][1]/dense.xIonDense[ipHYDROGEN][0];
		}
		else
		{
			OHIIoHI = 0.;
		}
		drHion = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* "NextDR keys from H maser, dt, ij=" possible hydrogen maser action */
	if( dtMase.dTauMase < -0.01 )
	{
		/* maser so powerful, must limit inc in tay
		 * >>chng 96 aug 08, from 0.3 to 0.1 due to large maser crash */
		drHMase = radius.drad*MAX2(0.1,-0.2/dtMase.dTauMase);
	}
	else
	{
		drHMase = BigRadius;
	}

	/* NextDR keys from change in He0/He+ ionization, old=%11.3e sv=%11.3e */
	/* >>chng 02 dec 12, add logic for increase in He0/He+ as I-front is approached */
	if( nzone <= 1 )
	{
		if( dense.xIonDense[ipHELIUM][1] > SMALLFLOAT )
		{
			Old_He_atom_ov_ion = dense.xIonDense[ipHELIUM][0]/dense.xIonDense[ipHELIUM][1];
		}
		else
		{
			Old_He_atom_ov_ion = 0.;
		}
		drHe1ovHe2 = BigRadius;
	}
	else
	{
		if( dense.xIonDense[ipHELIUM][1] > SMALLFLOAT )
		{
			if( dense.xIonDense[ipHELIUM][0]/dense.xIonDense[ipHELIUM][1] > 1000. )
			{
				/* >>chng 97 jul 12, He mostly neutral but Lya photoionized,
				* oscillation could
				* take place - do not trigger dr from this oscillation */
				drHe1ovHe2 = BigRadius;
			}
			else
			{
				double he1ov2_incr;
				double atom_ov_ion = dense.xIonDense[ipHELIUM][0]/dense.xIonDense[ipHELIUM][1];
				/* >>chng 02 dec 12, new logic, just look at change in ionization */
				he1ov2_incr = atom_ov_ion / MAX2(SMALLFLOAT, Old_He_atom_ov_ion);
				if( he1ov2_incr > 1. )
				{
					double fac = 1.3;
					drHe1ovHe2 = radius.drad / MIN2(5., he1ov2_incr / fac );
				}
				else
				{
					drHe1ovHe2 = BigRadius;
				}
			}
			Old_He_atom_ov_ion = dense.xIonDense[ipHELIUM][0]/dense.xIonDense[ipHELIUM][1];
		}
		else
		{
			Old_He_atom_ov_ion = 0.;
			drHe1ovHe2 = BigRadius;
		}
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* check on how much outward flux is being added relative to
	 * attenuated incident beam
	 *  1    '' NextDR keys from ratio out2 tot continuua'',e9.1)') */
	if( (rfield.SumIncCon > 0. && rfield.SumOutCon > 0.) && 
	  !thermal.lgTSetOn )
	{
		/* evaluated in metdif */
		Out2Tot = rfield.SumOutCon/rfield.SumIncCon;
		/* Out2HTot = SumHOutCon / SumHIncCon
		 * try to keep outward frac less than 5% of total */
		drOutFrac = radius.drad*MAX2(0.7,0.05/MAX2(1e-10,Out2Tot));
	}
	else
	{
		Out2Tot = FLT_MAX;
		drOutFrac = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* check how heating is changing
	 * '' NextDR keys from change in heating; current, delta='', */
	if( nzone <= 1 || thermal.lgTSetOn )
	{
		drdHdStep = BigRadius;
		dHdStep = FLT_MAX;
	}
	else
	{
		dHdStep = fabs(heat.htot-OldHeat)/heat.htot;
		if( dHdStep > 0. )
		{
			if( dense.gas_phase[ipHYDROGEN] >= 1e13 )
			{
				/* drdHdStep = drad * MAX( 0.8 , 0.05/dHdStep ) */
				drdHdStep = radius.drad*MAX2(0.8,0.075/dHdStep);
			}
			else if( dense.gas_phase[ipHYDROGEN] >= 1e11 )
			{
				/* drdHdStep = drad * MAX( 0.8 , 0.075/dHdStep ) */
				drdHdStep = radius.drad*MAX2(0.8,0.100/dHdStep);
			}
			else
			{
				/* changed from .15 to .12 for outer edge of coolhii too steep dT
				 * changed to .10 for symmetry, big change in some rates, 95nov14
				 * changed from .10 to .125 since parispn seemed to waste zones
				 * >>chng 96 may 21, from .125 to .15 since pn's still waste zones */
				drdHdStep = radius.drad*MAX2(0.8,0.15/dHdStep);
			}
		}
		else
		{
			drdHdStep = BigRadius;
		}
	}
	OldHeat = heat.htot;

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* pressure due to incident continuum if in eos */
	if( strcmp(dense.chDenseLaw,"CPRE") == 0 && pressure.lgContRadPresOn )
	{
		if( nzone > 2 && pressure.pinzon > 0. )
		{
			/* pinzon is pressrue from acceleration onto previos zone
			 * want this less than some fraction of total pressure */
			drConPres = 0.05*pressure.PresTotlInit/(wind.AccelTot*
			  phycon.xMassDensity*geometry.FillFac);
		}
		else
		{
			drConPres = BigRadius;
		}
	}
	else
	{
		drConPres = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* check how temperature is changing
	 *  1    '' NextDR keys from change in temperature; current, delta='', */
	if( nzone <= 1 )
	{
		drdTdStep = BigRadius;
		dTdStep = FLT_MAX;
		OlddTdStep = dTdStep;
	}
	else
	{
		dTdStep = (phycon.te-OldTe)/phycon.te;
		/* >>chng 02 dec 08, desired change in temperature per zone must not
		 * be smaller than allower error in temperature.  For now use relative error
		 * in heating - - cooling balance.  Better would be to also use c-h deriv wrt t
		 * to get slope */
		x = conv.HeatCoolRelErrorAllowed*2.;
		x = MAX2( 0.01 , x ); 
		x = MIN2( 0.05 , x );
		/* >>chng 02 dec 11 rjrw change back to 0.03 -- improve dynamics.dRad criterion instead */
		x = 0.03;
		/* >>chng 02 dec 07, do not do this if there is mild te jitter, 
		 * so that dT is changing sign - this happens
		 * in ism.ini, very stable temperature with slight noise up and down */
		if( dTdStep*OlddTdStep > 0. )
		{
			/* >>chng 96 may 30, variable depending on temp
			 * >>chng 96 may 31, allow 0.7 smaller, was 0.8
			 * >>chng 97 may 05, from 0.7 to 0.5 stop from punching through thermal front */
			double absdt = fabs(dTdStep);
			drdTdStep = radius.drad*MAX2(0.7,x/absdt);
		}
		else
		{
			drdTdStep = BigRadius;
		}
	}
	OlddTdStep = dTdStep;
	OldTe = phycon.te;

	/* >>chng 02 oct 06, only check on opacity - interaction if not
	 * constant temperature - there were constant temperature models that
	 * extended to infinity but hung with last few photons and this test.
	 * better to ignore this test which is really for thermal models */
	if( !thermal.lgTSetOn )
	{
		/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
		/* find freq where opacity largest and interaction rate is fastest
		* "cont inter nu=%10.2e opac=%10.2e\n" */
		ContRate(&freqm,&opacm);

		/* use optical depth at max interaction energy
		* >>chng 96 jun 06 was drChange=0.15 changed to 0.3 for high Z models
		* taking too many zones
		* drInter = drChange / MAX(1e-30,opacm*FillFac) */

		drInter = 0.3/MAX2(1e-30,opacm*geometry.FillFac*geometry.AngleIllum);
	}
	else
	{
		drInter = BigRadius;
		freqm = 0.;
		opacm = 1.;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* check on changes in destruction rates for various atoms */
	ChkRate(ipCARBON,&dDRCarbon,&DestOldCarb, &DestNewCarb,&icarstag);
	ChkRate(ipNITROGEN,&dDRNitrogen,&DestOldNit, &DestNewNit,&initstag);
	ChkRate(ipOXYGEN,&dDROxygen,&DestOldOxy, &DestNewOxy,&ioxystag);
	ChkRate(ipIRON,&dDRIron,&DestOldIron, &DestNewIron,&iironstag);

	dDestRate = vfmax(dDROxygen,dDRIron,dDRCarbon,dDRNitrogen,FEND);

	if( dDRCarbon == dDestRate )
	{
		dDestRate = dDRCarbon;
		DestOld = DestOldCarb;
		DestNew = DestNewCarb;
		istage = icarstag;
		strcpy( chDestAtom, "Carbon  " );
	}

	else if( dDRNitrogen == dDestRate )
	{
		dDestRate = dDRNitrogen;
		DestOld = DestOldNit;
		DestNew = DestNewNit;
		istage = initstag;
		strcpy( chDestAtom, "Nitrogen" );
	}

	else if( dDROxygen == dDestRate )
	{
		dDestRate = dDROxygen;
		DestOld = DestOldOxy;
		DestNew = DestNewOxy;
		strcpy( chDestAtom, "Oxygen  " );
		istage = ioxystag;
	}

	else if( dDRIron == dDestRate )
	{
		dDestRate = dDRIron;
		DestOld = DestOldIron;
		DestNew = DestNewIron;
		istage = iironstag;
		strcpy( chDestAtom, "Iron    " );
	}

	else
	{
		fprintf( ioQQQ, " insanity following ChkRate\n" );
		ShowMe();
		puts( "[Stop in nextdr]" );
		cdEXIT(EXIT_FAILURE);
	}

	/*  NextDR keys from change in dest rates, atom= */
	if( dDestRate > 0. )
	{
		/* if( te.gt.40 000. ) then
		 * added different accuracy for hot gas since tend to jump over
		 * big te range for small change in heat (intrinsically unstable)
		 * drDest = drad * MAX( 0.5 , 0.10/dDestRate )
		 * else
		 * was drChange, changed to .15 for parishii go through HeII=HeI I front
		 * drDest = drad * MAX( 0.5 , 0.15/dDestRate )
		 * >>chng 95 dec 18 from above to below to stop oscillations
		 * >>chng 95 dec 27 from min of .5 to .75 to stop zone size from changing rapidly
		 * drDest = drad * MAX( 0.8 , 0.20 /dDestRate )
		 * >>chng 96 jan 14 from .2 to .25 to use less zones
		 * >>chng 96 may 30 from min of 0.8 to 0.5 to prevent crashing into He i-front */
		drDest = radius.drad*MAX2(0.5,0.20/dDestRate);
		/* endif */
	}
	else
	{
		drDest = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* check whether change in wind velocity constrains DRAD */
	if( wind.windv!=0. )
	{
		double v = fabs(wind.windv);
		/* this is fractional accel over length */
		dVeldRad = fabs(wind.windv-OldWindVelocity)/
			MAX2(v,0.1*timesc.sound_speed_isothermal)/radius.drad;

		if( 1.1*dVeldRad*radius.drad > 0.03  )
		{
			/* dVeldRad is D(vel)/vel / DRAD, computed in convpres */
			winddr = 0.03/dVeldRad;
		}
		else
		{
			winddr = 1.1*radius.drad;
		}

		/* >>chng 02 nov 05, add dr from advective term,
		 * the 1/500 came from looking at one set of structure plots */
		if( dynamics.lgAdvection )
		{
			/* >>chng 02 dec 11, from 5 to 20 */
			winddr = MIN2( winddr , dynamics.dRad / 20. );
		}
	}
	else
	{
		winddr = BigRadius;
		dVeldRad = 0.;
	}
	/* remember old velocity */
	OldWindVelocity = wind.windv;

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* inside out globule */
	if( strcmp(dense.chDenseLaw,"GLOB") == 0 )
	{
#		define	DNGLOB	0.10
		if( radius.glbdst < 0. )
		{
			fprintf( ioQQQ, " Globule distance is negative, internal overflow has occured,  sorry.\n" );
			fprintf( ioQQQ, " This is routine NextDR, GLBDST is%10.2e\n", 
			  radius.glbdst );
			puts( "[Stop in nextdr]" );
			cdEXIT(EXIT_FAILURE);
		}
		factor = radius.glbden*pow(radius.glbrad/radius.glbdst,radius.glbpow);
		fac2 = radius.glbden*pow(radius.glbrad/(radius.glbdst - (float)radius.drad),radius.glbpow);
		if( fac2/factor > 1. + DNGLOB )
		{
			/* DNGLOB is relative change in density allowed this zone, 0.10 */
			GlobDr = radius.drad*DNGLOB/(fac2/factor - 1.);
		}
		else
		{
			GlobDr = BigRadius;
		}
		/* GlobDr = GLBDST * DNGLOB * (GLBRAD/GLBDST)**(-GLBPOW) / GLBPOW */
		GlobDr = MIN2(GlobDr,radius.glbdst/20.);
	}
	else
	{
		GlobDr = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	hdnew = 0.;
	if( strncmp( dense.chDenseLaw , "DLW" , 3) == 0 )
	{
		/* one of the special density laws, first get density at possible next dr */
		if( strcmp(dense.chDenseLaw,"DLW1") == 0 )
		{
			hdnew = fabden(radius.Radius+radius.drad,radius.depth+
			  radius.drad);
		}
		else if( strcmp(dense.chDenseLaw,"DLW2") == 0 )
		{
			hdnew = tabden(radius.Radius+radius.drad,radius.depth+
			  radius.drad);
		}
		else
		{
			fprintf( ioQQQ, " dlw insanity in NextDR\n" );
			puts( "[Stop in nextdr]" );
			cdEXIT(EXIT_FAILURE);
		}
		drTab = fabs(hdnew-dense.gas_phase[ipHYDROGEN])/MAX2(hdnew,dense.gas_phase[ipHYDROGEN]);
		drTab = radius.drad*MAX2(0.2,0.10/MAX2(0.01,drTab));
	}
	else
	{
		drTab = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* special density law */
	if( strcmp(dense.chDenseLaw,"DLW1") == 0 )
	{
		dnew = fabs(fabden(radius.Radius+radius.drad,radius.depth+
		  radius.drad)/dense.gas_phase[ipHYDROGEN]-1.);
		/* DNGLOB is relative change in density allowed this zone, 0.10 */
		if( dnew == 0. )
		{
			SpecDr = radius.drad*3.;
		}
		else if( dnew/DNGLOB > 1.0 )
		{
			SpecDr = radius.drad/(dnew/DNGLOB);
		}
		else
		{
			SpecDr = BigRadius;
		}
	}
	else
	{
		SpecDr = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* check grain line heating dominates
	 * this is important in PDR and HII region calculations
	 * >>chng 97 jul 03, added following check */
	if( heat.heating[0][13]/heat.htot > 0.2 )
	{
		/* >>chng 01 jan 03, following returns 0 when NO light at all,
		 * had failed with botched assert */
		GrainRateDr(&grfreqm,&gropacm);
		DrGrainHeat = 1.0/MAX2(1e-30,gropacm*geometry.FillFac*geometry.AngleIllum);
	}
	else
	{
		gropacm = 0.;
		grfreqm = 0.;
		DrGrainHeat = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* check if line heating dominates
	 * this is important in high metallicity models */
	if( heat.heating[0][22]/heat.htot > 0.2 )
	{
		FndLineHt(&level,&ipStrong,&Strong);
		if( Strong/heat.htot > 0.1 )
		{
			if( level == 1 )
			{
				/* a level1 line was the heat source (usual case)
				 * drLineHeat = MAX(1.0,TauLines(ipLnTauIn,ipStrong)*0.2) /
				 *  1      TauLines(ipLnDTau,ipStrong) */
				TauInwd = TauLines[ipStrong].TauIn;
				TauDTau = TauLines[ipStrong].dTau;
			}
			else if( level == 2 )
			{
				/* a level2 line was the heat source
				 * (bad case since not as well treated)
				 * drLineHeat = MAX(1.0,WindLine(ipLnTauIn,ipStrong)*0.2) /
				 *  1      WindLine(ipLnDTau,ipStrong) */
				TauInwd = TauLine2[ipStrong].TauIn;
				TauDTau = TauLine2[ipStrong].dTau;
			}
			else if( level == 3 )
			{
				/* a 12CO line was the heat source
				 * (bad case since not as well treated)
				 * drLineHeat = MAX(1.0,WindLine(ipLnTauIn,ipStrong)*0.2) /
				 *  1      WindLine(ipLnDTau,ipStrong) */
				TauInwd = C12O16Rotate[ipStrong].TauIn;
				TauDTau = C12O16Rotate[ipStrong].dTau;
			}
			else if( level == 4 )
			{
				/* a 13CO line was the heat source
				 * (bad case since not as well treated)
				 * drLineHeat = MAX(1.0,WindLine(ipLnTauIn,ipStrong)*0.2) /
				 *  1      WindLine(ipLnDTau,ipStrong) */
				TauInwd = C13O16Rotate[ipStrong].TauIn;
				TauDTau = C13O16Rotate[ipStrong].dTau;
			}
			else
			{
				/* this is insane, since Strong was set, but not level */
				fprintf( ioQQQ, " PROBLEM NextDR Strong line heating set, but not level.\n" );
				TotalInsanity();
			}
			/* in following logic cannot use usual inverse opacity,
			 * since line heating competes with escape probability,
			 * so is effective at surprising optical depths */
			if( TauDTau > 0. )
			{
				drLineHeat = MAX2(1.,TauInwd)*0.4/TauDTau;
			}
			else
			{
				drLineHeat = BigRadius;
			}
		}
		else
		{
			TauInwd = 0.;
			drLineHeat = BigRadius;
			ipStrong = 0;
			Strong = 0.;
		}
	}
	else
	{
		TauInwd = 0.;
		drLineHeat = BigRadius;
		ipStrong = 0;
		level = 0;
		Strong = 0.;
	}

	/* >>chng rm next two - we are now using the solomon process rate,
	 * and collisional cooling within H2 - so these should be covered */
#	if 0
	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* Lyman band optical depth
	 * second term total lyman band optical depth, do not check if thick
	 * H2Opacity is opacity evaluated in hmole */
	if( hmi.Molec[ipMH2]*hmi.H2Opacity > 1e-20 && colden.colden[ipCOLH2]*
	  hmi.H2Opacity < 5. )
	{
		if( colden.colden[ipCOLH2]*hmi.H2Opacity < 2. )
		{
			/* special extra 2 since H2 will change by this amount early on */
			drH2Band = (didz.drChange/2.)/(hmi.Molec[ipMH2]*hmi.H2Opacity*
			  geometry.FillFac);
		}
		else
		{
			drH2Band = didz.drChange/(hmi.Molec[ipMH2]*hmi.H2Opacity*geometry.FillFac);
		}
	}
	else
	{
		drH2Band = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* if H2 cooling - if important cooling source then keep track of abundance */
	if( nzone <= 1 || thermal.lgTSetOn  )
	{
		drH2Abund = BigRadius;
	}
	else
	{
		if( OldH2Abund > SMALLFLOAT )
		{
			dHdStep = fabs(hmi.Molec[ipMH2]/dense.gas_phase[ipHYDROGEN] -OldH2Abund)/OldH2Abund;
			dHdStep = MAX2( SMALLFLOAT , dHdStep );
			/* do this only if significant */
			if( CoolHeavy.h2line/heat.htot > 0.05 )
			{
				fprintf(ioQQQ,"trip h2line\n");fixit();/*rm this line */
				drH2Abund = radius.drad*MAX2(0.1,0.075/dHdStep);
			}
			else
			{
				drH2Abund = radius.drad*MAX2(0.8,0.2/dHdStep);
				drH2Abund = BigRadius;
			}
		}
		else 
		{
			drH2Abund = BigRadius;
		}
		/*fprintf(punch.ipDRout,"h2dr dr%.2e\t%.2e\t%.2e\n",CoolHeavy.h2line/heat.htot, dHdStep, 0.075/dHdStep);*/
	}
	OlderH2Abund = OldH2Abund;
	OldH2Abund = hmi.Molec[ipMH2]/dense.gas_phase[ipHYDROGEN];
#	endif

	/* >>chng 03 mar 03, add this logic */
	/* do not let change in cooling/heating due to H2 become too large */
	drH2_heat_cool = BigRadius;
	if( nzone<1 )
	{
		Old_H2_heat_cool = 0.;
	}
	else if( !thermal.lgTSetOn )
	{
		/* this is case where temperature has not been set */
		/* compare total heating - cooling due to h2 with total due to everything */
		double H2_heat_cool = (fabs(hmi.HeatH2Dexc_used)+fabs(hmi.HeatH2Dish_used)) / heat.htot;
		if( H2_heat_cool > 0.1 )
		{
			dH2_heat_cool = fabs( H2_heat_cool - Old_H2_heat_cool );
			drH2_heat_cool = radius.drad*MAX2(0.2,0.05/dH2_heat_cool);
		}
		else
		{
			drH2_heat_cool = BigRadius;
		}
	}
	Old_H2_heat_cool = (fabs(hmi.HeatH2Dexc_used)+fabs(hmi.HeatH2Dish_used)) / heat.htot;

	/* >>chng 03 mar 04, add this logic */
	/* do not let change in H2 abundance become too large */
	drH2_abund = BigRadius;
	if( nzone<1 )
	{
		Old_H2_abund = 0.;
	}
	else 
	{
		if( 2.*hmi.htwo_total/dense.gas_phase[ipHYDROGEN] > 0.01 )
		{
			/* this is percentage change in H2 density */
			dH2_abund = 2.*fabs( hmi.htwo_total - Old_H2_abund ) / hmi.htwo_total;
			/* in testing th85ism the dH2_abund did come out zero */
			drH2_abund = radius.drad*MAX2(0.2,0.05/MAX2(SMALLFLOAT,dH2_abund) );
		}
		else
		{
			drH2_abund = BigRadius;
		}
	}
	Old_H2_abund = hmi.htwo_total ;

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* can't make drmax large deep in model since causes feedback
	 * oscillations with changes in heating or destruction rates
	 * >>chng 96 oct 15, change from 5 to 10 */
	if( nzone < 11 )
	{
		lgOscil = FALSE;
		if( nzone < 5 )
		{
			/* >>chng 96 nov 11, had been 4 * drad up to 11, change to following
			 * to be similar to c90.01, max of 1.3 between 5 and 11 */
			drmax = 4.*radius.drad/geometry.AngleIllum;
		}
		else
		{
			drmax = 1.3*radius.drad/geometry.AngleIllum;
		}
	}
	else
	{
		/* >>chng 96 oct 15, do not let zones increase if oscillations present */
		lgOscil = FALSE;
		/* this routine called before RTOptDepthIncre stores variables
		 * >>chng 96 oct 31, error to declare oscillation propto toler, the 
		 *heating cooling tolerance */
		error = conv.HeatCoolRelErrorAllowed*conv.HeatCoolRelErrorAllowed;
		/*limit = MIN2((long)NZLIM-2,nzone-2);*/
		limit = nzone -2;
		ASSERT( limit < struc.nzlim );
		for( k=nzone - 10; k < limit; k++ )
		{
			/* small residiual is to allow 0.01 rel error */
			if( (struc.testr[k-1] - struc.testr[k])/struc.testr[k]*
			  (struc.testr[k] - struc.testr[k+1])/struc.testr[k] < 
			  -(float)error )
			{
				lgOscil = TRUE;
			}
			/* small residiual is to allow 0.01 rel error */
			if( (struc.ednstr[k-1] - struc.ednstr[k])/struc.ednstr[k]*
			  (struc.ednstr[k] - struc.ednstr[k+1])/struc.ednstr[k] < 
			  -(float)error )
			{
				lgOscil = TRUE;
			}
		}
		if( lgOscil )
		{
			/* >>chng 96 oct 15, do not let zones increase if oscillations present */
			drmax = radius.drad;
		}
		else
		{
			/* >>chng 96 jan 15 was 1.15 - why so small? */
			drmax = 1.3*radius.drad;
		}
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* check on several convergence criteria */

	/* is current pressure ok? */
	if( conv.lgConvPres )
	{
		drPressure = BigRadius;
	}
	else if( nzone > 0 )
	{
		/* first zone usually has pressure fail due to bug in ptot */
		drPressure = radius.drad/2.;
	}
	else
	{
		drPressure = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* >>chng 01 apr 05, add this logic */
	/* an ionization convergence failure of some kind did occur, although
	 * it may no longer be a problem.  Do not let zone get larger */
	if( conv.lgConvIonizThisZone )
	{
		drConvergeIoniz = radius.drad;
	}
	else 
	{
		drConvergeIoniz = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	if( !conv.lgConvTemp )
	{
		drFail = radius.drad/2.;
	}
	else
	{
		drFail = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* change in electron density - NextDR keys from change in elec den,
	 * remember old electron density during first call */
	if( nzone == 0 )
	{
		OldEden = dense.eden;
	}

	/* this is low ionization solution */
	if( dense.eden/dense.gas_phase[ipHYDROGEN] <= 1. )
	{
		dEden = fabs(dense.eden-OldEden)/dense.eden;
		if( dEden > 0. )
		{
			/* >>chng 96 may 15, down to .03 from 0.04
			 * >>chng 96 may 17, up to 0.1 from .03 */
			/* >>chng 99 nov 23, smallest change is 0.3, had been 0.7,
			 * smaller value needed due to large initial changes in ionization
			 * in 94 since code does not make attempt to identify drnext
			 * and stay in middle of zone, as result some near-lte dense models
			 * can have rapid changes in ionization when due to pumped lines,
			 * see dalton.in for examples */
			/* >>chng 99 nov 25, allow dr to become as much as 0.2 smaller,
			 * for dense moels where initial change in eden is large at ill face.
			 * had been 0.2 */
			/* >>chng 01 mar 24, ratio had allowed 10% changes in electron fraction,
			 * in pdr models this was too big, and ConvIoniz did not converge,
			 * change to 0.04 */
			/*drEden = radius.drad*MAX2(0.2,0.10/dEden);*/
			drEden = radius.drad*MAX2(0.2,0.04/dEden);
		}
		else
		{
			drEden = BigRadius;
		}
	}
	else
	{
		dEden = 0.;
		drEden = BigRadius;
	}
	OldestEden = OldEden;
	OldEden = dense.eden;

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* do not let thickness get too large
	 *  1    '' NextDR keys from relative depth'',e11.3)') */
	if( nzone > 20 )
	{
		/*drDepth = radius.depth/20.;*/
		/* >>chng 02 nov 05, change from 1/20 to 1/10 wasted zones early on */
		drDepth = radius.depth/10.;
	}
	else
	{
		drDepth = BigRadius;
	}
	
	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* case where stopping thickness or edge specified, need to approach slowly */
	ThicknessSet = -1.;
	dThickness = -1.;
	DepthToGo = -1.;
	if( StopCalc.HColStop < 5e29 )
	{
		ThicknessSet = StopCalc.HColStop;
		dThickness = dense.gas_phase[ipHYDROGEN]*geometry.FillFac;
		DepthToGo = StopCalc.HColStop-colden.colden[ipCOLUMNDENSITY];
	}

	else if( StopCalc.colpls < 5e29 )
	{
		ThicknessSet = StopCalc.colpls;
		dThickness = (double)(dense.xIonDense[ipHYDROGEN][1])*geometry.FillFac;
		DepthToGo = StopCalc.colpls-colden.colden[ipCHII];
	}

	else if( StopCalc.colnut < 5e29 )
	{
		ThicknessSet = StopCalc.colnut;
		dThickness = (double)(dense.xIonDense[ipHYDROGEN][0])*geometry.FillFac;
		DepthToGo = StopCalc.colnut - colden.colden[ipCHI] ;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* this is case where outer radius is set */
	if( radius.router[iteration-1] < 5e29 )
	{
		ThicknessSet = radius.router[iteration-1] ;
		dThickness = 1.;
		DepthToGo = radius.router[iteration-1] - radius.depth ;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* set dr if one of above tests have triggered */
	if( ThicknessSet > 0. && DepthToGo > 0.)
	{
		/* want to approach the outer edge slowly,
		 * the need for this logic is most evident in brems.in - 
		 * HI fraction varies across coronal model */
		DepthToGo = MIN2( ThicknessSet/10. , DepthToGo );
		drThickness = DepthToGo / dThickness ;
	}
	else
	{
		drThickness = BigRadius ;
	}
	/*fprintf(ioQQQ," drThickness = %e %e %e \n", drThickness, DepthToGo , dThickness );*/

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* spherical models, do not want delta R/R big */
	drSphere = radius.Radius*0.03;
	drSphere = radius.Radius*0.04;

	/* optical depth to electron scattering */
	dRTaue = didz.drChange/(dense.eden*6.65e-25);
	/* >>chng 02 oct 06, increase dr when constant temperature,
	 * to prevent some ct models from taking too many cells */
	if( thermal.lgTSetOn ) 
		dRTaue *= 3.;

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	if( fluct.flong != 0. )
	{
		/* FacAbunSav = (cfirst * COS( depth*flong ) + csecnd)
		 * this is option for rapid density fluctuations
		 * set DR to roughly 0.1 of fluctuation length
		 * read flcsub first parameter (log of period) is used to get flong
		 * flong = 6.2831853 / period in flcsub */
		drFluc = 0.628/2./fluct.flong;
	}
	else
	{
		drFluc = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/* if density fluctuations in place then override change in heat
	 * for dr set */
	if( strcmp(dense.chDenseLaw,"SINE") == 0 && fluct.lgDenFluc )
	{
		drdHdStep = BigRadius;
	}

	/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*/
	/*active dr sets */
	/* we are deep into model, use logic since we have several zones
	 * of old data */
	radius.drNext = vfmin(drmax,drInter,drLineHeat,winddr,drFluc,GlobDr,
	  DrGrainHeat,/*drTotalPower,*/FEND);
	radius.drNext = vfmin(radius.drNext,SpecDr,drFail,drSphere,radius.sdrmax,
	  drPressure,drConvergeIoniz,dRTaue,FEND);
	radius.drNext = vfmin(radius.drNext,/*drH2Band,drH2Abund,*/drDest,drdTdStep,drdHdStep,
	  drConPres,drTab,FEND);
	radius.drNext = vfmin(radius.drNext,drOutFrac,drHion,drDepth,
	  drEden,drHMase,drThickness,drHe1ovHe2,drH2_heat_cool,drH2_abund,FEND);

	/* keep dr constant in first two zones, if it wants to increase,
	 * but always allow it to decrease.
	 * to guard against large increases in efrac for balmer cont photo dominated models,
	 */
	if( nzone <= 1 && radius.drNext > OldDR)
	{
		radius.drNext = OldDR;
	}

	/* option to force min drad */
	if( radius.drNext < radius.sdrmin )
	{
		radius.drNext = radius.sdrmin;
	}

	/* dr = zero is a logical mistake */
	if( radius.drNext <= 0. )
	{
		fprintf( ioQQQ, " NextDR finds insane drNext:%10.2e\n", 
		  radius.drNext );
		fprintf( ioQQQ, " all drs follow:%10.2e%10.2e%10.2e%10.2e%10.2e%10.2e\n all drs follow:%10.2e%10.2e%10.2e%10.2e%10.2e\n all drs follow:%10.2e%10.2e%10.2e%10.2e%10.2e\n", 
		  drmax, drInter, drLineHeat, winddr, drFluc, GlobDr, SpecDr, 
		  drFail, drSphere, radius.sdrmax, drPressure, drConvergeIoniz,dRTaue, 
		  OldH2Abund,drOutFrac, drDepth );
		puts( "[Stop in nextdr]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* all this is to only punch on last iteration
	 * the punch dr command is not really a punch command, making this necessary
	 * lgDRon is set true if "punch dr" entered */
	if( punch.lgDROn )
	{
		if( punch.lgDRPLst )
		{
			/* lgDRPLst was set true if "punch" had "last" on it */
			if( IterCnt.lgLastIt )
			{
				lgDoPun = TRUE;
			}
			else
			{
				lgDoPun = FALSE;
			}
		}
		else
		{
			lgDoPun = TRUE;
		}
	}
	else
	{
		lgDoPun = FALSE;
	}
	if( (trace.lgTrace && trace.lgDrBug) || lgDoPun )
	{
		if( !conv.lgConvTemp && nzone > 0 )
		{
			fprintf( punch.ipDRout, " >>>> A temperature failure occured.\n" );
		}
		if( !conv.lgConvPres && nzone > 0 )
		{
			fprintf( punch.ipDRout, " >>>> A pressure failure occured.\n" );
		}

		/* this is common part of each line, the zone count, depth, chosen dr, and depth2go */
		fprintf( punch.ipDRout , "%ld\t%.3e\t%.3e\t%.3e\t", nzone, radius.depth, radius.drNext, radius.Depth2Go );

		/*=======begin active dr sets */
		if( radius.drNext == drLineHeat )
		{
			if( level == 1 )
			{
				strcpy( chLbl, chLineLbl(&TauLines[ipStrong]) );
				fprintf( punch.ipDRout, "level 1 line heating,%10.10s TauIn%10.2e%10.2e%10.2e\n", 
				  chLbl, TauInwd, TauLines[ipStrong].pump, 
				  TauLines[ipStrong].Pesc );
			}
			else if( level == 2 )
			{
				strcpy( chLbl, chLineLbl(&TauLine2[ipStrong]));
				fprintf( punch.ipDRout, "level 2 line heating,%10.10s TauIn%10.2e%10.2e%10.2e\n", 
				  chLbl, TauInwd, TauLine2[ipStrong].pump, 
				  TauLine2[ipStrong].Pesc );
			}
			else
			{
				fprintf( ioQQQ, " insanity pr line heat\n" );
				puts( "[Stop in nextdr]" );
				cdEXIT(EXIT_FAILURE);
			}
		}

		else if( radius.drNext == drDepth )
		{
			fprintf( punch.ipDRout, "relative depth\n");
		}

		else if( radius.drNext == drThickness )
		{
			fprintf( punch.ipDRout, "depth to go\n");
		}

		else if( radius.drNext == drTab )
		{
			fprintf( punch.ipDRout, "spec den law, new old den%10.2e%10.2e\n", 
			  hdnew, dense.gas_phase[ipHYDROGEN] );
		}

		else if( radius.drNext == drHMase )
		{
			fprintf( punch.ipDRout, "H maser, dt, ij=%10.2e%5ld\n", 
			  dtMase.dTauMase, dtMase.ijMase );
		}

		else if( radius.drNext == drHe1ovHe2 )
		{
			/* NextDR keys from change in He2/He1 ionization, old=%11.3e sv=%11.3e FrLya*/
			fprintf( punch.ipDRout, "change in He0/He+ ionization, ratio %.2e\n", 
			  Old_He_atom_ov_ion );
		}

		else if( radius.drNext == drH2_heat_cool )
		{
			/* NextDR keys from change in H2 heating, old=%11.3e sv=%11.3e FrLya*/
			fprintf( punch.ipDRout, "change in H2 heating/cooling, d(c,h)/H %.2e\n", 
			  dH2_heat_cool );
		}

		else if( radius.drNext == drH2_abund )
		{
			/* NextDR keys from change in H2 abundance, old=%11.3e sv=%11.3e FrLya*/
			fprintf( punch.ipDRout, "change in H2 abundance, d(H2)/H %.2e\n", 
			  dH2_abund );
		}

		else if( radius.drNext == drHion )
		{
			fprintf( punch.ipDRout, "change in H ionization fm to%11.3e%11.3e\n", 
			  SaveOHIIoHI, OHIIoHI );
		}

		else if( radius.drNext == drEden )
		{
			fprintf( punch.ipDRout, 
				"change in elec den, rel chng:%11.3e, cur %g old=%g\n", 
			  dEden , OldEden , OldestEden );
		}

		else if( radius.drNext == drDest )
		{
			fprintf( punch.ipDRout, 
				"change in dest rates, atom=%s%3ld old rate %.2e new %.2e\n", 
			  chDestAtom, istage ,DestOld , DestNew );
		}

		else if( radius.drNext == drOutFrac )
		{
			fprintf( punch.ipDRout, 
				"ratio out2 tot continuua%10.2e%10.2e%7ld %4.4s %4.4s\n", 
			  Out2Tot, 
			  rfield.SumOutMax, 
			  rfield.ipSumOutMax, 
			  rfield.chLineLabel[rfield.ipSumOutMax-1], 
			  rfield.chContLabel[rfield.ipSumOutMax-1] );
			/* in above SumOutMax is anu where peak occured */
		}

		else if( radius.drNext == drdHdStep )
		{
			fprintf( punch.ipDRout, 
				"change in heating; current %10.3e delta=%10.3e\n", 
			  heat.htot, dHdStep );
		}

		else if( radius.drNext == drConPres )
		{
			fprintf( punch.ipDRout, "change in con accel\n"  );
		}

		else if( radius.drNext == drdTdStep )
		{
			fprintf( punch.ipDRout, 
				"change in temperature; current= %10.3e, dT/T= %.3f\n", 
			  phycon.te, dTdStep );
		}

		else if( radius.drNext == radius.sdrmin )
		{
			fprintf( punch.ipDRout, "sdrmin\n"  );
		}

#		if 0
		else if( radius.drNext == drH2Band )
		{
			fprintf( punch.ipDRout, "H2 Lyman Band - opacity=%10.2e\n", 
			  hmi.Molec[ipMH2]*hmi.H2Opacity );
		}
		else if( radius.drNext == drH2Abund )
		{
			fprintf( punch.ipDRout, "H2 abundance - H2/H old\t%.2e\tnew\t%.2e\n", 
			  OlderH2Abund ,
			  OldH2Abund );
		}
#		endif

		else if( radius.drNext == radius.sdrmax )
		{
			fprintf( punch.ipDRout, "sdrmax\n" 
			  );
		}

		else if( radius.drNext == drSphere )
		{
			fprintf( punch.ipDRout, "sphericity\n" 
			  );
		}

		else if( radius.drNext == dRTaue )
		{
			fprintf( punch.ipDRout, 
				"optical depth to electron scattering\n" 
			  );
		}

		else if( radius.drNext == drPressure )
		{
			fprintf( punch.ipDRout, 
				"pressure failure\n" 
			   );
		}

		else if( radius.drNext == drConvergeIoniz )
		{
			fprintf( punch.ipDRout, 
				"ionization convergence inhibited\n" 
			   );
		}

		else if( radius.drNext == drFail )
		{
			fprintf( punch.ipDRout, 
				"temperature failure.\n" 
			  );
		}

		else if( radius.drNext == drmax )
		{
			fprintf( punch.ipDRout, 
				"DRMAX; nu opc dr=%10.2e%10.2e%10.2e oscil=%2c\n", 
			  freqm, opacm, didz.drChange/
			  opacm, TorF(lgOscil) );
		}

		else if( radius.drNext == drInter )
		{
			fprintf( punch.ipDRout, 
				"cont inter nu=%10.2e opac=%10.2e\n", 
			  freqm, opacm );
		}

		else if( radius.drNext == DrGrainHeat )
		{

			fprintf( punch.ipDRout, 
				"grain heating nu=%10.2e opac=%10.2e\n", 
			  grfreqm, gropacm );
		}

		else if( radius.drNext == winddr )
		{
			fprintf( punch.ipDRout, 
				"Wind, dVeldRad=%10.3e\n", 
			   dVeldRad );
		}

		else if( radius.drNext == drFluc )
		{
			fprintf( punch.ipDRout, 
				"density fluctuations\n" 
			   );
		}

		else if( radius.drNext == GlobDr )
		{
			fprintf( punch.ipDRout, 
				"GLOB law new dr=%10.2e HDEN=%10.2e\n", 
				GlobDr,
			  dense.gas_phase[ipHYDROGEN] );
		}

		else if( radius.drNext == SpecDr )
		{
			fprintf( punch.ipDRout, 
				"special law new dr=%10.2e HDEN=%10.2e\n", 
				SpecDr,
			  dense.gas_phase[ipHYDROGEN] );
		}

		else if( radius.drNext == OldDR )
		{
			fprintf( punch.ipDRout, "old DR.\n" 
			  );
		}

		else
		{
			fprintf( punch.ipDRout, 
				" %4ld NextDR keys from insanity %10.2e\n", 
			  nzone, radius.drNext );

			fprintf( ioQQQ, 
				" %4ld NextDR keys from insanity %10.2e\n", 
			  nzone, radius.drNext );
			puts( "[Stop in nextdr]" );
			cdEXIT(EXIT_FAILURE);
		}

		/*======end active dr sets */
	}

	/* this is general code that prevents zone thickness drNext from
	 * becoming too thin, something that can happen for various bad reasons
	 * HOWEVER we do not want to do this test for certain density laws,
	 * for which very small zone thicknesses are unavoidable
	 * the special cases are:
	 * special density law,
	 * globule density law,
	 * not at carbon +-0 i front
	 * not flucuations command
	 * drMinimum was set in FirstDR to either sdrmin (set drmin) or
	 * some fraction of the initial radius - it is always set
	 * to something non-trivial.  
	 * sdrmin is only set wih the "set dr" command */

	if( ((strcmp(dense.chDenseLaw,"DLW1") != 0 && 
		strcmp(dense.chDenseLaw ,"GLOB") != 0) && 
		dense.xIonDense[ipCARBON][0]/dense.gas_phase[ipCARBON] < 0.05) && 
		(fluct.flong == 0.) &&
		/* >>chng 01 aug 11, add check against stopping on depth to go */
		radius.drNext != DepthToGo )
	{
		/* don't let dr get smaller than drMinimum, if this resets drNext
		 * then code stops with warning that zones got too thin */
		if( radius.drNext < radius.drMinimum )
		{
			radius.drNext = radius.drMinimum;
			/* leaving this at true will cause the model to stop with a warning
			 * that the zone thickness is too small */
			radius.lgDrMinUsed = TRUE;
			fprintf( ioQQQ, 
				" NextDR finds dr too small and aborting.  This is zone %ld iteration %ld\n", 
				nzone, 
				iteration);
		}
	}

	/* factor to allow for slop in floating numbers */
#	define	Z	1.0001

	/* following is to make thickness of model exact
	 * n.b., on last zone, drNext can be NEGATIVE!!
	 * DEPTH was incremented at start of zone calc in newrad,
	 * has been outer edge of zone all throughout */
	radius.drNext = MIN2(radius.drNext,(radius.router[iteration-1]-
	  radius.depth)*Z);

	/* this means outer limit exceeded */
	if( radius.drNext < 0. )
	{
		radius.lgDrNeg = TRUE;
	}
	else
	{
		radius.lgDrNeg = FALSE;
	}

	if( StopCalc.HColStop < 5e29 )
	{
		coleff = dense.gas_phase[ipHYDROGEN]*geometry.FillFac;
		radius.drNext = 
			MIN2(
			radius.drNext,
			(StopCalc.HColStop-colden.colden[ipCOLUMNDENSITY]-radius.drad*coleff)*Z/coleff);
	}

	else if( StopCalc.colpls < 5e29 )
	{
		coleff = dense.xIonDense[ipHYDROGEN][1]*geometry.FillFac;
		radius.drNext = 
			MIN2(
			radius.drNext,
			(StopCalc.colpls-colden.colden[ipCHII]- radius.drad*coleff)*Z/coleff);
	}

	else if( StopCalc.colnut < 5e29 )
	{
		coleff = dense.xIonDense[ipHYDROGEN][0]*geometry.FillFac;
		/* >>chng 97 oct 30, prevent overflow for very high U models */
		if( (StopCalc.colnut - colden.colden[ipCHI] - (float)(radius.drad)*
		  coleff) < coleff*radius.drNext )
		{
			radius.drNext = (StopCalc.colnut - colden.colden[ipCHI] - 
			  radius.drad*coleff)*Z/coleff;
		}
	}

	if( StopCalc.iptnu != rfield.nupper )
	{
		/* end optical depth has been specified */
		dt = opac.opacity_abs[StopCalc.iptnu-1]*geometry.FillFac;
		radius.drNext = 
			MIN2(radius.drNext,(StopCalc.tauend-opac.TauAbsGeo[0][StopCalc.iptnu-1]-
		  radius.drad*dt)/MAX2(SMALLFLOAT,dt) );
	}

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, " NextDR chooses next drad=%12.4e; this drad was%12.4e\n", 
		  radius.drNext, radius.drad );
	}

#	ifdef DEBUG_FUN
	fputs( " <->NextDR()\n", debug_fp );
#	endif
	return;
}

/*ContRate called by nextdr to find energy of maximum continuum-gas interaction */
static void ContRate(double *freqm, 
  double *opacm)
{
	long int i, 
	  ipHi,
	  iplow, 
	  limit;
	double FreqH, 
	  FreqSub, 
	  OpacH, 
	  OpacSub, 
	  xMaxH, 
	  xMaxSub;

#	ifdef DEBUG_FUN
	fputs( "<+>ContRate()\n", debug_fp );
#	endif

	/* 
	 * find maximum continuum interaction rate,
	 * these should be reset in following logic without exception,
	 * if they are still zero at the end we have a logical error 
	 */
	xMaxSub = 0.;
	FreqSub = 0.;
	OpacSub = 0.;

	/* do up to carbon photo edge if carbon is turned on */
	/* >>>chng 00 apr 07, add test for whether element is turned on */
	if( dense.lgElmtOn[ipCARBON] )
	{
		/* carbon is turned on, use carbon 1 edge */
		ipHi = Heavy.ipHeavy[ipCARBON][0] - 1;
	}
	else
	{
		/* carbon truned off, use hydrogen balmer continuum */
		ipHi = iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][ipH2s]-1;
	}

	for( i=1; i < ipHi; i++ )
	{
		/* this does not have grain opacity since grains totally passive
		 * at energies below CI edge */
		if( rfield.anu[i]*rfield.flux[i]/rfield.widflx[i]*(opac.opacity_abs[i] - 
		  gv.dstab[i]*dense.gas_phase[ipHYDROGEN]) > xMaxSub )
		{
			xMaxSub = rfield.anu[i]*rfield.flux[i]/rfield.widflx[i]*
			  (opac.opacity_abs[i] - gv.dstab[i]*dense.gas_phase[ipHYDROGEN]);
			FreqSub = rfield.anu[i];
			OpacSub = opac.opacity_abs[i] - gv.dstab[i]*dense.gas_phase[ipHYDROGEN];
		}
	}

	/* not every continuum extends beyond C edge-this whole loop can add to zero
	 * use total opacity here
	 * test is to put in fir continuum if free free heating is important */
	if( CoolHeavy.brems_heat_total/heat.htot > 0.05 )
	{
		/* this is index for energy where cloud free free optical depth is unity,
		 * is zero if no freq are opt thin */
		iplow = MAX2(1 , rfield.ipEnergyBremsThin );
	}
	else
	{
		/* >>>chng 00 apr 07, from Heavy.ipHeavy[0][5] to ipHi defined above, since
		 * would crash if element not defined */
		iplow = ipHi;
	}

	/* this energy range from carbon edge to hydrogen edge */
	limit = MIN2(rfield.nflux,iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][ipH1s]-1);
	for( i=iplow; i < limit; i++ )
	{
		if( rfield.anu[i]*rfield.flux[i]/rfield.widflx[i]*(opac.opacity_abs[i] - 
		  gv.dstab[i]*dense.gas_phase[ipHYDROGEN]) > xMaxSub )
		{
			xMaxSub = rfield.anu[i]*rfield.flux[i]/rfield.widflx[i]*
			  (opac.opacity_abs[i] - gv.dstab[i]*dense.gas_phase[ipHYDROGEN]);
			FreqSub = rfield.anu[i];
			OpacSub = opac.opacity_abs[i] - gv.dstab[i]*dense.gas_phase[ipHYDROGEN];
		}
	}

	/* variables to check continuum interactions over lyman continuum */
	xMaxH = 0.;
	OpacH = 0.;
	FreqH = 0.;

	/* not every continuum extends beyond 1 Ryd-this whole loop can add to zero */
	for( i=iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][ipH1s]-1; i < rfield.nflux; i++ )
	{
		if( rfield.anu[i]*rfield.flux[i]/rfield.widflx[i]*(opac.opacity_abs[i] - 
		  gv.dstab[i]*dense.gas_phase[ipHYDROGEN]) > xMaxH )
		{
			/* xMaxH = anu(i)*flux(i)/widflx(i)*opac(i) */
			xMaxH = rfield.anu[i]*rfield.flux[i]/rfield.widflx[i]*
			  (opac.opacity_abs[i] - gv.dstab[i]*dense.gas_phase[ipHYDROGEN]);
			FreqH = rfield.anu[i];
			OpacH = opac.opacity_abs[i] - gv.dstab[i]*dense.gas_phase[ipHYDROGEN];
		}
	}


	/* use lyman continuum if its opacity is larger than non-h ion */
	if( xMaxSub < 1e-30 )
	{
		/* this happens for laser source - use Lyman continuum */
		*opacm = OpacH;
		*freqm = FreqH;
	}
	else if( OpacH > OpacSub && xMaxH/xMaxSub > 1e-10 )
	{
		/* use Lyman continuum */
		*opacm = OpacH;
		*freqm = FreqH;
	}
	else
	{
		/* not much rate in the lyman continuum, stick with low energy */
		*opacm = OpacSub;
		*freqm = FreqSub;
	}

	if( xMaxH > xMaxSub )
	{
	}
	else
	{
		*opacm = OpacSub;
		*freqm = FreqSub;
	}

	{
		/* following should be set true to print contributors */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			fprintf(ioQQQ,"conratedebug \t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
			xMaxSub,FreqSub,OpacSub,
			xMaxH,FreqH ,OpacH,*freqm,*opacm
			);
	
		}
	}

	/* these were set to zero at start, and must have been reset if one of the
	 * two loops.  Logic error if still zero. */
	ASSERT( *opacm > 0. && *freqm > 0. );

#	ifdef DEBUG_FUN
	fputs( " <->ContRate()\n", debug_fp );
#	endif
	return;
}

/*GrainRateDr called by nextdr to find grain heating rate dr */
static void GrainRateDr(double *grfreqm, 
  double *gropacm)
{
	long int i, 
	  iplow;
	double xMax;

#	ifdef DEBUG_FUN
	fputs( "<+>GrainRateDr()\n", debug_fp );
#	endif

	/* in all following changed from anu2 to anu  july 25 95
	 *
	 * find maximum continuum interaction rate */

	/* not every continuum extends beyond C edge-this whole loop can add to zero
	 * use total opacity here
	 * test is to put in fir continuum if free free heating is important */
	if( CoolHeavy.brems_heat_total/heat.htot > 0.05 )
	{
		/* this is pointer to energy where cloud free free optical depth is unity,
		 * is zero if no freq are opt thin */
		iplow = MAX2(1 , rfield.ipEnergyBremsThin );
	}
	else
	{
		/* do up to carbon photo edge if carbon is turned on */
		/* >>>chng 00 apr 07, add test for whether element is turned on */
		if( dense.lgElmtOn[ipCARBON] )
		{
			/* carbon is turned on, use carbon 1 edge */
			iplow = Heavy.ipHeavy[ipCARBON][0];
		}
		else
		{
			/* carbon truned off, use hydrogen balmer continuum */
			iplow = iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][ipH2s];
		}

	}

	xMax = -1.;
	/* integrate up to H edge */
	for( i=iplow-1; i < Heavy.ipHeavy[ipHYDROGEN][0]; i++ )
	{
		if( rfield.anu[i]*rfield.flux[i]/rfield.widflx[i]*opac.opacity_abs[i] > xMax )
		{
			xMax = rfield.anu[i]*rfield.flux[i]/rfield.widflx[i]*
			  opac.opacity_abs[i];
			*grfreqm = rfield.anu[i];
			*gropacm = opac.opacity_abs[i];
		}
	}
	/* integrate up to heii edge if he is turned on,
	 * this logic will not make sense if grains on but he off, which in itself makes no sense*/
	if( dense.lgElmtOn[ipHELIUM] )
	{
		for( i=Heavy.ipHeavy[ipHYDROGEN][0]-1; i < Heavy.ipHeavy[ipHELIUM][1]; i++ )
		{
			if( rfield.anu[i]*rfield.flux[i]/rfield.widflx[i]*opac.opacity_abs[i] > xMax )
			{
				xMax = rfield.anu[i]*rfield.flux[i]/rfield.widflx[i]*
				  opac.opacity_abs[i];
				*grfreqm = rfield.anu[i];
				*gropacm = opac.opacity_abs[i];
			}
		}
	}

	/* possible that there is NO ionizing radiation, in extreme cases,
	 * if so then xMax will still be negative */
	if( xMax <= 0. )
	{
		*gropacm = 0.;
		*grfreqm = 0.;
	}

#	ifdef DEBUG_FUN
	fputs( " <->GrainRateDr()\n", debug_fp );
#	endif
	return;
}

/*ChkRate called by nextdr to check how rates of destruction of various species changes */
static void ChkRate(
	  /* element number on C scale */
	  long int nelem, 
	  /* change in destruction rate */
	  double *dDestRate, 
	  /* old and new destruction rates */
	  double *DestRateOld,
	  double *DestRateNew,
	  /* stage of ionization on the physical scale */
	  long int *istage)
{
	long int i;

	double average, 
	  dDest;

	static double OldDest[LIMELM][LIMELM];

#	ifdef DEBUG_FUN
	fputs( "<+>ChkRate()\n", debug_fp );
#	endif

	/* return if this element is not turned on */
	if( !dense.lgElmtOn[nelem] )
	{
		*dDestRate = 1e-3;
		*DestRateOld = 1e-3;
		*DestRateNew = 1e-3;
		*istage = 0;
		
#		ifdef DEBUG_FUN
		fputs( " <->ChkRate()\n", debug_fp );
#		endif
		return;
	}

	/* for first zone, and during search, we will do nothing but
	 * still must return finite numbers */
	*istage = 1;
	*dDestRate = 0.;
	*DestRateOld = 0.;
	*DestRateNew = 0.;
	*dDestRate = 0.;

	if( nzone <= 1 )
	{
		for( i=0; i < nelem+1; i++ )
		{
			OldDest[nelem][i] = ionrec.TotIonizRate[nelem][i];
		}
	}

	else if( dense.xIonDense[nelem][0]/dense.gas_phase[nelem] <  0.9 )
	{
		/* do not use this method if everything is atomic */
		for( i=0; i < (nelem); i++ )
		{
			/* last check below, .5 chosen so that do not key off
			 * predominantly neutral species where self-opacity
			 * could cause oscillations */
			if( ((dense.xIonDense[nelem][i]/dense.gas_phase[nelem] > 0.01 && 
				dense.xIonDense[nelem][i]/dense.gas_phase[nelem] < 0.9) && 
				dense.xIonDense[nelem][i+1]/dense.gas_phase[nelem] > .05) && 
				OldDest[nelem][i] > 0. )
			{
				/* last check on old dest in case just lowered ionization
				 * stage, so no history */
				/* check that rate is positive */
				if( ionrec.TotIonizRate[nelem][i] <= 0. )
				{
					fprintf( ioQQQ, " ChkRate gets insane destruction rate for ion%4ld%4ld%10.2e\n", 
					  nelem+1, i, ionrec.TotIonizRate[nelem][i] );
					puts( "[Stop in chkrate]" );
					cdEXIT(EXIT_FAILURE);
				}

				/* do not consider unless of middling ionization, and
				 * rate is going down (to prevent dr osciallating)
				 * no absolute value in following since do not want to
				 * consider cases where ionization rate increases */
				average = (OldDest[nelem][i] + ionrec.TotIonizRate[nelem][i])* 0.5;

				dDest = (OldDest[nelem][i] - ionrec.TotIonizRate[nelem][i])/ average;
				/* ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ term + if rate going down */

				if( *dDestRate < dDest )
				{
					/* biggest rate so far, remember change in rates and ionization stage */
					*dDestRate = dDest;
					*istage = i+1;
					*DestRateOld = OldDest[nelem][i];
					*DestRateNew = ionrec.TotIonizRate[nelem][i];
				}

			}
			OldDest[nelem][i] = ionrec.TotIonizRate[nelem][i];
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->ChkRate()\n", debug_fp );
#	endif
	return;
}
/*lint +e777 floating tests for equality are ok and frequent here */
