/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*ParseInterp parse parameters on interpolate command */
#include "cddefines.h"
#include "called.h"
#include "bndsok.h"
#include "rfield.h"
#include "egray.h"
#include "trace.h"
#include "bit32.h"
#include "input.h"
#include "readar.h"
#include "parse.h"

void ParseInterp(char *chCard, 
  int *lgEOF)
{
	char chLab4[5];
	int lgDONE, 
	  lgEOL;
	long int i, 
		k ,
	  npairs;
	double cmax, 
	  cmin, 
	  fac;

#	ifdef DEBUG_FUN
	fputs( "<+>ParseInterp()\n", debug_fp );
#	endif

	/* 
	 * this sub reads in the "interpolate" command, and has
	 * logic for the "continue" lines as well.
	 * OUTPUT:
	 * TNU is vector of eneriges where the grid is defined
	 * TSLOP initially is vector of log fnu at each freq
	 * converted into slopes here too
	 */

	if( rfield.nspec >= LIMSPC-1 )
	{
		fprintf( ioQQQ, " Too many spectra entered.  Increase LIMSPC\n" );
		puts( "[Stop in ParseInterp]" );
		cdEXIT(EXIT_FAILURE);
	}

	strcpy( rfield.chSpType[rfield.nspec], "INTER" );

	/* read all of the numbers on a line */
	npairs = 0;

	/* this is flag saying that all numbers are in */
	lgDONE = FALSE;

	/* this flag says we hit end of command stream */
	*lgEOF = FALSE;
	while( !lgDONE && !*lgEOF )
	{
		i = 5;
		lgEOL = FALSE;

		/* keep scanning numbers til we hit eol for current line image */
		/* >>chng 01 aug 10, add check on npairs not exceeding NC_ELL as per 
		 * Ilse van Bemmel bug report */
		/*while( !lgEOL && npairs<rfield.nupper )*/
		while( !lgEOL && npairs<NCELL )
		{
			rfield.tNuRyd[npairs][rfield.nspec] = 
				FFmtRead(chCard ,&i,INPUT_LINE_LENGTH,&lgEOL);
			rfield.tFluxLog[npairs][rfield.nspec] = 
				FFmtRead(chCard ,&i,INPUT_LINE_LENGTH,&lgEOL);
			++npairs;
		}
		/* check if we fell through due to too many points, did not hit EOL */
		if( !lgEOL )
		{
			fprintf( ioQQQ, "Too many continuum points were entered.\n" );
			fprintf( ioQQQ, 
				"The current logic limits the number of possible points to the value of NCELL, which is %i.\n",NCELL );
			fprintf( ioQQQ, 
				"Increase the value of NCELL in rfield.h.\nSorry.\n" );
			puts( "[Stop in ParseInterp]" );
			cdEXIT(EXIT_FAILURE);
		}

		/* added too many above, so back off */
		--npairs;

		/* read a new line, checking for EOF */
		readar(chCard,lgEOF);

		/* option to ignore all lines starting with #, *, or %. */
		while( !*lgEOF && ((chCard[0] == '#' || chCard[0] == '*') || chCard[0] == '%') )
		{
			readar(chCard,lgEOF);
		}

		/* get cap'd first four char of chCard */
		cap4( chLab4 , chCard );

		/* print the line, but only if it is a continue line */
		if( called.lgTalk && (strncmp(chLab4,"CONT",4)==0) )
		{
			fprintf( ioQQQ, "                       * ");
			/* next logic will print command plus spaces to right justified * */
			k=0;
			while( chCard[k]!='\0' )
			{
				fprintf(ioQQQ,"%c",chCard[k]);
				++k;
			}
			while( k<80 )
			{
				fprintf(ioQQQ,"%c",' ');
				++k;
			}
			fprintf( ioQQQ,"*\n"); 
		}

		/* now convert entire line to caps */
		caps(chCard);

		/* is this a continue line? */
		if( strncmp(chCard,"CONT",4) != 0 )
		{
			/* we have a line but next command, not continue */
			lgDONE = TRUE;
		}

		/* this is another way to hit end of input stream - blank lines */
		if( chCard[0] == ' ' )
			*lgEOF = TRUE;
	}

	/* if valid next line, backup one line */
	if( lgDONE )
	{
		input.nRead -= input.iReadWay;
	}

	/* done reading all of the possible lines */
	--npairs;
	if( npairs < 1 )
	{
		fprintf( ioQQQ, "There must be at least 2 pairs to interpolate,\nSorry\n" );
		puts( "[Stop in ParseInterp]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* >>chng 02 apr 19, from > to >=, test did not trigger due to overrun protection
	 * in main loop */
	else if( npairs >= NCELL - 2 )
	{
		fprintf( ioQQQ, " Too many continuum points entered.\n" );
		fprintf( ioQQQ, 
			"The current logic limits the number of possible points to the value of NCELL, which is %i.\nSorry.\n",NCELL );
		fprintf( ioQQQ, " Increase NCELL (in cddefines.h) to more than the number of continuum points.\n" );
		puts( "[Stop in ParseInterp]" );
		cdEXIT(EXIT_FAILURE);
	}

	if( rfield.tNuRyd[0][rfield.nspec] == 0. )
	{
		/* special case - if first energy is zero then it is low energy */
		if( rfield.tNuRyd[1][rfield.nspec] > 0. )
		{
			/* second energy positive, assume linear Ryd */
			rfield.tNuRyd[0][rfield.nspec] = rfield.emm;
		}
		else if( rfield.tNuRyd[1][rfield.nspec] < 0. )
		{
			/* second energy negative, assume log of Ryd */
			rfield.tNuRyd[0][rfield.nspec] = log10(rfield.emm);
		}
		else
		{
			/* second energy zero, not allowed */
			fprintf( ioQQQ, 
				"An energy of zero was entered for element%3ld in INTERPOLATE and is not allowed.\nSorry\n", 
			  rfield.nspec );
			puts( "[Stop in ParseInterp]" );
			cdEXIT(EXIT_FAILURE);
		}
	}

	/* convert from log(Hz) to Ryd if first nu>5 */
	if( rfield.tNuRyd[0][rfield.nspec] >= 5. )
	{
		for( i=0; i < (npairs + 1); i++ )
		{
			rfield.tNuRyd[i][rfield.nspec] = 
				pow(10.,rfield.tNuRyd[i][rfield.nspec] - 15.517);
		}
	}

	else if( rfield.tNuRyd[0][rfield.nspec] < 0. )
	{
		/* energies entered as logs */
		for( i=0; i < (npairs + 1); i++ )
		{
			rfield.tNuRyd[i][rfield.nspec] = pow(10.,rfield.tNuRyd[i][rfield.nspec]);
		}
	}

	else
	{
		/* numbers are linear Rydbergs */
		for( i=0; i < (npairs + 1); i++ )
		{
			if( rfield.tNuRyd[i][rfield.nspec] == 0. )
			{
				fprintf( ioQQQ, "An energy of zero was entered for element%3ld in INTERPOLATE and is not allowed.\nSorry\n", 
				  i );
				puts( "[Stop in ParseInterp]" );
				cdEXIT(EXIT_FAILURE);
			}
		}
	}

	/* option to print debugging file then exit */
	{
		/* following should be set true to print information */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC  )
		{
			for( i=0; i < npairs; i++ )
			{
				fprintf(ioQQQ,"%.4e\t%.3e\n",
					rfield.tNuRyd[i][rfield.nspec],
					pow(10.,rfield.tFluxLog[i][rfield.nspec])  * rfield.tNuRyd[i][rfield.nspec]);
			}
			cdEXIT(EXIT_SUCCESS);
		}
	}

	for( i=0; i < npairs; i++ )
	{
		/* check that frequencies are monotonically increasing */
		if( rfield.tNuRyd[i+1][rfield.nspec] <= rfield.tNuRyd[i][rfield.nspec] )
		{
			fprintf( ioQQQ, "The energies MUST be in increasing order.  Energy #%3ld=%10.2e Ryd was greater than or equal to the next one.\nSorry.\n", 
			  i, rfield.tNuRyd[i][rfield.nspec] );
			puts( "[Stop in ParseInterp]" );
			cdEXIT(EXIT_FAILURE);
		}

		/* TFAC is energy, and TSLOP is slope in f_nu; not photons */
		/*rfield.tFluxLog[i][rfield.nspec] = rfield.tslop[i][rfield.nspec];*/

		/*rfield.tslop[i][rfield.nspec] = (float)((rfield.tslop[i+1][rfield.nspec] - 
		  rfield.tslop[i][rfield.nspec])/log10(rfield.tNuRyd[i+1][rfield.nspec]/
		  rfield.tNuRyd[i][rfield.nspec]));*/
		rfield.tslop[i][rfield.nspec] = ((rfield.tFluxLog[i+1][rfield.nspec] - 
		  rfield.tFluxLog[i][rfield.nspec])/log10(rfield.tNuRyd[i+1][rfield.nspec]/
		  rfield.tNuRyd[i][rfield.nspec]));
	}

	if( npairs + 2 < rfield.nupper )
	{
		/* zero out remainder of array */
		for( i=npairs + 1; i < rfield.nupper; i++ )
		{
			rfield.tNuRyd[i][rfield.nspec] = 0.;
		}
	}

	/* now check that array is defined over all energies */
	if( rfield.tNuRyd[0][rfield.nspec] > rfield.emm )
	{
		/* not defined over low energy part of array */
		fprintf( ioQQQ, 
			" Warning:  the input continuum was not defined over the entire energy range. Some energies are set to zero.\n" );
		fprintf( ioQQQ, 
			" >>>>>>>>>>You are making a BIG mistake.\n" );
	}

	/* check on IR */
	if( rfield.tNuRyd[0][rfield.nspec] > rfield.emm )
	{
		bndsok.lgMMok = FALSE;
	}

	if( rfield.tNuRyd[0][rfield.nspec] > 1/36. )
	{
		bndsok.lgHPhtOK = FALSE;
	}

	/* gamma ray, EGAMRY is roughly 100MeV */
	if( rfield.tNuRyd[npairs][rfield.nspec] < rfield.egamry )
	{
		bndsok.lgGamrOK = FALSE;
	}

	/* EnerGammaRay is roughly 100keV; high is gamma ray */
	if( rfield.tNuRyd[npairs][rfield.nspec] < egray.EnerGammaRay )
	{
		bndsok.lgXRayOK = FALSE;
	}

	/* find min and max of continuum */
	cmax = -38.;
	cmin = 28;

	/* tfac can be very large or small */
	/* >>chng 01 jul 11, from <npairs to <=npairs, [nparis] is highest point in table */
	for( i=0; i <= npairs; i++ )
	{
		cmax = MAX2(cmax,rfield.tFluxLog[i][rfield.nspec]);
		cmin = MIN2(cmin,rfield.tFluxLog[i][rfield.nspec]);
	}

	/* check on dynamic range of input continuum */
	if( cmax - cmin > 74. && bit32.lgBit32 )
	{
		fprintf( ioQQQ, "The dynamic range of the specified continuumis too large for a 32-bit computer.\nSorry.\n" );
		puts( "[Stop in ParseInterp]" );
		cdEXIT(EXIT_FAILURE);
	}

	if( cmin < -37. )
	{
		fac = -cmin - 37.;
		/* >>chng 01 jul 11, from <npairs to <=npairs, [nparis] is highest point in table */
		for( i=0; i <= npairs; i++ )
		{
			rfield.tFluxLog[i][rfield.nspec] += fac;
		}
	}

	else if( cmax > 37. )
	{
		fac = cmax - 37.;
		/* >>chng 01 jul 11, from <npairs to <=npairs, [nparis] is highest point in table */
		for( i=0; i <= npairs; i++ )
		{
			rfield.tFluxLog[i][rfield.nspec] -= fac;
		}
	}

	/* option to print out results at this stage - "trace continuum" */
	if( trace.lgConBug && trace.lgTrace )
	{
		fprintf( ioQQQ, " Table for this continuum; TNU, TFAC, TSLOP\n" );
		for( i=0; i < (npairs + 1); i++ )
		{
			fprintf( ioQQQ, "%12.4e %12.4e %12.4e\n", rfield.tNuRyd[i][rfield.nspec], 
			  rfield.tFluxLog[i][rfield.nspec], rfield.tslop[i][rfield.nspec] );
		}
	}

	/* renormalize continuum so that flux we will interpolate upon passes through unity
	 * at near 1 Ryd.  but first we must find 1 Ryd in the array.
	 * find 1 ryd, npairs is one less than number of continuum pairs */
	i = 0;
	while( rfield.tNuRyd[i][rfield.nspec] <= 1. && i < npairs-1 )
	{
		i += 1;
	}
	/* i is now the table point where tnu[i-1] is <= 1 ryd,
	 * and tnu[i] > 1 ryd */

	/* following is impossible but sanity check */
	if( i < 1 )
	{
		fprintf( ioQQQ, "ParseInput finds insane i after tnu loop\n");
		ShowMe();
		puts( "[Stop in ParseInterp]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* do the renormalization so 1 ryd is about unity */
	fac = rfield.tFluxLog[i-1][rfield.nspec];
	for( i=0; i <= npairs; i++ )
	{
		rfield.tFluxLog[i][rfield.nspec] -= fac;
	}

	/* now increment number of continua */
	++rfield.nspec;

#	ifdef DEBUG_FUN
	fputs( " <->ParseInterp()\n", debug_fp );
#	endif
	return;
}
