/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*RTDiffuse evaluate local diffuse emission for this zone,
 * fill in ConEmitLocal and ThrowOut arrays with diffuse emission,
 * ots rates for this zone were set in RT_OTS,
 * called by Cloudy, this routine adds energy to the outward beam */
#include "cddefines.h"
#include "physconst.h"
#include "taulines.h"
#include "qheat.h"
#include "grainvar.h"
#include "lines_service.h"
#include "iso.h"
#include "dense.h"
#include "opacity.h"
#include "trace.h"
#include "rfield.h"
#include "phycon.h"
#include "hmi.h"
#include "radius.h"
#include "outer.h"
#include "ionrange.h"
#include "heavy.h"
#include "fekems.h"
#include "plasnu.h"
#include "atomfeii.h"
#include "h2.h"
#include "rt.h"
/* only used during debug print */
#include "ipoint.h"

#define	TwoS	(1+ipISO)

void RTDiffuse(void)
{
	/* If this flag is set to TRUE, a table of 2 photon emission coefficients
	 * comparable to Brown and Mathews (1971) will be printed.	*/
	static long lgPrt2PhtEmisCoef = FALSE;

	/*int lgHeTypeSave;*/
	long int i, 
	  ip, 
	  ipISO,
	  ipHi, 
	  ipLo, 
	  ipla,
	  nelem,
	  ion,
	  limit, 
	  n;

	long ipLimit,maxi=0;
	double EnergyLimit;
	float Aul;

	double EdenAbund, 
	  difflya, 
	  esc, 
	  ots, 
	  xInWrd,
	  arg, 
	  cor, 
	  fac, 
	  factor, 
	  gamma, 
	  gion, 
	  gn, 
	  photon, 
	  sum,
	  Sum1level,
	  SumCaseB;
	  /* following usually not used for anything, and lint will say so,
	   * but this info is important diagnostic
	  SumTotalBrems */

	float Dilution , fach;

#	ifdef DEBUG_FUN
	fputs( "<+>RTDiffuse()\n", debug_fp );
#	endif

	/* this routine evaluates the local diffuse fields
	 * it fills in the vector ConEmitLocal */
#	if 0
	for( i=0; i < rfield.nupper; i++ )
	{
		rfield.ThrowOut[i] = 0.;
		rfield.ConEmitLocal[i] = 0.;
		rfield.ConOTS_local_photons[i] = 0.;
		rfield.TotDiff2Pht[i] = 0.;
	}
#	endif
	memset(rfield.ThrowOut             , 0 , rfield.nupper*sizeof(float) );
	memset(rfield.ConEmitLocal         , 0 , rfield.nupper*sizeof(float) );
	memset(rfield.ConOTS_local_photons , 0 , rfield.nupper*sizeof(float) );
	memset(rfield.TotDiff2Pht          , 0 , rfield.nupper*sizeof(float) );

	/* loop over hydrogenic and helium-like species of all elements 
	 * to add all recombination continua and all lines*/
	for( ipISO=ipH_LIKE; ipISO<NISO; ++ipISO )
	{
		/* >>chng 01 sep 23, rewrote for iso sequences */
		for( nelem=ipISO; nelem < LIMELM; nelem++ )
		{
			/* this will be the sum of recombinations to all excited levels */
			SumCaseB = 0.;

			/* the product of the densities of the parent ion and electrons */
			EdenAbund = dense.eden*dense.xIonDense[nelem][nelem+2-ipISO-1];

			/* recombination continua for all iso seq - 
			 * if they exist */
			if( IonRange.IonHigh[nelem] >= nelem + 1 - ipISO  )
			{
				/* loop over all levels to include recombination diffuse continua */
				limit = iso.numLevels[ipISO][nelem];

				/* pick highest energy continuum point that opacities extend to 
				 * for ground continuum, go up to highest defined boltzmann factor,
				 * at bottom of loop will be reset to ground state photo edge */
				/* >>chng 03 feb 03, upper limit for ground state from ipMaxBolt
				 * to nflux */
				ipHi = rfield.nflux;
				for( n=0; n < limit; n++ )
				{
					Sum1level = 0.;

					/* >>chng 02 nov 20 - pull the plug on old he treatment */
					/* the number is (2 pi me k/h^2) ^ -3/2 * 8 pi/c^2 / ge - it includes
					 * the stat weight of the free electron in the demominator */
					gamma = 2.0618684e11*iso.stat[ipISO][nelem][n]/iso.stat_ion[ipISO]/phycon.te/phycon.sqrte;

					/* following is boltz fac for second cell, check not zero,*/
					/*bolt2 = rfield.anu[iso.ipIsoLevNIonCon[ipISO][nelem][n]]/phycon.te_ryd;*/

					/* only escaping part of recombinations are thrown into ThrowOut here,
					 * added to ConInterOut.  The lost photons are never put back in.*/
#					if 0
					if( iso.ConBoltz[ipISO][nelem][n] > 0. &&
						rfield.ContBoltz[iso.ipIsoLevNIonCon[ipISO][nelem][n]-1]>SMALLFLOAT )
					{
						for( i=iso.ipIsoLevNIonCon[ipISO][nelem][n]-1; i < ipHi; i++ )
						{
							/* flux is in photons per sec per ryd
							 * ContBoltz is ratio of Boltz fac for each freq */
							photon = gamma*rfield.ContBoltz[i]/iso.ConBoltz[ipISO][nelem][n]*
							  rfield.widflx[i]*rfield.anu2[i]*
							  opac.OpacStack[i-iso.ipIsoLevNIonCon[ipISO][nelem][n]+iso.ipOpac[ipISO][nelem][n]];
							Sum1level += photon;
							ASSERT( photon >= 0. );
							rfield.ConEmitLocal[i] += (float)(photon*EdenAbund);
							rfield.ThrowOut[i] += 
								(float)(photon*EdenAbund*iso.RadRecomb[ipISO][nelem][n][ipRecEsc]/**/);
						}
					}
					/* >>chng 03 feb 03, test on bolt2 allowed cases to fall through
					 * where recomb continuum was not evaluated */
					else
					/*else if( bolt2 < 3. )*/
					{
					}
#					endif

					/*   boltzmann factor is zero, but still emission within continuum */
					for( i=iso.ipIsoLevNIonCon[ipISO][nelem][n]-1; i < ipHi; i++ )
					{
						/* dwid used to adjust where within WIDFLX exp is evaluated -
						 * weighted to lower energy due to exp(-energy/T) */
						double dwid = 0.2;
						double arg = (rfield.anu[i]-iso.xIsoLevNIonRyd[ipISO][nelem][n]+
							rfield.widflx[i]*dwid)/phycon.te_ryd;
						arg = MAX2(0.,arg);
						/*arg = MAX2(0.,1.57890e5*
							(rfield.anu[i]-iso.xIsoLevNIonRyd[ipISO][nelem][n]+
							rfield.widflx[i]*dwid)/phycon.te);*/
						/* this should probably be the same number that appears in tfidle and sexp	*/
						if( arg > 84. ) 
							break;
						/* flux is in photons per sec per ryd */
						photon = gamma*sexp(arg)*rfield.widflx[i]*
							opac.OpacStack[i-iso.ipIsoLevNIonCon[ipISO][nelem][n]+iso.ipOpac[ipISO][nelem][n]]*
							rfield.anu2[i];
						ASSERT( photon >= 0. );
						Sum1level += photon;
						rfield.ConEmitLocal[i] += (float)(photon*EdenAbund);
						rfield.ThrowOut[i] += 
							(float)(photon*EdenAbund*iso.RadRecomb[ipISO][nelem][n][ipRecEsc]);
					}
					/* reset this counter at 2s, lowest excited state, will use below
					 * to confirm case B sum */
					if( n > 0 )
					{
						/* SumCaseB will be sum to all excited */
						SumCaseB += Sum1level;
					}

					/* for excited states go either up to ground continuum edge 
					 * or highest defined Boltzmann factor */
					/* >>chng 00 dec 15, form ipHn to ipHn-1, since ipHn is on F scale */
					/* >>chng 03 feb 03, upper limit for ground state from ipMaxBolt
					 * to nflux */
					ipHi = iso.ipIsoLevNIonCon[ipISO][nelem][0]-1;
				}
				/* code for NHPLPHOT < 400 is at end of routine */
				/* following is check on self-consistency */
				iso.CaseBCheck[ipISO][nelem] = MAX2(iso.CaseBCheck[ipISO][nelem],
				  (float)(SumCaseB/iso.RadRec_caseB[ipISO][nelem]));

				/* >>chng 02 nov 20 - pull the plug on old he treatment */
				/* >>chng 02 may 16, moved this here, from loops below */
				/* >>chng 02 nov 18, do not include very highest level since funky */
				for( ipLo=0; ipLo < (iso.numLevels[ipISO][nelem] - 2); ipLo++ )
				{
					for( ipHi=ipLo+1 ; ipHi < iso.numLevels[ipISO][nelem] - 1; ipHi++ )
					{
						/* must not include 2s-1s two photon in this since is actually a continuum,
						* or the 2p-2s transition which is off our energy grid */
						if( EmisLines[ipISO][nelem][ipHi][ipLo].ipCont < 1 )
							continue;

						/* pointer to line energy in continuum array,
						* in most all cases the 2p-2s transition will have an energy very far
						* off our energy grid.  These were assigned an ipCont of 1, 0 here,
						* and the A to such a small value that these lines should not appear 
						* in the resulting spectrum */
						ip = EmisLines[ipISO][nelem][ipHi][ipLo].ipCont-1;

						/* number of photons in the line has not been defined up til now,
						* do so now.  this is redone in lines.  */
						EmisLines[ipISO][nelem][ipHi][ipLo].phots = 
							EmisLines[ipISO][nelem][ipHi][ipLo].Aul*
							EmisLines[ipISO][nelem][ipHi][ipLo].PopHi*
							EmisLines[ipISO][nelem][ipHi][ipLo].Pesc*
							dense.xIonDense[nelem][nelem+2-ipISO-1];

						/* inward fraction of line */
						xInWrd = EmisLines[ipISO][nelem][ipHi][ipLo].phots*
							EmisLines[ipISO][nelem][ipHi][ipLo].FracInwd;

						/* reflected part of line */
						rfield.reflin[ip] += (float)(xInWrd*radius.BeamInIn);

						/* inward beam that goes out since sphere set */
						/* in all this the ColOvTot term has been commented out,
						* since this is not meaningful for something like the hydrogen atom,
						* where most forms by recombination */
						rfield.outlin[ip] += (float)(xInWrd*radius.BeamInOut*opac.tmn[ip]/*
						EmisLines[ipISO][nelem][ipHi][ipLo].ColOvTot*/);

						/* outward part of line */
						rfield.outlin[ip] += 
							(float)(EmisLines[ipISO][nelem][ipHi][ipLo].phots*
							(1. - EmisLines[ipISO][nelem][ipHi][ipLo].FracInwd)*
							radius.BeamOutOut*opac.tmn[ip]/*
							EmisLines[ipISO][nelem][ipHi][ipLo].ColOvTot*/);
					}
				}
			}
		}
	}

	/* at this stage the diffuse arrays have all the isoelectronic sequence recombination continua */

	/*Iso treatment of two photon emission.  */
	for( ipISO = ipH_LIKE; ipISO < NISO; ipISO++ )
	{
		for( nelem=ipISO; nelem < LIMELM; nelem++ )
		{
			/* if an element is turned off then IonHigh is -1 */
			if( (IonRange.IonHigh[nelem] >= nelem + 1 - ipISO)  )
			{
				/* NISO could in the future be increased, but we want this assert to blow
				 * so that it is understood this may not be correct for other iso sequences	*/
				ASSERT( ipISO <= ipHE_LIKE );
					
				/* upper limit to 2-phot is energy of 2s to ground */
				ipLimit = iso.ipTwoPhoE[ipISO][nelem];
				EnergyLimit = EmisLines[ipISO][nelem][TwoS][0].EnergyWN/RYD_INF;
				Aul = EmisLines[ipISO][nelem][TwoS][0].Aul;
				
				/* this could fail when pops very low and Pop2Ion is zero */
				ASSERT( ipLimit>0 && EnergyLimit>0. && Aul>0.f /*&& EdenAbund>0.*/ );
				
				sum = 0.;
			
				maxi = MAX2( maxi, ipLimit );

				/* iplimit is index for 2s - 1s energy	*/
				for( i=0; i<ipLimit; i++ )
				{
					float rate_ind;

					/* We do not assert rfield.anu[i]<=EnergyLimit because the maximum 
					 * index could be set to point to the next highest bin.	*/
					ASSERT( rfield.anu[i]/EnergyLimit < 1.01 || rfield.anu[i-1]<EnergyLimit);

					/* iso.As2nu[ipISO][nelem][i] is transition probability A per bin
					 * So sum is the total transition probability - this sum should
					 * add up to the A two photon */
					sum += iso.As2nu[ipISO][nelem][i];

					/* iso.Pop2Ion[ipISO][nelem][TwoS] is dimensionless and represents the population 
					 * of the TwoS level relative to that of the ion.  dense.xIonDense[nelem][nelem+1-ipISO]
					 * is the density of the current ion (cm^-3).  The factor of 2 is for two photons per 
					 * transition. Thus fach has dimension photons cm-3 s-1.	*/
					fach = 2.f*iso.Pop2Ion[ipISO][nelem][TwoS]*dense.xIonDense[nelem][nelem+1-ipISO]
						*iso.As2nu[ipISO][nelem][i];

					/* >>chng 02 aug 14, add induced two photon emission */
					/* product of occupation numbers for induced emission */
					rate_ind = (1.f + rfield.SummedOcc[i]) *
						(1.f+rfield.SummedOcc[iso.ipSym2nu[ipISO][nelem][i]-1]);
					/* this is the total rate */
					fach *= rate_ind;

					rfield.ConEmitLocal[i] += fach;
					rfield.TotDiff2Pht[i] += fach;

					/* this is escaping part, as determined from optical depth to illuminated face */
					rfield.ThrowOut[i] += fach*opac.ExpmTau[i];
					/* save locally destroyed ots two-photon continuum */
					rfield.ConOTS_local_photons[i] += fach*(1.f - opac.ExpmTau[i]);
				}

				/* a sanity check on the code, see Spitzer and Greenstein (1951), eqn 4.	*/
				/* >>refer	HI	2nu	Spitzer, L., & Greenstein, J., 1951, ApJ, 114, 407 */
				ASSERT( fabs( 1.f - (float)sum/Aul ) < 0.01f );
			}

			/* Print hydrogen and helium two-photon emission coefficients?	*/
			if( lgPrt2PhtEmisCoef )
			{
				long yTimes20;
				double y, E2nu;

				lgPrt2PhtEmisCoef = FALSE;

				fprintf( ioQQQ, "\ny\tGammaNot(2q)\n");

				for( yTimes20=1; yTimes20<=10; yTimes20++ )
				{
					y = yTimes20/20.;

					fprintf( ioQQQ, "%.3e\t", (double)y );	

					E2nu = EmisLines[0][0][1][0].EnergyWN/RYD_INF;
					i = ipoint(y*E2nu);
					fprintf( ioQQQ, "%.3e\t", 
						8./3.*HPLANCK*iso.Pop2Ion[0][0][1]/dense.eden
						*y*iso.As2nu[0][0][i]*E2nu/rfield.widflx[i] );
					
					E2nu = EmisLines[1][1][2][0].EnergyWN/RYD_INF;
					i = ipoint(y*E2nu);
					fprintf( ioQQQ, "%.3e\n", 
						8./3.*HPLANCK*iso.Pop2Ion[1][1][2]/dense.eden
						*y*iso.As2nu[1][1][i]*E2nu/rfield.widflx[i] );

					/*
					fprintf( ioQQQ, "%.3e\t%.3e\n", 
						rfield.TotDiff2Pht[i]*rfield.anu[i]*EN1RYD
						/E2nu/rfield.widflx[i]/dense.eden/dense.xIonDense[nelem][nelem+1-ipISO]/FR1RYD,
						8./3.*HPLANCK*iso.Pop2Ion[ipISO][nelem][TwoS]/dense.eden
						*y*iso.As2nu[ipISO][nelem][i]*E2nu/rfield.widflx[i]);	*/
				}
				fprintf( ioQQQ, "eden%.3e\n", dense.eden );
			}
		}
	}

	/* free-free free free brems emission for all ions */
	/* ipEnergyBremsThin is index to energy where gas goes optically thin to brems,
	 * so this loop is over optically thick frequencies */
	/* >>chng 01 jul 08, combine loops into one with if branch to
	 * only add optically thin part to outward beam in ThrowOut */
	/* >>chng 01 aug 21, put limit to continuum defined with photons, ipMaxBolt can
	 * be larger than that */
	limit = MIN2( rfield.ipMaxBolt , rfield.nflux );
	/*SumTotalBrems = 0.;*/
	for( i=0; i < limit; i++ )
	{
		double TotBremsAllIons = 0., BremsThisIon;

		/* do hydrogen first, before main loop since want to add on H- brems */
		nelem = ipHYDROGEN;
		ion = 1;
		BremsThisIon = POW2( (float)ion )*dense.xIonDense[nelem][ion]*rfield.gff[ion][i];
		ASSERT( BremsThisIon >= 0. );
		
		/* for case of hydrogen, add H- brems - OpacStack contains the ratio
		 * of the H- to H brems cross section - multiply by this and the fraction in ground */
		BremsThisIon *= (1.+opac.OpacStack[i-1+opac.iphmra]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s]);
		TotBremsAllIons = BremsThisIon;

		/* chng 02 may 16, by Ryan...do all brems for all ions in one fell swoop,
		 * using gaunt factors from rfield.gff.	*/
		for( nelem=ipHELIUM; nelem < LIMELM; nelem++ )
		{
			/* for H, IonLow is 0 and IonHigh is 1 */
			/* MAX2 occurs because we want to start at first ion (or above)
			 * and do not want atom */
			for( ion=MAX2(1,IonRange.IonLow[nelem]); ion<=IonRange.IonHigh[nelem]; ++ion )
			{
				/* eff. charge is ion, so first rfield.gff argument must be "ion".	*/
				BremsThisIon = POW2( (float)ion )*dense.xIonDense[nelem][ion]*rfield.gff[ion][i];
				ASSERT( BremsThisIon >= 0. );

				TotBremsAllIons += BremsThisIon;
			}
		}

		TotBremsAllIons *= dense.eden*1.032e-11*rfield.widflx[i]*rfield.ContBoltz[i]/rfield.anu[i]/phycon.sqrte;
		ASSERT( TotBremsAllIons >= 0.);
		
		/* >>chng 01 jul 01, move thick brems back to ConEmitLocal but do not add
		 * to outward beam - ConLocNoInter array removed as result
		 * if problems develop with very dense blr clouds, this may be reason */
		/*rfield.ConLocNoInter[i] += (float)fac;*/
		rfield.ConEmitLocal[i] += (float)TotBremsAllIons;

		/* do not add optically thick part to outward beam since self absorbed
		 * >>chng 96 feb 27, put back into outward beam since do not integrate
		 * over it anyway. */
		/* >>chng 99 may 28, take back out of beam since DO integrate over it
		 * in very dense BLR clouds */
		/* >>chng 01 jul 10, add here, in only one loop, where opticall thin */
		if( i >= rfield.ipEnergyBremsThin )
		{
			rfield.ThrowOut[i] += (float)TotBremsAllIons;

			/* the total cooling due to brems 
			SumTotalBrems += TotBremsAllIons * rfield.anu[i];*/
		}
	}

	/* grain dust emission */
	/* >>chng 01 nov 22, moved calculation of grain flux to qheat.c, PvH */
	if( gv.lgDustOn )
	{
		/* this calculates diffuse emission from grains,
		 * and stores the result in gv.GrainEmission */
		GrainMakeDiffuse();

		for( i=0; i < rfield.nflux; i++ )
		{
			rfield.ConEmitLocal[i] += gv.GrainEmission[i];
			rfield.ThrowOut[i] += gv.GrainEmission[i];
		}
	}

	/* hminus emission */
	fac = dense.eden*(double)dense.xIonDense[ipHYDROGEN][0];
	gn = 1.;
	gion = 2.;
	gamma = 2.0618684e11*gn/gion/phycon.te/phycon.sqrte;
	cor = rfield.ContBoltz[hmi.iphmin-1];
	/* >>chng 00 dec 15 change limit to -1 of H edge */
	limit = MIN2(iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][ipH1s]-1,rfield.nflux);

	if( cor > 0. )
	{
		for( i=hmi.iphmin-1; i < limit; i++ )
		{
			/* flux is in photons per sec per ryd
			 * ContBoltz is ratio of Boltz fac for each freq */
			factor = gamma*rfield.ContBoltz[i]/cor*rfield.widflx[i]*
			  opac.OpacStack[i-hmi.iphmin+opac.iphmop]*
			  rfield.anu2[i]*fac;
			rfield.ConEmitLocal[i] += (float)factor;
			rfield.ThrowOut[i] += (float)factor;
		}
	}
	else
	{
		for( i=hmi.iphmin-1; i < limit; i++ )
		{
			arg = MAX2(0.,1.57890e5*(rfield.anu[i]-0.05544)/phycon.te);
			/* flux is in photons per sec per ryd */
			factor = gamma*sexp(arg)*rfield.widflx[i]*
				opac.OpacStack[i-hmi.iphmin+opac.iphmop]*
			  rfield.anu2[i]*fac;
			rfield.ConEmitLocal[i] += (float)factor;
			rfield.ThrowOut[i] += (float)factor;
		}
	}

	/* this is a unit of energy that will be passed through the code as a test
	 * that all integrations are carried out.  A similar test is set in lineset1
	 * and verified in PrtFinal.  The opacity at this cell is zero so only
	 * geometrical dilution will affect the integral
	 * Radius is currently outer edge of zone, so radius-drad/2 is radius
	 * of center of zone */

	/* this dilution is needed to conserve volume in spherical models.  tests such
	 * as parispn.in will fault if this is removed */
	Dilution = (float)POW2( radius.rinner / (radius.Radius-radius.drad/2.) );/**/
	rfield.ConEmitLocal[rfield.nflux] = 1.e-10f * Dilution /* */ ;
	rfield.ThrowOut[rfield.nflux] = 1.e-10f * Dilution /* */;

	/* opacity should be zero at this energy */
	ASSERT( opac.opacity_abs[rfield.nflux] == 0. );

	/* many tmn added to conserve energy in very high Z models
	 * rerun highZ qso model if any tmn ever deleted here
	 *
	 * tmn set in StartZone and includes both opacity and dilution 
	 *
	 * use duffuse lines and continuum to add flux to outward beam
	 *
	 * NB!!!  this routine adds flux to the outward beam
	 *  it can only be called once per zone
	 *
	 * covrt is radiative transfer covering factor
	 * covrt = 0 for open geometry
	 * covrt = 1 for closed geometry
	 *
	 * outwrd is fraction that goes outward, of the escaping line radiation
	 * default covrt is zero for open geometry, is 1 for closed
	 * outwrd is 1/2 for opten, 1 for closed */

	/*outwrd = (1. + cover.covrt)/2.;*/

	/* radius is the outer edge of the current zone
	 * dVolOutwrd = outwrd * dreff
	 * >>chng 95 nov 16
	 * dVolOutwrd = outwrd * dreff * ( radius/(radius+drad) )**2
	 * dreff in upper since fill fac enters, but not in lower
	 * >>chng 96 may 08, moved to zonsrt
	 * dVolOutwrd = outwrd * radius * dreff / (radius + 2.*drad) */
	 
	/* save contents of ConInterOut for checking how much was added */
	for( i=0; i < rfield.nflux; i++ )
	{
		rfield.SavOutCon[i] = rfield.ConInterOut[i] + rfield.outlin[i];
	}

	/* >>chng 96 nov 19, do not consider energies below plasma freq
	 * ipPlasmaFreq points to lowest energy thin to brems and plas freq 
	ipPlasmaFreq = MAX2(plasnu.ipPlasma,1)-1;*/

	/* >>chng 01 jul 04, this is a major change in logic */
	/* add the local diffuse continuum to the outward beam contained
	 * in rfield.ConInterOut - this will be added to the interactive part
	 * of the continuum if outward-only is set, and ignored otherwise.
	 * this is done by multiplying ConInterOut by lgOutOnly which is 1 if outward,
	 * zero otherwise */

	/* add rfield.ThrowOut continuum (set in RTDiffuse) to ConInterOut,
	 * lower limit of plasnu.ipPlasma-1 since continuum is zero below plasnu.ipPlasma-1 
	 * due to plasma frequency
	 * note that upper range of sum is <= nflux, which contains the unit
	 * verification token in the highest cell*/
	for( i=plasnu.ipPlasma-1; i <= rfield.nflux; i++ )
	{
		/* ConInterOut is the interactive continuum
		 * tmn is attenuation across one zone
		 * ThrowOut contains all radiation thrown into outward beam, 
		 * eval in RTDiffuse */
		/* NB opac.tmn is needed for FIR brems emission in dense BLR clouds -
		 * each zone is vastly optically thick to 1 cm radiation */
		rfield.ConInterOut[i] += rfield.ThrowOut[i]*(float)radius.dVolOutwrd;
		ASSERT( rfield.ConInterOut[i] >= 0.);
	}

	/* now save the outward continua to track changes */
	for( i=0; i < rfield.nflux; i++ )
	{
		/* >>chng 96 jul 13, add extra grain emiss to save out, so it is
		 * not counted in effective sum - benign source of radiation */
		rfield.SavOutCon[i] += gv.GrainEmission[i]*
		  (float)radius.dVolOutwrd;
	}

	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			fprintf(ioQQQ,"rtdiffusedebugg %li\t%.2e\t%.2e\t%.2e\n", 
				nzone, rfield.ConInterOut[1158] , rfield.ThrowOut[1158],radius.dVolOutwrd);
		}
	}

#	if !defined(NDEBUG)
	/*begin sanity check for level 1 lines */
	for( i=1; i <= nLevel1; i++ )
	{
		if( TauLines[i].phots*
			TauLines[i].ColOvTot < 0. )
		{
			fprintf( ioQQQ, " RTDiffuse finds insane NPohts*ColovTot for level 1 line, vals=%10.1e%10.1e\n", 
			  TauLines[i].phots, 
			  TauLines[i].ColOvTot );
			DumpLine(&TauLines[i]);
			ShowMe();
			puts( "[Stop in RTDiffuse]" );
			cdEXIT(EXIT_FAILURE);
		}
	}
	/*end sanity check */
#	endif

	/* outward level 1 line photons, 0 is dummy line */
	for( i=1; i <= nLevel1; i++ )
	{
		/* pointer to line energy in continuum array */
		ip = TauLines[i].ipCont-1;

		/* last factor does not accout for frac of lines pumped */
		xInWrd = TauLines[i].phots*TauLines[i].FracInwd;
		rfield.reflin[ip] += (float)(xInWrd*radius.BeamInIn);

		/* inward beam that goes out since sphere set */
		rfield.outlin[ip] += (float)(xInWrd*radius.BeamInOut*opac.tmn[ip]*
		  TauLines[i].ColOvTot);

		rfield.outlin[ip] += (float)(TauLines[i].phots*
		  (1. - TauLines[i].FracInwd)*radius.BeamOutOut*
		  /*opac.tmn[ip]**/   TauLines[i].ColOvTot);
	}

#	if !defined(NDEBUG)
	/*begin sanity check for level 2 lines */
	for( i=0; i < nWindLine; i++ )
	{
		if( TauLine2[i].phots*TauLine2[i].ColOvTot < 0. )
		{
			fprintf( ioQQQ, " RTDiffuse finds insane NPohts*ColovTot for level 2 line, vals=%10.1e%10.1e\n", 
			  TauLine2[i].phots, TauLine2[i].ColOvTot );
			DumpLine(&TauLine2[i]);
			ShowMe();
			puts( "[Stop in RTDiffuse]" );
			cdEXIT(EXIT_FAILURE);
		}
	}
	for( i=0; i < nHFLines; i++ )
	{
		if( HFLines[i].phots*HFLines[i].ColOvTot < 0. )
		{
			fprintf( ioQQQ, " RTDiffuse finds insane NPohts*ColovTot for level 2 line, vals=%10.1e%10.1e\n", 
			  HFLines[i].phots, HFLines[i].ColOvTot );
			DumpLine(&HFLines[i]);
			ShowMe();
			puts( "[Stop in RTDiffuse]" );
			cdEXIT(EXIT_FAILURE);
		}
	}
	/*end sanity check */
#	endif

	/* outward level 2 line photons */
	for( i=0; i < nWindLine; i++ )
	{
		if( TauLine2[i].nelem != TauLine2[i].IonStg )
		{
			/* NB if this changes, then FeIIEmitOut in pop371 
			 * probably needs to change too
			 * pointer to line energy in continuum array */
			ip = TauLine2[i].ipCont-1;

			/* last factor does not accout for frac of lines pumped */
			xInWrd = TauLine2[i].phots*TauLine2[i].FracInwd;
			rfield.reflin[ip] += (float)(xInWrd*radius.BeamInIn);

			/* inward beam that goes out since sphere set */
			rfield.outlin[ip] += (float)(xInWrd*radius.BeamInOut*opac.tmn[ip]*
			  TauLine2[i].ColOvTot);

			rfield.outlin[ip] += (float)(TauLine2[i].phots*
			  (1. - TauLine2[i].FracInwd)*radius.BeamOutOut*
			  /*opac.tmn[ip]**/TauLine2[i].ColOvTot);
			/* >>chng 96 jul 16, added ipLnColovTOT */
		}
	}

	/* outward hyperfine structure line photons */
	for( i=0; i < nHFLines; i++ )
	{
		if( HFLines[i].nelem != HFLines[i].IonStg )
		{
			/* NB if this changes, then FeIIEmitOut in pop371 
			 * probably needs to change too
			 * pointer to line energy in continuum array */
			ip = HFLines[i].ipCont-1;

			/* last factor does not accout for frac of lines pumped */
			xInWrd = HFLines[i].phots*HFLines[i].FracInwd;
			rfield.reflin[ip] += (float)(xInWrd*radius.BeamInIn);

			/* inward beam that goes out since sphere set */
			rfield.outlin[ip] += (float)(xInWrd*radius.BeamInOut*opac.tmn[ip]*
			  HFLines[i].ColOvTot);

			rfield.outlin[ip] += (float)(HFLines[i].phots*
			  (1. - HFLines[i].FracInwd)*radius.BeamOutOut*
			  /*opac.tmn[ip]**/HFLines[i].ColOvTot);
			/* >>chng 96 jul 16, added ipLnColovTOT */
		}
	}

	/* carbon monoxide */
	for( i=0; i < nCORotate; i++ )
	{
		/* C12O16 */
		/* pointer to line energy in continuum array */
		ip = C12O16Rotate[i].ipCont-1;

		/* last factor does not accout for frac of lines pumped */
		xInWrd = C12O16Rotate[i].phots*C12O16Rotate[i].FracInwd;
		rfield.reflin[ip] += (float)(xInWrd*radius.BeamInIn);

		/* inward beam that goes out since sphere set */
		rfield.outlin[ip] += (float)(xInWrd*radius.BeamInOut*opac.tmn[ip]*
		  C12O16Rotate[i].ColOvTot);

		rfield.outlin[ip] += (float)(C12O16Rotate[i].phots*
		  (1. - C12O16Rotate[i].FracInwd)*radius.BeamOutOut*C12O16Rotate[i].ColOvTot);

		/* C13O16 */
		/* pointer to line energy in continuum array */
		ip = C13O16Rotate[i].ipCont-1;

		/* last factor does not accout for frac of lines pumped */
		xInWrd = C13O16Rotate[i].phots*C13O16Rotate[i].FracInwd;
		rfield.reflin[ip] += (float)(xInWrd*radius.BeamInIn);

		/* inward beam that goes out since sphere set */
		rfield.outlin[ip] += (float)(xInWrd*radius.BeamInOut*opac.tmn[ip]*
		  C13O16Rotate[i].ColOvTot);

		rfield.outlin[ip] += (float)(C13O16Rotate[i].phots*
		  (1. - C13O16Rotate[i].FracInwd)*radius.BeamOutOut*C13O16Rotate[i].ColOvTot);
	}

	/* H2 emission */
	H2_RTDiffuse(radius.dVolOutwrd,radius.dVolReflec);

	/* do the large FeII atom if it is turned on */
	if( FeII.lgFeIION )
	{
		/* this will do the above, adding FeII to reflected, outward beam */
		if( dense.xIonDense[ipIRON][1] > 0. && FeII.lgFeIION  )
		{
			FeIIEmitOut(radius.dVolOutwrd,radius.dVolReflec);
		}
	}

	/* add recombination continua for elements heavier than those done with iso seq */
	/* nelem = 2 is Li */
	/* >>chng 03 feb 04 move to NISO so totally general */
	/*for( nelem=ipLITHIUM; nelem < LIMELM; nelem++ )*/
	for( nelem=NISO; nelem < LIMELM; nelem++ )
	{
		/* do not include hydrogenic species in following */
		for( ion=0; ion < nelem; ion++ )
		{
			if( dense.xIonDense[nelem][ion+1] > 0. )
			{
				long int ipNu, ns, nshell,igRec , igIon,
					iplow , iphi , ipop;

				ip = Heavy.ipHeavy[nelem][ion]-1;

				/* nflux was reset upward in ConvInitTemp to encompass all
				 * possible line and continuum emission.  this test should not
				 * possibly fail.  It could if the ionization were to increase with depth.
				 * This is important because the nflux cell in ConInterOut is used to carry out the
				 * unit integration, and if it gets clobbered by diffuse emission the code
				 * will declare insanity in PrtComment */
				ASSERT( ip >= 0 && ip < rfield.nflux );

				/* >>chng 01 jul 01, removed ConOutRecInter array, added ground state recombination
				 * into outward beam using Milne relation below */

				/* get shell number, stat weights for this species */
				outer( nelem+1 , nelem+1-ion , &nshell, &igRec , &igIon );
				gn = (double)igRec;
				/* >>chng 01 jun 30, following should have been igIon
				gion = (double)igRec; */
				gion = (double)igIon;

				/* shell number */
				ns = Heavy.nsShells[nelem][ion]-1;
				ASSERT( ns == (nshell-1) );

				/* lower and upper energies, and offset for opacity stack */
				iplow = opac.ipElement[nelem][ion][ns][0]-1;
				iphi = opac.ipElement[nelem][ion][ns][1];
				iphi = MIN2( iphi , rfield.nflux );
				ipop = opac.ipElement[nelem][ion][ns][2];

				/* now convert ipop to the offset in the opacity stack from threshold */
				ipop = ipop - iplow;

				/*bolt2 = 1.57890e5*rfield.anu[iplow]/phycon.te;*/

				gamma = 2.0618684e11*gn/gion/phycon.te/phycon.sqrte*dense.eden*dense.xIonDense[nelem][ion+1];

				/* this is ground state continuum from stored opacities */
				if( rfield.ContBoltz[iplow] > SMALLFLOAT )
				{
					for( ipNu=iplow; ipNu < iphi; ++ipNu )
					{
						photon = gamma*rfield.ContBoltz[ipNu]/rfield.ContBoltz[iplow]*
							rfield.widflx[ipNu]*opac.OpacStack[ipNu+ipop]*rfield.anu2[ipNu];
						/* >>chng 01 jul 01, add heavy rec to ground in active beam,
						 * had been in before but not in active beam */
						/* fixit();should use ConEmitLocal for all continua, but not followed
								 * by ThrowOut - put that at the end.  Once continua all
								 * bundled this way, it will be easy to save them as a function
								 * of depth and then do exact rt */
						rfield.ConEmitLocal[ipNu] += (float)photon;
						rfield.ThrowOut[i] += (float)photon;
					}
				}

				/* now do the recombination Lya */
				ipla = Heavy.ipLyHeavy[nelem][ion]-1;
				ASSERT( ipla >= 0 );
				esc = opac.ExpmTau[ipla];
				/* xLyaHeavy is set to a fraction of the total rad rec in MakeRecomb, includes eden */
				difflya = Heavy.xLyaHeavy[nelem][ion]*dense.xIonDense[nelem][ion+1];
				rfield.outlin[ipla] += (float)(difflya*radius.dVolOutwrd*opac.tmn[ipla]*esc);
				rfield.reflin[ipla] += (float)(difflya*radius.dVolReflec*opac.tmn[ipla]*esc);

				/* now do the recombination Balmer photons */
				ipla = Heavy.ipBalHeavy[nelem][ion]-1;
				ASSERT( ipla >= 0 );
				esc = opac.ExpmTau[ipla];
				/* xLyaHeavy is set to fraction of total rad rec in MakeRecomb, includes eden */
				difflya = Heavy.xLyaHeavy[nelem][ion]*dense.xIonDense[nelem][ion+1];
				rfield.outlin[ipla] += (float)(difflya*radius.dVolOutwrd*opac.tmn[ipla]*esc);
				rfield.reflin[ipla] += (float)(difflya*radius.dVolReflec*opac.tmn[ipla]*esc);
			}
		}
	}

	/* Fe Auger K-alpha in outward only
	 * first is rec plus hot auger
	 * this is pointer to valence shell Lya
	 * >>chng 97 apr 28, had been 26,20, so picked up vastly wrong energy for Ka
	 * improve to real pointers for line
	 * >>chng 96 nov 20, added tmn to two below, to prevent line from
	 * overwhelming very thick models
	 * >>chng 97 apr 28, put in reflected Ka, had been only outward
	 * outlin(ip) = outlin(ip) + fekhot/1.11e-8 * dReff * */
	rfield.outlin[fekems.ipkhot-1] += fekems.fekhot*(float)radius.dVolOutwrd*
	  opac.tmn[fekems.ipkhot-1];

	rfield.reflin[fekems.ipkhot-1] += fekems.fekhot*(float)radius.dVolReflec;

	/* >>chng 97 apr 28, put in reflected Ka, had been only outward
	 * this is cold iron, one cell below hot
	 * outlin(ip-1) = outlin(ip-1) +fekcld/1.03e-8*dReff* */
	/* >>chng 02 jan 14, add grain fe to both cold Ka */
	rfield.outlin[fekems.ipkcld-1] += (fekems.fekcld+fekems.fegrain)*(float)radius.dVolOutwrd*
	  opac.tmn[fekems.ipkcld-2];

	rfield.reflin[fekems.ipkcld-1] += (fekems.fekcld+fekems.fegrain)*(float)radius.dVolReflec;

	if( FeII.lgFeIION )
	{
		/* do OTS and outward parts of FeII lines, if large atom is turned on */
		FeIIRTOut();
	}

	/**********************************************************************
	 *                                                                    *
	 * now analyse what we have done to the outward beam                  *
	 *                                                                    *
	 **********************************************************************/

	/* now figure out how much we added to the beam
	 * this is used for setting zone thickness - do not want to
	 * add too much energy in outward beam */
	rfield.SumOutCon = 0.;
	rfield.SumIncCon = 0.;
	ots = 0.;

	/* this will be photon energy where this peak occurs */
	rfield.SumOutMax = 0.;

	/* radiation field may not extend up to H ionization limit */
	limit = MIN2( rfield.nflux , (iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][ipH1s] - 1) );
	for( i=0; i < limit; i++ )
	{
		/* >>chng 96 jan 14, was photon number, now energy
		 * >>chng 96 nov 19, save difference in SavOutCon array, for later punch */
		rfield.SavOutCon[i] = (float)(MAX2(0.,rfield.outlin[i]+rfield.ConInterOut[i]-
			rfield.SavOutCon[i]));

		esc = rfield.SavOutCon[i]*opac.opacity_abs[i]*rfield.anu[i];

		if( esc > ots )
		{
			ots = esc;
			rfield.SumOutMax = rfield.anu[i];
			rfield.ipSumOutMax = i+1;
		}

		/* the ratio of SumOutCon to SumIncCon is kept below 5% in nextdr */
		rfield.SumOutCon += (float)(esc*0.);/*fixit();remove *0. */

		/* >>chng 96 nov 18, add ConInterOut, outlin to sum
		 * SumIncCon = SumIncCon + (flux(i)+otscon(i)+otslin(i)) *opac(i)* */
		rfield.SumIncCon += (float)((rfield.flux[i] + rfield.otscon[i] + 
		  rfield.otslin[i] + rfield.ConInterOut[i] + rfield.outlin[i])*
		  opac.opacity_abs[i]*rfield.anu[i]);
	}

	rfield.SumHOutCon = 0.;
	rfield.SumHIncCon = 0.;
	/* this will be sum of H-ionizing continuum
	 * >>chng 96 jan 14, was photon number, now energy */
	for( i=limit; i < rfield.nflux; i++ )
	{
		/* esc = ( outlin(i)+ConInterOut(i)-SavOutCon(i) )* opac(i)* anu(i) */
		rfield.SavOutCon[i] = (float)MAX2(0.,rfield.outlin[i]+rfield.ConInterOut[i]-
		  rfield.SavOutCon[i]);

		esc = rfield.SavOutCon[i]*opac.opacity_abs[i]*rfield.anu[i];

		if( esc > ots )
		{
			ots = esc;
			rfield.SumOutMax = rfield.anu[i];
			rfield.ipSumOutMax = i+1;
		}

		rfield.SumHOutCon += (float)esc;

		/* >>chng 96 nov 18, add ConInterOut, outlin to sum */
		rfield.SumHIncCon += (float)((rfield.flux[i] + rfield.otscon[i] + 
		  rfield.otslin[i] + rfield.ConInterOut[i] + rfield.outlin[i])*
		  opac.opacity_abs[i]*rfield.anu[i]);
	}
	rfield.SumOutCon += rfield.SumHOutCon;
	rfield.SumIncCon += rfield.SumHIncCon;

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, " RTDiffuse returns.\n" );
	}

	/* >>chng 02 jul 25, zero out below plasma freq */
	for( i=0; i<plasnu.ipPlasma-1; i++ )
	{
		rfield.flux[i] = 0.;
		rfield.ConEmitLocal[i] = 0.;
		rfield.ThrowOut[i] = 0.;
		rfield.otscon[i] =0.;
		rfield.otslin[i] =0.;
		rfield.outlin[i] =0.;
		rfield.reflin[i] = 0.;
		rfield.TotDiff2Pht[i] = 0.;
		rfield.ConOTS_local_photons[i] = 0.;
		rfield.ConInterOut[i] = 0.;
	}

	for( i=0; i < rfield.nflux; i++ )
	{
		/* >>chng 00 oct 03, add diffuse continua */
		/* local diffuse continua */
		rfield.OccNumbDiffCont[i] = rfield.ConEmitLocal[i]*rfield.convoc[i];

		/* confirm that all are non-negative */
		ASSERT( rfield.flux[i] >=0.);
		ASSERT( rfield.otscon[i] >=0.) ;
		ASSERT( rfield.otslin[i] >=0.) ;
		ASSERT( rfield.ConInterOut[i] >=0.) ;
		ASSERT( rfield.outlin[i] >=0.) ;
	}

#	ifdef DEBUG_FUN
	fputs( " <->RTDiffuse()\n", debug_fp );
#	endif
	return;
}

