/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*RT_OTS compute diffuse fields due to H, He atoms, ion, triplets, metal recombination,
 * called by ConvIonizeOpacityDo  */
/*RT_OTS_AddLine add local destruction of lines to ots field */
/*RT_OTS_AddCont add local destruction of continuum to ots field */
/*RT_OTS_Update sum flux, otscon, otslin, ConInterOut, outlin, to form SummeDif, SummedCon SummedOcc */
/*RT_OTS_Zero - zero out some vectors - 
 * this is only called when code initialized by ContSetIntensity */
/*RT_OTS_ChkSum sanity check confirms summed continua reflect contents of individuals */
#include "cddefines.h"
#include "taulines.h"
#include "opacity.h"
#include "dense.h"
#include "iso.h"
#include "plasnu.h"
#include "rfield.h"
#include "converge.h"
#include "rt.h"
#include "atomfeii.h"
#include "heavy.h"
#include "pbowen.h"
#include "ionrange.h"
#include "radius.h"
#include "trace.h"

/* =================================================================== */

void RT_OTS(void)
{
	long int
		ipla,
		ipISO ,
		nelem,
		n;
	float 
		difflya,
		esc,
		ots;

	/* the Bowen HeII yield */
#	define BOWEN 0.3f
	long int ipHi, 
	  ipLo;

	/*int lgHeTypeSave;*/

	float bwnfac ,
	  ots660 ,
	  otsnew;

#	ifdef DEBUG_FUN
	fputs( "<+>RT_OTS()\n", debug_fp );
#	endif

	/**************************************************************************
	 *
	 * the bowen HeII - OIII fluorescense problem
	 *
	 **************************************************************************/

	nelem = ipHELIUM;
	if( dense.lgElmtOn[nelem] )
	{
		/* conversion per unit atom to OIII, at end of sub we divide by it,
		 * to fix lines back to proper esc/dest probs */
		bwnfac = BOWEN * MAX2(0.f,1.f- EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pesc - 
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pelec_esc );

		/* the last factor accounts for fact that two photons are produced,
		 * and the branching ratio */
		ots660 = (float)(EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Aul*
		  EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].PopHi*dense.xIonDense[nelem][nelem+1]*
		  EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest *BOWEN*2.0 * 0.8 );

		/* now add this to the ots field */
		RT_OTS_AddLine(ots660 , pbowen.ip660 );

		/* decrease the destruction prob by the amount we will add elsewhere,
		 * ok since dest probs updated on every iteration*/
		EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest *= bwnfac;
		ASSERT( EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest >= 0. );
	}

	else
	{
		bwnfac = 1.;
	}

	/* make ots fields due to lines and continua of species treated with unified 
	 * isoelectronic sequence */
	/* loop over all elements */
	for( ipISO=0; ipISO<NISO; ++ipISO )
	{
		for( nelem=ipISO; nelem < LIMELM; nelem++ )
		{
			/* do if this stage exists */
			if( (IonRange.IonHigh[nelem] >= nelem+1-ipISO )  )
			{
				/* generate line ots rates */
				/* now loop over all possible levels, but cannot include two photon
				* since there is no pointer to this continuum */
				for( ipHi=1; ipHi < iso.numLevels[ipISO][nelem]; ipHi++ )
				{
					for( ipLo=0; ipLo < ipHi; ipLo++ )
					{
						/* this signifies a fake line */
						if( EmisLines[ipISO][nelem][ipHi][ipLo].ipCont< 1 )
							continue;
						/* ots rates, the destp prob was set in hydropesc */
						EmisLines[ipISO][nelem][ipHi][ipLo].ots = (float)(
							EmisLines[ipISO][nelem][ipHi][ipLo].Aul*
							EmisLines[ipISO][nelem][ipHi][ipLo].PopHi*dense.xIonDense[nelem][nelem+2-ipISO-1]*
							EmisLines[ipISO][nelem][ipHi][ipLo].Pdest);

						ASSERT( EmisLines[ipISO][nelem][ipHi][ipLo].ots >= 0. );
						/* option to kill lyalpha ots rates 
						if( nelem==0 && ipHi==ipH2p && ipLo==ipH1s )
						{
							EmisLines[ipISO][nelem][ipHi][ipLo].ots = 0.;
						}*/

						/* finally dump the ots rate into the stack */
						RT_OTS_AddLine(EmisLines[ipISO][nelem][ipHi][ipLo].ots,
							EmisLines[ipISO][nelem][ipHi][ipLo].ipCont );
					}
				}
				{
					/* debugging code for line oscillation problems */
					/*@-redef@*/
					enum {DEBUG_LOC=FALSE};
					/*@+redef@*/
					if( DEBUG_LOC )
					{
						if( nelem==0  && iteration==2 ) 
						{
							fprintf(ioQQQ,"%li hlyaots\t%g\t%g\n",
								nzone, EmisLines[ipISO][nelem][4][3].ots,
								opac.opacity_abs[EmisLines[ipISO][nelem][4][3].ipCont-1]);
						}
					}
				}

				/**************************************************************************
				*
				* ots recombination bound-free b-f continua continuum
				*
				**************************************************************************/

				/* put in OTS continuum */
				for( n=0; n < iso.numLevels[ipISO][nelem]; n++ )
				{
					otsnew = (float)(iso.RadRecomb[ipISO][nelem][n][ipRecRad]*
						(1. - iso.RadRecomb[ipISO][nelem][n][ipRecEsc])*dense.eden*
						dense.xIonDense[nelem][nelem+1]);
					ASSERT( otsnew >= 0. );

					/* continuum energy index used in this routine is decremented by one there */
					RT_OTS_AddCont(otsnew,iso.ipIsoLevNIonCon[ipISO][nelem][n]);
					/* debugging code for rec continua */
					{
						/*@-redef@*/
						enum {DEBUG_LOC=FALSE};
						/*@+redef@*/
						if( DEBUG_LOC )
						{
							if( nzone > 234 && nelem==0 && n==0) 
							{
								fprintf(ioQQQ,"hotsdebugg %li\t%.4e\th con ots\t%g\t%g\t%g\t%g\t%g\t%g\n",
									nzone, 
									radius.depth ,
									EmisLines[ipISO][ipHYDROGEN][2][0].PopHi*dense.xIonDense[ipHYDROGEN][1],
									otsnew,
									iso.RadRecomb[ipISO][nelem][n][ipRecEsc] , 
									opac.opacity_abs[iso.ipIsoLevNIonCon[ipISO][nelem][n]-1] ,
									rfield.otsconNew[iso.ipIsoLevNIonCon[ipISO][nelem][n]-1] ,
									opac.e2TauAbs[iso.ipIsoLevNIonCon[ipISO][nelem][n]-1] );
							}
						}
					}
				}
			}
		}
	}
	/* more debugging code for rec continua */
	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			nelem = 0;
			fprintf(ioQQQ,"hotsdebugg %li \t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n",
				nzone, 
				rfield.otsconNew[iso.ipIsoLevNIonCon[ipH_LIKE][nelem][0]-1],
				rfield.otsconNew[iso.ipIsoLevNIonCon[ipH_LIKE][nelem][1]-1],
				rfield.otsconNew[iso.ipIsoLevNIonCon[ipH_LIKE][nelem][3]-1],
				rfield.otsconNew[iso.ipIsoLevNIonCon[ipH_LIKE][nelem][4]-1],
				rfield.otsconNew[iso.ipIsoLevNIonCon[ipH_LIKE][nelem][5]-1],
				rfield.otsconNew[iso.ipIsoLevNIonCon[ipH_LIKE][nelem][6]-1],
				       opac.opacity_abs[iso.ipIsoLevNIonCon[ipH_LIKE][nelem][6]-1]);
		}
	}

	nelem = ipHELIUM;
	if( dense.lgElmtOn[nelem] && bwnfac > 0. )
	{
		/* increase the destruction prob by the amount we decreased it above */
		EmisLines[ipH_LIKE][ipHELIUM][ipH2p][ipH1s].Pdest /= bwnfac;
	}

	if( trace.lgTrace )
	{
		fprintf(ioQQQ,"     RT_OTS Pdest %.2e ots rate %.2e in otslinNew %.2e con opac %.2e\n",
			EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pdest , EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].ots , 
			rfield.otslinNew[EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].ipCont-1]  ,
			opac.opacity_abs[EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].ipCont-1]
			);
	}


	/* recombination continua for all elements not yet converted into std isoelectronc form */
	for( nelem=NISO ; nelem < LIMELM; nelem++ )
	{
		long int ion;
		/* do not include species treated in iso-electronic fashon in the following,
		 * these were treated above */
		for( ion=0; ion < nelem+1-NISO; ion++ )
		{
			if( dense.xIonDense[nelem][ion+1] > 0. )
			{
				/* now do the recombination Lya */
				ipla = Heavy.ipLyHeavy[nelem][ion];
				esc = opac.ExpmTau[ipla-1];
				/* xLyaHeavy is set to a fraction of total rad rec in MakeRecomb, includes eden */
				difflya = Heavy.xLyaHeavy[nelem][ion]*dense.xIonDense[nelem][ion+1];
				/* >>chng 00 dec 22, from MIN2 to MAX2, MIN2 had effect of always
				 * setting the ots rates to zero */
				ots = difflya*MAX2(0.f,1.f-esc);
				ASSERT( ots >= 0.);
				RT_OTS_AddLine(ots,ipla);

				/* now do the recombination balmer lines */
				ipla = Heavy.ipBalHeavy[nelem][ion];
				esc = opac.ExpmTau[ipla-1];
				/* xLyaHeavy is set to a fraction of total rad rec in MakeRecomb, includes eden */
				difflya = Heavy.xLyaHeavy[nelem][ion]*dense.xIonDense[nelem][ion+1];
				/* >>chng 00 dec 22, from MIN2 to MAX2, MIN2 had effect of always
				 * setting the ots rates to zero */
				ots = difflya*MAX2(0.f,1.f-esc);
				ASSERT( ots >= 0.);
				RT_OTS_AddLine(ots,ipla);
			}
		}
	}

	if( FeII.lgFeIION )
	{
		/* do OTS and outward parts of FeII lines, if large atom is turned on */
		FeIIRTOTS();
	}

#	ifdef DEBUG_FUN
	fputs( " <->RT_OTS()\n", debug_fp );
#	endif

	return;
}

/* =================================================================== */

void RT_OTS_AddLine(float ots, 
	/* pointer on the f scale */
  long int ip )
{

#	ifdef DEBUG_FUN
	fputs( "<+>RT_OTS_AddLine()\n", debug_fp );
#	endif

	/* add ots due to line destruction to radiation field */

	/* return if outside bounds of this continuum source, ip > rfield.nflux
	 * first case ip==0 happens when called with dummy line */
	if( ip==0 || ip > rfield.nflux )
	{ 
#		ifdef DEBUG_FUN
		fputs( " <->RT_OTS_AddLine()\n", debug_fp );
#		endif
		return;
	}

	/*the local ots rate must be non-negative */
	ASSERT( ots >= 0. );
	/* continuum pointer must be positive */
	ASSERT( ip > 0 );

	/* add locally destroyed flux of photons to line OTS array */
	/* check whether local gas opacity  (units cm-1) is positive, if so
	 * convert line destruction rate into ots rate by dividing by it */
	if( opac.opacity_abs[ip-1] > 0. )
	{
		rfield.otslinNew[ip-1] += (float)(ots/opac.opacity_abs[ip-1]);
	}

#	ifdef DEBUG_FUN
	fputs( " <->RT_OTS_AddLine()\n", debug_fp );
#	endif
	return;
}

/* =================================================================== */

/*add local destruction of continuum to ots field */
void RT_OTS_AddCont(
					/* the ots rate itself */
					float ots, 
					/* pointer to continuum cell for ots, on f scale */
					long int ip)
{

#	ifdef DEBUG_FUN
	fputs( "<+>RT_OTS_AddCont()\n", debug_fp );
#	endif

	/* 
	 * routine called to add ots due to continuum destruction to
	 * radiation field
	 */
	 
	/* check if outside bounds of this continuum source */
	if( ip > rfield.nflux )
	{ 
#		ifdef DEBUG_FUN
		fputs( " <->RT_OTS_AddCont()\n", debug_fp );
#		endif
		return;
	}

	ASSERT( ip> 0 );

	ASSERT( ots >= 0. );

	ASSERT( ip <= rfield.nupper );

	/* add locally destroyed flux of photons to continuum OTS array */
	/* check whether local gas opacity  (units cm-1) is positive, if so
	 * convert continuum destruction rate into ots rate by dividing by it */
	if( opac.opacity_abs[ip-1] > 0. )
	{
		rfield.otsconNew[ip-1] += (float)(ots/opac.opacity_abs[ip-1]);
		/*if( ots>0. ) fprintf(ioQQQ,
			"buggg ip %li ots %.2e opac %.2e otsnew %.2e\n",
			ip, ots , opac.opacity_abs[ip-1] ,rfield.otsconNew[ip-1]);*/
	}

#	ifdef DEBUG_FUN
	fputs( " <->RT_OTS_AddCont()\n", debug_fp );
#	endif
	return;
}

/* =================================================================== */

void RT_OTS_Update(
	/* summed ots rates */
	double *SumOTS , 
	/* array index for constituent that changed the most */
	long int * ipOTSchange, 
	/* limit on how large a change to allow, 0 for no limit */
	double BigFrac)
{
	long int i;

	/* when setting new to sum of current and old, this is the fraction of old to take*/
	double FracOld , FracNew ,
		changeOTS ,
		chng,
		OTSOld ,
		OTSNew,
		BigOTSNew;

#	ifdef DEBUG_FUN
	fputs( "<+>RT_OTS_Update()\n", debug_fp );
#	endif

	*SumOTS = 0.;

	OTSOld = 0.;
	OTSNew = 0.;
	BigOTSNew = 0.;
	/* find summed ots rates, old and new */
	for( i=0; i < rfield.nflux; i++ )
	{
		OTSOld += rfield.otscon[i] + rfield.otslin[i];
		OTSNew += rfield.otsconNew[i] + rfield.otslinNew[i];
		if( BigOTSNew < rfield.otsconNew[i] + rfield.otslinNew[i] )
		{
			BigOTSNew = rfield.otsconNew[i] + rfield.otslinNew[i];
		}
	}
	
	/* we now have old and new rates, what is the ratio, and by how much will
	 * we allow this to change? */
	if( BigFrac == 0. || conv.lgSearch || OTSOld < SMALLFLOAT )
	{
		/* this branch, want a clear update of ots rates, either because
		 * requested with BigFrac == 0, or we are in search phase */
		FracOld = 0.;
	}
	else
	{
		if( OTSNew > OTSOld )
		{
			chng = fabs(1. - OTSOld/MAX2(SMALLFLOAT,OTSNew) );
		}
		else
		{
			chng = fabs(1. - OTSNew/MAX2(SMALLFLOAT,OTSOld) );
		}

		if( chng < BigFrac )
		{
			/* this branch, old and new do not differ by much, so use new */
			FracOld = 0.;
		}
		else
		{
			/* this branch, too large a difference between old and new, cap it to BigFrac */
			FracOld = (1. - BigFrac / chng);
			ASSERT( FracOld >= 0. );
			FracOld = MIN2( 0.25 , FracOld );
		}
	}

	/* fraction old and new ots rates */
	FracNew = 1. - FracOld ;

	/* remember largest change in ots rates */
	changeOTS = 0.;
	*ipOTSchange = -100000;
	/*fprintf(ioQQQ," sumcontinuum zone%li 1168=%e\n", nzone,rfield.otslin[1167]);*/

	for( i=0; i < rfield.nflux; i++ )
	{
		/* >>chng 01 feb 01, define inverse opacity in safe manner */
		double CurrentInverseOpacity = 1./MAX2( SMALLDOUBLE , opac.opacity_abs[i] ) ;

		/* remember which energy had largest change in ots rates */
		if( fabs( rfield.otscon[i]+rfield.otslin[i]-rfield.otsconNew[i]-rfield.otslinNew[i])> changeOTS)
		{
			*ipOTSchange = i;

			changeOTS = 
			  fabs( rfield.otscon[i]+rfield.otslin[i]-rfield.otsconNew[i]-rfield.otslinNew[i]);
		}

		/* >>chng 01 apr 10, only change ots with means if FracOld not zero,
		 * this is to avoid taking means when this routine called by StartEndIter 
		 * to reset sums of rates */
		if( BigFrac > 0. && conv.nTotalIoniz > 0 )
		{
			/* the New vectors are the ones updated by the AddOTS routines,
			 * and have the current OTS rates.  The otscon and otslin vectors
			 * have the rates from the previous refresh of the New vectors*/
			/* here is the refresh.  all were initially set to zero in call to 
			 * RT_OTS_Zero (below) from zero */
			rfield.otscon[i] = (float)((rfield.otscon[i]*FracOld +rfield.otsconNew[i]*FracNew));

			/* this is local ots continuum created by destroyed diffuse continua, 
			 * currently only two-photon */
			/* >>chng 00 dec 27, define this continuum and do not count 2-photon in old var */
			rfield.ConOTS_local_OTS_rate[i] = (float)((double)rfield.ConOTS_local_photons[i]*CurrentInverseOpacity);

			/* current ots will be average of old and new */
			rfield.otslin[i] = (float)((rfield.otslin[i]*FracOld + rfield.otslinNew[i]*FracNew));
		}

		/* zero out the new otscon vector*/
		rfield.otsconNew[i] = 0.;

		/* zero out the new otslin vector*/
		rfield.otslinNew[i] = 0.;

		/* remember sum of ots rates for convergence criteria */
		*SumOTS += (rfield.otscon[i] + rfield.otslin[i])*opac.opacity_abs[i];
		{
			/* following should be set true to print strongest ots contributors */
			/*@-redef@*/
			enum {DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC && (nzone==378)/**/ )
			{
				if( conv.nPres2Ioniz > 3 ) 
					exit(1);
				fprintf(ioQQQ,"rtotsbugggg\t%li\t%.3e\t%.3e\t%.3e\t%.3e\n",
					conv.nPres2Ioniz,
					rfield.anu[i],
					opac.opacity_abs[i],
					rfield.otscon[i],
					rfield.otslin[i]);
			}
		}

		/* >>chng 00 dec 27, include ConOTS_local_OTS_rate */
		/* >>chng 01 jul 04, add *rfield.lgOutOnly logic */
		rfield.SummedDif[i] = rfield.otscon[i] + rfield.otslin[i] + 
		  rfield.ConInterOut[i]*rfield.lgOutOnly + rfield.outlin[i] +
		  rfield.ConOTS_local_OTS_rate[i];

		rfield.SummedCon[i] = rfield.flux[i] + rfield.SummedDif[i];
		rfield.SummedOcc[i] = rfield.SummedCon[i]*rfield.convoc[i];
	}

	/* >>chng 02 oct 18, add this */
	/* sum of accumulated flux from particular frequency to infinity */
	rfield.flux_accum[rfield.nflux-1] = 0;
	for( i=1; i < rfield.nflux; i++ )
	{
		rfield.flux_accum[rfield.nflux-i-1] = rfield.flux_accum[rfield.nflux-i] +
			rfield.SummedCon[rfield.nflux-i-1];
	}
	/* this has to be positive since is sum of all photons in SummedCon */
	ASSERT( rfield.flux_accum[0]> 0. );

	/* >>chng 02 jul 23, set to black body at local temp if in optically thick continuum,
	 * between plasma freqnecy and energy where brems is thin */
	ASSERT(plasnu.ipPlasma>0 );

	/* >>chng 02 jul 25, set all radiation fields to zero below plasma frequence */
	for( i=0; i < plasnu.ipPlasma-1; i++ )
	{
		rfield.otscon[i] = 0.;
		rfield.ConOTS_local_OTS_rate[i] = 0.;
		rfield.ConOTS_local_photons[i] = 0.;
		rfield.otslin[i] = 0.;
		rfield.SummedDif[i] = 0.;
		rfield.OccNumbBremsCont[i] = 0.;
		rfield.SummedCon[i] = 0.;
		rfield.SummedOcc[i] = 0.;
		rfield.ConInterOut[i] = 0.;
	}
	/* this loop evaluates occupation number for brems continuum,
	 * only used for induced two photon emission */
	if( rfield.ipEnergyBremsThin > 0 )
	{
		for( i=plasnu.ipPlasma-1; i < rfield.nflux; i++ )
		{
			/* this corrects for opacity / optical depth in brems - brems opacity goes as
			 * energy squared. */
			/* when totally optically thin to brems rfield.ipEnergyBremsThin is zero,
			 * so need this max */
			float factor = MIN2(1.f,rfield.anu2[MAX2(0,rfield.ipEnergyBremsThin-1)] / rfield.anu2[i]);

			/* occupation number for black body is 1/ (exp hn/kT) -1) */
			rfield.OccNumbBremsCont[i] = (float)(1./(1./MAX2(SMALLFLOAT,rfield.ContBoltz[i]) - 1.)) * factor;
		}
	}

#	if 0
	/* ipPlasma is 1 if plasma frequence does not occur within the grid, 
	 * so loop starts at 0 once decremented by 1 
	 * this sets the continuum between the plasma frequency and where the gas becomes
	 * optically thin to brems absorption equal to a black body at the local temperature.*/
	for( i=plasnu.ipPlasma-1; i < rfield.ipEnergyBremsThin; i++ )
	{
		/* occupation number for black body is 1/ (exp hn/kT) -1) */
		rfield.SummedOcc[i] = (float)(1./(1./MAX2(SMALLFLOAT,rfield.ContBoltz[i]) - 1.));
		rfield.SummedCon[i] = rfield.SummedOcc[i]/rfield.convoc[i];
		/* >>chng 02 jul 29, most routines will get total radiation field from above, 
		 * but to be safe, will put the black body into this local ots field */
		rfield.ConOTS_local_OTS_rate[i] = rfield.SummedCon[i];
	}

	/* ipPlasma is 1 if plasma frequence does not occur within the grid, 
	 * so loop starts at 0 once decremented by 1 */
	/* THINK - SummedCon gets hosed - this will replace everything in the array 
	 * so this is from where thing begins, and adds flux */
	/* only do this is gas is optically thick to part of bresms */
	if( rfield.ipEnergyBremsThin > 0 )
	{
		for( i=rfield.ipEnergyBremsThin; i < rfield.nflux; i++ )
		{
			/* this corrects for opacity / optical depth in brems - brems opacity goes as
			* energy squared. */
			/* when totally optically thin to brems rfield.ipEnergyBremsThin is zero,
			* so need this max */
			float factor = rfield.anu2[MAX2(0,rfield.ipEnergyBremsThin-1)] / rfield.anu2[i];
			ASSERT( factor >= 0. && factor <= 1. );

			/* occupation number for black body is 1/ (exp hn/kT) -1) */
			rfield.SummedOcc[i] += (float)(1./(1./MAX2(SMALLFLOAT,rfield.ContBoltz[i]) - 1.)) * factor;
			rfield.SummedCon[i] = rfield.SummedOcc[i]/MAX2(SMALLFLOAT,rfield.convoc[i]);
			/* >>chng 02 jul 29, most routines will get total radiation field from above, 
			* but to be safe, will put the black body into this local ots field */
			rfield.ConOTS_local_OTS_rate[i] = rfield.SummedCon[i];
		}
	}
#	endif

	{
		/* following should be set true to print strongest ots contributors */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && (nzone>0)/**/ )
		{
			double BigOTS;
			int ipOTS=0;
			BigOTS = -1.;
			/* find the biggest ots contributor */
			/*for( i=iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][0]; i < rfield.nflux; i++ )*/
			for( i=0; i < rfield.nflux; i++ )
			{
				if( (rfield.otscon[i] + rfield.otslin[i])*opac.opacity_abs[i] > BigOTS )
				{
					BigOTS = (rfield.otscon[i] + rfield.otslin[i])*opac.opacity_abs[i];
					ipOTS = (int)i;
				}
			}
			fprintf(ioQQQ,
				" sumcontinuunots zone %li SumOTS %.2e %.2eRyd opc:%.2e OTS%.2e lin%.2e con:%.2e %s %s cell%i FracNew %.2f \n",
				nzone,
				*SumOTS,
				rfield.anu[ipOTS] , 
				opac.opacity_abs[ipOTS],
				BigOTS ,
				rfield.otslin[ipOTS],
				rfield.otscon[ipOTS] , 
				/* line	label */
				rfield.chLineLabel[ipOTS] ,
				/* cont label*/
				rfield.chContLabel[ipOTS] ,
				ipOTS,
				FracNew);
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->RT_OTS_Update()\n", debug_fp );
#	endif
	return;
}

/* =================================================================== */

/*RT_OTS_Zero zero out some vectors - this is only called when code initialized by ContSetIntensity */
void RT_OTS_Zero( void )
{
	long int i;

#	ifdef DEBUG_FUN
	fputs( "<+>RT_OTS_Zero()\n", debug_fp );
#	endif

	/* this loop goes up to nflux itself (<=) since the highest cell
	 * will be used to pass unity through the code to verify integrations */
	for( i=0; i <= rfield.nflux; i++ )
	{
		rfield.SummedDif[i] = 0.;
		/* the four main ots vectors */
		rfield.otscon[i] = 0.;
		rfield.otslin[i] = 0.;
		rfield.otslinNew[i] = 0.;
		rfield.otsconNew[i] = 0.;

		rfield.ConInterOut[i] = 0.;
		rfield.outlin[i] = 0.;
		rfield.SummedDif[i] = 0.;
		/* "zero" for the summed con will be just the incident radiation field */
		rfield.SummedCon[i] = rfield.flux[i];
		rfield.SummedOcc[i] = rfield.SummedCon[i]*rfield.convoc[i];
		rfield.ConOTS_local_photons[i] = 0.;
		rfield.ConOTS_local_OTS_rate[i] = 0.;
	}

#	ifdef DEBUG_FUN
	fputs( " <->RT_OTS_Zero()\n", debug_fp );
#	endif
	return;
}

/* =================================================================== */

/*RT_OTS_ChkSum sanity check confirms summed continua reflect contents of individuals */
void RT_OTS_ChkSum(long int ipPnt)
{
	static long int nInsane=0;
	long int i;
	double phisig;

#	ifdef DEBUG_FUN
	fputs( "<+>RT_OTS_ChkSum()\n", debug_fp );
#	endif

	/* this entire sub is a sanity check */
	/* >>chng 02 jul 23, lower bound from 0 to rfield.ipEnergyBremsThin - since now
	 * set radiation field to black body below this energy */
	for( i=rfield.ipEnergyBremsThin; i < rfield.nflux; i++ )
	{
		phisig = rfield.otscon[i] + rfield.otslin[i] + rfield.ConInterOut[i]*rfield.lgOutOnly + 
		  rfield.outlin[i]+
		  rfield.ConOTS_local_OTS_rate[i];
		/* >>chng 02 sep 19, add sec test on SummedDif since it can be zero whild 
		 * phisig is just above small float */
		if( phisig > 0. && rfield.SummedDif[i]> 0.)
		{
			if( fabs(rfield.SummedDif[i]/phisig-1.) > 1e-3 )
			{
				fprintf( ioQQQ, " PROBLEM RT_OTS_ChkSum insane SummedDif at energy %.5e error= %.2e i=%4ld\n", 
				  rfield.anu[i], rfield.SummedDif[i]/phisig - 1., i );
				fprintf( ioQQQ, " SummedDif, sum are%11.4e%11.4e\n", 
				  rfield.SummedDif[i], phisig );
				fprintf( ioQQQ, " otscon otslin ConInterOut outlin are%11.4e%11.4e%11.4e%11.4e\n", 
				  rfield.otscon[i], rfield.otslin[i], rfield.ConInterOut[i], 
				  rfield.outlin[i] );
				fprintf( ioQQQ, " line continuum here are %4.4s %4.4s\n", 
				  rfield.chLineLabel[i], rfield.chContLabel[i] );

				++nInsane;
			}
		}
		phisig += rfield.flux[i];
		/* >>chng 02 sep 19, add sec test on SummedDif since it can be zero when 
		 * phisig is just above small float */
		if( phisig > 0. && rfield.SummedDif[i]> 0. )
		{
			if( fabs(rfield.SummedCon[i]/phisig-1.) > 1e-3 )
			{
				++nInsane;
				fprintf( ioQQQ, " PROBLEM RT_OTS_ChkSum%3ld, insane SummedCon at energy%.5e error=%.2e i=%ld\n", 
				  ipPnt, rfield.anu[i], rfield.SummedCon[i]/phisig - 1., i );
				fprintf( ioQQQ, " SummedCon, sum are %.4e %.4e\n", 
				  rfield.SummedCon[i], phisig );
				fprintf( ioQQQ, " otscon otslin ConInterOut outlin flux are%.4e %.4e %.4e %.4e %.4e\n", 
				  rfield.otscon[i], rfield.otslin[i], rfield.ConInterOut[i], 
				  rfield.outlin[i], rfield.flux[i] );
				fprintf( ioQQQ, " line continuum here are %s %s\n", 
				  rfield.chLineLabel[i], rfield.chContLabel[i]
				   );
			}
		}
	}

	if( nInsane > 0 )
	{
		fprintf( ioQQQ, " PROBLEM RT_OTS_ChkSum too much insanity to continue.\n");
		/* TotalInsanity exits after announcing a problem */
		TotalInsanity();
	}

#	ifdef DEBUG_FUN
	fputs( " <->RT_OTS_ChkSum()\n", debug_fp );
#	endif
	return;
}

/* =================================================================== */

/*RT_OTS_PrtRate print continuum and line ots rates when trace ots is on */
void RT_OTS_PrtRate(
	  /* weakest rate to print */
	  double weak ,
	  /* flag, 'c' continuum, 'l' line, 'b' both */
	  int chFlag )
{
	long int i;

#	ifdef DEBUG_FUN
	fputs( "<+>RT_OTS_PrtRate()\n", debug_fp );
#	endif

	/* arg must be one of these three */
	ASSERT( chFlag=='l' || chFlag=='c' || chFlag=='b' );

	/* 
	 * both printouts have cell number (on C array scale)
	 * energy in ryd
	 * the actual value of the ots rate 
	 * the ots rate relative to the continuum at that energy 
	 * rate times opacity 
	 * all are only printed if greater than weak 
	 */

	/*===================================================================*/
	/* first print ots continua                                          */
	/*===================================================================*/
	if( chFlag == 'c' || chFlag == 'b' )
	{
		fprintf( ioQQQ, "     OTSCON array, ANU, OTSCON, OTS/INCID OTS*opac limit=%.2e\n",weak );

		for( i=0; i < rfield.nupper; i++ )
		{
			if( rfield.otscon[i]*opac.opacity_abs[i] > weak )
			{
				fprintf( ioQQQ, "     %4ld%12.4e%12.4e%12.4e%12.4e %s \n", 
				  i, 
				  rfield.anu[i], 
				  rfield.otscon[i], 
				  rfield.otscon[i]/MAX2(1e-30,rfield.flux[i]),
				  rfield.otscon[i]*opac.opacity_abs[i], 
				  rfield.chContLabel[i]);

			}
		}
	}

	/*===================================================================*/
	/* second print ots line rates                                       */
	/*===================================================================*/
	if( chFlag == 'l' || chFlag == 'b' )
	{
		fprintf( ioQQQ, "     OTSLIN array, ANU, OTSLIN, OTS/INCID OTS*opac limit=%.2e\n",weak );

		for( i=0; i < rfield.nupper; i++ )
		{
			if( rfield.otslin[i]*opac.opacity_abs[i] > weak )
			{
				fprintf( ioQQQ, "     %4ld%12.4e%12.4e%12.4e%12.4e %s \n", 
				  i, 
				  rfield.anu[i], 
				  rfield.otslin[i], 
				  rfield.otslin[i]/MAX2(1e-30,rfield.flux[i]),
				  rfield.otslin[i]*opac.opacity_abs[i],
				  rfield.chLineLabel[i] );
			}
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->RT_OTS_PrtRate()\n", debug_fp );
#	endif
	return;
}
